/*
 * Copyright (c) 2019 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#pragma once

#include <endian.h>

/*
 * Android 9 device tree overlays support, for more info,
 * please visit https://source.android.com/devices/architecture/dto
 */

#define DT_TABLE_MAGIC 0xd7b7ab1e

enum {
    ANDROID_DTBO_V0 = 0,
    ANDROID_DTBO_V1,
};

struct dt_table_header {
    uint32_t magic;             /* DT_TABLE_MAGIC */
    uint32_t total_size;        /* includes dt_table_header +
                                   all dt_table_entry and all dtb/dtbo */
    uint32_t header_size;       /* sizeof(dt_table_header) */
    uint32_t dt_entry_size;     /* sizeof(dt_table_entry) */
    uint32_t dt_entry_count;    /* number of dt_table_entry */
    uint32_t dt_entries_offset; /* offset to the first dt_table_entry
                                   from head of dt_table_header */
    uint32_t page_size;         /* flash page size we assume */
    uint32_t version;           /* DTBO image version */
};

/* struct of dt_table_entry v0 */
struct dt_table_entry {
    uint32_t dt_size;           /* size of corresponding dtbo entry */
    uint32_t dt_offset;         /* offset from head of dt_table_header */
    uint32_t id;                /* optional, must be zero if unused */
    uint32_t rev;               /* optional, must be zero if unused */
    uint32_t custom[4];         /* optional, must be zero if unused */
};

enum dt_compression_info {
    NO_COMPRESSION,
    ZLIB_COMPRESSION,
    GZIP_COMPRESSION
};

struct dt_table_entry_v1 {
    uint32_t dt_size;           /* size of corresponding dtbo entry */
    uint32_t dt_offset;         /* offset from head of dt_table_header */
    uint32_t id;                /* optional, must be zero if unused */
    uint32_t rev;               /* optional, must be zero if unused */
    uint32_t flags;             /* For version 1 of dt_table_header, the 4
                                   least significant bits of 'flags' will be
                                   used to indicate the compression format
                                   of the DT entry as per the enum
                                   'dt_compression_info' */
    uint32_t custom[3];         /* optional, must be zero if unused */
};

/* android dtbo table header helper function */
#define _DEFINE_ANDROID_DTBO_READ_FUNC(field)                           \
    static inline uint32_t android_dtbo_ ## field(const void *dtbo)     \
    {                                                                   \
        return (BE32(((const struct dt_table_header *)(dtbo))->field)); \
    }

#define _DEFINE_ANDROID_DTBO_WRITE_FUNC(field)                              \
    static inline void android_dtbo_set_ ## field(void *dtbo,               \
            uint32_t val)                                                   \
    {                                                                       \
        struct dt_table_header *header = (struct dt_table_header *)dtbo;    \
        header->field = BE32(val);                                          \
    }

#define DEFINE_ANDROID_DTBO_RW_FUNC(field) \
    _DEFINE_ANDROID_DTBO_READ_FUNC(field)  \
    _DEFINE_ANDROID_DTBO_WRITE_FUNC(field)

DEFINE_ANDROID_DTBO_RW_FUNC(magic)
DEFINE_ANDROID_DTBO_RW_FUNC(total_size)
DEFINE_ANDROID_DTBO_RW_FUNC(header_size)
DEFINE_ANDROID_DTBO_RW_FUNC(dt_entry_size)
DEFINE_ANDROID_DTBO_RW_FUNC(dt_entry_count)
DEFINE_ANDROID_DTBO_RW_FUNC(dt_entries_offset)
DEFINE_ANDROID_DTBO_RW_FUNC(page_size)
DEFINE_ANDROID_DTBO_RW_FUNC(version)

/* android dtbo table entry macros */
#define android_dtbo_tbl_entry_v0(table_entry, field) \
     (BE32(((const struct dt_table_entry *)table_entry)->field))

#define android_dtbo_tbl_entry_v1(table_entry, field) \
     (BE32(((const struct dt_table_entry_v1 *)table_entry)->field))

#define android_dtbo_tbl_entry(table_entry, dt_ver, field) \
    (android_dtbo_tbl_entry_v ## dt_ver(table_entry, field))

/* macros to get table entry field */
#define android_dtbo_tbl_entry_dt_size(table_entry) \
    (android_dtbo_tbl_entry(table_entry, 0, dt_size))

#define android_dtbo_tbl_entry_dt_offset(table_entry) \
    (android_dtbo_tbl_entry(table_entry, 0, dt_offset))

#define android_dtbo_tbl_entry_id(table_entry) \
    (android_dtbo_tbl_entry(table_entry, 0, id))

#define android_dtbo_tbl_entry_rev(table_entry) \
    (android_dtbo_tbl_entry(table_entry, 0, rev))

#define android_dtbo_tbl_entry_flags(table_entry) \
    (android_dtbo_tbl_entry(table_entry, 1, flags))

#define android_dtbo_tbl_entry_compress_algo(table_entry) \
    (android_dtbo_tbl_entry(table_entry, 1, flags) & 0xf)

#define android_dtbo_tbl_entry_custom(table_entry, dt_ver, index) \
    (android_dtbo_tbl_entry_v ## dt_ver(table_entry, custom)[index])

/* android dtbo return code */
#define DTBO_RET_OK         0
#define DTBO_ERR_NULL       1
#define DTBO_ERR_INITIAL    2
#define DTBO_ERR_ENTRY_CNT  3
#define DTBO_ERR_BADMAGIC   9

/**
 * android_dtbo_check_header() - check android dtbo header
 *
 * @dtbo: android dtbo image start address
 *
 * returns:
 *     DTB_RET_OK, on success
 *     -DTBO_ERR_BADMAGIC, on magic check failiure
 */
int android_dtbo_check_header(const void *dtbo);

/**
* android_dtbo_get_dt_table_entry() - get specified dt_table_entry struct
*
* @dtbo:  android dtbo image start address
* @index: the table entry index
* @entry: return pointer to pointer to the table entry specified by 'index'
*
* returns:
*     DTB_RET_OK, on sucess
*     otherwise, on failure
*/
int android_dtbo_get_dt_table_entry(const void *dtbo, uint32_t index,
                                    struct dt_table_entry **entry);

/**
 * android_dtbo_get_dt_dtbo_entry() - get specified dtbo entry
 *
 * @dtbo:  android dtbo image start address
 * @index: the dtbo entry index
 *
 * returns:
 *     pointer to the dtbo entry, on success
 *     NULL, on failure
 */
void *android_dtbo_get_dt_dtbo_entry(const void *dtbo, uint32_t index);
