/*
 * Copyright (c) 2016 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _BOOTCTRL_H_
#define _BOOTCTRL_H_

#include <stdint.h>

/* struct boot_ctrl occupies the slot_suffix field of
 * struct bootloader_message */

#define OFFSETOF_METADATA_INFO 2048

#define BOOTCTRL_MAGIC 0x19191100
#define BOOTCTRL_SUFFIX_A      "_a"
#define BOOTCTRL_SUFFIX_B      "_b"
#define BOOTCTRL_SUFFIX_MAXLEN 2

#define BOOT_CONTROL_VERSION    1
#define READ_PARTITION          0
#define WRITE_PARTITION         1

typedef struct slot_metadata {
    uint8_t priority : 3;
    uint8_t retry_count : 3;
    uint8_t successful_boot : 1;
    uint8_t normal_boot : 1;
} slot_metadata_t;
typedef struct boot_ctrl {
    /* Magic for identification */
    uint32_t magic;

    /* Version of struct. */
    uint8_t version;

    /* Information about each slot. */
    slot_metadata_t slot_info[2];

    uint8_t recovery_retry_count;
} boot_ctrl_t;


typedef enum {
    BOOT_PART_NONE = 0,
    BOOT_PART_A,
    BOOT_PART_B
} boot_part_t;

#define AVB_AB_MAGIC "\0AB0"
#define AVB_AB_MAGIC_LEN 4

/* Versioning for the on-disk A/B metadata - keep in sync with avbtool. */
#define AVB_AB_MAJOR_VERSION 1
#define AVB_AB_MINOR_VERSION 0

/* Size of AvbABData struct. */
#define AVB_AB_DATA_SIZE 32

/* Maximum values for slot data */
#define AVB_AB_MAX_PRIORITY 15
#define AVB_AB_MAX_TRIES_REMAINING 7
/* bootctrl API */
/**
* get_suffix() returns the current slot' suffix.
*/
const char *get_suffix(void);
/**
* get_current_slot() returns the current slot used by partitions.
*/
int get_current_slot(void);
/**
* ab_metadata_init() init two slot information and marks the slot passed in
* parameter as the active boot slot.
* Returns whether the command succeeded.
*/
int ab_metadata_init(int slot);
/**
* get_retry_count() get the slot's boot retry count. if retry count < 0,
* should switch to next slot.
* Returns the retry count.
*/
uint8_t get_retry_count(int slot);
/**
* reduce_retry_count() reduce the slot's boot retry count.
*/
int reduce_retry_count(int slot);
/**
* check_valid_slot() check whether valid slot exist for boot.
* Returns whether the command succeeded.
*/
int check_valid_slot(void);
/**
* mark_slot_invalid() mark the slot as invalid.
* Returns whether the command succeeded.
*/
int mark_slot_invalid(int slot);
/**
* get_bootup_status() get whether the slot have been successfully booted.
* Returns whether the command succeeded.
*/
int get_bootup_status(int slot);
/**
* get_efuse_status() get the efuse status of the current slot.
* Returns the efuse status of the current slot.
*/
int get_efuse_status(int slot);
/**
* clear_efuse_status() clear the efuse status of the current slot.
* Returns update success or fail.
*/
int clear_efuse_status(int slot);
/**
* get_bl_ver_data() get the bl_ver data of the current slot.
* Returns the bl_ver data of the current slot.
*/
int get_bl_ver_data(int slot);

//xf.li@20230313 modify for ab_rollback start
/**
* mark_slot_unsuccessful() mark the slot unsuccessful.
* Returns mark success or fail.
*/
int mark_slot_unsuccessful(int slot);
//xf.li@20230313 modify for ab_rollback start

/**
 * check_ab_boot() - check ab boot status
 *
 * check ab boot status, and reduce retry count on current slot if necessary.
 * If the retry count reaches zero, bootctrl will switch the active slot to
 * the other slot.
 *
 * returns:
 *     0: success
 *     -EBADSLT: current boot slot marked as invalid, and set new boot slot done
 *     -ENOTSUP: ab boot device not supported
 */
int check_ab_boot(void);
#endif /* _BOOTCTRL_H_ */
