/*
 * Copyright (c) 2016 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#pragma once

#include <stdbool.h>
#include <sys/types.h>

/**
 * fit_get_image() - load fit image from a partition
 *
 * the function will use bio to access a partition from a storage and
 * check fdt header. If pass, allocate memory buffer to read the fit image
 * to load_buf
 *
 * @label:      partition name
 * @load_buf:   pointer to buffer pointer, the address of allocated memory
 *              buffer with fit image loaded was passing back to the caller
 *              via this argument.
 *
 * returns:
 *     0, on success
 *     otherwise, on failure
 *
 */
int fit_get_image(const char *label, void **load_buf);

/**
 * fit_image_get_data() - get data property of a subimage node
 * @fit:        fit image start address
 * @noffset:    the offset to the subimage node
 * @data:       return data pointer
 * @size:       return data size
 *
 * returns:
 *     0, on success
 *     otherwise, on failure
 */
int fit_image_get_data(const void *fit, int noffset,
                       const void **data, uint32_t *size);
/**
 * fit_conf_verify_sig() - verify fit configuration signature
 *
 * @conf:   configuration name
 * @fit:    fit image start address
 *
 * returns:
 *    0, on success
 *    otherwise, on failure
 */
int fit_conf_verify_sig(const char *conf, void *fit);

/**
 * fit_load_image() - load fit image to proper address
 *
 * This checks FIT configuration to find sub-image nodes image
 * and load the image to right address
 *
 * @conf:           configuration name
 * @img_pro:        image property name
 * @fit:            fit image start address
 * @load:           returned load address
 * @load_size:      returned loaded raw image size
 * @entry:          returned entry address
 * @need_verified:  whether to check image integrity
 *
 * returns:
 *    0, on success
 *    otherwise, on failure
 *
 */
int fit_load_image(const char *conf, const char *img_pro, void *fit,
                   addr_t *load, size_t *load_size, paddr_t *entry,
                   bool need_verified);

/**
 * fit_load_loadable_image() - load "loadable" images to "load" address
 *
 * This function finds "sub_node_name" loadable image nodes, do integrity check
 * per hash_check_enabled(), and load images to "load" address.
 *
 * @fit:            fit image start address
 * @sub_node_name:  loadable image subnode name
 * @load:           returned loadable image load address
 *
 * return:
 *      0: success
 *      otherwise: failure error code
 *
 */
int fit_load_loadable_image(void *fit, const char *sub_node_name, addr_t *load);
