
#pragma once

#define BYTE u8
#define __force
#define __bitwise

#define ML_BITS  4
#define COPYLENGTH 8
#define RUN_BITS (8 - ML_BITS)
#define RUN_MASK ((1U << RUN_BITS) - 1)
#define ML_MASK  ((1U << ML_BITS) - 1)

/*
 * Note: Uncompressed chunk size is used in the compressor side
 * (userspace side for compression).
 * It is hardcoded because there is not proper way to extract it
 * from the binary stream which is generated by the preliminary
 * version of LZ4 tool so far.
 */
#define ARCHIVE_MAGICNUMBER 0x184C2102
#define LZ4_DEFAULT_UNCOMPRESSED_CHUNK_SIZE (8 << 20)

#define unlikely(x) __builtin_expect(!!(x), 0)

static const int dec32table[] = {0, 3, 2, 3, 0, 0, 0, 0};
#if LZ4_ARCH64
static const int dec64table[] = {0, 0, 0, -1, 0, 1, 2, 3};
#endif

typedef unsigned int __u32;
typedef unsigned short __u16;
typedef __u16 __bitwise __le16;
typedef __u32 __bitwise __le32;

typedef struct _U16_S { u16 v; } U16_S;
typedef struct _U32_S { u32 v; } U32_S;
typedef struct _U64_S { u64 v; } U64_S;

#define le16_to_cpup __le16_to_cpup
#define le32_to_cpup __le32_to_cpup

#define A16(x) (((U16_S *)(x))->v)
#define A32(x) (((U32_S *)(x))->v)
#define A64(x) (((U64_S *)(x))->v)

#define PUT4(s, d) (A32(d) = A32(s))
#define PUT8(s, d) (A64(d) = A64(s))

#define LZ4_READ_LITTLEENDIAN_16(d, s, p) \
        (d = s - get_unaligned_le16(p))

#define LZ4_WILDCOPY(s, d, e)       \
        do {                        \
            LZ4_COPYPACKET(s, d);   \
        } while (d < e)

#if LZ4_ARCH64 /* 64-bit */
#define STEPSIZE 8

#define LZ4_COPYSTEP(s, d)  \
    do {            \
        PUT8(s, d); \
        d += 8;     \
        s += 8;     \
    } while (0)

#define LZ4_COPYPACKET(s, d)    LZ4_COPYSTEP(s, d)

#define LZ4_SECURECOPY(s, d, e)         \
    do {                    \
        if (d < e) {            \
            LZ4_WILDCOPY(s, d, e);  \
        }               \
    } while (0)

#else   /* 32-bit */
#define STEPSIZE 4

#define LZ4_COPYSTEP(s, d)  \
    do {            \
        PUT4(s, d); \
        d += 4;     \
        s += 4;     \
    } while (0)

#define LZ4_COPYPACKET(s, d)        \
    do {                \
        LZ4_COPYSTEP(s, d); \
        LZ4_COPYSTEP(s, d); \
    } while (0)

#define LZ4_SECURECOPY  LZ4_WILDCOPY

#endif

#define LZ4_OK 0 /* lz4 decode ok */
#define LZ4_FAIL 1 /* lz4 decode failure */
#define ELZ4WO 2 /* write overflow */
#define ELZ4IR 3 /* invalid return value */
#define ELZ4NP 4 /* null pointer */
#define ELZ4IH 5 /* invalid header */
#define ELZ4DF 6 /* decode failure */
#define ELZ4DC 7 /* data corrupt */

/**
 * unlz4() - do decompression for lz4 compressed format
 *
 * This decodes from input keeping compressed data
 * to output with uncompressed data. If input is equal to
 * output, the decompression will be failure.
 *
 * @input: lz4 compressed data address
 * @output: lz4 uncompressed data address
 * @in_len: size of lz4 compressed data
 *
 * returns:
 *    0 (i.e. LZ4_OK), on success
 *    otherwise, on failure
 *
 */
int unlz4(const void *input, long in_len, void *output);
