/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "spi.h"
#include <gpio.h>
#include <stdlib.h>
#include <mtk_spi.h>
#include <mtk_wrapper.h>

static struct mtk_spi_bus mtk_spi[] = {
    {
        .reg_addr = (void *)MTK_SPI0_BASE,
        .state = MTK_SPI_IDLE,
        .ismap = false
    },
    {
        .reg_addr = (void *)MTK_SPI2_BASE,
        .state = MTK_SPI_IDLE,
        .ismap = false
    },
    {
        .reg_addr = (void *)MTK_SPI3_BASE,
        .state = MTK_SPI_IDLE,
        .ismap = false
    },
    {
        .reg_addr = (void *)MTK_SPI4_BASE,
        .state = MTK_SPI_IDLE,
        .ismap = false
    },
    {
        .reg_addr = (void *)MTK_SPI5_BASE,
        .state = MTK_SPI_IDLE,
        .ismap = false
    },
};

void mtk_spi_dump_data(const char *name, const char *data, int size)
{
    int i;

    dprintf(SPEW, "%s:\r\n", name);
    for (i = 0; i < size; i++) {
        if (!(i % 6))
            dprintf(SPEW, "\r\n");
        dprintf(SPEW, "%x ", data[i]);
    }
    dprintf(SPEW, "\r\n");
}

void mtk_spi_dump_register_addr(struct mtk_spi_bus *bus)
{
    struct mtk_spi_regs *regs = bus->reg_addr;

    dprintf(SPEW, "spi_cfg0_reg:0x%lx\r\n", plat_wrap_kvaddr_to_paddr(&regs->spi_cfg0_reg));
    dprintf(SPEW, "spi_cfg1_reg:0x%lx\r\n", plat_wrap_kvaddr_to_paddr(&regs->spi_cfg1_reg));
    dprintf(SPEW, "spi_tx_src_reg:0x%lx\r\n", plat_wrap_kvaddr_to_paddr(&regs->spi_tx_src_reg));
    dprintf(SPEW, "spi_rx_dst_reg:0x%lx\r\n", plat_wrap_kvaddr_to_paddr(&regs->spi_rx_dst_reg));
    dprintf(SPEW, "spi_cmd_reg:0x%lx\r\n", plat_wrap_kvaddr_to_paddr(&regs->spi_cmd_reg));
    dprintf(SPEW, "spi_status1_reg:0x%lx\r\n", plat_wrap_kvaddr_to_paddr(&regs->spi_status1_reg));
}

void mtk_spi_dump_register(struct mtk_spi_bus *bus)
{
    struct mtk_spi_regs *regs = bus->reg_addr;

    dprintf(SPEW, "spi_cfg0_reg:0x%x\r\n", readl(&regs->spi_cfg0_reg));
    dprintf(SPEW, "spi_cfg1_reg:0x%x\r\n", readl(&regs->spi_cfg1_reg));
    dprintf(SPEW, "spi_tx_src_reg:0x%x\r\n", readl(&regs->spi_tx_src_reg));
    dprintf(SPEW, "spi_rx_dst_reg:0x%x\r\n", readl(&regs->spi_rx_dst_reg));
    dprintf(SPEW, "spi_cmd_reg:0x%x\r\n", readl(&regs->spi_cmd_reg));
    dprintf(SPEW, "spi_status1_reg:0x%x\r\n", readl(&regs->spi_status1_reg));
}

void mtk_spi_set_gpio_mode(int bus_num)
{
    if (bus_num == 0) {
        gpio_cust_set_mode(GPIO153,1);
        gpio_cust_set_mode(GPIO154,1);
        gpio_cust_set_mode(GPIO155,1);
        gpio_cust_set_mode(GPIO156,1);
    } else if (bus_num == 2) {
        gpio_cust_set_mode(GPIO127,1);
        gpio_cust_set_mode(GPIO128,1);
        gpio_cust_set_mode(GPIO129,1);
        gpio_cust_set_mode(GPIO130,1);
    } else if (bus_num == 3) {
        gpio_cust_set_mode(GPIO131,1);
        gpio_cust_set_mode(GPIO132,1);
        gpio_cust_set_mode(GPIO133,1);
        gpio_cust_set_mode(GPIO134,1);
    } else if (bus_num == 4) {
        gpio_cust_set_mode(GPIO165,1);
        gpio_cust_set_mode(GPIO166,1);
        gpio_cust_set_mode(GPIO167,1);
        gpio_cust_set_mode(GPIO168,1);
    } else if (bus_num == 5) {
        gpio_cust_set_mode(GPIO157,1);
        gpio_cust_set_mode(GPIO158,1);
        gpio_cust_set_mode(GPIO159,1);
        gpio_cust_set_mode(GPIO160,1);
    }
}

struct mtk_spi_bus *mtk_spi_bus_pa_to_kv(int bus_num)
{
    struct mtk_spi_bus *bus;

    if (bus_num >= 2)
        bus_num--;
    bus = &mtk_spi[bus_num];
    if (!bus->ismap) {
        bus->reg_addr = plat_wrap_paddr_to_kvaddr((unsigned long)bus->reg_addr);
        bus->ismap = true;
    }

    return bus;
}

struct mtk_spi_bus *mtk_spi_bus_get(int bus_num)
{
    if (bus_num >= 2)
        bus_num--;
    return &mtk_spi[bus_num];
}

void mtk_spi_reset(struct mtk_spi_bus *bus)
{
    int reg_val;
    struct mtk_spi_regs *regs = bus->reg_addr;

    reg_val = readl(&regs->spi_cmd_reg);
    reg_val |= 1 << SPI_CMD_RST_SHIFT;
    writel(reg_val, &regs->spi_cmd_reg);

    reg_val = readl(&regs->spi_cmd_reg);
    reg_val &= ~(1 << SPI_CMD_RST_SHIFT);
    writel(reg_val, &regs->spi_cmd_reg);
}

void mtk_spi_start(struct mtk_spi_bus *bus)
{
    int reg_val;
    struct mtk_spi_regs *regs = bus->reg_addr;

    bus->state = MTK_SPI_IDLE;

    /* set pause mode */
    reg_val = readl(&regs->spi_cmd_reg);
    reg_val |= 1 << SPI_CMD_PAUSE_EN_SHIFT;
    writel(reg_val, &regs->spi_cmd_reg);
}

void mtk_spi_stop(struct mtk_spi_bus *bus)
{
    int reg_val;
    struct mtk_spi_regs *regs = bus->reg_addr;

    mtk_spi_reset(bus);
    bus->state = MTK_SPI_IDLE;

    /* clear pause mode */
    reg_val = readl(&regs->spi_cmd_reg);
    reg_val &= ~(1 << SPI_CMD_PAUSE_EN_SHIFT);
    writel(reg_val, &regs->spi_cmd_reg);
}

void mtk_spi_trigger(struct mtk_spi_bus *bus)
{
    int reg_val;
    struct mtk_spi_regs *regs = bus->reg_addr;

    if (bus->state == MTK_SPI_IDLE) {
        reg_val = readl(&regs->spi_cmd_reg);
        reg_val |= 1 << SPI_CMD_ACT_SHIFT;
        writel(reg_val, &regs->spi_cmd_reg);
        bus->state = MTK_SPI_PAUSE_IDLE;
    } else if (bus->state == MTK_SPI_PAUSE_IDLE) {
        reg_val = readl(&regs->spi_cmd_reg);
        reg_val |= 1 << SPI_CMD_RESUME_SHIFT;
        writel(reg_val, &regs->spi_cmd_reg);
    }
}

void mtk_spi_packet(struct mtk_spi_bus *bus, int size)
{
    int reg_val, packet_len, packet_loop;
    struct mtk_spi_regs *regs = bus->reg_addr;

    packet_len = MIN(size, MTK_PACKET_SIZE);
    packet_loop = MTK_SPI_ROUNDUP_DIV(size, packet_len);

    reg_val = readl(&regs->spi_cfg1_reg);
    reg_val &= ~(SPI_CFG1_PACKET_LENGTH_MASK | SPI_CFG1_PACKET_LOOP_MASK);
    reg_val |= ((packet_len - 1) << SPI_CFG1_PACKET_LENGTH_SHIFT) |
               ((packet_loop - 1) << SPI_CFG1_PACKET_LOOP_SHIFT);
    writel(reg_val, &regs->spi_cfg1_reg);
}

int mtk_spi_transfer_is_end(struct mtk_spi_bus *bus)
{
    int i;
    struct mtk_spi_regs *regs = bus->reg_addr;

    /*
    *spi sw should wait for status1 register to idle before polling
    * status0 register for rx/tx finish.
    */

    i = 0;
    while ((readl(&regs->spi_status1_reg) & MTK_SPI_BUSY_STATUS) == 0) {
        i++;
        plat_wrap_delay(10);
        if (i > MTK_TXRX_TIMEOUT_US) {
            dprintf(INFO, "Timeout waiting for spi status1 register.\r\n");
            goto error;
        }
    }

    i = 0;
    while ((readl(&regs->spi_status0_reg) & MTK_SPI_PAUSE_FINISH_INT_STATUS) == 0) {
        i++;
        plat_wrap_delay(10);
        if (i > MTK_TXRX_TIMEOUT_US) {
            dprintf(INFO, "Timeout waiting for spi status0 register.\r\n");
            goto error;
        }
    }

    return 0;

error:
    mtk_spi_reset(bus);
    bus->state = MTK_SPI_IDLE;

    return -1;
}

int mtk_spi_fifo_transfer(struct mtk_spi_bus *bus, unsigned char *in, unsigned char *out, int size)
{
    int i, reg_val = 0, word_count, ret, remaind;
    struct mtk_spi_regs *regs = bus->reg_addr;

    if (!size || size > MTK_FIFO_DEPTH)
        return -1;

    mtk_spi_packet(bus, size);

    remaind = size & 0x03;//size % 4

    if (in && !out) {
        word_count = size >> 2;//size / 4

        if (remaind)
            word_count++;

        for (i = 0; i < word_count; i++)
            writel(MTK_ARBITRARY_VALUE, &regs->spi_tx_data_reg);
    }

    if (out) {
        reg_val = 0;
        for (i = 0; i < size - remaind; i++) {
            reg_val |= *(out + i) << ((i & 0x03) << 3);//<< 3 is * 8
            if ((i & 0x03) == 3) {
                writel(reg_val, &regs->spi_tx_data_reg);
                reg_val = 0;
            }

        }

        if (remaind) {
            reg_val = 0;
            for (i = 0; i < remaind; i++) {
                reg_val |= *(out + size - remaind + i) << ((i & 0x03) << 3);
            }
            writel(reg_val, &regs->spi_tx_data_reg);
        }
    }

    dprintf(SPEW, "spi before transfer:\r\n");
    mtk_spi_dump_register(bus);

    mtk_spi_trigger(bus);
    ret = mtk_spi_transfer_is_end(bus);

    if (!ret && in) {
        for (i = 0; i < size; i++) {
            if ((i & 0x03) == 0)
                reg_val = readl(&regs->spi_rx_data_reg);
            *(in + i) = (reg_val >> ((i & 0x03) << 3)) & 0xff;
        }
        mtk_spi_dump_data("spi after transfer in data is", (const char *)in, size);
    }

    return ret;
}

int mtk_spi_dma_transfer(struct mtk_spi_bus *bus, unsigned char *in, unsigned char *out, int size)
{
    int reg_val, ret;
    struct mtk_spi_regs *regs = bus->reg_addr;

    if (!in && out) {
        reg_val = readl(&regs->spi_cmd_reg);
        reg_val |= 1 << SPI_CMD_TX_DMA_SHIFT;
        writel(reg_val, &regs->spi_cmd_reg);
    } else {
        reg_val = readl(&regs->spi_cmd_reg);
        reg_val |= (1 << SPI_CMD_RX_DMA_SHIFT) |
                   (1 << SPI_CMD_TX_DMA_SHIFT);
        writel(reg_val, &regs->spi_cmd_reg);
    }

    if (out)
        writel((unsigned int)plat_wrap_kvaddr_to_paddr(out), &regs->spi_tx_src_reg);

    if (in)
        writel((unsigned int)plat_wrap_kvaddr_to_paddr(in), &regs->spi_rx_dst_reg);

    mtk_spi_packet(bus, size);

    dprintf(SPEW, "spi before transfer:\r\n");
    mtk_spi_dump_register(bus);

    mtk_spi_trigger(bus);
    ret = mtk_spi_transfer_is_end(bus);

    reg_val = readl(&regs->spi_cmd_reg);
    reg_val &= ~(1 << SPI_CMD_RX_DMA_SHIFT | 1 << SPI_CMD_TX_DMA_SHIFT);
    writel(reg_val, &regs->spi_cmd_reg);

    if (in)
        mtk_spi_dump_data("spi after transfer in data is", (const char *)in, size);

    return ret;
}

int mtk_spi_transfer(int bus_num, unsigned char *in, unsigned char *out, int size)
{
    int ret, min_size;
    struct mtk_spi_bus *bus = mtk_spi_bus_get(bus_num);

    if (!bus->ismap) {
        dprintf(SPEW, "paddr isn't map to vaddr,Maybe you didn't call spi_cust_init function before transfer.\n");
        return -1;
    }

    if (in)
        dprintf(SPEW, "in(0x%lx),size(%d)\r\n", plat_wrap_kvaddr_to_paddr((void *)in), size);
    if (out)
        dprintf(SPEW, "out(0x%lx),size(%d)\r\n", plat_wrap_kvaddr_to_paddr((void *)out), size);

    mtk_spi_start(bus);

    while (size) {
        min_size = MIN(size, MTK_PACKET_SIZE);

        if (size > MTK_FIFO_DEPTH)
            ret = mtk_spi_dma_transfer(bus, in, out, min_size);
        else
            ret = mtk_spi_fifo_transfer(bus, in, out, min_size);

        if (ret)
            return ret;

        size -= min_size;
        if (in)
            in = in + min_size;
        if (out)
            out = out + min_size;
    }

    mtk_spi_stop(bus);

    return ret;
}

int spi_cust_tx(int bus_num, void *tx, int size)
{
    return mtk_spi_transfer(bus_num, NULL, tx, size);
}

int spi_cust_rx(int bus_num, void *rx, int size)
{
    return mtk_spi_transfer(bus_num, rx, NULL, size);
}

int spi_cust_rx_tx(int bus_num, void *rx, void *tx, int size)
{
    return mtk_spi_transfer(bus_num, rx, tx, size);
}

void spi_cust_init(int bus_num, unsigned int speed_hz)
{
    uint32_t div, sck_ticks, cs_ticks, reg_val;
    struct mtk_spi_bus *bus = mtk_spi_bus_pa_to_kv(bus_num);
    struct mtk_spi_regs *regs = bus->reg_addr;

    if (speed_hz < (SPI_HZ / 2))
        div = MTK_SPI_ROUNDUP_DIV(SPI_HZ, speed_hz);
    else
        div = 1;

    sck_ticks = MTK_SPI_ROUNDUP_DIV(div, 2);
    cs_ticks = sck_ticks * 2;

    dprintf(SPEW,"spi current bus: %d, frequence initialized at %u Hz\r\n", bus_num, SPI_HZ / (sck_ticks * 2));
    dprintf(SPEW,"spi reg phy addr:\r\n");
    mtk_spi_dump_register_addr(bus);

    mtk_spi_set_gpio_mode(bus_num);

    /* set the timing */
    writel(((sck_ticks - 1) << SPI_CFG0_SCK_HIGH_SHIFT) |
           ((sck_ticks - 1) << SPI_CFG0_SCK_LOW_SHIFT) |
           ((cs_ticks - 1) << SPI_CFG0_CS_HOLD_SHIFT) |
           ((cs_ticks - 1) << SPI_CFG0_CS_SETUP_SHIFT),
           &regs->spi_cfg0_reg);

    reg_val = readl(&regs->spi_cfg1_reg);
    reg_val &= ~SPI_CFG1_CS_IDLE_MASK;
    reg_val |= (cs_ticks - 1) << SPI_CFG1_CS_IDLE_SHIFT;
    writel(reg_val, &regs->spi_cfg1_reg);

    reg_val = readl(&regs->spi_cmd_reg);

    reg_val &= ~SPI_CMD_CPHA_EN;
    reg_val &= ~SPI_CMD_CPOL_EN;

    /* set the mlsbx and mlsbtx */
    reg_val |= SPI_CMD_TXMSBF_EN;
    reg_val |= SPI_CMD_RXMSBF_EN;

    /* set the tx/rx endian */
    reg_val &= ~SPI_CMD_TX_ENDIAN_EN;
    reg_val &= ~SPI_CMD_RX_ENDIAN_EN;

    /* clear pause mode */
    reg_val &= ~SPI_CMD_PAUSE_EN;

    /* set finish interrupt always enable */
    reg_val |= SPI_CMD_FINISH_IE_EN;

    /* set pause interrupt always enable */
    reg_val |= SPI_CMD_PAUSE_IE_EN;

    /* disable dma mode */
    reg_val &= ~(SPI_CMD_TX_DMA_EN | SPI_CMD_RX_DMA_EN);

    /* set deassert mode */
    reg_val &= ~SPI_CMD_DEASSERT_EN;

    writel(reg_val, &regs->spi_cmd_reg);

    dprintf(SPEW, "spi after init:\r\n");
    mtk_spi_dump_register(bus);
}
