/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <mtk_wrapper.h>
#include "eint.h"

void *vaddr = NULL;

typedef struct {
    eint_handler eint_func[EINT_MAX_CHANNEL];
    void *arg[EINT_MAX_CHANNEL];
} eint_func;

eint_func EINT_FUNC;

/*
 * eint_cust_get_status: To get the interrupt status
 * @eint_num: the EINT number to get
 */
unsigned int eint_cust_get_status(unsigned int eint_num)
{
    unsigned long int base = 0;
    unsigned int st = 0;

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (eint_num < EINT_AP_MAXNUMBER) {
        base = (eint_num / 32) * 4 + EINT_STA_OFFSET;
    } else {
        dprintf(SPEW, "[EINT NUMBER ERROR] eint_num :%d \n",eint_num);
        return -1;
    }

    st = readl(vaddr+base);
    return st;
}

/*
 * eint_cust_ack: To ack the interrupt
 * @eint_num: the EINT number to set
 */
int eint_cust_ack(unsigned int eint_num)
{
    unsigned long int base = 0;
    unsigned int bit = 1 << (eint_num % 32);

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (eint_num < EINT_AP_MAXNUMBER) {
        base = (eint_num / 32) * 4 + EINT_INTACK_OFFSET;
    } else {
        dprintf (SPEW, "[EINT NUMBER ERROR] eint_num :%d \n",eint_num);
        return -1;
    }

    writel(bit, (vaddr+base));
    return 1;
}

/*
 * eint_cust_mask: Mask the specified EINT number.
 * @eint_num: EINT number to mask
 */
int eint_cust_mask(unsigned int eint_num)
{
    unsigned long int base = 0;
    unsigned int bit = 1 << (eint_num % 32);

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (eint_num < EINT_AP_MAXNUMBER) {
        base = (eint_num / 32) * 4 + EINT_MASK_SET_OFFSET;
    } else {
        dprintf(SPEW, "[EINT NUMBER ERROR] eint_num :%d \n",eint_num);
        return -1;
    }

    writel(bit, (vaddr+base));
    return 1;
}

/*
 * eint_cust_unmask: Unmask the specified EINT number.
 * @eint_num: EINT number to unmask
 */
int eint_cust_unmask(unsigned int eint_num)
{
    unsigned long int base = 0;
    unsigned int bit = 1 << (eint_num % 32);

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (eint_num < EINT_AP_MAXNUMBER) {
        base = (eint_num / 32) * 4 + EINT_MASK_CLR_OFFSET;
    } else {
        dprintf(SPEW, "[EINT NUMBER ERROR] eint_num :%d \n",eint_num);
        return -1;
    }

    writel(bit, (base+vaddr));
    return 1;
}

/*
 * eint_cust_set_sens: Set the sensitivity for the EINT number.
 * @eint_num: EINT number to set
 * @sens: sensitivity to set
 */
int eint_cust_set_sens(unsigned int eint_num, unsigned int sens)
{
    unsigned long int base =0;
    unsigned int bit = 1 << (eint_num % 32);

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (sens == EDGE_SENSITIVE) {
        base = (eint_num / 32) * 4 + EINT_SENS_CLR_OFFSET;
    } else if (sens == LEVEL_SENSITIVE) {
        base = (eint_num / 32) * 4 + EINT_SENS_SET_OFFSET;
    } else {
        dprintf(SPEW, "EINT sensitivity set ERROR\n");
        return -1;
    }

    writel(bit, (base+vaddr));
    return 1;
}

/*
 * eint_cust_set_polarity: Set the polarity for the EINT number.
 * @eint_num: EINT number to set
 * @pol: polarity to set
 */
int eint_cust_set_polarity(unsigned int eint_num, unsigned int pol)
{
    unsigned long int base;
    unsigned int bit = 1 << (eint_num % 32);

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (pol == LOW_POLARITY_TRIGGER) {
        base = (eint_num / 32) * 4 + EINT_POL_CLR_OFFSET;
    } else if (pol == HIGH_POLARITY_TRIGGER) {
        base = (eint_num / 32) * 4 + EINT_POL_SET_OFFSET;
    } else {
        dprintf(SPEW, "EINT polarity set ERROR\n");
        return -1;
    }

    writel(bit, (base+vaddr));
    return 1;
}

static enum wrap_handler_return eint_lisr(void *arg)
{
    unsigned int index;
    unsigned int status = 0;
    unsigned int status_check = 0;

    for (index = 0; index < EINT_MAX_CHANNEL; index++) {
        /* read status register every 32 interrupts */
        if (!(index % 32)) {
            status = eint_cust_get_status(index);

            if ((index == EINT_MAX_CHANNEL-1) && EINT_MAX_CHANNEL == 229) {
                status = eint_cust_get_status(index);
                dprintf(SPEW, "EINT Module - index:%d,EINT_STA = 0x%x\n",
                         index+1, status);
            }
        }

        status_check = status & (1 << (index % 32));

        if (status_check) {
            eint_cust_mask(index);
            dprintf(SPEW, "receive predicted index of EINT Module:%d,EINT_STA = 0x%x\n",
                    index, status);

            if (EINT_FUNC.eint_func[index]) {
                EINT_FUNC.eint_func[index] (EINT_FUNC.arg[index]);
            }
            eint_cust_ack(index);
            status &= ~ (1 << (index % 32));
        }
    }
    return WRAP_INT_RESCHEDULE;
}


void eint_cust_registration(unsigned int eint_num, eint_handler handler, void *arg)
{
    eint_cust_mask(eint_num);

    EINT_FUNC.eint_func[eint_num] = handler;
    EINT_FUNC.arg[eint_num] = 0;

    eint_cust_ack(eint_num);
}


int eint_cust_set_domain0 (void)
{
    unsigned int base = 0;
    unsigned int val = 0xFFFFFFFF;
    unsigned int ap_cnt = (EINT_MAX_CHANNEL / 32);
    unsigned int i;

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    if (EINT_AP_MAXNUMBER % 32)
        ap_cnt++;

    base = EINT_D0_EN_OFFSET;

    for (i = 0; i < ap_cnt; i++) {
        writel (val,(vaddr + base + (i * 4)));
    }

    return 1;
}

int eint_cust_init(void)
{
    unsigned int i;
    vaddr = plat_wrap_paddr_to_kvaddr(EINT_BASE);

    if (!vaddr) {
        dprintf(SPEW, "null or invalid EINT vaddr pointer\n");
        return -1;
    }

    for (i = 0; i < EINT_MAX_CHANNEL; i++) {
        EINT_FUNC.eint_func[i] = NULL;
        EINT_FUNC.arg[i] = NULL;
    }

    /* assign to domain for AP */
    eint_cust_set_domain0();

    plat_wrap_irq_set_sensitive(EINT_IRQ_BIT_ID, LEVEL_SENSITIVE);
    plat_wrap_irq_set_polarity(EINT_IRQ_BIT_ID, HIGH_POLARITY_TRIGGER);
    plat_wrap_register_int_handler(EINT_IRQ_BIT_ID, eint_lisr, NULL);
    plat_wrap_unmask_interrupt(EINT_IRQ_BIT_ID);

    return 1;
}
