/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
#include <gpio.h>
#include "gpio_in.h"

#define WRITE_REGISTER_UINT16(reg, val) (*(volatile unsigned short * const)(reg)) = (val)
#define OUTREG16(x, y) WRITE_REGISTER_UINT16((unsigned short *)(x),(unsigned short)(y))
#define DRV_WriteReg16(addr, data) OUTREG16(addr, data)

#define READ_REGISTER_UINT16(reg) (*(volatile unsigned short * const)(reg))
#define INREG16(x) READ_REGISTER_UINT16((unsigned short *)(x))
#define DRV_Reg16(addr) INREG16(addr)

/******************************************************************************
 MACRO Definition
******************************************************************************/
#define GPIO_DEVICE "mt-gpio"
#define VERSION     "$Revision$"
/*---------------------------------------------------------------------------*/
#define ARRAY_SIZE(a)       (sizeof(a) / sizeof(a[0]))
#define GPIO_WR32(addr, data)   DRV_WriteReg16(addr,data)
#define GPIO_RD32(addr)         DRV_Reg16(addr)
#define GPIO_SET_BITS(BIT,REG)   ((*(volatile u16*)(REG)) = (u16)(BIT))
#define GPIO_CLR_BITS(BIT,REG)   ((*(volatile u16*)(REG)) &= ~((u16)(BIT)))
/*---------------------------------------------------------------------------*/
#define TRUE                   1
#define FALSE                  0
/*---------------------------------------------------------------------------*/
#define MAX_GPIO_REG_BITS      16
#define MAX_GPIO_MODE_PER_REG  5
#define GPIO_MODE_BITS         3
/*---------------------------------------------------------------------------*/

/*
 * auto2712m1v1 io power domain
 * define 1: 3.3v
 * define 0: 1.8v
 */
#define VCC3IO_GPIO    1
#define VCC3IO_GBE     1
#define VCC3IO_NOR     1
#define VCC3IO_MSDC1   1
#define VCC3IO_MSDC2   1
#define VCC3IO_UART4B  1
#define VCC3IO_SPICTP  0
#define VCC3IO_SPIMCU  1
#define VCC3IO_I2S102  1
#define VCC3IO_I2SIO1  1
#define VCC3IO_I2SO02  1
#define VCC3IO_TDMO    1

enum gpio_rdsel_type {
    GPIO,   /* VCC3IO_GPIO */
    GBE,    /* VCC3IO_GBE */
    NOR,    /* VCC3IO_NOR */
    MSDC1,  /* VCC3IO_MSDC1 */
    MSDC2,  /* VCC3IO_MSDC2 */
    UART4B, /* VCC3IO_UART4B */
    SPICTP, /* VCC3IO_SPICTP */
    SPIMCU, /* VCC3IO_SPIMCU */
    I2S102, /* VCC3IO_I2S102 */
    I2SIO1, /* VCC3IO_I2SIO1 */
    I2SO02, /* VCC3IO_I2SO02 */
    TDMO,   /* VCC3IO_TDMO */
};

int gpio_type_in[] = {
    [GPIO] = VCC3IO_GPIO,
    [GBE] = VCC3IO_GBE,
    [NOR] = VCC3IO_NOR,
    [MSDC1] = VCC3IO_MSDC1,
    [MSDC2] = VCC3IO_MSDC2,
    [UART4B] = VCC3IO_UART4B,
    [SPICTP] = VCC3IO_SPICTP,
    [SPIMCU] = VCC3IO_SPIMCU,
    [I2S102] = VCC3IO_I2S102,
    [I2SIO1] = VCC3IO_I2SIO1,
    [I2SO02] = VCC3IO_I2SO02,
    [TDMO] = VCC3IO_TDMO
};

static inline void gpio_set_rdsel(enum gpio_rdsel_type type, addr_t reg, int start_pos)
{
    u32 val = 0;

    if (gpio_type_in[type] == 1) {
        val |= 0x0c << start_pos;
    }
    writel((readl(reg) & (~(0x3f << start_pos))) | val, reg);
}

/**
 * struct mtk_pin_spec_pupd_set_samereg
 * - For special pins' pull up/down setting which resides in same register
 * @pin: The pin number.
 * @offset: The offset of special pull up/down setting register.
 * @pupd_bit: The pull up/down bit in this register.
 * @r0_bit: The r0 bit of pull resistor.
 * @r1_bit: The r1 bit of pull resistor.
 */
struct mtk_pin_spec_pupd_set_samereg {
    unsigned short pin;
    unsigned short offset;
    unsigned char pupd_bit;
    unsigned char r1_bit;
    unsigned char r0_bit;
};

#define MTK_PIN_PUPD_SPEC_SR(_pin, _offset, _pupd, _r1, _r0)    \
    {   \
        .pin = _pin,    \
        .offset = _offset,  \
        .pupd_bit = _pupd,  \
        .r1_bit = _r1,      \
        .r0_bit = _r0,      \
    }

static const struct mtk_pin_spec_pupd_set_samereg mt2712_spec_pupd[] = {
    MTK_PIN_PUPD_SPEC_SR(18, 0xe50, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(19, 0xe60, 12, 11, 10),
    MTK_PIN_PUPD_SPEC_SR(20, 0xe50, 5, 4, 3),
    MTK_PIN_PUPD_SPEC_SR(21, 0xe60, 15, 14, 13),
    MTK_PIN_PUPD_SPEC_SR(22, 0xe50, 8, 7, 6),
    MTK_PIN_PUPD_SPEC_SR(23, 0xe70, 2, 1, 0),

    MTK_PIN_PUPD_SPEC_SR(30, 0xf30, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(31, 0xf30, 6, 5, 4),
    MTK_PIN_PUPD_SPEC_SR(32, 0xf30, 10, 9, 8),
    MTK_PIN_PUPD_SPEC_SR(33, 0xf30, 14, 13, 12),
    MTK_PIN_PUPD_SPEC_SR(34, 0xf40, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(35, 0xf40, 6, 5, 4),
    MTK_PIN_PUPD_SPEC_SR(36, 0xf40, 10, 9, 8),
    MTK_PIN_PUPD_SPEC_SR(37, 0xc40, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(38, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(39, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(40, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(41, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(42, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(43, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(44, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(45, 0xc60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(46, 0xc50, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(47, 0xda0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(48, 0xd90, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(49, 0xdf0, 14, 13, 12),
    MTK_PIN_PUPD_SPEC_SR(50, 0xdf0, 10, 9, 8),
    MTK_PIN_PUPD_SPEC_SR(51, 0xdf0, 6, 5, 4),
    MTK_PIN_PUPD_SPEC_SR(52, 0xdf0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(53, 0xd50, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(54, 0xd80, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(55, 0xe00, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(56, 0xd40, 2, 1, 0),

    MTK_PIN_PUPD_SPEC_SR(63, 0xc80, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(64, 0xdb0, 14, 13, 12),
    MTK_PIN_PUPD_SPEC_SR(65, 0xdb0, 6, 5, 4),
    MTK_PIN_PUPD_SPEC_SR(66, 0xdb0, 10, 9, 8),
    MTK_PIN_PUPD_SPEC_SR(67, 0xcd0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(68, 0xdb0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(69, 0xc90, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(70, 0xcc0, 2, 1, 0),

    MTK_PIN_PUPD_SPEC_SR(89, 0xce0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(90, 0xdd0, 14, 13, 12),
    MTK_PIN_PUPD_SPEC_SR(91, 0xdd0, 10, 9, 8),
    MTK_PIN_PUPD_SPEC_SR(92, 0xdd0, 6, 5, 4),
    MTK_PIN_PUPD_SPEC_SR(93, 0xdd0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(94, 0xd20, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(95, 0xcf0, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(96, 0xd30, 2, 1, 0),

    MTK_PIN_PUPD_SPEC_SR(135, 0xe50, 11, 10, 9),
    MTK_PIN_PUPD_SPEC_SR(136, 0xe50, 14, 13, 12),
    MTK_PIN_PUPD_SPEC_SR(137, 0xe70, 5, 4, 3),
    MTK_PIN_PUPD_SPEC_SR(138, 0xe70, 8, 7, 6),
    MTK_PIN_PUPD_SPEC_SR(139, 0xe70, 11, 10, 9),
    MTK_PIN_PUPD_SPEC_SR(140, 0xe70, 14, 13, 12),
    MTK_PIN_PUPD_SPEC_SR(141, 0xe60, 2, 1, 0),
    MTK_PIN_PUPD_SPEC_SR(142, 0xe60, 5, 4, 3)
};

const struct mtk_pin_spec_pupd_set_samereg *mt_get_spec_pupd_ctrl(unsigned long pin)
{
    unsigned int i;

    for (i = 0; i < ARRAY_SIZE(mt2712_spec_pupd); i++) {
        if (pin == mt2712_spec_pupd[i].pin) {
            return &mt2712_spec_pupd[i];
        }
    }
    return NULL;
}

#define PUPD_OFFSET_TO_REG(spec_pupd_ctrl) ((VAL_REGS*)(GPIO_BASE + spec_pupd_ctrl->offset))

struct mt_gpio_obj {
    GPIO_REGS       *reg;
};

static struct mt_gpio_obj gpio_dat;

static struct mt_gpio_obj *gpio_obj = &gpio_dat;
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_dir_chip(u32 pin, u32 dir)
{
    u32 pos;
    u32 bit;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    if (dir >= GPIO_DIR_MAX)
        return -ERINVAL;

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    if (dir == GPIO_DIR_IN)
        GPIO_SET_BITS((1L << bit), &obj->reg->dir[pos].rst);
    else
        GPIO_SET_BITS((1L << bit), &obj->reg->dir[pos].set);
    return RSUCCESS;

}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_dir_chip(u32 pin)
{
    u32 pos;
    u32 bit;
    u32 reg;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    reg = GPIO_RD32(&obj->reg->dir[pos].val);
    return (((reg & (1L << bit)) != 0)? 1: 0);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_pull_enable_chip(u32 pin, u32 enable, u32 r1r0)
{
    u32 pos;
    u32 bit;

    struct mt_gpio_obj *obj = gpio_obj;
    const struct mtk_pin_spec_pupd_set_samereg *spec_pupd_ctrl;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN) {
        return -ERINVAL;
    }

    if (enable >= GPIO_PULL_EN_MAX) {
        return -ERINVAL;
    }

    if (r1r0 >= GPIO_SPEC_PULL_EN_MAX) {
        return -ERINVAL;
    }

    /*for special pin pupd*/
    if (NULL != (spec_pupd_ctrl = mt_get_spec_pupd_ctrl(pin))) {
        switch (r1r0) {
            case MTK_PUPD_SET_R1R0_00:
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r0_bit)), &(PUPD_OFFSET_TO_REG(spec_pupd_ctrl)->rst));
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r1_bit)), &(PUPD_OFFSET_TO_REG(spec_pupd_ctrl)->rst));
                break;
            case MTK_PUPD_SET_R1R0_01:
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r0_bit)), &((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->set));
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r1_bit)), &((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->rst));
                break;
            case MTK_PUPD_SET_R1R0_10:
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r0_bit)), &((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->rst));
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r1_bit)), &((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->set));
                break;
            case MTK_PUPD_SET_R1R0_11:
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r0_bit)), &((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->set));
                GPIO_SET_BITS((1L << (spec_pupd_ctrl->r1_bit)), &((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->set));
                break;
            default:
                return -ERINVAL;
        }
        return RSUCCESS;
    }
    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    if (enable == GPIO_PULL_DISABLE)
        GPIO_SET_BITS((1L << bit), &obj->reg->pullen[pos].rst);
    else
        GPIO_SET_BITS((1L << bit), &obj->reg->pullen[pos].set);
    return RSUCCESS;

}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_pull_enable_chip(u32 pin)
{
    u32 pos;
    u32 bit;
    u32 reg=0;
    struct mt_gpio_obj *obj = gpio_obj;
    const struct mtk_pin_spec_pupd_set_samereg *spec_pupd_ctrl;
    unsigned long data=0;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    /*for special msdc pupd*/
    if (NULL != (spec_pupd_ctrl = mt_get_spec_pupd_ctrl(pin))) {
        data = GPIO_RD32(&PUPD_OFFSET_TO_REG(spec_pupd_ctrl)->val);
        return ((data >> (spec_pupd_ctrl->r0_bit)) & 3L);
    }


    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    reg = GPIO_RD32(&obj->reg->pullen[pos].val);
    return (((reg & (1L << bit)) != 0)? 1: 0);
}

/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_pull_select_chip(u32 pin, u32 select)
{
    u32 pos;
    u32 bit;
    struct mt_gpio_obj *obj = gpio_obj;
    const struct mtk_pin_spec_pupd_set_samereg *spec_pupd_ctrl;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    if (select >= GPIO_PULL_MAX)
        return -ERINVAL;

    /*for special msdc pupd*/
    if (NULL != (spec_pupd_ctrl = mt_get_spec_pupd_ctrl(pin))) {
        if (select == GPIO_PULL_DOWN) {
            GPIO_SET_BITS((1L << (spec_pupd_ctrl->pupd_bit)), &(PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->set);
        } else {
            GPIO_SET_BITS((1L << (spec_pupd_ctrl->pupd_bit)), &(PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->rst);
        }
        return RSUCCESS;
    }

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    if (select == GPIO_PULL_DOWN)
        GPIO_SET_BITS((1L << bit), &obj->reg->pullsel[pos].rst);
    else
        GPIO_SET_BITS((1L << bit), &obj->reg->pullsel[pos].set);
    return RSUCCESS;
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_pull_select_chip(u32 pin)
{
    u32 pos;
    u32 bit;
    u32 reg=0;
    struct mt_gpio_obj *obj = gpio_obj;
    const struct mtk_pin_spec_pupd_set_samereg *spec_pupd_ctrl;
    unsigned long data=0;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    /*for special msdc pupd*/
    if (NULL != (spec_pupd_ctrl = mt_get_spec_pupd_ctrl(pin))) {
        data = GPIO_RD32(&((PUPD_OFFSET_TO_REG(spec_pupd_ctrl))->val));
        return (((data & (1L << (spec_pupd_ctrl->pupd_bit))) != 0)? 0: 1);
    }

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    reg = GPIO_RD32(&obj->reg->pullsel[pos].val);
    return (((reg & (1L << bit)) != 0)? 1: 0);

}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_out_chip(u32 pin, u32 output)
{

    u32 pos;
    u32 bit;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    if (output >= GPIO_OUT_MAX)
        return -ERINVAL;

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    if (output == GPIO_OUT_ZERO)
        GPIO_SET_BITS((1L << bit), &obj->reg->dout[pos].rst);
    else
        GPIO_SET_BITS((1L << bit), &obj->reg->dout[pos].set);
    return RSUCCESS;


}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_out_chip(u32 pin)
{
    u32 pos;
    u32 bit;
    u32 reg;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    reg = GPIO_RD32(&obj->reg->dout[pos].val);
    return (((reg & (1L << bit)) != 0)? 1: 0);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_in_chip(u32 pin)
{
    u32 pos;
    u32 bit;
    u32 reg;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    pos = pin / MAX_GPIO_REG_BITS;
    bit = pin % MAX_GPIO_REG_BITS;

    reg = GPIO_RD32(&obj->reg->din[pos].val);
    return (((reg & (1L << bit)) != 0)? 1: 0);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_mode_chip(u32 pin, u32 mode)
{
    u32 pos;
    u32 bit;
    u32 reg;
    u32 mask = (1L << GPIO_MODE_BITS) - 1;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN) {
        return -ERINVAL;
    }

    if (mode >= GPIO_MODE_MAX) {
        return -ERINVAL;
    }

    pos = pin / MAX_GPIO_MODE_PER_REG;
    bit = pin % MAX_GPIO_MODE_PER_REG;

    reg = GPIO_RD32(&obj->reg->mode[pos].val);

    reg &= ~(mask << (GPIO_MODE_BITS*bit));
    reg |= (mode << (GPIO_MODE_BITS*bit));

    GPIO_WR32(&obj->reg->mode[pos].val, reg);
    return RSUCCESS;
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_mode_chip(u32 pin)
{
    u32 pos;
    u32 bit;
    u32 reg;
    u32 mask = (1L << GPIO_MODE_BITS) - 1;
    struct mt_gpio_obj *obj = gpio_obj;

    if (!obj->reg) {
        return -ERACCESS;
    }

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    pos = pin / MAX_GPIO_MODE_PER_REG;
    bit = pin % MAX_GPIO_MODE_PER_REG;

    reg = GPIO_RD32(&obj->reg->mode[pos].val);
    return ((reg >> (GPIO_MODE_BITS*bit)) & mask);
}

/*set driving*/
struct mtk_drv_group_desc {
    unsigned char min_drv;
    unsigned char max_drv;
    unsigned char low_bit;
    unsigned char high_bit;
    unsigned char step;
};
struct mtk_pin_drv_grp {
    unsigned short pin;
    unsigned short offset;
    unsigned char bit;
    unsigned char grp;
};

#define MTK_PIN_DRV_GRP(_pin, _offset, _bit, _grp)  \
    {   \
        .pin = _pin,    \
        .offset = _offset,  \
        .bit = _bit,    \
        .grp = _grp,    \
    }

#define MTK_DRV_GRP(_min, _max, _low, _high, _step) \
    {   \
        .min_drv = _min,    \
        .max_drv = _max,    \
        .low_bit = _low,    \
        .high_bit = _high,  \
        .step = _step,      \
    }

static const struct mtk_drv_group_desc mt2712_drv_grp[] =  {
    /* 0E4E8SR 4/8/12/16 */
    MTK_DRV_GRP(4, 16, 1, 2, 4),
    /* 0E2E4SR  2/4/6/8 */
    MTK_DRV_GRP(2, 8, 1, 2, 2),
    /* E8E4E2  2/4/6/8/10/12/14/16 */
    MTK_DRV_GRP(2, 16, 0, 2, 2)
};
#define DRV_GROUP_MAX 3

static const struct mtk_pin_drv_grp mt2712_pin_drv[] = {
    MTK_PIN_DRV_GRP(0, 0xc10, 4, 0),
    MTK_PIN_DRV_GRP(1, 0xc10, 4, 0),
    MTK_PIN_DRV_GRP(2, 0xc10, 4, 0),
    MTK_PIN_DRV_GRP(3, 0xc10, 4, 0),
    MTK_PIN_DRV_GRP(4, 0xc00, 12, 0),
    MTK_PIN_DRV_GRP(5, 0xc00, 12, 0),
    MTK_PIN_DRV_GRP(6, 0xc00, 12, 0),
    MTK_PIN_DRV_GRP(7, 0xc00, 12, 0),
    MTK_PIN_DRV_GRP(8, 0xc10, 0, 0),
    MTK_PIN_DRV_GRP(9, 0xc10, 0, 0),
    MTK_PIN_DRV_GRP(10, 0xc10, 0, 0),
    MTK_PIN_DRV_GRP(11, 0xc10, 0, 0),
    MTK_PIN_DRV_GRP(12, 0xb60, 0, 0),
    MTK_PIN_DRV_GRP(13, 0xb60, 4, 0),
    MTK_PIN_DRV_GRP(14, 0xb60, 0, 0),
    MTK_PIN_DRV_GRP(15, 0xb60, 4, 0),
    MTK_PIN_DRV_GRP(18, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(19, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(20, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(21, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(22, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(23, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(24, 0xb40, 4, 0),
    MTK_PIN_DRV_GRP(25, 0xb40, 8, 0),
    MTK_PIN_DRV_GRP(26, 0xb40, 12, 0),
    MTK_PIN_DRV_GRP(27, 0xb50, 0, 0),
    MTK_PIN_DRV_GRP(28, 0xb40, 12, 0),
    MTK_PIN_DRV_GRP(29, 0xb40, 12, 0),
    MTK_PIN_DRV_GRP(30, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(31, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(32, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(33, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(34, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(35, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(36, 0xf50, 8, 2),
    MTK_PIN_DRV_GRP(37, 0xc40, 8, 2),
    MTK_PIN_DRV_GRP(38, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(39, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(40, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(41, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(42, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(43, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(44, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(45, 0xc60, 8, 2),
    MTK_PIN_DRV_GRP(46, 0xc50, 8, 2),
    MTK_PIN_DRV_GRP(47, 0xda0, 8, 2),
    MTK_PIN_DRV_GRP(48, 0xd90, 8, 2),
    MTK_PIN_DRV_GRP(49, 0xd60, 8, 2),
    MTK_PIN_DRV_GRP(50, 0xd60, 8, 2),
    MTK_PIN_DRV_GRP(51, 0xd60, 8, 2),
    MTK_PIN_DRV_GRP(52, 0xd60, 8, 2),
    MTK_PIN_DRV_GRP(53, 0xd50, 8, 2),
    MTK_PIN_DRV_GRP(54, 0xd80, 8, 2),
    MTK_PIN_DRV_GRP(55, 0xe00, 8, 2),
    MTK_PIN_DRV_GRP(56, 0xd40, 8, 2),

    MTK_PIN_DRV_GRP(63, 0xc80, 8, 2),
    MTK_PIN_DRV_GRP(64, 0xca0, 8, 2),
    MTK_PIN_DRV_GRP(65, 0xca0, 8, 2),
    MTK_PIN_DRV_GRP(66, 0xca0, 8, 2),
    MTK_PIN_DRV_GRP(67, 0xcd0, 8, 2),
    MTK_PIN_DRV_GRP(68, 0xca0, 8, 2),
    MTK_PIN_DRV_GRP(69, 0xc90, 8, 2),
    MTK_PIN_DRV_GRP(70, 0xcc0, 8, 2),
    MTK_PIN_DRV_GRP(71, 0xb60, 8, 1),
    MTK_PIN_DRV_GRP(72, 0xb60, 8, 1),
    MTK_PIN_DRV_GRP(73, 0xb60, 8, 1),
    MTK_PIN_DRV_GRP(74, 0xb60, 8, 1),
    MTK_PIN_DRV_GRP(75, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(76, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(77, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(78, 0xb70, 0, 1),
    MTK_PIN_DRV_GRP(79, 0xb70, 0, 1),
    MTK_PIN_DRV_GRP(80, 0xb70, 0, 1),
    MTK_PIN_DRV_GRP(81, 0xb70, 0, 1),
    MTK_PIN_DRV_GRP(82, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(83, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(84, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(85, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(86, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(87, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(88, 0xb60, 12, 1),
    MTK_PIN_DRV_GRP(89, 0xce0, 8, 2),
    MTK_PIN_DRV_GRP(90, 0xd00, 8, 2),
    MTK_PIN_DRV_GRP(91, 0xd00, 8, 2),
    MTK_PIN_DRV_GRP(92, 0xd00, 8, 2),
    MTK_PIN_DRV_GRP(93, 0xd00, 8, 2),
    MTK_PIN_DRV_GRP(94, 0xd20, 8, 2),
    MTK_PIN_DRV_GRP(95, 0xcf0, 8, 2),
    MTK_PIN_DRV_GRP(96, 0xd30, 8, 2),
    MTK_PIN_DRV_GRP(97, 0xb70, 4, 0),
    MTK_PIN_DRV_GRP(98, 0xb70, 4, 0),
    MTK_PIN_DRV_GRP(99, 0xb70, 4, 0),
    MTK_PIN_DRV_GRP(100, 0xb70, 4, 0),
    MTK_PIN_DRV_GRP(101, 0xb70, 8, 0),
    MTK_PIN_DRV_GRP(102, 0xb70, 8, 0),
    MTK_PIN_DRV_GRP(103, 0xb70, 8, 0),
    MTK_PIN_DRV_GRP(104, 0xb70, 8, 0),
    MTK_PIN_DRV_GRP(135, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(136, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(137, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(138, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(139, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(140, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(141, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(142, 0xb40, 0, 1),
    MTK_PIN_DRV_GRP(143, 0xba0, 12, 0),
    MTK_PIN_DRV_GRP(144, 0xba0, 12, 0),
    MTK_PIN_DRV_GRP(145, 0xba0, 12, 0),
    MTK_PIN_DRV_GRP(146, 0xba0, 12, 0),
    MTK_PIN_DRV_GRP(147, 0xba0, 12, 0),
    MTK_PIN_DRV_GRP(148, 0xbb0, 0, 0),
    MTK_PIN_DRV_GRP(149, 0xbb0, 0, 0),
    MTK_PIN_DRV_GRP(150, 0xbb0, 0, 0),
    MTK_PIN_DRV_GRP(151, 0xbb0, 0, 0),
    MTK_PIN_DRV_GRP(152, 0xbb0, 0, 0),
    MTK_PIN_DRV_GRP(153, 0xbb0, 4, 0),
    MTK_PIN_DRV_GRP(154, 0xbb0, 4, 0),
    MTK_PIN_DRV_GRP(155, 0xbb0, 4, 0),
    MTK_PIN_DRV_GRP(156, 0xbb0, 4, 0),
    MTK_PIN_DRV_GRP(157, 0xbb0, 8, 0),
    MTK_PIN_DRV_GRP(158, 0xbb0, 8, 0),
    MTK_PIN_DRV_GRP(159, 0xbb0, 8, 0),
    MTK_PIN_DRV_GRP(160, 0xbb0, 8, 0),
    MTK_PIN_DRV_GRP(161, 0xbb0, 12, 0),
    MTK_PIN_DRV_GRP(162, 0xbb0, 12, 0),
    MTK_PIN_DRV_GRP(163, 0xbb0, 12, 0),
    MTK_PIN_DRV_GRP(164, 0xbb0, 12, 0),
    MTK_PIN_DRV_GRP(165, 0xbc0, 0, 0),
    MTK_PIN_DRV_GRP(166, 0xbc0, 0, 0),
    MTK_PIN_DRV_GRP(167, 0xbc0, 0, 0),
    MTK_PIN_DRV_GRP(168, 0xbc0, 0, 0),
    MTK_PIN_DRV_GRP(169, 0xbc0, 4, 0),
    MTK_PIN_DRV_GRP(170, 0xbc0, 4, 0),
    MTK_PIN_DRV_GRP(171, 0xbc0, 8, 0),
    MTK_PIN_DRV_GRP(172, 0xbc0, 8, 0),
    MTK_PIN_DRV_GRP(173, 0xbc0, 12, 0),
    MTK_PIN_DRV_GRP(174, 0xbd0, 0, 0),
    MTK_PIN_DRV_GRP(175, 0xbd0, 0, 0),
    MTK_PIN_DRV_GRP(176, 0xbc0, 12, 0),
    MTK_PIN_DRV_GRP(177, 0xbd0, 4, 0),
    MTK_PIN_DRV_GRP(178, 0xbd0, 8, 0),
    MTK_PIN_DRV_GRP(179, 0xbd0, 4, 0),
    MTK_PIN_DRV_GRP(180, 0xbd0, 8, 0),
    MTK_PIN_DRV_GRP(181, 0xbd0, 12, 0),
    MTK_PIN_DRV_GRP(182, 0xbe0, 0, 0),
    MTK_PIN_DRV_GRP(183, 0xbd0, 12, 0),
    MTK_PIN_DRV_GRP(184, 0xbe0, 0, 0),
    MTK_PIN_DRV_GRP(185, 0xbe0, 4, 0),
    MTK_PIN_DRV_GRP(186, 0xbe0, 8, 0),
    MTK_PIN_DRV_GRP(187, 0xbe0, 12, 0),
    MTK_PIN_DRV_GRP(188, 0xbf0, 0, 0),
    MTK_PIN_DRV_GRP(189, 0xbe0, 12, 0),
    MTK_PIN_DRV_GRP(190, 0xbf0, 0, 0),
    MTK_PIN_DRV_GRP(191, 0xbf0, 4, 0),
    MTK_PIN_DRV_GRP(192, 0xbf0, 8, 0),
    MTK_PIN_DRV_GRP(193, 0xbf0, 4, 0),
    MTK_PIN_DRV_GRP(194, 0xbf0, 4, 0),
    MTK_PIN_DRV_GRP(195, 0xbf0, 8, 0),
    MTK_PIN_DRV_GRP(196, 0xbf0, 12, 0),
    MTK_PIN_DRV_GRP(197, 0xbf0, 12, 0),
    MTK_PIN_DRV_GRP(198, 0xbf0, 12, 0),
    MTK_PIN_DRV_GRP(199, 0xbf0, 12, 0),
    MTK_PIN_DRV_GRP(200, 0xc00, 0, 0),
    MTK_PIN_DRV_GRP(201, 0xc00, 0, 0),
    MTK_PIN_DRV_GRP(202, 0xc00, 0, 0),
    MTK_PIN_DRV_GRP(203, 0xc00, 0, 0),
    MTK_PIN_DRV_GRP(204, 0xc00, 4, 0),
    MTK_PIN_DRV_GRP(205, 0xc00, 4, 0),
    MTK_PIN_DRV_GRP(206, 0xc00, 4, 0),
    MTK_PIN_DRV_GRP(207, 0xc00, 8, 0),
    MTK_PIN_DRV_GRP(208, 0xc00, 8, 0),
    MTK_PIN_DRV_GRP(209, 0xc00, 8, 0),
};

static const struct mtk_pin_drv_grp *mt_find_pin_drv_grp_by_pin(unsigned long pin)
{
    int i;

    if (pin >= MAX_GPIO_PIN)
        return NULL;

    for (i = 0; i < MAX_GPIO_PIN; i++) {
        const struct mtk_pin_drv_grp *pin_drv = ((struct mtk_pin_drv_grp *)mt2712_pin_drv) + i;
        if (pin == pin_drv->pin)
            return pin_drv;
    }
    return NULL;
}

#define GET_VALUE_FROM_DRIVING(val) ((val) % MASK_SPEC_DRIVING)
#define GET_SPECIFY_BIT(val, bit_n) (((val) >> (bit_n)) & 0x1)
#define SET_SPECIFY_BIT(val, bit_n) (val << bit_n)
#define MTK_BIT_RESHAPE(D) (SET_SPECIFY_BIT(GET_SPECIFY_BIT(GET_VALUE_FROM_DRIVING(D), 0), 3) | \
                            SET_SPECIFY_BIT(GET_SPECIFY_BIT(GET_VALUE_FROM_DRIVING(D), 1), 2) | \
                            SET_SPECIFY_BIT(GET_SPECIFY_BIT(GET_VALUE_FROM_DRIVING(D), 2), 0) | \
                            SET_SPECIFY_BIT(GET_SPECIFY_BIT(GET_VALUE_FROM_DRIVING(D), 3), 1))

/* for spec gpio 57-62 */
static signed int gpio_set_driving_spec(u32 pin, unsigned char driving)
{
    unsigned int val;
    unsigned short offset = 0xB50; /* 0x10005B50[7] & [10:8] */
    unsigned int mask = 0x780; /* bit[10:7] all 1 */

    /* 10005B50[10:7]--> E8/E4/E8NRF1/E8NRF2 */
    if (pin >= 57 && pin <= 62) {
        if (driving > MTK_DRIVING_SET_F2F1E8E4_1111) {
            return -ERINVAL;
        }
        val = (MTK_BIT_RESHAPE(driving) & 0xf) << 7;
        GPIO_WR32(GPIO_BASE+offset,((GPIO_RD32(GPIO_BASE+offset))&(~mask)));
        GPIO_WR32(GPIO_BASE+offset,((GPIO_RD32(GPIO_BASE+offset))|(val)));
        return RSUCCESS;
    }
    return -ERINVAL;
}

signed int gpio_cust_set_driving(u32 pin, unsigned char driving)
{
    unsigned int val;
    unsigned int bits, mask, shift;
    const struct mtk_drv_group_desc *drv_grp;
    const struct mtk_pin_drv_grp *pin_drv;

    if (pin >= MAX_GPIO_PIN)
        return -ERINVAL;

    /* for spec gpio 57-62 */
    val = gpio_set_driving_spec(pin, driving);
    if (!val)
        return RSUCCESS;

    pin_drv = mt_find_pin_drv_grp_by_pin(pin);
    if (!pin_drv || pin_drv->grp >= DRV_GROUP_MAX)
        return -ERINVAL;

    drv_grp = ((struct mtk_drv_group_desc *)mt2712_drv_grp) + pin_drv->grp;

    if (driving >= drv_grp->min_drv && driving <= drv_grp->max_drv
            && !(driving % drv_grp->step)) {
        val = driving / drv_grp->step - 1;
        bits = drv_grp->high_bit - drv_grp->low_bit + 1;
        mask = (1UL << (bits)) -1;
        shift = pin_drv->bit + drv_grp->low_bit;;
        mask <<= shift;
        val <<= shift;
        GPIO_WR32(GPIO_BASE+pin_drv->offset,((GPIO_RD32(GPIO_BASE+pin_drv->offset))&(~mask)));
        GPIO_WR32(GPIO_BASE+pin_drv->offset,((GPIO_RD32(GPIO_BASE+pin_drv->offset))|(val)));
        return RSUCCESS;
    }
    return -ERINVAL;
}

#define MTK_LK_GPIO_TO_EINT(_gpio, _eint)   \
    {   \
        .gpionum = _gpio,   \
        .eintnum = _eint,   \
    }

struct mtk_lk_gpio_to_eint mt2712_lk_gpio_to_eint[] = {
    MTK_LK_GPIO_TO_EINT(0, 6),
    MTK_LK_GPIO_TO_EINT(1, 7),
    MTK_LK_GPIO_TO_EINT(2, 8),
    MTK_LK_GPIO_TO_EINT(3, 9),
    MTK_LK_GPIO_TO_EINT(4, 10),
    MTK_LK_GPIO_TO_EINT(5, 11),
    MTK_LK_GPIO_TO_EINT(6, 12),
    MTK_LK_GPIO_TO_EINT(7, 13),
    MTK_LK_GPIO_TO_EINT(8, 14),
    MTK_LK_GPIO_TO_EINT(9, 15),
    MTK_LK_GPIO_TO_EINT(10, 16),
    MTK_LK_GPIO_TO_EINT(11, 17),
    MTK_LK_GPIO_TO_EINT(12, 42),
    MTK_LK_GPIO_TO_EINT(13, 43),
    MTK_LK_GPIO_TO_EINT(14, 44),
    MTK_LK_GPIO_TO_EINT(15, 45),
    MTK_LK_GPIO_TO_EINT(16, 46),
    MTK_LK_GPIO_TO_EINT(17, 47),
    MTK_LK_GPIO_TO_EINT(18, 18),
    MTK_LK_GPIO_TO_EINT(19, 19),
    MTK_LK_GPIO_TO_EINT(20, 48),
    MTK_LK_GPIO_TO_EINT(21, 49),
    MTK_LK_GPIO_TO_EINT(22, 50),
    MTK_LK_GPIO_TO_EINT(23, 51),
    MTK_LK_GPIO_TO_EINT(24, 52),
    MTK_LK_GPIO_TO_EINT(25, 53),
    MTK_LK_GPIO_TO_EINT(26, 54),
    MTK_LK_GPIO_TO_EINT(27, 55),
    MTK_LK_GPIO_TO_EINT(28, 56),
    MTK_LK_GPIO_TO_EINT(29, 57),
    MTK_LK_GPIO_TO_EINT(30, 58),
    MTK_LK_GPIO_TO_EINT(31, 59),
    MTK_LK_GPIO_TO_EINT(32, 60),
    MTK_LK_GPIO_TO_EINT(33, 61),
    MTK_LK_GPIO_TO_EINT(34, 62),
    MTK_LK_GPIO_TO_EINT(35, 63),
    MTK_LK_GPIO_TO_EINT(36, 64),
    MTK_LK_GPIO_TO_EINT(37, -1),
    MTK_LK_GPIO_TO_EINT(38, -1),
    MTK_LK_GPIO_TO_EINT(39, -1),
    MTK_LK_GPIO_TO_EINT(40, -1),
    MTK_LK_GPIO_TO_EINT(41, -1),
    MTK_LK_GPIO_TO_EINT(42, -1),
    MTK_LK_GPIO_TO_EINT(43, -1),
    MTK_LK_GPIO_TO_EINT(44, -1),
    MTK_LK_GPIO_TO_EINT(45, -1),
    MTK_LK_GPIO_TO_EINT(46, -1),
    MTK_LK_GPIO_TO_EINT(47, -1),
    MTK_LK_GPIO_TO_EINT(48, 142),
    MTK_LK_GPIO_TO_EINT(49, 65),
    MTK_LK_GPIO_TO_EINT(50, 66),
    MTK_LK_GPIO_TO_EINT(51, 67),
    MTK_LK_GPIO_TO_EINT(52, 68),
    MTK_LK_GPIO_TO_EINT(53, 69),
    MTK_LK_GPIO_TO_EINT(54, 20),
    MTK_LK_GPIO_TO_EINT(55, 70),
    MTK_LK_GPIO_TO_EINT(56, 71),
    MTK_LK_GPIO_TO_EINT(57, 72),
    MTK_LK_GPIO_TO_EINT(58, 73),
    MTK_LK_GPIO_TO_EINT(59, 74),
    MTK_LK_GPIO_TO_EINT(60, 75),
    MTK_LK_GPIO_TO_EINT(61, 76),
    MTK_LK_GPIO_TO_EINT(62, 77),
    MTK_LK_GPIO_TO_EINT(63, 78),
    MTK_LK_GPIO_TO_EINT(64, 79),
    MTK_LK_GPIO_TO_EINT(65, 80),
    MTK_LK_GPIO_TO_EINT(66, 81),
    MTK_LK_GPIO_TO_EINT(67, 82),
    MTK_LK_GPIO_TO_EINT(68, 83),
    MTK_LK_GPIO_TO_EINT(69, 84),
    MTK_LK_GPIO_TO_EINT(70, 85),
    MTK_LK_GPIO_TO_EINT(71, 86),
    MTK_LK_GPIO_TO_EINT(72, 87),
    MTK_LK_GPIO_TO_EINT(73, 88),
    MTK_LK_GPIO_TO_EINT(74, 89),
    MTK_LK_GPIO_TO_EINT(75, 90),
    MTK_LK_GPIO_TO_EINT(76, 91),
    MTK_LK_GPIO_TO_EINT(77, 92),
    MTK_LK_GPIO_TO_EINT(78, 93),
    MTK_LK_GPIO_TO_EINT(79, 94),
    MTK_LK_GPIO_TO_EINT(80, 95),
    MTK_LK_GPIO_TO_EINT(81, 96),
    MTK_LK_GPIO_TO_EINT(82, 97),
    MTK_LK_GPIO_TO_EINT(83, 98),
    MTK_LK_GPIO_TO_EINT(84, 99),
    MTK_LK_GPIO_TO_EINT(85, 100),
    MTK_LK_GPIO_TO_EINT(86, 101),
    MTK_LK_GPIO_TO_EINT(87, 102),
    MTK_LK_GPIO_TO_EINT(88, 21),
    MTK_LK_GPIO_TO_EINT(89, 103),
    MTK_LK_GPIO_TO_EINT(90, 104),
    MTK_LK_GPIO_TO_EINT(91, 105),
    MTK_LK_GPIO_TO_EINT(92, 106),
    MTK_LK_GPIO_TO_EINT(93, 107),
    MTK_LK_GPIO_TO_EINT(94, 108),
    MTK_LK_GPIO_TO_EINT(95, 109),
    MTK_LK_GPIO_TO_EINT(96, 110),
    MTK_LK_GPIO_TO_EINT(97, 111),
    MTK_LK_GPIO_TO_EINT(98, 112),
    MTK_LK_GPIO_TO_EINT(99, 113),
    MTK_LK_GPIO_TO_EINT(100, 114),
    MTK_LK_GPIO_TO_EINT(101, 30),
    MTK_LK_GPIO_TO_EINT(102, 31),
    MTK_LK_GPIO_TO_EINT(103, 32),
    MTK_LK_GPIO_TO_EINT(104, 33),
    MTK_LK_GPIO_TO_EINT(105, 115),
    MTK_LK_GPIO_TO_EINT(106, 116),
    MTK_LK_GPIO_TO_EINT(107, 117),
    MTK_LK_GPIO_TO_EINT(108, 118),
    MTK_LK_GPIO_TO_EINT(109, 119),
    MTK_LK_GPIO_TO_EINT(110, 34),
    MTK_LK_GPIO_TO_EINT(111, 120),
    MTK_LK_GPIO_TO_EINT(112, 121),
    MTK_LK_GPIO_TO_EINT(113, 122),
    MTK_LK_GPIO_TO_EINT(114, 123),
    MTK_LK_GPIO_TO_EINT(115, 124),
    MTK_LK_GPIO_TO_EINT(116, 35),
    MTK_LK_GPIO_TO_EINT(117, 125),
    MTK_LK_GPIO_TO_EINT(118, 126),
    MTK_LK_GPIO_TO_EINT(119, 127),
    MTK_LK_GPIO_TO_EINT(120, 128),
    MTK_LK_GPIO_TO_EINT(121, 129),
    MTK_LK_GPIO_TO_EINT(122, 130),
    MTK_LK_GPIO_TO_EINT(123, 131),
    MTK_LK_GPIO_TO_EINT(124, 132),
    MTK_LK_GPIO_TO_EINT(125, 133),
    MTK_LK_GPIO_TO_EINT(126, 134),
    MTK_LK_GPIO_TO_EINT(127, 135),
    MTK_LK_GPIO_TO_EINT(128, 136),
    MTK_LK_GPIO_TO_EINT(129, 137),
    MTK_LK_GPIO_TO_EINT(130, 138),
    MTK_LK_GPIO_TO_EINT(131, 139),
    MTK_LK_GPIO_TO_EINT(132, 143),
    MTK_LK_GPIO_TO_EINT(133, 144),
    MTK_LK_GPIO_TO_EINT(134, 145),
    MTK_LK_GPIO_TO_EINT(135, 146),
    MTK_LK_GPIO_TO_EINT(136, 36),
    MTK_LK_GPIO_TO_EINT(137, 147),
    MTK_LK_GPIO_TO_EINT(138, 37),
    MTK_LK_GPIO_TO_EINT(139, 38),
    MTK_LK_GPIO_TO_EINT(140, 39),
    MTK_LK_GPIO_TO_EINT(141, 40),
    MTK_LK_GPIO_TO_EINT(142, 41),
    MTK_LK_GPIO_TO_EINT(143, 148),
    MTK_LK_GPIO_TO_EINT(144, 149),
    MTK_LK_GPIO_TO_EINT(145, 150),
    MTK_LK_GPIO_TO_EINT(146, 151),
    MTK_LK_GPIO_TO_EINT(147, 152),
    MTK_LK_GPIO_TO_EINT(148, 153),
    MTK_LK_GPIO_TO_EINT(149, 154),
    MTK_LK_GPIO_TO_EINT(150, 155),
    MTK_LK_GPIO_TO_EINT(151, 156),
    MTK_LK_GPIO_TO_EINT(152, 157),
    MTK_LK_GPIO_TO_EINT(153, 158),
    MTK_LK_GPIO_TO_EINT(154, 159),
    MTK_LK_GPIO_TO_EINT(155, 160),
    MTK_LK_GPIO_TO_EINT(156, 161),
    MTK_LK_GPIO_TO_EINT(157, 162),
    MTK_LK_GPIO_TO_EINT(158, 163),
    MTK_LK_GPIO_TO_EINT(159, 164),
    MTK_LK_GPIO_TO_EINT(160, 165),
    MTK_LK_GPIO_TO_EINT(161, 166),
    MTK_LK_GPIO_TO_EINT(162, 167),
    MTK_LK_GPIO_TO_EINT(163, 168),
    MTK_LK_GPIO_TO_EINT(164, 169),
    MTK_LK_GPIO_TO_EINT(165, 170),
    MTK_LK_GPIO_TO_EINT(166, 171),
    MTK_LK_GPIO_TO_EINT(167, 172),
    MTK_LK_GPIO_TO_EINT(168, 173),
    MTK_LK_GPIO_TO_EINT(169, 174),
    MTK_LK_GPIO_TO_EINT(170, 175),
    MTK_LK_GPIO_TO_EINT(171, 176),
    MTK_LK_GPIO_TO_EINT(172, 177),
    MTK_LK_GPIO_TO_EINT(173, 178),
    MTK_LK_GPIO_TO_EINT(174, 179),
    MTK_LK_GPIO_TO_EINT(175, 180),
    MTK_LK_GPIO_TO_EINT(176, 181),
    MTK_LK_GPIO_TO_EINT(177, 182),
    MTK_LK_GPIO_TO_EINT(178, 183),
    MTK_LK_GPIO_TO_EINT(179, 184),
    MTK_LK_GPIO_TO_EINT(180, 185),
    MTK_LK_GPIO_TO_EINT(181, 186),
    MTK_LK_GPIO_TO_EINT(182, 187),
    MTK_LK_GPIO_TO_EINT(183, 188),
    MTK_LK_GPIO_TO_EINT(184, 189),
    MTK_LK_GPIO_TO_EINT(185, 190),
    MTK_LK_GPIO_TO_EINT(186, 191),
    MTK_LK_GPIO_TO_EINT(187, 192),
    MTK_LK_GPIO_TO_EINT(188, 193),
    MTK_LK_GPIO_TO_EINT(189, 194),
    MTK_LK_GPIO_TO_EINT(190, 195),
    MTK_LK_GPIO_TO_EINT(191, 196),
    MTK_LK_GPIO_TO_EINT(192, 197),
    MTK_LK_GPIO_TO_EINT(193, 198),
    MTK_LK_GPIO_TO_EINT(194, 199),
    MTK_LK_GPIO_TO_EINT(195, 200),
    MTK_LK_GPIO_TO_EINT(196, 201),
    MTK_LK_GPIO_TO_EINT(197, 202),
    MTK_LK_GPIO_TO_EINT(198, 203),
    MTK_LK_GPIO_TO_EINT(199, 204),
    MTK_LK_GPIO_TO_EINT(200, 205),
    MTK_LK_GPIO_TO_EINT(201, 206),
    MTK_LK_GPIO_TO_EINT(202, 207),
    MTK_LK_GPIO_TO_EINT(203, 208),
    MTK_LK_GPIO_TO_EINT(204, 209),
    MTK_LK_GPIO_TO_EINT(205, 210),
    MTK_LK_GPIO_TO_EINT(206, 211),
    MTK_LK_GPIO_TO_EINT(207, 212),
    MTK_LK_GPIO_TO_EINT(208, 213),
    MTK_LK_GPIO_TO_EINT(209, 214),
    MTK_LK_GPIO_TO_EINT(210, 215),
    MTK_LK_GPIO_TO_EINT(211, 216),
    MTK_LK_GPIO_TO_EINT(212, 217),
    MTK_LK_GPIO_TO_EINT(213, 218),
    MTK_LK_GPIO_TO_EINT(214, 219),
    MTK_LK_GPIO_TO_EINT(215, 220),
    MTK_LK_GPIO_TO_EINT(216, 221),
    MTK_LK_GPIO_TO_EINT(217, 222),
    MTK_LK_GPIO_TO_EINT(218, 223),
    MTK_LK_GPIO_TO_EINT(219, 224),
    MTK_LK_GPIO_TO_EINT(220, 225),
    MTK_LK_GPIO_TO_EINT(221, 226),
    MTK_LK_GPIO_TO_EINT(222, 227),
    MTK_LK_GPIO_TO_EINT(223, 228)
};

signed int gpio_cust_get_irq(unsigned int gpio)
{
    if (gpio >= MAX_GPIO_PIN)
        return -ERINVAL;

    return mt2712_lk_gpio_to_eint[gpio].eintnum;
}

/*---------------------------------------------------------------------------*/
void mt_gpio_pin_decrypt(u32 *cipher)
{
    *cipher &= ~(0x80000000);
    return;
}

/*set GPIO function in fact*/
/*---------------------------------------------------------------------------*/

signed int gpio_cust_set_dir(u32 pin, u32 dir)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_set_dir_chip(pin,dir);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_dir(u32 pin)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_get_dir_chip(pin);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_pull_enable(u32 pin, u32 enable, u32 r1r0)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_set_pull_enable_chip(pin, enable, r1r0);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_pull_enable(u32 pin)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_get_pull_enable_chip(pin);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_pull_select(u32 pin, u32 select)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_set_pull_select_chip(pin,select);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_pull_select(u32 pin)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_get_pull_select_chip(pin);
}
/*---------------------------------------------------------------------------*/
signed int mt_set_gpio_inversion(u32 pin, u32 enable)
{
    return -ERACCESS;
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_inversion(u32 pin)
{
    return -ERACCESS;
}
/*---------------------------------------------------------------------------*/

signed int gpio_cust_set_out(u32 pin, u32 output)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_set_out_chip(pin,output);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_out(u32 pin)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_get_out_chip(pin);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_in(u32 pin)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_get_in_chip(pin);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_set_mode(u32 pin, u32 mode)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_set_mode_chip(pin,mode);
}
/*---------------------------------------------------------------------------*/
signed int gpio_cust_get_mode(u32 pin)
{
    mt_gpio_pin_decrypt(&pin);

    return gpio_cust_get_mode_chip(pin);
}

void gpio_cust_init(void)
{
    gpio_dat.reg = (GPIO_REGS *)GPIO_BASE;
    gpio_set_rdsel(GPIO, (addr_t)GPIO_RDSEL11_EN, 0);
    gpio_set_rdsel(GPIO, (addr_t)GPIO_RDSEL11_EN, 6);
    gpio_set_rdsel(GPIO, (addr_t)GPIO_EXMD_CTRL0, 4);
    gpio_set_rdsel(GPIO, (addr_t)GPIO_RDSEL10_EN, 0);
    gpio_set_rdsel(GPIO, (addr_t)GPIO_RDSEL10_EN, 6);
    gpio_set_rdsel(GPIO, (addr_t)GPIO_RDSEL2_EN, 0);
    gpio_set_rdsel(GPIO, (addr_t)GPIO_RDSEL2_EN, 6);
    gpio_set_rdsel(GBE, (addr_t)GPIO_RDSEL3_EN, 0);
    gpio_set_rdsel(GBE, (addr_t)GPIO_RDSEL3_EN, 6);
    gpio_set_rdsel(GBE, (addr_t)GPIO_RDSEL2_EN, 0);
    gpio_set_rdsel(GBE, (addr_t)GPIO_RDSEL2_EN, 6);
    gpio_set_rdsel(GBE, (addr_t)GPIO_RDSEL4_EN, 0);
    gpio_set_rdsel(NOR, (addr_t)GPIO_RDSEL1_EN, 6);
    gpio_set_rdsel(MSDC1, (addr_t)GPIO_MSDC1_CTRL5, 4);
    gpio_set_rdsel(MSDC2, (addr_t)GPIO_MSDC2_CTRL5, 4);
    gpio_set_rdsel(UART4B, (addr_t)GPIO_RDSEL4_EN, 6);
    gpio_set_rdsel(UART4B, (addr_t)GPIO_RDSEL5_EN, 0);
    gpio_set_rdsel(SPICTP, (addr_t)GPIO_RDSEL6_EN, 0);
    gpio_set_rdsel(SPICTP, (addr_t)GPIO_RDSEL6_EN, 6);
    gpio_set_rdsel(SPIMCU, (addr_t)GPIO_RDSEL7_EN, 0);
    gpio_set_rdsel(SPIMCU, (addr_t)GPIO_RDSEL7_EN, 6);
    gpio_set_rdsel(I2S102, (addr_t)GPIO_RDSEL8_EN, 0);
    gpio_set_rdsel(I2S102, (addr_t)GPIO_RDSEL8_EN, 6);
    gpio_set_rdsel(I2S102, (addr_t)GPIO_RDSEL9_EN, 0);
    gpio_set_rdsel(I2S102, (addr_t)GPIO_RDSEL9_EN, 6);
    gpio_set_rdsel(I2SIO1, (addr_t)GPIO_RDSELA_EN, 0);
    gpio_set_rdsel(I2SIO1, (addr_t)GPIO_RDSELA_EN, 6);
    gpio_set_rdsel(I2SIO1, (addr_t)GPIO_RDSELB_EN, 0);
    gpio_set_rdsel(I2SIO1, (addr_t)GPIO_RDSELB_EN, 6);
    gpio_set_rdsel(I2SO02, (addr_t)GPIO_RDSELC_EN, 0);
    gpio_set_rdsel(I2SO02, (addr_t)GPIO_RDSELC_EN, 6);
    gpio_set_rdsel(I2SO02, (addr_t)GPIO_RDSELD_EN, 0);
    gpio_set_rdsel(I2SO02, (addr_t)GPIO_RDSELD_EN, 6);
    gpio_set_rdsel(I2SO02, (addr_t)GPIO_RDSELE_EN, 0);
    gpio_set_rdsel(I2SO02, (addr_t)GPIO_RDSELE_EN, 6);
    gpio_set_rdsel(TDMO, (addr_t)GPIO_RDSELF_EN, 0);
    gpio_set_rdsel(TDMO, (addr_t)GPIO_RDSELF_EN, 6);
}
