/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <err.h>
#include <mtk_spi.h>
#include <unittest.h>
#include <lib/mempool.h>
#include <mtk_wrapper.h>

#define DEBUG_DATA_SIZE 60

unsigned char *test_data(int data_size, bool is_init)
{
    unsigned char *data;

    if (data_size < 1)
        dprintf(INFO, "exception---size range is < 1.");

    data = (unsigned char *)mempool_alloc(data_size, MEMPOOL_UNCACHE);

    if (!data) {
        dprintf(INFO, "mempool_alloc is fail.\r\n");
        goto error;
    }

    if (is_init)
        for (int i = 0; i < data_size ; i++)
            data[i] = (unsigned char)i;

    return data;

error:
    mempool_free(data);
    return NULL;

}

int spi_loopback_check(unsigned char *rx, unsigned char *tx, int size)
{
    int i, err = 0;

    for (i = 0; i <size; i++) {
        if (tx[i] != rx[i]) {
            dprintf(SPEW, "tx[%d]:0x%x, rx[%d]:0x%x\r\n", i, tx[i], i, rx[i]);
            err++;
        }
    }

    if (err) {
        dprintf(SPEW, "spim loopback test fail\r\n\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n");
        return -1;
    } else {
        dprintf(SPEW, "spim loopback test pass\r\n\n-------------------------------------------------\n\n");
        return 0;
    }
}

int spi_dma_loopback(int bus_num, int size)
{
    int ret;
    unsigned char *rx, * tx;

    tx = test_data(size,true);
    rx = test_data(size,false);

    if (!tx || !rx)
        return -1;

    for (int i = 0; i < size ; i++)
        tx[i] = (unsigned char)i;

    ret = spi_cust_rx_tx(bus_num, rx, tx, size);
    if (ret) {
        dprintf(INFO, "mtk_spi_transfer fail, ret(%d)\r\n", ret);
        return ret;
    }

    ret = spi_loopback_check(rx, tx, size);
    mempool_free(tx);
    mempool_free(rx);

    return ret;
}

int spi_loopback(void)
{
    int size = 1, ret = 0, bus_num = 0;

    for (; size <= DEBUG_DATA_SIZE;) {

        dprintf(SPEW, "bus_num is:%d-----size is:%d\r\n\n", bus_num, size);

        ret = spi_dma_loopback(bus_num, size);
        if (ret) {
            dprintf(SPEW, "test fail-----bus_num is:%d-----size is:%d\r\n", bus_num, size);
            return ret;
        }

        if (bus_num >= 2)
            bus_num++;
        else
            bus_num = 2;

        if (bus_num > 5) {
            bus_num = 0;
            size++;
        }
    }

    dprintf(SPEW, "TEST SUCCESS!\r\n");

    return 0;
}

int spi_bus_init(void)
{
    spi_cust_init(0, 1*1000*1000);
    spi_cust_init(2, 1*1000*1000);
    spi_cust_init(3, 1*1000*1000);
    spi_cust_init(4, 1*1000*1000);
    spi_cust_init(5, 1*1000*1000);
    return 0;
}

static bool test_spi_init(void)
{
    BEGIN_TEST;

    EXPECT_EQ(NO_ERROR, spi_bus_init(), "fail to test spi_bus_init");

    END_TEST;
}

static bool test_spi_rx(void)
{
    BEGIN_TEST;

    unsigned char *rx = test_data(DEBUG_DATA_SIZE, false);
    EXPECT_EQ(NO_ERROR, spi_cust_rx(0, rx, DEBUG_DATA_SIZE), "fail to test spi_cust_rx");
    mempool_free(rx);

    END_TEST;
}


static bool test_spi_tx(void)
{
    BEGIN_TEST;

    unsigned char *tx = test_data(DEBUG_DATA_SIZE, true);
    EXPECT_EQ(NO_ERROR, spi_cust_tx(0, tx, DEBUG_DATA_SIZE), "fail to test spi_cust_tx");
    mempool_free(tx);

    END_TEST;
}

static bool test_spi_rx_tx(void)
{
    BEGIN_TEST;

    unsigned char *tx = test_data(DEBUG_DATA_SIZE, true);
    unsigned char *rx = test_data(DEBUG_DATA_SIZE, false);
    EXPECT_EQ(NO_ERROR, spi_cust_rx_tx(0, rx, tx, DEBUG_DATA_SIZE), "fail to test spi_cust_rx_tx");
    mempool_free(tx);
    mempool_free(rx);

    END_TEST;
}

static bool test_spi_loopback(void)
{
    BEGIN_TEST;

    EXPECT_EQ(NO_ERROR, spi_loopback(), "fail to test spi_loopback");

    END_TEST;
}


BEGIN_TEST_CASE(spi_tests);
RUN_TEST(test_spi_init);
RUN_TEST(test_spi_rx);
RUN_TEST(test_spi_tx);
RUN_TEST(test_spi_rx_tx);
RUN_TEST(test_spi_loopback);
END_TEST_CASE(spi_tests);
