/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <unittest.h>
#include <uart_cust.h>
#include <lib/mempool.h>

/**
 * lib/unittest defines several macros for unittest,
 * EXPECT_EQ, EXPECT_NEQ, EXPECT_LE, EXPECT_LT, EXPECT_GE, EXPECT_GT,
 * EXPECT_TRUE, EXPECT_FALSE, EXPECT_BYTES_EQ, EXPECT_BYTES_NE, EXPECT_EQ_LL,
 * ASSERT_NOT_NULL.
 * For detail, please refer to lib/unittest/include/unittest.h
 */

#define UART_TX_BUFF_SIZE     4096
#define UART_RX_BUFF_SIZE     4096
#define UART_DMA_BUFF_SIZE    4096
static bool test_uart_cust_init(void)
{
    uint32_t tx_buff_size, rx_buff_size;
    char *tx_buff, *rx_buff;

    BEGIN_TEST;

    tx_buff_size = UART_TX_BUFF_SIZE;
    rx_buff_size = UART_RX_BUFF_SIZE;

    tx_buff = (char *)mempool_alloc(tx_buff_size, MEMPOOL_UNCACHE);
    rx_buff = (char *)mempool_alloc(rx_buff_size, MEMPOOL_UNCACHE);
    printf("%s with tx_buff:%p, rx_buff:%p\n", __FUNCTION__, tx_buff, rx_buff);
    memset(tx_buff, 0x0 , tx_buff_size);
    memset(rx_buff, 0x0 , rx_buff_size);

    EXPECT_EQ(NO_ERROR, uart_cust_init(UART_PORT4, tx_buff, tx_buff_size, rx_buff, rx_buff_size), "fail to init uart_cust_init");

    END_TEST;
}

static bool test_uart_cust_config(void)
{
    uint32_t dma_len;
    char *dma_buff;

    BEGIN_TEST;

    dma_len = UART_DMA_BUFF_SIZE;
    dma_buff = (char *)mempool_alloc(dma_len, MEMPOOL_UNCACHE);
    printf("%s with dma_buff:%p, dma_len:0x%x\n", __FUNCTION__, dma_buff, dma_len);
    EXPECT_EQ(NO_ERROR, uart_cust_config(UART_PORT4, BAUD_RATE_921600, dma_buff, dma_len), "fail to config uart_cust_config");

    END_TEST;
}

static bool test_uart_cust_tx(void)
{
    int cnt = 20;
    const uint8_t buf[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, \
                         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
                       };

    BEGIN_TEST;

    printf(" %s with cnt:%d\n", __func__, cnt);
    EXPECT_EQ(cnt, uart_cust_tx(UART_PORT4, buf, cnt), "fail to test uart_cust_tx");

    END_TEST;
}

static bool test_uart_cust_rx(void)
{
    BEGIN_TEST;

    int i = 0, cnt = 20;
    uint8_t buf[20] = {0x0};

    EXPECT_EQ(cnt, uart_cust_rx(UART_PORT4, &buf[0], cnt), "fail to test uart_cust_rx");

    for (i = 0; i < cnt; i++)
        printf(" %s with value[%d] 0x%x\n", __func__, i, buf[i]);

    END_TEST;
}

static bool test_uart_tx_large_buf(void)
{
    int i = 0, cnt = 1000, real_cnt = 0;
    uint8_t buf[1000] = {0x0};

    BEGIN_TEST;
    for (i = 0; i < cnt; i++)
        buf[i] = i;

    printf(" %s with cnt:%d\n", __func__, cnt);
    real_cnt = uart_cust_tx(UART_PORT4, buf, cnt);
    printf(" %s with real_cnt:%d\n", __func__, real_cnt);

    END_TEST;
}

static bool test_uart_rx_large_buf(void)
{
    BEGIN_TEST;

    int i = 0, cnt = 1000, real_cnt = 0;
    uint8_t buf[1000] = {0x0};

    real_cnt = uart_cust_rx(UART_PORT4, &buf[0], cnt);
    printf(" %s read cnt is %d\n", __func__, real_cnt);

    for (i = 0; i < real_cnt; i++)
        printf("RX value[%d] 0x%x\n", i, buf[i]);

    END_TEST;
}

static bool test_uart_tx_data_continue_send(void)
{
    unsigned int i = 0, real_cnt = 0, loop_cnt = 100;
    uint8_t buf[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,};

    BEGIN_TEST;

    printf(" %s with cnt:%d\n", __func__, loop_cnt);
    for (i = 0; i < loop_cnt; i++) {
        real_cnt = uart_cust_tx(UART_PORT4, buf, 10);
        printf("Send[%d] cnt is %d\n", i, real_cnt);
    }

    END_TEST;
}

static bool test_uart_rx_data_continue_read(void)
{
    BEGIN_TEST;

    int i = 0, j = 0, cnt = 10, real_cnt = 0, total_cnt = 0, loop_cnt = 100;
    uint8_t buf[10] = {0x0};

    for (i = 0; i < loop_cnt; i++) {
        real_cnt = uart_cust_rx(UART_PORT4, &buf[0], cnt);
        total_cnt += real_cnt;
        printf("Read[%d] cnt is %d, total: %d\n", i, real_cnt, real_cnt);

        for (j = 0; j < real_cnt; j++)
            printf("Loop[%d] rx value[%d] 0x%x\n", i, j, buf[j]);
    }

    END_TEST;
}

BEGIN_TEST_CASE(uart_cust_tests);
RUN_TEST(test_uart_cust_init);
RUN_TEST(test_uart_cust_config);
RUN_TEST(test_uart_cust_tx);
RUN_TEST(test_uart_cust_rx);
RUN_TEST(test_uart_tx_large_buf);
RUN_TEST(test_uart_rx_large_buf);
RUN_TEST(test_uart_tx_data_continue_send);
RUN_TEST(test_uart_rx_data_continue_read);
/* cross test */
RUN_TEST(test_uart_tx_large_buf);
RUN_TEST(test_uart_rx_data_continue_read);
RUN_TEST(test_uart_tx_data_continue_send);
RUN_TEST(test_uart_rx_large_buf);
END_TEST_CASE(uart_cust_tests);
