/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * MediaTek Inc. (C) 2017. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */

#include <kernel/misc.h>
#include <kernel/mutex.h>
#include <kernel/panic.h>
#include <mm/core_memprot.h>
#include <io.h>
#include <initcall.h>
#include "emi_mpu_api.h"
#include "emi_reg.h"
#include "trace.h"

#define MOD "[EMI_MPU_API]"

#define IOMEM(reg) (reg)
#define EMI_MPU_VA_BUFFER	emi_mpu_va_buffer

vaddr_t pericfg_va_buffer;
vaddr_t emi_mpu_va_buffer;

register_phys_mem(MEM_AREA_IO_NSEC, PERICFG_BASE_ADDR, SIZE_4KB);
register_phys_mem(MEM_AREA_IO_NSEC, EMI_MPU_BASE, SIZE_4KB);

inline unsigned int readl(volatile unsigned int addr)
{
    return *((volatile unsigned int *)addr);
}

inline void writel(unsigned int val, volatile unsigned int addr)
{
    *((volatile unsigned int *)addr) = val;
}

TEE_Result emi_mpu_set_protection(struct emi_region_info_t *region_info)
{
    int i;
    unsigned int start;
    unsigned int end;

    if (region_info->region >= EMI_MPU_REGION_NUM) {
        EMSG("[emi_mpu_set_protection] region %d out of range\n",
            region_info->region);
        return TEE_ERROR_BAD_PARAMETERS;
    }

    start = (unsigned int)(region_info->start >> EMI_MPU_ALIGN_BITS);
    end = (unsigned int)(region_info->end >> EMI_MPU_ALIGN_BITS);
    //pericfg_va_buffer = (vaddr_t)phys_to_virt_io(PERICFG_BASE_ADDR);
    //according 4G mode, to caculate EMI address(need map PERIAXI first, then read CTL3)

    if ((readl((unsigned int)pericfg_va_buffer + PERIAXI_CTL3_OFF) & PERISYS_4G_SUPPORT) == 0) {
		if (start>DRAM_OFFSET){
        	start = start - DRAM_OFFSET;
		}else{
			start = 0;
		}
		if (end>DRAM_OFFSET){
        	end = end - DRAM_OFFSET;
		}else{
			end = 0;
		}
        DMSG("%s, MPU 2GB mode start/end:[0x%x 0x%x].\n", MOD, start, end);
    } else {
        start &= EMI_4GB_CLR_BIT32;
        end &= EMI_4GB_CLR_BIT32;
        DMSG("%s, MPU 4GB mode start/end:[0x%x 0x%x].\n", MOD, start, end);
    }
    //EMI_MPU_VA_BUFFER = (vaddr_t)phys_to_virt_io(EMI_MPU_BASE);
    //map mpu register set

    writel(start, EMI_MPU_SA(region_info->region));
    writel(end, EMI_MPU_EA(region_info->region));
    writel(region_info->apc, EMI_MPU_APC(region_info->region, 0));
	//Make sure the setting is take effect before return!
	dsb();
	isb();
    return TEE_SUCCESS;
}

TEE_Result emi_mpu_get_protection_reg(int region, struct emi_region_info_t *region_info)
{
    if ( (region < 0) || (region >= EMI_MPU_REGION_NUM)) {
        EMSG("region %d out of range\n", region);
        return TEE_ERROR_BAD_PARAMETERS;
    }
    if (!region_info){
        EMSG("bad region info struct\n");
        return TEE_ERROR_BAD_PARAMETERS;
    }

    region_info->region = region;
	region_info->apc = readl(EMI_MPU_APC(region, 0));
	region_info->start = readl(EMI_MPU_SA(region));
	region_info->end  = readl(EMI_MPU_EA(region));
    
    return TEE_SUCCESS;
}


TEE_Result emi_mpu_get_protection(int region, struct emi_region_info_t *region_info)
{
	TEE_Result res;
	paddr_t start;
	paddr_t end;
    //EMI_MPU_VA_BUFFER = (vaddr_t)phys_to_virt_io(EMI_MPU_BASE);

    res = emi_mpu_get_protection_reg(region, region_info);
	if(TEE_SUCCESS != res){
		return res;
	}
	start = region_info->start;
	end = region_info->end;

	if ((readl((unsigned int)pericfg_va_buffer + PERIAXI_CTL3_OFF) & PERISYS_4G_SUPPORT) == 0) {
        start += DRAM_OFFSET;
        end  += DRAM_OFFSET;
        DMSG("%s, MPU 2GB mode start/end:[0x%x 0x%x].\n", MOD, start, end);
    } else {
        start |= EMI_4GB_SET_BIT32;
        end |= EMI_4GB_SET_BIT32;
        DMSG("%s, MPU 4GB mode start/end:[0x%x 0x%x].\n", MOD, start, end);
    }
	start <<= EMI_MPU_ALIGN_BITS;
	end <<= EMI_MPU_ALIGN_BITS;
    region_info->start = start;
    region_info->end  = end;
    
    return TEE_SUCCESS;
}

static struct mutex mpu_mu = MUTEX_INITIALIZER;

void dump_region_info(int region)
{
    struct emi_region_info_t region_info;
    emi_mpu_get_protection_reg(region, &region_info);
	DMSG("-------region %d--------", region);
    DMSG("region_info.start:%p", region_info.start);
    DMSG("region_info.end:%p", region_info.end);
    DMSG("region_info.region:%d", region_info.region);
    DMSG("region_info.apc:%x", region_info.apc);
	emi_mpu_get_protection(region, &region_info);
    DMSG("region_info.start address:%p", region_info.start);
    DMSG("region_info.end address:%p", region_info.end);

}
void dump_region_info_all(void)
{
    int i = 0;
    for(i=0; i < EMI_MPU_REGION_NUM; i++){
        dump_region_info(i);
    }
}

TEE_Result check_record_unlocked(int region, bool* is_used)
{
    struct emi_region_info_t region_info;
	TEE_Result res = emi_mpu_get_protection(region, &region_info);
    if(TEE_SUCCESS != res){
        return res;
    }
    // dump_region_info(&region_info);
    if (region_info.apc !=0 ){
        is_used = 1;
    }else{
        is_used = 0;
    }
    return TEE_SUCCESS;
}

TEE_Result emi_mpu_set_unlock(paddr_t start, paddr_t end, int region, tz_mpu_permission per){
    struct emi_region_info_t region_info;
    region_info.start = start;
    region_info.end = end;
    region_info.region = region;
    region_info.apc = SET_ACCESS_PERMISSON(per, EMI_MPU_FORBIDDEN, EMI_MPU_FORBIDDEN, per);
    return emi_mpu_set_protection(&region_info);
}

TEE_Result emi_mpu_set_clear(int region){
    struct emi_region_info_t region_info;
    region_info.start = 0;
    region_info.end = 0;
    region_info.region = region;
    region_info.apc = 0;
    return emi_mpu_set_protection(&region_info);
}

inline bool is_dyn_region(int region){
	return (DYN_MPU_REGION_START <= region) && (region < EMI_MPU_REGION_NUM);
}
TEE_Result emi_mpu_unprotect(paddr_t addr)
{
    int i = 0;
    int select_region = -1;
    struct emi_region_info_t region_info;
    TEE_Result err = TEE_ERROR_GENERIC;
    mutex_lock(&mpu_mu);
    for(i=0; i < DYN_MPU_NUM; i++){
        err = emi_mpu_get_protection(TO_DYN_REGION_NUM(i), &region_info);
		if (err != TEE_SUCCESS){
			break;
		}
		DMSG("MPU protect addr:%p", region_info.start);
        if((region_info.start == addr) && (region_info.apc != 0)){
            select_region = TO_DYN_REGION_NUM(i);
            err = emi_mpu_set_clear(select_region);
            break;
        }
    }
    mutex_unlock(&mpu_mu);
    return err;
}

TEE_Result emi_mpu_protect_unlocked(paddr_t start, size_t len, tz_mpu_permission per)
{
    int i = 0;
    int select_region = -1;
    bool is_used;
    TEE_Result err=TEE_ERROR_GENERIC;
    for(i=0; i < DYN_MPU_NUM; i++){
        if ( TEE_SUCCESS != check_record_unlocked(TO_DYN_REGION_NUM(i), &is_used)){
			break;
		}
        if(!is_used){
            select_region = TO_DYN_REGION_NUM(i);
            break;
        }
    }
    if (is_dyn_region(select_region)){
        err = emi_mpu_set_unlock(start, start+len-1, select_region, per);
    }else{
    	EMSG("Can't find proper region.");
        err = TEE_ERROR_OUT_OF_MEMORY;
    }
    return err;
}

static TEE_Result emi_mpu_record_init(void)
{
    int i = 0;
    bool is_used;
	TEE_Result err=TEE_SUCCESS;
    //mutex_lock(&mpu_mu);
    pericfg_va_buffer = (vaddr_t)phys_to_virt_io(PERICFG_BASE_ADDR);
	EMI_MPU_VA_BUFFER = (vaddr_t)phys_to_virt_io(EMI_MPU_BASE);
    for(i=0; i < DYN_MPU_NUM; i++){
        err = check_record_unlocked(TO_DYN_REGION_NUM(i), &is_used);
        if(is_used){
            panic("The mpu region for dynamic use is used already.");
        }
    }
	IMSG("emi_mpu_record_init finished.");
    //mutex_unlock(&mpu_mu);
    return err;
}

//Check the pa ~ pa+len is in the protected region or not.
//Called from fast smc.
TEE_Result check_mem_protect(paddr_t pa, uint32_t len, uint32_t* is_protected)
{
	struct emi_region_info_t region_info;
	int i = 0;
	uint32_t protected = 0;
	TEE_Result err=TEE_SUCCESS;
	//mutex_lock(&mpu_mu);
	for(i=0; i < DYN_MPU_NUM; i++){
		err = emi_mpu_get_protection(TO_DYN_REGION_NUM(i), &region_info);
        if(TEE_SUCCESS != err){
			break;
		}
        if(region_info.apc != 0){
			if((region_info.start<=pa) && (pa<region_info.end)){
				protected = 1;
				break;
			}
			if(region_info.start<=(pa+len) || (pa+len)<region_info.end){
				protected = 1;
				break;
			}
			if((pa<region_info.start) && (region_info.end<=(pa+len))){
				protected = 1;
				break;
			}
		}
    }
	//mutex_unlock(&mpu_mu);
	*is_protected=protected;
	return err;
}


inline bool check_address_64k_aligin(paddr_t pa)
{
	return (pa & ((1<<EMI_MPU_ALIGN_BITS)-1));
}

TEE_Result syscall_emi_mpu_protect_unlocked(paddr_t pa, size_t len, uint32_t per)
{
	TEE_Result res;
	DMSG("address:%p, len: %zu, permission:%d", pa, len, per);
	if(check_address_64k_aligin(pa)) {
		EMSG("The protect memory must 128K alignment, but the input is:%p",pa);
		return TEE_ERROR_BAD_PARAMETERS;
	}
	res = emi_mpu_protect_unlocked(pa, len, per);
    DMSG("res:%x",res);
    // dump_region_info_all();
	return res;
}

TEE_Result syscall_emi_mpu_protect(paddr_t pa, size_t len, uint32_t per)
{
	TEE_Result res;
	mutex_lock(&mpu_mu);
	res = syscall_emi_mpu_protect_unlocked(pa,len,per);
    mutex_unlock(&mpu_mu);
	return res;
}

TEE_Result syscall_emi_mpu_unprotect(paddr_t pa)
{
	TEE_Result res;

	DMSG("pa:%p",pa);
	if(check_address_64k_aligin(pa)) {
		EMSG("The protect memory must 128K alignment, but the input is:%p",pa);
		return TEE_ERROR_BAD_PARAMETERS;
	}
	res = emi_mpu_unprotect(pa);
    DMSG("res:%x",res);
    // dump_region_info_all();
	return res;
}

driver_init(emi_mpu_record_init);

