// SPDX-License-Identifier: BSD-2-Clause
/*
 * Copyright (c) 2015, Linaro Limited
 */

#include <console.h>
#include <drivers/gic.h>
#include <drivers/serial8250_uart.h>
#include <kernel/generic_boot.h>
#include <kernel/panic.h>
#include <kernel/pm_stubs.h>
#include <mm/core_memprot.h>
#include <platform_config.h>
#include <stdint.h>
#include <tee/entry_std.h>
#include <tee/entry_fast.h>
#include <io.h>

static void main_fiq(void);

register_phys_mem(MEM_AREA_IO_NSEC,
		  MT2735_UART0_BASE, SERIAL8250_UART_REG_SIZE);
register_phys_mem(MEM_AREA_IO_SEC, BASE_GICC_BASE, GICC_SIZE);
register_phys_mem(MEM_AREA_IO_SEC, BASE_GICD_BASE, GICD_SIZE);
register_phys_mem(MEM_AREA_IO_SEC, MCUCFG_BASE, SIZE_MCUCFG);

static const struct thread_handlers handlers = {
	.std_smc = tee_entry_std,
	.fast_smc = tee_entry_fast,
	.nintr = main_fiq,
	.cpu_on = cpu_on_handler,
	.cpu_off = pm_do_nothing,
	.cpu_suspend = pm_do_nothing,
	.cpu_resume = pm_do_nothing,
	.system_off = pm_do_nothing,
	.system_reset = pm_do_nothing,
};

static struct gic_data gic_data;

void main_init_gic(void)
{
	vaddr_t gicc_base;
	vaddr_t gicd_base;

	gicc_base = (vaddr_t)phys_to_virt_io(BASE_GICC_BASE);
	gicd_base = (vaddr_t)phys_to_virt_io(BASE_GICD_BASE);

	if (!gicc_base || !gicd_base)
		panic();

	gic_init_base_addr(&gic_data, gicc_base, gicd_base);
	itr_init(&gic_data.chip);
}

void main_secondary_init_gic(void)
{
	gic_cpu_init(&gic_data);
}

const struct thread_handlers *generic_boot_get_handlers(void)
{
	return &handlers;
}

static void main_fiq(void)
{
	gic_it_handle(&gic_data);
}

static struct serial8250_uart_data console_data;

void console_init(void)
{
	serial8250_uart_init(&console_data, CONSOLE_UART_BASE,
			     CONSOLE_UART_CLK_IN_HZ, CONSOLE_BAUDRATE);
	register_serial_console(&console_data.chip);
}


//!!!!!!!!!!!!!!!!!!!!!!!!!!!Attention!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// This is for mtk mt2712 platform specificlly.
// If the irq is low level trigger, need to invert the trigger polarity.
// If not, the interrupt wil be triggered incorrectly.
//For example:
// itr_add(&gpt_itr); //Register interrupt handler.
// Set the polarity before enable the interrupt.
// itr_set_polarity(IT_GPT_NUMBER, (gpt_itr.flags & ITRF_TRIGGER_LEVEL)==ITRF_TRIGGER_LEVEL);
// Enable the interrupt.
// itr_enable(IT_GPT_NUMBER);
//!!!!!!!!!!!!!!!!!!!!!!!!!!!Attention!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
bool itr_set_polarity(unsigned int irq, unsigned int polarity)
{
    unsigned int offset;
    unsigned int reg_index;
    unsigned int value;
	// DMSG("irq:%d polarity:%d!", irq, polarity);
    /* spi only */
    if (irq < GIC_PRIVATE_SIGNALS)
        return false;
	vaddr_t mcucfg_base_va = (vaddr_t)phys_to_virt_io(MCUCFG_BASE);

    if (!mcucfg_base_va){
		EMSG("Map mcucfg registers failed!");
		panic();
	}
    offset = (irq - GIC_PRIVATE_SIGNALS) & 0x1F;
    reg_index = (irq - GIC_PRIVATE_SIGNALS) >> 5;
	vaddr_t reg_addr = (vaddr_t)(mcucfg_base_va + INT_POL_CTL0 + (reg_index * 4));
    if (polarity == 1) {
        value = read32(reg_addr);
        value |= (1 << offset); /* always invert the incoming IRQ's polarity */
        write32(value, reg_addr);
    } else {
        value = read32(reg_addr);
        value &= ~(0x1 << offset);
        write32(value, reg_addr);
    }
    return true;
}

TEE_Result syscall_emi_mpu_protect(paddr_t pa, size_t len, uint32_t per)
{
	return TEE_ERROR_NOT_IMPLEMENTED;
}

TEE_Result syscall_emi_mpu_unprotect(paddr_t pa)
{
	return TEE_ERROR_NOT_IMPLEMENTED;
}
