// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2018, Linaro Limited
 */

#include <sks_ck_debug.h>
#include <inttypes.h>
#include <pkcs11.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <util.h>

#include "xtest_test.h"
#include "xtest_helpers.h"

/*
 * Some PKCS#11 object resources used in the tests
 */
static const CK_BYTE cktest_aes128_key[16];

static const CK_BYTE cktest_aes128_iv[16];

static const CK_AES_CTR_PARAMS cktest_aes_ctr_params = {
	.ulCounterBits = 1,
};

static CK_MECHANISM cktest_aes_ecb_mechanism = {
	CKM_AES_ECB,
	NULL, 0,
};
static CK_MECHANISM cktest_aes_cbc_mechanism = {
	CKM_AES_CBC,
	(CK_BYTE_PTR)cktest_aes128_iv, sizeof(cktest_aes128_iv),
};
static CK_MECHANISM cktest_aes_ctr_mechanism = {
	CKM_AES_CTR,
	(CK_BYTE_PTR)&cktest_aes_ctr_params, sizeof(cktest_aes_ctr_params),
};
static CK_MECHANISM cktest_aes_cts_mechanism = {
	CKM_AES_CTS,
	(CK_BYTE_PTR)cktest_aes128_iv, sizeof(cktest_aes128_iv),
};

/*
 * Util to find a slot on which to open a session
 */
static CK_RV close_lib(void)
{
	return C_Finalize(0);
}

static CK_RV init_lib_and_find_token_slot(CK_SLOT_ID *slot)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID_PTR slots = NULL;
	CK_ULONG count = 0;

	rv = C_Initialize(0);
	if (rv)
		return rv;

	rv = C_GetSlotList(CK_TRUE, NULL, &count);
	if (rv != CKR_OK)
		goto bail;

	if (count < 1) {
		rv = CKR_GENERAL_ERROR;
		goto bail;
	}

	slots = malloc(count * sizeof(CK_SLOT_ID));
	if (!slots) {
		rv = CKR_HOST_MEMORY;
		goto bail;
	}

	rv = C_GetSlotList(CK_TRUE, slots, &count);
	if (rv)
		goto bail;

	/* Use the last slot */
	*slot = slots[count - 1];

bail:
	free(slots);
	if (rv)
		close_lib();

	return rv;
}

static CK_RV find_by_attr(CK_SESSION_HANDLE hSession, CK_ATTRIBUTE *template, CK_ULONG attr_count, CK_ULONG *count,
                          CK_OBJECT_HANDLE_PTR *hObject) {
	CK_RV rv;

	if (NULL == hObject || NULL == template || NULL == count) {
		return CKR_ARGUMENTS_BAD;
	}

	rv = C_FindObjectsInit(hSession, template, attr_count);
	if (rv != CKR_OK) {
		fprintf(stderr, "Can't initialize search\n");
		return rv;
	}

	CK_ULONG max_objects = 25;
	bool searching = 1;
	*count = 0;
	while (searching) {
		CK_ULONG found = 0;
		*hObject = realloc(*hObject, (*count + max_objects) * sizeof(CK_OBJECT_HANDLE));
		if (NULL == *hObject) {
			fprintf(stderr, "Could not allocate memory for objects\n");
			return CKR_HOST_MEMORY;
		}

		CK_OBJECT_HANDLE_PTR loc = *hObject;
		rv = C_FindObjects(hSession, &loc[*count], max_objects, &found);
		if (rv != CKR_OK) {
			fprintf(stderr, "Can't run search\n");
			C_FindObjectsFinal(hSession);
			return rv;
		}

		(*count) += found;

		if (0 == found)
			searching = 0;
	}

	rv = C_FindObjectsFinal(hSession);
	if (rv != CKR_OK) {
		fprintf(stderr, "Can't finalize search\n");
		return rv;
	}

	if (0 == *count) {
		fprintf(stderr, "Didn't find requested key\n");
		return rv;
	}

	return CKR_OK;
}

static void xtest_pkcs11_test_1000(ADBG_Case_t *c)
{
	CK_RV rv;

	rv = C_Initialize(NULL);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_Finalize(NULL);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_Initialize(NULL);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_Initialize(NULL);
	ADBG_EXPECT_CK_RESULT(c, CKR_CRYPTOKI_ALREADY_INITIALIZED, rv);

	rv = C_Finalize(NULL);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = C_Finalize(NULL);
	ADBG_EXPECT_CK_RESULT(c, CKR_CRYPTOKI_NOT_INITIALIZED, rv);
}

ADBG_CASE_DEFINE(pkcs11, 1000, xtest_pkcs11_test_1000,
		 "Initialize and close Cryptoki library");

static void xtest_pkcs11_test_1001(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID_PTR slot_ids = NULL;
	CK_ULONG slot_count = 0;
	CK_ULONG present_slot_count = 0;
	CK_INFO lib_info = { };
	CK_SLOT_INFO slot_info = { };
	CK_TOKEN_INFO token_info = { };
	CK_FUNCTION_LIST_PTR ckfunc_list = NULL;
	size_t i = 0;
	CK_SLOT_ID max_slot_id = 0;
	CK_MECHANISM_TYPE_PTR mecha_types = NULL;
	CK_ULONG mecha_count = 0;
	CK_MECHANISM_INFO mecha_info = { };

	rv = C_Initialize(NULL);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	Do_ADBG_BeginSubCase(c, "Test C_GetFunctionList()");

	rv = C_GetFunctionList(&ckfunc_list);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	if (!ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetInfo) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetSlotList) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetSlotInfo) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetTokenInfo) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetMechanismList) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetMechanismInfo))
		goto out;

	Do_ADBG_EndSubCase(c, "Test C_GetFunctionList()");
	Do_ADBG_BeginSubCase(c, "Test C_GetInfo()");

	rv = C_GetInfo(&lib_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_EndSubCase(c, "Test C_GetInfo()");
	Do_ADBG_BeginSubCase(c, "Test C_GetSlotList()");

	rv = C_GetSlotList(0, NULL, &slot_count);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	if (!ADBG_EXPECT_COMPARE_UNSIGNED(c, slot_count, !=, 0))
		goto out;

	if (slot_count > 1) {
		/* Ensure case non-NULL-buffer and zero-count is tested */
		CK_SLOT_ID id = 0;

		slot_count = 0;
		rv = C_GetSlotList(0, &id, &slot_count);
		if (!ADBG_EXPECT_CK_RESULT(c, CKR_BUFFER_TOO_SMALL, rv))
			goto out;
	}

	rv = C_GetSlotList(1, NULL, &present_slot_count);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	if (!ADBG_EXPECT_COMPARE_UNSIGNED(c, slot_count, ==,
					  present_slot_count))
		goto out;

	slot_ids = calloc(slot_count, sizeof(CK_SLOT_ID));
	if (!ADBG_EXPECT_NOT_NULL(c, slot_ids))
		goto out;

	slot_count--;
	rv = C_GetSlotList(1, slot_ids, &slot_count);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_BUFFER_TOO_SMALL, rv))
		goto out;

	rv = C_GetSlotList(1, slot_ids, &slot_count);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_EndSubCase(c, "Test C_GetSlotList()");
	Do_ADBG_BeginSubCase(c, "Test C_Get{Slot|Token}Info()");

	for (i = 0; i < slot_count; i++) {
		CK_SLOT_ID slot = slot_ids[i];

		rv = C_GetSlotInfo(slot, &slot_info);
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out;

		rv = C_GetTokenInfo(slot, &token_info);
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out;

		if (max_slot_id < slot)
			max_slot_id = slot;
	}

	Do_ADBG_EndSubCase(c, "Test C_Get{Slot|Token}Info()");
	Do_ADBG_BeginSubCase(c, "Test C_GetMechanism{List|Info}()");

	for (i = 0; i < slot_count; i++) {
		CK_SLOT_ID slot = slot_ids[i];
		size_t j = 0;

		mecha_count = 0;
		rv = C_GetMechanismList(slot, NULL, &mecha_count);
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out;

		if (mecha_count == 0)
			continue;

		free(mecha_types);
		mecha_types = calloc(mecha_count, sizeof(*mecha_types));
		if (!ADBG_EXPECT_NOT_NULL(c, mecha_types))
			goto out;

		/* Test specific case: valid buffer reference with 0 count */
		mecha_count = 0;
		rv = C_GetMechanismList(slot, mecha_types, &mecha_count);
		if (!ADBG_EXPECT_CK_RESULT(c, CKR_BUFFER_TOO_SMALL, rv))
			goto out;

		rv = C_GetMechanismList(slot, mecha_types, &mecha_count);
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out;

		for (j = 0; j < mecha_count; j++) {
			rv = C_GetMechanismInfo(slot, mecha_types[j],
						&mecha_info);
			if (!ADBG_EXPECT_CK_OK(c, rv))
				goto out;
		}
	}

	Do_ADBG_EndSubCase(c, "Test C_GetMechanism{List|Info}()");
	Do_ADBG_BeginSubCase(c, "Test C_Get*Info() with invalid reference");

	rv = C_GetSlotInfo(max_slot_id + 1, &slot_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	rv = C_GetTokenInfo(max_slot_id + 1, &token_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	mecha_count = 1;
	if (!mecha_types)
		mecha_types = malloc(sizeof(*mecha_types));
	if (!ADBG_EXPECT_NOT_NULL(c, mecha_types))
		goto out;

	rv = C_GetMechanismList(max_slot_id + 1, mecha_types, &mecha_count);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	rv = C_GetMechanismInfo(max_slot_id + 1, CKM_AES_KEY_GEN, &mecha_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	rv = C_GetSlotInfo(ULONG_MAX, &slot_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	rv = C_GetTokenInfo(ULONG_MAX, &token_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	mecha_count = 1;
	rv = C_GetMechanismList(ULONG_MAX, mecha_types, &mecha_count);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

	rv = C_GetMechanismInfo(ULONG_MAX, CKM_AES_KEY_GEN, &mecha_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SLOT_ID_INVALID, rv))
		goto out;

out:
	Do_ADBG_EndSubCase(c, NULL);
	free(slot_ids);
	free(mecha_types);

	rv = C_Finalize(NULL);
	ADBG_EXPECT_CK_OK(c, rv);
}

ADBG_CASE_DEFINE(pkcs11, 1001, xtest_pkcs11_test_1001,
		 "PKCS11: List PKCS#11 slots and get information from");

static void xtest_pkcs11_test_1002(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session[3] = { 0 };
	CK_FLAGS session_flags = 0;
	CK_SESSION_INFO session_info = { };
	CK_FUNCTION_LIST_PTR ckfunc_list = NULL;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_GetFunctionList(&ckfunc_list);
	if (!ADBG_EXPECT_CK_OK(c, rv) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_OpenSession) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_CloseSession) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_CloseAllSessions) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_GetSessionInfo))
		goto bail;

	Do_ADBG_BeginSubCase(c, "Test C_OpenSession()/C_GetSessionInfo()");

	session_flags = CKF_RW_SESSION;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session[0]);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_PARALLEL_NOT_SUPPORTED, rv))
		goto bail;

	session_flags = CKF_SERIAL_SESSION;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session[0]);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	rv = C_GetSessionInfo(session[0], &session_info);
	if (!ADBG_EXPECT_CK_OK(c, rv) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.slotID, ==, slot) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.flags, ==,
					  session_flags) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.state, ==,
					  CKS_RO_PUBLIC_SESSION) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.ulDeviceError, ==, 0))
		goto bail;

	session_flags = CKF_SERIAL_SESSION | CKF_RW_SESSION;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session[1]);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	rv = C_GetSessionInfo(session[1], &session_info);
	if (!ADBG_EXPECT_CK_OK(c, rv) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.slotID, ==, slot) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.flags, ==,
					  session_flags) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.state, ==,
					  CKS_RW_PUBLIC_SESSION) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.ulDeviceError, ==, 0))
		goto bail;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session[2]);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	rv = C_GetSessionInfo(session[2], &session_info);
	if (!ADBG_EXPECT_CK_OK(c, rv) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.slotID, ==, slot) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.flags, ==,
					  session_flags) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.state, ==,
					  CKS_RW_PUBLIC_SESSION) ||
	    !ADBG_EXPECT_COMPARE_UNSIGNED(c, session_info.ulDeviceError, ==, 0))
		goto bail;

	Do_ADBG_EndSubCase(c, "Test C_OpenSession()/C_GetSessionInfo()");
	Do_ADBG_BeginSubCase(c, "Test C_CloseSession()");

	/* Close 2 of them */
	rv = C_CloseSession(session[0]);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	rv = C_GetSessionInfo(session[0], &session_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv))
		goto bail;

	rv = C_GetSessionInfo(session[1], &session_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	rv = C_GetSessionInfo(session[2], &session_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	/* Close all remaining sessions, later calls should failed on session */
	rv = C_CloseAllSessions(slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

	rv = C_CloseSession(session[1]);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv))
		goto bail;

	rv = C_CloseSession(session[2]);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv))
		goto bail;

	rv = C_GetSessionInfo(session[1], &session_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv))
		goto bail;

	rv = C_GetSessionInfo(session[2], &session_info);
	if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv))
		goto bail;

	/* Open a session, should be closed from library closure */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session[0]);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto bail;

bail:
	Do_ADBG_EndSubCase(c, NULL);
	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}

ADBG_CASE_DEFINE(pkcs11, 1002, xtest_pkcs11_test_1002,
		 "PKCS11: Open and close PKCS#11 sessions");

/*
 * Helpers for tests where we must log into the token.
 * These define the genuine PINs and label to be used with the test token.
 */
static CK_UTF8CHAR test_token_so_pin[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8 , 9, 10, };
static CK_UTF8CHAR test_token_user_pin[] = {
	1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12,
};
static CK_UTF8CHAR test_token_label[] = "PKCS11 TA test token";

static CK_RV init_test_token(CK_SLOT_ID slot)
{
	return C_InitToken(slot, test_token_so_pin, sizeof(test_token_so_pin),
			   test_token_label);
}

/* Login as user, eventually reset user PIN if needed */
static CK_RV init_user_test_token(CK_SLOT_ID slot)
{
	CK_FLAGS session_flags = CKF_SERIAL_SESSION | CKF_RW_SESSION;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_RV rv = CKR_GENERAL_ERROR;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (rv)
		return rv;

	rv = C_Login(session, CKU_USER,	test_token_user_pin,
		     sizeof(test_token_user_pin));
	if (rv == CKR_OK) {
		C_Logout(session);
		C_CloseSession(session);
		return rv;
	}

	rv = C_Login(session, CKU_SO, test_token_so_pin,
		     sizeof(test_token_so_pin));
	if (rv) {
		C_CloseSession(session);

		rv = init_test_token(slot);
		if (rv)
			return rv;

		rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
		if (rv)
			return rv;

		rv = C_Login(session, CKU_SO, test_token_so_pin,
			     sizeof(test_token_so_pin));
		if (rv) {
			C_CloseSession(session);
			return rv;
		}
	}

	rv = C_InitPIN(session, test_token_user_pin,
		       sizeof(test_token_user_pin));

	C_Logout(session);
	C_CloseSession(session);

	return rv;
}

static CK_RV test_already_initialized_token(ADBG_Case_t *c, CK_SLOT_ID slot)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_TOKEN_INFO token_info = { };
	/* Same content as test_token_so_pin[] but 1 more byte */
	CK_UTF8CHAR pin1[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, };
	/* Same content as test_token_so_pin[] but 1 different byte */
	CK_UTF8CHAR pin2[] = { 0, 1, 2, 3, 4, 5, 6, 6, 8, 9, 10, };
	CK_FLAGS flags = 0;

	Do_ADBG_BeginSubCase(c, "C_InitToken() on initialized token");

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_InitToken(slot, test_token_so_pin,
			 sizeof(test_token_so_pin) - 1, test_token_label);
	if (!ADBG_EXPECT_COMPARE_UNSIGNED(c, rv, !=, CKR_OK))
		goto out;

	rv = C_InitToken(slot, pin1, sizeof(pin1), test_token_label);
	if (!ADBG_EXPECT_COMPARE_UNSIGNED(c, rv, !=, CKR_OK))
		goto out;

	rv = C_InitToken(slot, pin2, sizeof(pin2), test_token_label);
	if (!ADBG_EXPECT_COMPARE_UNSIGNED(c, rv, !=, CKR_OK))
		goto out;

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	flags = token_info.flags;

	/* Token should have set CKF_SO_PIN_COUNT_LOW to 1 */
	if (!ADBG_EXPECT_TRUE(c, !!(flags & CKF_SO_PIN_COUNT_LOW))) {
		rv = CKR_GENERAL_ERROR;
		goto out;
	}

	rv = init_test_token(slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	flags = token_info.flags;

	/*
	 * Token should have reset CKF_SO_PIN_COUNT_LOW to 0.
	 * Other flags should show a sane initialized state.
	 */
	if (!ADBG_EXPECT_TRUE(c, !(flags & CKF_SO_PIN_COUNT_LOW)) ||
	    !ADBG_EXPECT_TRUE(c, !!(flags & CKF_TOKEN_INITIALIZED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_ERROR_STATE)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_INITIALIZED))) {
		rv = CKR_GENERAL_ERROR;
		goto out;
	}

	rv = init_user_test_token(slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	flags = token_info.flags;

	if (!ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_COUNT_LOW)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_FINAL_TRY)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_LOCKED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_TO_BE_CHANGED)) ||
	    !ADBG_EXPECT_TRUE(c, !!(flags & CKF_USER_PIN_INITIALIZED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_ERROR_STATE))) {
		rv = CKR_GENERAL_ERROR;
		goto out;
	}

out:
	Do_ADBG_EndSubCase(c, "C_InitToken() on initialized token");

	return rv;
}

static CK_RV test_uninitialized_token(ADBG_Case_t *c, CK_SLOT_ID slot)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_TOKEN_INFO token_info = { };
	CK_FLAGS flags = 0;

	Do_ADBG_BeginSubCase(c, "C_InitToken() on uninitialized token");

	rv = init_test_token(slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	flags = token_info.flags;

	if (!ADBG_EXPECT_TRUE(c, !!(flags & CKF_TOKEN_INITIALIZED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_ERROR_STATE)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_INITIALIZED))) {
		rv = CKR_GENERAL_ERROR;
		goto out;
	}

	rv = init_user_test_token(slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	flags = token_info.flags;

	if (!ADBG_EXPECT_TRUE(c, !!(flags & CKF_TOKEN_INITIALIZED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_COUNT_LOW)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_FINAL_TRY)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_LOCKED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_USER_PIN_TO_BE_CHANGED)) ||
	    !ADBG_EXPECT_TRUE(c, !!(flags & CKF_USER_PIN_INITIALIZED)) ||
	    !ADBG_EXPECT_TRUE(c, !(flags & CKF_ERROR_STATE)))
		rv = CKR_GENERAL_ERROR;

out:
	Do_ADBG_EndSubCase(c, "C_InitToken() on uninitialized token");

	return rv;
}

static CK_RV test_login_logout(ADBG_Case_t *c, CK_SLOT_ID slot)
{
	CK_FLAGS session_flags = CKF_SERIAL_SESSION | CKF_RW_SESSION;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_RV rv = CKR_GENERAL_ERROR;

	Do_ADBG_BeginSubCase(c, "Test C_Login()/C_Logout()");

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	/* Logout: should fail as we did not log in yet */
	rv = C_Logout(session);
	ADBG_EXPECT_CK_RESULT(c, CKR_USER_NOT_LOGGED_IN, rv);

	/* Login/re-log/logout user */
	rv = C_Login(session, CKU_USER, test_token_user_pin,
		     sizeof(test_token_user_pin));
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_Login(session, CKU_USER, test_token_user_pin,
		     sizeof(test_token_user_pin));
	ADBG_EXPECT_CK_RESULT(c, CKR_USER_ALREADY_LOGGED_IN, rv);

	rv = C_Logout(session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	/* Login/re-log/logout security officer */
	rv = C_Login(session, CKU_SO, test_token_so_pin,
		     sizeof(test_token_so_pin));
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_Login(session, CKU_SO, test_token_so_pin,
		     sizeof(test_token_so_pin));
	ADBG_EXPECT_CK_RESULT(c, CKR_USER_ALREADY_LOGGED_IN, rv);

	rv = C_Logout(session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	/* Login user then SO and reverse */
	rv = C_Login(session, CKU_SO, test_token_so_pin,
		     sizeof(test_token_so_pin));
	ADBG_EXPECT_CK_OK(c, rv);

	rv = C_Login(session, CKU_USER, test_token_user_pin,
		     sizeof(test_token_user_pin));
	ADBG_EXPECT_CK_RESULT(c, CKR_USER_ANOTHER_ALREADY_LOGGED_IN, rv);

	rv = C_Logout(session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_Login(session, CKU_USER, test_token_user_pin,
		     sizeof(test_token_user_pin));
	ADBG_EXPECT_CK_OK(c, rv);

	rv = C_Login(session, CKU_SO, test_token_so_pin,
		     sizeof(test_token_so_pin));
	ADBG_EXPECT_CK_RESULT(c, CKR_USER_ANOTHER_ALREADY_LOGGED_IN, rv);

	rv = C_Logout(session);
	ADBG_EXPECT_CK_OK(c, rv);

	/* Login context specifc, in an invalid case (need an operation) */
	rv = C_Login(session, CKU_CONTEXT_SPECIFIC, test_token_user_pin,
		     sizeof(test_token_user_pin));
	ADBG_EXPECT_CK_RESULT(c, CKR_OPERATION_NOT_INITIALIZED, rv);

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

out:
	Do_ADBG_EndSubCase(c, "Test C_Login()/C_Logout()");
	return rv;
}

static CK_RV test_set_pin(ADBG_Case_t *c, CK_SLOT_ID slot,
			  CK_USER_TYPE user_type)
{
	CK_FLAGS session_flags = CKF_SERIAL_SESSION | CKF_RW_SESSION;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_UTF8CHAR some_pin[] = { 7, 6, 5, 4, 3, 2, 1, 2, 3, 4, 5, 6, 7 };
	CK_UTF8CHAR_PTR old_pin = NULL;
	CK_USER_TYPE ut = user_type;
	size_t old_pin_sz = 0;
	CK_RV rv2 = CKR_OK;
	CK_RV rv = CKR_OK;

	Do_ADBG_BeginSubCase(c, "Test C_SetPIN() user_type %lu", user_type);

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	if (user_type == CKU_SO) {
		old_pin = (CK_UTF8CHAR_PTR)test_token_so_pin;
		old_pin_sz = sizeof(test_token_so_pin);
	} else {
		old_pin = (CK_UTF8CHAR_PTR)test_token_user_pin;
		old_pin_sz = sizeof(test_token_user_pin);
		ut = CKU_USER;
	}

	if (ut == user_type) {
		rv = C_Login(session, ut, old_pin, old_pin_sz);
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out_session;
	}

	rv = C_SetPIN(session, old_pin, old_pin_sz, some_pin, sizeof(some_pin));
	if (!ADBG_EXPECT_CK_OK(c, rv)) {
		if (ut == user_type)
			goto out_logout;
		else
			goto out_session;
	}

	if (ut == user_type) {
		rv = C_Logout(session);
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out_session;
	}

	rv = C_Login(session, ut, some_pin, sizeof(some_pin));
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out_session;

	rv = C_SetPIN(session, some_pin, sizeof(some_pin), old_pin, old_pin_sz);
	ADBG_EXPECT_CK_OK(c, rv);

out_logout:
	rv2 = C_Logout(session);
	if (!ADBG_EXPECT_CK_OK(c, rv2) && !rv)
		rv = rv2;
out_session:
	rv2 = C_CloseSession(session);
	if (!ADBG_EXPECT_CK_OK(c, rv2) && !rv)
		rv = rv2;
out:
	Do_ADBG_EndSubCase(c, "Test C_SetPIN() user_type %lu", user_type);

	return rv;
}

static void xtest_pkcs11_test_1003(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_FUNCTION_LIST_PTR ckfunc_list = NULL;
	CK_SLOT_ID slot = 0;
	CK_TOKEN_INFO token_info = { };

	rv = C_GetFunctionList(&ckfunc_list);
	if (!ADBG_EXPECT_CK_OK(c, rv) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_InitToken) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_InitPIN) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_SetPIN) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_Login) ||
	    !ADBG_EXPECT_NOT_NULL(c, ckfunc_list->C_Logout))
		goto out;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_GetTokenInfo(slot, &token_info);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	/* Abort test if token is about to lock */
	if (!ADBG_EXPECT_TRUE(c, !(token_info.flags & CKF_SO_PIN_FINAL_TRY)))
		goto out;

	if (!(token_info.flags & CKF_TOKEN_INITIALIZED)) {
		rv = test_uninitialized_token(c, slot);
		if (rv != CKR_OK)
			goto out;
	}

	rv = test_already_initialized_token(c, slot);
	if (rv != CKR_OK)
		goto out;

	rv = test_login_logout(c, slot);
	if (rv != CKR_OK)
		goto out;

	rv = test_set_pin(c, slot, CKU_USER);
	if (rv != CKR_OK)
		goto out;

	rv = test_set_pin(c, slot, CKU_SO);
	if (rv != CKR_OK)
		goto out;

	/*
	 * CKU_CONTEXT_SPECIFIC is anything not CKU_USER or CKU_SO in order
	 * to skip the initial login.
	 */
	test_set_pin(c, slot, CKU_CONTEXT_SPECIFIC);
out:
	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}

ADBG_CASE_DEFINE(pkcs11, 1003, xtest_pkcs11_test_1003,
		 "PKCS11: Login to PKCS#11 token");

static CK_ATTRIBUTE cktest_token_object[] = {
	{ CKA_DECRYPT,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) },
	{ CKA_TOKEN,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) },
	{ CKA_MODIFIABLE, &(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) },
	{ CKA_CLASS,	&(CK_OBJECT_CLASS){CKO_SECRET_KEY},
						sizeof(CK_OBJECT_CLASS) },
	{ CKA_KEY_TYPE,	&(CK_KEY_TYPE){CKK_AES}, sizeof(CK_KEY_TYPE) },
	{ CKA_VALUE,	(void *)cktest_aes128_key, sizeof(cktest_aes128_key) },
};

static CK_ATTRIBUTE cktest_session_object[] = {
	{ CKA_DECRYPT,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) },
	{ CKA_TOKEN,	&(CK_BBOOL){CK_FALSE}, sizeof(CK_BBOOL) },
	{ CKA_MODIFIABLE, &(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) },
	{ CKA_KEY_TYPE,	&(CK_KEY_TYPE){CKK_AES}, sizeof(CK_KEY_TYPE) },
	{ CKA_CLASS,	&(CK_OBJECT_CLASS){CKO_SECRET_KEY},
						sizeof(CK_OBJECT_CLASS) },
	{ CKA_VALUE,	(void *)cktest_aes128_key, sizeof(cktest_aes128_key) },
};

/* Create session object and token object from a session */
static void test_create_destroy_single_object(ADBG_Case_t *c, bool persistent)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE obj_hdl = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_SERIAL_SESSION | CKF_RW_SESSION;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	if (persistent) {
		rv = C_CreateObject(session, cktest_token_object,
				    ARRAY_SIZE(cktest_token_object),
				    &obj_hdl);
	} else {
		rv = C_CreateObject(session, cktest_session_object,
				    ARRAY_SIZE(cktest_session_object),
				    &obj_hdl);
	}

	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_DestroyObject(session, obj_hdl);
	ADBG_EXPECT_CK_OK(c, rv);
out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}

static void test_create_destroy_session_objects(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE obj_hdl[512] = { 0 };
	CK_FLAGS session_flags = CKF_SERIAL_SESSION | CKF_RW_SESSION;
	size_t n = 0;

	for (n = 0; n < ARRAY_SIZE(obj_hdl); n++)
		obj_hdl[n] = CK_INVALID_HANDLE;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	for (n = 0; n < ARRAY_SIZE(obj_hdl); n++) {
		rv = C_CreateObject(session, cktest_session_object,
				    ARRAY_SIZE(cktest_session_object),
				    obj_hdl + n);

		if (rv == CKR_DEVICE_MEMORY || !ADBG_EXPECT_CK_OK(c, rv))
			break;
	}

	Do_ADBG_Log("    created object count: %zu", n);

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_CreateObject(session, cktest_session_object,
			    ARRAY_SIZE(cktest_session_object),
			    obj_hdl);

	ADBG_EXPECT_CK_OK(c, rv);

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}

/* Create session object and token object from a session */
static void test_create_objects_in_session(ADBG_Case_t *c, bool readwrite)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE token_obj_hld = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE session_obj_hld = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_SERIAL_SESSION;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	if (readwrite)
		session_flags |= CKF_RW_SESSION;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_CreateObject(session, cktest_token_object,
			    ARRAY_SIZE(cktest_token_object),
			    &token_obj_hld);

	if (readwrite) {
		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out;
	} else {
		if (!ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_READ_ONLY, rv))
			goto out;
	}

	rv = C_CreateObject(session, cktest_session_object,
			    ARRAY_SIZE(cktest_session_object),
			    &session_obj_hld);

	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out_tobj;

	rv = C_DestroyObject(session, session_obj_hld);
	ADBG_EXPECT_CK_OK(c, rv);

out_tobj:
	if (readwrite) {
		rv = C_DestroyObject(session, token_obj_hld);
		ADBG_EXPECT_CK_OK(c, rv);
	}
out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}

static void xtest_pkcs11_test_1004(ADBG_Case_t *c)
{
	Do_ADBG_BeginSubCase(c, "Create and destroy a volatile object");
	test_create_destroy_single_object(c, false /*!persistent*/);
	Do_ADBG_EndSubCase(c, "Create and destroy a volatile object");

	Do_ADBG_BeginSubCase(c, "Create and destroy a persistent object");
	test_create_destroy_single_object(c, true /*persistent*/);
	Do_ADBG_EndSubCase(c, "Create and destroy a persistent object");

	Do_ADBG_BeginSubCase(c, "Create and destroy many session objects");
	test_create_destroy_session_objects(c);
	Do_ADBG_EndSubCase(c, "Create and destroy many session objects");

	Do_ADBG_BeginSubCase(c, "Create objects in a read-only session");
	test_create_objects_in_session(c, false /*!readwrite*/);
	Do_ADBG_EndSubCase(c, "Create objects in a read-only session");

	Do_ADBG_BeginSubCase(c, "Create objects in a read/write session");
	test_create_objects_in_session(c, true /*readwrite*/);
	Do_ADBG_EndSubCase(c, "Create objects in a read/write session");
}

ADBG_CASE_DEFINE(pkcs11, 1004, xtest_pkcs11_test_1004,
		 "PKCS11: create/destroy PKCS#11 simple objects");


static const CK_MECHANISM_TYPE allowed_only_aes_ecb[] = {
	CKM_AES_ECB,
};
static const CK_MECHANISM_TYPE allowed_not_aes_ecb[] = {
	CKM_AES_CBC, CKM_AES_CBC_PAD, CKM_AES_CTR, CKM_AES_CTS,
	CKM_AES_GCM, CKM_AES_CCM,
};
static const CK_MECHANISM_TYPE allowed_only_aes_cbcnopad[] = {
	CKM_AES_CBC,
};
static const CK_MECHANISM_TYPE allowed_not_aes_cbcnopad[] = {
	CKM_AES_ECB, CKM_AES_CBC_PAD, CKM_AES_CTR, CKM_AES_CTS,
	CKM_AES_GCM, CKM_AES_CCM,
};
static const CK_MECHANISM_TYPE allowed_only_aes_ctr[] = {
	CKM_AES_CTR,
};
static const CK_MECHANISM_TYPE allowed_not_aes_ctr[] = {
	CKM_AES_ECB, CKM_AES_CBC, CKM_AES_CBC_PAD, CKM_AES_CTS,
	CKM_AES_GCM, CKM_AES_CCM,
};
static const CK_MECHANISM_TYPE allowed_only_aes_cts[] = {
	CKM_AES_CTS,
};
static const CK_MECHANISM_TYPE allowed_not_aes_cts[] = {
	CKM_AES_ECB, CKM_AES_CBC, CKM_AES_CBC_PAD, CKM_AES_CTR,
	CKM_AES_GCM, CKM_AES_CCM,
};

#define CKTEST_AES_KEY \
	{ CKA_CLASS,	&(CK_OBJECT_CLASS){CKO_SECRET_KEY},	\
			sizeof(CK_OBJECT_CLASS) },		\
	{ CKA_KEY_TYPE,	&(CK_KEY_TYPE){CKK_AES},		\
			sizeof(CK_KEY_TYPE) },			\
	{ CKA_VALUE,	(void *)cktest_aes128_key,		\
			sizeof(cktest_aes128_key) }

#define CKTEST_AES_ALLOWED_KEY(_allowed) \
	{ CKA_ALLOWED_MECHANISMS, (void *)_allowed, sizeof(_allowed), }

#define CK_KEY_ALLOWED_AES_TEST(_label, _allowed) \
	static CK_ATTRIBUTE _label[] = {				\
		CKTEST_AES_KEY,						\
		{ CKA_ENCRYPT,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) }, \
		{ CKA_DECRYPT,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) }, \
		CKTEST_AES_ALLOWED_KEY(_allowed),			\
	}

#define CK_KEY_ALLOWED_AES_ENC_TEST(_label, _allowed) \
	static CK_ATTRIBUTE _label[] = {				\
		CKTEST_AES_KEY,						\
		{ CKA_ENCRYPT,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) }, \
		CKTEST_AES_ALLOWED_KEY(_allowed),			\
	}
#define CK_KEY_ALLOWED_AES_DEC_TEST(_label, _allowed) \
	static CK_ATTRIBUTE _label[] = {				\
		CKTEST_AES_KEY,						\
		{ CKA_DECRYPT,	&(CK_BBOOL){CK_TRUE}, sizeof(CK_BBOOL) }, \
		CKTEST_AES_ALLOWED_KEY(_allowed),			\
	}

CK_KEY_ALLOWED_AES_TEST(cktest_aes_only_ecb, allowed_only_aes_ecb);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_not_ecb, allowed_not_aes_ecb);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_only_cbcnopad, allowed_only_aes_cbcnopad);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_not_cbcnopad, allowed_not_aes_cbcnopad);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_only_cts, allowed_only_aes_cts);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_not_cts, allowed_not_aes_cts);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_only_ctr, allowed_only_aes_ctr);
CK_KEY_ALLOWED_AES_TEST(cktest_aes_not_ctr, allowed_not_aes_ctr);

struct cktest_allowed_test {
	CK_ATTRIBUTE_PTR attr_key;
	CK_ULONG attr_count;
	CK_MECHANISM_PTR mechanism;
};

#define CKTEST_KEY_MECHA(key, mecha) {	\
		.attr_key = key,		\
		.attr_count = ARRAY_SIZE(key),	\
		.mechanism = mecha,		\
	}

static const struct cktest_allowed_test cktest_allowed_valid[] = {
	CKTEST_KEY_MECHA(cktest_aes_only_ecb, &cktest_aes_ecb_mechanism),
	CKTEST_KEY_MECHA(cktest_aes_only_cbcnopad, &cktest_aes_cbc_mechanism),
	CKTEST_KEY_MECHA(cktest_aes_only_cts, &cktest_aes_cts_mechanism),
	CKTEST_KEY_MECHA(cktest_aes_only_ctr, &cktest_aes_ctr_mechanism),
};

static const struct cktest_allowed_test cktest_allowed_invalid[] = {
	CKTEST_KEY_MECHA(cktest_aes_not_ecb, &cktest_aes_ecb_mechanism),
	CKTEST_KEY_MECHA(cktest_aes_not_cbcnopad, &cktest_aes_cbc_mechanism),
	CKTEST_KEY_MECHA(cktest_aes_not_cts, &cktest_aes_cts_mechanism),
	CKTEST_KEY_MECHA(cktest_aes_not_ctr, &cktest_aes_ctr_mechanism),
};

/* Create session object and token object from a session */
static CK_RV cipher_init_final(ADBG_Case_t *c, CK_SESSION_HANDLE session,
				CK_ATTRIBUTE_PTR attr_key, CK_ULONG attr_count,
				CK_MECHANISM_PTR mechanism, uint32_t mode,
				CK_RV expected_rc)
{
	CK_RV rv2 = CKR_GENERAL_ERROR;
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_OBJECT_HANDLE object = CK_INVALID_HANDLE;

	switch (mode) {
	case TEE_MODE_ENCRYPT:
	case TEE_MODE_DECRYPT:
		break;
	default:
		ADBG_EXPECT_TRUE(c, false);
	}

	rv = C_CreateObject(session, attr_key, attr_count, &object);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return rv;

	if (mode == TEE_MODE_ENCRYPT)
		rv = C_EncryptInit(session, mechanism, object);
	if (mode == TEE_MODE_DECRYPT)
		rv = C_DecryptInit(session, mechanism, object);

	if (!ADBG_EXPECT_CK_RESULT(c, expected_rc, rv)) {
		rv = CKR_GENERAL_ERROR;
		goto out;
	}

	if (rv) {
		/*
		 * If we're here it was the expected error code above and
		 * we're supposed to return OK below.
		 */
		rv = CKR_OK;
	} else {
		if (mode == TEE_MODE_ENCRYPT)
			rv = C_EncryptFinal(session, NULL, NULL);
		if (mode == TEE_MODE_DECRYPT)
			rv = C_DecryptFinal(session, NULL, NULL);

		/* Only check that the operation is no more active */
		if (ADBG_EXPECT_TRUE(c, (rv == CKR_ARGUMENTS_BAD) ||
					(rv == CKR_DATA_LEN_RANGE)))
			rv = CKR_OK;
		else
			rv = CKR_GENERAL_ERROR;
	}

out:
	rv2 = C_DestroyObject(session, object);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv)
		return rv;
	else
		return rv2;
}

CK_KEY_ALLOWED_AES_ENC_TEST(cktest_aes_enc_only_cts, allowed_only_aes_cts);

CK_KEY_ALLOWED_AES_DEC_TEST(cktest_aes_dec_only_ctr, allowed_only_aes_ctr);

static void xtest_pkcs11_test_1005(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_SERIAL_SESSION;
	size_t n = 0;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	for (n = 0; n < ARRAY_SIZE(cktest_allowed_valid); n++) {

		Do_ADBG_BeginSubCase(c, "valid usage #%zu", n);

		rv = cipher_init_final(c, session,
					cktest_allowed_valid[n].attr_key,
					cktest_allowed_valid[n].attr_count,
					cktest_allowed_valid[n].mechanism,
					TEE_MODE_ENCRYPT,
					CKR_OK);

		ADBG_EXPECT_CK_OK(c, rv);

		Do_ADBG_EndSubCase(c, NULL);
		if (rv)
			goto out;

	}

	for (n = 0; n < ARRAY_SIZE(cktest_allowed_invalid); n++) {
		Do_ADBG_BeginSubCase(c, "invalid usage #%zu", n);

		rv = cipher_init_final(c, session,
					cktest_allowed_invalid[n].attr_key,
					cktest_allowed_invalid[n].attr_count,
					cktest_allowed_invalid[n].mechanism,
					TEE_MODE_ENCRYPT,
					CKR_KEY_FUNCTION_NOT_PERMITTED);

		ADBG_EXPECT_CK_OK(c, rv);

		Do_ADBG_EndSubCase(c, NULL);
		if (rv)
			goto out;

	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}


ADBG_CASE_DEFINE(pkcs11, 1005, xtest_pkcs11_test_1005,
		"PKCS11: Check ciphering with valid and invalid keys #1");

static void xtest_pkcs11_test_1006(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_SERIAL_SESSION;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	/* Encrypt only AES CTS key */
	rv = cipher_init_final(c, session,
				cktest_aes_enc_only_cts,
				ARRAY_SIZE(cktest_aes_enc_only_cts),
				&cktest_aes_cts_mechanism,
				TEE_MODE_ENCRYPT,
				CKR_OK);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = cipher_init_final(c, session,
				cktest_aes_enc_only_cts,
				ARRAY_SIZE(cktest_aes_enc_only_cts),
				&cktest_aes_cts_mechanism,
				TEE_MODE_DECRYPT,
				CKR_KEY_FUNCTION_NOT_PERMITTED);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	/* Decrypt only AES CTR key */
	rv = cipher_init_final(c, session,
				cktest_aes_dec_only_ctr,
				ARRAY_SIZE(cktest_aes_dec_only_ctr),
				&cktest_aes_ctr_mechanism,
				TEE_MODE_ENCRYPT,
				CKR_KEY_FUNCTION_NOT_PERMITTED);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = cipher_init_final(c, session,
				cktest_aes_dec_only_ctr,
				ARRAY_SIZE(cktest_aes_dec_only_ctr),
				&cktest_aes_ctr_mechanism,
				TEE_MODE_DECRYPT,
				CKR_OK);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}
ADBG_CASE_DEFINE(pkcs11, 1006, xtest_pkcs11_test_1006,
		"PKCS11: Check ciphering with valid and invalid keys #2");

static CK_RV open_cipher_session(ADBG_Case_t *c,
				 CK_SLOT_ID slot, CK_SESSION_HANDLE_PTR session,
				 CK_ATTRIBUTE_PTR attr_key, CK_ULONG attr_count,
				 CK_MECHANISM_PTR mechanism, uint32_t mode)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_OBJECT_HANDLE object = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_SERIAL_SESSION;

	switch (mode) {
	case TEE_MODE_ENCRYPT:
	case TEE_MODE_DECRYPT:
		break;
	default:
		ADBG_EXPECT_TRUE(c, false);
		return CKR_GENERAL_ERROR;
	}

	rv = C_OpenSession(slot, session_flags, NULL, 0, session);
	if (rv == CKR_DEVICE_MEMORY)
		return rv;
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return rv;

	rv = C_CreateObject(*session, attr_key, attr_count, &object);
	if (rv == CKR_DEVICE_MEMORY)
		return rv;
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return rv;

	if (mode == TEE_MODE_ENCRYPT)
		rv = C_EncryptInit(*session, mechanism, object);
	if (mode == TEE_MODE_DECRYPT)
		rv = C_DecryptInit(*session, mechanism, object);

	if (rv == CKR_DEVICE_MEMORY)
		return rv;
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return CKR_GENERAL_ERROR;

	return rv;
}

static void xtest_pkcs11_test_1007(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE sessions[128];
	size_t n = 0;

	for (n = 0; n < ARRAY_SIZE(sessions); n++)
		sessions[n] = CK_INVALID_HANDLE;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	for (n = 0; n < ARRAY_SIZE(sessions); n++) {

		rv = open_cipher_session(c, slot, &sessions[n],
					 cktest_allowed_valid[0].attr_key,
					 cktest_allowed_valid[0].attr_count,
					 cktest_allowed_valid[0].mechanism,
					 TEE_MODE_ENCRYPT);

		/* Failure due to memory allocation is not a error case */
		if (rv == CKR_DEVICE_MEMORY)
			break;

		if (!ADBG_EXPECT_CK_OK(c, rv))
			goto out;
	}

	if (!ADBG_EXPECT_COMPARE_UNSIGNED(c, n, >, 0))
		goto out;

	Do_ADBG_Log("    created sessions count: %zu", n);

	/* Closing session with out bound and invalid IDs (or negative ID) */
	rv = C_CloseSession(sessions[n - 1] + 1024);
	ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv);
	rv = C_CloseSession(CK_INVALID_HANDLE);
	ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv);
	rv = C_CloseSession(~0);
	ADBG_EXPECT_CK_RESULT(c, CKR_SESSION_HANDLE_INVALID, rv);

	/* Closing each session: all related resources shall be free */
	for (n = 0; n < ARRAY_SIZE(sessions); n++) {
		if (sessions[n] == CK_INVALID_HANDLE)
			continue;

		rv = C_CloseSession(sessions[n]);
		ADBG_EXPECT_CK_OK(c, rv);
		sessions[n] = CK_INVALID_HANDLE;
	}

	/* Open and close another session */
	rv = open_cipher_session(c, slot, &sessions[0],
				 cktest_allowed_valid[0].attr_key,
				 cktest_allowed_valid[0].attr_count,
				 cktest_allowed_valid[0].mechanism,
				 TEE_MODE_ENCRYPT);

	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = C_CloseSession(sessions[0]);
	ADBG_EXPECT_CK_OK(c, rv);
	sessions[0] = CK_INVALID_HANDLE;

out:
	for (n = 0; n < ARRAY_SIZE(sessions); n++) {
		if (sessions[n] == CK_INVALID_HANDLE)
			continue;

		rv = C_CloseSession(sessions[n]);
		ADBG_EXPECT_CK_OK(c, rv);
	}

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);
}
ADBG_CASE_DEFINE(pkcs11, 1007, xtest_pkcs11_test_1007,
		"PKCS11: Check operations release at session closure");

uint8_t origin_8K_0[8192] = {0};

uint8_t hmac_256_key[] = {
	0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
};

uint8_t hmac_256_mac[] = {
	0x22, 0x78, 0xCE, 0x63, 0x7A, 0x4A, 0x86, 0x10, 0x54, 0xA0, 0xEF, 0x9C, 0x31, 0xFA, 0x57, 0xE8,
	0xA7, 0x81, 0xA9, 0xB3, 0x74, 0x4A, 0x70, 0xF5, 0xB3, 0xC8, 0x69, 0x65, 0x40, 0xE6, 0x4E, 0xA5,
};

uint8_t hmac_384_key[] = {
	0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
};

uint8_t hmac_384_mac[] = {
	0x4D, 0x02, 0xB6, 0xE0, 0x2B, 0x85, 0x19, 0x6F, 0x0E, 0xDF, 0x0D, 0x16, 0x2C, 0x00, 0x9D, 0x35,
	0x8B, 0x1C, 0x12, 0x69, 0x6D, 0x86, 0xAD, 0xD4, 0x17, 0xE3, 0x71, 0x34, 0x65, 0x18, 0x2B, 0xBD,
	0xFF, 0xFC, 0xE2, 0xF0, 0x8D, 0xA2, 0x9D, 0xCB, 0x27, 0xBE, 0xC2, 0xEF, 0xEA, 0x08, 0xCD, 0xFD,
};

uint8_t cmac_128_key[] = {
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c,
};

uint8_t cmac_128_mac[] = {
	0x27, 0x4D, 0x63, 0x3F, 0xD8, 0xD7, 0x66, 0xA6, 0xD9, 0xDA, 0x5A, 0xD5, 0x0F, 0x5A, 0x52, 0x2A,
};

uint8_t cmac_192_key[] = {
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b,
};

uint8_t cmac_192_mac[] = {
	0x03, 0x02, 0xCF, 0xFB, 0x13, 0x3D, 0x74, 0xD9, 0xC5, 0xC9, 0xE6, 0x3C, 0xAF, 0xE7, 0xFB, 0xD6,
};

uint8_t cmac_256_key[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4,
};

uint8_t cmac_256_mac[] = {
	0x66, 0x00, 0x9F, 0xF9, 0x91, 0xA5, 0xDC, 0x5E, 0x11, 0xF5, 0x91, 0xBB, 0xC9, 0x60, 0x06, 0xB6,
};


//uint8_t aes_IV[16] = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff};
uint8_t aes_IV[16] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f};

//uint8_t aes_CTR[16] = {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff};
uint8_t aes_CTR[16] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9};

uint8_t aes_msg[] = {
// static uint8_t aes256_cbc_ctr_plaintext[256] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};

uint8_t aes_gcm_128_mac[] = {
	0x49, 0xAC, 0xD5, 0xA3, 0x39, 0xFF, 0xB2, 0xD4, 0xD7, 0x89, 0x52, 0xC8, 0xAA, 0x98, 0x2F, 0x80,
};

uint8_t aes_gcm_192_mac[] = {
	0x60, 0x63, 0x6A, 0xAE, 0x5C, 0x3D, 0x90, 0xEA, 0x96, 0x7E, 0x29, 0x64, 0x83, 0x16, 0x53, 0x87,
};

uint8_t aes_gcm_256_mac[] = {
	0x89, 0x45, 0x46, 0xAA, 0x7E, 0xDB, 0x92, 0x51, 0x17, 0x25, 0xFA, 0xE9, 0xA1, 0xC3, 0xB6, 0x56,
};


// Used by CBC stream mode Encyption
static uint8_t aes_msg0[]= {
//static uint8_t aes256_cbc_ctr_plaintext[256] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
};
static uint8_t aes_msg1[]= {
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
};
static uint8_t aes_msg2[]= {
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
};
static uint8_t aes_msg3[]= {
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};

static uint8_t sha1_digest_1M_0_output[20] = {
	0x3B, 0x71, 0xF4, 0x3F, 0xF3, 0x0F, 0x4B, 0x15, 0xB5, 0xCD, 0x85, 0xDD, 0x9E, 0x95, 0xEB, 0xC7,
	0xE8, 0x4E, 0xB5, 0xA3,
};
static uint8_t sha224_digest_1M_0_output[28] = {
	0xAA, 0xAA, 0xDE, 0xDC, 0xCB, 0x99, 0x8D, 0xDB, 0x99, 0xD2, 0xC0, 0x20, 0xB6, 0x58, 0x5A, 0x5E,
	0xCE, 0xAD, 0xCF, 0xF0, 0xC3, 0x48, 0xF3, 0x5F, 0xE5, 0x98, 0xB4, 0x18,
};
static uint8_t sha256_digest_1M_0_output[32] = {
	0x30 ,0xE1 ,0x49 ,0x55 ,0xEB ,0xF1 ,0x35 ,0x22 ,0x66 ,0xDC ,0x2F ,0xF8 ,0x06 ,0x7E ,0x68 ,0x10 ,
	0x46 ,0x07 ,0xE7 ,0x50 ,0xAB ,0xB9 ,0xD3 ,0xB3 ,0x65 ,0x82 ,0xB8 ,0xAF ,0x90 ,0x9F ,0xCB ,0x58 ,
};
static uint8_t sha384_digest_1M_0_output[48] = {
	0x31, 0x64, 0x67, 0x3A, 0x8A, 0xC2, 0x75, 0x76, 0xAB, 0x5F, 0xC0, 0x6B, 0x9A, 0xDC, 0x4C, 0xE0,
	0xAC, 0xA5, 0xBD, 0x30, 0x25, 0x38, 0x4B, 0x1C, 0xF2, 0x12, 0x8A, 0x87, 0x95, 0xE7, 0x47, 0xC4,
	0x31, 0xE8, 0x82, 0x78, 0x5A, 0x0B, 0xF8, 0xDC, 0x70, 0xB4, 0x29, 0x95, 0xDB, 0x38, 0x85, 0x75,
};
static uint8_t sha512_digest_1M_0_output[64] = {
	0xD6, 0x29, 0x26, 0x85, 0xB3, 0x80, 0xE3, 0x38, 0xE0, 0x25, 0xB3, 0x41, 0x5A, 0x90, 0xFE, 0x8F,
	0x9D, 0x39, 0xA4, 0x6E, 0x7B, 0xDB, 0xA8, 0xCB, 0x78, 0xC5, 0x0A, 0x33, 0x8C, 0xEF, 0xCA, 0x74,
	0x1F, 0x69, 0xE4, 0xE4, 0x64, 0x11, 0xC3, 0x2D, 0xE1, 0xAF, 0xDE, 0xDF, 0xB2, 0x68, 0xE5, 0x79,
	0xA5, 0x1F, 0x81, 0xFF, 0x85, 0xE5, 0x6F, 0x55, 0xB0, 0xEE, 0x7C, 0x33, 0xFE, 0x8C, 0x25, 0xC9,
};

// 512 bytes
static uint8_t sha_origin_text[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x22, 0x66,
};

//static uint8_t sha_stream_origin_text[128] = {
static uint8_t sha_stream_origin_text_1[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
};

static uint8_t sha_stream_origin_text_11[] = {
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
};
static uint8_t sha_stream_origin_text_12[] = {
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
};
static uint8_t sha_stream_origin_text_13[] = {
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};
//static uint8_t sha_stream_origin_text[256] = {
static uint8_t sha_stream_origin_text_2[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};
//static uint8_t sha_stream_origin_text[128] = {
static uint8_t sha_stream_origin_text_3[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
	0x22, 0x66,
};

static uint8_t msg_buf[64] = {
	0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

static uint8_t sha1_digest_output[20] = {
	0x40,0x5f,0xb7,0x69,0x31,0x2f,0x2f,0x56,0x4,0x9a,0x17,0x5,0xd9,0x33,0xad,0x1d,
	0xc2,0x4,0x5,0x12,
};

static uint8_t sha224_digest_output[28] = {
	0xd2,0x79,0x6c,0x24,0x1d,0xeb,0x8d,0x11,0x7d,0x53,0x62,0x90,0xae,0x7f,0x5c,0x6c,
	0xe8,0xf9,0xb6,0xad,0x55,0xe2,0x65,0x22,0x8c,0x16,0xf6,0x68,
};

static uint8_t sha256_digest_output[32] = {
	0xd6,0x72,0xd0,0x0,0xa2,0x9d,0x7d,0x7,0x32,0x45,0xf0,0x2a,0x8c,0x73,0xb6,0x96,
	0xee,0x71,0x52,0x99,0xd8,0x2e,0x74,0xda,0xf0,0x96,0x9d,0xaa,0x1a,0x37,0xfb,0xf,
};

static uint8_t sha384_digest_output[48] = {
	0x6,0xb5,0xb8,0x1c,0xc4,0x59,0x3d,0x59,0xfd,0xed,0x6d,0x5d,0xcb,0x21,0x1b,0x4e,
	0x5a,0x2c,0x46,0xbe,0xc6,0xf2,0x9,0x5b,0xe2,0x9c,0x9c,0x9a,0xa5,0x74,0x5b,0x28,
	0xfa,0xb5,0x9b,0xd8,0xb4,0xea,0xbf,0xdb,0x8e,0x86,0xf4,0xda,0x8a,0x63,0x19,0x34,
};

static uint8_t sha512_digest_output[64] = {
	0x82,0x6d,0xf,0x91,0xa7,0x29,0xb6,0x5f,0x74,0xe4,0xaa,0x91,0xa,0x73,0xc9,0xb,
	0xcb,0x5b,0x4,0xde,0x80,0x62,0xd0,0xac,0x55,0x6e,0x35,0x9f,0xab,0x2e,0xd4,0xd5,
	0x91,0xba,0xb1,0x17,0x47,0x6e,0x66,0x5b,0x79,0xc7,0x79,0x3a,0x3,0x8,0xb5,0xf5,
	0xd1,0x15,0x1d,0x16,0x17,0x9b,0xf6,0x7e,0x77,0x2d,0xd2,0x58,0x50,0xdf,0x4c,0x69,
};

static uint8_t digest_golden[64] = { //512 bits
	0x19, 0xfa, 0x61, 0xd7, 0x55, 0x22, 0xa4, 0x66, 0x9b, 0x44, 0xe3, 0x9c, 0x1d, 0x2e, 0x17, 0x26,
	0xc5, 0x30, 0x23, 0x21, 0x30, 0xd4, 0x07, 0xf8, 0x9a, 0xfe, 0xe0, 0x96, 0x49, 0x97, 0xf7, 0xa7,
	0x3e, 0x83, 0xbe, 0x69, 0x8b, 0x28, 0x8f, 0xeb, 0xcf, 0x88, 0xe3, 0xe0, 0x3c, 0x4f, 0x07, 0x57,
	0xea, 0x89, 0x64, 0xe5, 0x9b, 0x63, 0xd9, 0x37, 0x08, 0xb1, 0x38, 0xcc, 0x42, 0xa6, 0x6e, 0xb3,
};

#define BUFFERSIZ    1024
#define XTEST_NULL_KEY_ID        (0x0000)
#define XTEST_AES_128_KEY_ID_0   (0x1000)
#define XTEST_AES_192_KEY_ID_0   (0x1001)
#define XTEST_AES_256_KEY_ID_0   (0x1002)
#define XTEST_AES_KEY_ID_0       (0x1010)
#define XTEST_AES_KEY_ID_1       (0x1011)
#define XTEST_EC_PUB_KEY_ID_0    (0x2000)
#define XTEST_EC_PUB_KEY_ID_1    (0x2001)
#define XTEST_EC_PUB_KEY_ID_2    (0x2002)
#define XTEST_EC_PUB_KEY_ID_3    (0x2003)
#define XTEST_EC_PUB_KEY_ID_4    (0x2004)
#define XTEST_EC_PRIV_KEY_ID_0   (0x3000)
#define XTEST_EC_PRIV_KEY_ID_1   (0x3001)
#define XTEST_EC_PRIV_KEY_ID_2   (0x3002)
#define XTEST_EC_PRIV_KEY_ID_3   (0x3003)
#define XTEST_EC_PRIV_KEY_ID_4   (0x3004)
#define XTEST_HMAC_256_KEY_ID_0  (0x4001)
#define XTEST_HMAC_384_KEY_ID_0  (0x4002)
#define XTEST_CMAC_128_KEY_ID_0  (0x5001)
#define XTEST_CMAC_192_KEY_ID_0  (0x5002)
#define XTEST_CMAC_256_KEY_ID_0  (0x5003)

/* OP-TEE TEE client API (built by optee_client) */
#include <tee_client_api.h>

#if 0
/* To the the UUID (found the the TA's h-file(s)) */
#define PTA_MBOX_UUID	{0x598e9628, 0xf233, 0x4c83, {0x8d, 0x4c, 0xbd, 0x2f, 0x0a, 0xd4, 0xb2, 0xe4}}

static int MyGptTest(int cmd)
{
	printf("\x1b[1;31m  ==> MyGptTest send 0x%x  \033[0m \n", cmd);

	TEEC_Result res;
	TEEC_Context ctx;
	TEEC_Session sess;
	TEEC_Operation op;
	TEEC_UUID uuid = PTA_MBOX_UUID;
	uint32_t err_origin;

	/* Initialize a context connecting us to the TEE */
	res = TEEC_InitializeContext(NULL, &ctx);
	if (res != TEEC_SUCCESS)
		printf("TEEC_InitializeContext failed with code 0x%x\n", res);

	/*
	 * Open a session to the "hello world" TA, the TA will print "hello
	 * world!" in the log when the session is created.
	 */
	res = TEEC_OpenSession(&ctx, &sess, &uuid,
				   TEEC_LOGIN_PUBLIC, NULL, NULL, &err_origin);
	if (res != TEEC_SUCCESS)
		printf("TEEC_Opensession failed with code 0x%x origin 0x%x\n",
			res, err_origin);

	/*
	 * Execute a function in the TA by invoking it, in this case
	 * we're incrementing a number.
	 *
	 * The value of command ID part and how the parameters are
	 * interpreted is part of the interface provided by the TA.
	 */

	/* Clear the TEEC_Operation struct */
	memset(&op, 0, sizeof(op));

	/*
	 * Prepare the argument. Pass a value in the first parameter,
	 * the remaining three parameters are unused.
	 */
	op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_INPUT, TEEC_NONE,
					 TEEC_NONE, TEEC_NONE);
//	op.params[0].value.a = aes_IV;
//	op.params[0].value.b = (uint32_t)16;
	op.params[0].tmpref.buffer = aes_IV;
	op.params[0].tmpref.size   = 16;

	/*
	 * PTA_CMD_GPT_INIT is the actual function in the TA to be
	 * called.
	 */
//	printf("Invoking TA to enable gpt %d\n", op.params[0].value.a);
	res = TEEC_InvokeCommand(&sess, cmd, &op,
				 &err_origin);
	if (res != TEEC_SUCCESS)
		printf("TEEC_InvokeCommand failed with code 0x%x origin 0x%x\n",
			res, err_origin);
	// printf("TA incremented value to %d\n", op.params[0].value.a);
//	printf("Waited %d seconds for gpt interrupt.\n", op.params[0].value.b);
	/*
	 * We're done with the TA, close the session and
	 * destroy the context.
	 *
	 * The TA will print "Goodbye!" in the log when the
	 * session is closed.
	 */

	TEEC_CloseSession(&sess);

	TEEC_FinalizeContext(&ctx);

	return 0;
}


void *child(void *arg);
void* child(void *arg)
{
	int *_ptr = (int *)arg;
	for(int i = 0;i < 10;++i)
	{
	Do_ADBG_Log("\x1b[1;32m  Slave: 0x%x  \033[0m", *_ptr);
	}

	MyGptTest(0x78);

	pthread_exit(NULL);
}

#endif
#define SKS_DEBUG_PRINT (1)
void __SHOW_VAL(const char *c, uint8_t *ptr, uint32_t len);
void __SHOW_VAL(const char *c, uint8_t *ptr, uint32_t len)
{
#if SKS_DEBUG_PRINT

    uint8_t line[8] = {0};
    printf(" --- %s ( %d ) bytes ---\r\n", c, len);
    uint32_t j = 0;
    for (j = 0; j < len; j++)
    {
        line[j % 8] = *(ptr + j);
        if ((j % 8) == 7)
            printf(" %02X %02X %02X %02X - %02X %02X %02X %02X  \r\n"
                 , line[0], line[1], line[2], line[3]
                 , line[4], line[5], line[6], line[7]);
    }
    if ((j % 8) != 0)
    {
        switch (j % 8)
        {
        case 1: printf(" %02X -- -- -- - -- -- -- --  \r\n", line[0]); break;
        case 2: printf(" %02X %02X -- -- - -- -- -- --  \r\n", line[0], line[1]); break;
        case 3: printf(" %02X %02X %02X -- - -- -- -- --  \r\n", line[0], line[1], line[2]); break;
        case 4: printf(" %02X %02X %02X %02X - -- -- -- --  \r\n", line[0], line[1], line[2], line[3]); break;
        case 5: printf(" %02X %02X %02X %02X - %02X -- -- --  \r\n", line[0], line[1], line[2], line[3], line[4]); break;
        case 6: printf(" %02X %02X %02X %02X - %02X %02X -- --  \r\n", line[0], line[1], line[2], line[3], line[4], line[5]); break;
        case 7: printf(" %02X %02X %02X %02X - %02X %02X %02X --  \r\n", line[0], line[1], line[2], line[3], line[4], line[5], line[6]); break;
	default: break;
        }
    }
    printf(" --------------------\r\n");

#endif
}

//static bool __CHECK_VAL(const char *c, uint8_t *pTarget, uint8_t *pSrc, uint32_t len)
//{
//	__SHOW_VAL("data to check", pSrc, len);
//	if (memcmp(pTarget, pSrc, len) != 0)
//	{
//		Do_ADBG_Log(" %s compare Fail!!  \n", c);
//		return false;
//	}
//
//	Do_ADBG_Log(" AES %s compare PASS!!  \n", c);
//	return true;
//}

static CK_RV generate_symmetric_key(CK_SESSION_HANDLE session,
                                 CK_ULONG key_length,
                                 CK_ULONG key_id,
                                 CK_OBJECT_HANDLE_PTR key)
{
	CK_RV rv;
	CK_MECHANISM mech = {CKM_MTK_HSM_AES_KEY_GEN, NULL, 0};
	CK_KEY_TYPE keyType_aes = CKK_AES;
	CK_BBOOL true_value = CK_TRUE;
	CK_ULONG key_name = key_id;
	// CK_BBOOL false_value = CK_FALSE;

	CK_ATTRIBUTE attr[] = {
		{CKA_SENSITIVE, &true_value,           sizeof(CK_BBOOL)},
		{CKA_TOKEN,     &true_value,           sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType_aes,          sizeof(keyType_aes)},
		{CKA_ENCRYPT,   &true_value,           sizeof(CK_BBOOL)},
		{CKA_DECRYPT,   &true_value,           sizeof(CK_BBOOL)},
		{CKA_VALUE_LEN, &key_length,           sizeof(CK_ULONG)},
		{CKA_ID,        &key_name,             sizeof(key_name)}
	};

	rv = C_GenerateKey(session, &mech, attr, sizeof(attr) / sizeof(CK_ATTRIBUTE), key);

	return rv;
}

static CK_RV generate_cmac_symmetric_key(CK_SESSION_HANDLE session,
                                 CK_ULONG key_length,
                                 CK_ULONG key_id,
                                 CK_OBJECT_HANDLE_PTR key)
{
	CK_RV rv;
	CK_MECHANISM mech = {CKM_MTK_HSM_AES_KEY_GEN, NULL, 0};
	CK_KEY_TYPE keyType_aes = CKK_AES;
	CK_BBOOL true_value = CK_TRUE;
	CK_ULONG key_name = key_id;
	// CK_BBOOL false_value = CK_FALSE;

	CK_ATTRIBUTE attr[] = {
		{CKA_SENSITIVE, &true_value,           sizeof(CK_BBOOL)},
		{CKA_TOKEN,     &true_value,           sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType_aes,          sizeof(keyType_aes)},
		{CKA_SIGN,   	&true_value,           sizeof(CK_BBOOL)},
		{CKA_VERIFY,    &true_value,           sizeof(CK_BBOOL)},
		{CKA_VALUE_LEN, &key_length,           sizeof(CK_ULONG)},
		{CKA_ID,        &key_name,             sizeof(key_name)}
	};

	rv = C_GenerateKey(session, &mech, attr, sizeof(attr) / sizeof(CK_ATTRIBUTE), key);

	return rv;
}

static CK_RV generate_hmac_symmetric_key(CK_SESSION_HANDLE session,
                                 CK_ULONG key_length,
                                 CK_ULONG key_id,
                                 CK_OBJECT_HANDLE_PTR key)
{
	CK_RV rv;
	CK_MECHANISM mech = {CKM_MTK_HSM_AES_KEY_GEN, NULL, 0};
	CK_KEY_TYPE keyType_aes = CKK_SHA256_HMAC;
	CK_BBOOL true_value = CK_TRUE;
	CK_ULONG key_name = key_id;

	Do_ADBG_Log("  Gen key len(byte): %ld  \n", key_length);
	switch (key_length)
	{
	case 48: // 384 bits
		keyType_aes = CKK_SHA384_HMAC;
		break;

	case 32: // 256 bits
	default:
		keyType_aes = CKK_SHA256_HMAC;
		break;
	}


	CK_ATTRIBUTE attr[] = {
		{CKA_SENSITIVE, &true_value,           sizeof(CK_BBOOL)},
		{CKA_TOKEN,     &true_value,           sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType_aes,          sizeof(keyType_aes)},
		{CKA_SIGN,   	&true_value,           sizeof(CK_BBOOL)},
		{CKA_VERIFY,    &true_value,           sizeof(CK_BBOOL)},
		{CKA_VALUE_LEN, &key_length,           sizeof(CK_ULONG)},
		{CKA_ID,        &key_name,             sizeof(key_name)}
	};

	rv = C_GenerateKey(session, &mech, attr, sizeof(attr) / sizeof(CK_ATTRIBUTE), key);

	return rv;
}

static CK_RV find_symmetric_key(CK_SESSION_HANDLE session,
                                CK_OBJECT_HANDLE_PTR key,
                                CK_ULONG key_id)
{
	CK_RV rv;
	CK_ULONG count = 0;
	CK_OBJECT_HANDLE *found_objects = NULL;
	CK_ULONG key_name = key_id;

	Do_ADBG_Log("[%s][%d] key_id: %ld\n", __FUNCTION__, __LINE__, key_id);
	CK_ATTRIBUTE attr[] = {
		{CKA_ID, &key_name, sizeof(key_name)}
	};

	rv = find_by_attr(session, attr, 1, &count, &found_objects);
	if (rv != CKR_OK) {
		Do_ADBG_Log("[find key fail.0]");
		return rv;
	}
	if (count == 0) {
		Do_ADBG_Log("[find key fail.1]");
		return -1;
	}
	Do_ADBG_Log("[find key pass.]");

	*key = found_objects[0];
	Do_ADBG_Log("[%s][%d] *key : %ld\n", __FUNCTION__, __LINE__, *key);
	Do_ADBG_Log("[%s][%d] count : %ld\n", __FUNCTION__, __LINE__, count);
	return CKR_OK;

}

CK_RV _AES_one_shot(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _AES_one_shot(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_AES_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_AES_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_AES_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_symmetric_key(session, key_length, aes_id, &key);
		// rv = C_GenerateKey(session, &mech, template, sizeof(template) / sizeof(CK_ATTRIBUTE), &key);
		// // rv = C_GenerateKey(session, &mech, NULL, 0, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	unsigned char cipher[BUFFERSIZ] = {0};
	unsigned char plain[BUFFERSIZ]  = {0};
	CK_ULONG cipher_len = BUFFERSIZ;


	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	CK_AES_CTR_PARAMS mtktest_aes_ctr_params;
	const char *mech_type = "CKM_UNKNOWN_TYPE";

	if ((mechanism.mechanism == CKM_AES_ECB) || (mechanism.mechanism == CKM_MTK_HSM_AES_ECB) )
	{
		mech_type = "CKM_AES_ECB";
	}
	else if ((mechanism.mechanism == CKM_AES_CBC) || (mechanism.mechanism == CKM_MTK_HSM_AES_CBC) )
	{
		mech_type = "CKM_AES_CBC";
		mechanism.pParameter = aes_IV;
		mechanism.ulParameterLen = 16;
	}
	else if ((mechanism.mechanism == CKM_AES_CTR) || (mechanism.mechanism == CKM_MTK_HSM_AES_CTR) )
	{
		mech_type = "CKM_AES_CTR";
		memcpy(mtktest_aes_ctr_params.cb, aes_CTR, 16);

		mechanism.pParameter = (CK_BYTE_PTR)&mtktest_aes_ctr_params;
		mechanism.ulParameterLen = sizeof(mtktest_aes_ctr_params);
	}

	cipher_len = sizeof(aes_msg);

	Do_ADBG_Log("  C_EncryptInit \n");
	/* Initialize the encryption operation in the session */
	rv = C_EncryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_EncryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	// Encrypt one-shot
	Do_ADBG_Log("  C_Encrypt  \n");
	/* Do the encryption */
	rv = C_Encrypt(session, (CK_BYTE_PTR)aes_msg, sizeof(aes_msg),
				   (CK_BYTE_PTR)cipher, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES one-shot cipher", cipher, cipher_len);

	Do_ADBG_Log("  C_DecryptInit  \n");
	/* Decryption */
	rv = C_DecryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_DecryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	/* Do the decryption */
	rv = C_Decrypt(session, (CK_BYTE_PTR)cipher, cipher_len,
				   (CK_BYTE_PTR)plain, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Decrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES one-shot plain", plain, cipher_len);

	if (memcmp(aes_msg, plain, sizeof(aes_msg)) != 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("AES Decrypt fail!!");
		goto out;
	}
	Do_ADBG_Log("  [%s %d one-shot Encrypt -> Decrypt pass.]  \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}

CK_RV _AES_stream_mode(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _AES_stream_mode(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_AES_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_AES_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_AES_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_symmetric_key(session, key_length, aes_id, &key);
		// rv = C_GenerateKey(session, &mech, template, sizeof(template) / sizeof(CK_ATTRIBUTE), &key);
		// // rv = C_GenerateKey(session, &mech, NULL, 0, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	unsigned char cipher0[sizeof(aes_msg0)];
	unsigned char cipher1[sizeof(aes_msg1)];
	unsigned char cipher2[sizeof(aes_msg2)];
	unsigned char cipher3[sizeof(aes_msg3)];
	unsigned char plain0[sizeof(aes_msg0)];
	unsigned char plain1[sizeof(aes_msg1)];
	unsigned char plain2[sizeof(aes_msg2)];
	unsigned char plain3[sizeof(aes_msg3)];

	memset(cipher0, 0, sizeof(cipher0));
	memset(cipher1, 0, sizeof(cipher1));
	memset(cipher2, 0, sizeof(cipher2));
	memset(cipher3, 0, sizeof(cipher3));
	memset(plain0, 0, sizeof(plain0));
	memset(plain1, 0, sizeof(plain1));
	memset(plain2, 0, sizeof(plain2));
	memset(plain3, 0, sizeof(plain3));

	CK_ULONG cipher_len = BUFFERSIZ;

	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	CK_AES_CTR_PARAMS mtktest_aes_ctr_params;
	const char *mech_type = "CKM_UNKNOWN_TYPE";

	if ((mechanism.mechanism == CKM_AES_ECB) || (mechanism.mechanism == CKM_MTK_HSM_AES_ECB) )
	{
		mech_type = "CKM_AES_ECB";
	}
	else if ((mechanism.mechanism == CKM_AES_CBC) || (mechanism.mechanism == CKM_MTK_HSM_AES_CBC) )
	{
		mech_type = "CKM_AES_CBC";
		mechanism.pParameter = aes_IV;
		mechanism.ulParameterLen = 16;
	}
	else if ((mechanism.mechanism == CKM_AES_CTR) || (mechanism.mechanism == CKM_MTK_HSM_AES_CTR) )
	{
		mech_type = "CKM_AES_CTR";
		memcpy(mtktest_aes_ctr_params.cb, aes_CTR, 16);

		mechanism.pParameter = (CK_BYTE_PTR)&mtktest_aes_ctr_params;
		mechanism.ulParameterLen = sizeof(mtktest_aes_ctr_params);
	}

	cipher_len = sizeof(aes_msg);

	Do_ADBG_Log("  C_EncryptInit  \n");
	/* Initialize the encryption operation in the session */
	rv = C_EncryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_EncryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	// Encrypt update
	Do_ADBG_Log("  C_EncryptUpdate  \n");
	/* Do the encryption */
	cipher_len = sizeof(aes_msg0);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg0, sizeof(aes_msg0),
						 (CK_BYTE_PTR)cipher0, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream cipher0", cipher0, cipher_len);

	cipher_len = sizeof(aes_msg1);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg1, sizeof(aes_msg1),
						 (CK_BYTE_PTR)cipher1, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream cipher1", cipher1, cipher_len);

	cipher_len = sizeof(aes_msg2);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg2, sizeof(aes_msg2),
						 (CK_BYTE_PTR)cipher2, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream cipher2", cipher2, cipher_len);

	cipher_len = sizeof(aes_msg3);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg3, sizeof(aes_msg3),
						 (CK_BYTE_PTR)cipher3, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream cipher3", cipher3, cipher_len);

	rv = C_EncryptFinal(session, (CK_BYTE_PTR)cipher3, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptFinal(): %lu", mech_type, rv);
		goto out;
	}

	Do_ADBG_Log("  C_DecryptInit  \n");
	/* Decryption */
	rv = C_DecryptInit(session, &mechanism, key);

	// Decrypt update
	Do_ADBG_Log("  C_DecryptUpdate  \n");

	/* Do the decryption */
	cipher_len = sizeof(aes_msg0);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher0, sizeof(aes_msg0),
						 (CK_BYTE_PTR)plain0, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Decrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream plain0", plain0, cipher_len);

	cipher_len = sizeof(aes_msg1);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher1, sizeof(aes_msg1),
						 (CK_BYTE_PTR)plain1, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Decrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream plain1", plain1, cipher_len);

	cipher_len = sizeof(aes_msg2);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher2, sizeof(aes_msg2),
						 (CK_BYTE_PTR)plain2, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Decrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream plain2", plain2, cipher_len);

	cipher_len = sizeof(aes_msg3);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher3, sizeof(aes_msg3),
						 (CK_BYTE_PTR)plain3, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Decrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("AES stream plain3", plain3, cipher_len);

	rv = C_DecryptFinal(session, (CK_BYTE_PTR)cipher3, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptFinal(): %lu", mech_type, rv);
		goto out;
	}

	if (memcmp(aes_msg0, plain0, sizeof(aes_msg0)) != 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("AES Decrypt aes_msg0 fail!!");
		goto out;
	}
	if (memcmp(aes_msg1, plain1, sizeof(aes_msg1)) != 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("AES Decrypt aes_msg1 fail!!");
		goto out;
	}
	if (memcmp(aes_msg2, plain2, sizeof(aes_msg2)) != 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("AES Decrypt aes_msg2 fail!!");
		goto out;
	}
	if (memcmp(aes_msg3, plain3, sizeof(aes_msg3)) != 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("AES Decrypt aes_msg3 fail!!");
		goto out;
	}
	Do_ADBG_Log("  [%s %d stream Encrypt -> Decrypt pass.]  \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}

static unsigned char checksum (unsigned char *ptr, size_t sz)
{
    unsigned char chk = 0;
    while (sz-- != 0)
        chk -= *ptr++;
    return chk;
}

#define LEN_1M (1048576) // 1M
CK_RV _AES_stream_mode_1M_0(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _AES_stream_mode_1M_0(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_AES_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_AES_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_AES_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_symmetric_key(session, key_length, aes_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	uint8_t aes_enc_buf[LEN_1M] = {0};
	uint8_t aes_dec_buf[LEN_1M] = {0};
	memset(aes_dec_buf, 0xFF, LEN_1M);
	CK_ULONG cipher_len = BUFFERSIZ;

	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	CK_AES_CTR_PARAMS mtktest_aes_ctr_params;
	const char *mech_type = "CKM_UNKNOWN_TYPE";

	if ((mechanism.mechanism == CKM_AES_ECB) || (mechanism.mechanism == CKM_MTK_HSM_AES_ECB) )
	{
		mech_type = "CKM_AES_ECB";
	}
	else if ((mechanism.mechanism == CKM_AES_CBC) || (mechanism.mechanism == CKM_MTK_HSM_AES_CBC) )
	{
		mech_type = "CKM_AES_CBC";
		mechanism.pParameter = aes_IV;
		mechanism.ulParameterLen = 16;
	}
	else if ((mechanism.mechanism == CKM_AES_CTR) || (mechanism.mechanism == CKM_MTK_HSM_AES_CTR) )
	{
		mech_type = "CKM_AES_CTR";
		memcpy(mtktest_aes_ctr_params.cb, aes_CTR, 16);

		mechanism.pParameter = (CK_BYTE_PTR)&mtktest_aes_ctr_params;
		mechanism.ulParameterLen = sizeof(mtktest_aes_ctr_params);
	}

	Do_ADBG_Log("  C_EncryptInit  \n");
	/* Initialize the encryption operation in the session */
	rv = C_EncryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_EncryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	// Encrypt update
	Do_ADBG_Log("  C_EncryptUpdate  \n");
	/* Do the encryption */
	cipher_len = sizeof(origin_8K_0);
	for (int i = 0; i < 128; i++) // 8K * 128 = 1M
		rv = C_EncryptUpdate(session, (CK_BYTE_PTR)origin_8K_0, sizeof(origin_8K_0),
						(CK_BYTE_PTR)(&aes_enc_buf[i * sizeof(origin_8K_0)]), &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	rv = C_EncryptFinal(session, (CK_BYTE_PTR)aes_enc_buf, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptFinal(): %lu", mech_type, rv);
		goto out;
	}

	Do_ADBG_Log("  C_DecryptInit  \n");
	/* Decryption */
	rv = C_DecryptInit(session, &mechanism, key);

	// Decrypt update
	Do_ADBG_Log("  C_DecryptUpdate \n");

	/* Do the decryption */
	for (int i = 0; i < 128; i++) // 8K * 128 = 1M
		rv = C_DecryptUpdate(session, (CK_BYTE_PTR)(&aes_enc_buf[i * sizeof(origin_8K_0)]), sizeof(origin_8K_0),
						 (CK_BYTE_PTR)(&aes_dec_buf[i * sizeof(origin_8K_0)]), &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	rv = C_DecryptFinal(session, (CK_BYTE_PTR)aes_dec_buf, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptFinal(): %lu", mech_type, rv);
		goto out;
	}

	if (0 != checksum(aes_dec_buf, cipher_len)) {
		Do_ADBG_Log("AES Decrypt fail!!");
		goto out;
	}
	Do_ADBG_Log("  [%s %d stream 1M_0 Encrypt -> Decrypt pass.]  \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}


static void xtest_hsm_pkcs_11_test_1008(ADBG_Case_t *c)
{
//	pthread_t t;
//	int _no = 0x11;
//	for(int i = 0;i < 10;++i)
//	{
//		Do_ADBG_Log("  Master: %d  ", i);
//	}
//	pthread_create(&t, NULL, child, &_no);
//	pthread_join(t, NULL);

//	MyGptTest(0x56);

	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 1008 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_ECB, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_ECB, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_CBC, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_CBC, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_CTR, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_CTR, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1008, xtest_hsm_pkcs_11_test_1008,
		"PKCS11: AES ECB/CBC/CTR 256 encrypt + decrypt");

static void xtest_hsm_pkcs_11_test_1009(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 1009 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_ECB, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_ECB, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_CBC, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_CBC, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_CTR, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_CTR, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1009, xtest_hsm_pkcs_11_test_1009,
		"PKCS11: AES ECB/CBC/CTR 128 encrypt + decrypt");

static void xtest_hsm_pkcs_11_test_1010(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 1010 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_ECB, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_ECB, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_CBC, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_CBC, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_one_shot(CKM_MTK_HSM_AES_CTR, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode(CKM_MTK_HSM_AES_CTR, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1010, xtest_hsm_pkcs_11_test_1010,
		"PKCS11: AES ECB/CBC/CTR 192 encrypt + decrypt");

static void xtest_hsm_pkcs_11_test_1013(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 1013 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_ECB, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_CBC, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_CTR, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_ECB, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_CBC, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_CTR, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_ECB, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_CBC, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_stream_mode_1M_0(CKM_MTK_HSM_AES_CTR, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1013, xtest_hsm_pkcs_11_test_1013,
		"PKCS11: AES ECB/CBC/CTR encrypt/decrypt 1M_0 stream mode");

// 256
static uint8_t aes_gcm_iv_256[12] = {
    0X95, 0X5B, 0X58, 0X97, 0XF6, 0XB9, 0X80, 0X6B, 0XBE, 0XC5, 0XC3, 0X3E,
};
static uint8_t aes256_gcm_aad[16] = {
    0X33, 0XE6, 0X18, 0XEC, 0XBB, 0XE5, 0XEB, 0X05, 0X66, 0XDF, 0X21, 0XC3, 0XC3, 0X4B, 0X7E, 0X25,
};
static uint8_t aes256_gcm_tag[16] = {
    0X89, 0X45, 0X46, 0XAA, 0X7E, 0XDB, 0X92, 0X51, 0X17, 0X25, 0XFA, 0XE9, 0XA1, 0XC3, 0XB6, 0X56,
};

CK_RV _AES_GCM_one_shot(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _AES_GCM_one_shot(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_CMAC_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_CMAC_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_CMAC_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_symmetric_key(session, key_length, aes_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	unsigned char cipher[BUFFERSIZ] = {0};
	unsigned char plain[BUFFERSIZ]  = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	CK_GCM_PARAMS mtktest_aes_gcm_params;
	const char *mech_type = "CKM_UNKNOWN_TYPE";

	if ((mechanism.mechanism == CKM_AES_GCM) || (mechanism.mechanism == CKM_MTK_HSM_AES_GCM) )
	{
		mech_type = "CKM_AES_GCM";
		mtktest_aes_gcm_params.pIv 	 = (CK_BYTE_PTR)aes_gcm_iv_256;
		mtktest_aes_gcm_params.ulIvLen	 = sizeof(aes_gcm_iv_256);
//		mtktest_aes_gcm_params.ulIvBits  = xx;
		mtktest_aes_gcm_params.pAAD	 = (CK_BYTE_PTR)aes256_gcm_aad;
		mtktest_aes_gcm_params.ulAADLen  = sizeof(aes256_gcm_aad);
		mtktest_aes_gcm_params.ulTagBits = sizeof(aes256_gcm_tag) * 8;

		mechanism.pParameter = (CK_BYTE_PTR)&mtktest_aes_gcm_params;
		mechanism.ulParameterLen = sizeof(mtktest_aes_gcm_params);
	}

	Do_ADBG_Log("  C_EncryptInit  \n");
	/* Initialize the encryption operation in the session */
	rv = C_EncryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_EncryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	// Encrypt
	Do_ADBG_Log("  C_Encrypt \n");
	/* Do the encryption */
	cipher_len = sizeof(aes_msg) + sizeof(aes256_gcm_tag);
	rv = C_Encrypt(session, (CK_BYTE_PTR)aes_msg, sizeof(aes_msg),
			(CK_BYTE_PTR)cipher, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("GCM output", cipher, cipher_len);

	/* Decryption */
	Do_ADBG_Log("  C_DecryptInit  \n");
	rv = C_DecryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_DecryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	// Decrypt
	Do_ADBG_Log("  C_Decrypt  \n");
	/* Do the decryption */
	rv = C_Decrypt(session, (CK_BYTE_PTR)cipher, sizeof(aes_msg),
			(CK_BYTE_PTR)plain, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Decrypt(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("GCM output", plain, cipher_len);

	if (memcmp(aes_msg, plain, sizeof(aes_msg)) != 0) {
		Do_ADBG_Log("AES Decrypt fail!!");
		goto out;
	}

	if (memcmp(&cipher[sizeof(aes_msg)], &plain[sizeof(aes_msg)], sizeof(aes256_gcm_tag)) != 0) {
		Do_ADBG_Log("AES MAC gen fail!!");
		goto out;
	}

	switch (key_len)
	{
	case 128:
		if (memcmp(aes_gcm_128_mac, &plain[sizeof(aes_msg)], sizeof(aes256_gcm_tag)) != 0) {
			rv = CKR_GENERAL_ERROR;
			Do_ADBG_Log("AES GCM-128 MAC comparison fail!!");
			goto out;
		}
		break;

	case 192:
		if (memcmp(aes_gcm_192_mac, &plain[sizeof(aes_msg)], sizeof(aes256_gcm_tag)) != 0) {
			rv = CKR_GENERAL_ERROR;
			Do_ADBG_Log("AES GCM-192 MAC comparison fail!!");
			goto out;
		}
		break;

	default:
	case 256:
		if (memcmp(aes_gcm_256_mac, &plain[sizeof(aes_msg)], sizeof(aes256_gcm_tag)) != 0) {
			rv = CKR_GENERAL_ERROR;
			Do_ADBG_Log("AES GCM-256 MAC comparison fail!!");
			goto out;
		}
		break;
	}
	Do_ADBG_Log("  [%s %d one-shot Encrypt -> Decrypt & MAC pass.]  \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}

CK_RV _AES_GCM_stream_mode(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _AES_GCM_stream_mode(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_CMAC_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_CMAC_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_CMAC_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_symmetric_key(session, key_length, aes_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	unsigned char cipher0[sizeof(aes_msg0)];
	unsigned char cipher1[sizeof(aes_msg1)];
	unsigned char cipher2[sizeof(aes_msg2)];
	unsigned char cipher3[sizeof(aes_msg3)];
	unsigned char aes_gcm_tag[16];
	unsigned char plain0[sizeof(aes_msg0)];
	unsigned char plain1[sizeof(aes_msg1)];
	unsigned char plain2[sizeof(aes_msg2)];
	unsigned char plain3[sizeof(aes_msg3)];
	unsigned char aes_gcm_tag_veri[16];

	memset(cipher0, 0, sizeof(cipher0));
	memset(cipher1, 0, sizeof(cipher1));
	memset(cipher2, 0, sizeof(cipher2));
	memset(cipher3, 0, sizeof(cipher3));
	memset(aes_gcm_tag, 0, sizeof(aes_gcm_tag));
	memset(plain0, 0, sizeof(plain0));
	memset(plain1, 0, sizeof(plain1));
	memset(plain2, 0, sizeof(plain2));
	memset(plain3, 0, sizeof(plain3));
	memset(aes_gcm_tag_veri, 0, sizeof(aes_gcm_tag_veri));

	CK_ULONG cipher_len = BUFFERSIZ;


	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	CK_GCM_PARAMS mtktest_aes_gcm_params;
	const char *mech_type = "CKM_UNKNOWN_TYPE";

	if ((mechanism.mechanism == CKM_AES_GCM) || (mechanism.mechanism == CKM_MTK_HSM_AES_GCM) )
	{
		mech_type = "CKM_AES_GCM";
		mtktest_aes_gcm_params.pIv 	 = (CK_BYTE_PTR)aes_gcm_iv_256;
		mtktest_aes_gcm_params.ulIvLen	 = sizeof(aes_gcm_iv_256);
//		mtktest_aes_gcm_params.ulIvBits  = xx;
		mtktest_aes_gcm_params.pAAD	 = (CK_BYTE_PTR)aes256_gcm_aad;
		mtktest_aes_gcm_params.ulAADLen  = sizeof(aes256_gcm_aad);
		mtktest_aes_gcm_params.ulTagBits = sizeof(aes256_gcm_tag) * 8;

		mechanism.pParameter = (CK_BYTE_PTR)&mtktest_aes_gcm_params;
		mechanism.ulParameterLen = sizeof(mtktest_aes_gcm_params);
	}

	Do_ADBG_Log("  C_EncryptInit  \n");
	/* Initialize the encryption operation in the session */
	rv = C_EncryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_EncryptInit: rv = 0x%lX\n", rv);
		goto out;
	}


	// Encrypt update
	Do_ADBG_Log("  C_EncryptUpdate  \n");
	/* Do the encryption */
	cipher_len = sizeof(aes_msg0);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg0, sizeof(aes_msg0),
						 (CK_BYTE_PTR)cipher0, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_msg1);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg1, sizeof(aes_msg1),
						 (CK_BYTE_PTR)cipher1, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_msg2);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg2, sizeof(aes_msg2),
						 (CK_BYTE_PTR)cipher2, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_msg3);
	rv = C_EncryptUpdate(session, (CK_BYTE_PTR)aes_msg3, sizeof(aes_msg3),
						 (CK_BYTE_PTR)cipher3, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_gcm_tag);
	rv = C_EncryptFinal(session, (CK_BYTE_PTR)aes_gcm_tag, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : EncryptFinal(): %lu", mech_type, rv);
		goto out;
	}

	/* Decryption */
	rv = C_DecryptInit(session, &mechanism, key);

	// Decrypt update
	Do_ADBG_Log("  C_DecryptUpdate  \n");
	/* Do the decryption */
	cipher_len = sizeof(aes_msg0);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher0, sizeof(aes_msg0),
						 (CK_BYTE_PTR)plain0, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_msg1);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher1, sizeof(aes_msg1),
						 (CK_BYTE_PTR)plain1, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_msg2);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher2, sizeof(aes_msg2),
						 (CK_BYTE_PTR)plain2, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_msg3);
	rv = C_DecryptUpdate(session, (CK_BYTE_PTR)cipher3, sizeof(aes_msg3),
						 (CK_BYTE_PTR)plain3, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptUpdate(): %lu", mech_type, rv);
		goto out;
	}

	cipher_len = sizeof(aes_gcm_tag);
	rv = C_DecryptFinal(session, (CK_BYTE_PTR)aes_gcm_tag_veri, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : DecryptFinal(): %lu", mech_type, rv);
		goto out;
	}

	if (memcmp(aes_msg0, plain0, sizeof(aes_msg0)) != 0) {
		Do_ADBG_Log("AES Decrypt aes_msg0 fail!!");
		goto out;
	}
	if (memcmp(aes_msg1, plain1, sizeof(aes_msg1)) != 0) {
		Do_ADBG_Log("AES Decrypt aes_msg1 fail!!");
		goto out;
	}
	if (memcmp(aes_msg2, plain2, sizeof(aes_msg2)) != 0) {
		Do_ADBG_Log("AES Decrypt aes_msg2 fail!!");
		goto out;
	}
	if (memcmp(aes_msg3, plain3, sizeof(aes_msg3)) != 0) {
		Do_ADBG_Log("AES Decrypt aes_msg3 fail!!");
		goto out;
	}

	if (memcmp((const void *)aes_gcm_tag_veri, (const void *)aes_gcm_tag, cipher_len) != 0) {
		Do_ADBG_Log("AES MAC fail!!");
		goto out;
	}

	switch (key_len)
	{
	case 128:
		if (memcmp(aes_gcm_128_mac, (const void *)aes_gcm_tag, cipher_len) != 0) {
			rv = CKR_GENERAL_ERROR;
			Do_ADBG_Log("AES GCM-128 MAC comparison fail!!");
			goto out;
		}
		break;

	case 192:
		if (memcmp(aes_gcm_192_mac, (const void *)aes_gcm_tag, cipher_len) != 0) {
			rv = CKR_GENERAL_ERROR;
			Do_ADBG_Log("AES GCM-192 MAC comparison fail!!");
			goto out;
		}
		break;

	default:
	case 256:
		if (memcmp(aes_gcm_256_mac, (const void *)aes_gcm_tag, cipher_len) != 0) {
			rv = CKR_GENERAL_ERROR;
			Do_ADBG_Log("AES GCM-256 MAC comparison fail!!");
			goto out;
		}
		break;
	}

	Do_ADBG_Log("  [%s %d stream Encrypt -> Decrypt pass & MAC pass.]  \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}

static void xtest_hsm_pkcs_11_test_1011(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 1011 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_GCM_one_shot(CKM_MTK_HSM_AES_GCM/*fixed*/, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_GCM_one_shot(CKM_MTK_HSM_AES_GCM/*fixed*/, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_GCM_one_shot(CKM_MTK_HSM_AES_GCM/*fixed*/, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1011, xtest_hsm_pkcs_11_test_1011,
		"PKCS11: AES GCM encrypt/decrypt one-shot mode");

static void xtest_hsm_pkcs_11_test_1012(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 1012 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_GCM_stream_mode(CKM_MTK_HSM_AES_GCM/*fixed*/, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_GCM_stream_mode(CKM_MTK_HSM_AES_GCM/*fixed*/, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _AES_GCM_stream_mode(CKM_MTK_HSM_AES_GCM/*fixed*/, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1012, xtest_hsm_pkcs_11_test_1012,
		"PKCS11: AES GCM encrypt/decrypt stream mode");

static void xtest_hsm_pkcs_11_test_2001(ADBG_Case_t *c)
{
	Do_ADBG_Log("======== 2001 Test ========");

	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG hmac_id = XTEST_HMAC_256_KEY_ID_0;
	CK_ULONG key_length = 256 / 8;
	CK_OBJECT_HANDLE key;

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, hmac_id);
	if (rv != CKR_OK)
	{
		rv = generate_hmac_symmetric_key(session, key_length, hmac_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	uint8_t hmac256_tag[32] = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* HMAC mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = CKM_MTK_HSM_SHA256_HMAC;

	const char *mech_type = "CKM_HSM_SHA256_HMAC";

	cipher_len = sizeof(hmac256_tag);

	Do_ADBG_Log("  C_SignInit  \n");
	rv = C_SignInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_SignInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_Sign  \n");
	rv = C_Sign(session, (CK_BYTE_PTR)sha_origin_text, sizeof(sha_origin_text),
				   (CK_BYTE_PTR)hmac256_tag, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Sign(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("HMAC256", hmac256_tag, cipher_len);
	if (memcmp(hmac256_tag, hmac_256_mac, sizeof(hmac256_tag)) != 0) {
		Do_ADBG_Log("HMAC 256 gen MAC fail!!");
		goto out;
	}

	Do_ADBG_Log("  C_VerifyInit  \n");
	rv = C_VerifyInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_VerifyInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_Verify  \n");
	rv = C_Verify(session, (CK_BYTE_PTR)sha_origin_text, sizeof(sha_origin_text),
				   (CK_BYTE_PTR)hmac256_tag, cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log(" %s : C_Verify(): %lu \n", mech_type, rv);
		goto out;
	}
	Do_ADBG_Log(" [%s %d: C_Verify() one-shot pass] \n", mech_type, (uint32_t)key_length * 8);

out:

	rv2 = C_CloseSession(session);
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	rv2 = close_lib();
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	if (rv != 0)
		ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 2001, xtest_hsm_pkcs_11_test_2001,
		"PKCS11: Check HMAC-256 one-shot gen MAC & verify");

static void xtest_hsm_pkcs_11_test_2002(ADBG_Case_t *c)
{
	Do_ADBG_Log("======== 2002 Test ========");

	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG hmac_id = XTEST_HMAC_384_KEY_ID_0;
	CK_ULONG key_length = 384 / 8;
	CK_OBJECT_HANDLE key;

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, hmac_id);
	if (rv != CKR_OK)
	{
		rv = generate_hmac_symmetric_key(session, key_length, hmac_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	uint8_t hmac384_tag[48] = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* HMAC mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = CKM_MTK_HSM_SHA384_HMAC;

	const char *mech_type = "CKM_HSM_SHA384_HMAC";

	cipher_len = sizeof(hmac384_tag);

	Do_ADBG_Log("  C_SignInit  \n");
	rv = C_SignInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_SignInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_Sign  \n");
	rv = C_Sign(session, (CK_BYTE_PTR)sha_origin_text, sizeof(sha_origin_text),
				   (CK_BYTE_PTR)hmac384_tag, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Sign(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("HMAC384", hmac384_tag, cipher_len);
	if (memcmp(hmac384_tag, hmac_384_mac, sizeof(hmac384_tag)) != 0) {
		Do_ADBG_Log("HMAC 384 gen MAC fail!!");
		goto out;
	}

	Do_ADBG_Log("  C_VerifyInit  \n");
	rv = C_VerifyInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_VerifyInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_Verify  \n");
	rv = C_Verify(session, (CK_BYTE_PTR)sha_origin_text, sizeof(sha_origin_text),
				   (CK_BYTE_PTR)hmac384_tag, cipher_len);
	if (rv != CKR_OK) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log(" %s : C_Verify(): %lu \n", mech_type, rv);
		goto out;
	}
	Do_ADBG_Log(" [%s %d: C_Verify() one-shot pass] \n", mech_type, (uint32_t)key_length * 8);

out:

	rv2 = C_CloseSession(session);
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	rv2 = close_lib();
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	if (rv != 0)
		ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 2002, xtest_hsm_pkcs_11_test_2002,
		"PKCS11: Check HMAC-384 one-shot gen MAC & verify");

static void xtest_hsm_pkcs_11_test_2003(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG hmac_id = XTEST_HMAC_256_KEY_ID_0;
	CK_ULONG key_length = 256 / 8;
	CK_OBJECT_HANDLE key;

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, hmac_id);
	if (rv != CKR_OK)
	{
		rv = generate_hmac_symmetric_key(session, key_length, hmac_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	uint8_t hmac256_tag[32] = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* HMAC mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = CKM_MTK_HSM_SHA256_HMAC;

	const char *mech_type = "CKM_HSM_SHA256_HMAC";

	cipher_len = sizeof(hmac256_tag);

	Do_ADBG_Log("  C_SignInit  \n");
	rv = C_SignInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_SignInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_SignUpdate  \n");
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_1, sizeof(sha_stream_origin_text_1));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_11, sizeof(sha_stream_origin_text_11));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_12, sizeof(sha_stream_origin_text_12));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_13, sizeof(sha_stream_origin_text_13));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_2, sizeof(sha_stream_origin_text_2));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_3, sizeof(sha_stream_origin_text_3));
	rv = C_SignFinal(session, (CK_BYTE_PTR)hmac256_tag, &cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Sign(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("HMAC256", hmac256_tag, cipher_len);
	if (memcmp(hmac256_tag, hmac_256_mac, sizeof(hmac256_tag)) != 0) {
		Do_ADBG_Log("HMAC 256 gen MAC fail!!");
		goto out;
	}

	Do_ADBG_Log("  C_VerifyInit  \n");
	rv = C_VerifyInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2003 Fail] C_VerifyInit: rv = 0x%lX\n", rv);
		goto out;
	}

	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_1, sizeof(sha_stream_origin_text_1));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_11, sizeof(sha_stream_origin_text_11));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_12, sizeof(sha_stream_origin_text_12));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_13, sizeof(sha_stream_origin_text_13));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_2, sizeof(sha_stream_origin_text_2));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_3, sizeof(sha_stream_origin_text_3));
	rv = C_VerifyFinal(session, (CK_BYTE_PTR)hmac256_tag, cipher_len);

	if (rv != CKR_OK) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log(" %s : C_Verify(): %lu \n", mech_type, rv);
		goto out;
	}
	Do_ADBG_Log(" [%s %d: C_Verify() stream pass] \n", mech_type, (uint32_t)key_length * 8);

out:

	rv2 = C_CloseSession(session);
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	rv2 = close_lib();
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	if (rv != 0)
		ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 2003, xtest_hsm_pkcs_11_test_2003,
		"PKCS11: Check HMAC-256 stream mode gen MAC & verify");


static void xtest_hsm_pkcs_11_test_2004(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG hmac_id = XTEST_HMAC_384_KEY_ID_0;
	CK_ULONG key_length = 384 / 8;
	CK_OBJECT_HANDLE key;

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, hmac_id);
	if (rv != CKR_OK)
	{
		rv = generate_hmac_symmetric_key(session, key_length, hmac_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	uint8_t hmac384_tag[48] = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* HMAC mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = CKM_MTK_HSM_SHA384_HMAC;

	const char *mech_type = "CKM_HSM_SHA384_HMAC";

	cipher_len = sizeof(hmac384_tag);

	Do_ADBG_Log("  C_SignInit  \n");
	rv = C_SignInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_SignInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_SignUpdate  \n");
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_1, sizeof(sha_stream_origin_text_1));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_11, sizeof(sha_stream_origin_text_11));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_12, sizeof(sha_stream_origin_text_12));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_13, sizeof(sha_stream_origin_text_13));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_2, sizeof(sha_stream_origin_text_2));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_3, sizeof(sha_stream_origin_text_3));
	rv = C_SignFinal(session, (CK_BYTE_PTR)hmac384_tag, &cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Sign(): %lu", mech_type, rv);
		goto out;
	}
//	__SHOW_VAL("HMAC384", hmac384_tag, cipher_len);
	if (memcmp(hmac384_tag, hmac_384_mac, sizeof(hmac384_tag)) != 0) {
		Do_ADBG_Log("HMAC 384 gen MAC fail!!");
		goto out;
	}

	Do_ADBG_Log("  C_VerifyInit  \n");
	rv = C_VerifyInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2004 Fail] C_VerifyInit: rv = 0x%lX\n", rv);
		goto out;
	}

	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_1, sizeof(sha_stream_origin_text_1));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_11, sizeof(sha_stream_origin_text_11));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_12, sizeof(sha_stream_origin_text_12));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_13, sizeof(sha_stream_origin_text_13));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_2, sizeof(sha_stream_origin_text_2));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_3, sizeof(sha_stream_origin_text_3));
	rv = C_VerifyFinal(session, (CK_BYTE_PTR)hmac384_tag, cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log(" %s : C_Verify(): %lu \n", mech_type, rv);
		goto out;
	}
	Do_ADBG_Log(" [%s %d: C_Verify() stream pass] \n", mech_type, (uint32_t)key_length * 8);

out:

	rv2 = C_CloseSession(session);
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	rv2 = close_lib();
	if (!ADBG_EXPECT_CK_OK(c, rv2))
		goto out;

	if (rv != 0)
		ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 2004, xtest_hsm_pkcs_11_test_2004,
		"PKCS11: Check HMAC-384 stream mode gen MAC & verify");

CK_RV _CMAC_one_shot(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _CMAC_one_shot(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_CMAC_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_CMAC_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_CMAC_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_cmac_symmetric_key(session, key_length, aes_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	unsigned char cmac_tag[16] = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	const char *mech_type = "CKM_AES_CMAC";

	cipher_len = sizeof(cmac_tag);

	Do_ADBG_Log("  C_SignInit  \n");
	rv = C_SignInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_SignInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_Sign  \n");
	rv = C_Sign(session, (CK_BYTE_PTR)aes_msg, sizeof(aes_msg),
				   (CK_BYTE_PTR)cmac_tag, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Sign(): %lu", mech_type, rv);
		goto out;
	}
	__SHOW_VAL("CMAC MAC", cmac_tag, cipher_len);
	switch (key_len)
	{
	case 128:
		if (memcmp(cmac_tag, cmac_128_mac, sizeof(cmac_tag)) != 0) {
			Do_ADBG_Log("CMAC 128 gen MAC fail!!");
			goto out;
		}
		break;

	case 192:
		if (memcmp(cmac_tag, cmac_192_mac, sizeof(cmac_tag)) != 0) {
			Do_ADBG_Log("CMAC 192 gen MAC fail!!");
			goto out;
		}
		break;

	default:
	case 256:
		if (memcmp(cmac_tag, cmac_256_mac, sizeof(cmac_tag)) != 0) {
			Do_ADBG_Log("CMAC 256 gen MAC fail!!");
			goto out;
		}
		break;

	}

	Do_ADBG_Log("  C_VerifyInit  \n");
	rv = C_VerifyInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[3001 Fail] C_VerifyInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_Verify  \n");
	rv = C_Verify(session, (CK_BYTE_PTR)aes_msg, sizeof(aes_msg),
				   (CK_BYTE_PTR)cmac_tag, cipher_len);
	if (rv != CKR_OK) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log(" %s : C_Verify(): %lu \n", mech_type, rv);
		goto out;
	}
	Do_ADBG_Log(" [%s %d: C_Verify() one-shot pass] \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}

CK_RV _CMAC_stream_mode(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _CMAC_stream_mode(CK_MECHANISM_TYPE in_mechanism, uint32_t key_len, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_ULONG key_length = key_len / 8;
	CK_OBJECT_HANDLE key;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	switch (key_len)
	{
	case 128: aes_id = XTEST_AES_128_KEY_ID_0;	break;
	case 192: aes_id = XTEST_AES_192_KEY_ID_0;	break;

	default:
	case 256: aes_id = XTEST_AES_256_KEY_ID_0;	break;
	}

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_cmac_symmetric_key(session, key_length, aes_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	unsigned char cmac_tag[16] = {0};
	CK_ULONG cipher_len = BUFFERSIZ;

	/* AES mechanism */
	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	mechanism.mechanism = in_mechanism;

	const char *mech_type = "CKM_AES_CMAC";

	cipher_len = sizeof(cmac_tag);

	Do_ADBG_Log("  C_SignInit  \n");
	rv = C_SignInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[2001 Fail] C_SignInit: rv = 0x%lX\n", rv);
		goto out;
	}

	Do_ADBG_Log("  C_SignUpdate  \n");
	rv = C_SignUpdate(session, (CK_BYTE_PTR)aes_msg0, sizeof(aes_msg0));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)aes_msg1, sizeof(aes_msg1));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)aes_msg2, sizeof(aes_msg2));
	rv = C_SignUpdate(session, (CK_BYTE_PTR)aes_msg3, sizeof(aes_msg3));
	rv = C_SignFinal(session, (CK_BYTE_PTR)cmac_tag, &cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Sign(): %lu", mech_type, rv);
		goto out;
	}
	__SHOW_VAL("CMAC MAC", cmac_tag, cipher_len);

	Do_ADBG_Log("  C_VerifyInit  \n");
	rv = C_VerifyInit(session, &mechanism, key);

	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)aes_msg0, sizeof(aes_msg0));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)aes_msg1, sizeof(aes_msg1));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)aes_msg2, sizeof(aes_msg2));
	rv = C_VerifyUpdate(session, (CK_BYTE_PTR)aes_msg3, sizeof(aes_msg3));
	rv = C_VerifyFinal(session, (CK_BYTE_PTR)cmac_tag, cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log(" %s : C_Verify(): %lu \n", mech_type, rv);
		goto out;
	}
	Do_ADBG_Log(" %s %d: C_Verify() stream OK \n", mech_type, key_len);

out:
	rv2 = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv2);

	if (rv != 0)
		return rv;
	else
		return rv2;
}

static void xtest_hsm_pkcs_11_test_3001(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 3001 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _CMAC_one_shot(CKM_MTK_HSM_AES_CMAC/*fixed*/, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _CMAC_one_shot(CKM_MTK_HSM_AES_CMAC/*fixed*/, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _CMAC_one_shot(CKM_MTK_HSM_AES_CMAC/*fixed*/, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 3001, xtest_hsm_pkcs_11_test_3001,
		"PKCS11: Check CMAC one-shot gen MAC & verify");

static void xtest_hsm_pkcs_11_test_3002(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 3002 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _CMAC_stream_mode(CKM_MTK_HSM_AES_CMAC/*fixed*/, 128, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _CMAC_stream_mode(CKM_MTK_HSM_AES_CMAC/*fixed*/, 192, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _CMAC_stream_mode(CKM_MTK_HSM_AES_CMAC/*fixed*/, 256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 3002, xtest_hsm_pkcs_11_test_3002,
		"PKCS11: Check CMAC stream mode gen MAC & verify");

CK_RV _SHA_test(CK_MECHANISM_TYPE in_mechanism, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _SHA_test(CK_MECHANISM_TYPE in_mechanism, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv  = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;

	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */

	unsigned char cipher[BUFFERSIZ];
	CK_ULONG cipher_len = BUFFERSIZ;
	const char *mech_type = "CKM_SHA";

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	//------------------------------
	// SHA
	Do_ADBG_Log("  C_DigestInit \n");
	switch(in_mechanism)
	{
		case CKM_MTK_HSM_SHA1:      cipher_len = sizeof(sha1_digest_output);    break;
		case CKM_MTK_HSM_SHA224:    cipher_len = sizeof(sha224_digest_output);  break;
		case CKM_MTK_HSM_SHA384:    cipher_len = sizeof(sha384_digest_output);  break;
		case CKM_MTK_HSM_SHA512:    cipher_len = sizeof(sha512_digest_output);  break;
		case CKM_MTK_HSM_WHP:       cipher_len = sizeof(msg_buf);  break;
		default:
		case CKM_MTK_HSM_SHA256:    cipher_len = sizeof(sha256_digest_output);  break;
	}
	mechanism.mechanism = in_mechanism;

	// SHA one-shot mode
	rv = C_DigestInit(session, &mechanism);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[4001 Fail] C_DigestInit \n");
		goto out;
	}

	Do_ADBG_Log("  C_Digest one-shot \n");
	if (in_mechanism == CKM_MTK_HSM_WHP){
		rv = C_Digest(session, (CK_BYTE_PTR)msg_buf, sizeof(msg_buf), (CK_BYTE_PTR)cipher, &cipher_len);
	}else {
		rv = C_Digest(session, (CK_BYTE_PTR)sha_origin_text, sizeof(sha_origin_text), (CK_BYTE_PTR)cipher, &cipher_len);
	}

	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Digest(): %lu", mech_type, rv);
		goto out;
	}

	__SHOW_VAL("SHA one-shot output", cipher, cipher_len);
	switch(in_mechanism) {

		case CKM_MTK_HSM_SHA1:
			rv = memcmp(sha1_digest_output, cipher, sizeof(sha1_digest_output));
			mech_type = "CKM_MTK_HSM_SHA1";
			break;
		case CKM_MTK_HSM_SHA224:
			rv = memcmp(sha224_digest_output, cipher, sizeof(sha224_digest_output));
			mech_type = "CKM_MTK_HSM_SHA224";
			break;
		case CKM_MTK_HSM_SHA384:
			rv = memcmp(sha384_digest_output, cipher, sizeof(sha384_digest_output));
			mech_type = "CKM_MTK_HSM_SHA384";
			break;
		case CKM_MTK_HSM_SHA512:
			rv = memcmp(sha512_digest_output, cipher, sizeof(sha512_digest_output));
			mech_type = "CKM_MTK_HSM_SHA512";
			break;
		case CKM_MTK_HSM_WHP:
			rv = memcmp(digest_golden, cipher, sizeof(digest_golden));
			mech_type = "CKM_MTK_HSM_WHP";
			break;
		default:
		case CKM_MTK_HSM_SHA256:
			rv = memcmp(sha256_digest_output, cipher, sizeof(sha256_digest_output));
			mech_type = "CKM_MTK_HSM_SHA256";
			break;
	}

	if (0 == rv){
		Do_ADBG_Log("  [%s Digest one-shot comparison OK!!.]  \n", mech_type);
	} else {
		Do_ADBG_Log("  [%s Digest one-shot comparison fail!!.]  \n", mech_type);
		goto out;
	}

	if(in_mechanism == CKM_MTK_HSM_WHP)
		goto out;

	// SHA stream mode
	rv = C_DigestInit(session, &mechanism);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[4001 Fail] C_DigestInit \n");
		goto out;
	}

	Do_ADBG_Log("  C_Digest SHA stream mode \n");
	rv = C_DigestUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_1, sizeof(sha_stream_origin_text_1));
	rv = C_DigestUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_11, sizeof(sha_stream_origin_text_11));
	rv = C_DigestUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_12, sizeof(sha_stream_origin_text_12));
	rv = C_DigestUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_13, sizeof(sha_stream_origin_text_13));
	rv = C_DigestUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_2, sizeof(sha_stream_origin_text_2));
	rv = C_DigestUpdate(session, (CK_BYTE_PTR)sha_stream_origin_text_3, sizeof(sha_stream_origin_text_3));
	rv = C_DigestFinal(session, (CK_BYTE_PTR)cipher, &cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Digest(): %lu", mech_type, rv);
		goto out;
	}

	__SHOW_VAL("SHA stream output", cipher, cipher_len);
	switch(in_mechanism) {

		case CKM_MTK_HSM_SHA1:
			rv = memcmp(sha1_digest_output, cipher, sizeof(sha1_digest_output));
			break;
		case CKM_MTK_HSM_SHA224:
			rv = memcmp(sha224_digest_output, cipher, sizeof(sha224_digest_output));
			break;
		case CKM_MTK_HSM_SHA384:
			rv = memcmp(sha384_digest_output, cipher, sizeof(sha384_digest_output));
			break;
		case CKM_MTK_HSM_SHA512:
			rv = memcmp(sha512_digest_output, cipher, sizeof(sha512_digest_output));
			break;
		default:
		case CKM_MTK_HSM_SHA256:
			rv = memcmp(sha256_digest_output, cipher, sizeof(sha256_digest_output));
			break;
	}

	if (0 == rv){
		Do_ADBG_Log("  [%s Digest stream comparison OK!!.]  \n", mech_type);
	} else {
		Do_ADBG_Log("  [%s Digest stream comparison fail!!.]  \n", mech_type);
		goto out;
	}
out:
	rv2 = C_CloseSession(session);

	if (rv != 0)
		return rv;
	else
		return rv2;

}

static void xtest_hsm_pkcs_11_test_4001(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 4001 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test(CKM_MTK_HSM_SHA1,   slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test(CKM_MTK_HSM_SHA224, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test(CKM_MTK_HSM_SHA256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test(CKM_MTK_HSM_SHA384, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test(CKM_MTK_HSM_SHA512, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 4001, xtest_hsm_pkcs_11_test_4001,
		"PKCS11: SHA 1/224/256/384/512 one-shot & stream mode");

static void xtest_hsm_pkcs_11_test_4003(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 4003 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test(CKM_MTK_HSM_WHP,   slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 4003, xtest_hsm_pkcs_11_test_4003,
		"PKCS11: Whirlpool test");

CK_RV _SHA_test_1M(CK_MECHANISM_TYPE in_mechanism, CK_SLOT_ID slot, ADBG_Case_t *c);
CK_RV _SHA_test_1M(CK_MECHANISM_TYPE in_mechanism, CK_SLOT_ID slot, ADBG_Case_t *c)
{
	CK_RV rv  = CKR_GENERAL_ERROR;
	CK_RV rv2 = CKR_OK;

	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */

	unsigned char cipher[BUFFERSIZ];
	CK_ULONG cipher_len = BUFFERSIZ;
	const char *mech_type = "CKM_SHA";

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	//------------------------------
	// SHA
	Do_ADBG_Log("  C_DigestInit \n");
	switch(in_mechanism)
	{
		case CKM_MTK_HSM_SHA1:      cipher_len = sizeof(sha1_digest_1M_0_output);    break;
		case CKM_MTK_HSM_SHA224:    cipher_len = sizeof(sha224_digest_1M_0_output);  break;
		case CKM_MTK_HSM_SHA384:    cipher_len = sizeof(sha384_digest_1M_0_output);  break;
		case CKM_MTK_HSM_SHA512:    cipher_len = sizeof(sha512_digest_1M_0_output);  break;

		default:
		case CKM_MTK_HSM_SHA256:    cipher_len = sizeof(sha256_digest_1M_0_output);  break;
	}
	mechanism.mechanism = in_mechanism;

	// SHA stream mode
	rv = C_DigestInit(session, &mechanism);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[4001 Fail] C_DigestInit \n");
		goto out;
	}

	Do_ADBG_Log("  C_Digest SHA stream mode \n");
	for (int i = 0; i < 128; i++) // 8K * 128 = 1M
		rv = C_DigestUpdate(session, (CK_BYTE_PTR)origin_8K_0, sizeof(origin_8K_0));

	rv = C_DigestFinal(session, (CK_BYTE_PTR)cipher, &cipher_len);

	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : C_Digest(): %lu", mech_type, rv);
		goto out;
	}

	__SHOW_VAL("SHA stream output", cipher, cipher_len);
	switch(in_mechanism) {

		case CKM_MTK_HSM_SHA1:
			rv = memcmp(sha1_digest_1M_0_output, cipher, sizeof(sha1_digest_1M_0_output));
			mech_type = "CKM_MTK_HSM_SHA1";
			break;
		case CKM_MTK_HSM_SHA224:
			rv = memcmp(sha224_digest_1M_0_output, cipher, sizeof(sha224_digest_1M_0_output));
			mech_type = "CKM_MTK_HSM_SHA224";
			break;
		case CKM_MTK_HSM_SHA384:
			rv = memcmp(sha384_digest_1M_0_output, cipher, sizeof(sha384_digest_1M_0_output));
			mech_type = "CKM_MTK_HSM_SHA384";
			break;
		case CKM_MTK_HSM_SHA512:
			rv = memcmp(sha512_digest_1M_0_output, cipher, sizeof(sha512_digest_1M_0_output));
			mech_type = "CKM_MTK_HSM_SHA512";
			break;
		default:
		case CKM_MTK_HSM_SHA256:
			rv = memcmp(sha256_digest_1M_0_output, cipher, sizeof(sha256_digest_1M_0_output));
			mech_type = "CKM_MTK_HSM_SHA256";
			break;
	}

	if (0 == rv){
		Do_ADBG_Log("  [%s Digest 1M comparison OK!!.]  \n", mech_type);
	} else {
		Do_ADBG_Log("  [%s Digest 1M comparison fail!!.]  \n", mech_type);
		goto out;
	}
out:
	rv2 = C_CloseSession(session);

	if (rv != 0)
		return rv;
	else
		return rv2;

}
static void xtest_hsm_pkcs_11_test_4002(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;

	Do_ADBG_Log("======== 4002 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test_1M(CKM_MTK_HSM_SHA1,   slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test_1M(CKM_MTK_HSM_SHA224, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test_1M(CKM_MTK_HSM_SHA256, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test_1M(CKM_MTK_HSM_SHA384, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = _SHA_test_1M(CKM_MTK_HSM_SHA512, slot, c);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 4002, xtest_hsm_pkcs_11_test_4002,
		"PKCS11: SHA 256 stream mode 1M(8K * 128)");

static void xtest_hsm_pkcs_11_test_5001(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	unsigned char random1[BUFFERSIZ] = {0};
	unsigned char random2[BUFFERSIZ] = {0};

	Do_ADBG_Log("======== 5001 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	uint32_t size = 16;
	C_GenerateRandom(session, random1, size);
	C_GenerateRandom(session, random2, size);
	if (memcmp(random1, random2, size) == 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("TRNG gen 16 bytes fail!!");
		goto out;
	}

	size = 18;
	C_GenerateRandom(session, random1, size);
	C_GenerateRandom(session, random2, size);
	if (memcmp(random1, random2, size) == 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("TRNG gen 18 bytes fail!!");
		goto out;
	}

	size = 38;
	C_GenerateRandom(session, random1, size);
	C_GenerateRandom(session, random2, size);
	if (memcmp(random1, random2, size) == 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("TRNG gen 38 bytes fail!!");
		goto out;
	}

	size = 512;
	C_GenerateRandom(session, random1, size);
	C_GenerateRandom(session, random2, size);
	if (memcmp(random1, random2, size) == 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("TRNG gen 512 bytes fail!!");
		goto out;
	}

#ifdef PLATFORM_FLAVOR_mt2735
	//Do RNG Test
	size = 4;
	C_GenerateRandom(session, random1, size);
	C_GenerateRandom(session, random2, size);
	if (memcmp(random1, random2, size) == 0) {
		rv = CKR_GENERAL_ERROR;
		Do_ADBG_Log("RNG gen 4 bytes fail!!");
		goto out;
	}
#endif

	Do_ADBG_Log("  [TRNG generate keys OK!!.]  \n");


out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 5001, xtest_hsm_pkcs_11_test_5001,
		"PKCS11: TRNG");

CK_BBOOL true_val = CK_TRUE;
CK_BBOOL false_val = CK_FALSE;
CK_ULONG key_length_bytes = 32;
static CK_RV generate_ec_keypair(CK_SESSION_HANDLE session,
                                 CK_BYTE_PTR named_curve_oid,
                                 CK_ULONG named_curve_oid_len,
                                 CK_ULONG pub_key_id,
                                 CK_ULONG priv_key_id,
                                 CK_OBJECT_HANDLE_PTR public_key,
                                 CK_OBJECT_HANDLE_PTR private_key)
{
	CK_RV rv;
	CK_MECHANISM mech = {CKM_MTK_HSM_EC_KEY_PAIR_GEN, NULL, 0};

	CK_ULONG pub_key_name = pub_key_id;
	CK_ULONG priv_key_name = priv_key_id;
	CK_KEY_TYPE keyType = CKK_EC;

	CK_ATTRIBUTE public_key_template[] = {
		{CKA_SENSITIVE, &true_val,             sizeof(CK_BBOOL)},
		{CKA_VERIFY,    &true_val,           sizeof(CK_BBOOL)},
		{CKA_EC_PARAMS, named_curve_oid,     named_curve_oid_len},
		{CKA_TOKEN,     &true_val,           sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType,          sizeof(keyType)},
		{CKA_ID,        &pub_key_name,       sizeof(CK_ULONG)}
	};

	CK_ATTRIBUTE private_key_template[] = {
		{CKA_SENSITIVE, &true_val,             sizeof(CK_BBOOL)},
		{CKA_SIGN,      &true_val,        sizeof(CK_BBOOL)},
		{CKA_TOKEN,     &true_val,        sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType,         sizeof(keyType)},
		{CKA_ID,        &priv_key_name,   sizeof(CK_ULONG)}
	};

	rv = C_GenerateKeyPair(session,
	                       &mech,
	                       public_key_template, sizeof(public_key_template) / sizeof(CK_ATTRIBUTE),
	                       private_key_template, sizeof(private_key_template) / sizeof(CK_ATTRIBUTE),
	                       public_key,
	                       private_key);
	return rv;
}

static CK_RV find_ec_keypair(CK_SESSION_HANDLE session,
                             CK_OBJECT_HANDLE_PTR public_key,
                             CK_OBJECT_HANDLE_PTR private_key,
                             CK_ULONG pubkey_id,
                             CK_ULONG privkey_id)
{

	CK_RV rv;
	CK_ULONG count0 = 0;
	CK_OBJECT_HANDLE *found_objects0 = NULL;
	// CK_ULONG pub_key_name = XTEST_EC_PUB_KEY_ID_0;
	CK_ULONG pub_key_name = pubkey_id;

	CK_ATTRIBUTE attr0[] = {
		{CKA_ID, &pub_key_name, sizeof(CK_ULONG)}
	};

	rv = find_by_attr(session, attr0, 1, &count0, &found_objects0);
	if (rv != CKR_OK) {
		Do_ADBG_Log("[find public_key fail.1]");
		return rv;
	}
	if (count0 == 0) {
		Do_ADBG_Log("[find public_key fail.2]");
		return -1;
	}
	Do_ADBG_Log("[find public_key pass.]");

	*public_key = found_objects0[0];
	Do_ADBG_Log("[%s][%d] *public_key : %ld\n", __FUNCTION__, __LINE__, *public_key);
	Do_ADBG_Log("[%s][%d] count0 : %ld\n", __FUNCTION__, __LINE__, count0);

	/* ---------------------------------------- */

	CK_ULONG count1 = 0;
	CK_OBJECT_HANDLE *found_objects1 = NULL;
	// CK_ULONG priv_key_name = XTEST_EC_PRIV_KEY_ID_0;
	CK_ULONG priv_key_name = privkey_id;

	CK_ATTRIBUTE attr1[] = {
		{CKA_ID, &priv_key_name, sizeof(CK_ULONG)}
	};

	rv = find_by_attr(session, attr1, 1, &count1, &found_objects1);
	if (rv != CKR_OK) {
		Do_ADBG_Log("[find private_key fail.1]");
		return rv;
	}
	if (count1 == 0) {
		Do_ADBG_Log("[find private_key fail.2]");
		return 0xffff;
	}
	Do_ADBG_Log("[find key pass.]");
	*private_key = found_objects1[0];
	Do_ADBG_Log("[%s][%d] *private_key : %ld\n", __FUNCTION__, __LINE__, *private_key);
	Do_ADBG_Log("[%s][%d] count1 : %ld\n", __FUNCTION__, __LINE__, count1);

	return CKR_OK;

}

static CK_RV generate_signature(CK_SESSION_HANDLE session,
                                CK_OBJECT_HANDLE key,
                                CK_MECHANISM_TYPE mechanism,
                                CK_BYTE_PTR data,
                                CK_ULONG data_length,
                                CK_BYTE_PTR signature,
                                CK_ULONG_PTR signature_length)
{
	CK_RV rv;
	CK_MECHANISM mech;

	mech.mechanism = mechanism;
	mech.ulParameterLen = 0;
	mech.pParameter = NULL;

	rv = C_SignInit(session, &mech, key);
	if (rv != CKR_OK) {
		return !CKR_OK;
	}

	rv = C_Sign(session, data, data_length, signature, signature_length);
	return rv;
}

//--------------------------------------------------------------------------------
// ECDSA 192 sign
static uint8_t ecdsa192_sign_hashval_e[24] = {
	0x1F, 0x1E, 0x1C, 0xF8, 0x92, 0x92, 0x6C, 0xFC, 0xCF, 0xC5, 0xA2, 0x8F,
	0xD2, 0x3F, 0x77, 0x80, 0x08, 0xDB, 0xA4, 0xB3, 0x5F, 0x04, 0xB2, 0xFD,
};
//--------------------------------------------------------------------------------
// ECDSA 192 verify
//static uint8_t ecdsa192_verify_hashval_e[24] = {
//	0x00, 0x00, 0x00, 0x00, 0x1B, 0x37, 0x6F, 0x0B, 0x73, 0x5C, 0x61, 0x5C, 0xEE, 0xEB, 0x31, 0xBA,
//	0xEE, 0x65, 0x4B, 0x0A, 0x37, 0x48, 0x25, 0xDB,
//};
// static uint8_t ecdsa192_trm_pubkey_x[24] = {
//     0xFB, 0xA2, 0xAA, 0xC6, 0x47, 0x88, 0x4B, 0x50, 0x4E, 0xB8, 0xCD, 0x5A, 0x0A, 0x12, 0x87, 0xBA,
//     0xBC, 0xC6, 0x21, 0x63, 0xF6, 0x06, 0xA9, 0xA2,
// };
// static uint8_t ecdsa192_trm_pubkey_y[24] = {
//     0xDA, 0xE6, 0xD4, 0xCC, 0x05, 0xEF, 0x4F, 0x27, 0xD7, 0x9E, 0xE3, 0x8B, 0x71, 0xC9, 0xC8, 0xEF,
//     0x48, 0x65, 0xD9, 0x88, 0x50, 0xD8, 0x4A, 0xA5,
// };
//static uint8_t ecdsa192_verify_signature_rs[48] = {
//	// r
//	0xF0, 0xEC, 0xBA, 0x72, 0xB8, 0x8C, 0xDE, 0x39, 0x9C, 0xC5, 0xA1, 0x8E, 0x2A, 0x8B, 0x7D, 0xA5,
//	0x4D, 0x81, 0xD0, 0x4F, 0xB9, 0x80, 0x28, 0x21,
//	// s
//	0x1E, 0x6D, 0x3D, 0x4A, 0xE2, 0xB1, 0xFA, 0xB2, 0xBD, 0x20, 0x40, 0xF5, 0xDA, 0xBF, 0x00, 0xF8,
//	0x54, 0xFA, 0x14, 0x0B, 0x6D, 0x21, 0xE8, 0xED,
//};

//--------------------------------------------------------------------------------
// ECDSA 224 sign
static uint8_t ecdsa224_sign_hashval_e[28] = {
	0x1F, 0x1E, 0x1C, 0xF8, 0x92, 0x92, 0x6C, 0xFC, 0xCF, 0xC5, 0xA2, 0x8F, 0xEE, 0xF3, 0xD8, 0x07,
	0xD2, 0x3F, 0x77, 0x80, 0x08, 0xDB, 0xA4, 0xB3, 0x5F, 0x04, 0xB2, 0xFD,
};
// static uint8_t ecc224_d_key[28] = {
//     0x3F, 0x0C, 0x48, 0x8E, 0x98, 0x7C, 0x80, 0xBE, 0x0F, 0xEE, 0x52, 0x1F, 0x8D, 0x90,
//     0xBE, 0x60, 0x34, 0xEC, 0x69, 0xAE, 0x11, 0xCA, 0x72, 0xAA, 0x77, 0x74, 0x81, 0xE8,
// };
// static uint8_t ecc224_k_key[28] = {
//     0xA5, 0x48, 0x80, 0x3B, 0x79, 0xDF, 0x17, 0xC4, 0x0C, 0xDE, 0x3F, 0xF0, 0xE3, 0x6D,
//     0x02, 0x51, 0x43, 0xBC, 0xBB, 0xA1, 0x46, 0xEC, 0x32, 0x90, 0x8E, 0xB8, 0x49, 0x37,
// };
//static uint8_t ecdsa224_sign_signature_rs[56] = {
//	// r
//	0xC3, 0xA3, 0xF5, 0xB8, 0x27, 0x12, 0x53, 0x20, 0x04, 0xC6, 0xF6, 0xD1, 0xDB, 0x67, 0x2F, 0x55,
//	0xD9, 0x31, 0xC3, 0x40, 0x9E, 0xA1, 0x21, 0x6D, 0x0B, 0xE7, 0x73, 0x80,
//	// s
//	0xC5, 0xAA, 0x1E, 0xAE, 0x60, 0x95, 0xDE, 0xA3, 0x4C, 0x9B, 0xD8, 0x4D, 0xA3, 0x85, 0x2C, 0xCA,
//	0x41, 0xA8, 0xBD, 0x9D, 0x55, 0x48, 0xF3, 0x6D, 0xAB, 0xDF, 0x66, 0x17,
//};

//--------------------------------------------------------------------------------
// ECDSA 224 verify

//--------------------------------------------------------------------------------
// ECDSA 256 sign
static uint8_t ecdsa256_sign_hashval_e[32] = {
    0xA4, 0x1A, 0x41, 0xA1, 0x2A, 0x79, 0x95, 0x48, 0x21, 0x1C, 0x41, 0x0C, 0x65, 0xD8, 0x13, 0x3A,
    0xFD, 0xE3, 0x4D, 0x28, 0xBD, 0xD5, 0x42, 0xE4, 0xB6, 0x80, 0xCF, 0x28, 0x99, 0xC8, 0xA8, 0xC4,
};
// static uint8_t ecc256_d_key[32] = {
//     0xC4, 0x77, 0xF9, 0xF6, 0x5C, 0x22, 0xCC, 0xE2, 0x06, 0x57, 0xFA, 0xA5, 0xB2, 0xD1, 0xD8, 0x12,
//     0x23, 0x36, 0xF8, 0x51, 0xA5, 0x08, 0xA1, 0xED, 0x04, 0xE4, 0x79, 0xC3, 0x49, 0x85, 0xBF, 0x96,
// };
// static uint8_t ecc256_k_key[32] = {
//     0x7A, 0x1A, 0x7E, 0x52, 0x79, 0x7F, 0xC8, 0xCA, 0xAA, 0x43, 0x5D, 0x2A, 0x4D, 0xAC, 0xE3, 0x91,
//     0x58, 0x50, 0x4B, 0xF2, 0x04, 0xFB, 0xE1, 0x9F, 0x14, 0xDB, 0xB4, 0x27, 0xFA, 0xEE, 0x50, 0xAE,
// };
// static uint8_t ecdsa256_sign_signature_rs[64] = {
//     // r
//     0x2B, 0x42, 0xF5, 0x76, 0xD0, 0x7F, 0x41, 0x65, 0xFF, 0x65, 0xD1, 0xF3, 0xB1, 0x50, 0x0F, 0x81,
//     0xE4, 0x4C, 0x31, 0x6F, 0x1F, 0x0B, 0x3E, 0xF5, 0x73, 0x25, 0xB6, 0x9A, 0xCA, 0x46, 0x10, 0x4F,
//     // s
//     0xDC, 0x42, 0xC2, 0x12, 0x2D, 0x63, 0x92, 0xCD, 0x3E, 0x3A, 0x99, 0x3A, 0x89, 0x50, 0x2A, 0x81,
//     0x98, 0xC1, 0x88, 0x6F, 0xE6, 0x9D, 0x26, 0x2C, 0x4B, 0x32, 0x9B, 0xDB, 0x6B, 0x63, 0xFA, 0xF1,
// };

//-------------------------------------------------------------------------------
// ECDSA 256 verify
// static uint8_t ecdsa256_verify_hashval_e[32] = {
//     0x14, 0x2c, 0xea, 0x9d, 0x21, 0x15, 0xb7, 0x54, 0x43, 0x4a, 0x0e, 0x66, 0x98, 0x0e, 0xf0, 0xd8,
//     0x79, 0xfa, 0x5c, 0xdb, 0x87, 0x93, 0xf7, 0xf9, 0x01, 0x82, 0x0a, 0xdf, 0x4a, 0x4e, 0xac, 0x0c,
// };
// static uint8_t ecdsa256_trm_pubkey_x[32] = {
//     0x2e, 0xcc, 0x5c, 0x84, 0x0b, 0x62, 0x83, 0x86, 0x02, 0x5e, 0x74, 0xb6, 0x91, 0x80, 0xce, 0x2a,
//     0xfc, 0xed, 0x3c, 0x4b, 0xcd, 0x53, 0xa3, 0x1d, 0x53, 0xe6, 0xa0, 0xdc, 0x63, 0x1d, 0x91, 0x20,
// };
// static uint8_t ecdsa256_trm_pubkey_y[32] = {
//     0x22, 0x3f, 0x3e, 0x9d, 0x4c, 0x98, 0xb8, 0xe2, 0x43, 0xdb, 0xe2, 0x27, 0x19, 0x04, 0x6b, 0xd9,
//     0x90, 0xa8, 0x1d, 0x54, 0xc9, 0xf5, 0xb7, 0x5b, 0x74, 0x4a, 0xd4, 0xca, 0xcc, 0xfa, 0xe5, 0x0b,
// };
// static uint8_t ecdsa256_verify_signature_rs[64] = {
//     // r
//     0x90, 0xef, 0x91, 0x84, 0xdb, 0x77, 0x19, 0x01, 0x55, 0x64, 0xe0, 0x10, 0xa1, 0xb8, 0x9a, 0xf6,
//     0xe8, 0x58, 0x2b, 0x47, 0xfa, 0xdf, 0xd0, 0x45, 0x23, 0xb5, 0xa0, 0x13, 0x4e, 0xd5, 0x5c, 0x68,
//     // s
//     0x54, 0x2b, 0xc1, 0x87, 0xc7, 0xed, 0x2c, 0xd6, 0x25, 0x92, 0x91, 0x6e, 0x44, 0x47, 0xde, 0x50,
//     0x3b, 0x69, 0x18, 0xae, 0x2b, 0x7c, 0x7e, 0x8e, 0x32, 0x12, 0x7d, 0x9f, 0x2f, 0xa1, 0xf5, 0x5e,
// };


//--------------------------------------------------------------------------------
// ECDSA 384 sign
static uint8_t ecdsa384_sign_hashval_e[48] = {
    0x5A, 0xEA, 0x18, 0x7D, 0x1C, 0x4F, 0x6E, 0x1B, 0x35, 0x05, 0x7D, 0x20, 0x12, 0x6D, 0x83, 0x6C,
    0x6A, 0xDB, 0xBC, 0x70, 0x49, 0xEE, 0x02, 0x99, 0xC9, 0x52, 0x9F, 0x5E, 0x0B, 0x3F, 0x8B, 0x5A,
    0x74, 0x11, 0x14, 0x9D, 0x6C, 0x30, 0xD6, 0xCB, 0x2B, 0x8A, 0xF7, 0x0E, 0x0A, 0x78, 0x1E, 0x89,
};
// static uint8_t ecc384_d_key[48] = {
//     0xF9, 0x2C, 0x02, 0xED, 0x62, 0x9E, 0x4B, 0x48, 0xC0, 0x58, 0x4B, 0x1C, 0x6C, 0xE3, 0xA3, 0xE3,
//     0xB4, 0xFA, 0xAE, 0x4A, 0xFC, 0x6A, 0xCB, 0x04, 0x55, 0xE7, 0x3D, 0xFC, 0x39, 0x2E, 0x6A, 0x0A,
//     0xE3, 0x93, 0xA8, 0x56, 0x5E, 0x6B, 0x97, 0x14, 0xD1, 0x22, 0x4B, 0x57, 0xD8, 0x3F, 0x8A, 0x08,
// };
// static uint8_t ecc384_k_key[48] = {
//     0x2E, 0x44, 0xEF, 0x1F, 0x8C, 0x0B, 0xEA, 0x83, 0x94, 0xE3, 0xDD, 0xA8, 0x1E, 0xC6, 0xA7, 0x84,
//     0x2A, 0x45, 0x9B, 0x53, 0x47, 0x01, 0x74, 0x9E, 0x2E, 0xD9, 0x5F, 0x05, 0x4F, 0x01, 0x37, 0x68,
//     0x08, 0x78, 0xE0, 0x74, 0x9F, 0xC4, 0x3F, 0x85, 0xED, 0xCA, 0xE0, 0x6C, 0xC2, 0xF4, 0x3F, 0xEF,
// };
// static uint8_t ecdsa384_sign_signature_rs[96] = {
//     // r
//     0x30, 0xEA, 0x51, 0x4F, 0xC0, 0xD3, 0x8D, 0x82, 0x08, 0x75, 0x6F, 0x06, 0x81, 0x13, 0xC7, 0xCA,
//     0xDA, 0x9F, 0x66, 0xA3, 0xB4, 0x0E, 0xA3, 0xB3, 0x13, 0xD0, 0x40, 0xD9, 0xB5, 0x7D, 0xD4, 0x1A,
//     0x33, 0x27, 0x95, 0xD0, 0x2C, 0xC7, 0xD5, 0x07, 0xFC, 0xEF, 0x9F, 0xAF, 0x01, 0xA2, 0x70, 0x88,
//     // s
//     0xCC, 0x80, 0x8E, 0x50, 0x4B, 0xE4, 0x14, 0xF4, 0x6C, 0x90, 0x27, 0xBC, 0xBF, 0x78, 0xAD, 0xF0,
//     0x67, 0xA4, 0x39, 0x22, 0xD6, 0xFC, 0xAA, 0x66, 0xC4, 0x47, 0x68, 0x75, 0xFB, 0xB7, 0xB9, 0x4E,
//     0xFD, 0x1F, 0x7D, 0x5D, 0xBE, 0x62, 0x0B, 0xFB, 0x82, 0x1C, 0x46, 0xD5, 0x49, 0x68, 0x3A, 0xD8,
// };

//-------------------------------------------------------------------------------
// ECDSA 384 verify
// static uint8_t ecdsa384_verify_hashval_e[48] = {
//     0xcb, 0xc6, 0xbb, 0xa3, 0xe5, 0x87, 0x7a, 0x6d, 0xff, 0xfd, 0xb2, 0xdc, 0xb7, 0x7a, 0xbb, 0xa9,
//     0xe3, 0x30, 0x6a, 0xf3, 0xdd, 0x57, 0x4d, 0xfc, 0x82, 0x32, 0x66, 0x95, 0x6c, 0x00, 0xa2, 0x51,
//     0x30, 0x1d, 0xb5, 0x32, 0xd6, 0xc6, 0x1c, 0x40, 0x13, 0xf7, 0xe8, 0x7d, 0x1f, 0xbb, 0x09, 0x22,
// };
// static  uint8_t ecdsa384_trm_pubkey_x[48] = {
//     0x06, 0xde, 0x1f, 0x7f, 0x02, 0xdf, 0x3a, 0xcd, 0x90, 0x4b, 0x68, 0x75, 0x5e, 0x94, 0xc1, 0xdc,
//     0x48, 0x6f, 0x23, 0x51, 0x1c, 0x60, 0x69, 0x8e, 0x2d, 0xbc, 0x0b, 0x87, 0x4c, 0x6b, 0x89, 0x50,
//     0x2b, 0xe6, 0x83, 0xd7, 0xcd, 0x04, 0x69, 0x26, 0x43, 0x09, 0xb4, 0xf2, 0x9d, 0x49, 0x80, 0xb9,
// };
// static  uint8_t ecdsa384_trm_pubkey_y[48] = {
//     0x4c, 0x18, 0xb5, 0xe9, 0x6f, 0x64, 0xcc, 0x4e, 0xe9, 0x21, 0xc7, 0x1f, 0x19, 0x90, 0x56, 0x8f,
//     0x35, 0xdd, 0x3a, 0x34, 0xae, 0x3c, 0x54, 0x19, 0x8c, 0x9d, 0x9c, 0x17, 0x4b, 0x82, 0x22, 0x14,
//     0xb0, 0x94, 0xae, 0x82, 0xe7, 0xf0, 0x72, 0xdd, 0xc3, 0xc2, 0x24, 0x0a, 0xe3, 0xbc, 0xfd, 0x81,
// };
// static uint8_t ecdsa384_verify_signature_rs[96] = {
//     // r
//     0x1c, 0x5a, 0xd5, 0xd9, 0x42, 0x86, 0xc3, 0xf6, 0xdc, 0x9e, 0x88, 0x25, 0x8f, 0x4e, 0x27, 0xfe,
//     0x0a, 0x89, 0x3c, 0x52, 0xb9, 0xa5, 0xaf, 0xe0, 0x9d, 0x84, 0xec, 0x94, 0xcb, 0xb0, 0xaa, 0x64,
//     0x3e, 0x17, 0x58, 0xa5, 0x04, 0x46, 0xcd, 0x20, 0x91, 0x20, 0x69, 0x05, 0xf0, 0x83, 0xfd, 0xa9,
//     // s
//     0xb6, 0x24, 0xa6, 0xe7, 0x34, 0xbe, 0xb2, 0x88, 0x34, 0x74, 0x2f, 0x76, 0xac, 0x05, 0x24, 0x9d,
//     0x3f, 0x3a, 0x52, 0x84, 0x16, 0x90, 0x1f, 0xc6, 0x6e, 0xe6, 0xca, 0xf1, 0x58, 0x69, 0x87, 0x9c,
//     0x1d, 0xfc, 0x64, 0xe5, 0x1e, 0xaf, 0xc4, 0xc5, 0x58, 0xf7, 0x29, 0x32, 0x61, 0x66, 0xfb, 0x7a,
// };
// ECDSA sm2 sign
static uint8_t ecdsasm2_sign_hashval_e[32] = {
    0x5B, 0xF4, 0xBD, 0x14, 0xA1, 0x6B, 0x7E, 0x3C, 0xE0, 0xD7, 0x77, 0xFC, 0xB1, 0x46, 0x8B, 0xB0,
    0x88, 0xD5, 0xB4, 0x80, 0x1C, 0x00, 0x78, 0xA3, 0x87, 0xE9, 0x78, 0x71, 0x7B, 0xC8, 0x3F, 0x7A,
};

static CK_RV verify_signature(CK_SESSION_HANDLE session,
                              CK_OBJECT_HANDLE key,
                              CK_MECHANISM_TYPE mechanism,
                              CK_BYTE_PTR data,
                              CK_ULONG data_length,
                              CK_BYTE_PTR signature,
                              CK_ULONG signature_length)
{
	CK_RV rv;
	CK_MECHANISM mech;

	mech.mechanism = mechanism;
	mech.ulParameterLen = 0;
	mech.pParameter = NULL;

	rv = C_VerifyInit(session, &mech, key);
	if (rv != CKR_OK) {
		return !CKR_OK;
	}

	rv = C_Verify(session, data, data_length, signature, signature_length);
	return rv;
}

//static int bytes_to_new_hexstring(char *bytes, size_t bytes_len, unsigned char **hex_array)
//{
//	if (!bytes || !hex_array) {
//		return -1;
//	}
//
//	unsigned char *tmp = realloc(*hex_array, bytes_len * 2 + 1);
//	if (!tmp) {
//		if (*hex_array) {
//			free(*hex_array);
//		}
//		return -1;
//	}
//	*hex_array = tmp;
//	memset(*hex_array, 0, bytes_len * 2 + 1);
//
//	char values[16] = "0123456789ABCDEF";
//	for (int i = 0, j = 0; i < (int)bytes_len; i++, j += 2) {
//		*((*hex_array) + j) = values[bytes[i] >> 4 & 0x0f];
//		*((*hex_array) + j + 1) = values[bytes[i] & 0x0f];
//	}
//
//	return 0;
//}

#define MAX_SIGNATURE_LENGTH 256
static CK_RV ec_main(CK_SESSION_HANDLE session, uint32_t key_len) {
	CK_RV rv;

	CK_BYTE_PTR data = NULL;
	CK_ULONG data_length = 0;

	CK_OBJECT_HANDLE pubkey = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE privkey = CK_INVALID_HANDLE;
	CK_ULONG pubkey_id;
	CK_ULONG privkey_id;

	CK_BYTE prime192v1[] = {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x01};
	CK_BYTE secp224r1[] = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x21};
	CK_BYTE prime256v1[] = {0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07};
	CK_BYTE secp384r1[] = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22};

	CK_BYTE_PTR curve = NULL;
	CK_ULONG curve_oid_len = 0;

	clock_t t0, t1;

	// Assign signature data
	switch (key_len)
	{
	case 192:
		curve = prime192v1;
		curve_oid_len = sizeof(prime192v1);
		data = ecdsa192_sign_hashval_e;
		data_length = sizeof(ecdsa192_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_0;
		privkey_id = XTEST_EC_PRIV_KEY_ID_0;
		break;

	case 224:
		curve = secp224r1;
		curve_oid_len = sizeof(secp224r1);
		data = ecdsa224_sign_hashval_e;
		data_length = sizeof(ecdsa224_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_1;
		privkey_id = XTEST_EC_PRIV_KEY_ID_1;
		break;

	case 256:
		curve = prime256v1;
		curve_oid_len = sizeof(prime256v1);
		data = ecdsa256_sign_hashval_e;
		data_length = sizeof(ecdsa256_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_2;
		privkey_id = XTEST_EC_PRIV_KEY_ID_2;
		break;

	default:
	case 384:
		curve = secp384r1;
		curve_oid_len = sizeof(secp384r1);
		data = ecdsa384_sign_hashval_e;
		data_length = sizeof(ecdsa384_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_3;
		privkey_id = XTEST_EC_PRIV_KEY_ID_3;
		break;
	}

	CK_BYTE signature[MAX_SIGNATURE_LENGTH];
	CK_ULONG signature_length = key_len / 8 * 2; // Including r + s

	// Set the PKCS11 signature mechanism type.
	CK_MECHANISM_TYPE mechanism = CKM_MTK_HSM_ECDSA;

	/**
	 * Curve OIDs generated using OpenSSL on the command line.
	 * Visit https://docs.aws.amazon.com/cloudhsm/latest/userguide/pkcs11-key-types.html for a list
	 * of supported curves.
	 * openssl ecparam -name prime256v1 -outform DER | hexdump -C
	 */

	rv = find_ec_keypair(session, &pubkey, &privkey, pubkey_id, privkey_id);
	if (rv != CKR_OK)
	{
		rv = generate_ec_keypair(session, curve, curve_oid_len, pubkey_id, privkey_id, &pubkey, &privkey);
		if (rv == CKR_OK) {
			printf("prime256v1 key generated. Public key handle: %lu, Private key handle: %lu\n", pubkey,
			       privkey);
		} else {
			printf("prime256v1 key generation failed: %lu\n", rv);
			return rv;
		}
	}

	rv = generate_signature(session, privkey, mechanism,
	                        data, data_length, signature, &signature_length);

	printf(" == ECDSA Sign %d == \r\n", key_len);
	__SHOW_VAL("ECDSA output", signature, signature_length);


//	if (rv == CKR_OK) {
//		unsigned char *hex_signature = NULL;
//		bytes_to_new_hexstring((char*)signature, signature_length, &hex_signature);
//		if (!hex_signature) {
//			printf("Could not allocate hex array\n");
//			return 1;
//		}
//		printf("Data: %s\n", data);
//		printf("Signature: %s\n", hex_signature);
//		free(hex_signature);
//		hex_signature = NULL;
//	} else {
//		printf("Signature generation failed: %lu\n", rv);
//		return rv;
//	}


	printf(" == ECDSA Verify %d == \r\n", key_len);
	t0 = clock();
	rv = verify_signature(session, pubkey, mechanism, data, data_length, signature, signature_length);
	if (rv == CKR_OK) {
		printf(" Verification successful PASS!!  \n");
	} else {
		printf(" Verification successful FAIL!!  \n");
		return rv;
	}
	t1 = clock();
	float time_diff = (float)(t1 - t0)/CLOCKS_PER_SEC;
	Do_ADBG_Log("ECDSA verify time = %lf\n", time_diff);

	return 0;
}

static CK_RV ec_sha_main(CK_SESSION_HANDLE session, uint32_t key_len) {
	CK_RV rv;

	CK_BYTE_PTR data = NULL;
	CK_ULONG data_length = 0;

	CK_OBJECT_HANDLE pubkey = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE privkey = CK_INVALID_HANDLE;
	CK_ULONG pubkey_id;
	CK_ULONG privkey_id;

	CK_BYTE prime192v1[] = {0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x01};
	CK_BYTE secp224r1[] = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x21};
	CK_BYTE prime256v1[] = {0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07};
	CK_BYTE secp384r1[] = {0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22};

	CK_BYTE_PTR curve = NULL;
	CK_ULONG curve_oid_len = 0;

	// Assign signature data
	switch (key_len)
	{
	case 192:
		curve = prime192v1;
		curve_oid_len = sizeof(prime192v1);
		data = ecdsa192_sign_hashval_e;
		data_length = sizeof(ecdsa192_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_0;
		privkey_id = XTEST_EC_PRIV_KEY_ID_0;
		break;

	case 224:
		curve = secp224r1;
		curve_oid_len = sizeof(secp224r1);
		data = ecdsa224_sign_hashval_e;
		data_length = sizeof(ecdsa224_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_1;
		privkey_id = XTEST_EC_PRIV_KEY_ID_1;
		break;

	case 256:
		curve = prime256v1;
		curve_oid_len = sizeof(prime256v1);
		data = ecdsa256_sign_hashval_e;
		data_length = sizeof(ecdsa256_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_2;
		privkey_id = XTEST_EC_PRIV_KEY_ID_2;
		break;

	default:
	case 384:
		curve = secp384r1;
		curve_oid_len = sizeof(secp384r1);
		data = ecdsa384_sign_hashval_e;
		data_length = sizeof(ecdsa384_sign_hashval_e);
		pubkey_id = XTEST_EC_PUB_KEY_ID_3;
		privkey_id = XTEST_EC_PRIV_KEY_ID_3;
		break;
	}

	CK_BYTE signature[MAX_SIGNATURE_LENGTH];
	CK_ULONG signature_length = key_len / 8 * 2; // Including r + s

	// Set the PKCS11 signature mechanism type.
	CK_MECHANISM_TYPE mechanism = CKM_MTK_HSM_ECDSA_SHA256;

	/**
	 * Curve OIDs generated using OpenSSL on the command line.
	 * Visit https://docs.aws.amazon.com/cloudhsm/latest/userguide/pkcs11-key-types.html for a list
	 * of supported curves.
	 * openssl ecparam -name prime256v1 -outform DER | hexdump -C
	 */

	rv = find_ec_keypair(session, &pubkey, &privkey, pubkey_id, privkey_id);
	if (rv != CKR_OK)
	{
		rv = generate_ec_keypair(session, curve, curve_oid_len, pubkey_id, privkey_id, &pubkey, &privkey);
		if (rv == CKR_OK) {
			printf("prime256v1 key generated. Public key handle: %lu, Private key handle: %lu\n", pubkey,
			       privkey);
		} else {
			printf("prime256v1 key generation failed: %lu\n", rv);
			return rv;
		}
	}

	rv = generate_signature(session, privkey, mechanism,
	                        data, data_length, signature, &signature_length);

	printf(" == ECDSA Sign %d == \r\n", key_len);
	__SHOW_VAL("ECDSA output", signature, signature_length);


//	if (rv == CKR_OK) {
//		unsigned char *hex_signature = NULL;
//		bytes_to_new_hexstring((char*)signature, signature_length, &hex_signature);
//		if (!hex_signature) {
//			printf("Could not allocate hex array\n");
//			return 1;
//		}
//		printf("Data: %s\n", data);
//		printf("Signature: %s\n", hex_signature);
//		free(hex_signature);
//		hex_signature = NULL;
//	} else {
//		printf("Signature generation failed: %lu\n", rv);
//		return rv;
//	}

	printf(" == ECDSA Verify %d == \r\n", key_len);
	rv = verify_signature(session, pubkey, mechanism, data, data_length, signature, signature_length);
	if (rv == CKR_OK) {
		printf(" Verification successful PASS!!  \n");
	} else {
		printf(" Verification successful FAIL!!  \n");
		return rv;
	}

	return 0;
}

static CK_RV ec_sm2_main(CK_SESSION_HANDLE session, uint32_t key_len) {
	CK_RV rv;

	CK_BYTE_PTR data = NULL;
	CK_ULONG data_length = 0;

	CK_OBJECT_HANDLE pubkey = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE privkey = CK_INVALID_HANDLE;
	CK_ULONG pubkey_id;
	CK_ULONG privkey_id;
	CK_BYTE prime256v1[] = {0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07};

	CK_BYTE_PTR curve = NULL;
	CK_ULONG curve_oid_len = 0;

	// Assign signature data
	curve = prime256v1;
	curve_oid_len = sizeof(prime256v1);
	data = ecdsasm2_sign_hashval_e;
	data_length = sizeof(ecdsasm2_sign_hashval_e);

	pubkey_id = XTEST_EC_PUB_KEY_ID_2;
	privkey_id = XTEST_EC_PRIV_KEY_ID_2;

	CK_BYTE signature[MAX_SIGNATURE_LENGTH];
	CK_ULONG signature_length = key_len / 8 * 2; // Including r + s

	// Set the PKCS11 signature mechanism type.
	CK_MECHANISM_TYPE mechanism = CKM_MTK_HSM_ECDSA_SM2;

	/**
	 * Curve OIDs generated using OpenSSL on the command line.
	 * Visit https://docs.aws.amazon.com/cloudhsm/latest/userguide/pkcs11-key-types.html for a list
	 * of supported curves.
	 * openssl ecparam -name prime256v1 -outform DER | hexdump -C
	 */

	rv = find_ec_keypair(session, &pubkey, &privkey, pubkey_id, privkey_id);
	if (rv != CKR_OK)
	{
		rv = generate_ec_keypair(session, curve, curve_oid_len, pubkey_id, privkey_id, &pubkey, &privkey);
		if (rv == CKR_OK) {
			printf("prime256v1 key generated. Public key handle: %lu, Private key handle: %lu\n", pubkey,
			       privkey);
		} else {
			printf("prime256v1 key generation failed: %lu\n", rv);
			return rv;
		}
	}

	rv = generate_signature(session, privkey, mechanism,
	                        data, data_length, signature, &signature_length);

	printf(" == ECDSA-SM2 Sign %d == \r\n", key_len);
	__SHOW_VAL("ECDSA-SM2 output", signature, signature_length);


	printf(" == ECDSA-SM2 Verify %d == \r\n", key_len);
	rv = verify_signature(session, pubkey, mechanism, data, data_length, signature, signature_length);
	if (rv == CKR_OK) {
		printf(" Verification successful PASS!!  \n");
	} else {
		printf(" Verification successful FAIL!!  \n");
		return rv;
	}

	return 0;
}

static void xtest_hsm_pkcs_11_test_6001(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	Do_ADBG_Log("======== Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = ec_main(session, 384);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_Log("[ECDSA 384 Sign --> Verify pass]\n");

out:

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 6001, xtest_hsm_pkcs_11_test_6001,
                 "PKCS11: ECDSA - 384");

static void xtest_hsm_pkcs_11_test_6002(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	Do_ADBG_Log("======== Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = ec_main(session, 256);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_Log("[ECDSA 256 Sign --> Verify pass]\n");

out:

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 6002, xtest_hsm_pkcs_11_test_6002,
                 "PKCS11: ECDSA-256");

static void xtest_hsm_pkcs_11_test_6003(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	Do_ADBG_Log("======== Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = ec_main(session, 224);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_Log("[ECDSA 224 Sign --> Verify pass] \n");

out:

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 6003, xtest_hsm_pkcs_11_test_6003,
                 "PKCS11: ECDSA-224");

static void xtest_hsm_pkcs_11_test_6004(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	Do_ADBG_Log("======== Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = ec_main(session, 192);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_Log("[ECDSA 192 Sign --> Verify pass]\n");

out:

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 6004, xtest_hsm_pkcs_11_test_6004,
                 "PKCS11: ECDSA-192");


static void xtest_hsm_pkcs_11_test_6005(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	Do_ADBG_Log("======== 6005 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = ec_sha_main(session, 256);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_Log(" [ECDSA SHA Sign --> Verify pass] \n");

out:

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 6005, xtest_hsm_pkcs_11_test_6005,
                 "PKCS11: ECDSA-SHA");

static void xtest_hsm_pkcs_11_test_6006(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	Do_ADBG_Log("======== Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = ec_sm2_main(session, 256);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	Do_ADBG_Log("[ECDSA SM2(256) Sign --> Verify pass]\n");

out:

	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 6006, xtest_hsm_pkcs_11_test_6006,
                 "PKCS11: ECDSA-SM2");

static CK_RV import_symmetric_key(CK_SESSION_HANDLE session,
                                  CK_KEY_TYPE keyType,
                                  CK_BYTE_PTR key_content,
                                  CK_ULONG key_length,
                                  CK_ULONG key_id,
                                  CK_BYTE_PTR keyblob,
                                  CK_ULONG_PTR keyblob_length)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_BBOOL ck_true = CK_TRUE;
	CK_ULONG key_name = key_id;
	// CK_KEY_TYPE keyType = CKK_AES;
	CK_OBJECT_CLASS obj_class_secret = CKO_SECRET_KEY;

	__SHOW_VAL("symmetirc key content", key_content, key_length);

	/* Set the key object */
	CK_ATTRIBUTE attrs[] = {
		{CKA_SENSITIVE, &ck_true,          sizeof(CK_BBOOL)},
		{CKA_CLASS,     &obj_class_secret, sizeof(CK_OBJECT_CLASS)},
		{CKA_KEY_TYPE,  &keyType,          sizeof(CK_KEY_TYPE)},
		{CKA_TOKEN,     &ck_true,          sizeof(CK_BBOOL)},
		{CKA_VALUE,     key_content,       key_length},
		{CKA_VALUE_LEN, &key_length,       sizeof(CK_ULONG)},
		{CKA_ENCRYPT,   &ck_true,          sizeof(CK_BBOOL)},
		{CKA_DECRYPT,   &ck_true,          sizeof(CK_BBOOL)},
		{CKA_ID,        &key_name,         sizeof(CK_ULONG)}
	};

	rv = C_ImportKey(session, attrs, sizeof(attrs) / sizeof(CK_ATTRIBUTE), keyblob, keyblob_length);

	__SHOW_VAL("key blob", keyblob, *keyblob_length);

	return rv;
}

#define MAX_BLOB_LENGTH (256)
static void xtest_hsm_pkcs_11_test_1014(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_HMAC_256_KEY_ID_0;
	CK_OBJECT_HANDLE key;

	CK_BYTE keyblob[MAX_BLOB_LENGTH] = {0};
	CK_ULONG keyblob_length = sizeof(keyblob);

	Do_ADBG_Log("======== 1014 Test ========");

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = import_symmetric_key(session, CKK_SHA256_HMAC, hmac_256_key, sizeof(hmac_256_key), aes_id, keyblob, &keyblob_length);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1014, xtest_hsm_pkcs_11_test_1014,
                 "PKCS11: import hmac 256 symmetric key");

static void xtest_hsm_pkcs_11_test_1016(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_HMAC_384_KEY_ID_0;
	CK_OBJECT_HANDLE key;

	CK_BYTE keyblob[MAX_BLOB_LENGTH];
	CK_ULONG keyblob_length = sizeof(keyblob);

	Do_ADBG_Log("======== 1016 Test ========");

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = import_symmetric_key(session, CKK_SHA384_HMAC, hmac_384_key, sizeof(hmac_384_key), aes_id, keyblob, &keyblob_length);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1016, xtest_hsm_pkcs_11_test_1016,
                 "PKCS11: import hmac 384 symmetric key");

static void xtest_hsm_pkcs_11_test_1017(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_ULONG aes_id = XTEST_NULL_KEY_ID;
	CK_OBJECT_HANDLE key;

	CK_BYTE keyblob[MAX_BLOB_LENGTH];
	CK_ULONG keyblob_length = sizeof(keyblob);

	Do_ADBG_Log("======== 1017 Test ========");

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	// Import CMAC 128 key
	aes_id = XTEST_CMAC_128_KEY_ID_0;
	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = import_symmetric_key(session, CKK_AES, cmac_128_key, sizeof(cmac_128_key), aes_id, keyblob, &keyblob_length);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
	}

	// Import CMAC 192 key
	aes_id = XTEST_CMAC_192_KEY_ID_0;
	keyblob_length = sizeof(keyblob);
	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = import_symmetric_key(session, CKK_AES, cmac_192_key, sizeof(cmac_192_key), aes_id, keyblob, &keyblob_length);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
	}

	// Import CMAC 256 key
	aes_id = XTEST_CMAC_256_KEY_ID_0;
	keyblob_length = sizeof(keyblob);
	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = import_symmetric_key(session, CKK_AES, cmac_256_key, sizeof(cmac_256_key), aes_id, keyblob, &keyblob_length);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1017, xtest_hsm_pkcs_11_test_1017,
                 "PKCS11: import cmac 128/192/256 symmetric key");

static CK_RV import_ec_keypair(CK_SESSION_HANDLE session,
                               CK_BYTE_PTR named_curve_oid, CK_ULONG named_curve_oid_len,
                               CK_ULONG pubkey_id, CK_BYTE_PTR pubkey_content, CK_ULONG pubkey_length,
                               CK_ULONG privkey_id, CK_BYTE_PTR privkey_content, CK_ULONG privkey_length,
                               CK_BYTE_PTR keyblob, CK_ULONG_PTR keyblob_length)
{
	CK_RV rv;
	CK_MECHANISM mech = {CKM_MTK_HSM_EC_KEY_PAIR_IMPORT, NULL, 0};

	CK_ULONG pub_key_name = pubkey_id;
	CK_ULONG priv_key_name = privkey_id;
	CK_KEY_TYPE keyType = CKK_EC;

	CK_ATTRIBUTE public_key_template[] = {
		{CKA_SENSITIVE, &true_val,             sizeof(CK_BBOOL)},
		{CKA_VERIFY,    &true_val,             sizeof(CK_BBOOL)},
		{CKA_EC_PARAMS, named_curve_oid,       named_curve_oid_len},
		{CKA_TOKEN,     &true_val,             sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType,              sizeof(keyType)},
		{CKA_ID,        &pub_key_name,         sizeof(CK_ULONG)},
		{CKA_VALUE,     pubkey_content,        pubkey_length},
		{CKA_VALUE_LEN, &pubkey_length,        sizeof(CK_ULONG)},
	};

	CK_ATTRIBUTE private_key_template[] = {
		{CKA_SENSITIVE, &true_val,             sizeof(CK_BBOOL)},
		{CKA_SIGN,      &true_val,             sizeof(CK_BBOOL)},
		{CKA_TOKEN,     &true_val,             sizeof(CK_BBOOL)},
		{CKA_KEY_TYPE,  &keyType,              sizeof(keyType)},
		{CKA_ID,        &priv_key_name,        sizeof(CK_ULONG)},
		{CKA_VALUE,     privkey_content,       privkey_length},
		{CKA_VALUE_LEN, &privkey_length,       sizeof(CK_ULONG)},
	};

	__SHOW_VAL("pub key content", pubkey_content, pubkey_length);
	__SHOW_VAL("priv key content", privkey_content, privkey_length);

	rv = C_ImportKeyPair(session,
	                     &mech,
	                     public_key_template, sizeof(public_key_template) / sizeof(CK_ATTRIBUTE),
	                     private_key_template, sizeof(private_key_template) / sizeof(CK_ATTRIBUTE),
	                     keyblob,
	                     keyblob_length);

	__SHOW_VAL("key blob", keyblob, *keyblob_length);

	return rv;
}

/* ec key pair golden sample */
static uint8_t _private_key[] =
{
	0x62, 0xFB, 0x7C, 0x52, 0x2D, 0x02, 0x86, 0x31,
	0x90, 0x93, 0xAB, 0xA6, 0x23, 0xF4, 0x00, 0xF4,
	0xC8, 0xCE, 0xAC, 0x50, 0xFC, 0x93, 0xFF, 0x9B,
	0xE7, 0x53, 0xD1, 0x4B, 0x5A, 0x6B, 0x51, 0xED
};
static uint8_t _public_key[] =
{
	0xAB, 0x83, 0xF9, 0x93, 0x6D, 0x57, 0x07, 0x31,
	0x30, 0x47, 0x8C, 0x45, 0xEB, 0x45, 0xC3, 0x94,
	0xCE, 0x7E, 0x9B, 0x13, 0x24, 0x3B, 0x74, 0xAD,
	0x51, 0xEF, 0xCF, 0x4C, 0x8E, 0x32, 0x4F, 0x83,
	0x11, 0x3B, 0xF3, 0xD9, 0x19, 0xD0, 0xB6, 0x26,
	0x9C, 0x4E, 0xC2, 0x99, 0xBC, 0x74, 0xFB, 0xCE,
	0xB3, 0xE8, 0x4A, 0xEA, 0xFA, 0xE3, 0x1D, 0xE0,
	0x57, 0x27, 0x3C, 0xFA, 0x1E, 0xDB, 0x37, 0xE5,
};

static int store_blob_to_temp_file(uint8_t *data, uint32_t data_length)
{
	FILE *pFile;

	pFile = fopen( "/usr/local/test_blob", "w" );
	if ( NULL == pFile ) {
		printf( "open failure\n" );
		return -1;
	} else {
		fwrite(data, 1, data_length, pFile);
	}

	fclose(pFile);
	return 0;
}

static int read_blob_from_temp_file(uint8_t *buffer, uint32_t *pbufferlen)
{

	FILE *pFile;
	int readsize = 0;
	if ( !buffer || !pbufferlen )
		return -1;

	readsize = *pbufferlen;
	pFile = fopen( "/usr/local/test_blob", "r" );
	if ( NULL == pFile ) {
		printf( "open failure\n" );
		return -1;
	} else {
		readsize = fread(buffer, 1, readsize, pFile);
	}
	*pbufferlen = readsize;
	fclose(pFile);

	return readsize;
}
static void xtest_hsm_pkcs_11_test_1015(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_BYTE keyblob[MAX_BLOB_LENGTH] = {0};
	CK_ULONG keyblob_length = sizeof(keyblob);

	CK_OBJECT_HANDLE pubkey = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE privkey = CK_INVALID_HANDLE;
	CK_ULONG pubkey_id = XTEST_EC_PUB_KEY_ID_4;
	CK_ULONG privkey_id = XTEST_EC_PRIV_KEY_ID_4;

	Do_ADBG_Log("======== 1015 Test ========");

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	CK_BYTE prime256v1[] = {0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07};

	rv = find_ec_keypair(session, &pubkey, &privkey, pubkey_id, privkey_id);
	if (rv != CKR_OK) {
		rv = import_ec_keypair(session, prime256v1, sizeof(prime256v1),
		                       pubkey_id, _public_key, sizeof(_public_key),
		                       privkey_id, _private_key, sizeof(_private_key),
		                       keyblob, &keyblob_length);
		if (rv == CKR_OK) {
			printf("prime256v1 key generated. Public key handle: %lu, Private key handle: %lu\n", pubkey,
			       privkey);

			store_blob_to_temp_file(keyblob, keyblob_length);
		} else {
			printf("prime256v1 key generation failed: %lu\n", rv);
		}
	}

	if (!ADBG_EXPECT_CK_OK(c, rv)) {
		goto out;
	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1015, xtest_hsm_pkcs_11_test_1015,
                 "PKCS11: import ecc keypair");


static void xtest_hsm_pkcs_11_test_1018(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	CK_BYTE pubkeybuf[96] = {0};
	CK_ULONG pubkeylength = sizeof(pubkeybuf);

	Do_ADBG_Log("======== 1018 Test ========");

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;


	CK_BYTE keyblob[MAX_BLOB_LENGTH] = {0};
	CK_ULONG keyblob_length = sizeof(keyblob);
	CK_OBJECT_HANDLE pubkey = CK_INVALID_HANDLE;
	CK_OBJECT_HANDLE privkey = CK_INVALID_HANDLE;
	CK_ULONG pubkey_id = XTEST_EC_PUB_KEY_ID_4;
	CK_ULONG privkey_id = XTEST_EC_PRIV_KEY_ID_4;

	CK_BYTE prime256v1[] = {0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07};

	rv = find_ec_keypair(session, &pubkey, &privkey, pubkey_id, privkey_id);
	if (rv != CKR_OK) {
		rv = import_ec_keypair(session, prime256v1, sizeof(prime256v1),
		                       pubkey_id, _public_key, sizeof(_public_key),
		                       privkey_id, _private_key, sizeof(_private_key),
		                       keyblob, &keyblob_length);
		if (rv == CKR_OK) {
			printf("prime256v1 key imported. Public key handle: %lu, Private key handle: %lu\n", pubkey,
			       privkey);
			store_blob_to_temp_file(keyblob, keyblob_length);
		} else {
			printf("prime256v1 key imported failed: %lu\n", rv);
		}
	}

	rv = read_blob_from_temp_file(keyblob, (uint32_t *)&keyblob_length);

	__SHOW_VAL("key blob to export", keyblob, keyblob_length);
	rv = C_ExportKey(session, keyblob, keyblob_length, pubkeybuf, &pubkeylength);
	if (!ADBG_EXPECT_CK_OK(c, rv)) {
		goto out;
	}
	__SHOW_VAL("pub key from export", pubkeybuf, pubkeylength);

	if (0 != memcmp(pubkeybuf, _public_key, pubkeylength)) {
		rv = CKR_GENERAL_ERROR;
		printf("[%s][%d] export key result unexpected.\n", __FUNCTION__, __LINE__);
		goto out;
	} else {
		printf("[%s][%d] export key PASS!\n", __FUNCTION__, __LINE__);
	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1018, xtest_hsm_pkcs_11_test_1018,
                 "PKCS11: export key");


static void xtest_hsm_pkcs_11_test_1098(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;

	const char *mech_type;

	Do_ADBG_Log("======== Test ========");


	CK_ULONG aes_id = XTEST_AES_KEY_ID_1;
	CK_ULONG key_length = 24;
	CK_OBJECT_HANDLE key;

	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	rv = find_symmetric_key(session, &key, aes_id);
	if (rv != CKR_OK)
	{
		rv = generate_symmetric_key(session, key_length, aes_id, &key);
		Do_ADBG_Log("[Generate Key] %ld \n" , rv);
		if (!ADBG_EXPECT_CK_OK(c, rv)) {
			goto out;
		}
		Do_ADBG_Log("[Generate Key pass.]");
	}

	/* AES mechanism */
	CK_MECHANISM_TYPE aes_allow_mech = CKM_MTK_HSM_AES_ECB;

	CK_MECHANISM mechanism = {0, NULL, 0}; /* Default vals */
	/* AES mechanism */
	mechanism.mechanism = aes_allow_mech;
	mech_type = "CKM_AES_ECB";

	unsigned char cipher[BUFFERSIZ];
	CK_ULONG cipher_len = BUFFERSIZ;

	Do_ADBG_Log("  C_EncryptInit  \n");
	/* Initialize the encryption operation in the session */
	rv = C_EncryptInit(session, &mechanism, key);

	if (rv != CKR_OK) {
		Do_ADBG_Log("[1008 Fail]C_EncryptInit: rv = 0x%lX\n", rv);
		goto out;
	}

	// Encrypt one-shot
	Do_ADBG_Log("  C_Encrypt  \n");
	/* Do the encryption */
	rv = C_Encrypt(session, (CK_BYTE_PTR)aes_msg, sizeof(aes_msg),
				   (CK_BYTE_PTR)cipher, &cipher_len);
	if (rv != CKR_OK) {
		Do_ADBG_Log("%s : Encrypt(): %lu", mech_type, rv);
		goto out;
	}

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 1098, xtest_hsm_pkcs_11_test_1098,
                 "PKCS11: Check operations release at session closure");


static void xtest_hsm_pkcs_11_test_7001(ADBG_Case_t *c)
{
	CK_RV rv = CKR_GENERAL_ERROR;
	CK_SLOT_ID slot = 0;
	CK_SESSION_HANDLE session = CK_INVALID_HANDLE;
	CK_FLAGS session_flags = CKF_RW_SESSION | CKF_SERIAL_SESSION;
	CK_ULONG out_len = 0;

	CK_MECHANISM mechanism = {CKM_MTK_HSM_DUMP_LOG, NULL, 0}; /* Default vals */

	Do_ADBG_Log("======== 7001 Test ========");

	/* Initial */
	rv = init_lib_and_find_token_slot(&slot);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		return;

	/* open session */
	rv = C_OpenSession(slot, session_flags, NULL, 0, &session);
	if (!ADBG_EXPECT_CK_OK(c, rv))
		goto out;

	C_Utils(session, &mechanism, NULL, 0, NULL, &out_len);

out:
	rv = C_CloseSession(session);
	ADBG_EXPECT_CK_OK(c, rv);

	rv = close_lib();
	ADBG_EXPECT_CK_OK(c, rv);

}
ADBG_CASE_DEFINE(hsm_pkcs_11, 7001, xtest_hsm_pkcs_11_test_7001,
		"Dump HSM log");