/*
 * Copyright (c) 2014, STMicroelectronics International N.V.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include "testframework.h"

#define MOD_RED(opstr, nstr, resstr)             \
do {                                             \
	TEE_BigIntConvertFromString(op, opstr);  \
	TEE_BigIntConvertFromString(n, nstr);    \
	TEE_BigIntMod(r, op, n);                 \
	TB_ASSERT_HEX_PRINT_VALUE(r, resstr);    \
} while (0)

static void test_modular_reduction(void)
{
	TB_INFO("   Testing modular reduction");

	DEF_BIGINT(op, 2048);
	DEF_BIGINT(n, 2048);
	DEF_BIGINT(r, 2048);

	MOD_RED("10", "A", "6");
	MOD_RED("10", "20", "10");
	MOD_RED("-1", "123", "122");
	MOD_RED("-122", "123", "1");
	MOD_RED("-1", "1_0000_0000_0000_0000_0000", "FFFFFFFFFFFFFFFFFFFF");

	MOD_RED("42176E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A",
		"A9A9C49F0015DBEF9EB0D927F379821AE2C59",
		"3460EB99F07BC4617A7586E6522B98F66E5B2");

	MOD_RED("42176E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A",
		"1234", "1082");

	MOD_RED("FFFF6E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A",
		"2", "0");

	MOD_RED("FFFF6E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A",
		"FFFF6E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A",
		"0");

	MOD_RED("FFFF6E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A",
		"FFFF6E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828B",
		"FFFF6E7CD0386C94194C3C3DACFA7237"
		"CCF4E0829E2EBEAB59DE12CAE3305C9E"
		"576F837A42D9C13BB1D83E4DFD330B65"
		"76FD516B2AE9FBDD152F0A820C5DD18E"
		"D048D476EC791DC989D6D8C30B2B30FF"
		"C2A4ABAC082C672B38E138610B74CE0C"
		"30D1B378BE6FE0C44E09EC11586245CD"
		"8F1216619392E6B2C857C71EA78DD4D8"
		"61A482D9D1251C341D1D6205AFB2BFB5"
		"3DD1F272B5624FA61B020936745387CD"
		"F1F3EBC4435CBE6C6B323497098FDE9D"
		"61AA0CC87416235E706A71F73EC3C291"
		"D9ACA2EECD712DA40966DD251310AE94"
		"9713F69B2A32E42732A7F6669547D88A"
		"415B25C582FDBBF06C5EC5F5750053D4"
		"9F7ED68DED168C92F9C531828A");

	DEL_BIGINT(op);
	DEL_BIGINT(n);
	DEL_BIGINT(r);
}

#define MOD_ADDSUB(op1str, op2str, nstr, result)          \
	do {                                              \
		TEE_BigIntConvertFromString(op1, op1str); \
		TEE_BigIntConvertFromString(op2, op2str); \
		TEE_BigIntConvertFromString(n, nstr);     \
		TEE_BigIntAddMod(r, op1, op2, n);         \
		TB_ASSERT_HEX_PRINT_VALUE(r, result);     \
		 TEE_BigIntSubMod(tmp1, r, op2, n);       \
		TB_ASSERT_BIGINT_EQ(tmp1, op1);           \
	} while (0)

static void test_modular_addsub(void)
{
	const char *N2048str = "F9000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "000000000000000000000000000005C5";

	TB_INFO("   Testing modular addition and subtraction");

	DEF_BIGINT(op1, 2048);
	DEF_BIGINT(op2, 2048);
	DEF_BIGINT(n, 2048);
	DEF_BIGINT(r, 2048);
	DEF_BIGINT(tmp1, 2048);

	MOD_ADDSUB("1", "1", "2", "0");
	MOD_ADDSUB("123", "123", "200", "46");
	MOD_ADDSUB("0", "123", "200", "123");
	MOD_ADDSUB("123", "0", "200", "123");

	/* Testing some 2048 bit numbers */
	MOD_ADDSUB("F9000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   ,
		   "F9000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000001",
		   N2048str,
		   "F8FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
		   "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFA3C");

	MOD_ADDSUB("F9000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000"
		   "00000000000000000000000000000000",
		   "5C5", N2048str, "0");

	DEL_BIGINT(op1);
	DEL_BIGINT(op2);
	DEL_BIGINT(n);
	DEL_BIGINT(r);
	DEL_BIGINT(tmp1);

}

#define MOD_MUL(op1str, op2str, nstr, result)     \
do {                                              \
	TEE_BigIntConvertFromString(op1, op1str); \
	TEE_BigIntConvertFromString(op2, op2str); \
	TEE_BigIntConvertFromString(n, nstr);     \
	TEE_BigIntMulMod(r, op1, op2, n);         \
	TB_ASSERT_HEX_PRINT_VALUE(r, result);     \
} while (0)

static void test_modular_mul(void)
{
	const char *N2048str = "F9000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "00000000000000000000000000000000"
			       "000000000000000000000000000005C5";

	TB_INFO("   Testing modular multiplication");

	DEF_BIGINT(op1, 2048);
	DEF_BIGINT(op2, 2048);
	DEF_BIGINT(n, 2048);
	DEF_BIGINT(r, 2048);
	DEF_BIGINT(tmp1, 2048);

	MOD_MUL("0", "0", "2", "0");
	MOD_MUL("1", "1", "2", "1");
	MOD_MUL("123", "123", "200", "C9");

	MOD_MUL("F9000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		,
		"F9000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000000"
		"00000000000000000000000000000001",
		N2048str, "2143D4");

	DEL_BIGINT(op1);
	DEL_BIGINT(op2);
	DEL_BIGINT(n);
	DEL_BIGINT(r);
	DEL_BIGINT(tmp1);
}

#define MOD_INV(op1str, nstr, result)               \
do {                                                \
	TEE_BigIntConvertFromString(op1, op1str);   \
	TEE_BigIntConvertFromString(n, nstr);       \
	TEE_BigIntInvMod(r, op1, n);                \
	TB_ASSERT_HEX_PRINT_VALUE(r, result);       \
	TEE_BigIntMulMod(tmp1, r, op1, n);          \
	TB_ASSERT_HEX_PRINT_VALUE(tmp1, "1");       \
	TEE_BigIntInvMod(tmp1, r, n);               \
	TB_ASSERT_HEX_PRINT_VALUE(tmp1, op1str);    \
} while (0)

static void test_modular_inv(void)
{
	const char *N2048str =
	    "F9000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "00000000000000000000000000000000"
	    "000000000000000000000000000005C5";

	TB_INFO("   Testing modular inversion");

	DEF_BIGINT(op1, 2048);
	DEF_BIGINT(op2, 2048);
	DEF_BIGINT(n, 2048);
	DEF_BIGINT(r, 2048);
	DEF_BIGINT(tmp1, 2048);

	MOD_INV("2", "3", "2");
	MOD_INV("3", "7", "5");
	MOD_INV("5", "7", "3");
	MOD_INV("7B", "FA1", "8E5");
	MOD_INV("462D578CF0F88", "AB54A98CEB1F0AD3", "5ABBF22B54DBD000");
	MOD_INV("7", N2048str,
		"B1DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6"
		"DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB"
		"6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6D"
		"B6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6"
		"DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB"
		"6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6D"
		"B6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6"
		"DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB"
		"6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6D"
		"B6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6"
		"DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB"
		"6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6D"
		"B6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6"
		"DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB"
		"6DB6DB6DB6DB6DB6DB6DB6DB6DB6DB6D"
		"B6DB6DB6DB6DB6DB6DB6DB6DB6DB71D6");

	DEL_BIGINT(op1);
	DEL_BIGINT(op2);
	DEL_BIGINT(n);
	DEL_BIGINT(r);
	DEL_BIGINT(tmp1);
}

void tb_modulus(void)
{
	const char *TEST_NAME = "Modular arithmetic";

	TB_HEADER(TEST_NAME);
	test_modular_reduction();
	test_modular_addsub();
	test_modular_mul();
	test_modular_inv();
	TB_FOOTER(TEST_NAME);
}
