// SPDX-License-Identifier: BSD-2-Clause
/*
 * Copyright (c) 2016, Linaro Limited
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <err.h>
#include <stdio.h>
#include <string.h>
/* OP-TEE TEE client API (built by optee_client) */
#include <tee_client_api.h>

/* To the the UUID (found the the TA's h-file(s)) */
#include <vm_map_ta.h>
#include <stdlib.h>

TEEC_Result call_ta(uint32_t cmd, uint64_t *inout1, uint64_t *intout2, uint32_t attr)
{
	TEEC_Result res;
	TEEC_Context ctx;
	TEEC_Session sess;
	TEEC_Operation op;
	TEEC_UUID uuid = TA_VM_MAP_UUID;
	uint32_t err_origin;

	/* Initialize a context connecting us to the TEE */
	res = TEEC_InitializeContext(NULL, &ctx);
	if (res != TEEC_SUCCESS)
		errx(1, "TEEC_InitializeContext failed with code 0x%x", res);

	/*
	 * Open a session to the "hello world" TA, the TA will print "hello
	 * world!" in the log when the session is created.
	 */
	res = TEEC_OpenSession(&ctx, &sess, &uuid,
			       TEEC_LOGIN_PUBLIC, NULL, NULL, &err_origin);
	if (res != TEEC_SUCCESS)
		errx(1, "TEEC_Opensession failed with code 0x%x origin 0x%x",
			res, err_origin);

	/*
	 * Execute a function in the TA by invoking it, in this case
	 * we're incrementing a number.
	 *
	 * The value of command ID part and how the parameters are
	 * interpreted is part of the interface provided by the TA.
	 */

	/* Clear the TEEC_Operation struct */
	memset(&op, 0, sizeof(op));

	/*
	 * Prepare the argument. Pass a value in the first parameter,
	 * the remaining three parameters are unused.
	 */
	op.paramTypes = TEEC_PARAM_TYPES(TEEC_VALUE_INOUT, TEEC_VALUE_INOUT,
					 TEEC_VALUE_INPUT, TEEC_VALUE_INPUT);
	op.params[0].value.a = *inout1;
	op.params[1].value.a = *intout2;
	op.params[2].value.a = attr;
	op.params[3].value.a = *inout1>>32;

	printf("TEEC_InvokeCommand cmd:%x, inout1:%x, inout2:%x\n", cmd, op.params[0].value.a,op.params[1].value.a);
	res = TEEC_InvokeCommand(&sess, cmd, &op,
				 &err_origin);
	printf("TEEC_InvokeCommand res:%x, inout1:%x, inout2:%x\n", res, op.params[0].value.a,op.params[1].value.a);

	TEEC_CloseSession(&sess);

	TEEC_FinalizeContext(&ctx);

	return res;
}

#define CHECK_RESULT(res, exp, info) \
	do { \
		if (res!=exp){ \
			fprintf(stderr, "[%s:%d]Error: %s expect 0x%x bug got 0x%x.\n", __FUNCTION__, __LINE__, info, exp, res); \
		}else{ \
			printf("Success: %s\n", info); \
		} \
	}while(0)

void test_all_default(){
	TEEC_Result res = TEEC_SUCCESS;
	uint64_t inout1 = 0x47000000;
	uint64_t inout2 = 0x10000;
	uint32_t attr = 1;
	res = call_ta(TA_VM_MAP_CMD_MPU_PROTECT, &inout1, &inout2, attr);
	CHECK_RESULT(res, TEEC_SUCCESS, "EMI MPU protect");
	inout1 = 0x47000000;
	inout2 = 0x1000000;
	attr = 0;
	res = call_ta(TA_VM_MAP_CMD_MEM_MAP_RANDOM, &inout1, &inout2, attr);
	CHECK_RESULT(res, TEEC_SUCCESS, "EMI memory map/umap");
	inout1 = 0x47000000;
	inout2 = 0x1000000;
	res = call_ta(TA_VM_MAP_CMD_MEM_MAP_TWICE, &inout1, &inout2, attr);
	CHECK_RESULT(res, TEEC_SUCCESS, "EMI memory map twice");
	inout1 = 0x47000000;
	inout2 = 0x1000000;
	res = call_ta(TA_VM_MAP_CMD_MEM_MAP_TWO_REGION, &inout1, &inout2, attr);
	CHECK_RESULT(res, TEEC_ERROR_BAD_PARAMETERS, "EMI memory map two region");
	inout1 = 0x47000000;
	inout2 = 0x10000;
	res = call_ta(TA_VM_MAP_CMD_MPU_UNPROTECT, &inout1, &inout2, attr);
	CHECK_RESULT(res, TEEC_SUCCESS, "EMI MPU unprotect");
}

int main(int argc, char *argv[])
{
	int i;
	uint32_t err_origin;
	uint64_t start_addr=0;
	size_t   len=0;
	uint32_t attr=0;
	TEEC_Result res = TEEC_SUCCESS;
	printf("argc %d \n", argc);
	for (i=0; i<argc; i++){
		printf("%d is %s\n", i, argv[i]);
	}
	if (argc == 3){
		start_addr = strtoull(argv[2], NULL, 0);
		printf("Input start_addr:0x%llx\n", start_addr);
		if (0==memcmp(argv[1],"unprotect",strlen("unprotect"))){
            res = call_ta(TA_VM_MAP_CMD_MPU_UNPROTECT, &start_addr, &len, attr);
			CHECK_RESULT(res, TEEC_SUCCESS, "EMI MPU unprotect");
		}
		if (0==memcmp(argv[1],"unmap",strlen("unmap"))){
            res = call_ta(TA_VM_MAP_CMD_MEM_UNMAP, &start_addr, &len, attr);
			CHECK_RESULT(res, TEEC_SUCCESS, "EMI memory map unmap");
		}
	}else if (argc == 5){
		start_addr = strtoull(argv[2], NULL, 0);
		len = strtoul(argv[3], NULL, 0);
		attr = strtoul(argv[4], NULL, 0);
		printf("Input start_addr:0x%llx\n", start_addr);
		printf("Input start_addr:0x%llx len:0x%llx, attr:0x%x\n", start_addr, len, attr);
		if (0==memcmp(argv[1],"protect",strlen("protect"))){
            res = call_ta(TA_VM_MAP_CMD_MPU_PROTECT, &start_addr, &len, attr);
			CHECK_RESULT(res, TEEC_SUCCESS, "EMI MPU protect");
		}
		if (0==memcmp(argv[1],"map",strlen("map"))){
            res = call_ta(TA_VM_MAP_CMD_MEM_MAP, &start_addr, &len, attr);
			CHECK_RESULT(res, TEEC_SUCCESS, "");
		}
		if (0==memcmp(argv[1],"simple",strlen("simple"))){
            res = call_ta(TA_VM_MAP_CMD_MEM_MAP_SIMPLE, &start_addr, &len, attr);
			CHECK_RESULT(res, TEEC_SUCCESS, "");
		}
	}else{
		test_all_default();
	}

	return 0;
}
