// SPDX-License-Identifier: BSD-2-Clause
/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * MediaTek Inc. (C) 2020. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */

#include "processing_mtk_hsm.h"
#include "hsm_mbox_if.h"
#include "mtk_crypto.h"
#include "hsm_job.h"
#include "sks_hsm_debug.h"

#define GET_BE32(a) ((((uint32_t) (a)[0]) << 24) | \
                    (((uint32_t) (a)[1]) << 16) | \
                    (((uint32_t) (a)[2]) << 8) | \
                    ((uint32_t) (a)[3]))

#define PUT_BE32(a, val) {	\
		(a)[0] = (uint8_t) ((((uint32_t) (val)) >> 24) & 0xff);	\
		(a)[1] = (uint8_t) ((((uint32_t) (val)) >> 16) & 0xff);	\
		(a)[2] = (uint8_t) ((((uint32_t) (val)) >> 8) & 0xff);	\
		(a)[3] = (uint8_t) (((uint32_t) (val)) & 0xff);		\
}

#define PUT_BE64(a, val) { \
		(a)[0] = (uint8_t) (((uint64_t) (val)) >> 56 & 0xff);	\
		(a)[1] = (uint8_t) (((uint64_t) (val)) >> 48 & 0xff);	\
		(a)[2] = (uint8_t) (((uint64_t) (val)) >> 40 & 0xff);	\
		(a)[3] = (uint8_t) (((uint64_t) (val)) >> 32 & 0xff);	\
		(a)[4] = (uint8_t) (((uint64_t) (val)) >> 24 & 0xff);	\
		(a)[5] = (uint8_t) (((uint64_t) (val)) >> 16 & 0xff);	\
		(a)[6] = (uint8_t) (((uint64_t) (val)) >> 8 & 0xff);	\
		(a)[7] = (uint8_t) (((uint64_t) (val)) & 0xff);	\
}

#define LOOKBIT(x) (1UL << (x))

extern  TEE_Result Send_job(job_struct *pjob);
extern  TEE_Result dump_hsm_log(void);

void __SHOW_VAL(char *c, uint8_t *ptr, uint32_t len);
void __SHOW_VAL(char *c, uint8_t *ptr, uint32_t len)
{
#if HSM_TA_DEBUG

	uint8_t line[8] = {0};
	SKS_HSM_DEBUG(" --- %s ( %d ) bytes ---\r\n", c, len);
	int j = 0;
	for (j = 0; j < len; j++)
	{
		line[j % 8] = *(ptr + j);
		if ((j % 8) == 7)
			SKS_HSM_DEBUG(" %02X %02X %02X %02X - %02X %02X %02X %02X  \r\n"
				 , line[0], line[1], line[2], line[3]
				 , line[4], line[5], line[6], line[7]);
	}
	if ((j % 8) != 0)
	{
		switch (j % 8)
		{
		case 1: SKS_HSM_DEBUG(" %02X -- -- -- - -- -- -- --  \r\n", line[0]); break;
		case 2: SKS_HSM_DEBUG(" %02X %02X -- -- - -- -- -- --  \r\n", line[0], line[1]); break;
		case 3: SKS_HSM_DEBUG(" %02X %02X %02X -- - -- -- -- --  \r\n", line[0], line[1], line[2]); break;
		case 4: SKS_HSM_DEBUG(" %02X %02X %02X %02X - -- -- -- --  \r\n", line[0], line[1], line[2], line[3]); break;
		case 5: SKS_HSM_DEBUG(" %02X %02X %02X %02X - %02X -- -- --  \r\n", line[0], line[1], line[2], line[3], line[4]); break;
		case 6: SKS_HSM_DEBUG(" %02X %02X %02X %02X - %02X %02X -- --  \r\n", line[0], line[1], line[2], line[3], line[4], line[5]); break;
		case 7: SKS_HSM_DEBUG(" %02X %02X %02X %02X - %02X %02X %02X --  \r\n", line[0], line[1], line[2], line[3], line[4], line[5], line[6]); break;
		}
	}
	SKS_HSM_DEBUG(" --------------------\r\n");

#endif
}

static void inc32(uint8_t *block, uint32_t i)
{
	uint32_t val = 0;
	uint32_t org_val = 0;

	val = GET_BE32(block + 12);
	org_val = val;
	val += i;
	PUT_BE32(block + 12, val);

	if (org_val > val) // Overflow
	{
		for (int i = 11; i >= 0; i--)
		{
			block[i]++;
			if (block[i] != 0)
				break;
		}
	}
}

static void xor_block(uint8_t *dst, const uint8_t *src)
{
	uint32_t *d = (uint32_t *) dst;
	uint32_t *s = (uint32_t *) src;
	*d++ ^= *s++;
	*d++ ^= *s++;
	*d++ ^= *s++;
	*d++ ^= *s++;
}

static void shift_right_block(uint8_t *v)
{
	uint32_t val = 0;

	val = GET_BE32(v + 12);
	val >>= 1;
	if (v[11] & 0x01)
		val |= 0x80000000;
	PUT_BE32(v + 12, val);

	val = GET_BE32(v + 8);
	val >>= 1;
	if (v[7] & 0x01)
		val |= 0x80000000;
	PUT_BE32(v + 8, val);

	val = GET_BE32(v + 4);
	val >>= 1;
	if (v[3] & 0x01)
		val |= 0x80000000;
	PUT_BE32(v + 4, val);

	val = GET_BE32(v);
	val >>= 1;
	PUT_BE32(v, val);
}


/* Multiplication in GF(2^128) */
static void gf_mult(const uint8_t *x, const uint8_t *y, uint8_t *z)
{
	uint8_t v[16] = {0};
	int i = 0;
	int j = 0;

	memset(z, 0, 16); /* Z_0 = 0^128 */
	memcpy(v, y, 16); /* V_0 = Y */

	for (i = 0; i < 16; i++) {
		for (j = 0; j < 8; j++) {
			if (x[i] & LOOKBIT(7 - j)) {
				/* Z_(i + 1) = Z_i XOR V_i */
				xor_block(z, v);
			} else {
				/* Z_(i + 1) = Z_i */
			}

			if (v[15] & 0x01) {
				/* V_(i + 1) = (V_i >> 1) XOR R */
				shift_right_block(v);
				/* R = 11100001 || 0^120 */
				v[0] ^= 0xe1;
			} else {
				/* V_(i + 1) = V_i >> 1 */
				shift_right_block(v);
			}
		}
	}
}


static void ghash_start(uint8_t *y)
{
	/* Y_0 = 0^128 */
	memset(y, 0, 16);
}


static void ghash(const uint8_t *h, const uint8_t *x, size_t xlen, uint8_t *y)
{
	size_t m = 0;
	size_t i = 0;
	const uint8_t *xpos = x;
	uint8_t tmp[16] = {0};

	m = xlen / 16;

	for (i = 0; i < m; i++) {
		/* Y_i = (Y^(i-1) XOR X_i) dot H */
		xor_block(y, xpos);
		xpos += 16;

		/* dot operation:
		 * multiplication operation for binary Galois (finite) field of
		 * 2^128 elements */
		gf_mult(y, h, tmp);
		memcpy(y, tmp, 16);
	}

	if (x + xlen > xpos) {
		/* Add zero padded last block */
		size_t last = x + xlen - xpos;
		memcpy(tmp, xpos, last);
		memset(tmp + last, 0, sizeof(tmp) - last);

		/* Y_i = (Y^(i-1) XOR X_i) dot H */
		xor_block(y, tmp);

		/* dot operation:
		 * multiplication operation for binary Galois (finite) field of
		 * 2^128 elements */
		gf_mult(y, h, tmp);
		memcpy(y, tmp, 16);
	}

	/* Return Y_m */
}


static void aes_gcm_calculate_j0(const uint8_t *H, const uint8_t *iv, size_t iv_len, uint8_t *J0)
{
	uint8_t len_buf[16] = {0};

	if (iv_len == 12)
	{
		/* calculate block J_0 = IV || 0^31 || 1 */
		memcpy(J0, iv, iv_len);
		memset(J0 + iv_len, 0, 16 - iv_len);
		J0[15] = 0x01;
	}
	else
	{
		/*
		 * s = 128 * ceil(len(IV)/128) - len(IV)
		 * J_0 = GHASH_H(IV || 0^(s+64) || [len(IV)]_64)
		 */
		ghash_start(J0);
		ghash(H, iv, iv_len, J0);
		PUT_BE64(len_buf, 0);
		PUT_BE64(len_buf + 8, iv_len * 8);
		ghash(H, len_buf, sizeof(len_buf), J0);
	}
}


static void aes_gcm_ghash(const uint8_t *H, const uint8_t *aad, size_t aad_len,
                          const uint8_t *crypt, size_t crypt_len, uint8_t *S)
{
	uint8_t len_buf[16] = {0};

	/*
	 * u = 128 * ceil[len(C)/128] - len(C)
	 * v = 128 * ceil[len(A)/128] - len(A)
	 * S = GHASH_H(A || 0^v || C || 0^u || [len(A)]64 || [len(C)]64)
	 * (i.e., zero padded to block size A || C and lengths of each in bits)
	 */
	ghash_start(S);
	ghash(H, aad, aad_len, S);
	ghash(H, crypt, crypt_len, S);
	PUT_BE64(len_buf, aad_len * 8);
	PUT_BE64(len_buf + 8, crypt_len * 8);
	ghash(H, len_buf, sizeof(len_buf), S);
}

static uint32_t get_symmetric_key_from_attribute(struct sks_object *key, MTK_HSM_State *state)
{
	int key_id = 0;
	int key_id_size = sizeof(key_id);
	int key_len = 0;
	int key_len_size = sizeof(key_len);
	uint32_t rv = 0;

	if ((key == NULL) || (state == NULL))
		 return SKS_CKR_ARGUMENTS_BAD;

	rv = get_attribute(key->attributes, SKS_CKA_HSM_KEY_ID, &key_id, &key_id_size);
	if (rv != 0)
		return rv;
	rv = get_attribute(key->attributes, SKS_CKA_HSM_KEY_LEN, &key_len, &key_len_size);
	if (rv != 0)
		return rv;

	state->key1_id = key_id;
	state->key1_len = key_len;

	SKS_HSM_DEBUG(" ==> state->key1_len: %d\n", state->key1_len);
	return SKS_OK;
}

uint32_t mtk_AES_ECB_init(struct pkcs11_session *session,
                          struct sks_attribute_head *proc_params,
                          struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	uint32_t rv = 0;

	if ((session == NULL) || (proc_params == NULL) || (key == NULL))
		 return SKS_BAD_PARAM;

	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	// Get key
	rv = get_symmetric_key_from_attribute(key, _ptr);
	if(rv != SKS_OK)
		return rv;

	session->processing->mtk_hsm_handle = _ptr;


	return SKS_OK;
}

uint32_t mtk_AES_CBC_init(struct pkcs11_session *session,
						  struct sks_attribute_head *proc_params,
						  struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	uint8_t *IV = NULL;
	uint32_t rv = 0;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	IV = proc_params->data;

	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	memcpy(_ptr->CBC_IV, IV, 16);

	// Get key
	rv = get_symmetric_key_from_attribute(key, _ptr);
	if(rv != SKS_OK)
		return rv;

	session->processing->mtk_hsm_handle = _ptr;

	return SKS_OK;
}

uint32_t mtk_AES_CTR_init(struct pkcs11_session *session,
				struct sks_attribute_head *proc_params,
				struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	uint8_t *CTR = NULL;
	struct serialargs args;
	uint32_t rv = 0;
	/* CTR parameters */
	uint32_t incr_counter = 0;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	TEE_MemFill(&args, 0, sizeof(args));

	if (!proc_params)
		return SKS_BAD_PARAM;

	serialargs_init(&args, proc_params->data, proc_params->size);

	rv = serialargs_get(&args, &incr_counter, sizeof(uint32_t));
	if (rv)
		goto bail;

	rv = serialargs_get_ptr(&args, &CTR, 16);
	if (rv)
		goto bail;

	__SHOW_VAL("CTR", CTR, 16);


	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	memcpy(_ptr->CTR, CTR, 16);

	// Get key
	rv = get_symmetric_key_from_attribute(key, _ptr);
	if(rv != SKS_OK)
		return rv;

	session->processing->mtk_hsm_handle = _ptr;

bail:
	return SKS_OK;

}

uint32_t mtk_AES_GCM_init(struct pkcs11_session *session,
				struct sks_attribute_head *proc_params,
				struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	uint8_t *pIv = NULL;
	uint32_t ulIvLen = 0;
	uint8_t *pAAD = NULL;
	uint32_t ulAADLen = 0;
	uint8_t ulTagBits = 0;

	struct serialargs args;
	uint32_t rv = 0;
	/* GCM parameters */
	uint32_t incr_counter = 0;

	static const org_0[16] = {0};
	uint8_t H[16] = {0};
	job_struct _job = {0};
	uint8_t J0[16] = {0};
	uint8_t S[16] = {0};

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	TEE_MemFill(&args, 0, sizeof(args));

	if (!proc_params)
		return SKS_BAD_PARAM;

	serialargs_init(&args, proc_params->data, proc_params->size);

	rv = serialargs_get(&args, &ulIvLen, sizeof(uint32_t));
	if (rv)
		goto bail;

	rv = serialargs_get_ptr(&args, &pIv, ulIvLen);
	if (rv)
		goto bail;

	rv = serialargs_get(&args, &ulAADLen, sizeof(uint32_t));
	if (rv)
		goto bail;

	rv = serialargs_get_ptr(&args, &pAAD, ulAADLen);
	if (rv)
		goto bail;

	rv = serialargs_get(&args, &ulTagBits, sizeof(uint8_t));
	if (rv)
		goto bail;

	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	// Get key
	rv = get_symmetric_key_from_attribute(key, _ptr);
	if(rv != SKS_OK)
		return rv;

	/* H = ECB(0^128) */
	SKS_HSM_DEBUG("-> H = ECB(0^128)\r\n");

	_job.inputPtr               = org_0;
	_job.inputLength            = 16;
	_job.outputPtr              = H;
	_job.outputLength           = 16;

	_job.service		    = CRYPTO_ENCRYPT;
	_job.family                 = CRYPTO_ALGOFAM_AES;
	_job.mode                   = CRYPTO_ALGOMODE_ECB;
	_job.operation_mode         = CRYPTO_OPERATIONMODE_START;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;

	Send_job(&_job);

	/* calculate_j0 */
	SKS_HSM_DEBUG("-> Calculate j0 \r\n");
	aes_gcm_calculate_j0(H, pIv, ulIvLen, J0);
//	__SHOW_VAL("J0", J0, 16);

	/* calculate S of add data*/
	ghash(H, pAAD, ulAADLen, S);

	memcpy(_ptr->H,   H,  16);
	memcpy(_ptr->S,   S,  16);
	memcpy(_ptr->J0,  J0, 16);
	inc32(J0, 1);
	memcpy(_ptr->CTR, J0, 16);
	_ptr->add_len = ulAADLen;
	_ptr->total_len = 0;

	session->processing->mtk_hsm_handle = _ptr;

bail:
	return SKS_OK;

}

uint32_t mtk_HMAC_init(struct pkcs11_session *session,
				struct sks_attribute_head *proc_params,
				struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	uint32_t rv = 0;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	 _ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	memset(_ptr->SHA_IV, 0, SHA_MAX_OUTPUT_SIZE);
	memset(_ptr->SHA_buf, 0, SHA_MAX_BLOCK_SIZE);
	_ptr->SHA_remain_len = 0;
	_ptr->isFirstUpdate  = true;

	// Get key
	rv = get_symmetric_key_from_attribute(key, _ptr);
	if(rv != SKS_OK)
		return rv;

	session->processing->mtk_hsm_handle = _ptr;

	return SKS_OK;
}

uint32_t mtk_AES_CMAC_init(struct pkcs11_session *session,
				struct sks_attribute_head *proc_params,
				struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	uint32_t rv = 0;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	memset(_ptr->CMAC_Buf, 0, 16);
	_ptr->isFirstUpdate    = true;

	// Get key
	rv = get_symmetric_key_from_attribute(key, _ptr);
	if(rv != SKS_OK)
		return rv;

	session->processing->mtk_hsm_handle = _ptr;

	return SKS_OK;
}

uint32_t mtk_SHA_init(struct pkcs11_session *session,
					  struct sks_attribute_head *proc_params)
{
	MTK_HSM_State *_ptr = NULL;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	 _ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	memset(_ptr->SHA_IV, 0, SHA_MAX_OUTPUT_SIZE);
	memset(_ptr->SHA_buf, 0, SHA_MAX_BLOCK_SIZE);
	_ptr->SHA_remain_len = 0;
	_ptr->isFirstUpdate  = true;

	session->processing->mtk_hsm_handle = _ptr;

	return SKS_OK;
}

uint32_t mtk_ECDSA_init(struct pkcs11_session *session,
						struct sks_attribute_head *proc_params,
						struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	int key_id = 0;
	int key_id_size = sizeof(key_id);
	int key_len = 0;
	int key_len_size = sizeof(key_len);
	uint32_t rv = 0;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	// Get key
	rv = get_attribute(key->attributes, SKS_CKA_HSM_KEY_ID, &key_id, &key_id_size);
	rv = get_attribute(key->attributes, SKS_CKA_HSM_KEY_LEN, &key_len, &key_len_size);

	session->processing->mtk_hsm_handle = _ptr;
	session->processing->mtk_hsm_handle->key1_id = key_id;
	session->processing->mtk_hsm_handle->key1_len = key_len;

	return SKS_OK;
}

uint32_t mtk_ECDSA_SHA_init(struct pkcs11_session *session,
						struct sks_attribute_head *proc_params,
						struct sks_object *key)
{
	MTK_HSM_State *_ptr = NULL;
	int key_id = 0;
	int key_id_size = sizeof(key_id);
	int key_len = 0;
	int key_len_size = sizeof(key_len);
	uint32_t rv = 0;

	if ((session == NULL) || (proc_params == NULL))
		 return SKS_BAD_PARAM;

	_ptr = TEE_Malloc(sizeof(MTK_HSM_State), TEE_MALLOC_FILL_ZERO);
	if (!_ptr)
		return SKS_MEMORY;

	memset(_ptr->SHA_IV, 0, SHA_MAX_OUTPUT_SIZE);
	memset(_ptr->SHA_buf, 0, SHA_MAX_BLOCK_SIZE);
	_ptr->SHA_remain_len = 0;
	_ptr->isFirstUpdate  = true;

	// Get key
	rv = get_attribute(key->attributes, SKS_CKA_HSM_KEY_ID, &key_id, &key_id_size);
	rv = get_attribute(key->attributes, SKS_CKA_HSM_KEY_LEN, &key_len, &key_len_size);

	session->processing->mtk_hsm_handle = _ptr;
	session->processing->mtk_hsm_handle->key1_id = key_id;
	session->processing->mtk_hsm_handle->key1_len = key_len;

	return SKS_OK;
}

enum Crypto_OperationModeType _Step_to_OPmode(enum processing_step step, uint8_t isFirstUpdate)
{
	switch (step)
	{
	case SKS_FUNC_STEP_ONESHOT:
		return CRYPTO_OPERATIONMODE_SINGLECALL;

	case SKS_FUNC_STEP_UPDATE:
		if (isFirstUpdate == true)
			return CRYPTO_OPERATIONMODE_START;
		else
			return CRYPTO_OPERATIONMODE_UPDATE;

	case SKS_FUNC_STEP_FINAL:
		return CRYPTO_OPERATIONMODE_FINISH;

	default:
		return CRYPTO_OPERATIONMODE_INVALID;
	}
}

uint32_t mtk_AES_ECB_step(struct pkcs11_session *session,
							  TEE_Param *in, TEE_Param *out,
							  enum processing_func function,
							  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	job_struct _job = {0};

	if (step == SKS_FUNC_STEP_FINAL)
		return SKS_OK;

	if ((session == NULL) || (out == NULL))
		return SKS_BAD_PARAM;

	_job.inputPtr               = in ? (uint8_t *)in->memref.buffer  : NULL;
	_job.inputLength            = in ? in->memref.size : 0;

	_job.outputPtr              = out ? (uint8_t *)out->memref.buffer : NULL;
	_job.outputLength           = out ? out->memref.size : 0;

	if (function == SKS_FUNCTION_ENCRYPT)
		_job.service            = CRYPTO_ENCRYPT;
	else // SKS_FUNCTION_DECRYPT
		_job.service            = CRYPTO_DECRYPT;

	_job.family                 = CRYPTO_ALGOFAM_AES;
	_job.mode                   = CRYPTO_ALGOMODE_ECB;
	_job.operation_mode         = CRYPTO_OPERATIONMODE_START;


	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;


	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	return SKS_OK;
}

uint32_t mtk_AES_CBC_step(struct pkcs11_session *session,
						  TEE_Param *in, TEE_Param *out,
						  enum processing_func function,
						  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	uint8_t next_IV[16] = {0};  // Used by decryption
	job_struct _job = {0};
	uint8_t *in_buf = NULL;
	uint8_t *out_buf = NULL;

	if (step == SKS_FUNC_STEP_FINAL)
		return SKS_OK;

	if ((session == NULL) || (out == NULL))
		return SKS_BAD_PARAM;

	_job.inputPtr               = in ? (uint8_t *)in->memref.buffer  : NULL;
	_job.inputLength            = in ? in->memref.size : 0;
	_job.secondaryInputPtr      = (uint8_t *)_ptr->CBC_IV;
	_job.secondaryInputLength   = 16;

	_job.outputPtr              = out ? (uint8_t *)out->memref.buffer : NULL;
	_job.outputLength           = out ? out->memref.size : 0;
	if (function == SKS_FUNCTION_ENCRYPT)
		_job.service            = CRYPTO_ENCRYPT;
	else // SKS_FUNCTION_DECRYPT
		_job.service            = CRYPTO_DECRYPT;

	_job.family                 = CRYPTO_ALGOFAM_AES;
	_job.mode                   = CRYPTO_ALGOMODE_CBC;
	_job.operation_mode         = CRYPTO_OPERATIONMODE_START;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;

	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	SKS_HSM_DEBUG(" ==> _job.secondaryInputLength: %d\n", _job.secondaryInputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	in_buf = (uint8_t *)_job.inputPtr;
	out_buf = (uint8_t *)_job.outputPtr;
	if (step == SKS_FUNC_STEP_UPDATE)
	{
		// Update IV[]
		for (int i = 16; i >= 1; i--)
		{
			if (function == SKS_FUNCTION_ENCRYPT)
				_ptr->CBC_IV[16 - i] = out_buf[_job.outputLength - i];
			else
				_ptr->CBC_IV[16 - i] = in_buf[_job.outputLength - i];
		}
	}

	return SKS_OK;
}

uint32_t mtk_AES_CTR_step(struct pkcs11_session *session,
						  TEE_Param *in, TEE_Param *out,
						  enum processing_func function,
						  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	job_struct _job = {0};

	if (step == SKS_FUNC_STEP_FINAL)
		return SKS_OK;

	if ((session == NULL) || (in == NULL) || (out == NULL))
		return SKS_BAD_PARAM;

	_job.inputPtr               = in ? (uint8_t *)in->memref.buffer  : NULL;
	_job.inputLength            = in ? in->memref.size : 0;
	_job.secondaryInputPtr      = (uint8_t *)_ptr->CTR;
	_job.secondaryInputLength   = 16;

	_job.outputPtr              = out ? (uint8_t *)out->memref.buffer : NULL;
	_job.outputLength           = out ? out->memref.size : 0;
	if (function == SKS_FUNCTION_ENCRYPT)
		_job.service            = CRYPTO_ENCRYPT;
	else // SKS_FUNCTION_DECRYPT
		_job.service            = CRYPTO_DECRYPT;

	_job.family                 = CRYPTO_ALGOFAM_AES;
	_job.mode                   = CRYPTO_ALGOMODE_CTR;
	_job.operation_mode         = CRYPTO_OPERATIONMODE_START;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;


	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	SKS_HSM_DEBUG(" ==> _job.secondaryInputLength: %d\n", _job.secondaryInputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	// Update CTR ! _ptr->CTR[] += (_job.inputLength / 16);
	if (step == SKS_FUNC_STEP_UPDATE)
	{
		// Update CTR !!!!
		inc32(_ptr->CTR, (_job.inputLength / 16));
	}

	return SKS_OK;
}

uint32_t mtk_AES_GCM_step(struct pkcs11_session *session,
				  TEE_Param *in, TEE_Param *out,
				  enum processing_func function,
				  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	uint8_t tmp_S[64] = {0};
	uint8_t len_buf[16] = {0};
	job_struct _job = {0};

	SKS_HSM_DEBUG(" mtk_AES_GCM_step!! function: %d, step: %d  \n", function, step);

	memcpy(tmp_S, _ptr->S, 16);

	if (step != SKS_FUNC_STEP_FINAL)
	{
		_ptr->total_len += in->memref.size;

		if (function == SKS_FUNCTION_DECRYPT)	// Hash before decrypt
		{
			ghash(_ptr->H, in->memref.buffer, in->memref.size, tmp_S);
//			__SHOW_VAL("Ghash Decrypt input", out->memref.buffer, in->memref.size);
//			__SHOW_VAL("Decrypt S", tmp_S, 16);
		}

		mtk_AES_CTR_step(session, in, out, function, step);

		if (function == SKS_FUNCTION_ENCRYPT)   // Hash after encrypt
		{
			ghash(_ptr->H, out->memref.buffer, in->memref.size, tmp_S);
//			__SHOW_VAL("Ghash Encrypt input", out->memref.buffer, in->memref.size);
//			__SHOW_VAL("Encrypt S", tmp_S, 16);
		}
	}

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		memcpy(_ptr->S, tmp_S, 16);
	}
	else // (step == SKS_FUNC_STEP_FINAL) or (step == SKS_FUNC_STEP_ONESHOT)
	{
		PUT_BE64(len_buf, (_ptr->add_len) * 8);
		PUT_BE64(len_buf + 8, _ptr->total_len * 8);
		ghash(_ptr->H, len_buf, sizeof(len_buf), tmp_S);

		/* T = MSB_t(GCTR_K(J_0, S)) */
		_job.inputPtr               = tmp_S;
		_job.inputLength            = 16;
		_job.secondaryInputPtr      = _ptr->J0;
		_job.secondaryInputLength   = 16;
		_job.outputPtr              = len_buf;
		_job.outputLength           = 16;

		_job.service		    = CRYPTO_ENCRYPT;
		_job.family                 = CRYPTO_ALGOFAM_AES;
		_job.mode                   = CRYPTO_ALGOMODE_CTR;
		_job.operation_mode         = CRYPTO_OPERATIONMODE_START;

		_job.cryptoKeyId            = _ptr->key1_id;
		_job.keyLength              = _ptr->key1_len * 8;

		Send_job(&_job);

		__SHOW_VAL("Fianl tag", len_buf, 16);

		if (step == SKS_FUNC_STEP_ONESHOT)
			memcpy(&out->memref.buffer[out->memref.size - 16], len_buf, 16);
		else
			memcpy(out->memref.buffer, len_buf, 16);

	}

	return SKS_OK;
}

uint32_t mtk_AES_CMAC_step(struct pkcs11_session *session,
						   TEE_Param *in, TEE_Param *inout,
						   enum processing_func function,
						   enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	job_struct _job = {0};
	uint8_t veri_buf[64] = {0};
	uint8_t _buf[16] = {0};
	uint8_t *verify_buf = NULL;
	uint8_t *out_buf = NULL;

	if ((session == NULL))
		return SKS_BAD_PARAM;

	if (step == SKS_FUNC_STEP_ONESHOT)
	{
		_job.inputPtr               = in ? (uint8_t *)in->memref.buffer  : NULL;
		_job.inputLength            = in ? in->memref.size : 0;
	}
	else if (step == SKS_FUNC_STEP_UPDATE)
	{
		_job.inputPtr               = in ? (uint8_t *)in->memref.buffer  : NULL;
		_job.inputLength            = in ? in->memref.size : 0;

		if (_ptr->isFirstUpdate == true)
		{
			memcpy(_ptr->CMAC_Buf, &((uint8_t *)_job.inputPtr)[_job.inputLength - 16], 16);
			_job.inputLength -= 16;

			if (_job.inputLength == 0)
				return SKS_OK;
		}
		else
		{
			memcpy(_buf, &((uint8_t *)_job.inputPtr)[_job.inputLength - 16], 16);

			if (_job.inputLength > 16)
			{
				for (int i = _job.inputLength; i >= 16; i-- )
					((uint8_t *)_job.inputPtr)[i] = ((uint8_t *)_job.inputPtr)[i - 16];
			}

			memcpy((uint8_t *)_job.inputPtr, _ptr->CMAC_Buf, 16);
			memcpy(_ptr->CMAC_Buf, _buf, 16);
		}
	}
	else // (step == SKS_FUNC_STEP_FINAL)
	{
		_job.inputPtr               = _ptr->CMAC_Buf;
		_job.inputLength            = 16;
	}


	_job.secondaryInputPtr      = (uint8_t *)_ptr->CBC_IV;
	_job.secondaryInputLength   = 16;

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		_job.outputPtr              = veri_buf;
	}
	else
	{
		if (function == SKS_FUNCTION_VERIFY)
			_job.outputPtr  = veri_buf;
		else
			_job.outputPtr  = inout ? (uint8_t *)inout->memref.buffer : NULL;
	}
	_job.outputLength           = 16;

	_job.service                = CRYPTO_MACGENERATE;   // No matter the function is sign or verify, HSM do the same thing.
	_job.family                 = CRYPTO_ALGOFAM_AES;
	_job.mode                   = CRYPTO_ALGOMODE_CMAC;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;
	SKS_HSM_DEBUG(" ==> _job.keyLength: %d\n", _job.keyLength);


	_job.operation_mode         = _Step_to_OPmode(step, _ptr->isFirstUpdate);
	SKS_HSM_DEBUG(" step: %d --> op_mode: %d  \n", step, _job.operation_mode);

	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	if (step == SKS_FUNC_STEP_ONESHOT)
		verify_buf = inout ? (uint8_t *)inout->memref.buffer : NULL;
	else
		verify_buf = in ? (uint8_t *)in->memref.buffer : NULL;

	out_buf = (uint8_t *)_job.outputPtr;

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		// Update IV[]
		memcpy(_ptr->CBC_IV, out_buf, 16);
//		__SHOW_VAL("CMAC Updated IV[]", _ptr->CBC_IV, 16);
	}
	else
	{
		if (function == SKS_FUNCTION_VERIFY)
		{
			if (memcmp(verify_buf, out_buf, _job.outputLength) != 0)
			{
				SKS_HSM_ERROR(" CMAC%d verify ERROR!!  \n", _job.keyLength);
				return SKS_ERROR;
			}
		}
	}

	if (_ptr->isFirstUpdate == true)
		_ptr->isFirstUpdate = false;

	return SKS_OK;
}

static uint32_t _merge_remain_data(TEE_Param *in, TEE_Param *new_in, MTK_HSM_State *mtk_ptr, uint32_t _block_size)
{
	uint8_t *in_buf     = (uint8_t *)in->memref.buffer;
	uint32_t in_size    = in->memref.size;

	uint32_t rlen       = mtk_ptr->SHA_remain_len;
	uint32_t merged_len = in_size + rlen;

	if (merged_len < _block_size)
	{
		memcpy(&mtk_ptr->SHA_buf[rlen], in_buf, in_size);
		mtk_ptr->SHA_remain_len = merged_len;

		SKS_HSM_DEBUG("  ==> No copy, new rlen: %d   \n", mtk_ptr->SHA_remain_len);
		return SKS_OK;
	}

	if (rlen != 0)
	{
		new_in->memref.buffer = TEE_Malloc(merged_len, TEE_MALLOC_FILL_ZERO);
		if (!new_in->memref.buffer)
		{
			SKS_HSM_ERROR(" %s [%d] Memory allocate fail!!  \n", __FUNCTION__, __LINE__);
			return SKS_MEMORY;
		}

		memcpy(new_in->memref.buffer, mtk_ptr->SHA_buf, rlen);
		memcpy((&new_in->memref.buffer[rlen]), in_buf, in_size);
	}

	rlen    = merged_len % _block_size;
	in_size = merged_len - rlen;

	if (new_in->memref.buffer != NULL)
		memcpy(mtk_ptr->SHA_buf, &new_in->memref.buffer[in_size], rlen);
	else
		memcpy(mtk_ptr->SHA_buf, &in_buf[in_size], rlen);

	new_in->memref.size     = in_size;
	mtk_ptr->SHA_remain_len = rlen;
	SKS_HSM_DEBUG("  ==> new in_size: %d   \n", new_in->memref.size);
	SKS_HSM_DEBUG("  ==> new rlen: %d   \n", mtk_ptr->SHA_remain_len);

	__SHOW_VAL("Remain data", mtk_ptr->SHA_buf, mtk_ptr->SHA_remain_len);

	return SKS_OK;

}

#define SHA_MAX_OUTPUT_SIZE      (64)
uint32_t mtk_SHA_step(uint32_t mecha_type,
					  struct pkcs11_session *session,
					  TEE_Param *in, TEE_Param *out,
					  enum processing_func function,
					  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;

	job_struct _job = {0};
	uint8_t _block_size = 0;
	uint8_t tmp_SHA_block[SHA_MAX_OUTPUT_SIZE] = {0};

	TEE_Param new_in_body = {0};
	uint8_t *out_buf = NULL;

	switch (mecha_type)
	{
	case SKS_CKM_MTK_HSM_SHA1:
		_job.family = CRYPTO_ALGOFAM_SHA1;
		_block_size = 64;
		break;

	case SKS_CKM_MTK_HSM_SHA224:
		_job.family = CRYPTO_ALGOFAM_SHA2_224;
		_block_size = 64;
		break;

	case SKS_CKM_MTK_HSM_SHA256:
		_job.family = CRYPTO_ALGOFAM_SHA2_256;
		_block_size = 64;
		break;

	case SKS_CKM_MTK_HSM_SHA384:
		_job.family = CRYPTO_ALGOFAM_SHA2_384;
		_block_size = 128;
		break;

	case SKS_CKM_MTK_HSM_SHA512:
		_job.family = CRYPTO_ALGOFAM_SHA2_512;
		_block_size = 128;
		break;
	case SKS_CKM_MTK_HSM_WHP:
		_job.family = CRYPTO_ALGOFAM_WHP;
		_block_size = 128;
		break;
	default:
		return SKS_BAD_PARAM;
		break;
	}

	// Process the data left last time that is not block size alignment.
	if (step == SKS_FUNC_STEP_ONESHOT)
	{
		_job.inputPtr       = in ? (uint8_t *)in->memref.buffer  : NULL;
		_job.inputLength    = in ? in->memref.size : 0;
	}
	else if (step == SKS_FUNC_STEP_UPDATE)
	{
		if (_merge_remain_data(in, &new_in_body, _ptr, _block_size) != SKS_OK)
			return SKS_MEMORY;

		if (new_in_body.memref.size == 0)
		{
			return SKS_OK;
		}

		if (new_in_body.memref.buffer == NULL)
			_job.inputPtr       = in ? (uint8_t *)in->memref.buffer  : NULL;
		else
			_job.inputPtr       = new_in_body.memref.buffer;

		_job.inputLength    = new_in_body.memref.size;
	}
	else
	{
		if (_ptr->SHA_remain_len != 0)
		{
			_job.inputPtr       = _ptr->SHA_buf;
			_job.inputLength    = _ptr->SHA_remain_len;
		}
		else
		{
			_job.inputPtr       = NULL;
			_job.inputLength    = 0;
		}
	}

	_job.secondaryInputPtr      = (uint8_t *)_ptr->SHA_IV;
	_job.secondaryInputLength   = _block_size / 2;

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		_job.outputPtr          = tmp_SHA_block;
		_job.outputLength       = _block_size / 2;
	}
	else
	{
		_job.outputPtr          = out ? (uint8_t *)out->memref.buffer : NULL;
		_job.outputLength       = out ? out->memref.size : 0;
	}

	_job.service                = CRYPTO_HASH;
	_job.mode                   = CRYPTO_ALGOMODE_NOT_SET;

	_job.cryptoKeyId            = 0xFF;
	_job.keyLength              = 0;

	_job.operation_mode         = _Step_to_OPmode(step, _ptr->isFirstUpdate);

	SKS_HSM_DEBUG(" ==> _job.secondaryInputLength: %d\n", _job.secondaryInputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	out_buf = (uint8_t *)_job.outputPtr;

	__SHOW_VAL("out_buf", out_buf, 32);

	if (step == SKS_FUNC_STEP_UPDATE)
		memcpy(_ptr->SHA_IV, out_buf, _job.outputLength); // Update SHA_IV[]

	if (_ptr->isFirstUpdate == true)
		_ptr->isFirstUpdate = false;

	if (new_in_body.memref.buffer != NULL)
		TEE_Free(new_in_body.memref.buffer);

	return SKS_OK;
}

uint32_t mtk_SHA256_HMAC_step(struct pkcs11_session *session,
							  TEE_Param *in, TEE_Param *inout,
							  enum processing_func function,
							  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	uint8_t *verify_buf = NULL;
	uint8_t *out_buf =    NULL;
	job_struct _job = {0};
	uint8_t _block_size = 64;
	uint8_t veri_buf[64] = {0};

	TEE_Param new_in_body = {0};

	if ((session == NULL))
		return SKS_BAD_PARAM;

	// Process the data left last time that is not block size alignment.
	if (step == SKS_FUNC_STEP_ONESHOT)
	{
		_job.inputPtr       = in ? (uint8_t *)in->memref.buffer  : NULL;
		_job.inputLength    = in ? in->memref.size : 0;
	}
	else if (step == SKS_FUNC_STEP_UPDATE)
	{
		if (_merge_remain_data(in, &new_in_body, _ptr, _block_size) != SKS_OK)
			return SKS_MEMORY;

		if (new_in_body.memref.size == 0)
		{
			return SKS_OK;
		}

		if (new_in_body.memref.buffer == NULL)
			_job.inputPtr       = in ? (uint8_t *)in->memref.buffer  : NULL;
		else
			_job.inputPtr       = new_in_body.memref.buffer;

		_job.inputLength    = new_in_body.memref.size;
	}
	else
	{
		if (_ptr->SHA_remain_len != 0)
		{
			_job.inputPtr       = _ptr->SHA_buf;
			_job.inputLength    = _ptr->SHA_remain_len;
		}
		else
		{
			_job.inputPtr       = NULL;
			_job.inputLength    = 0;
		}
	}

	_job.secondaryInputPtr      = (uint8_t *)_ptr->SHA_IV;
	_job.secondaryInputLength   = 32;

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		_job.outputPtr      = veri_buf;
		_job.outputLength   = _block_size / 2;
	}
	else
	{
		if (function == SKS_FUNCTION_VERIFY)
			_job.outputPtr  = veri_buf;
		else
			_job.outputPtr  = inout ? (uint8_t *)inout->memref.buffer : NULL;

		_job.outputLength       = 32;
	}

	_job.service                = CRYPTO_MACGENERATE;   // No matter the function is sign or verify, HSM do the same thing.
	_job.family                 = CRYPTO_ALGOFAM_SHA2_256;
	_job.mode                   = CRYPTO_ALGOMODE_HMAC;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;

	_job.operation_mode         = _Step_to_OPmode(step, _ptr->isFirstUpdate);

	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	SKS_HSM_DEBUG(" ==> _job.secondaryInputLength: %d\n", _job.secondaryInputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	if (step == SKS_FUNC_STEP_ONESHOT)
		verify_buf = inout ? (uint8_t *)inout->memref.buffer : NULL;
	else
		verify_buf = in ? (uint8_t *)in->memref.buffer : NULL;

	out_buf = (uint8_t *)_job.outputPtr;

	__SHOW_VAL("out_buf", out_buf, 32);

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		// Update SHA_IV[]
		memcpy(_ptr->SHA_IV, out_buf, 32);
	}
	else
	{
		if (function == SKS_FUNCTION_VERIFY)
		{
			__SHOW_VAL("Verification out_buf", verify_buf, _job.outputLength);
			if (memcmp(verify_buf, out_buf, _job.outputLength) != 0)
			{
				SKS_HSM_ERROR(" HMAC256 verify ERROR!!  \n");
				return SKS_ERROR;
			}
		}
	}

	if (_ptr->isFirstUpdate == true)
		_ptr->isFirstUpdate = false;

	if (new_in_body.memref.buffer != NULL)
		TEE_Free(new_in_body.memref.buffer);

	return SKS_OK;
}

uint32_t mtk_SHA384_HMAC_step(struct pkcs11_session *session,
							  TEE_Param *in, TEE_Param *inout,
							  enum processing_func function,
							  enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	job_struct _job = {0};
	uint8_t _block_size = 128;
	uint8_t veri_buf[64] = {0};

	TEE_Param new_in_body = {0};
	uint8_t *verify_buf = NULL;
	uint8_t *out_buf = NULL;

	if ((session == NULL))
		return SKS_BAD_PARAM;

	// Process the data left last time that is not block size alignment.
	if (step == SKS_FUNC_STEP_ONESHOT)
	{
		_job.inputPtr       = in ? (uint8_t *)in->memref.buffer  : NULL;
		_job.inputLength    = in ? in->memref.size : 0;
	}
	else if (step == SKS_FUNC_STEP_UPDATE)
	{
		if (_merge_remain_data(in, &new_in_body, _ptr, _block_size) != SKS_OK)
			return SKS_MEMORY;

		if (new_in_body.memref.size == 0)
		{
			return SKS_OK;
		}

		if (new_in_body.memref.buffer == NULL)
			_job.inputPtr       = in ? (uint8_t *)in->memref.buffer  : NULL;
		else
			_job.inputPtr       = new_in_body.memref.buffer;

		_job.inputLength    = new_in_body.memref.size;
	}
	else
	{
		if (_ptr->SHA_remain_len != 0)
		{
			_job.inputPtr       = _ptr->SHA_buf;
			_job.inputLength    = _ptr->SHA_remain_len;
		}
		else
		{
			_job.inputPtr       = NULL;
			_job.inputLength    = 0;
		}
	}

	_job.secondaryInputPtr      = (uint8_t *)_ptr->SHA_IV;
	_job.secondaryInputLength   = 64;  // A specific length of HMAC-384

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		_job.outputPtr      = veri_buf;
		_job.outputLength   = _block_size / 2;
	}
	else
	{
		if (function == SKS_FUNCTION_VERIFY)
			_job.outputPtr  = veri_buf;
		else
			_job.outputPtr  = inout ? (uint8_t *)inout->memref.buffer : NULL;

		_job.outputLength       = 48;
	}

	_job.service                = CRYPTO_MACGENERATE;   // No matter the function is sign or verify, HSM do the same thing.
	_job.family                 = CRYPTO_ALGOFAM_SHA2_384;
	_job.mode                   = CRYPTO_ALGOMODE_HMAC;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len * 8;

	_job.operation_mode         = _Step_to_OPmode(step, _ptr->isFirstUpdate);

	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	SKS_HSM_DEBUG(" ==> _job.secondaryInputLength: %d\n", _job.secondaryInputLength);
	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	if (step == SKS_FUNC_STEP_ONESHOT)
		verify_buf = inout ? (uint8_t *)inout->memref.buffer : NULL;
	else
		verify_buf = in ? (uint8_t *)in->memref.buffer : NULL;

	out_buf = (uint8_t *)_job.outputPtr;

	__SHOW_VAL("out_buf", out_buf, _job.outputLength);

	if (step == SKS_FUNC_STEP_UPDATE)
	{
		// Update SHA_IV[]
		memcpy(_ptr->SHA_IV, out_buf, _job.outputLength);
	}
	else
	{
		if (function == SKS_FUNCTION_VERIFY)
		{
			__SHOW_VAL("Verification out_buf", verify_buf, _job.outputLength);
			if (memcmp(verify_buf, out_buf, _job.outputLength) != 0)
			{
    			SKS_HSM_ERROR(" HMAC384 verify ERROR!!  \n");
				return SKS_ERROR;
			}
		}
	}

	if (_ptr->isFirstUpdate == true)
		_ptr->isFirstUpdate = false;

	if (new_in_body.memref.buffer != NULL)
		TEE_Free(new_in_body.memref.buffer);

	return SKS_OK;
}

uint32_t mtk_ECDSA_step(struct pkcs11_session *session,
									 TEE_Param *in, TEE_Param *inout,
									 enum processing_func function,
									 enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	job_struct _job = {0};
	uint8_t r[64] = {0};
	uint8_t s[64] = {0};
	uint8_t *out_buf = NULL;
	uint8_t rs_len = 0;

	if ((session == NULL) || (in == NULL))
		return SKS_BAD_PARAM;

	_job.inputPtr               = in ? (uint8_t *)in->memref.buffer  : NULL;
	_job.inputLength            = in ? in->memref.size : 0;

	if (function == SKS_FUNCTION_SIGN)
	{
		_job.outputPtr              = r;
		_job.outputLength           = inout ? (inout->memref.size / 2): 0;
		_job.secondaryOutputPtr     = s;
		_job.secondaryOutputLength  = inout ? (inout->memref.size / 2): 0;
	}
	else // (function == SKS_FUNCTION_VERIFY)
	{
		uint8_t rs_len				= inout ? (inout->memref.size / 2): 0;
		uint8_t *in_buf2			= inout ? (uint8_t *)inout->memref.buffer  : NULL;
		memcpy(r, in_buf2, rs_len);
		memcpy(s, &in_buf2[rs_len], rs_len);

		_job.secondaryInputPtr      = r;
		_job.secondaryInputLength   = rs_len;
		_job.tertiaryInputPtr	    = s;
		_job.tertiaryInputLength    = rs_len;
	}

	if (function == SKS_FUNCTION_SIGN)
		_job.service                = CRYPTO_SIGNATUREGENERATE;
	else
		_job.service                = CRYPTO_SIGNATUREVERIFY;
	
	uint32_t mecha_type = session->processing->mecha_type;
    
	if (mecha_type == SKS_CKM_MTK_HSM_ECDSA_SM2){
		_job.family                 = CRYPTO_ALGOFAM_ECCNIST_SM2;
	}else{
		_job.family                 = CRYPTO_ALGOFAM_ECCNIST;
	}
	SKS_HSM_DEBUG(" ==> _job.family: %d\n", _job.family);
	
	_job.mode                   = CRYPTO_ALGOMODE_NOT_SET;

	_job.cryptoKeyId            = _ptr->key1_id;
	_job.keyLength              = _ptr->key1_len* 8;

	_job.operation_mode         = _Step_to_OPmode(step, _ptr->isFirstUpdate);

	SKS_HSM_DEBUG(" ==> _job.inputLength: %d\n", _job.inputLength);
	if (function == SKS_FUNCTION_SIGN)
		SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);
	else
		SKS_HSM_DEBUG(" ==> _job.secondaryInputLength: %d\n", _job.secondaryInputLength);


	Send_job(&_job);

	out_buf = inout ? (uint8_t *)inout->memref.buffer : NULL;
	rs_len   = _job.outputLength;

	__SHOW_VAL("r", r, rs_len);
	__SHOW_VAL("s", s, rs_len);

	if (function == SKS_FUNCTION_SIGN)
	{
		// Merge 2 output buffer into 1
		memcpy(out_buf, r, rs_len);
		memcpy(&out_buf[rs_len], s, rs_len);
	}

	return SKS_OK;
}

uint32_t mtk_ECDSA_SHA_step(struct pkcs11_session *session,
									 TEE_Param *in, TEE_Param *inout,
									 enum processing_func function,
									 enum processing_step step)
{
	MTK_HSM_State *_ptr = session->processing->mtk_hsm_handle;
	uint32_t mecha_type = session->processing->mecha_type;
	uint32_t digesting_mecha_type = 0;
	uint8_t digestedOutBuf[512] = {0};
	uint32_t digestedDataLen    = 0;
	TEE_Param digestedOut       = {0};

	if ((session == NULL) || (in == NULL))
		return SKS_BAD_PARAM;

	// ASSERT SHA one-shot only

	if (step != SKS_FUNC_STEP_ONESHOT)
	{
		SKS_HSM_ERROR(" ASSERT SHA one-shot only!! \n");
		return SKS_BAD_PARAM;
	}

	switch (mecha_type)
	{
	case SKS_CKM_MTK_HSM_ECDSA_SHA224:
		digesting_mecha_type = SKS_CKM_MTK_HSM_SHA224;
		digestedOut.memref.size = 28;
		break;

	case SKS_CKM_MTK_HSM_ECDSA_SHA256:
		digesting_mecha_type = SKS_CKM_MTK_HSM_SHA256;
		digestedOut.memref.size = 32;
		break;

	case SKS_CKM_MTK_HSM_ECDSA_SHA384:
		digesting_mecha_type = SKS_CKM_MTK_HSM_SHA384;
		digestedOut.memref.size = 48;
		break;
	case SKS_CKM_MTK_HSM_ECDSA_SHA512:
		digesting_mecha_type = SKS_CKM_MTK_HSM_SHA512;
		digestedOut.memref.size = 64;
		break;
	default:
	case SKS_CKM_MTK_HSM_ECDSA_SHA1:
		digesting_mecha_type = SKS_CKM_MTK_HSM_SHA1;
		digestedOut.memref.size = 20;
		break;
	}
	digestedOut.memref.buffer = digestedOutBuf;

	mtk_SHA_step(digesting_mecha_type, session, in, &digestedOut, function, step);
	mtk_ECDSA_step(session, &digestedOut, inout, function, step);

	return SKS_OK;
}

uint32_t mtk_generate_random(TEE_Param *out)
{
	job_struct _job = {0};

	if (out == NULL)
		return SKS_BAD_PARAM;

	_job.outputPtr              = (uint8_t *)out->memref.buffer;
	_job.outputLength           = out->memref.size;

	_job.service                = CRYPTO_RANDOMGENERATE;
	_job.family                 = CRYPTO_ALGOFAM_RNG;
	_job.mode                   = CRYPTO_ALGOMODE_NOT_SET;

	SKS_HSM_DEBUG(" ==> _job.outputLength: %d\n", _job.outputLength);

	Send_job(&_job);

	__SHOW_VAL("TRNG val", _job.outputPtr, _job.outputLength);

	return SKS_OK;
}

uint32_t mtk_dump_hsm_log(void)
{
	dump_hsm_log();

	return 0;
}

TEE_Result Send_job(job_struct *pjob)
{
	TEE_TASessionHandle cryp_session;
	TEE_Result res = TEE_SUCCESS;
	uint32_t origin = 0;
	TEE_Param params[4];
	TEE_UUID uuid = PTA_MBOX_UUID;
	int hsm_ret = -1;

	uint32_t types = TEE_PARAM_TYPES( TEE_PARAM_TYPE_VALUE_INPUT
					, TEE_PARAM_TYPE_VALUE_INPUT
					, TEE_PARAM_TYPE_VALUE_INPUT
					, TEE_PARAM_TYPE_VALUE_INPUT);

	TEE_MemFill(params, 0, sizeof(TEE_Param) * 4);
	params[0].value.a = (uint32_t)pjob;
	params[0].value.b = (uint32_t)sizeof(job_struct);
	params[1].value.a = (uint32_t)&hsm_ret;

	res = TEE_OpenTASession(&uuid, 0, types, params, &cryp_session,
							&origin);

	if (res != TEE_SUCCESS) {
		SKS_HSM_ERROR("rpc - TEE_OpenTASession returned 0x%x\n",
			 (unsigned int)res);
		return res;
	}

	res = TEE_InvokeTACommand(cryp_session, 0, PTA_CMD_HSM_SEND_JOB, types, params, &origin);
	if (res != TEE_SUCCESS) {
		SKS_HSM_ERROR("rpc_call_cryp - TEE_InvokeTACommand returned 0x%x\n",
			 (unsigned int)res);
	}

	TEE_CloseTASession(cryp_session);

	return res;

}

TEE_Result dump_hsm_log(void)
{
	TEE_TASessionHandle cryp_session;
	TEE_Result res = TEE_SUCCESS;
	uint32_t origin = 0;
	TEE_Param params[4];
	TEE_UUID uuid = PTA_MBOX_UUID;
	int hsm_ret = -1;

	uint32_t types = TEE_PARAM_TYPES( TEE_PARAM_TYPE_VALUE_INPUT
					, TEE_PARAM_TYPE_VALUE_INPUT
					, TEE_PARAM_TYPE_VALUE_INPUT
					, TEE_PARAM_TYPE_VALUE_INPUT);

	TEE_MemFill(params, 0, sizeof(TEE_Param) * 4);

	res = TEE_OpenTASession(&uuid, 0, types, params, &cryp_session,
							&origin);

	if (res != TEE_SUCCESS) {
		SKS_HSM_ERROR("rpc - TEE_OpenTASession returned 0x%x\n",
			 (unsigned int)res);
		return res;
	}

	res = TEE_InvokeTACommand(cryp_session, 0, PTA_CMD_HSM_DUMP_LOG, types, params, &origin);
	if (res != TEE_SUCCESS) {
		SKS_HSM_ERROR("rpc_call_cryp - TEE_InvokeTACommand returned 0x%x\n",
			 (unsigned int)res);
	}

	TEE_CloseTASession(cryp_session);

	return res;

}
