// SPDX-License-Identifier: BSD-2-Clause
/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * MediaTek Inc. (C) 2020. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */
#include <assert.h>
#include <sks_internal_abi.h>
#include <sks_ta.h>
#include <string.h>
#include <tee_api_defines.h>
#include <tee_internal_api.h>
#include <tee_internal_api_extensions.h>
#include <util.h>
#include <tee_api.h>

#include "attributes.h"
#include "object.h"
#include "pkcs11_token.h"
#include "pkcs11_attributes.h"
#include "processing.h"
#include "serializer.h"
#include "sks_helpers.h"
#include "mtk_key.h"
#include "sks_hsm_debug.h"

#define KEY_TABLE_STRING "key_table"
#define KEY_TABLE_SIZE_STRING "key_tbl_size"

#define ALIGN_TO_16(x) (((x+15)>>4)<<4)

static int pta_gen_sym_key(int size, int *pkey_id);
static int pta_get_key_blob(uint8_t *dst, int size, int key_id, int algo_id, int *pblobsize);
static int pta_get_key_table(uint8_t *dst, int size, int *ptablesize);
static int pta_gen_key_pair(int size, int uECC_curve_id, int *pkey_id);
static int pta_rebuild_key_table(uint8_t *buffer, int tablesize);
static int pta_rebuild_key_blob(uint8_t *buffer, int blob_size, int key_id);
static int pta_import_key(uint8_t *pkeybuffer, uint32_t size, int *pkeyid, int algo_id);
static int pta_export_key(uint8_t *pblob, uint32_t blobsize, uint8_t *ppubkey, uint32_t *ppubkeylength);
static int pta_delete_key(int key_id);

static int _create_file(uint8_t *pPOName, uint32_t namelen)
{
	TEE_ObjectHandle ohKey = TEE_HANDLE_NULL;
	TEE_ObjectHandle hPersistObj = TEE_HANDLE_NULL;
	TEE_Result nResult = TEE_SUCCESS;


	nResult = TEE_CreatePersistentObject(
	              TEE_STORAGE_PRIVATE,
	              pPOName, namelen,
	              (TEE_DATA_FLAG_ACCESS_READ | TEE_DATA_FLAG_ACCESS_WRITE | TEE_DATA_FLAG_ACCESS_WRITE_META),
	              ohKey,
	              TEE_HANDLE_NULL, 0,
	              &hPersistObj);

	if (nResult != 0)
	{
		if (nResult != TEE_ERROR_ITEM_NOT_FOUND)
			SKS_HSM_ERROR("TEE_CreatePersistentObject error, nResult 0x%x!!!\n", nResult);
		return nResult;
	}

	TEE_CloseObject(hPersistObj);

	return nResult;
}

static int _delete_file(uint8_t *pPOName, uint32_t namelen)
{
	TEE_ObjectHandle hPersistObj = TEE_HANDLE_NULL;
	TEE_Result nResult = TEE_SUCCESS;

	nResult = TEE_OpenPersistentObject(
	              TEE_STORAGE_PRIVATE,
	              pPOName, namelen,
	              TEE_DATA_FLAG_ACCESS_READ | TEE_DATA_FLAG_ACCESS_WRITE | TEE_DATA_FLAG_ACCESS_WRITE_META,
	              &hPersistObj);

	if (nResult != 0)
	{
		if (nResult != TEE_ERROR_ITEM_NOT_FOUND)
			SKS_HSM_ERROR("TEE_OpenPersistentObject error, nResult 0x%x!!!\n", nResult);
		return nResult;
	}

	TEE_CloseAndDeletePersistentObject(hPersistObj);

	return TEE_SUCCESS;

}

static int _write_file(uint8_t *pPOName, uint32_t namelen, uint8_t *Content, uint32_t size)
{
	TEE_ObjectHandle hPersistObj = TEE_HANDLE_NULL;
	TEE_Result nResult = TEE_SUCCESS;

	nResult = TEE_OpenPersistentObject(
	              TEE_STORAGE_PRIVATE,
	              pPOName, namelen,
	              TEE_DATA_FLAG_ACCESS_READ | TEE_DATA_FLAG_ACCESS_WRITE | TEE_DATA_FLAG_ACCESS_WRITE_META,
	              &hPersistObj);

	if (nResult != 0)
	{
		if (nResult != TEE_ERROR_ITEM_NOT_FOUND)
			SKS_HSM_ERROR("TEE_OpenPersistentObject error, nResult 0x%x!!!\n", nResult);
		return nResult;
	}

	nResult = TEE_WriteObjectData(
	              hPersistObj,
	              (void*)Content,
	              size);

	if (nResult != 0)
	{
		SKS_HSM_ERROR("TEE_WriteObjectData error, nResult 0x%x!!!\n", nResult);
		return nResult;
	}

	TEE_CloseObject(hPersistObj);

	return nResult;
}
static int _read_file(uint8_t *pPOName, uint32_t namelen, uint8_t *Content, uint32_t size)
{
	TEE_ObjectHandle hPersistObj = TEE_HANDLE_NULL;
	TEE_Result nResult = TEE_SUCCESS;
	uint32_t count = 0;
	uint32_t rsize = size;

	nResult = TEE_OpenPersistentObject(
	              TEE_STORAGE_PRIVATE,
	              pPOName, namelen,
	              TEE_DATA_FLAG_ACCESS_READ | TEE_DATA_FLAG_ACCESS_WRITE | TEE_DATA_FLAG_ACCESS_WRITE_META,
	              &hPersistObj);

	if (nResult != 0)
	{
		if (nResult != TEE_ERROR_ITEM_NOT_FOUND)
			SKS_HSM_ERROR("TEE_CreatePersistentObject error, nResult 0x%x!!!\n", nResult);
		return nResult;
	}

	nResult = TEE_ReadObjectData(hPersistObj, Content, rsize, &count);
	if ((rsize != count) || (nResult != 0))
	{
		SKS_HSM_ERROR("error rsize 0x%x, count 0x%x!!!\n", rsize, count);
		SKS_HSM_ERROR("nResult 0x%x!!!\n", nResult);
		TEE_CloseObject(hPersistObj);
		return -1;
	}
	TEE_CloseObject(hPersistObj);

	return nResult;
}

char *itoa_32(int n, char *str, int radix)
{
	char digit[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	char *p = str;
	char *head = str;
	unsigned int val;

	if (str == NULL) {
		return NULL;
	}

	if (n == 0) {
		*p++ = '0';
		*p = 0;
		return str;
	}

	if (radix == 10 && n < 0)
		val = (-n);
	else
		val = n;

	while (val) {
		*p++ = digit[val % radix];
		val /= radix;
	}

	if (radix == 10 && n < 0)
		*p++ = '-';
	*p = 0;
	for (--p; head < p ; ++head, --p)
	{
		char temp = *head;
		*head = *p;
		*p = temp;
	}
	return str;
}

static void _make_blob_filename(uint8_t *dst, int key_id)
{
	int8_t *prefix = "blob_";
	int8_t strkeyid[64] = {0};

	if (dst == NULL) {
		return;
	}
	itoa_32(key_id, strkeyid, 16);

	memcpy(dst, prefix, strlen(prefix));
	memcpy(dst + strlen((char *)prefix), strkeyid, strlen(strkeyid));
}

static void _write_key_blob_to_storage(uint8_t *buffer, uint32_t blobsize, int key_id)
{
	int8_t *prefix = "blob_";
	int8_t pPOName[128] = {0};
	uint32_t namelen = 0;

	if (buffer == NULL) {
		return;
	}
	_make_blob_filename(pPOName, key_id);

	namelen = strlen((char *)pPOName);

	_create_file(pPOName, namelen);
	_write_file(pPOName, namelen, buffer, blobsize);
}


static void _remove_key_blob_from_storage(key_id)
{
	int8_t *prefix = "blob_";
	int8_t pPOName[128] = {0};
	uint32_t namelen = 0;

	_make_blob_filename(pPOName, key_id);

	namelen = strlen((char *)pPOName);
	_delete_file(pPOName, namelen);

}

static void _write_key_table_to_storage(uint8_t *buffer, uint32_t tablesize)
{
	int8_t pTableName[] = KEY_TABLE_STRING;
	int8_t pTableSizeName[] = KEY_TABLE_SIZE_STRING;
	uint32_t namelen = 0;

	if (buffer == NULL) {
		return;
	}
	namelen = strlen((char *)pTableName);
	_create_file(pTableName, namelen);
	_write_file(pTableName, namelen, buffer, tablesize);

	namelen = strlen((char *)pTableSizeName);
	_create_file(pTableSizeName, namelen);
	_write_file(pTableSizeName, namelen, &tablesize, sizeof(uint32_t));
}

static void _read_key_table_size(uint32_t *ptablesize)
{
	int8_t pTableSizeName[128] = KEY_TABLE_SIZE_STRING;
	uint8_t Content[4] = {0};
	uint32_t dstsize = sizeof(Content);
	uint32_t namelen = 0;

	if (ptablesize == NULL) {
		return;
	}
	namelen = strlen((char *)pTableSizeName);
	_read_file(pTableSizeName, namelen, Content, dstsize);

	*ptablesize = *((uint32_t *)Content);
}

static void _read_key_table(uint8_t *buffer, uint32_t tablesize)
{
	int8_t pTableName[128] = KEY_TABLE_STRING;
	uint32_t namelen = 0;

	namelen = strlen((char *)pTableName);
	_read_file(pTableName, namelen, buffer, tablesize);
}

static void _read_key_blob(uint8_t *buffer, uint32_t buffersize, int key_id)
{
	int8_t pBlobName[128] = {0};
	uint32_t namelen = 0;

	_make_blob_filename(pBlobName, key_id);

	namelen = strlen((char *)pBlobName);
	_read_file(pBlobName, namelen, buffer, buffersize);
}

static int _get_key_info_in_table(key_table *ptbl, int idx, uint32_t *pkeyid, uint32_t *pkeysize)
{
	if (ptbl == NULL) {
		return -1;
	}
	*pkeyid = ptbl->slot[idx].id;
	*pkeysize = ptbl->slot[idx].key_w.size;
	return 0;
}

static inline void _estimate_blob_size(uint32_t key_size, uint32_t *pblobsize)
{
	*pblobsize = ALIGN_TO_16(key_size) + BLOB_MAC_SIZE + BLOB_HEADER_SIZE;
}

static inline int _check_symmetric_key_size_valid(uint32_t size)
{
	return (size == 16) || (size == 24) || (size == 32) || (size == 48);
}

int mtk_delete_key(struct sks_object *object)
{
	uint8_t buffer[sizeof(key_table)] = {0};
	int tablesize = 0;
	uint32_t rv = 0;
	int key_id = 0;
	int key_id_size = sizeof(key_id);

	if ((object == NULL))
		 return SKS_CKR_ARGUMENTS_BAD;

	rv = get_attribute(object->attributes, SKS_CKA_HSM_KEY_ID, &key_id, &key_id_size);
	if (rv != 0) /* not mtk key object */
		return SKS_CKR_ARGUMENTS_BAD;
	if (pta_delete_key(key_id) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	_remove_key_blob_from_storage(key_id);

	if (pta_get_key_table(buffer, sizeof(buffer), &tablesize) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	_write_key_table_to_storage(buffer, tablesize);

	return TEE_SUCCESS;
}

int mtk_generate_symmetric_key(int size, int *pkeyid)
{
	int key_id = 0;
	uint8_t buffer[sizeof(key_table)] = {0};
	int blobsize = 0;
	int tablesize = 0;

	int _rv = _check_symmetric_key_size_valid(size);
	SKS_HSM_DEBUG("  %s [%d] size: %d, rv: %d \n", __FUNCTION__, __LINE__, size, _rv);
	if (!_check_symmetric_key_size_valid(size))
		return TEE_ERROR_GENERIC;

	if (pkeyid == NULL)
		return TEE_ERROR_GENERIC;

	if (pta_gen_sym_key(size, &key_id) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	if (pta_get_key_blob(buffer, sizeof(buffer), key_id, KEY_ALGO_ID_AES, &blobsize) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	_write_key_blob_to_storage(buffer, blobsize, key_id);

	if (pta_get_key_table(buffer, sizeof(buffer), &tablesize) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	_write_key_table_to_storage(buffer, tablesize);

	*pkeyid = key_id;

	return TEE_SUCCESS;
}

int mtk_generate_ecc_key_pair(int size, int curve_id, int *pkeyid)
{
	int key_id = 0;
	uint8_t buffer[sizeof(key_table)] = {0};
	int blobsize = 0;
	int tablesize = 0;
	int uECC_curve_id = curve_id;

	if (pkeyid == NULL)
		return TEE_ERROR_GENERIC;

	if (pta_gen_key_pair(size, uECC_curve_id, &key_id) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	if (pta_get_key_blob(buffer, sizeof(buffer), key_id, KEY_ALGO_ID_ECC, &blobsize) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	_write_key_blob_to_storage(buffer, blobsize, key_id);

	if (pta_get_key_table(buffer, sizeof(buffer), &tablesize) != TEE_SUCCESS)
		return TEE_ERROR_GENERIC;

	_write_key_table_to_storage(buffer, tablesize);

	*pkeyid = key_id;

	return TEE_SUCCESS;
}


int mtk_rebuild_key_table(void)
{
	uint8_t *table_buffer = NULL;
	uint8_t *blob_buffer = NULL;
	int table = 0;
	uint32_t tablesize = 0, blob_size = 0, key_size = 0, key_id = 0;
	uint32_t maxblobsize = MAX_BLOB_SIZE;
	key_table *ptable = NULL;

	_read_key_table_size(&tablesize);

	if (tablesize <= 0) {
		SKS_HSM_ERROR("[%s][%d] invalid table size : %d\n", __FUNCTION__, __LINE__, tablesize);
		goto fail;
	}

	table_buffer = TEE_Malloc(tablesize, TEE_MALLOC_FILL_ZERO);
	if (table_buffer == NULL) {
		SKS_HSM_ERROR("[%s][%d] allocate memory fail\n", __FUNCTION__, __LINE__);
		goto fail;
	}

	blob_buffer = TEE_Malloc(maxblobsize, TEE_MALLOC_FILL_ZERO);
	if (blob_buffer == NULL) {
		SKS_HSM_ERROR("[%s][%d] allocate memory fail\n", __FUNCTION__, __LINE__);
		goto fail;
	}

	_read_key_table(table_buffer, tablesize);
	ptable = (key_table *)table_buffer;
	SKS_HSM_INFO("[%s][%d] ptable->nkeys : %d\n", __FUNCTION__, __LINE__, ptable->nkeys);

	/* clean garbage slot address info in table*/
	for (int i = 0; i < ptable->nkeys; ++i) {
		ptable->slot[i].key_w.addr32 = 0;
	}

	if (pta_rebuild_key_table(table_buffer, tablesize) != 0) {
		SKS_HSM_ERROR("[%s][%d] rebuild key table fail!\n", __FUNCTION__, __LINE__);
		goto fail;
	}

	/* Rebuild the key blob for each key_id in the key table*/
	for (int i = 0; i < ptable->nkeys; ++i) {
		memset(blob_buffer, 0, maxblobsize);
		blob_size = key_id = key_size = 0;

		_get_key_info_in_table(ptable, i, &key_id, &key_size);
		if (key_size == 0 || key_id == 0)
			goto fail;

		_estimate_blob_size(key_size, &blob_size);
		_read_key_blob(blob_buffer, blob_size, key_id);
		pta_rebuild_key_blob(blob_buffer, blob_size, key_id);
	}


	if (table_buffer != NULL)
		TEE_Free(table_buffer);
	if (blob_buffer != NULL)
		TEE_Free(blob_buffer);

	return 0;

fail:
	if (table_buffer != NULL)
		TEE_Free(table_buffer);
	if (blob_buffer != NULL)
		TEE_Free(blob_buffer);

	return -1;
}

int mtk_import_key(uint8_t *pkey, uint32_t size,
				int *pkeyid, int *pblob,
				int *inblobsize, int algo_id)
{
	uint8_t buffer[sizeof(key_table)] = {0};
	int blobsize = 0;
	int key_id = 0;
	int tablesize = 0;
	int rv = 0;
	ecc_keypair_datastruct *pecckey = NULL;

	if ((pkey == NULL) || (pkeyid == NULL) || (inblobsize == NULL)) {
		SKS_HSM_ERROR("[%s][%d] bad argument\n", __FUNCTION__, __LINE__);
		goto fail;
	}

	switch (algo_id)
	{
	case KEY_ALGO_ID_ECC:
		pecckey = (ecc_keypair_datastruct *)pkey;
		if (pecckey->curve_id == UECC_CURVE_ID_UNSUPPORT) {
			SKS_HSM_ERROR("[%s][%d] not support curve\n", __FUNCTION__, __LINE__);
			goto fail;
		}
		break;
	case KEY_ALGO_ID_AES:
		if (!_check_symmetric_key_size_valid(size)) {
			SKS_HSM_ERROR("[%s][%d] not symmetric key size (%d)\n", __FUNCTION__, __LINE__, size);
			goto fail;
		}
		break;
	default:
		SKS_HSM_ERROR("[%s][%d] not algorithm type\n", __FUNCTION__, __LINE__);
		break;
	}

	rv = pta_import_key(pkey, size, &key_id, algo_id);
	if (rv != TEE_SUCCESS) {
		SKS_HSM_ERROR("[%s][%d] import key fail (%d)\n", __FUNCTION__, __LINE__, rv);
		goto fail;
	}

	rv = pta_get_key_blob(buffer, sizeof(buffer), key_id, algo_id, &blobsize);
	if (rv != TEE_SUCCESS) {
		SKS_HSM_ERROR("[%s][%d] get blob fail (%d)\n", __FUNCTION__, __LINE__, rv);
		goto fail;
	}

	if (blobsize > *inblobsize) {
		SKS_HSM_ERROR("[%s][%d] in buf too small (%d > %d)\n", __FUNCTION__, __LINE__, blobsize, inblobsize);
		goto fail;
	}
	_write_key_blob_to_storage(buffer, blobsize, key_id);
	memcpy(pblob, buffer, blobsize);
	*inblobsize = blobsize;

	rv = pta_get_key_table(buffer, sizeof(buffer), &tablesize);
	if (rv != TEE_SUCCESS) {
		SKS_HSM_ERROR("[%s][%d] get table fail (%d)\n", __FUNCTION__, __LINE__, rv);
		goto fail;
	}

	_write_key_table_to_storage(buffer, tablesize);

	*pkeyid = key_id;

	return TEE_SUCCESS;
fail:
	return TEE_ERROR_GENERIC;
}

int mtk_export_key(uint8_t *pblob, uint32_t blobsize, uint8_t *ppubkey, uint32_t *ppubkeylength)
{
	int rv = 0;
	if ((pblob == NULL) || (ppubkey == NULL) || (ppubkeylength == NULL)) {
		SKS_HSM_ERROR("[%s][%d] bad argument\n", __FUNCTION__, __LINE__);
		goto fail;
	}

	rv = pta_export_key(pblob, blobsize, ppubkey, ppubkeylength);
	if (rv != TEE_SUCCESS) {
		SKS_HSM_ERROR("[%s][%d] export key fail (%d)\n", __FUNCTION__, __LINE__, rv);
		goto fail;
	}

	return TEE_SUCCESS;
fail:
	return TEE_ERROR_GENERIC;

}

static void _fill_pta_params(TEE_Param *params,
                             uint32_t a0, uint32_t b0,
                             uint32_t a1, uint32_t b1,
                             uint32_t a2, uint32_t b2,
                             uint32_t a3, uint32_t b3)
{
	TEE_MemFill(params, 0, sizeof(TEE_Param) * 4);

	params[0].value.a = a0;
	params[0].value.b = b0;

	params[1].value.a = a1;
	params[1].value.b = b1;

	params[2].value.a = a2;
	params[2].value.b = b2;

	params[3].value.a = a3;
	params[3].value.b = b3;
}

static int _transmit_to_pta(TEE_Param *params, uint32_t cmd_id)
{
	TEE_TASessionHandle pta_session;
	uint32_t origin;
	TEE_UUID uuid = PTA_MBOX_UUID;
	TEE_Result res = TEE_SUCCESS;

	uint32_t types = TEE_PARAM_TYPES( TEE_PARAM_TYPE_VALUE_INPUT
	                                  , TEE_PARAM_TYPE_VALUE_INPUT
	                                  , TEE_PARAM_TYPE_VALUE_INPUT
	                                  , TEE_PARAM_TYPE_VALUE_INPUT);

	res = TEE_OpenTASession(&uuid, 0, types, params, &pta_session,
	                        &origin);

	if (res != TEE_SUCCESS) {
		SKS_HSM_ERROR("rpc_sha256 - TEE_OpenTASession returned 0x%x\n",
		     (unsigned int)res);
		return res;
	}


	res = TEE_InvokeTACommand(pta_session, 0, cmd_id, types, params, &origin);
	if (res != TEE_SUCCESS) {
		SKS_HSM_ERROR("rpc_call_cryp - TEE_InvokeTACommand returned 0x%x\n",
		     (unsigned int)res);
	}

	TEE_CloseTASession(pta_session);
	return res;
}

static int pta_gen_key_pair(int size, int uECC_curve_id, int *pkey_id) // TODO: ADD CURVE ID
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4] = {0};

	int hsm_ret = -1;
	int key_id = -2;
	int key_size = size;

	if (pkey_id == NULL) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)size,
	                 (uint32_t)uECC_curve_id, (uint32_t)&key_id,
	                 NULL, NULL,
	                 NULL, NULL);

	res = _transmit_to_pta(params, PTA_CMD_HSM_GEN_KEY_PAIR);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	*pkey_id = key_id;

	return hsm_ret;
}

static int pta_gen_sym_key(int size, int *pkey_id)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4] = {0};

	int hsm_ret = -1;
	int key_id = -2;
	int key_size = size;

	if (pkey_id == NULL) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)&key_id,
	                 (uint32_t)key_size , NULL,
	                 NULL, NULL,
	                 NULL, NULL);

	res = _transmit_to_pta(params, PTA_CMD_HSM_GEN_SYMM_KEY);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	*pkey_id = key_id;

	return hsm_ret;
}

static int pta_delete_key(int key_id)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4] = {0};

	int hsm_ret = -1;

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)key_id,
	                 NULL , NULL,
	                 NULL, NULL,
	                 NULL, NULL);

	res = _transmit_to_pta(params, PTA_CMD_HSM_DELETE_KEY);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	return hsm_ret;
}

static int pta_export_key(uint8_t *pblob, uint32_t blobsize, uint8_t *ppubkey, uint32_t *ppubkeylength)
{
	TEE_Result res;
	TEE_Param params[4];

	int hsm_ret = -1;

	if ((pblob == NULL) || (ppubkey == NULL) || (ppubkeylength == NULL)) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)pblob,
	                 (uint32_t)blobsize , (uint32_t)ppubkey,
	                 (uint32_t)ppubkeylength, NULL,
	                 NULL, NULL);


	res = _transmit_to_pta(params, PTA_CMD_HSM_EXPORT_KEY);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	return hsm_ret;
}

static int pta_import_key(uint8_t *pkeybuffer, uint32_t size, int *pkeyid, int algo_id)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4];

	int hsm_ret = -1;
	int key_id = 0;

	if ((pkeybuffer == NULL) || (pkeyid == NULL)) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)pkeybuffer,
	                 (uint32_t)size , (uint32_t)&key_id,
	                 (uint32_t)algo_id, NULL,
	                 NULL, NULL);


	res = _transmit_to_pta(params, PTA_CMD_HSM_IMPORT_KEY);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	*pkeyid = key_id;

	return hsm_ret;
}

static int pta_get_key_blob(uint8_t *dst, int size, int key_id, int algo_id, int *pblobsize)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4];

	int hsm_ret = -1;
	uint32_t blobsize;

	if (dst == NULL) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)key_id,
	                 (uint32_t)dst , (uint32_t)size,
	                 (uint32_t)algo_id, (uint32_t)&blobsize,
	                 NULL, NULL);


	res = _transmit_to_pta(params, PTA_CMD_HSM_GET_KEY_BLOB);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	*pblobsize = blobsize;

	return hsm_ret;
}

static int pta_get_key_table(uint8_t *dst, int size, int *ptablesize)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4];

	int hsm_ret = -1;
	uint32_t tablesize;

	if (dst == NULL) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)dst,
	                 (uint32_t)size , (uint32_t)&tablesize,
	                 NULL, NULL,
	                 NULL, NULL);

	res = _transmit_to_pta(params, PTA_CMD_HSM_GET_KEY_TABLE);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	*ptablesize = tablesize;

	return hsm_ret;
}


static int pta_rebuild_key_table(uint8_t *buffer, int tablesize)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4] = {0};
	int hsm_ret = -1;

	if (buffer == NULL) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)buffer,
	                 (uint32_t)tablesize , NULL,
	                 NULL, NULL,
	                 NULL, NULL);

	res = _transmit_to_pta(params, PTA_CMD_HSM_REBUILD_KEY_TABLE);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	return hsm_ret;
}

static int pta_rebuild_key_blob(uint8_t *buffer, int blob_size, int key_id)
{
	TEE_Result res = TEE_SUCCESS;
	TEE_Param params[4] = {0};
	int hsm_ret = -1;

	if (buffer == NULL) {
		return TEE_ERROR_BAD_PARAMETERS;
	}

	_fill_pta_params(params,
	                 (uint32_t)&hsm_ret, (uint32_t)buffer,
	                 (uint32_t)blob_size , (uint32_t)key_id,
	                 NULL, NULL,
	                 NULL, NULL);

	res = _transmit_to_pta(params, PTA_CMD_HSM_REBUILD_KEY_BLOB);

	if (res != TEE_SUCCESS) {
		return TEE_ERROR_GENERIC;
	}

	return hsm_ret;
}
