/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * MediaTek Inc. (C) 2020. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stdint.h>
#include <mm/core_memprot.h>
#include <tee/cache.h>
#include "buddy.h"
#include "buddy_config.h"
#include "share_memmap.h"

#if BUDDY_MEMPOOL_SIZE > BUDDY_MEMPOOL_BUF_LEN
#error "[buddy system] not enough memory..."
#endif

#define max(a, b) (((a)>(b))?(a):(b))
#define min(a, b) (((a)<(b))?(a):(b))

typedef uint16_t node_len_t ;

struct buddy {
	uint32_t size;
	node_len_t longest[];
};

static struct buddy *__instance = NULL;
// static uint8_t buddy_table[sizeof(struct buddy) + 2 * BUDDY_TOTAL_BLOCKS * sizeof(node_len_t)];
static uint8_t *memory_pool = NULL;
static vaddr_t v_memory_pool;

static inline int left_child(int index)
{
	/* index * 2 + 1 */
	return ((index << 1) + 1);
}

static inline int right_child(int index)
{
	/* index * 2 + 2 */
	return ((index << 1) + 2);
}

static inline int parent(int index)
{
	/* (index+1)/2 - 1 */
	return (((index + 1) >> 1) - 1);
}

static inline bool is_power_of_2(int index)
{
	return !(index & (index - 1));
}

static inline unsigned next_power_of_2(unsigned size)
{
	/* depend on the fact that size < 2^32 */
	size -= 1;
	size |= (size >> 1);
	size |= (size >> 2);
	size |= (size >> 4);
	size |= (size >> 8);
	size |= (size >> 16);
	return size + 1;
}

static inline uint8_t *get_va_in_mempool(int idx)
{
	return (v_memory_pool + (idx << BUDDY_BLOCK_SIZE_ORDER));
}

static inline uint8_t *get_pa_in_mempool(int idx)
{
	return (memory_pool + (idx << BUDDY_BLOCK_SIZE_ORDER));
}

static inline int map_pa_2_idx_in_mempool(uint8_t * ptr)
{
	return (((int)ptr - (int)memory_pool) >> BUDDY_BLOCK_SIZE_ORDER);
}

void buddy_init(void)
{
	if (__instance != NULL) {
		return;
	}

	struct buddy *self = NULL;
	node_len_t node_size;
	unsigned num_of_fragments = BUDDY_TOTAL_BLOCKS;
	int i;
	int table_size = sizeof(struct buddy) + 2 * BUDDY_TOTAL_BLOCKS * sizeof(node_len_t);

	/* alloacte an array to represent a complete binary tree */
	self = (struct buddy *) malloc(table_size);
	memset(self, 0, table_size);

	self->size = num_of_fragments;
	node_size = num_of_fragments * 2;

	/* initialize *longest* array for buddy structure */
	int iter_end = num_of_fragments * 2 - 1;
	for (i = 0; i < iter_end; i++) {
		if (is_power_of_2(i + 1)) {
			node_size >>= 1;
		}
		self->longest[i] = node_size;
	}

	core_mmu_add_mapping(MEM_AREA_RAM_SEC, TEE_BUDDY_MEMPOOL_BUF_BASE, TEE_BUDDY_MEMPOOL_BUF_LEN);
	v_memory_pool = (vaddr_t)phys_to_virt(TEE_BUDDY_MEMPOOL_BUF_BASE, MEM_AREA_RAM_SEC);

	if (cache_operation(TEE_CACHEINVALIDATE, v_memory_pool, TEE_BUDDY_MEMPOOL_BUF_LEN) != TEE_SUCCESS) {
		EMSG("[\x1b[31m%s\033[0m][%d] cache invalidate fail\n", __FUNCTION__, __LINE__);
		return;
	}

	memory_pool = (uint8_t *)TEE_BUDDY_MEMPOOL_BUF_BASE;
	__instance = (void*)self;


}

void buddy_deinit(void)
{
	memory_pool = NULL;
	__instance = NULL;
}

/* choose the child with smaller longest value which is still larger
 * than *size* */
unsigned choose_better_child(struct buddy *self, unsigned index, size_t size)
{
	struct compound {
		size_t size;
		unsigned index;
	} children[2];

	children[0].index = left_child(index);
	children[0].size = self->longest[children[0].index];
	children[1].index = right_child(index);
	children[1].size = self->longest[children[1].index];

	int min_idx = (children[0].size <= children[1].size) ? 0 : 1;

	if (size > children[min_idx].size) {
		min_idx = 1 - min_idx;
	}

	return children[min_idx].index;
}

/** allocate *size* from a buddy system *self*
 * @return the offset from the beginning of memory to be managed */
int buddy_alloc_blk(size_t size)
{
	struct buddy *self = __instance;

	if (self == NULL || self->size < size || size <= 0) {
		return -1;
	}

	size = next_power_of_2(size);

	unsigned index = 0;
	if (self->longest[index] < size) {
		return -1;
	}

	/* search recursively for the child */
	unsigned node_size = 0;
	for (node_size = self->size; node_size != size; node_size >>= 1) {
		/* choose the child with smaller longest value which is still larger
		 * than *size* */
		/* TODO */
		index = choose_better_child(self, index, size);
	}

	/* update the *longest* value back */
	self->longest[index] = 0;
	int offset = (index + 1) * node_size - self->size;

	while (index) {
		index = parent(index);
		self->longest[index] =
		    max(self->longest[left_child(index)],
		        self->longest[right_child(index)]);
	}

	return offset;
}

void buddy_free_blk(int offset)
{
	struct buddy *self = __instance;

	if (self == NULL || offset < 0 || offset > self->size) {
		return;
	}

	node_len_t node_size;
	unsigned index;

	/* get the corresponding index from offset */
	node_size = 1;
	index = offset + self->size - 1;

	for (; self->longest[index] != 0; index = parent(index)) {
		node_size <<= 1;    /* node_size *= 2; */

		if (index == 0) {
			break;
		}
	}

	self->longest[index] = node_size;

	while (index) {
		index = parent(index);
		node_size <<= 1;

		node_len_t left_longest = self->longest[left_child(index)];
		node_len_t right_longest = self->longest[right_child(index)];

		if (left_longest + right_longest == node_size) {
			self->longest[index] = node_size;
		} else {
			self->longest[index] = max(left_longest, right_longest);
		}
	}
}

void buddy_dump(void)
{
	struct buddy *self = __instance;

	int len = self->size << 1;
	int max_col = self->size << 1;
	int level = 0;
	int i, j;

	char cs[] = {'/', '\\'};
	int idx = 0;
	char c;

	for (i = 0, max_col = len, level = 0; i < len - 1; i++) {
		if (is_power_of_2(i + 1)) {
			max_col >>= 1;
			level ++;
			idx = 0;
			// LOGE("\n%d(%3d): ", level, max_col);
		}

		// LOGE("%*d", max_col, self->longest[i]);
	}

	for (i = 0, max_col = len, level = 0; i < len - 1; i++) {
		if (is_power_of_2(i + 1)) {
			max_col >>= 1;
			level ++;
			idx = 0;
			// LOGE("\n%d(%3d): ", level, max_col);
		}

		if (self->longest[i] > 0) {
			c = '-';
		} else {
			c = cs[idx];
			idx ^= 0x1;
		}

		for (j = 0; j < max_col; j++) {
			// LOGE("%c", c);
		}
	}
	// LOGE("\n");
}

int buddy_size(int offset)
{
	struct buddy *self = __instance;
	unsigned node_size = 1;
	unsigned index = offset + self->size - 1;

	for (; self->longest[index]; index = parent(index)) {
		node_size >>= 1;
	}

	return node_size;
}

int buddy_malloc(uint32_t size, addr_t *addr)
{
	int idx = -1;
	int blks = 0;

	blks = (size >> BUDDY_BLOCK_SIZE_ORDER) + 1;
	idx = buddy_alloc_blk(blks);
	if (idx < 0) {
		// LOGE("[\x1b[34m%s\033[0m][%d] allocate fail\r\n", __FUNCTION__, __LINE__);
		return BUDDY_FAIL;
	}

	addr->pa = get_pa_in_mempool(idx);
	addr->va = get_va_in_mempool(idx);

	return BUDDY_OK;
}


void buddy_free(addr_t *addr)
{
	int idx = -1;

	idx = map_pa_2_idx_in_mempool(addr->pa);

	buddy_free_blk(idx);
}