/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <kernel/interrupt.h>
#include <mm/core_memprot.h>

#include "mbox_drv.h"
#include "mbox_serv.h"
#include "system_event.h"
#include "processor.h"

#define MBOX_SERV_MAGIC_NUM     0x5AA52731UL
#define MBOX_SERV_MAGIC_ERROR   0xCCE2911BUL

#define MBOX_SERV_TX_DELAY_MS     1UL
#define MBOX_SERV_TX_DELAY_CNT    1UL
#define MBOX_SERV_TX_DELAY	(MBOX_SERV_TX_DELAY_MS / MBOX_SERV_TX_DELAY_CNT)

struct mbox_domain {
	uint8_t domain;
	uint8_t security;
};

struct mbox_serv_table {
	enum mbox_regions region;
	enum CPU_SERIAL cpu_serial;
};

static const struct mbox_serv_table mbox_serv_tb[][MBOX_SERV_ENTRY_CNT] = {
	/* CA53 */
	{
		{MBOX_REGION_0,  CPU_SERIAL_HSM},
		{MBOX_REGION_1,  CPU_SERIAL_CA53},
	},

	/* HSM */
	{
		{MBOX_REGION_0,  CPU_SERIAL_HSM},
		{MBOX_REGION_1,  CPU_SERIAL_CA53},
	},
};

static enum CPU_SERIAL mbox_serv_serial;
static const struct mbox_serv_table *p_mbox_serv;
static struct mbox_serv_registry mbox_serv_registry_tb[SYS_EVENT_CNT];
static int8_t mbox_serv_mutex[MBOX_DOMAIN_CNT];

static inline void mbox_serv_ack(uint8_t msg_id)
{
	/* clear macgic number. Let sender knows that we received the msg */
	mbox_msg_tx(mbox_serv_serial, msg_id, MBOX_SERV_MAGIC_NUM);
}

static inline void mbox_serv_ack_error(uint8_t msg_id)
{
	/* clear macgic number. Let sender knows that there is something wrong in isr */
	mbox_msg_tx(mbox_serv_serial, msg_id, MBOX_SERV_MAGIC_ERROR);
}

static int8_t wait_remote_ack(enum CPU_SERIAL dest, uint8_t msg_id)
{
	uint32_t data = 0;
	int8_t result = MBOX_SERV_ERR_FAIL;

	/* wait remote set MBOX_SERV_MAGIC_NUM or MBOX_SERV_MAGIC_ERROR to mail box*/
	while(1)
	{
		mbox_msg_rx(dest, msg_id, &data);
		if (data == MBOX_SERV_MAGIC_NUM)
		{
			result = MBOX_SERV_OK;
			break;
		}
		else if (data == MBOX_SERV_MAGIC_ERROR)
		{
			result = MBOX_SERV_ERR_REMOTE_FAIL;
			// mbox_msg_rx(dest, msg_id + 1, &data);
			mbox_msg_tx(dest, msg_id, MBOX_SERV_MAGIC_NUM);
			mbox_msg_rx(dest, msg_id + 1, &data);
			result = (int8_t)data;
			break;
		}
		// drApiThreadSleep(1/*ms*/); //TODO
	}

	return result;
}

static int8_t mbox_serv_get_res(enum mbox_regions region)
{
	uint8_t i;
	int8_t result = MBOX_SERV_ERR_BUSY;

	for (i = 0; i < MBOX_SERV_TX_DELAY_CNT; i++) {
		// SuspendAllInterrupts();
		if (mbox_serv_mutex[(uint8_t)region] == false) {
			mbox_serv_mutex[(uint8_t)region] = true;
			// ResumeAllInterrupts();
			result = 0;
			break;
		}
		// ResumeAllInterrupts();

		// drApiThreadSleep(MBOX_SERV_TX_DELAY); //TODO
	}
	return result;
}

static void mbox_serv_release_res(enum mbox_regions region)
{
	mbox_serv_mutex[(uint8_t)region] = false;
}

static int8_t mbox_serv_is_ack(enum CPU_SERIAL dest, uint8_t msg_id)
{
	int8_t result = false;
	uint8_t i;
	uint32_t data = 0;


	for (i = 0; i < MBOX_SERV_TX_DELAY_CNT; i++) {
		/* Get target header to check that it received the msg or not */
		mbox_msg_rx(dest, msg_id, &data);

		/* Data is cleared after msg is taken */
		if (data == MBOX_SERV_MAGIC_NUM)  {
			result = true;
			break;
		}

		// drApiThreadSleep(MBOX_SERV_TX_DELAY); //TODO
	}

	return result;
}

static inline uint8_t mbox_serv_convert_msg_id(uint8_t idx)
{
	return idx * MBOX_ENTRY_SIZE;
}

static int8_t mbox_serv_search_msg_id(enum CPU_SERIAL src,
                                      enum CPU_SERIAL dest,
                                      enum mbox_regions *region,
                                      uint8_t *msg_id)
{
	uint8_t i;
	int8_t result = MBOX_SERV_ERR_FAIL;

	for (i = 0; i < MBOX_SERV_ENTRY_CNT; i++) {
		if (mbox_serv_tb[dest][i].cpu_serial == src) {
			*msg_id = mbox_serv_convert_msg_id(i);
			*region = mbox_serv_tb[dest][i].region;
			result = 0;
			break;
		}
	}

	return result;
}

static inline int8_t mbox_serv_q_is_empty(const struct mbox_serv_queue *q)
{
	int8_t result = false;

	if (q->in_idx == q->out_idx) {
		result = true;
	}

	return result;
}

static inline int8_t mbox_serv_q_is_full(const struct mbox_serv_queue *q)
{
	uint8_t idx;
	int8_t result = false;

	idx = (q->in_idx + 1U) % MBOX_SERV_QUEUE_SIZE;

	if (idx == q->out_idx) {
		result = true;
	}

	return result;
}

static inline void mbox_serv_enqueue(struct mbox_serv_queue *const q)
{
	if (q != NULL) {
		q->in_idx = (q->in_idx + 1U) % MBOX_SERV_QUEUE_SIZE;
	}
}

void mbox_serv_dequeue(struct mbox_serv_queue *const q)
{
	if (q != NULL) {
		q->out_idx = (q->out_idx + 1U) % MBOX_SERV_QUEUE_SIZE;
	}
}

void mbox_serv_isr(uint8_t msg_id)
{
	/* receive msg and put to rx queue */
	uint8_t i;
	uint8_t idx;
	uint8_t event_id;
	uint8_t offset;
	uint32_t header = 0;
	const struct mbox_serv_registry *p_registry;
	struct sys_event *p_event;
	int8_t result = MBOX_SERV_OK;

	/*
	 * Interrupt at lastest message queue.
	 * Get the first message queue on each entry
	 */
	offset = msg_id - MBOX_DATA_SIZE;

	/* decode msg */
	mbox_msg_rx(mbox_serv_serial, offset, &header);
	event_id = (uint8_t)(header & MBOX_SERV_EVENT_ID_MASK);
	idx = offset / MBOX_ENTRY_SIZE;

	/* if event id is not in the range or can't find callback, ignore it */
	if ((event_id < (uint8_t)SYS_EVENT_CNT) &&
	        (idx < MBOX_SERV_ENTRY_CNT)) {
		p_registry = &mbox_serv_registry_tb[event_id];

		if (p_registry->q != NULL) {
			/* if qeueu full, ignore the msg */
			if (mbox_serv_q_is_full(p_registry->q) == false) {
				p_event = &p_registry->q->event[p_registry->q->in_idx];

				/* put to user event */
				p_event->event_id = event_id;
				p_event->cpu_serial = p_mbox_serv[idx].cpu_serial;

				for (i = 0; i < MBOX_DATA_SIZE; i++) {
					mbox_msg_rx(mbox_serv_serial,
					            offset + 1U + i,
					            &p_event->data.data[i]);
				}

				/* update queue status */
				mbox_serv_enqueue(p_registry->q);

				if (p_registry->cbk.status == MBOX_SERV_CBK_STATUS_ENABLE)
					p_registry->cbk.fpCallback();

			} else {
				// TEE_MBOX_ERR("[\x1b[31m%s, %d\033[0m] MBOX_SERV_ERR_Q_FULL\n", __func__, __LINE__); //TODO
				result = MBOX_SERV_ERR_Q_FULL;
			}
		} else {
			// TEE_MBOX_ERR("[\x1b[31m%s, %d\033[0m] MBOX_SERV_ERR_NO_REGISTRY\n", __func__, __LINE__); //TODO
			result = MBOX_SERV_ERR_NO_REGISTRY;
		}
	} else {
		// TEE_MBOX_ERR("[\x1b[31m%s, %d\033[0m] MBOX_SERV_ERR_INVALID_ID\n", __func__, __LINE__); //TODO
		result = MBOX_SERV_ERR_INVALID_ID;
	}
	/* clear header as ACK. Let this region free */
	mbox_serv_ack(offset);

	// workaround, TEMP USED
	// if ((result == MBOX_SERV_OK) && (p_registry->cbk.status == MBOX_SERV_CBK_STATUS_ENABLE))
	// {
	// 	p_registry->cbk.fpCallback();
	// }

	(void)result;
}
void mbox_serv_init(void)
{
	uint8_t msg_id = -1;
	int i;
	enum mbox_regions region = MBOX_REGION_INVALID;

	/* init table pointer */
	mbox_serv_serial = CPU_SERIAL_CA53;
	p_mbox_serv = mbox_serv_tb[mbox_serv_serial];

	/* init event */
	(void)memset(&mbox_serv_registry_tb, 0, sizeof(mbox_serv_registry_tb));

	/* init mutex*/
	(void)memset(&mbox_serv_mutex, 0, sizeof(mbox_serv_mutex));

	/* init mbox driver and set callback function */
	mbox_init(mbox_serv_serial, mbox_serv_isr);

	for (i = 0; i < MBOX_SERV_ENTRY_CNT; i++) {
		/* enable latest message interrupt on each entry */
		// mbox_irq_enable(mbox_serv_convert_msg_id(i) + MBOX_DATA_SIZE);

		/* set Magic to show initial done */
		mbox_msg_tx(mbox_serv_serial,
		            mbox_serv_convert_msg_id(i),
		            MBOX_SERV_MAGIC_NUM);
	}


	mbox_serv_search_msg_id(CPU_SERIAL_HSM, CPU_SERIAL_CA53, &region, &msg_id);

	mbox_set_ca53_intr_msg_id(mbox_serv_convert_msg_id(region) + MBOX_DATA_SIZE);
}

int8_t mbox_serv_register_event(uint8_t event_id,
                                const struct mbox_serv_registry *registry)
{
	int8_t result = MBOX_SERV_ERR_FAIL;

	if ((event_id < (uint8_t)SYS_EVENT_CNT) && (registry != NULL)) {
		mbox_serv_registry_tb[event_id] = *registry;
		result = 0;
	}

	return result;
}

int8_t mbox_serv_tx(const struct sys_event *event)
{
	int8_t result;
	uint8_t msg_id = -1;
	uint8_t i;
	enum CPU_SERIAL dest;
	enum CPU_SERIAL src;
	enum mbox_regions region = MBOX_REGION_INVALID;
	uint32_t header;

	dest = event->cpu_serial;
	src = mbox_serv_serial;
	result = mbox_serv_search_msg_id(src, dest, &region, &msg_id);


	if ((result == 0) && (event->event_id < (uint8_t)SYS_EVENT_CNT)) {
		if (mbox_serv_get_res(region) == 0) {
			if (mbox_serv_is_ack(dest, msg_id) == true) {

				header = ((event->priority & 0xFFFF) << 16) | event->event_id;
				/* Put header first */
				mbox_msg_tx(dest,
				            msg_id,
				            header);
				/*
				 * Finally write the data to the latest message
				 * queue. We use latest msg q to trigger target
				 * MCU interrupt.
				 */
				for (i = 0; i < MBOX_DATA_SIZE; i++) {
					mbox_msg_tx(dest,
					            msg_id + i + 1U,
					            event->data.data[i]);
				}

				result = wait_remote_ack(dest, msg_id);

			} else {
				result = MBOX_SERV_ERR_BUSY;
			}

			mbox_serv_release_res(region);
		} else {
			result = MBOX_SERV_ERR_NO_RES;
		}
	} else {
		result = MBOX_SERV_ERR_INVALID_ID;
	}

	return result;
}


struct sys_event *mbox_serv_rx(struct mbox_serv_queue *const q)
{
	struct sys_event *event = NULL;

	if (q != NULL) {
		if (mbox_serv_q_is_empty(q) == false) {
			event = &q->event[q->out_idx];
		}
	}

	return event;
}

