// SPDX-License-Identifier: BSD-2-Clause
/*
 * Copyright (c) 2016, Linaro Limited
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <tee_internal_api.h>
#include <tee_internal_api_extensions.h>
#include <assert.h>
#include <vm_map_ta.h>
#include "string.h"
/*
 * Called when the instance of the TA is created. This is the first call in
 * the TA.
 */
TEE_Result TA_CreateEntryPoint(void)
{
	DMSG("has been called");

	return TEE_SUCCESS;
}

/*
 * Called when the instance of the TA is destroyed if the TA has not
 * crashed or panicked. This is the last call in the TA.
 */
void TA_DestroyEntryPoint(void)
{
	DMSG("has been called");
}

/*
 * Called when a new session is opened to the TA. *sess_ctx can be updated
 * with a value to be able to identify this session in subsequent calls to the
 * TA. In this function you will normally do the global initialization for the
 * TA.
 */
TEE_Result TA_OpenSessionEntryPoint(uint32_t param_types,
		TEE_Param __maybe_unused params[4],
		void __maybe_unused **sess_ctx)
{
	uint32_t exp_param_types = TEE_PARAM_TYPES(TEE_PARAM_TYPE_NONE,
						   TEE_PARAM_TYPE_NONE,
						   TEE_PARAM_TYPE_NONE,
						   TEE_PARAM_TYPE_NONE);

	DMSG("has been called");

	if (param_types != exp_param_types)
		return TEE_ERROR_BAD_PARAMETERS;

	/* Unused parameters */
	(void)&params;
	(void)&sess_ctx;

	/*
	 * The DMSG() macro is non-standard, TEE Internal API doesn't
	 * specify any means to logging from a TA.
	 */
	IMSG("Hello World!\n");

	/* If return value != TEE_SUCCESS the session will not be created. */
	return TEE_SUCCESS;
}

/*
 * Called when a session is closed, sess_ctx hold the value that was
 * assigned by TA_OpenSessionEntryPoint().
 */
void TA_CloseSessionEntryPoint(void __maybe_unused *sess_ctx)
{
	(void)&sess_ctx; /* Unused parameter */
	IMSG("Goodbye!\n");
}

void dbg_dump(uint8_t *buf, size_t buf_len)
{
	int i=0;
	DMSG("size:%zd", buf_len);
	for(i=0; i<buf_len/8; i++){
		DMSG("0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n", buf[i*8], buf[i*8+1], buf[i*8+2], buf[i*8+3], buf[i*8+4], buf[i*8+5], buf[i*8+6], buf[i*8+7]);
	}
	for(i*=8; i<buf_len; i++){
		DMSG("0x%02x", buf[i]);
	}
}

#define RANDOM_BUF_LEN  128
#define MAX_RW_LEN      0x200000
TEE_Result mem_rw_test(uint8_t* start, size_t len)
{
	len = len>MAX_RW_LEN ? MAX_RW_LEN : len;
	IMSG("test range:%p, len:%d", start, len);
	uint8_t randomBuffer[RANDOM_BUF_LEN] = { 0 };
	TEE_GenerateRandom(randomBuffer, RANDOM_BUF_LEN);
	// dbg_dump(randomBuffer, RANDOM_BUF_LEN);
	int i = 0;
	size_t i_len = len;
	i_len = (len/RANDOM_BUF_LEN)*RANDOM_BUF_LEN;
	for(i=0; i<i_len; i+=RANDOM_BUF_LEN){
		memcpy(start+i, randomBuffer, RANDOM_BUF_LEN);
		if (memcmp(start+i, randomBuffer, RANDOM_BUF_LEN)){
			EMSG("memory rw test fail at 0x%x.", start+i);
			EMSG("Read:");
			dbg_dump(start+i, RANDOM_BUF_LEN);
			EMSG("Expect:");
			dbg_dump(randomBuffer, RANDOM_BUF_LEN);
			return -1;
		}
	}
	return TEE_SUCCESS;
}

TEE_Result mem_rw_rand_pos_test(uint8_t* start, size_t len)
{
	DMSG("test range:%p, len:%d", start, len);
	if (len<0x100000){
		return mem_rw_test(start, len);
	}
	int i = 0;
	size_t rand_pos = 0;
	uint8_t* new_start=start;
	size_t new_len;
	size_t rest_len;
	TEE_Result res=TEE_SUCCESS;
	for(i=0; i<10; i++){
		TEE_GenerateRandom(&rand_pos, sizeof(rand_pos));
		rand_pos = rand_pos%len;
		DMSG("Generated rand position:%x", rand_pos);
		new_start = start+rand_pos;
		rest_len = (start+len)-new_start;
		new_len = rand_pos & (MAX_RW_LEN-1);
		new_len = new_len > len ? new_len%len : new_len;
		new_len = new_len > rest_len ? rest_len : new_len;
		res = mem_rw_test(new_start, new_len);
		if (res!=TEE_SUCCESS){
			break;
		}
	}
	return res;
}

static TEE_Result map_mem(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t va = 0;
	rc = TEE_MapPhysicalMemory(0x47000000, 0x1000000, 0, &va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	DMSG("Read back va address value:%x", *(uint64_t*)va);
	mem_rw_test((uint8_t*)va, 0x1000000);
	rc = TEE_UnmapPhysicalMemory(va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	// This should abort.
	// DMSG("Read back va address value:%x", *(uint64_t*)va);
	return rc;
}

void get_mem_para(uint32_t param_types, TEE_Param params[4], uint64_t* start, size_t* len, uint32_t* attr)
{
	uint32_t exp_param_types = TEE_PARAM_TYPES(TEE_PARAM_TYPE_VALUE_INOUT,
						   TEE_PARAM_TYPE_VALUE_INOUT,
						   TEE_PARAM_TYPE_VALUE_INPUT,
						   TEE_PARAM_TYPE_VALUE_INPUT);

	DMSG("has been called");

	if (param_types != exp_param_types)
		return TEE_ERROR_BAD_PARAMETERS;
	DMSG("0: 0x%x 1:0x%x 2:0x%x, 3:0x%x", params[0].value.a, params[1].value.a, params[2].value.a, params[3].value.a);
	uint64_t tmp = (uint64_t)(params[3].value.a)<<32;
	*start = tmp | params[0].value.a;
	*len   = params[1].value.a;
	*attr  = params[2].value.a;
	DMSG("start addr: 0x%llx, len:%d, attr:%x", *start, *len, *attr);
}

static TEE_Result mpu_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t va;
	rc = TEE_EMIMPUProtect(0x47000000, 0x10000, 1);
	DMSG("TEE_EMIMPUProtect res: %x", rc);

	rc = TEE_EMIMPUUnprotect(0x47000000);
	DMSG("TEE_EMIMPUUnprotect res: %x", rc);
	return rc;
}

static TEE_Result mpu_protect_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &pa, &len, &attr);
	rc = TEE_EMIMPUProtect(pa, len, attr);
	DMSG("TEE_EMIMPUProtect res: %x", rc);
	return rc;
}


static TEE_Result mpu_unprotect_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &pa, &len, &attr);
	rc = TEE_EMIMPUUnprotect(pa);
	DMSG("TEE_EMIMPUUnprotect res: %x", rc);
	return rc;
}

static TEE_Result mem_map_random_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &pa, &len, &attr);
	uint64_t va = 0;
	rc = TEE_MapPhysicalMemory(pa, len, attr, &va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	rc = mem_rw_rand_pos_test(va, len);
	DMSG("mem_rw_rand_pos_test res: %x", rc);
	if (rc){
		return rc;
	}
	rc = TEE_UnmapPhysicalMemory(va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	return rc;
}

static TEE_Result mem_map_simple_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	int i =0;
	get_mem_para(param_types, params, &pa, &len, &attr);
	uint64_t va = 0;
	uint32_t* va_off;
	uint32_t* va_a;
	rc = TEE_MapPhysicalMemory(pa, len, attr, &va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	va_a = (uint32_t*)va;
	for (i=0; i<64; i++){
		va_a[i] = i;
	}
	for (i=0; i<64; i++){
		DMSG("addr: %p, val:%d", (&va_a[i]), va_a[i]);
	}
    va_off = va_a+64;
	for (i=0; i<64; i++){
		DMSG("addr: %p, val:%d", (&va_off[i]), va_off[i]);
	}
	return rc;
}

static TEE_Result mem_map_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &pa, &len, &attr);
	uint64_t va = 0;
	DMSG("TEE_MapPhysicalMemory pa: %llx", pa);
	rc = TEE_MapPhysicalMemory(pa, len, attr, &va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	rc = mem_rw_rand_pos_test(va, len);
	DMSG("mem_rw_rand_pos_test res: %x", rc);
	return rc;
}

static TEE_Result mem_unmap_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc;
	uint64_t va;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &va, &len, &attr);
	rc = TEE_UnmapPhysicalMemory(va);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va);
	return rc;
}

static TEE_Result mem_map_twice_test(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc, rc0;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &pa, &len, &attr);
	uint64_t va1 = 0;
	uint64_t va2 = 0;
	size_t len1 = len/2;

	rc = TEE_MapPhysicalMemory(pa, len1, attr, &va1);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va1);
	rc = mem_rw_rand_pos_test(va1, len1);
	DMSG("mem_rw_rand_pos_test res: %x", rc);
	if (rc){
		return rc;
	}
	rc = TEE_UnmapPhysicalMemory(va1);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va1);
	//
	rc = TEE_MapPhysicalMemory(pa+len1, len1, attr, &va2);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va2);
	rc = mem_rw_rand_pos_test(va2, len1);
	DMSG("mem_rw_rand_pos_test res: %x", rc);
	if (rc){
		return rc;
	}
	rc = TEE_UnmapPhysicalMemory(va2);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va2);
	return rc;
}
static TEE_Result mem_map_two_region(uint32_t param_types,
	TEE_Param params[4])
{
	DMSG("has been called");
	TEE_Result rc, rc0;
	uint64_t pa;
	size_t len;
	uint32_t attr;
	get_mem_para(param_types, params, &pa, &len, &attr);
	uint64_t va1 = 0;
	uint64_t va2 = 0;
	size_t len1 = len/2;

	rc = TEE_MapPhysicalMemory(pa, len1, attr, &va1);
	DMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc, va1);
	rc0 = TEE_MapPhysicalMemory(pa+len1, len1, attr, &va2);
	//The seconde map should be fail.
	EMSG("TEE_MapPhysicalMemory res: %x, va: %p", rc0, va2);

	rc = TEE_UnmapPhysicalMemory(va1);
	DMSG("TEE_UnmapPhysicalMemory res: %x, va: %p", rc, va1);
	return rc0;
}

/*
 * Called when a TA is invoked. sess_ctx hold that value that was
 * assigned by TA_OpenSessionEntryPoint(). The rest of the paramters
 * comes from normal world.
 */
TEE_Result TA_InvokeCommandEntryPoint(void __maybe_unused *sess_ctx,
			uint32_t cmd_id,
			uint32_t param_types, TEE_Param params[4])
{
	(void)&sess_ctx; /* Unused parameter */

	switch (cmd_id) {
	case TA_VM_MAP_CMD_MPU:
		return mpu_test(param_types, params);
	case TA_VM_MAP_CMD_MPU_PROTECT:
		return mpu_protect_test(param_types, params);
	case TA_VM_MAP_CMD_MPU_UNPROTECT:
		return mpu_unprotect_test(param_types, params);
	case TA_VM_MAP_CMD_MAP_MEM:
		return map_mem(param_types, params);
	case TA_VM_MAP_CMD_MEM_MAP_RANDOM:
		return mem_map_random_test(param_types, params);
	case TA_VM_MAP_CMD_MEM_MAP_TWICE:
		return mem_map_twice_test(param_types, params);
	case TA_VM_MAP_CMD_MEM_MAP_TWO_REGION:
		return mem_map_two_region(param_types, params);
	case TA_VM_MAP_CMD_MEM_MAP:
		return mem_map_test(param_types, params);
	case TA_VM_MAP_CMD_MEM_UNMAP:
		return mem_unmap_test(param_types, params);
	case TA_VM_MAP_CMD_MEM_MAP_SIMPLE:
		return mem_map_simple_test(param_types, params);
	default:
		return TEE_ERROR_BAD_PARAMETERS;
	}
}
