// SPDX-License-Identifier: BSD-2-Clause
/*
 * Copyright (c) 2017-2018, Linaro Limited
 */

#include <assert.h>
#include <sks_internal_abi.h>
#include <sks_ta.h>
#include <string.h>
#include <tee_api_defines.h>
#include <tee_internal_api.h>
#include <tee_internal_api_extensions.h>
#include <util.h>

#include "attributes.h"
#include "object.h"
#include "pkcs11_token.h"
#include "pkcs11_attributes.h"
#include "processing.h"
#include "serializer.h"
#include "sks_helpers.h"

#include "processing_mtk_hsm.h"
#include "processing_mtk_key.h"

static uint32_t get_ready_session(struct pkcs11_session **sess,
				  uint32_t session_handle,
				  uintptr_t tee_session)
{
	struct pkcs11_session *session = NULL;

	session = sks_handle2session(session_handle, tee_session);
	if (!session)
		return SKS_CKR_SESSION_HANDLE_INVALID;

	if (session_is_active(session))
		return SKS_CKR_OPERATION_ACTIVE;

	*sess = session;

	return SKS_OK;
}

static bool func_matches_state(enum processing_func function,
				enum pkcs11_proc_state state)
{
	switch (function) {
	case SKS_FUNCTION_ENCRYPT:
		return (state == PKCS11_SESSION_ENCRYPTING ||
			state == PKCS11_SESSION_DIGESTING_ENCRYPTING ||
			state == PKCS11_SESSION_SIGNING_ENCRYPTING);
	case SKS_FUNCTION_DECRYPT:
		return (state == PKCS11_SESSION_DECRYPTING ||
			state == PKCS11_SESSION_DECRYPTING_DIGESTING ||
			state == PKCS11_SESSION_DECRYPTING_VERIFYING);
	case SKS_FUNCTION_DIGEST:
		return (state == PKCS11_SESSION_DIGESTING ||
			state == PKCS11_SESSION_DIGESTING_ENCRYPTING);
	case SKS_FUNCTION_SIGN:
		return (state == PKCS11_SESSION_SIGNING ||
			state == PKCS11_SESSION_SIGNING_ENCRYPTING);
	case SKS_FUNCTION_VERIFY:
		return (state == PKCS11_SESSION_VERIFYING ||
			state == PKCS11_SESSION_DECRYPTING_VERIFYING);
	case SKS_FUNCTION_SIGN_RECOVER:
		return state == PKCS11_SESSION_SIGNING_RECOVER;
	case SKS_FUNCTION_VERIFY_RECOVER:
		return state == PKCS11_SESSION_SIGNING_RECOVER;
	default:
		TEE_Panic(function);
		return false;
	}
}

static uint32_t get_active_session(struct pkcs11_session **sess,
				  uint32_t session_handle,
				  uintptr_t tee_session,
				  enum processing_func function)
{
	struct pkcs11_session *session = NULL;
	uint32_t rv = SKS_CKR_OPERATION_NOT_INITIALIZED;

	session = sks_handle2session(session_handle, tee_session);
	if (!session)
		return SKS_CKR_SESSION_HANDLE_INVALID;

	if (session->processing &&
	    func_matches_state(function, session->processing->state)) {
		*sess = session;
		rv = SKS_OK;
	}

	return rv;
}

void release_active_processing(struct pkcs11_session *session)
{
	if (!session->processing)
		return;

	switch (session->processing->mecha_type) {
	case SKS_CKM_AES_CTR:
		tee_release_ctr_operation(session->processing);
		break;
	case SKS_CKM_AES_GCM:
		tee_release_gcm_operation(session->processing);
		break;
	case SKS_CKM_AES_CCM:
		tee_release_ccm_operation(session->processing);
		break;
	case SKS_CKM_SHA1_RSA_PKCS_PSS:
	case SKS_CKM_SHA256_RSA_PKCS_PSS:
	case SKS_CKM_SHA384_RSA_PKCS_PSS:
	case SKS_CKM_SHA512_RSA_PKCS_PSS:
	case SKS_CKM_SHA224_RSA_PKCS_PSS:
		tee_release_rsa_pss_operation(session->processing);
		break;
	default:
		break;
	}

	if (session->processing->mtk_hsm_handle != TEE_HANDLE_NULL) {
		TEE_Free(session->processing->mtk_hsm_handle);
		session->processing->mtk_hsm_handle = TEE_HANDLE_NULL;
	}

	if (session->processing->tee_op_handle != TEE_HANDLE_NULL) {
		TEE_FreeOperation(session->processing->tee_op_handle);
		session->processing->tee_op_handle = TEE_HANDLE_NULL;
	}

	if (session->processing->extra_op_handle != TEE_HANDLE_NULL) {
		TEE_FreeOperation(session->processing->extra_op_handle);
		session->processing->extra_op_handle = TEE_HANDLE_NULL;
	}

	TEE_Free(session->processing);
	session->processing = NULL;
}

uint32_t entry_import_object(uintptr_t tee_session,
			     TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attrs_head *head = NULL;
	struct sks_object_head *template = NULL;
	size_t template_size = 0;
	uint32_t obj_handle = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	/*
	 * Collect the arguments of the request
	 */

	if (!ctrl || in || !out)
		return SKS_BAD_PARAM;

	if (out->memref.size < sizeof(uint32_t)) {
		out->memref.size = sizeof(uint32_t);
		return SKS_SHORT_BUFFER;
	}

	if ((uintptr_t)out->memref.buffer & 0x3UL)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, tee_session);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;

	/*
	 * Prepare a clean initial state for the requested object attributes.
	 * Free temporary template once done.
	 */
	rv = create_attributes_from_template(&head, template, template_size,
					     SKS_UNDEFINED_ID, NULL,
					     SKS_FUNCTION_IMPORT);
	TEE_Free(template);
	template = NULL;
	if (rv)
		goto bail;

	/*
	 * Check target object attributes match target processing
	 * Check target object attributes match token state
	 */
	rv = check_created_attrs_against_processing(SKS_PROCESSING_IMPORT,
						    head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, head);
	if (rv)
		goto bail;

	/*
	 * TODO: test object (will check all expected attributes are in place
	 */

	/*
	 * At this stage the object is almost created: all its attributes are
	 * referenced in @head, including the key value and are assume
	 * reliable. Now need to register it and get a handle for it.
	 */
	rv = create_object(session, head, &obj_handle);
	if (rv)
		goto bail;

	/*
	 * Now obj_handle (through the related struct sks_object instance)
	 * owns the serialised buffer that holds the object attributes.
	 * We reset attrs->buffer to NULL as serializer object is no more
	 * the attributes buffer owner.
	 */
	head = NULL;

	TEE_MemMove(out->memref.buffer, &obj_handle, sizeof(uint32_t));
	out->memref.size = sizeof(uint32_t);

	IMSG("SKSs%" PRIu32 ": import object 0x%" PRIx32,
	     session_handle, obj_handle);

bail:
	TEE_Free(template);
	TEE_Free(head);

	return rv;
}

size_t get_object_key_bit_size(struct sks_object *obj)
{
	void *a_ptr = NULL;
	uint32_t a_size = 0;
	struct sks_attrs_head *attrs = obj->attributes;

	switch (get_type(attrs)) {
	case SKS_CKK_AES:
	case SKS_CKK_GENERIC_SECRET:
	case SKS_CKK_MD5_HMAC:
	case SKS_CKK_SHA_1_HMAC:
	case SKS_CKK_SHA224_HMAC:
	case SKS_CKK_SHA256_HMAC:
	case SKS_CKK_SHA384_HMAC:
	case SKS_CKK_SHA512_HMAC:
		if (get_attribute_ptr(attrs, SKS_CKA_VALUE, NULL, &a_size))
			return 0;

		return a_size * 8;

	case SKS_CKK_RSA:
		if (get_attribute_ptr(attrs, SKS_CKA_MODULUS, NULL, &a_size))
			return 0;

		return a_size * 8;

	case SKS_CKK_EC:
		if (get_attribute_ptr(attrs, SKS_CKA_EC_PARAMS,
					&a_ptr, &a_size))
			return 0;

		return ec_params2tee_keysize(a_ptr, a_size);

	default:
		TEE_Panic(0);
		return 0;
	}
}

static uint32_t generate_random_key_value(struct sks_attrs_head **head)
{
	uint32_t rv = 0;
	void *data;
	uint32_t data_size;
	uint32_t value_len;
	void *value;
	int keyid = 0;

	if (!*head)
		return SKS_CKR_TEMPLATE_INCONSISTENT;

	rv = get_attribute_ptr(*head, SKS_CKA_VALUE_LEN, &data, &data_size);
	if (rv || data_size != sizeof(uint32_t)) {
		DMSG("%s", rv ? "No attribute value_len found" :
			"Invalid size for attribute VALUE_LEN");
		return SKS_CKR_ATTRIBUTE_VALUE_INVALID;
	}
	TEE_MemMove(&value_len, data, data_size);

	if (get_type(*head) == SKS_CKK_GENERIC_SECRET)
		value_len = (value_len + 7) / 8;

	value = TEE_Malloc(value_len, TEE_USER_MEM_HINT_NO_FILL_ZERO);
	if (!value)
		return SKS_MEMORY;

	TEE_GenerateRandom(value, value_len);

	rv = add_attribute(head, SKS_CKA_VALUE, value, value_len);

	TEE_Free(value);

	return rv;
}

static uint32_t generate_hsm_key(struct sks_attrs_head **head)
{
	uint32_t rv = 0;
	void *data;
	uint32_t data_size;
	uint32_t value_len;
	void *value;
	int keyid = 0;

	if (!*head)
		return SKS_CKR_TEMPLATE_INCONSISTENT;

	rv = get_attribute_ptr(*head, SKS_CKA_VALUE_LEN, &data, &data_size);
	if (rv || data_size != sizeof(uint32_t)) {
		DMSG("%s", rv ? "No attribute value_len found" :
			"Invalid size for attribute VALUE_LEN");
		return SKS_CKR_ATTRIBUTE_VALUE_INVALID;
	}
	TEE_MemMove(&value_len, data, data_size);

	if (get_type(*head) == SKS_CKK_GENERIC_SECRET)
		value_len = (value_len + 7) / 8;

	rv = mtk_generate_symmetric_key(value_len, &keyid);
	if (rv)
		return SKS_CKR_GENERAL_ERROR;

	rv = add_attribute(head, SKS_CKA_HSM_KEY_ID, &keyid, sizeof(keyid));
	if (rv)
		return SKS_CKR_GENERAL_ERROR;

	rv = add_attribute(head, SKS_CKA_HSM_KEY_LEN, &value_len, sizeof(value_len));
	if (rv )
		return SKS_CKR_GENERAL_ERROR;

	return rv;
}

uint32_t entry_generate_secret(uintptr_t tee_session,
			       TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	struct sks_attrs_head *head = NULL;
	struct sks_object_head *template = NULL;
	size_t template_size = 0;
	uint32_t obj_handle = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || in || !out)
		return SKS_BAD_PARAM;

	if (out->memref.size < sizeof(uint32_t)) {
		out->memref.size = sizeof(uint32_t);
		return SKS_SHORT_BUFFER;
	}

	if ((uintptr_t)out->memref.buffer & 0x3UL)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, tee_session);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	switch (proc_params->id) {
	case SKS_CKM_GENERIC_SECRET_KEY_GEN:
	case SKS_CKM_AES_KEY_GEN:
	case SKS_CKM_MTK_HSM_AES_KEY_GEN:
		break;
	default:
		rv = SKS_CKR_MECHANISM_INVALID;
		goto bail;
	}

	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;

	rv = check_mechanism_against_processing(session, proc_params->id,
						SKS_FUNCTION_GENERATE,
						SKS_FUNC_STEP_INIT);
	if (rv)
		goto bail;

	/*
	 * Prepare a clean initial state for the requested object attributes.
	 * Free temporary template once done.
	 */
	rv = create_attributes_from_template(&head, template, template_size,
					     proc_params->id, NULL,
					     SKS_FUNCTION_GENERATE);
	if (rv)
		goto bail;

	TEE_Free(template);
	template = NULL;

	rv = check_created_attrs(head, NULL);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_processing(proc_params->id, head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, head);
	if (rv)
		goto bail;

	/* TODO : Not finished */
	// rv = check_key_object_not_generated(session, head);
	// if (rv)
	// {
	// 	IMSG("[%s][%d] check_key_object_not_generated\n", __FUNCTION__, __LINE__);
	// 	goto bail;
	// }

	/*
	 * Execute target processing and add value as attribute SKS_CKA_VALUE.
	 * Symm key generation: depends on target processing to be used.
	 */
	switch (proc_params->id) {
	case SKS_CKM_GENERIC_SECRET_KEY_GEN:
	case SKS_CKM_AES_KEY_GEN:
		/* Generate random of size specified by attribute VALUE_LEN */
		rv = generate_random_key_value(&head);
		if (rv)
			goto bail;
		break;

	case SKS_CKM_MTK_HSM_AES_KEY_GEN:
		rv = generate_hsm_key(&head);
		if (rv)
			goto bail;
		break;
	default:
		rv = SKS_CKR_MECHANISM_INVALID;
		goto bail;
	}

	TEE_Free(proc_params);
	proc_params = NULL;

	/*
	 * Object is ready, register it and return a handle.
	 */
	rv = create_object(session, head, &obj_handle);
	if (rv)
		goto bail;

	/*
	 * Now obj_handle (through the related struct sks_object instance)
	 * owns the serialized buffer that holds the object attributes.
	 * We reset attrs->buffer to NULL as serializer object is no more
	 * the attributes buffer owner.
	 */
	head = NULL;

	TEE_MemMove(out->memref.buffer, &obj_handle, sizeof(uint32_t));
	out->memref.size = sizeof(uint32_t);

	IMSG("SKSs%" PRIu32 ": generate secret 0x%" PRIx32,
	     session_handle, obj_handle);

bail:
	TEE_Free(proc_params);
	TEE_Free(template);
	TEE_Free(head);

	return rv;
}

uint32_t alloc_get_tee_attribute_data(TEE_ObjectHandle tee_obj,
					     uint32_t attribute,
					     void **data, size_t *size)
{
	TEE_Result res = TEE_ERROR_GENERIC;
	void *ptr = NULL;
	uint32_t sz = 0;

	res = TEE_GetObjectBufferAttribute(tee_obj, attribute, NULL, &sz);
	if (res != TEE_ERROR_SHORT_BUFFER)
		return SKS_FAILED;

	ptr = TEE_Malloc(sz, TEE_USER_MEM_HINT_NO_FILL_ZERO);
	if (!ptr)
		return SKS_MEMORY;

	res = TEE_GetObjectBufferAttribute(tee_obj, attribute, ptr, &sz);
	if (res) {
		TEE_Free(ptr);
	} else {
		*data = ptr;
		*size = sz;
	}

	return tee2sks_error(res);
}

uint32_t tee2sks_add_attribute(struct sks_attrs_head **head, uint32_t sks_id,
				TEE_ObjectHandle tee_obj, uint32_t tee_id)
{
	uint32_t rv = 0;
	void *a_ptr = NULL;
	size_t a_size = 0;

	rv = alloc_get_tee_attribute_data(tee_obj, tee_id, &a_ptr, &a_size);
	if (rv)
		goto bail;

	rv = add_attribute(head, sks_id, a_ptr, a_size);

	TEE_Free(a_ptr);

bail:
	if (rv)
		EMSG("Failed TEE attribute 0x%" PRIx32 "for %s (0x%" PRIx32 ")",
				tee_id, sks2str_attr(sks_id), sks_id);
	return rv;
}

uint32_t entry_generate_key_pair(uintptr_t teesess,
				 TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	struct sks_attrs_head *pub_head = NULL;
	struct sks_attrs_head *priv_head = NULL;
	struct sks_object_head *template = NULL;
	size_t template_size = 0;
	uint32_t pubkey_handle = 0;
	uint32_t privkey_handle = 0;
	uint32_t *hdl_ptr = NULL;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || in || !out)
		return SKS_BAD_PARAM;

	if (out->memref.size < 2 * sizeof(uint32_t))
		return SKS_SHORT_BUFFER;

	// FIXME: cleaner way to test alignment of out buffer
	if ((uintptr_t)out->memref.buffer & 0x3UL)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, teesess);
	if (rv)
		return rv;

	/* Get mechanism parameters */
	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	rv = check_mechanism_against_processing(session, proc_params->id,
						SKS_FUNCTION_GENERATE_PAIR,
						SKS_FUNC_STEP_INIT);
	if (rv)
		goto bail;

	switch (proc_params->id) {
	case SKS_CKM_EC_KEY_PAIR_GEN:
	case SKS_CKM_RSA_PKCS_KEY_PAIR_GEN:
	case SKS_CKM_MTK_HSM_EC_KEY_PAIR_GEN:
		break;
	default:
		rv = SKS_CKR_MECHANISM_INVALID;
		goto bail;
	}

	/* Get and check public key attributes */
	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;

	rv = create_attributes_from_template(&pub_head, template, template_size,
					     proc_params->id, NULL,
					     SKS_FUNCTION_GENERATE_PAIR);
	if (rv)
		goto bail;

	TEE_Free(template);
	template = NULL;

	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;

	rv = create_attributes_from_template(&priv_head, template, template_size,
					     proc_params->id, NULL,
					     SKS_FUNCTION_GENERATE_PAIR);
	if (rv)
		goto bail;

	TEE_Free(template);
	template = NULL;

	/* Generate CKA_ID for keys if not specified by the templates */
	rv = add_missing_attribute_id(&pub_head, &priv_head);
	if (rv)
		goto bail;

	/* Check created object against processing and token state */
	rv = check_created_attrs(pub_head, priv_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_processing(proc_params->id, pub_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_processing(proc_params->id, priv_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, pub_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, priv_head);
	if (rv)
		goto bail;

	/* Generate key pair */
	switch (proc_params->id) {
	case SKS_CKM_EC_KEY_PAIR_GEN:
		rv = generate_ec_keys(proc_params, &pub_head, &priv_head);
		break;
	case SKS_CKM_MTK_HSM_EC_KEY_PAIR_GEN:
		rv = generate_hsm_ec_keys(proc_params, &pub_head, &priv_head);
		break;
	case SKS_CKM_RSA_PKCS_KEY_PAIR_GEN:
		rv = generate_rsa_keys(proc_params, &pub_head, &priv_head);
		break;
	default:
		rv = SKS_CKR_MECHANISM_INVALID;
		break;
	}
	if (rv)
		goto bail;

	TEE_Free(proc_params);
	proc_params = NULL;

	/*
	 * Object is ready, register it and return a handle.
	 */
	rv = create_object(session, pub_head, &pubkey_handle);
	if (rv)
		goto bail;

	rv = create_object(session, priv_head, &privkey_handle);
	if (rv)
		goto bail;

	/*
	 * Now obj_handle (through the related struct sks_object instance)
	 * owns the serialized buffer that holds the object attributes.
	 * We reset attrs->buffer to NULL as serializer object is no more
	 * the attributes buffer owner.
	 */
	pub_head = NULL;
	priv_head = NULL;
	hdl_ptr = (uint32_t *)out->memref.buffer;

	TEE_MemMove(hdl_ptr, &pubkey_handle, sizeof(uint32_t));
	TEE_MemMove(hdl_ptr + 1, &privkey_handle, sizeof(uint32_t));
	out->memref.size = 2 * sizeof(uint32_t);

	IMSG("SKSs%" PRIu32 ": create key pair 0x%" PRIx32 "/0x%" PRIx32,
	     session_handle, privkey_handle, pubkey_handle);

bail:
	TEE_Free(proc_params);
	TEE_Free(template);
	TEE_Free(pub_head);
	TEE_Free(priv_head);

	return rv;
}

uint32_t entry_digesting_init(uintptr_t tee_session, TEE_Param *ctrl,
				TEE_Param *in, TEE_Param *out,
				enum processing_func function)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	struct sks_object *obj = NULL;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || in || out)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, tee_session);
	if (rv)
		return rv;

	rv = set_processing_state(session, function, obj, NULL);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	if (proc_params->id & SKS_CKM_MTK_HSM_EXT)
	{
		switch (proc_params->id)
		{
		case SKS_CKM_MTK_HSM_SHA1:
		case SKS_CKM_MTK_HSM_SHA224:
		case SKS_CKM_MTK_HSM_SHA256:
		case SKS_CKM_MTK_HSM_SHA384:
		case SKS_CKM_MTK_HSM_SHA512:
		case SKS_CKM_MTK_HSM_WHP:
			rv = mtk_SHA_init(session, proc_params);
			break;

		default:
			rv = SKS_BAD_PARAM;
			break;
		}

		if (rv == SKS_OK) {
			session->processing->mecha_type = proc_params->id;
			DMSG("SKSs%" PRIu32 ": init processing %s %s",
			session_handle, sks2str_proc(proc_params->id),
			sks2str_function(function));
		} else {
			EMSG(" entry_digesting_init FAIL!!  \n");
		}

		goto bail;
	}


bail:
	if (rv && session)
		release_active_processing(session);

	TEE_Free(proc_params);

	return rv;
}

uint32_t entry_digesting_step(uintptr_t tee_session, TEE_Param *ctrl,
				TEE_Param *in, TEE_Param *out,
				enum processing_func function,
				enum processing_step step)
{

	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	uint32_t mecha_type = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_active_session(&session, session_handle, tee_session,
				function);
	if (rv)
		return rv;

	mecha_type = session->processing->mecha_type;
	if (mecha_type & SKS_CKM_MTK_HSM_EXT)
	{
		switch (mecha_type)
		{
		case SKS_CKM_MTK_HSM_SHA1:
		case SKS_CKM_MTK_HSM_SHA224:
		case SKS_CKM_MTK_HSM_SHA256:
		case SKS_CKM_MTK_HSM_SHA384:
		case SKS_CKM_MTK_HSM_SHA512:
		case SKS_CKM_MTK_HSM_WHP:
			rv = mtk_SHA_step(mecha_type, session, in, out, function, step);
			break;

		default:
			rv = SKS_BAD_PARAM;
			break;
		}

		goto bail;
	}

bail:
	switch (step) {
	case SKS_FUNC_STEP_UPDATE:
		if (rv != SKS_OK && rv != SKS_SHORT_BUFFER)
			release_active_processing(session);
		break;
	default:
		/* ONESHOT and FINAL terminates processing on success */
		if (rv != SKS_SHORT_BUFFER)
			release_active_processing(session);
		break;
	}

	return rv;
}


/*
 * entry_processing_init - Generic entry for initializing a processing
 *
 * @ctrl = [session-handle]
 * @in = input data or none
 * @out = output data or none
 * @function - encrypt, decrypt, sign, verify, digest, ...
 *
 * The generic part come that all the commands uses the same
 * input/output invocation parameters format (ctrl/in/out).
 */
uint32_t entry_processing_init(uintptr_t tee_session, TEE_Param *ctrl,
				TEE_Param *in, TEE_Param *out,
				enum processing_func function)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	uint32_t key_handle = 0;
	struct sks_object *obj = NULL;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || in || out)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, tee_session);
	if (rv)
		return rv;

	rv = serialargs_get(&ctrlargs, &key_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	obj = sks_handle2object(key_handle, session);
	if (!obj)
		return SKS_CKR_KEY_HANDLE_INVALID;

	rv = set_processing_state(session, function, obj, NULL);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	rv = check_mechanism_against_processing(session, proc_params->id,
						function, SKS_FUNC_STEP_INIT);
	if (rv)
		goto bail;

	rv = check_parent_attrs_against_processing(proc_params->id, function,
						   obj->attributes);
	if (rv)
		goto bail;

	// rv = check_access_attrs_against_token(session, obj->attributes);
	// if (rv)
	// {
	// 	IMSG("[%s][%d] rv %ld\n", __FUNCTION__, __LINE__, rv);
	// 	goto bail;
	// }

	if (proc_params->id & SKS_CKM_MTK_HSM_EXT)
	{
		switch (proc_params->id)
		{
		case SKS_CKM_MTK_HSM_AES_ECB:
			rv = mtk_AES_ECB_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_AES_CBC:
			rv = mtk_AES_CBC_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_AES_CTR:
			rv = mtk_AES_CTR_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_AES_GCM:
			rv = mtk_AES_GCM_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_AES_CMAC:
			rv = mtk_AES_CMAC_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_SHA256_HMAC:
		case SKS_CKM_MTK_HSM_SHA384_HMAC:
			rv = mtk_HMAC_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_ECDSA:
		case SKS_CKM_MTK_HSM_ECDSA_SM2:
			rv = mtk_ECDSA_init(session, proc_params, obj);
			break;

		case SKS_CKM_MTK_HSM_ECDSA_SHA1:
		case SKS_CKM_MTK_HSM_ECDSA_SHA224:
		case SKS_CKM_MTK_HSM_ECDSA_SHA256:
		case SKS_CKM_MTK_HSM_ECDSA_SHA384:
		case SKS_CKM_MTK_HSM_ECDSA_SHA512:
			rv = mtk_ECDSA_SHA_init(session, proc_params, obj);
			break;

		default:
			rv = SKS_BAD_PARAM;
			break;
		}

		if (rv == SKS_OK) {
			session->processing->mecha_type = proc_params->id;
			DMSG("SKSs%" PRIu32 ": init processing %s %s",
			session_handle, sks2str_proc(proc_params->id),
			sks2str_function(function));
		} else {
			EMSG(" entry_processing_init FAIL!!  \n");
		}

		goto bail;
	}

	rv = SKS_CKR_MECHANISM_INVALID;
	if (processing_is_tee_symm(proc_params->id)) {
		rv = init_symm_operation(session, function, proc_params, obj);
	}
	if (processing_is_tee_asymm(proc_params->id)) {
		rv = init_asymm_operation(session, function, proc_params, obj);
	}
	if (rv == SKS_OK) {
		session->processing->mecha_type = proc_params->id;
		IMSG("SKSs%" PRIu32 ": init processing %s %s",
		     session_handle, sks2str_proc(proc_params->id),
		     sks2str_function(function));
	}

bail:
	if (rv && session)
		release_active_processing(session);

	TEE_Free(proc_params);

	return rv;
}

/*
 * entry_processing_step - Generic entry on active processing
 *
 * @ctrl = [session-handle]
 * @in = input data or none
 * @out = output data or none
 * @function - encrypt, decrypt, sign, verify, digest, ...
 * @step - update, oneshot, final
 *
 * The generic part come that all the commands uses the same
 * input/output invocation parameters format (ctrl/in/out).
 */
uint32_t entry_processing_step(uintptr_t tee_session, TEE_Param *ctrl,
				TEE_Param *in, TEE_Param *out,
				enum processing_func function,
				enum processing_step step)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	uint32_t mecha_type = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_active_session(&session, session_handle, tee_session,
				function);
	if (rv)
		return rv;

	// TODO: check user authen and object activation dates
	rv = check_mechanism_against_processing(session, mecha_type,
						function, step);
	if (rv)
		goto bail;

	mecha_type = session->processing->mecha_type;
	rv = check_mechanism_against_processing(session, mecha_type,
						function, step);
	if (rv)
		goto bail;

	if (mecha_type & SKS_CKM_MTK_HSM_EXT)
	{
		switch (mecha_type)
		{
		case SKS_CKM_MTK_HSM_AES_ECB:
			rv = mtk_AES_ECB_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_AES_CBC:
			rv = mtk_AES_CBC_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_AES_CTR:
			rv = mtk_AES_CTR_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_AES_GCM:
			rv = mtk_AES_GCM_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_AES_CMAC:
			rv = mtk_AES_CMAC_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_SHA256_HMAC:
			rv = mtk_SHA256_HMAC_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_SHA384_HMAC:
			rv = mtk_SHA384_HMAC_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_ECDSA:
		case SKS_CKM_MTK_HSM_ECDSA_SM2:
			rv = mtk_ECDSA_step(session, in, out, function, step);
			break;

		case SKS_CKM_MTK_HSM_ECDSA_SHA1:
		case SKS_CKM_MTK_HSM_ECDSA_SHA224:
		case SKS_CKM_MTK_HSM_ECDSA_SHA256:
		case SKS_CKM_MTK_HSM_ECDSA_SHA384:
		case SKS_CKM_MTK_HSM_ECDSA_SHA512:
			rv = mtk_ECDSA_SHA_step(session, in, out, function, step);
			break;

		default:
			rv = SKS_BAD_PARAM;
			break;
		}

		if (rv == SKS_OK) {
			DMSG("SKSs%" PRIu32 ": init processing %s %s",
			session_handle, sks2str_proc(mecha_type),
			sks2str_function(function));
		} else {
			EMSG(" entry_processing_step FAIL!!  \n");
		}

		goto bail;
	}

	rv = SKS_CKR_MECHANISM_INVALID;
	if (processing_is_tee_symm(mecha_type)) {
		rv = step_symm_operation(session, function, step, in, out);
	}
	if (processing_is_tee_asymm(mecha_type)) {
		rv = step_asymm_operation(session, function, step, in, out);
	}
	if (rv == SKS_OK) {
		session->processing->updated = true;
		IMSG("SKSs%" PRIu32 ": processing %s %s",
		     session_handle, sks2str_proc(mecha_type),
		     sks2str_function(function));
	}

bail:
	switch (step) {
	case SKS_FUNC_STEP_UPDATE:
		if (rv != SKS_OK && rv != SKS_SHORT_BUFFER)
			release_active_processing(session);
		break;
	default:
		/* ONESHOT and FINAL terminates processing on success */
		if (rv != SKS_SHORT_BUFFER)
			release_active_processing(session);
		break;
	}

	return rv;
}

/*
 * entry_verify_oneshot - Generic entry on active processing
 *
 * @ctrl = [session-handle]
 * @in = input data or none
 * @out = output data or none
 * @function - encrypt, decrypt, sign, verify, digest, ...
 * @step - update, oneshot, final
 *
 * The generic part come that all the commands uses the same
 * input/output invocation parameters format (ctrl/in/out).
 */
uint32_t entry_verify_oneshot(uintptr_t tee_session, TEE_Param *ctrl,
				  TEE_Param *in, TEE_Param *in2,
				  enum processing_func function,
				  enum processing_step step)

{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	uint32_t mecha_type = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	assert(function == SKS_FUNCTION_VERIFY);
	if (!ctrl)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_active_session(&session, session_handle, tee_session,
				function);
	if (rv)
		return rv;

	// TODO: check user authen and object activation dates
	rv = check_mechanism_against_processing(session, mecha_type,
						function, step);
	if (rv)
		goto bail;


	mecha_type = session->processing->mecha_type;
	rv = check_mechanism_against_processing(session, mecha_type,
						function, step);
	if (rv)
		goto bail;

	if (mecha_type & SKS_CKM_MTK_HSM_EXT)
	{
		switch (mecha_type)
		{
		case SKS_CKM_MTK_HSM_AES_CMAC:
			rv = mtk_AES_CMAC_step(session, in, in2, function, step);
			break;

		case SKS_CKM_MTK_HSM_SHA256_HMAC:
			rv = mtk_SHA256_HMAC_step(session, in, in2, function, step);
			break;

		case SKS_CKM_MTK_HSM_SHA384_HMAC:
			rv = mtk_SHA384_HMAC_step(session, in, in2, function, step);
			break;

		case SKS_CKM_MTK_HSM_ECDSA:
		case SKS_CKM_MTK_HSM_ECDSA_SM2:
			rv = mtk_ECDSA_step(session, in, in2, function, step);
			break;

		case SKS_CKM_MTK_HSM_ECDSA_SHA1:
		case SKS_CKM_MTK_HSM_ECDSA_SHA224:
		case SKS_CKM_MTK_HSM_ECDSA_SHA256:
		case SKS_CKM_MTK_HSM_ECDSA_SHA384:
		case SKS_CKM_MTK_HSM_ECDSA_SHA512:
			rv = mtk_ECDSA_SHA_step(session, in, in2, function, step);
			break;

		default:
			rv = SKS_BAD_PARAM;
			break;
		}

		goto bail;
	}

	rv = SKS_CKR_MECHANISM_INVALID;
	if (processing_is_tee_symm(mecha_type)) {
		rv = step_symm_operation(session, function, step, in, in2);
	}
	if (processing_is_tee_asymm(mecha_type)) {
		rv = step_asymm_operation(session, function, step, in, in2);
	}

	IMSG("SKSs%" PRIu32 ": verify %s %s: %s", session_handle,
	     sks2str_proc(mecha_type), sks2str_function(function),
	     sks2str_rc(rv));

bail:
	if (rv != SKS_SHORT_BUFFER)
		release_active_processing(session);

	return rv;
}

uint32_t entry_derive_key(uintptr_t tee_session, TEE_Param *ctrl,
			  TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	uint32_t parent_handle = 0;
	struct sks_object *parent_obj;
	struct sks_attrs_head *head = NULL;
	struct sks_object_head *template = NULL;
	size_t template_size = 0;
	uint32_t out_handle = 0;
	uint32_t __maybe_unused mecha_id = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || in || !out)
		return SKS_BAD_PARAM;

	if (out->memref.size < sizeof(uint32_t)) {
		out->memref.size = sizeof(uint32_t);
		return SKS_SHORT_BUFFER;
	}

	if ((uintptr_t)out->memref.buffer & 0x3UL)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, tee_session);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	rv = serialargs_get(&ctrlargs, &parent_handle, sizeof(uint32_t));
	if (rv)
		goto bail;

	parent_obj = sks_handle2object(parent_handle, session);
	if (!parent_obj) {
		rv = SKS_CKR_KEY_HANDLE_INVALID;
		goto bail;
	}

	rv = set_processing_state(session, SKS_FUNCTION_DERIVE,
				  parent_obj, NULL);
	if (rv)
		goto bail;

	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;

	rv = check_mechanism_against_processing(session, proc_params->id,
						SKS_FUNCTION_DERIVE,
						SKS_FUNC_STEP_INIT);
	if (rv)
		goto bail;

	rv = create_attributes_from_template(&head, template, template_size,
					     proc_params->id, parent_obj->attributes,
					     SKS_FUNCTION_DERIVE);
	if (rv)
		goto bail;

	TEE_Free(template);
	template = NULL;

	rv = check_created_attrs(head, NULL);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_processing(proc_params->id, head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, head);
	if (rv)
		goto bail;

	// TODO: check_created_against_parent(session, parent, child);
	// This can handle DERVIE_TEMPLATE attributes from the parent key.

	rv = SKS_CKR_MECHANISM_INVALID;
	if (processing_is_tee_symm(proc_params->id)) {
		rv = init_symm_operation(session, SKS_FUNCTION_DERIVE,
					 proc_params, parent_obj);
		if (rv)
			goto bail;

		rv = do_symm_derivation(session, proc_params,
					parent_obj, &head);
	}
	if (processing_is_tee_asymm(proc_params->id)) {
		rv = init_asymm_operation(session, SKS_FUNCTION_DERIVE,
					  proc_params, parent_obj);
		if (rv)
			goto bail;

		rv = do_asymm_derivation(session, proc_params, &head);
	}
	if (rv)
		goto bail;

#if 0
	/* Exaustive list */
	switch (proc_params->id) {
	case SKS_CKM_ECDH1_DERIVE:	<--------------------------- TODO
	//case SKS_CKM_ECDH1_COFACTOR_DERIVE:
	case SKS_CKM_DH_PKCS_DERIVE:	<--------------------------- TODO
	case SKS_CKM_X9_42_DH_DERIVE:
	case SKS_CKM_X9_42_DH_HYBRID_DERIVE:
	case SKS_CKM_X9_42_MQV_DERIVE:
	case SKS_CKM_AES_GMAC
	case SKS_CKM_AES_ECB_ENCRYPT_DATA	<------------------- TODO
	case SKS_CKM_AES_CBC_ENCRYPT_DATA	<------------------- TODO
	case SKS_CKM_SHA1_KEY_DERIVATION
	case SKS_CKM_SHA224_KEY_DERIVATION
	case SKS_CKM_SHA256_KEY_DERIVATION
	case SKS_CKM_SHA384_KEY_DERIVATION
	case SKS_CKM_SHA512_KEY_DERIVATION
	case SKS_CKM_SHA512_224_KEY_DERIVATION
	case SKS_CKM_SHA512_256_KEY_DERIVATION
	case SKS_CKM_SHA512_T_KEY_DERIVATION
	// Exhaustive list is made of Camelia, Aria, Seed, KIP, GOSTR3410,
	// DES, 3DES, SSL3, TLS12, TLS-KDF, WTLS and concatenate  mechanisms.
	case SKS_CKM_ECMQV_DERIVE:
	}
#endif

	mecha_id = proc_params->id;
	TEE_Free(proc_params);
	proc_params = NULL;

	/*
	 * Object is ready, register it and return a handle.
	 */
	rv = create_object(session, head, &out_handle);
	if (rv)
		goto bail;

	/*
	 * Now out_handle (through the related struct sks_object instance)
	 * owns the serialized buffer that holds the object attributes.
	 * We reset attrs->buffer to NULL as serializer object is no more
	 * the attributes buffer owner.
	 */
	head = NULL;

	TEE_MemMove(out->memref.buffer, &out_handle, sizeof(uint32_t));
	out->memref.size = sizeof(uint32_t);

	IMSG("SKSs%" PRIu32 ": derive key Ox%" PRIx32 ", %s",
	     session_handle, out_handle, sks2str_proc(mecha_id));

bail:
	release_active_processing(session);
	TEE_Free(proc_params);
	TEE_Free(template);
	TEE_Free(head);

	return rv;
}


uint32_t entry_generate_random(uintptr_t teesess, TEE_Param *ctrl,
                               TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = SKS_OK;


	if (!ctrl || !out)
		return SKS_BAD_PARAM;

	rv = mtk_generate_random(out);

	return rv;
}

static uint32_t import_hsm_symmetric_key(struct sks_attrs_head **head, uint8_t *keyblob, int *blobsize)
{
	uint32_t rv = 0;
	int keyid = 0;
	uint8_t *pkeybuffer = NULL;
	int value_size = 0;
	// int size = 0;

	if (!*head || !keyblob)
		return SKS_CKR_TEMPLATE_INCONSISTENT;

	rv = get_attribute_ptr(*head, SKS_CKA_VALUE, &pkeybuffer, &value_size);
	if (rv) {
		EMSG("[%s][%d] rv : %d\n", __FUNCTION__, __LINE__, rv);
		return SKS_CKR_GENERAL_ERROR;
	}

	rv = mtk_import_key(pkeybuffer, value_size, &keyid, keyblob, blobsize, KEY_ALGO_ID_AES);
	if (rv)
		return SKS_CKR_GENERAL_ERROR;

	IMSG("[%s][%d] keyid : 0x%x\n", __FUNCTION__, __LINE__, keyid);

	rv = add_attribute(head, SKS_CKA_HSM_KEY_ID, &keyid, sizeof(keyid));
	if (rv)
		return SKS_CKR_GENERAL_ERROR;

	rv = add_attribute(head, SKS_CKA_HSM_KEY_LEN, &value_size, sizeof(value_size));
	if (rv )
		return SKS_CKR_GENERAL_ERROR;


	return rv;
}


uint32_t entry_import_key(uintptr_t tee_session,
			     TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attrs_head *head = NULL;
	struct sks_object_head *template = NULL;
	size_t template_size = 0;
	uint32_t obj_handle = 0;
	uint32_t key_type;
	uint8_t *blobbuf = NULL;
	uint32_t blobbufsize = 0;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	/*
	 * Collect the arguments of the request
	 */

	if (!ctrl || in || !out)
		return SKS_BAD_PARAM;

	if ((uintptr_t)out->memref.buffer & 0x3UL)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, tee_session);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;

	/*
	 * Prepare a clean initial state for the requested object attributes.
	 * Free temporary template once done.
	 */
	rv = create_attributes_from_template(&head, template, template_size,
					     SKS_UNDEFINED_ID, NULL,
					     SKS_FUNCTION_IMPORT);
	TEE_Free(template);
	template = NULL;
	if (rv)
		goto bail;

	/*
	 * Check target object attributes match target processing
	 * Check target object attributes match token state
	 */
	rv = check_created_attrs_against_processing(SKS_PROCESSING_IMPORT,
						    head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, head);
	if (rv)
		goto bail;


	blobbuf = out->memref.buffer;
	blobbufsize = out->memref.size;
	key_type = get_type(head);
	switch (key_type)
	{
	case SKS_CKK_AES:
	case SKS_CKK_GENERIC_SECRET:
	case SKS_CKK_SHA_1_HMAC:
	case SKS_CKK_SHA224_HMAC:
	case SKS_CKK_SHA256_HMAC:
	case SKS_CKK_SHA384_HMAC:
	case SKS_CKK_SHA512_HMAC:
		rv = import_hsm_symmetric_key(&head, blobbuf, &blobbufsize);
		break;
	default:
		EMSG("[%s][%d] unknown keytype : %d\n", __FUNCTION__, __LINE__, key_type);
		rv = SKS_CKR_KEY_TYPE_INCONSISTENT;
		break;
	}

	if (rv)
		goto bail;
	/*
	 * At this stage the object is almost created: all its attributes are
	 * referenced in @head, including the key value and are assume
	 * reliable. Now need to register it and get a handle for it.
	 */
	rv = create_object(session, head, &obj_handle);
	if (rv)
		goto bail;

	/*
	 * Now obj_handle (through the related struct sks_object instance)
	 * owns the serialised buffer that holds the object attributes.
	 * We reset attrs->buffer to NULL as serializer object is no more
	 * the attributes buffer owner.
	 */
	head = NULL;

	out->memref.size = blobbufsize;

	IMSG("SKSs%" PRIu32 ": import object 0x%" PRIx32,
	     session_handle, obj_handle);
bail:
	TEE_Free(template);
	TEE_Free(head);


	return rv;
}

uint32_t entry_import_key_pair(uintptr_t teesess,
			     TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	struct sks_attrs_head *pub_head = NULL;
	struct sks_attrs_head *priv_head = NULL;
	struct sks_object_head *template = NULL;
	size_t template_size = 0;
	uint32_t pubkey_handle = 0;
	uint32_t privkey_handle = 0;
	uint32_t *hdl_ptr = NULL;
	uint32_t blobbuflength;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));
	if (!ctrl || in || !out)
		return SKS_BAD_PARAM;

	if (out->memref.size < MAX_BLOB_SIZE)
		return SKS_SHORT_BUFFER;

	blobbuflength = out->memref.size;

	// FIXME: cleaner way to test alignment of out buffer
	if ((uintptr_t)out->memref.buffer & 0x3UL)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, teesess);
	if (rv)
		return rv;

	/* Get mechanism parameters */
	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	switch (proc_params->id) {
	case SKS_CKM_MTK_HSM_EC_KEY_PAIR_IMPORT:
		break;
	default:
		rv = SKS_CKR_MECHANISM_INVALID;
		goto bail;
	}

	/* Get and check public key attributes */
	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;
	rv = create_attributes_from_template(&pub_head, template, template_size,
										proc_params->id, NULL,
										SKS_FUNCTION_GENERATE_PAIR);
	if (rv)
		goto bail;

	TEE_Free(template);
	template = NULL;

	rv = serialargs_alloc_get_attributes(&ctrlargs, &template);
	if (rv)
		goto bail;

	template_size = sizeof(*template) + template->attrs_size;
	rv = create_attributes_from_template(&priv_head, template, template_size,
										proc_params->id, NULL,
										SKS_FUNCTION_GENERATE_PAIR);
	if (rv)
		goto bail;

	TEE_Free(template);
	template = NULL;

	/* Generate CKA_ID for keys if not specified by the templates */
	rv = add_missing_attribute_id(&pub_head, &priv_head);
	if (rv)
		goto bail;

	/* Check created object against processing and token state */
	rv = check_created_attrs(pub_head, priv_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_processing(proc_params->id, pub_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_processing(proc_params->id, priv_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, pub_head);
	if (rv)
		goto bail;

	rv = check_created_attrs_against_token(session, priv_head);
	if (rv)
		goto bail;

	rv = import_hsm_ecc_keypair(&pub_head, &priv_head, out->memref.buffer, &blobbuflength);
	if (rv)
		goto bail;

	TEE_Free(proc_params);
	proc_params = NULL;

	/*
	 * Object is ready, register it and return a handle.
	 */
	rv = create_object(session, pub_head, &pubkey_handle);
	if (rv)
		goto bail;

	rv = create_object(session, priv_head, &privkey_handle);
	if (rv)
		goto bail;

	/*
	 * Now obj_handle (through the related struct sks_object instance)
	 * owns the serialized buffer that holds the object attributes.
	 * We reset attrs->buffer to NULL as serializer object is no more
	 * the attributes buffer owner.
	 */
	pub_head = NULL;
	priv_head = NULL;

	out->memref.size = blobbuflength;

	IMSG("SKSs%" PRIu32 ": create key pair 0x%" PRIx32 "/0x%" PRIx32,
		session_handle, privkey_handle, pubkey_handle);

bail:
	TEE_Free(proc_params);
	TEE_Free(template);
	TEE_Free(pub_head);
	TEE_Free(priv_head);

	return rv;
}


uint32_t entry_export_key_pair(uintptr_t teesess,
			     TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	uint32_t mecha_type = 0;

	uint8_t *keyblob;
	uint32_t keybloblength;
	uint8_t *pubkey;
	uint32_t pubkeylength;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || !in || !out)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		goto bail;

	rv = get_ready_session(&session, session_handle, teesess);
	if (rv)
		goto bail;

	keyblob = (uint8_t *)in->memref.buffer;
	keybloblength = in->memref.size;
	pubkey = (uint8_t *)out->memref.buffer;
	pubkeylength = out->memref.size;

	rv = mtk_export_key(keyblob, keybloblength, pubkey, &pubkeylength);
	if (rv)
		goto bail;

	out->memref.size = pubkeylength;
bail:

	return rv;
}


uint32_t entry_utils(uintptr_t teesess,
                     TEE_Param *ctrl, TEE_Param *in, TEE_Param *out)
{
	uint32_t rv = 0;
	struct serialargs ctrlargs;
	uint32_t session_handle = 0;
	struct pkcs11_session *session = NULL;
	struct sks_attribute_head *proc_params = NULL;
	struct sks_object *obj = NULL;

	TEE_MemFill(&ctrlargs, 0, sizeof(ctrlargs));

	if (!ctrl || in || out)
		return SKS_BAD_PARAM;

	serialargs_init(&ctrlargs, ctrl->memref.buffer, ctrl->memref.size);

	rv = serialargs_get(&ctrlargs, &session_handle, sizeof(uint32_t));
	if (rv)
		return rv;

	rv = get_ready_session(&session, session_handle, teesess);
	if (rv)
		return rv;

	rv = serialargs_alloc_get_one_attribute(&ctrlargs, &proc_params);
	if (rv)
		goto bail;

	if (proc_params->id & SKS_CKM_MTK_HSM_EXT)
	{
		switch (proc_params->id)
		{
		case SKS_CKM_MTK_HSM_DUMP_LOG:
			rv = mtk_dump_hsm_log();
			break;

		default:
			rv = SKS_BAD_PARAM;
			break;
		}

		if (rv == SKS_OK) {
			DMSG("SKSs%" PRIu32 ": init processing %s %s",
			     session_handle, sks2str_proc(proc_params->id),
			     sks2str_function(function));
		} else {
			EMSG(" entry_utils FAIL!!  \n");
		}

		goto bail;
	}


bail:
	if (rv && session)
		release_active_processing(session);

	TEE_Free(proc_params);

	return rv;
}
