// SPDX-License-Identifier: BSD-2-Clause
/*
 * Copyright (c) 2017-2018, Linaro Limited
 */

#include <assert.h>
#include <sks_internal_abi.h>
#include <sks_ta.h>
#include <string.h>
#include <tee_api_defines.h>
#include <tee_internal_api.h>
#include <tee_internal_api_extensions.h>
#include <util.h>

#include "attributes.h"
#include "object.h"
#include "pkcs11_token.h"
#include "pkcs11_attributes.h"
#include "processing.h"
#include "serializer.h"
#include "sks_helpers.h"

#include "processing_mtk_key.h"

/*
 * DER encoded EC parameters, dumped from openssl tools using something like
 * openssl ecparam -name secp224r1 -param-enc [explicit]|
 *    openssl asn1parse -noout -out /dev/stdout | od -t x1
 */
static const uint8_t prime192v1_name_der[] = {
	0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x01,
};
static const uint8_t secp224r1_name_der[] = {
	0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x21,
};
static const uint8_t prime256v1_name_der[] = {
	0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07,
};
static const uint8_t secp384r1_name_der[] = {
	0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22,
};
static const uint8_t secp521r1_name_der[] = {
	0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x23,
};
static const uint8_t brainpoolP160r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x01,
};
static const uint8_t brainpoolP160t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x02,
};
static const uint8_t brainpoolP192r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x03,
};
static const uint8_t brainpoolP192t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x04,
};
static const uint8_t brainpoolP224r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x05,
};
static const uint8_t brainpoolP224t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x06,
};
static const uint8_t brainpoolP256r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x07,
};
static const uint8_t brainpoolP256t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x08,
};
static const uint8_t brainpoolP320r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x09,
};
static const uint8_t brainpoolP320t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0A,
};
static const uint8_t brainpoolP384r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0B,
};
static const uint8_t brainpoolP384t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0C,
};
static const uint8_t brainpoolP512r1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0D,
};
static const uint8_t brainpoolP512t1_name_der[] = {
	0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0E,
};

static const uint8_t secp224r1_oid_der[] = {
	0x30, 0x81, 0xDF, 0x02, 0x01, 0x01, 0x30, 0x28,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x1D, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x30, 0x53, 0x04, 0x1C, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
	0x04, 0x1C, 0xB4, 0x05, 0x0A, 0x85, 0x0C, 0x04,
	0xB3, 0xAB, 0xF5, 0x41, 0x32, 0x56, 0x50, 0x44,
	0xB0, 0xB7, 0xD7, 0xBF, 0xD8, 0xBA, 0x27, 0x0B,
	0x39, 0x43, 0x23, 0x55, 0xFF, 0xB4, 0x03, 0x15,
	0x00, 0xBD, 0x71, 0x34, 0x47, 0x99, 0xD5, 0xC7,
	0xFC, 0xDC, 0x45, 0xB5, 0x9F, 0xA3, 0xB9, 0xAB,
	0x8F, 0x6A, 0x94, 0x8B, 0xC5, 0x04, 0x39, 0x04,
	0xB7, 0x0E, 0x0C, 0xBD, 0x6B, 0xB4, 0xBF, 0x7F,
	0x32, 0x13, 0x90, 0xB9, 0x4A, 0x03, 0xC1, 0xD3,
	0x56, 0xC2, 0x11, 0x22, 0x34, 0x32, 0x80, 0xD6,
	0x11, 0x5C, 0x1D, 0x21, 0xBD, 0x37, 0x63, 0x88,
	0xB5, 0xF7, 0x23, 0xFB, 0x4C, 0x22, 0xDF, 0xE6,
	0xCD, 0x43, 0x75, 0xA0, 0x5A, 0x07, 0x47, 0x64,
	0x44, 0xD5, 0x81, 0x99, 0x85, 0x00, 0x7E, 0x34,
	0x02, 0x1D, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x16, 0xA2, 0xE0, 0xB8, 0xF0, 0x3E, 0x13,
	0xDD, 0x29, 0x45, 0x5C, 0x5C, 0x2A, 0x3D, 0x02,
	0x01, 0x01,
};

static const uint8_t secp384r1_oid_der[] = {
	0x30, 0x82, 0x01, 0x57, 0x02, 0x01, 0x01, 0x30,
	0x3C, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x31, 0x00, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF,
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x30, 0x7B, 0x04,
	0x30, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF,
	0xFC, 0x04, 0x30, 0xB3, 0x31, 0x2F, 0xA7, 0xE2,
	0x3E, 0xE7, 0xE4, 0x98, 0x8E, 0x05, 0x6B, 0xE3,
	0xF8, 0x2D, 0x19, 0x18, 0x1D, 0x9C, 0x6E, 0xFE,
	0x81, 0x41, 0x12, 0x03, 0x14, 0x08, 0x8F, 0x50,
	0x13, 0x87, 0x5A, 0xC6, 0x56, 0x39, 0x8D, 0x8A,
	0x2E, 0xD1, 0x9D, 0x2A, 0x85, 0xC8, 0xED, 0xD3,
	0xEC, 0x2A, 0xEF, 0x03, 0x15, 0x00, 0xA3, 0x35,
	0x92, 0x6A, 0xA3, 0x19, 0xA2, 0x7A, 0x1D, 0x00,
	0x89, 0x6A, 0x67, 0x73, 0xA4, 0x82, 0x7A, 0xCD,
	0xAC, 0x73, 0x04, 0x61, 0x04, 0xAA, 0x87, 0xCA,
	0x22, 0xBE, 0x8B, 0x05, 0x37, 0x8E, 0xB1, 0xC7,
	0x1E, 0xF3, 0x20, 0xAD, 0x74, 0x6E, 0x1D, 0x3B,
	0x62, 0x8B, 0xA7, 0x9B, 0x98, 0x59, 0xF7, 0x41,
	0xE0, 0x82, 0x54, 0x2A, 0x38, 0x55, 0x02, 0xF2,
	0x5D, 0xBF, 0x55, 0x29, 0x6C, 0x3A, 0x54, 0x5E,
	0x38, 0x72, 0x76, 0x0A, 0xB7, 0x36, 0x17, 0xDE,
	0x4A, 0x96, 0x26, 0x2C, 0x6F, 0x5D, 0x9E, 0x98,
	0xBF, 0x92, 0x92, 0xDC, 0x29, 0xF8, 0xF4, 0x1D,
	0xBD, 0x28, 0x9A, 0x14, 0x7C, 0xE9, 0xDA, 0x31,
	0x13, 0xB5, 0xF0, 0xB8, 0xC0, 0x0A, 0x60, 0xB1,
	0xCE, 0x1D, 0x7E, 0x81, 0x9D, 0x7A, 0x43, 0x1D,
	0x7C, 0x90, 0xEA, 0x0E, 0x5F, 0x02, 0x31, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xC7, 0x63, 0x4D, 0x81, 0xF4, 0x37, 0x2D, 0xDF,
	0x58, 0x1A, 0x0D, 0xB2, 0x48, 0xB0, 0xA7, 0x7A,
	0xEC, 0xEC, 0x19, 0x6A, 0xCC, 0xC5, 0x29, 0x73,
	0x02, 0x01, 0x01,
};
static const uint8_t secp521r1_oid_der[] = {
	0x30, 0x82, 0x01, 0xC2, 0x02, 0x01, 0x01, 0x30,
	0x4D, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x42, 0x01, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x30, 0x81,
	0x9E, 0x04, 0x42, 0x01, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFC, 0x04, 0x41, 0x51,
	0x95, 0x3E, 0xB9, 0x61, 0x8E, 0x1C, 0x9A, 0x1F,
	0x92, 0x9A, 0x21, 0xA0, 0xB6, 0x85, 0x40, 0xEE,
	0xA2, 0xDA, 0x72, 0x5B, 0x99, 0xB3, 0x15, 0xF3,
	0xB8, 0xB4, 0x89, 0x91, 0x8E, 0xF1, 0x09, 0xE1,
	0x56, 0x19, 0x39, 0x51, 0xEC, 0x7E, 0x93, 0x7B,
	0x16, 0x52, 0xC0, 0xBD, 0x3B, 0xB1, 0xBF, 0x07,
	0x35, 0x73, 0xDF, 0x88, 0x3D, 0x2C, 0x34, 0xF1,
	0xEF, 0x45, 0x1F, 0xD4, 0x6B, 0x50, 0x3F, 0x00,
	0x03, 0x15, 0x00, 0xD0, 0x9E, 0x88, 0x00, 0x29,
	0x1C, 0xB8, 0x53, 0x96, 0xCC, 0x67, 0x17, 0x39,
	0x32, 0x84, 0xAA, 0xA0, 0xDA, 0x64, 0xBA, 0x04,
	0x81, 0x85, 0x04, 0x00, 0xC6, 0x85, 0x8E, 0x06,
	0xB7, 0x04, 0x04, 0xE9, 0xCD, 0x9E, 0x3E, 0xCB,
	0x66, 0x23, 0x95, 0xB4, 0x42, 0x9C, 0x64, 0x81,
	0x39, 0x05, 0x3F, 0xB5, 0x21, 0xF8, 0x28, 0xAF,
	0x60, 0x6B, 0x4D, 0x3D, 0xBA, 0xA1, 0x4B, 0x5E,
	0x77, 0xEF, 0xE7, 0x59, 0x28, 0xFE, 0x1D, 0xC1,
	0x27, 0xA2, 0xFF, 0xA8, 0xDE, 0x33, 0x48, 0xB3,
	0xC1, 0x85, 0x6A, 0x42, 0x9B, 0xF9, 0x7E, 0x7E,
	0x31, 0xC2, 0xE5, 0xBD, 0x66, 0x01, 0x18, 0x39,
	0x29, 0x6A, 0x78, 0x9A, 0x3B, 0xC0, 0x04, 0x5C,
	0x8A, 0x5F, 0xB4, 0x2C, 0x7D, 0x1B, 0xD9, 0x98,
	0xF5, 0x44, 0x49, 0x57, 0x9B, 0x44, 0x68, 0x17,
	0xAF, 0xBD, 0x17, 0x27, 0x3E, 0x66, 0x2C, 0x97,
	0xEE, 0x72, 0x99, 0x5E, 0xF4, 0x26, 0x40, 0xC5,
	0x50, 0xB9, 0x01, 0x3F, 0xAD, 0x07, 0x61, 0x35,
	0x3C, 0x70, 0x86, 0xA2, 0x72, 0xC2, 0x40, 0x88,
	0xBE, 0x94, 0x76, 0x9F, 0xD1, 0x66, 0x50, 0x02,
	0x42, 0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFA, 0x51, 0x86, 0x87, 0x83, 0xBF,
	0x2F, 0x96, 0x6B, 0x7F, 0xCC, 0x01, 0x48, 0xF7,
	0x09, 0xA5, 0xD0, 0x3B, 0xB5, 0xC9, 0xB8, 0x89,
	0x9C, 0x47, 0xAE, 0xBB, 0x6F, 0xB7, 0x1E, 0x91,
	0x38, 0x64, 0x09, 0x02, 0x01, 0x01,
};
static const uint8_t prime192v1_oid_der[] = {
	0x30, 0x81, 0xC7, 0x02, 0x01, 0x01, 0x30, 0x24,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x19, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x30, 0x4B, 0x04, 0x18,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC,
	0x04, 0x18, 0x64, 0x21, 0x05, 0x19, 0xE5, 0x9C,
	0x80, 0xE7, 0x0F, 0xA7, 0xE9, 0xAB, 0x72, 0x24,
	0x30, 0x49, 0xFE, 0xB8, 0xDE, 0xEC, 0xC1, 0x46,
	0xB9, 0xB1, 0x03, 0x15, 0x00, 0x30, 0x45, 0xAE,
	0x6F, 0xC8, 0x42, 0x2F, 0x64, 0xED, 0x57, 0x95,
	0x28, 0xD3, 0x81, 0x20, 0xEA, 0xE1, 0x21, 0x96,
	0xD5, 0x04, 0x31, 0x04, 0x18, 0x8D, 0xA8, 0x0E,
	0xB0, 0x30, 0x90, 0xF6, 0x7C, 0xBF, 0x20, 0xEB,
	0x43, 0xA1, 0x88, 0x00, 0xF4, 0xFF, 0x0A, 0xFD,
	0x82, 0xFF, 0x10, 0x12, 0x07, 0x19, 0x2B, 0x95,
	0xFF, 0xC8, 0xDA, 0x78, 0x63, 0x10, 0x11, 0xED,
	0x6B, 0x24, 0xCD, 0xD5, 0x73, 0xF9, 0x77, 0xA1,
	0x1E, 0x79, 0x48, 0x11, 0x02, 0x19, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0x99, 0xDE, 0xF8, 0x36, 0x14,
	0x6B, 0xC9, 0xB1, 0xB4, 0xD2, 0x28, 0x31, 0x02,
	0x01, 0x01,
};
static const uint8_t __unused prime192v2_oid_der[] = {
	0x30, 0x81, 0xC7, 0x02, 0x01, 0x01, 0x30, 0x24,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x19, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x30, 0x4B, 0x04, 0x18,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC,
	0x04, 0x18, 0xCC, 0x22, 0xD6, 0xDF, 0xB9, 0x5C,
	0x6B, 0x25, 0xE4, 0x9C, 0x0D, 0x63, 0x64, 0xA4,
	0xE5, 0x98, 0x0C, 0x39, 0x3A, 0xA2, 0x16, 0x68,
	0xD9, 0x53, 0x03, 0x15, 0x00, 0x31, 0xA9, 0x2E,
	0xE2, 0x02, 0x9F, 0xD1, 0x0D, 0x90, 0x1B, 0x11,
	0x3E, 0x99, 0x07, 0x10, 0xF0, 0xD2, 0x1A, 0xC6,
	0xB6, 0x04, 0x31, 0x04, 0xEE, 0xA2, 0xBA, 0xE7,
	0xE1, 0x49, 0x78, 0x42, 0xF2, 0xDE, 0x77, 0x69,
	0xCF, 0xE9, 0xC9, 0x89, 0xC0, 0x72, 0xAD, 0x69,
	0x6F, 0x48, 0x03, 0x4A, 0x65, 0x74, 0xD1, 0x1D,
	0x69, 0xB6, 0xEC, 0x7A, 0x67, 0x2B, 0xB8, 0x2A,
	0x08, 0x3D, 0xF2, 0xF2, 0xB0, 0x84, 0x7D, 0xE9,
	0x70, 0xB2, 0xDE, 0x15, 0x02, 0x19, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFE, 0x5F, 0xB1, 0xA7, 0x24, 0xDC,
	0x80, 0x41, 0x86, 0x48, 0xD8, 0xDD, 0x31, 0x02,
	0x01, 0x01,
};
static const uint8_t __unused prime192v3_oid_der[] = {
	0x30, 0x81, 0xC7, 0x02, 0x01, 0x01, 0x30, 0x24,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x19, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x30, 0x4B, 0x04, 0x18,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC,
	0x04, 0x18, 0x22, 0x12, 0x3D, 0xC2, 0x39, 0x5A,
	0x05, 0xCA, 0xA7, 0x42, 0x3D, 0xAE, 0xCC, 0xC9,
	0x47, 0x60, 0xA7, 0xD4, 0x62, 0x25, 0x6B, 0xD5,
	0x69, 0x16, 0x03, 0x15, 0x00, 0xC4, 0x69, 0x68,
	0x44, 0x35, 0xDE, 0xB3, 0x78, 0xC4, 0xB6, 0x5C,
	0xA9, 0x59, 0x1E, 0x2A, 0x57, 0x63, 0x05, 0x9A,
	0x2E, 0x04, 0x31, 0x04, 0x7D, 0x29, 0x77, 0x81,
	0x00, 0xC6, 0x5A, 0x1D, 0xA1, 0x78, 0x37, 0x16,
	0x58, 0x8D, 0xCE, 0x2B, 0x8B, 0x4A, 0xEE, 0x8E,
	0x22, 0x8F, 0x18, 0x96, 0x38, 0xA9, 0x0F, 0x22,
	0x63, 0x73, 0x37, 0x33, 0x4B, 0x49, 0xDC, 0xB6,
	0x6A, 0x6D, 0xC8, 0xF9, 0x97, 0x8A, 0xCA, 0x76,
	0x48, 0xA9, 0x43, 0xB0, 0x02, 0x19, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0x7A, 0x62, 0xD0, 0x31, 0xC8,
	0x3F, 0x42, 0x94, 0xF6, 0x40, 0xEC, 0x13, 0x02,
	0x01, 0x01,
};
static const uint8_t prime256v1_oid_der[] = {
	0x30, 0x81, 0xF7, 0x02, 0x01, 0x01, 0x30, 0x2C,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x21, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0x30, 0x5B, 0x04, 0x20,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC,
	0x04, 0x20, 0x5A, 0xC6, 0x35, 0xD8, 0xAA, 0x3A,
	0x93, 0xE7, 0xB3, 0xEB, 0xBD, 0x55, 0x76, 0x98,
	0x86, 0xBC, 0x65, 0x1D, 0x06, 0xB0, 0xCC, 0x53,
	0xB0, 0xF6, 0x3B, 0xCE, 0x3C, 0x3E, 0x27, 0xD2,
	0x60, 0x4B, 0x03, 0x15, 0x00, 0xC4, 0x9D, 0x36,
	0x08, 0x86, 0xE7, 0x04, 0x93, 0x6A, 0x66, 0x78,
	0xE1, 0x13, 0x9D, 0x26, 0xB7, 0x81, 0x9F, 0x7E,
	0x90, 0x04, 0x41, 0x04, 0x6B, 0x17, 0xD1, 0xF2,
	0xE1, 0x2C, 0x42, 0x47, 0xF8, 0xBC, 0xE6, 0xE5,
	0x63, 0xA4, 0x40, 0xF2, 0x77, 0x03, 0x7D, 0x81,
	0x2D, 0xEB, 0x33, 0xA0, 0xF4, 0xA1, 0x39, 0x45,
	0xD8, 0x98, 0xC2, 0x96, 0x4F, 0xE3, 0x42, 0xE2,
	0xFE, 0x1A, 0x7F, 0x9B, 0x8E, 0xE7, 0xEB, 0x4A,
	0x7C, 0x0F, 0x9E, 0x16, 0x2B, 0xCE, 0x33, 0x57,
	0x6B, 0x31, 0x5E, 0xCE, 0xCB, 0xB6, 0x40, 0x68,
	0x37, 0xBF, 0x51, 0xF5, 0x02, 0x21, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xBC,
	0xE6, 0xFA, 0xAD, 0xA7, 0x17, 0x9E, 0x84, 0xF3,
	0xB9, 0xCA, 0xC2, 0xFC, 0x63, 0x25, 0x51, 0x02,
	0x01, 0x01,
};
static const uint8_t brainpoolP160r1_oid_der[] = {
	0x30, 0x81, 0x98, 0x02, 0x01, 0x01, 0x30, 0x20,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x15, 0x00, 0xE9, 0x5E, 0x4A, 0x5F,
	0x73, 0x70, 0x59, 0xDC, 0x60, 0xDF, 0xC7, 0xAD,
	0x95, 0xB3, 0xD8, 0x13, 0x95, 0x15, 0x62, 0x0F,
	0x30, 0x2C, 0x04, 0x14, 0x34, 0x0E, 0x7B, 0xE2,
	0xA2, 0x80, 0xEB, 0x74, 0xE2, 0xBE, 0x61, 0xBA,
	0xDA, 0x74, 0x5D, 0x97, 0xE8, 0xF7, 0xC3, 0x00,
	0x04, 0x14, 0x1E, 0x58, 0x9A, 0x85, 0x95, 0x42,
	0x34, 0x12, 0x13, 0x4F, 0xAA, 0x2D, 0xBD, 0xEC,
	0x95, 0xC8, 0xD8, 0x67, 0x5E, 0x58, 0x04, 0x29,
	0x04, 0xBE, 0xD5, 0xAF, 0x16, 0xEA, 0x3F, 0x6A,
	0x4F, 0x62, 0x93, 0x8C, 0x46, 0x31, 0xEB, 0x5A,
	0xF7, 0xBD, 0xBC, 0xDB, 0xC3, 0x16, 0x67, 0xCB,
	0x47, 0x7A, 0x1A, 0x8E, 0xC3, 0x38, 0xF9, 0x47,
	0x41, 0x66, 0x9C, 0x97, 0x63, 0x16, 0xDA, 0x63,
	0x21, 0x02, 0x15, 0x00, 0xE9, 0x5E, 0x4A, 0x5F,
	0x73, 0x70, 0x59, 0xDC, 0x60, 0xDF, 0x59, 0x91,
	0xD4, 0x50, 0x29, 0x40, 0x9E, 0x60, 0xFC, 0x09,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP160t1_oid_der[] = {
	0x30, 0x81, 0x98, 0x02, 0x01, 0x01, 0x30, 0x20,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x15, 0x00, 0xE9, 0x5E, 0x4A, 0x5F,
	0x73, 0x70, 0x59, 0xDC, 0x60, 0xDF, 0xC7, 0xAD,
	0x95, 0xB3, 0xD8, 0x13, 0x95, 0x15, 0x62, 0x0F,
	0x30, 0x2C, 0x04, 0x14, 0xE9, 0x5E, 0x4A, 0x5F,
	0x73, 0x70, 0x59, 0xDC, 0x60, 0xDF, 0xC7, 0xAD,
	0x95, 0xB3, 0xD8, 0x13, 0x95, 0x15, 0x62, 0x0C,
	0x04, 0x14, 0x7A, 0x55, 0x6B, 0x6D, 0xAE, 0x53,
	0x5B, 0x7B, 0x51, 0xED, 0x2C, 0x4D, 0x7D, 0xAA,
	0x7A, 0x0B, 0x5C, 0x55, 0xF3, 0x80, 0x04, 0x29,
	0x04, 0xB1, 0x99, 0xB1, 0x3B, 0x9B, 0x34, 0xEF,
	0xC1, 0x39, 0x7E, 0x64, 0xBA, 0xEB, 0x05, 0xAC,
	0xC2, 0x65, 0xFF, 0x23, 0x78, 0xAD, 0xD6, 0x71,
	0x8B, 0x7C, 0x7C, 0x19, 0x61, 0xF0, 0x99, 0x1B,
	0x84, 0x24, 0x43, 0x77, 0x21, 0x52, 0xC9, 0xE0,
	0xAD, 0x02, 0x15, 0x00, 0xE9, 0x5E, 0x4A, 0x5F,
	0x73, 0x70, 0x59, 0xDC, 0x60, 0xDF, 0x59, 0x91,
	0xD4, 0x50, 0x29, 0x40, 0x9E, 0x60, 0xFC, 0x09,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP192r1_oid_der[] = {
	0x30, 0x81, 0xB0, 0x02, 0x01, 0x01, 0x30, 0x24,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x19, 0x00, 0xC3, 0x02, 0xF4, 0x1D,
	0x93, 0x2A, 0x36, 0xCD, 0xA7, 0xA3, 0x46, 0x30,
	0x93, 0xD1, 0x8D, 0xB7, 0x8F, 0xCE, 0x47, 0x6D,
	0xE1, 0xA8, 0x62, 0x97, 0x30, 0x34, 0x04, 0x18,
	0x6A, 0x91, 0x17, 0x40, 0x76, 0xB1, 0xE0, 0xE1,
	0x9C, 0x39, 0xC0, 0x31, 0xFE, 0x86, 0x85, 0xC1,
	0xCA, 0xE0, 0x40, 0xE5, 0xC6, 0x9A, 0x28, 0xEF,
	0x04, 0x18, 0x46, 0x9A, 0x28, 0xEF, 0x7C, 0x28,
	0xCC, 0xA3, 0xDC, 0x72, 0x1D, 0x04, 0x4F, 0x44,
	0x96, 0xBC, 0xCA, 0x7E, 0xF4, 0x14, 0x6F, 0xBF,
	0x25, 0xC9, 0x04, 0x31, 0x04, 0xC0, 0xA0, 0x64,
	0x7E, 0xAA, 0xB6, 0xA4, 0x87, 0x53, 0xB0, 0x33,
	0xC5, 0x6C, 0xB0, 0xF0, 0x90, 0x0A, 0x2F, 0x5C,
	0x48, 0x53, 0x37, 0x5F, 0xD6, 0x14, 0xB6, 0x90,
	0x86, 0x6A, 0xBD, 0x5B, 0xB8, 0x8B, 0x5F, 0x48,
	0x28, 0xC1, 0x49, 0x00, 0x02, 0xE6, 0x77, 0x3F,
	0xA2, 0xFA, 0x29, 0x9B, 0x8F, 0x02, 0x19, 0x00,
	0xC3, 0x02, 0xF4, 0x1D, 0x93, 0x2A, 0x36, 0xCD,
	0xA7, 0xA3, 0x46, 0x2F, 0x9E, 0x9E, 0x91, 0x6B,
	0x5B, 0xE8, 0xF1, 0x02, 0x9A, 0xC4, 0xAC, 0xC1,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP192t1_oid_der[] = {
	0x30, 0x81, 0xB0, 0x02, 0x01, 0x01, 0x30, 0x24,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x19, 0x00, 0xC3, 0x02, 0xF4, 0x1D,
	0x93, 0x2A, 0x36, 0xCD, 0xA7, 0xA3, 0x46, 0x30,
	0x93, 0xD1, 0x8D, 0xB7, 0x8F, 0xCE, 0x47, 0x6D,
	0xE1, 0xA8, 0x62, 0x97, 0x30, 0x34, 0x04, 0x18,
	0xC3, 0x02, 0xF4, 0x1D, 0x93, 0x2A, 0x36, 0xCD,
	0xA7, 0xA3, 0x46, 0x30, 0x93, 0xD1, 0x8D, 0xB7,
	0x8F, 0xCE, 0x47, 0x6D, 0xE1, 0xA8, 0x62, 0x94,
	0x04, 0x18, 0x13, 0xD5, 0x6F, 0xFA, 0xEC, 0x78,
	0x68, 0x1E, 0x68, 0xF9, 0xDE, 0xB4, 0x3B, 0x35,
	0xBE, 0xC2, 0xFB, 0x68, 0x54, 0x2E, 0x27, 0x89,
	0x7B, 0x79, 0x04, 0x31, 0x04, 0x3A, 0xE9, 0xE5,
	0x8C, 0x82, 0xF6, 0x3C, 0x30, 0x28, 0x2E, 0x1F,
	0xE7, 0xBB, 0xF4, 0x3F, 0xA7, 0x2C, 0x44, 0x6A,
	0xF6, 0xF4, 0x61, 0x81, 0x29, 0x09, 0x7E, 0x2C,
	0x56, 0x67, 0xC2, 0x22, 0x3A, 0x90, 0x2A, 0xB5,
	0xCA, 0x44, 0x9D, 0x00, 0x84, 0xB7, 0xE5, 0xB3,
	0xDE, 0x7C, 0xCC, 0x01, 0xC9, 0x02, 0x19, 0x00,
	0xC3, 0x02, 0xF4, 0x1D, 0x93, 0x2A, 0x36, 0xCD,
	0xA7, 0xA3, 0x46, 0x2F, 0x9E, 0x9E, 0x91, 0x6B,
	0x5B, 0xE8, 0xF1, 0x02, 0x9A, 0xC4, 0xAC, 0xC1,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP224r1_oid_der[] = {
	0x30, 0x81, 0xC8, 0x02, 0x01, 0x01, 0x30, 0x28,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x1D, 0x00, 0xD7, 0xC1, 0x34, 0xAA,
	0x26, 0x43, 0x66, 0x86, 0x2A, 0x18, 0x30, 0x25,
	0x75, 0xD1, 0xD7, 0x87, 0xB0, 0x9F, 0x07, 0x57,
	0x97, 0xDA, 0x89, 0xF5, 0x7E, 0xC8, 0xC0, 0xFF,
	0x30, 0x3C, 0x04, 0x1C, 0x68, 0xA5, 0xE6, 0x2C,
	0xA9, 0xCE, 0x6C, 0x1C, 0x29, 0x98, 0x03, 0xA6,
	0xC1, 0x53, 0x0B, 0x51, 0x4E, 0x18, 0x2A, 0xD8,
	0xB0, 0x04, 0x2A, 0x59, 0xCA, 0xD2, 0x9F, 0x43,
	0x04, 0x1C, 0x25, 0x80, 0xF6, 0x3C, 0xCF, 0xE4,
	0x41, 0x38, 0x87, 0x07, 0x13, 0xB1, 0xA9, 0x23,
	0x69, 0xE3, 0x3E, 0x21, 0x35, 0xD2, 0x66, 0xDB,
	0xB3, 0x72, 0x38, 0x6C, 0x40, 0x0B, 0x04, 0x39,
	0x04, 0x0D, 0x90, 0x29, 0xAD, 0x2C, 0x7E, 0x5C,
	0xF4, 0x34, 0x08, 0x23, 0xB2, 0xA8, 0x7D, 0xC6,
	0x8C, 0x9E, 0x4C, 0xE3, 0x17, 0x4C, 0x1E, 0x6E,
	0xFD, 0xEE, 0x12, 0xC0, 0x7D, 0x58, 0xAA, 0x56,
	0xF7, 0x72, 0xC0, 0x72, 0x6F, 0x24, 0xC6, 0xB8,
	0x9E, 0x4E, 0xCD, 0xAC, 0x24, 0x35, 0x4B, 0x9E,
	0x99, 0xCA, 0xA3, 0xF6, 0xD3, 0x76, 0x14, 0x02,
	0xCD, 0x02, 0x1D, 0x00, 0xD7, 0xC1, 0x34, 0xAA,
	0x26, 0x43, 0x66, 0x86, 0x2A, 0x18, 0x30, 0x25,
	0x75, 0xD0, 0xFB, 0x98, 0xD1, 0x16, 0xBC, 0x4B,
	0x6D, 0xDE, 0xBC, 0xA3, 0xA5, 0xA7, 0x93, 0x9F,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP224t1_oid_der[] = {
	0x30, 0x81, 0xC8, 0x02, 0x01, 0x01, 0x30, 0x28,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x1D, 0x00, 0xD7, 0xC1, 0x34, 0xAA,
	0x26, 0x43, 0x66, 0x86, 0x2A, 0x18, 0x30, 0x25,
	0x75, 0xD1, 0xD7, 0x87, 0xB0, 0x9F, 0x07, 0x57,
	0x97, 0xDA, 0x89, 0xF5, 0x7E, 0xC8, 0xC0, 0xFF,
	0x30, 0x3C, 0x04, 0x1C, 0xD7, 0xC1, 0x34, 0xAA,
	0x26, 0x43, 0x66, 0x86, 0x2A, 0x18, 0x30, 0x25,
	0x75, 0xD1, 0xD7, 0x87, 0xB0, 0x9F, 0x07, 0x57,
	0x97, 0xDA, 0x89, 0xF5, 0x7E, 0xC8, 0xC0, 0xFC,
	0x04, 0x1C, 0x4B, 0x33, 0x7D, 0x93, 0x41, 0x04,
	0xCD, 0x7B, 0xEF, 0x27, 0x1B, 0xF6, 0x0C, 0xED,
	0x1E, 0xD2, 0x0D, 0xA1, 0x4C, 0x08, 0xB3, 0xBB,
	0x64, 0xF1, 0x8A, 0x60, 0x88, 0x8D, 0x04, 0x39,
	0x04, 0x6A, 0xB1, 0xE3, 0x44, 0xCE, 0x25, 0xFF,
	0x38, 0x96, 0x42, 0x4E, 0x7F, 0xFE, 0x14, 0x76,
	0x2E, 0xCB, 0x49, 0xF8, 0x92, 0x8A, 0xC0, 0xC7,
	0x60, 0x29, 0xB4, 0xD5, 0x80, 0x03, 0x74, 0xE9,
	0xF5, 0x14, 0x3E, 0x56, 0x8C, 0xD2, 0x3F, 0x3F,
	0x4D, 0x7C, 0x0D, 0x4B, 0x1E, 0x41, 0xC8, 0xCC,
	0x0D, 0x1C, 0x6A, 0xBD, 0x5F, 0x1A, 0x46, 0xDB,
	0x4C, 0x02, 0x1D, 0x00, 0xD7, 0xC1, 0x34, 0xAA,
	0x26, 0x43, 0x66, 0x86, 0x2A, 0x18, 0x30, 0x25,
	0x75, 0xD0, 0xFB, 0x98, 0xD1, 0x16, 0xBC, 0x4B,
	0x6D, 0xDE, 0xBC, 0xA3, 0xA5, 0xA7, 0x93, 0x9F,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP256r1_oid_der[] = {
	0x30, 0x81, 0xE0, 0x02, 0x01, 0x01, 0x30, 0x2C,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x21, 0x00, 0xA9, 0xFB, 0x57, 0xDB,
	0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
	0x9D, 0x83, 0x8D, 0x72, 0x6E, 0x3B, 0xF6, 0x23,
	0xD5, 0x26, 0x20, 0x28, 0x20, 0x13, 0x48, 0x1D,
	0x1F, 0x6E, 0x53, 0x77, 0x30, 0x44, 0x04, 0x20,
	0x7D, 0x5A, 0x09, 0x75, 0xFC, 0x2C, 0x30, 0x57,
	0xEE, 0xF6, 0x75, 0x30, 0x41, 0x7A, 0xFF, 0xE7,
	0xFB, 0x80, 0x55, 0xC1, 0x26, 0xDC, 0x5C, 0x6C,
	0xE9, 0x4A, 0x4B, 0x44, 0xF3, 0x30, 0xB5, 0xD9,
	0x04, 0x20, 0x26, 0xDC, 0x5C, 0x6C, 0xE9, 0x4A,
	0x4B, 0x44, 0xF3, 0x30, 0xB5, 0xD9, 0xBB, 0xD7,
	0x7C, 0xBF, 0x95, 0x84, 0x16, 0x29, 0x5C, 0xF7,
	0xE1, 0xCE, 0x6B, 0xCC, 0xDC, 0x18, 0xFF, 0x8C,
	0x07, 0xB6, 0x04, 0x41, 0x04, 0x8B, 0xD2, 0xAE,
	0xB9, 0xCB, 0x7E, 0x57, 0xCB, 0x2C, 0x4B, 0x48,
	0x2F, 0xFC, 0x81, 0xB7, 0xAF, 0xB9, 0xDE, 0x27,
	0xE1, 0xE3, 0xBD, 0x23, 0xC2, 0x3A, 0x44, 0x53,
	0xBD, 0x9A, 0xCE, 0x32, 0x62, 0x54, 0x7E, 0xF8,
	0x35, 0xC3, 0xDA, 0xC4, 0xFD, 0x97, 0xF8, 0x46,
	0x1A, 0x14, 0x61, 0x1D, 0xC9, 0xC2, 0x77, 0x45,
	0x13, 0x2D, 0xED, 0x8E, 0x54, 0x5C, 0x1D, 0x54,
	0xC7, 0x2F, 0x04, 0x69, 0x97, 0x02, 0x21, 0x00,
	0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC,
	0x3E, 0x66, 0x0A, 0x90, 0x9D, 0x83, 0x8D, 0x71,
	0x8C, 0x39, 0x7A, 0xA3, 0xB5, 0x61, 0xA6, 0xF7,
	0x90, 0x1E, 0x0E, 0x82, 0x97, 0x48, 0x56, 0xA7,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP256t1_oid_der[] = {
	0x30, 0x81, 0xE0, 0x02, 0x01, 0x01, 0x30, 0x2C,
	0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x01,
	0x01, 0x02, 0x21, 0x00, 0xA9, 0xFB, 0x57, 0xDB,
	0xA1, 0xEE, 0xA9, 0xBC, 0x3E, 0x66, 0x0A, 0x90,
	0x9D, 0x83, 0x8D, 0x72, 0x6E, 0x3B, 0xF6, 0x23,
	0xD5, 0x26, 0x20, 0x28, 0x20, 0x13, 0x48, 0x1D,
	0x1F, 0x6E, 0x53, 0x77, 0x30, 0x44, 0x04, 0x20,
	0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC,
	0x3E, 0x66, 0x0A, 0x90, 0x9D, 0x83, 0x8D, 0x72,
	0x6E, 0x3B, 0xF6, 0x23, 0xD5, 0x26, 0x20, 0x28,
	0x20, 0x13, 0x48, 0x1D, 0x1F, 0x6E, 0x53, 0x74,
	0x04, 0x20, 0x66, 0x2C, 0x61, 0xC4, 0x30, 0xD8,
	0x4E, 0xA4, 0xFE, 0x66, 0xA7, 0x73, 0x3D, 0x0B,
	0x76, 0xB7, 0xBF, 0x93, 0xEB, 0xC4, 0xAF, 0x2F,
	0x49, 0x25, 0x6A, 0xE5, 0x81, 0x01, 0xFE, 0xE9,
	0x2B, 0x04, 0x04, 0x41, 0x04, 0xA3, 0xE8, 0xEB,
	0x3C, 0xC1, 0xCF, 0xE7, 0xB7, 0x73, 0x22, 0x13,
	0xB2, 0x3A, 0x65, 0x61, 0x49, 0xAF, 0xA1, 0x42,
	0xC4, 0x7A, 0xAF, 0xBC, 0x2B, 0x79, 0xA1, 0x91,
	0x56, 0x2E, 0x13, 0x05, 0xF4, 0x2D, 0x99, 0x6C,
	0x82, 0x34, 0x39, 0xC5, 0x6D, 0x7F, 0x7B, 0x22,
	0xE1, 0x46, 0x44, 0x41, 0x7E, 0x69, 0xBC, 0xB6,
	0xDE, 0x39, 0xD0, 0x27, 0x00, 0x1D, 0xAB, 0xE8,
	0xF3, 0x5B, 0x25, 0xC9, 0xBE, 0x02, 0x21, 0x00,
	0xA9, 0xFB, 0x57, 0xDB, 0xA1, 0xEE, 0xA9, 0xBC,
	0x3E, 0x66, 0x0A, 0x90, 0x9D, 0x83, 0x8D, 0x71,
	0x8C, 0x39, 0x7A, 0xA3, 0xB5, 0x61, 0xA6, 0xF7,
	0x90, 0x1E, 0x0E, 0x82, 0x97, 0x48, 0x56, 0xA7,
	0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP320r1_oid_der[] = {
	0x30, 0x82, 0x01, 0x10, 0x02, 0x01, 0x01, 0x30,
	0x34, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x29, 0x00, 0xD3, 0x5E, 0x47,
	0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78,
	0x5E, 0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF,
	0xA6, 0xF6, 0xF4, 0x0D, 0xEF, 0x4F, 0x92, 0xB9,
	0xEC, 0x78, 0x93, 0xEC, 0x28, 0xFC, 0xD4, 0x12,
	0xB1, 0xF1, 0xB3, 0x2E, 0x27, 0x30, 0x54, 0x04,
	0x28, 0x3E, 0xE3, 0x0B, 0x56, 0x8F, 0xBA, 0xB0,
	0xF8, 0x83, 0xCC, 0xEB, 0xD4, 0x6D, 0x3F, 0x3B,
	0xB8, 0xA2, 0xA7, 0x35, 0x13, 0xF5, 0xEB, 0x79,
	0xDA, 0x66, 0x19, 0x0E, 0xB0, 0x85, 0xFF, 0xA9,
	0xF4, 0x92, 0xF3, 0x75, 0xA9, 0x7D, 0x86, 0x0E,
	0xB4, 0x04, 0x28, 0x52, 0x08, 0x83, 0x94, 0x9D,
	0xFD, 0xBC, 0x42, 0xD3, 0xAD, 0x19, 0x86, 0x40,
	0x68, 0x8A, 0x6F, 0xE1, 0x3F, 0x41, 0x34, 0x95,
	0x54, 0xB4, 0x9A, 0xCC, 0x31, 0xDC, 0xCD, 0x88,
	0x45, 0x39, 0x81, 0x6F, 0x5E, 0xB4, 0xAC, 0x8F,
	0xB1, 0xF1, 0xA6, 0x04, 0x51, 0x04, 0x43, 0xBD,
	0x7E, 0x9A, 0xFB, 0x53, 0xD8, 0xB8, 0x52, 0x89,
	0xBC, 0xC4, 0x8E, 0xE5, 0xBF, 0xE6, 0xF2, 0x01,
	0x37, 0xD1, 0x0A, 0x08, 0x7E, 0xB6, 0xE7, 0x87,
	0x1E, 0x2A, 0x10, 0xA5, 0x99, 0xC7, 0x10, 0xAF,
	0x8D, 0x0D, 0x39, 0xE2, 0x06, 0x11, 0x14, 0xFD,
	0xD0, 0x55, 0x45, 0xEC, 0x1C, 0xC8, 0xAB, 0x40,
	0x93, 0x24, 0x7F, 0x77, 0x27, 0x5E, 0x07, 0x43,
	0xFF, 0xED, 0x11, 0x71, 0x82, 0xEA, 0xA9, 0xC7,
	0x78, 0x77, 0xAA, 0xAC, 0x6A, 0xC7, 0xD3, 0x52,
	0x45, 0xD1, 0x69, 0x2E, 0x8E, 0xE1, 0x02, 0x29,
	0x00, 0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F,
	0xB7, 0xE1, 0x3C, 0x78, 0x5E, 0xD2, 0x01, 0xE0,
	0x65, 0xF9, 0x8F, 0xCF, 0xA5, 0xB6, 0x8F, 0x12,
	0xA3, 0x2D, 0x48, 0x2E, 0xC7, 0xEE, 0x86, 0x58,
	0xE9, 0x86, 0x91, 0x55, 0x5B, 0x44, 0xC5, 0x93,
	0x11, 0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP320t1_oid_der[] = {
	0x30, 0x82, 0x01, 0x10, 0x02, 0x01, 0x01, 0x30,
	0x34, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x29, 0x00, 0xD3, 0x5E, 0x47,
	0x20, 0x36, 0xBC, 0x4F, 0xB7, 0xE1, 0x3C, 0x78,
	0x5E, 0xD2, 0x01, 0xE0, 0x65, 0xF9, 0x8F, 0xCF,
	0xA6, 0xF6, 0xF4, 0x0D, 0xEF, 0x4F, 0x92, 0xB9,
	0xEC, 0x78, 0x93, 0xEC, 0x28, 0xFC, 0xD4, 0x12,
	0xB1, 0xF1, 0xB3, 0x2E, 0x27, 0x30, 0x54, 0x04,
	0x28, 0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F,
	0xB7, 0xE1, 0x3C, 0x78, 0x5E, 0xD2, 0x01, 0xE0,
	0x65, 0xF9, 0x8F, 0xCF, 0xA6, 0xF6, 0xF4, 0x0D,
	0xEF, 0x4F, 0x92, 0xB9, 0xEC, 0x78, 0x93, 0xEC,
	0x28, 0xFC, 0xD4, 0x12, 0xB1, 0xF1, 0xB3, 0x2E,
	0x24, 0x04, 0x28, 0xA7, 0xF5, 0x61, 0xE0, 0x38,
	0xEB, 0x1E, 0xD5, 0x60, 0xB3, 0xD1, 0x47, 0xDB,
	0x78, 0x20, 0x13, 0x06, 0x4C, 0x19, 0xF2, 0x7E,
	0xD2, 0x7C, 0x67, 0x80, 0xAA, 0xF7, 0x7F, 0xB8,
	0xA5, 0x47, 0xCE, 0xB5, 0xB4, 0xFE, 0xF4, 0x22,
	0x34, 0x03, 0x53, 0x04, 0x51, 0x04, 0x92, 0x5B,
	0xE9, 0xFB, 0x01, 0xAF, 0xC6, 0xFB, 0x4D, 0x3E,
	0x7D, 0x49, 0x90, 0x01, 0x0F, 0x81, 0x34, 0x08,
	0xAB, 0x10, 0x6C, 0x4F, 0x09, 0xCB, 0x7E, 0xE0,
	0x78, 0x68, 0xCC, 0x13, 0x6F, 0xFF, 0x33, 0x57,
	0xF6, 0x24, 0xA2, 0x1B, 0xED, 0x52, 0x63, 0xBA,
	0x3A, 0x7A, 0x27, 0x48, 0x3E, 0xBF, 0x66, 0x71,
	0xDB, 0xEF, 0x7A, 0xBB, 0x30, 0xEB, 0xEE, 0x08,
	0x4E, 0x58, 0xA0, 0xB0, 0x77, 0xAD, 0x42, 0xA5,
	0xA0, 0x98, 0x9D, 0x1E, 0xE7, 0x1B, 0x1B, 0x9B,
	0xC0, 0x45, 0x5F, 0xB0, 0xD2, 0xC3, 0x02, 0x29,
	0x00, 0xD3, 0x5E, 0x47, 0x20, 0x36, 0xBC, 0x4F,
	0xB7, 0xE1, 0x3C, 0x78, 0x5E, 0xD2, 0x01, 0xE0,
	0x65, 0xF9, 0x8F, 0xCF, 0xA5, 0xB6, 0x8F, 0x12,
	0xA3, 0x2D, 0x48, 0x2E, 0xC7, 0xEE, 0x86, 0x58,
	0xE9, 0x86, 0x91, 0x55, 0x5B, 0x44, 0xC5, 0x93,
	0x11, 0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP384r1_oid_der[] = {
	0x30, 0x82, 0x01, 0x40, 0x02, 0x01, 0x01, 0x30,
	0x3C, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x31, 0x00, 0x8C, 0xB9, 0x1E,
	0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F,
	0x7E, 0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71,
	0x09, 0xED, 0x54, 0x56, 0xB4, 0x12, 0xB1, 0xDA,
	0x19, 0x7F, 0xB7, 0x11, 0x23, 0xAC, 0xD3, 0xA7,
	0x29, 0x90, 0x1D, 0x1A, 0x71, 0x87, 0x47, 0x00,
	0x13, 0x31, 0x07, 0xEC, 0x53, 0x30, 0x64, 0x04,
	0x30, 0x7B, 0xC3, 0x82, 0xC6, 0x3D, 0x8C, 0x15,
	0x0C, 0x3C, 0x72, 0x08, 0x0A, 0xCE, 0x05, 0xAF,
	0xA0, 0xC2, 0xBE, 0xA2, 0x8E, 0x4F, 0xB2, 0x27,
	0x87, 0x13, 0x91, 0x65, 0xEF, 0xBA, 0x91, 0xF9,
	0x0F, 0x8A, 0xA5, 0x81, 0x4A, 0x50, 0x3A, 0xD4,
	0xEB, 0x04, 0xA8, 0xC7, 0xDD, 0x22, 0xCE, 0x28,
	0x26, 0x04, 0x30, 0x04, 0xA8, 0xC7, 0xDD, 0x22,
	0xCE, 0x28, 0x26, 0x8B, 0x39, 0xB5, 0x54, 0x16,
	0xF0, 0x44, 0x7C, 0x2F, 0xB7, 0x7D, 0xE1, 0x07,
	0xDC, 0xD2, 0xA6, 0x2E, 0x88, 0x0E, 0xA5, 0x3E,
	0xEB, 0x62, 0xD5, 0x7C, 0xB4, 0x39, 0x02, 0x95,
	0xDB, 0xC9, 0x94, 0x3A, 0xB7, 0x86, 0x96, 0xFA,
	0x50, 0x4C, 0x11, 0x04, 0x61, 0x04, 0x1D, 0x1C,
	0x64, 0xF0, 0x68, 0xCF, 0x45, 0xFF, 0xA2, 0xA6,
	0x3A, 0x81, 0xB7, 0xC1, 0x3F, 0x6B, 0x88, 0x47,
	0xA3, 0xE7, 0x7E, 0xF1, 0x4F, 0xE3, 0xDB, 0x7F,
	0xCA, 0xFE, 0x0C, 0xBD, 0x10, 0xE8, 0xE8, 0x26,
	0xE0, 0x34, 0x36, 0xD6, 0x46, 0xAA, 0xEF, 0x87,
	0xB2, 0xE2, 0x47, 0xD4, 0xAF, 0x1E, 0x8A, 0xBE,
	0x1D, 0x75, 0x20, 0xF9, 0xC2, 0xA4, 0x5C, 0xB1,
	0xEB, 0x8E, 0x95, 0xCF, 0xD5, 0x52, 0x62, 0xB7,
	0x0B, 0x29, 0xFE, 0xEC, 0x58, 0x64, 0xE1, 0x9C,
	0x05, 0x4F, 0xF9, 0x91, 0x29, 0x28, 0x0E, 0x46,
	0x46, 0x21, 0x77, 0x91, 0x81, 0x11, 0x42, 0x82,
	0x03, 0x41, 0x26, 0x3C, 0x53, 0x15, 0x02, 0x31,
	0x00, 0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D,
	0x28, 0x0F, 0x5D, 0x6F, 0x7E, 0x50, 0xE6, 0x41,
	0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56,
	0xB3, 0x1F, 0x16, 0x6E, 0x6C, 0xAC, 0x04, 0x25,
	0xA7, 0xCF, 0x3A, 0xB6, 0xAF, 0x6B, 0x7F, 0xC3,
	0x10, 0x3B, 0x88, 0x32, 0x02, 0xE9, 0x04, 0x65,
	0x65, 0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP384t1_oid_der[] = {
	0x30, 0x82, 0x01, 0x40, 0x02, 0x01, 0x01, 0x30,
	0x3C, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x31, 0x00, 0x8C, 0xB9, 0x1E,
	0x82, 0xA3, 0x38, 0x6D, 0x28, 0x0F, 0x5D, 0x6F,
	0x7E, 0x50, 0xE6, 0x41, 0xDF, 0x15, 0x2F, 0x71,
	0x09, 0xED, 0x54, 0x56, 0xB4, 0x12, 0xB1, 0xDA,
	0x19, 0x7F, 0xB7, 0x11, 0x23, 0xAC, 0xD3, 0xA7,
	0x29, 0x90, 0x1D, 0x1A, 0x71, 0x87, 0x47, 0x00,
	0x13, 0x31, 0x07, 0xEC, 0x53, 0x30, 0x64, 0x04,
	0x30, 0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D,
	0x28, 0x0F, 0x5D, 0x6F, 0x7E, 0x50, 0xE6, 0x41,
	0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56,
	0xB4, 0x12, 0xB1, 0xDA, 0x19, 0x7F, 0xB7, 0x11,
	0x23, 0xAC, 0xD3, 0xA7, 0x29, 0x90, 0x1D, 0x1A,
	0x71, 0x87, 0x47, 0x00, 0x13, 0x31, 0x07, 0xEC,
	0x50, 0x04, 0x30, 0x7F, 0x51, 0x9E, 0xAD, 0xA7,
	0xBD, 0xA8, 0x1B, 0xD8, 0x26, 0xDB, 0xA6, 0x47,
	0x91, 0x0F, 0x8C, 0x4B, 0x93, 0x46, 0xED, 0x8C,
	0xCD, 0xC6, 0x4E, 0x4B, 0x1A, 0xBD, 0x11, 0x75,
	0x6D, 0xCE, 0x1D, 0x20, 0x74, 0xAA, 0x26, 0x3B,
	0x88, 0x80, 0x5C, 0xED, 0x70, 0x35, 0x5A, 0x33,
	0xB4, 0x71, 0xEE, 0x04, 0x61, 0x04, 0x18, 0xDE,
	0x98, 0xB0, 0x2D, 0xB9, 0xA3, 0x06, 0xF2, 0xAF,
	0xCD, 0x72, 0x35, 0xF7, 0x2A, 0x81, 0x9B, 0x80,
	0xAB, 0x12, 0xEB, 0xD6, 0x53, 0x17, 0x24, 0x76,
	0xFE, 0xCD, 0x46, 0x2A, 0xAB, 0xFF, 0xC4, 0xFF,
	0x19, 0x1B, 0x94, 0x6A, 0x5F, 0x54, 0xD8, 0xD0,
	0xAA, 0x2F, 0x41, 0x88, 0x08, 0xCC, 0x25, 0xAB,
	0x05, 0x69, 0x62, 0xD3, 0x06, 0x51, 0xA1, 0x14,
	0xAF, 0xD2, 0x75, 0x5A, 0xD3, 0x36, 0x74, 0x7F,
	0x93, 0x47, 0x5B, 0x7A, 0x1F, 0xCA, 0x3B, 0x88,
	0xF2, 0xB6, 0xA2, 0x08, 0xCC, 0xFE, 0x46, 0x94,
	0x08, 0x58, 0x4D, 0xC2, 0xB2, 0x91, 0x26, 0x75,
	0xBF, 0x5B, 0x9E, 0x58, 0x29, 0x28, 0x02, 0x31,
	0x00, 0x8C, 0xB9, 0x1E, 0x82, 0xA3, 0x38, 0x6D,
	0x28, 0x0F, 0x5D, 0x6F, 0x7E, 0x50, 0xE6, 0x41,
	0xDF, 0x15, 0x2F, 0x71, 0x09, 0xED, 0x54, 0x56,
	0xB3, 0x1F, 0x16, 0x6E, 0x6C, 0xAC, 0x04, 0x25,
	0xA7, 0xCF, 0x3A, 0xB6, 0xAF, 0x6B, 0x7F, 0xC3,
	0x10, 0x3B, 0x88, 0x32, 0x02, 0xE9, 0x04, 0x65,
	0x65, 0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP512r1_oid_der[] = {
	0x30, 0x82, 0x01, 0xA2, 0x02, 0x01, 0x01, 0x30,
	0x4C, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x41, 0x00, 0xAA, 0xDD, 0x9D,
	0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6,
	0xAE, 0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D,
	0xB3, 0xB3, 0xC9, 0xD2, 0x0E, 0xD6, 0x63, 0x9C,
	0xCA, 0x70, 0x33, 0x08, 0x71, 0x7D, 0x4D, 0x9B,
	0x00, 0x9B, 0xC6, 0x68, 0x42, 0xAE, 0xCD, 0xA1,
	0x2A, 0xE6, 0xA3, 0x80, 0xE6, 0x28, 0x81, 0xFF,
	0x2F, 0x2D, 0x82, 0xC6, 0x85, 0x28, 0xAA, 0x60,
	0x56, 0x58, 0x3A, 0x48, 0xF3, 0x30, 0x81, 0x84,
	0x04, 0x40, 0x78, 0x30, 0xA3, 0x31, 0x8B, 0x60,
	0x3B, 0x89, 0xE2, 0x32, 0x71, 0x45, 0xAC, 0x23,
	0x4C, 0xC5, 0x94, 0xCB, 0xDD, 0x8D, 0x3D, 0xF9,
	0x16, 0x10, 0xA8, 0x34, 0x41, 0xCA, 0xEA, 0x98,
	0x63, 0xBC, 0x2D, 0xED, 0x5D, 0x5A, 0xA8, 0x25,
	0x3A, 0xA1, 0x0A, 0x2E, 0xF1, 0xC9, 0x8B, 0x9A,
	0xC8, 0xB5, 0x7F, 0x11, 0x17, 0xA7, 0x2B, 0xF2,
	0xC7, 0xB9, 0xE7, 0xC1, 0xAC, 0x4D, 0x77, 0xFC,
	0x94, 0xCA, 0x04, 0x40, 0x3D, 0xF9, 0x16, 0x10,
	0xA8, 0x34, 0x41, 0xCA, 0xEA, 0x98, 0x63, 0xBC,
	0x2D, 0xED, 0x5D, 0x5A, 0xA8, 0x25, 0x3A, 0xA1,
	0x0A, 0x2E, 0xF1, 0xC9, 0x8B, 0x9A, 0xC8, 0xB5,
	0x7F, 0x11, 0x17, 0xA7, 0x2B, 0xF2, 0xC7, 0xB9,
	0xE7, 0xC1, 0xAC, 0x4D, 0x77, 0xFC, 0x94, 0xCA,
	0xDC, 0x08, 0x3E, 0x67, 0x98, 0x40, 0x50, 0xB7,
	0x5E, 0xBA, 0xE5, 0xDD, 0x28, 0x09, 0xBD, 0x63,
	0x80, 0x16, 0xF7, 0x23, 0x04, 0x81, 0x81, 0x04,
	0x81, 0xAE, 0xE4, 0xBD, 0xD8, 0x2E, 0xD9, 0x64,
	0x5A, 0x21, 0x32, 0x2E, 0x9C, 0x4C, 0x6A, 0x93,
	0x85, 0xED, 0x9F, 0x70, 0xB5, 0xD9, 0x16, 0xC1,
	0xB4, 0x3B, 0x62, 0xEE, 0xF4, 0xD0, 0x09, 0x8E,
	0xFF, 0x3B, 0x1F, 0x78, 0xE2, 0xD0, 0xD4, 0x8D,
	0x50, 0xD1, 0x68, 0x7B, 0x93, 0xB9, 0x7D, 0x5F,
	0x7C, 0x6D, 0x50, 0x47, 0x40, 0x6A, 0x5E, 0x68,
	0x8B, 0x35, 0x22, 0x09, 0xBC, 0xB9, 0xF8, 0x22,
	0x7D, 0xDE, 0x38, 0x5D, 0x56, 0x63, 0x32, 0xEC,
	0xC0, 0xEA, 0xBF, 0xA9, 0xCF, 0x78, 0x22, 0xFD,
	0xF2, 0x09, 0xF7, 0x00, 0x24, 0xA5, 0x7B, 0x1A,
	0xA0, 0x00, 0xC5, 0x5B, 0x88, 0x1F, 0x81, 0x11,
	0xB2, 0xDC, 0xDE, 0x49, 0x4A, 0x5F, 0x48, 0x5E,
	0x5B, 0xCA, 0x4B, 0xD8, 0x8A, 0x27, 0x63, 0xAE,
	0xD1, 0xCA, 0x2B, 0x2F, 0xA8, 0xF0, 0x54, 0x06,
	0x78, 0xCD, 0x1E, 0x0F, 0x3A, 0xD8, 0x08, 0x92,
	0x02, 0x41, 0x00, 0xAA, 0xDD, 0x9D, 0xB8, 0xDB,
	0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE, 0x33,
	0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3,
	0xC9, 0xD2, 0x0E, 0xD6, 0x63, 0x9C, 0xCA, 0x70,
	0x33, 0x08, 0x70, 0x55, 0x3E, 0x5C, 0x41, 0x4C,
	0xA9, 0x26, 0x19, 0x41, 0x86, 0x61, 0x19, 0x7F,
	0xAC, 0x10, 0x47, 0x1D, 0xB1, 0xD3, 0x81, 0x08,
	0x5D, 0xDA, 0xDD, 0xB5, 0x87, 0x96, 0x82, 0x9C,
	0xA9, 0x00, 0x69, 0x02, 0x01, 0x01,
};
static const uint8_t brainpoolP512t1_oid_der[] = {
	0x30, 0x82, 0x01, 0xA2, 0x02, 0x01, 0x01, 0x30,
	0x4C, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
	0x01, 0x01, 0x02, 0x41, 0x00, 0xAA, 0xDD, 0x9D,
	0xB8, 0xDB, 0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6,
	0xAE, 0x33, 0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D,
	0xB3, 0xB3, 0xC9, 0xD2, 0x0E, 0xD6, 0x63, 0x9C,
	0xCA, 0x70, 0x33, 0x08, 0x71, 0x7D, 0x4D, 0x9B,
	0x00, 0x9B, 0xC6, 0x68, 0x42, 0xAE, 0xCD, 0xA1,
	0x2A, 0xE6, 0xA3, 0x80, 0xE6, 0x28, 0x81, 0xFF,
	0x2F, 0x2D, 0x82, 0xC6, 0x85, 0x28, 0xAA, 0x60,
	0x56, 0x58, 0x3A, 0x48, 0xF3, 0x30, 0x81, 0x84,
	0x04, 0x40, 0xAA, 0xDD, 0x9D, 0xB8, 0xDB, 0xE9,
	0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE, 0x33, 0xC9,
	0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3, 0xC9,
	0xD2, 0x0E, 0xD6, 0x63, 0x9C, 0xCA, 0x70, 0x33,
	0x08, 0x71, 0x7D, 0x4D, 0x9B, 0x00, 0x9B, 0xC6,
	0x68, 0x42, 0xAE, 0xCD, 0xA1, 0x2A, 0xE6, 0xA3,
	0x80, 0xE6, 0x28, 0x81, 0xFF, 0x2F, 0x2D, 0x82,
	0xC6, 0x85, 0x28, 0xAA, 0x60, 0x56, 0x58, 0x3A,
	0x48, 0xF0, 0x04, 0x40, 0x7C, 0xBB, 0xBC, 0xF9,
	0x44, 0x1C, 0xFA, 0xB7, 0x6E, 0x18, 0x90, 0xE4,
	0x68, 0x84, 0xEA, 0xE3, 0x21, 0xF7, 0x0C, 0x0B,
	0xCB, 0x49, 0x81, 0x52, 0x78, 0x97, 0x50, 0x4B,
	0xEC, 0x3E, 0x36, 0xA6, 0x2B, 0xCD, 0xFA, 0x23,
	0x04, 0x97, 0x65, 0x40, 0xF6, 0x45, 0x00, 0x85,
	0xF2, 0xDA, 0xE1, 0x45, 0xC2, 0x25, 0x53, 0xB4,
	0x65, 0x76, 0x36, 0x89, 0x18, 0x0E, 0xA2, 0x57,
	0x18, 0x67, 0x42, 0x3E, 0x04, 0x81, 0x81, 0x04,
	0x64, 0x0E, 0xCE, 0x5C, 0x12, 0x78, 0x87, 0x17,
	0xB9, 0xC1, 0xBA, 0x06, 0xCB, 0xC2, 0xA6, 0xFE,
	0xBA, 0x85, 0x84, 0x24, 0x58, 0xC5, 0x6D, 0xDE,
	0x9D, 0xB1, 0x75, 0x8D, 0x39, 0xC0, 0x31, 0x3D,
	0x82, 0xBA, 0x51, 0x73, 0x5C, 0xDB, 0x3E, 0xA4,
	0x99, 0xAA, 0x77, 0xA7, 0xD6, 0x94, 0x3A, 0x64,
	0xF7, 0xA3, 0xF2, 0x5F, 0xE2, 0x6F, 0x06, 0xB5,
	0x1B, 0xAA, 0x26, 0x96, 0xFA, 0x90, 0x35, 0xDA,
	0x5B, 0x53, 0x4B, 0xD5, 0x95, 0xF5, 0xAF, 0x0F,
	0xA2, 0xC8, 0x92, 0x37, 0x6C, 0x84, 0xAC, 0xE1,
	0xBB, 0x4E, 0x30, 0x19, 0xB7, 0x16, 0x34, 0xC0,
	0x11, 0x31, 0x15, 0x9C, 0xAE, 0x03, 0xCE, 0xE9,
	0xD9, 0x93, 0x21, 0x84, 0xBE, 0xEF, 0x21, 0x6B,
	0xD7, 0x1D, 0xF2, 0xDA, 0xDF, 0x86, 0xA6, 0x27,
	0x30, 0x6E, 0xCF, 0xF9, 0x6D, 0xBB, 0x8B, 0xAC,
	0xE1, 0x98, 0xB6, 0x1E, 0x00, 0xF8, 0xB3, 0x32,
	0x02, 0x41, 0x00, 0xAA, 0xDD, 0x9D, 0xB8, 0xDB,
	0xE9, 0xC4, 0x8B, 0x3F, 0xD4, 0xE6, 0xAE, 0x33,
	0xC9, 0xFC, 0x07, 0xCB, 0x30, 0x8D, 0xB3, 0xB3,
	0xC9, 0xD2, 0x0E, 0xD6, 0x63, 0x9C, 0xCA, 0x70,
	0x33, 0x08, 0x70, 0x55, 0x3E, 0x5C, 0x41, 0x4C,
	0xA9, 0x26, 0x19, 0x41, 0x86, 0x61, 0x19, 0x7F,
	0xAC, 0x10, 0x47, 0x1D, 0xB1, 0xD3, 0x81, 0x08,
	0x5D, 0xDA, 0xDD, 0xB5, 0x87, 0x96, 0x82, 0x9C,
	0xA9, 0x00, 0x69, 0x02, 0x01, 0x01,
};

struct supported_ecc_curve {
	const uint8_t *oid_der;
	size_t oid_size;
	const uint8_t *name_der;
	size_t name_size;
	size_t key_size;
	uint32_t tee_id;
	const char *label;
	size_t label_size;
};

#define ECC_CURVE(_tee_id, _key_size, _label)	\
	{							\
		.tee_id = _tee_id,				\
		.key_size = _key_size,				\
		.oid_der = _label ## _oid_der,			\
		.oid_size = sizeof(_label ## _oid_der),		\
		.name_der = _label ## _name_der,			\
		.name_size = sizeof(_label ## _name_der),	\
		.label = #_label,					\
		.label_size = sizeof(#_label) - 1,		\
	}

/* These are introduced with GPD TEE Core Internal API v1.2 */
#ifndef TEE_ECC_CURVE_BSI_P160r1
#define TEE_ECC_CURVE_BSI_P160r1	0x00000101 /* 160 bits */
#define TEE_ECC_CURVE_BSI_P192r1	0x00000102 /* 192 bits */
#define TEE_ECC_CURVE_BSI_P224r1	0x00000103 /* 224 bits */
#define TEE_ECC_CURVE_BSI_P256r1	0x00000104 /* 256 bits */
#define TEE_ECC_CURVE_BSI_P320r1	0x00000105 /* 320 bits */
#define TEE_ECC_CURVE_BSI_P384r1	0x00000106 /* 384 bits */
#define TEE_ECC_CURVE_BSI_P512r1	0x00000107 /* 512 bits */
#define TEE_ECC_CURVE_BSI_P160t1	0x00000201 /* 160 bits */
#define TEE_ECC_CURVE_BSI_P192t1	0x00000202 /* 192 bits */
#define TEE_ECC_CURVE_BSI_P224t1	0x00000203 /* 224 bits */
#define TEE_ECC_CURVE_BSI_P256t1	0x00000204 /* 256 bits */
#define TEE_ECC_CURVE_BSI_P320t1	0x00000205 /* 320 bits */
#define TEE_ECC_CURVE_BSI_P384t1	0x00000206 /* 384 bits */
#define TEE_ECC_CURVE_BSI_P512t1	0x00000207 /* 512 bits */
#define TEE_ECC_CURVE_25519		0x00000300 /* 256 bits */
#define TEE_ECC_CURVE_SM2		0x00000400 /* 256 bits */
#endif

static const struct supported_ecc_curve ec_curve_param[] = {
	ECC_CURVE(TEE_ECC_CURVE_NIST_P192, 192, prime192v1),
	ECC_CURVE(TEE_ECC_CURVE_NIST_P224, 224, secp224r1),
	ECC_CURVE(TEE_ECC_CURVE_NIST_P256, 256, prime256v1),
	ECC_CURVE(TEE_ECC_CURVE_NIST_P384, 384, secp384r1),
	ECC_CURVE(TEE_ECC_CURVE_NIST_P521, 521, secp521r1),
	/* Note: this will be for GPD TEE Core Internal API v1.2 */
	ECC_CURVE(TEE_ECC_CURVE_BSI_P160r1, 160, brainpoolP160r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P160t1, 160, brainpoolP160t1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P192r1, 192, brainpoolP192r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P192t1, 192, brainpoolP192t1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P224r1, 224, brainpoolP224r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P224t1, 224, brainpoolP224t1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P256r1, 256, brainpoolP256r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P256t1, 256, brainpoolP256t1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P320r1, 320, brainpoolP320r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P320t1, 320, brainpoolP320t1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P384r1, 384, brainpoolP384r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P384t1, 384, brainpoolP384t1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P512r1, 512, brainpoolP512r1),
	ECC_CURVE(TEE_ECC_CURVE_BSI_P512t1, 512, brainpoolP512t1),
};

static const struct supported_ecc_curve *get_curve(void *attr, size_t size)
{
	size_t idx = 0;

	/* Weak: not a real DER parser: try by params then by named curve */
	for (idx = 0; idx < ARRAY_SIZE(ec_curve_param); idx++) {
		const struct supported_ecc_curve *curve = &ec_curve_param[idx];

		if (size == curve->oid_size &&
		    !TEE_MemCompare(attr, curve->oid_der, curve->oid_size))
			return curve;

		if (size == curve->name_size &&
		    !TEE_MemCompare(attr, curve->name_der, curve->name_size))
			return curve;
	}

	return NULL;
}

size_t ec_params2tee_keysize(void *ec_params, size_t size)
{
	const struct supported_ecc_curve *curve = get_curve(ec_params, size);

	if (!curve)
		return 0;

	return curve->key_size;
}

/*
 * This function intentionally panics if the curve is not found.
 * Use ec_params2tee_keysize() to check the curve is supported by
 * the internal core API.
 */
uint32_t ec_params2tee_curve(void *ec_params, size_t size)
{
	const struct supported_ecc_curve *curve = get_curve(ec_params, size);

	assert(curve);

	return curve->tee_id;
}

uint32_t load_tee_ec_key_attrs(TEE_Attribute **tee_attrs, size_t *tee_count,
				struct sks_object *obj)
{
	TEE_Attribute *attrs = NULL;
	size_t count = 0;
	uint32_t rv = SKS_ERROR;

	assert(get_type(obj->attributes) == SKS_CKK_EC);

	switch (get_class(obj->attributes)) {
	case SKS_CKO_PUBLIC_KEY:
		attrs = TEE_Malloc(3 * sizeof(TEE_Attribute),
				   TEE_USER_MEM_HINT_NO_FILL_ZERO);
		if (!attrs)
			return SKS_MEMORY;

		if (sks2tee_load_attr(&attrs[count], TEE_ATTR_ECC_CURVE,
					obj, SKS_CKA_EC_PARAMS))
			count++;

		if (sks2tee_load_attr(&attrs[count],
					TEE_ATTR_ECC_PUBLIC_VALUE_X,
					obj, SKS_CKA_EC_POINT))
			count++;

		if (sks2tee_load_attr(&attrs[count],
					TEE_ATTR_ECC_PUBLIC_VALUE_Y,
					obj, SKS_CKA_EC_POINT))
			count++;

		if (count == 3)
			rv = SKS_OK;

		break;

	case SKS_CKO_PRIVATE_KEY:
		attrs = TEE_Malloc(4 * sizeof(TEE_Attribute),
				   TEE_USER_MEM_HINT_NO_FILL_ZERO);
		if (!attrs)
			return SKS_MEMORY;

		if (sks2tee_load_attr(&attrs[count], TEE_ATTR_ECC_CURVE,
					obj, SKS_CKA_EC_PARAMS))
			count++;


		if (sks2tee_load_attr(&attrs[count],
					TEE_ATTR_ECC_PRIVATE_VALUE,
					obj, SKS_CKA_VALUE))
			count++;

		if (sks2tee_load_attr(&attrs[count],
					TEE_ATTR_ECC_PUBLIC_VALUE_X,
					obj, SKS_CKA_EC_POINT))
			count++;

		if (sks2tee_load_attr(&attrs[count],
					TEE_ATTR_ECC_PUBLIC_VALUE_Y,
					obj, SKS_CKA_EC_POINT))
			count++;

		if (count == 4)
			rv = SKS_OK;

		break;

	default:
		assert(0);
		break;
	}

	if (rv == SKS_OK) {
		*tee_attrs = attrs;
		*tee_count = count;
	}

	return rv;
}

uint32_t sks2tee_algo_ecdh(uint32_t *tee_id,
			   struct sks_attribute_head *proc_params,
			   struct sks_object *obj)
{
	struct serialargs args;
	uint32_t rv = 0;
	uint32_t kdf = 0;

	TEE_MemFill(&args, 0, sizeof(args));

	serialargs_init(&args, proc_params->data, proc_params->size);

	rv = serialargs_get(&args, &kdf, sizeof(uint32_t));
	if (rv)
		return rv;

	if (kdf != SKS_CKD_NULL) {
		EMSG("Currently no support for hashed shared data");
		return SKS_CKR_MECHANISM_PARAM_INVALID;
	}

	switch (get_object_key_bit_size(obj)) {
	case 192:
		*tee_id = TEE_ALG_ECDH_P192;
		break;
	case 224:
		*tee_id = TEE_ALG_ECDH_P224;
		break;
	case 256:
		*tee_id = TEE_ALG_ECDH_P256;
		break;
	case 384:
		*tee_id = TEE_ALG_ECDH_P384;
		break;
	case 521:
		*tee_id = TEE_ALG_ECDH_P521;
		break;
	default:
		TEE_Panic(0);
		break;
	}

	return SKS_OK;
}

uint32_t sks2tee_ecdh_param_pub(struct sks_attribute_head *proc_params,
			        void **pub_data, size_t *pub_size)
{
	struct serialargs args;
	uint32_t rv = 0;
	uint32_t temp = 0;

	TEE_MemFill(&args, 0, sizeof(args));

	serialargs_init(&args, proc_params->data, proc_params->size);

	/* Skip KDF */
	rv = serialargs_get(&args, &temp, sizeof(uint32_t));
	if (rv)
		return rv;

	/* Shared data size, shall be 0 */
	rv = serialargs_get(&args, &temp, sizeof(uint32_t));
	if (rv || temp)
		return rv;

	/* Public data size and content */
	rv = serialargs_get(&args, &temp, sizeof(uint32_t));
	if (rv || !temp)
		return rv;

	*pub_size = temp;

	return serialargs_get_ptr(&args, pub_data, temp);
}

uint32_t sks2tee_algo_ecdsa(uint32_t *tee_id,
			   struct sks_attribute_head *proc_params,
			   struct sks_object *obj)
{
	switch (proc_params->id) {
	case SKS_CKM_ECDSA:
	case SKS_CKM_ECDSA_SHA1:
	case SKS_CKM_ECDSA_SHA224:
	case SKS_CKM_ECDSA_SHA256:
	case SKS_CKM_ECDSA_SHA384:
	case SKS_CKM_ECDSA_SHA512:
		break;
	default:
		return SKS_ERROR;
	}

	switch (get_object_key_bit_size(obj)) {
	case 192:
		*tee_id = TEE_ALG_ECDSA_P192;
		break;
	case 224:
		*tee_id = TEE_ALG_ECDSA_P224;
		break;
	case 256:
		*tee_id = TEE_ALG_ECDSA_P256;
		break;
	case 384:
		*tee_id = TEE_ALG_ECDSA_P384;
		break;
	case 521:
		*tee_id = TEE_ALG_ECDSA_P521;
		break;
	default:
		TEE_Panic(0);
		break;
	}

	return SKS_OK;
}

static uint32_t tee2sks_ec_attributes(struct sks_attrs_head **pub_head,
				 struct sks_attrs_head **priv_head,
				 TEE_ObjectHandle tee_obj, uint32_t tee_size)
{
	void *x_ptr = NULL;
	void *y_ptr = NULL;
	uint8_t *ecpoint = NULL;
	size_t x_size = 0;
	size_t y_size = 0;
	size_t psize = 0;
	size_t qsize = 0;
	size_t dersize = 0;
	size_t poffset = 0;
	uint32_t rv = 0;

	rv = tee2sks_add_attribute(priv_head, SKS_CKA_VALUE,
				   tee_obj, TEE_ATTR_ECC_PRIVATE_VALUE);
	if (rv)
		goto bail;

	rv = alloc_get_tee_attribute_data(tee_obj, TEE_ATTR_ECC_PUBLIC_VALUE_X,
					&x_ptr, &x_size);
	if (rv)
		goto bail;

	rv = alloc_get_tee_attribute_data(tee_obj, TEE_ATTR_ECC_PUBLIC_VALUE_Y,
					&y_ptr, &y_size);
	if (rv)
		goto x_cleanup;

	psize = (tee_size + 7) / 8;
	if (x_size > psize || y_size > psize) {
		rv = SKS_BAD_PARAM;
		goto p_cleanup;
	}

	qsize = 1 + 2 * psize;
	/* TODO: Support DER long definitive form, needed for 64 key size */
	if (qsize < 0x80) {
		dersize = qsize + 2;
	} else {
		EMSG("DER long definitive form not yet supported");
		rv = SKS_CKR_MECHANISM_INVALID;
		goto p_cleanup;
	}

	ecpoint = TEE_Malloc(dersize, TEE_MALLOC_FILL_ZERO);
	if (!ecpoint) {
		rv = SKS_MEMORY;
		goto p_cleanup;
	}

	/* DER encoding */
	ecpoint[0] = 0x04;
	ecpoint[1] = qsize & 0x7f;
	/* Only UNCOMPRESSED ECPOINT is currently supported */
	ecpoint[2] = 0x04;

	poffset = 0;
	if (x_size < psize)
		poffset = psize - x_size;
	TEE_MemMove(ecpoint + 3 + poffset, x_ptr, x_size);

	poffset = 0;
	if (y_size < psize)
		poffset = psize - y_size;
	TEE_MemMove(ecpoint + 3 + psize + poffset, y_ptr, y_size);

	/*
	 * Add EC_POINT on both private and public key objects as
	 * TEE_PopulateTransientObject requires public x/y values
	 * for TEE_TYPE_ECDSA_KEYPAIR.
	 */
	rv = add_attribute(priv_head, SKS_CKA_EC_POINT, ecpoint, dersize);
	if (rv)
		goto priv_cleanup;

	rv = add_attribute(pub_head, SKS_CKA_EC_POINT, ecpoint, dersize);

priv_cleanup:
	TEE_Free(ecpoint);
p_cleanup:
	TEE_Free(y_ptr);
x_cleanup:
	TEE_Free(x_ptr);
bail:
	return rv;
}

uint32_t generate_ec_keys(struct sks_attribute_head *proc_params,
			  struct sks_attrs_head **pub_head,
			  struct sks_attrs_head **priv_head)
{
	uint32_t rv = 0;
	void *a_ptr = NULL;
	uint32_t a_size = 0;
	uint32_t tee_size = 0;
	uint32_t tee_curve = 0;
	TEE_ObjectHandle tee_obj = TEE_HANDLE_NULL;
	TEE_Attribute tee_key_attr[1];
	TEE_Result res = TEE_ERROR_GENERIC;

	TEE_MemFill(tee_key_attr, 0, sizeof(tee_key_attr));

	if (!proc_params || !*pub_head || !*priv_head)
		return SKS_CKR_TEMPLATE_INCONSISTENT;

	if (!get_attribute_ptr(*pub_head, SKS_CKA_EC_POINT, NULL, NULL) ||
	    !get_attribute_ptr(*priv_head, SKS_CKA_VALUE, NULL, NULL) ||
	    !get_attribute_ptr(*priv_head, SKS_CKA_EC_POINT, NULL, NULL)) {
		EMSG("Unexpected attribute(s) found");
		trace_attributes("public-key", *pub_head);
		trace_attributes("privat-key", *priv_head);
		return SKS_CKR_TEMPLATE_INCONSISTENT;
	}

	if (get_attribute_ptr(*pub_head, SKS_CKA_EC_PARAMS, &a_ptr, &a_size)) {
		EMSG("Not EC_PARAMS attribute found");
		return SKS_CKR_ATTRIBUTE_TYPE_INVALID;
	}

	tee_size = ec_params2tee_keysize(a_ptr, a_size);
	if (!tee_size)
		return SKS_CKR_ATTRIBUTE_TYPE_INVALID;

	tee_curve = ec_params2tee_curve(a_ptr, a_size);

	TEE_InitValueAttribute(&tee_key_attr[0], TEE_ATTR_ECC_CURVE,
				tee_curve, 0);

	/* Create an ECDSA TEE key: will match PKCS11 ECDSA and ECDH */
	res = TEE_AllocateTransientObject(TEE_TYPE_ECDSA_KEYPAIR,
					  521/*tee_size*/, &tee_obj);
	if (res) {
		EMSG("TEE service failed, %" PRIx32, res);
		return tee2sks_error(res);
	}

	res = TEE_RestrictObjectUsage1(tee_obj, TEE_USAGE_EXTRACTABLE);
	if (res) {
		rv = tee2sks_error(res);
		goto bail;
	}

	res = TEE_GenerateKey(tee_obj, tee_size, &tee_key_attr[0], 1);
	if (res) {
		rv = tee2sks_error(res);
		goto bail;
	}

	/*
	 * Private key needs the same EC_PARAMS as used by the public key.
	 * Since it is also common for userspace to provide an empty private
	 * EC_PARAMS, make sure to remove before adding the correct one.
	 */
	remove_attribute(priv_head, SKS_CKA_EC_PARAMS);
	rv = add_attribute(priv_head, SKS_CKA_EC_PARAMS, a_ptr, a_size);
	if (rv)
		goto bail;

	rv = tee2sks_ec_attributes(pub_head, priv_head, tee_obj, tee_size);

bail:
	if (tee_obj != TEE_HANDLE_NULL)
		TEE_FreeTransientObject(tee_obj);

	return rv;
}

static int sks2uecc_curve_id(int curve_id)
{
	int uECC_curve_id = -1;
	switch (curve_id)
	{
	case TEE_ECC_CURVE_NIST_P192:
		uECC_curve_id = UECC_CURVE_ID_SECP192R1;
		break;
	case TEE_ECC_CURVE_NIST_P224:
		uECC_curve_id = UECC_CURVE_ID_SECP224R1;
		break;
	case TEE_ECC_CURVE_NIST_P256:
		uECC_curve_id = UECC_CURVE_ID_SECP256R1;
		break;
	case TEE_ECC_CURVE_NIST_P384:
		uECC_curve_id = UECC_CURVE_ID_SECP384R1;
		break;
	case TEE_ECC_CURVE_NIST_P521:
	default:
		EMSG("[%s][%d] not support curve id!", __FUNCTION__, __LINE__);
		uECC_curve_id = UECC_CURVE_ID_UNSUPPORT;
		break;
	}
	return uECC_curve_id;
}

uint32_t generate_hsm_ec_keys(struct sks_attribute_head *proc_params,
			  struct sks_attrs_head **pub_head,
			  struct sks_attrs_head **priv_head)
{
	uint32_t rv = 0;
	void *a_ptr = NULL;
	uint32_t a_size = 0;
	uint32_t tee_size = 0;
	uint32_t tee_curve = 0;
	TEE_ObjectHandle tee_obj = TEE_HANDLE_NULL;
	TEE_Attribute tee_key_attr[1];
	TEE_Result res = TEE_ERROR_GENERIC;
	int keyid = 0;
	int key_size = 0;
	int uECC_curve_id = -1;

	TEE_MemFill(tee_key_attr, 0, sizeof(tee_key_attr));

	if (!proc_params || !*pub_head || !*priv_head)
		return SKS_CKR_TEMPLATE_INCONSISTENT;

	if (!get_attribute_ptr(*pub_head, SKS_CKA_EC_POINT, NULL, NULL) ||
	    !get_attribute_ptr(*priv_head, SKS_CKA_VALUE, NULL, NULL) ||
	    !get_attribute_ptr(*priv_head, SKS_CKA_EC_POINT, NULL, NULL)) {
		EMSG("Unexpected attribute(s) found");
		trace_attributes("public-key", *pub_head);
		trace_attributes("privat-key", *priv_head);
		return SKS_CKR_TEMPLATE_INCONSISTENT;
	}

	if (get_attribute_ptr(*pub_head, SKS_CKA_EC_PARAMS, &a_ptr, &a_size)) {
		EMSG("Not EC_PARAMS attribute found");
		return SKS_CKR_ATTRIBUTE_TYPE_INVALID;
	}

	tee_size = ec_params2tee_keysize(a_ptr, a_size);
	if (!tee_size)
		return SKS_CKR_ATTRIBUTE_TYPE_INVALID;

	tee_curve = ec_params2tee_curve(a_ptr, a_size);

	key_size = tee_size >> 3;

	uECC_curve_id = sks2uecc_curve_id(tee_curve);
	if (uECC_curve_id == UECC_CURVE_ID_UNSUPPORT)
	{
		rv = SKS_CKR_ARGUMENTS_BAD;
		goto bail;
	}

	rv = mtk_generate_ecc_key_pair(tee_size, uECC_curve_id, &keyid);
	if (rv)
		goto bail;

	/*
	 * Private key needs the same EC_PARAMS as used by the public key.
	 * Since it is also common for userspace to provide an empty private
	 * EC_PARAMS, make sure to remove before adding the correct one.
	 */
	remove_attribute(priv_head, SKS_CKA_EC_PARAMS);
	rv = add_attribute(priv_head, SKS_CKA_EC_PARAMS, a_ptr, a_size);
	if (rv)
		goto bail;

	rv = add_attribute(pub_head, SKS_CKA_HSM_KEY_ID, &keyid, sizeof(keyid));
	if (rv)
		goto bail;

	rv = add_attribute(pub_head, SKS_CKA_HSM_KEY_LEN, &key_size, sizeof(key_size));
	if (rv )
		goto bail;

	rv = add_attribute(priv_head, SKS_CKA_HSM_KEY_ID, &keyid, sizeof(keyid));
	if (rv)
		goto bail;

	rv = add_attribute(priv_head, SKS_CKA_HSM_KEY_LEN, &key_size, sizeof(key_size));
	if (rv )
		goto bail;

bail:
	return rv;
}

uint32_t import_hsm_ecc_keypair(struct sks_attrs_head **pub,
                                struct sks_attrs_head **priv,
                                uint8_t *keyblob,
                                int *blobsize)
{
	uint32_t rv = 0;
	uint8_t *pubkeybuf = NULL;
	uint32_t pubkeylength;
	uint8_t *privkeybuf = NULL;
	uint32_t privkeylength;
	void *data = NULL;
	uint32_t data_size;
	void *a_ptr = NULL;
	uint32_t a_size = 0;
	uint32_t tee_size = 0;
	uint32_t tee_curve = 0;
	int key_size = 0;
	int keyid = 0;
	int uECC_curve_id = -1;
	ecc_keypair_datastruct ecckeypair;

	TEE_MemFill(&ecckeypair, 0, sizeof(ecckeypair));

	if (!*pub || !*priv || !keyblob || !blobsize)
		return SKS_CKR_TEMPLATE_INCONSISTENT;

	rv = get_attribute_ptr(*priv, SKS_CKA_VALUE, &privkeybuf, &privkeylength);
	if (rv) {
		EMSG("[%s][%d] rv : %d\n", __FUNCTION__, __LINE__, rv);
		return SKS_CKR_GENERAL_ERROR;
	}

	rv = get_attribute_ptr(*pub, SKS_CKA_VALUE, &pubkeybuf, &pubkeylength);
	if (rv) {
		EMSG("[%s][%d] rv : %d\n", __FUNCTION__, __LINE__, rv);
		return SKS_CKR_GENERAL_ERROR;
	}

	if (get_attribute_ptr(*pub, SKS_CKA_EC_PARAMS, &a_ptr, &a_size)) {
		EMSG("Not EC_PARAMS attribute found");
		return SKS_CKR_ATTRIBUTE_TYPE_INVALID;
	}

	tee_size = ec_params2tee_keysize(a_ptr, a_size);
	if (!tee_size)
		return SKS_CKR_ATTRIBUTE_TYPE_INVALID;

	tee_curve = ec_params2tee_curve(a_ptr, a_size);

	key_size = tee_size >> 3;

	uECC_curve_id = sks2uecc_curve_id(tee_curve);
	if (uECC_curve_id == UECC_CURVE_ID_UNSUPPORT)
	{
		rv = SKS_CKR_ARGUMENTS_BAD;
		goto bail;
	}

	ecckeypair.curve_id = uECC_curve_id;
	TEE_MemMove(ecckeypair.public, pubkeybuf, pubkeylength);
	TEE_MemMove(ecckeypair.private, privkeybuf, privkeylength);

	rv = mtk_import_key((uint8_t *)&ecckeypair, sizeof(ecckeypair), &keyid, keyblob, blobsize, KEY_ALGO_ID_ECC);
	if (rv)
		goto bail;

	rv = add_attribute(pub, SKS_CKA_HSM_KEY_ID, &keyid, sizeof(keyid));
	if (rv)
		return SKS_CKR_GENERAL_ERROR;

	rv = add_attribute(pub, SKS_CKA_HSM_KEY_LEN, &key_size, sizeof(key_size));
	if (rv )
		return SKS_CKR_GENERAL_ERROR;

	rv = add_attribute(priv, SKS_CKA_HSM_KEY_ID, &keyid, sizeof(keyid));
	if (rv)
		return SKS_CKR_GENERAL_ERROR;

	rv = add_attribute(priv, SKS_CKA_HSM_KEY_LEN, &key_size, sizeof(key_size));
	if (rv )
		return SKS_CKR_GENERAL_ERROR;
bail:
	return rv;
}