// SPDX-License-Identifier: BSD-2-Clause
/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * MediaTek Inc. (C) 2020. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */
#include <kernel/misc.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <tee/cache.h>
#include <kernel/delay.h>
#include "system_event.h"
#include "mbox_serv.h"
#include "km_ipc.h"
#include "address.h"
#include "sync_obj.h"
#include "buddy.h"
#include "hsm_job.h"
#include "mtk_crypto.h"
#include "share_memmap.h"
#include "hsm_pta_debug.h"

#define HSM_IN_BUF_OFFSET   (0)
#define HSM_IN_BUF_LEN      (0x2000)                                // IN BUF 1  --> 8KB

#define HSM_IN_BUF2_OFFSET  (HSM_IN_BUF_LEN)
#define HSM_IN_BUF2_LEN     (0x400)                                 // IN BUF 2  --> 1KB

#define HSM_IN_BUF3_OFFSET  (HSM_IN_BUF2_OFFSET + HSM_IN_BUF2_LEN)
#define HSM_IN_BUF3_LEN     (0x400)                                 // IN BUF 3  --> 1KB

#define HSM_OUT_BUF_OFFSET  (HSM_IN_BUF3_OFFSET + HSM_IN_BUF3_LEN)
#define HSM_OUT_BUF_LEN     (0x2000)                                // OUT BUF 1 --> 8KB

#define HSM_OUT_BUF2_OFFSET (HSM_OUT_BUF_OFFSET + HSM_OUT_BUF_LEN)
#define HSM_OUT_BUF2_LEN    (0x400)                                 // OUT BUF 2 --> 1KB

#define HSM_OUT_JOB_OFFSET  (HSM_OUT_BUF2_OFFSET + HSM_OUT_BUF2_LEN)
#define HSM_OUT_JOB_LEN     (0x80)                                  // JOB BUF   --> 128B

#define HSM_OUT_RET_OFFSET  (HSM_OUT_JOB_OFFSET + HSM_OUT_JOB_LEN)
#define HSM_OUT_RET_LEN     (0x10)                                  // RET BUF   --> 16B

vaddr_t mbox_io_buf_base;

static uint32_t algo_2_eventid(uint32_t algo_type)
{
	uint32_t event_id = 0xffffffff;

	switch (algo_type)
	{
	case CRYPTO_ALGOFAM_SHA1:
	case CRYPTO_ALGOFAM_SHA2_224:
	case CRYPTO_ALGOFAM_SHA2_256:
	case CRYPTO_ALGOFAM_SHA2_384:
	case CRYPTO_ALGOFAM_SHA2_512:
	case CRYPTO_ALGOFAM_WHP:
	case CRYPTO_ALGOFAM_AES:
		event_id = SYS_EVENT_CRY_AES_SHA;
		break;
	case CRYPTO_ALGOFAM_ECCNIST:
	case CRYPTO_ALGOFAM_ECCNIST_SM2:
		event_id = SYS_EVENT_CRY_ECC;
		break;
	case CRYPTO_ALGOFAM_RNG:
	case CRYPTO_ALGOFAM_SECURECOUNTER:
		event_id = SYS_EVENT_CRY_TRNG;
		break;
	case CRYPTO_ALGOFAM_NOT_SET:
	default:
		break;
	}

	return event_id;
}

static inline void _set_mbox_tx_header(struct sys_event *tx, int cpu_serial, int event_id)
{
	if (tx == NULL) {
		HSM_PTA_ERROR("[%s][%d] invalid input ptr\n", __FUNCTION__, __LINE__);
		return;
	}
	memset(tx, 0, sizeof(struct sys_event));
	tx->cpu_serial = CPU_SERIAL_HSM;
	tx->event_id = SYS_EVENT_KEY_MANAGEMENT;
}


int hsm_export_key(uint8_t *pblob, uint32_t blobsize, uint8_t *ppubkey, uint32_t *ppubkeylength) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	sync_handle sync = {0};
	addr_t in_share_buffer = {0};
	addr_t out_share_buffer = {0};
	addr_t keylength = {0};
	int hsm_ret = -1;

	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(blobsize, &in_share_buffer) != 0 ) {
		goto fail;
	}
	if (physical_malloc(*ppubkeylength, &out_share_buffer) != 0 ) {
		goto fail;
	}
	if (physical_malloc(sizeof(uint32_t), &keylength) != 0 ) {
		goto fail;
	}
	memput_and_cclean(&in_share_buffer, pblob, blobsize);
	memput_and_cclean(&keylength, ppubkeylength, sizeof(uint32_t));

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_EXPORT_KEY;
	tx.data.data[1] = get_hsm_address(&in_share_buffer);
	tx.data.data[2] = blobsize;
	tx.data.data[3] = get_hsm_address(&out_share_buffer);
	tx.data.data[4] = get_hsm_address(&keylength);
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}

	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);

	cclean_and_memcopy(ppubkeylength, &keylength, sizeof(int));
	cclean_and_memcopy(ppubkey, &out_share_buffer, *ppubkeylength);

	physical_free(&in_share_buffer);
	physical_free(&out_share_buffer);
	physical_free(&keylength);

	return hsm_ret;
fail:
	release_sync_handle(&sync);
	physical_free(&in_share_buffer);
	physical_free(&out_share_buffer);
	physical_free(&keylength);

	return hsm_ret;

}

int hsm_import_key(uint8_t *keybuffer, int size, int *pkey_id, int algo_id) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	sync_handle sync = {0};
	addr_t keyid = {0};
	addr_t share_buffer = {0};
	int hsm_ret = -1;

	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(sizeof(int), &keyid) != 0) {
		goto fail;
	}
	if (physical_malloc(size, &share_buffer) != 0 ) {
		goto fail;
	}
	memput_and_cclean(&share_buffer, keybuffer, size);

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_IMPORT_KEY;
	tx.data.data[1] = get_hsm_address(&share_buffer);
	tx.data.data[2] = size;
	tx.data.data[3] = get_hsm_address(&keyid);
	tx.data.data[4] = algo_id;
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}

	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);

	cclean_and_memcopy(pkey_id, &keyid, sizeof(int));

	physical_free(&keyid);
	physical_free(&share_buffer);

	return hsm_ret;
fail:
	release_sync_handle(&sync);
	physical_free(&keyid);
	physical_free(&share_buffer);

	return hsm_ret;
}

int hsm_delete_key(int key_id) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	sync_handle sync = {0};
	int hsm_ret = -1;
	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_DELETE_KEY;
	tx.data.data[1] = key_id;
	tx.data.data[14] = get_hsm_address(&sync);
	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}

	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}

	release_sync_handle(&sync);
	return hsm_ret;
fail:
	release_sync_handle(&sync);
	return hsm_ret;
}

int hsm_gen_key_pair(int size, int uECC_curve_id, int *pkey_id) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	sync_handle sync = {0};
	addr_t keyid = {0};
	int hsm_ret = -1;

	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(sizeof(int), &keyid) != 0) {
		goto fail;
	}

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_KEY_PAIR_GENERATION;
	tx.data.data[1] = size;
	tx.data.data[2] = uECC_curve_id;
	tx.data.data[3] = get_hsm_address(&keyid);
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}

	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);

	cclean_and_memcopy(pkey_id, &keyid, sizeof(int));

	physical_free(&keyid);
	return hsm_ret;
fail:
	release_sync_handle(&sync);
	physical_free(&keyid);

	return hsm_ret;
}

int hsm_gen_symm_key(int size, int *pkeyid) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	sync_handle sync = {0};
	addr_t keyid = {0};
	int hsm_ret = -1;

	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(sizeof(int), &keyid) != 0) {
		goto fail;
	}

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_SYMM_KEY_GENERATION;
	tx.data.data[1] = size;
	tx.data.data[2] = get_hsm_address(&keyid);
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}
	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);

	cclean_and_memcopy(pkeyid, &keyid, sizeof(int));

	physical_free(&keyid);
	return hsm_ret;
fail:
	HSM_PTA_ERROR("[%s][%d] mbox if fail ret : %d\n", __FUNCTION__, __LINE__, hsm_ret);
	release_sync_handle(&sync);
	physical_free(&keyid);

	return hsm_ret;
}


int hsm_get_key_blob(uint8_t *dst, int size, int key_id, int algo_id, int *pblobsize) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	sync_handle sync = {0};
	addr_t buffer = {0};
	addr_t blob_size = {0};
	int bsize = 0;
	int hsm_ret = -1;

	if (dst == NULL || pblobsize == NULL) {
		goto fail;
	}
	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(size, &buffer) != 0) {
		goto fail;
	}
	if (physical_malloc(sizeof(int), &blob_size) != 0) {
		goto fail;
	}

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_GET_KEY_BLOB;
	tx.data.data[1] = get_hsm_address(&buffer);
	tx.data.data[2] = size;
	tx.data.data[3] = key_id;
	tx.data.data[4] = algo_id;
	tx.data.data[5] = get_hsm_address(&blob_size);
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}
	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);

	cclean_and_memcopy(&bsize, &blob_size, sizeof(int));

	if (size < bsize) {
		goto fail;
	}
	*pblobsize = bsize;

	cclean_and_memcopy(dst, &buffer, bsize);

	physical_free(&buffer);
	physical_free(&blob_size);

	return hsm_ret;

fail:
	HSM_PTA_ERROR("[%s][%d] mbox if fail ret : %d\n", __FUNCTION__, __LINE__, hsm_ret);
	release_sync_handle(&sync);
	physical_free(&buffer);
	physical_free(&blob_size);

	return hsm_ret;

}

int hsm_get_key_table(uint8_t *dst, int size, int *ptablesize) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	int hsm_ret = -1;
	sync_handle sync = {0};
	addr_t buffer = {0};
	addr_t table_size = {0};
	int tblsize = 0;

	if (dst == NULL || ptablesize == NULL) {
		goto fail;
	}
	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(size, &buffer) != 0) {
		goto fail;
	}
	if (physical_malloc(sizeof(int), &table_size) != 0) {
		goto fail;
	}

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_GET_KEY_TABLE;
	tx.data.data[1] = get_hsm_address(&buffer);
	tx.data.data[2] = size;
	tx.data.data[3] = get_hsm_address(&table_size);
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}
	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);

	cclean_and_memcopy(&tblsize, &table_size, sizeof(int));

	if (size < tblsize) {
		goto fail;
	}
	*ptablesize = tblsize;

	cclean_and_memcopy(dst, &buffer, tblsize);

	physical_free(&buffer);
	physical_free(&table_size);

	return hsm_ret;

fail:
	HSM_PTA_ERROR("[%s][%d] mbox if fail ret : %d\n", __FUNCTION__, __LINE__, hsm_ret);
	release_sync_handle(&sync);
	physical_free(&buffer);
	physical_free(&table_size);

	return hsm_ret;
}

int hsm_rebuild_key_table(uint8_t *buffer, int table_size) {
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	int hsm_ret = -1;
	sync_handle sync = {0};
	addr_t share_buffer = {0};
	int tblsize = 0;

	if (buffer == NULL) {
		goto fail;
	}
	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(table_size, &share_buffer) != 0 ) {
		goto fail;
	}
	memput_and_cclean(&share_buffer, buffer, table_size);

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_REBUILD_KEY_TABLE;
	tx.data.data[1] = get_hsm_address(&share_buffer);
	tx.data.data[2] = table_size;
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}
	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);
	physical_free(&share_buffer);

	return hsm_ret;

fail:
	HSM_PTA_ERROR("[%s][%d] mbox if fail ret : %d\n", __FUNCTION__, __LINE__, hsm_ret);
	release_sync_handle(&sync);
	physical_free(&share_buffer);

	return hsm_ret;
}

int hsm_rebuild_key_blob(uint8_t *buffer, int blob_size, int key_id)
{
	struct sys_event tx;
	uint8_t mbox_ret = -1;
	int hsm_ret = -1;
	sync_handle sync = {0};
	addr_t share_buffer = {0};
	int tblsize = 0;

	if (buffer == NULL) {
		goto fail;
	}
	if (allocate_sync_handle(&sync) != 0 ) {
		goto fail;
	}
	if (physical_malloc(blob_size, &share_buffer) != 0 ) {
		goto fail;
	}
	memput_and_cclean(&share_buffer, buffer, blob_size);

	_set_mbox_tx_header(&tx, CPU_SERIAL_HSM, SYS_EVENT_KEY_MANAGEMENT);

	tx.data.data[0] = KM_MBOX_CMD_REBUILD_KEY_BLOB;
	tx.data.data[1] = get_hsm_address(&share_buffer);
	tx.data.data[2] = blob_size;
	tx.data.data[3] = key_id;
	tx.data.data[14] = get_hsm_address(&sync);

	mbox_ret = mbox_serv_tx(&tx);
	if (mbox_ret != MBOX_SERV_OK) {
		goto fail;
	}
	wait_remote_sync_done(&sync);
	hsm_ret = get_hsm_return_value(&sync);
	if (hsm_ret != 0) {
		goto fail;
	}
	release_sync_handle(&sync);
	physical_free(&share_buffer);

	return hsm_ret;

fail:
	HSM_PTA_ERROR("[%s][%d] mbox if fail ret : %d\n", __FUNCTION__, __LINE__, hsm_ret);
	release_sync_handle(&sync);
	physical_free(&share_buffer);

	return hsm_ret;
}

int hsm_send_job(job_struct *pjob)
{
	uint32_t in_len, in_len2, in_len3;
	uint8_t *in_buf = NULL;
	uint8_t *in_buf3 = NULL;
	uint32_t out_len = 0, out_len2 = 0;
	uint8_t *out_buf = NULL;
	uint8_t *out_buf2 = NULL;
	uint8_t *IV = NULL;
	job_struct job2hsm;
	uint32_t hsmaddr_job = 0;
	int ret = -1;
	struct sys_event tx = {0};

	HSM_PTA_DEBUG("[\x1b[31m%s\033[0m][%d]\n", __FUNCTION__, __LINE__);
	memcpy(&job2hsm, pjob, sizeof(job_struct));

	tx.event_id = algo_2_eventid(job2hsm.family);

	// In buffer #1
	HSM_PTA_DEBUG("\x1b[1;32m  inputLength: %d  \033[0m \n", job2hsm.inputLength);
	in_len = job2hsm.inputLength;
	if (in_len != 0)
	{
		in_buf = (uint8_t *)job2hsm.inputPtr;
		memcpy(mbox_io_buf_base + HSM_IN_BUF_OFFSET, in_buf, in_len);
		ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_IN_BUF_OFFSET, &hsmaddr_job);
		if (ret != 0)
		{
			HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
			return -1;
		}
		job2hsm.inputPtr = hsmaddr_job;
	}

	// In buffer #2
	HSM_PTA_DEBUG("\x1b[1;32m  secondaryInputLength: %d  \033[0m \n", job2hsm.secondaryInputLength);
	in_len2 = job2hsm.secondaryInputLength;
	if (in_len2 != 0)
	{
		IV = (uint8_t *)job2hsm.secondaryInputPtr;
		memcpy(mbox_io_buf_base + HSM_IN_BUF2_OFFSET, IV, in_len2);
		ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_IN_BUF2_OFFSET, &hsmaddr_job);
		if (ret != 0)
		{
			HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
			return -1;
		}
		job2hsm.secondaryInputPtr = hsmaddr_job;
	}

	// In buffer #3
	HSM_PTA_DEBUG("\x1b[1;32m  tertiaryInputLength: %d  \033[0m \n", job2hsm.tertiaryInputLength);
	in_len3 = job2hsm.tertiaryInputLength;
	if (in_len3 != 0)
	{
		in_buf3 = (uint8_t *)job2hsm.tertiaryInputPtr;
		memcpy(mbox_io_buf_base + HSM_IN_BUF3_OFFSET, in_buf3, in_len2);
		ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_IN_BUF3_OFFSET, &hsmaddr_job);
		if (ret != 0)
		{
			HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
			return -1;
		}
		job2hsm.tertiaryInputPtr = hsmaddr_job;
	}

	// Out buffer #1
	HSM_PTA_DEBUG("\x1b[1;32m  OutputLength: %d  \033[0m \n", job2hsm.outputLength);
	out_len = job2hsm.outputLength;
	out_buf = (uint8_t *)job2hsm.outputPtr;
	if (out_len != 0)
	{
		ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_OUT_BUF_OFFSET, &hsmaddr_job);
		if (ret != 0)
		{
			HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
			return -1;
		}
		job2hsm.outputPtr = hsmaddr_job;
	}

	// Out buffer #2
	HSM_PTA_DEBUG("\x1b[1;32m  secondaryOutputLength: %d  \033[0m \n", job2hsm.secondaryOutputLength);
	out_len2 = job2hsm.secondaryOutputLength;
	out_buf2 = (uint8_t *)job2hsm.secondaryOutputPtr;
	if (out_len2 != 0)
	{
		ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_OUT_BUF2_OFFSET, &hsmaddr_job);
		if (ret != 0)
		{
			HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
			return -1;
		}
		job2hsm.secondaryOutputPtr = hsmaddr_job;
	}

	// Return value pointer
	ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_OUT_RET_OFFSET, &hsmaddr_job);
	if (ret != 0)
	{
		HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
		return -1;
	}
	job2hsm.retValPtr = hsmaddr_job;


	HSM_PTA_DEBUG(" job2hsm.event_id :              0x%x\r\n", (uint32_t)job2hsm.event_id);
	HSM_PTA_DEBUG(" job2hsm.priority :              0x%x\r\n", (uint32_t)job2hsm.priority);
	HSM_PTA_DEBUG(" job2hsm.job_id :                0x%x\r\n", (uint32_t)job2hsm.job_id);
	HSM_PTA_DEBUG(" job2hsm.service :               0x%x\r\n", (uint32_t)job2hsm.service);
	HSM_PTA_DEBUG(" job2hsm.family :                0x%x\r\n", (uint32_t)job2hsm.family);
	HSM_PTA_DEBUG(" job2hsm.operation_mode :        0x%x\r\n", (uint32_t)job2hsm.operation_mode);
	HSM_PTA_DEBUG(" job2hsm.mode :                  0x%x\r\n", (uint32_t)job2hsm.mode);
	HSM_PTA_DEBUG(" job2hsm.cryptoKeyId :           0x%x\r\n", (uint32_t)job2hsm.cryptoKeyId);
	HSM_PTA_DEBUG(" job2hsm.keyLength :             0x%x\r\n", (uint32_t)job2hsm.keyLength);

	// Job buffer
	memcpy(mbox_io_buf_base + HSM_OUT_JOB_OFFSET, &job2hsm, sizeof(job_struct));
	if (cache_operation(TEE_CACHECLEAN, mbox_io_buf_base, HSM_OUT_RET_OFFSET) != TEE_SUCCESS)
	{
		HSM_PTA_ERROR("\x1b[31m == CACHE FAIL!! == \033[0m\n");
		return -1;
	}
	if (cache_operation(TEE_CACHEINVALIDATE, mbox_io_buf_base, (HSM_OUT_RET_OFFSET + HSM_OUT_RET_LEN)) != TEE_SUCCESS)
	{
		HSM_PTA_ERROR("\x1b[31m == CACHE INVALIDATE FAIL!! == \033[0m\n");
		return -1;
	}


	ret = hsm_addr_remap((uint32_t)HSM_ALL_IO_BUF_BASE + HSM_OUT_JOB_OFFSET, &hsmaddr_job);
	if (ret != 0)
	{
		HSM_PTA_ERROR("[\x1b[31m%s\033[0m] remap error : %d\n", __func__, ret);
		return -1;
	}
	tx.data.data[0] = hsmaddr_job;


	// Give return value a initial value
	write8(MBOX_SERV_FEEDBACK_INIT_VAL, mbox_io_buf_base + HSM_OUT_RET_OFFSET);

	// Send job with physical address to HSM, and wait for job done.
	ret = mbox_serv_tx(&tx);

	// Waiting for HSM done the job
	uint8_t hsm_feedback_result = MBOX_SERV_FEEDBACK_INIT_VAL;
	while (1)
	{
		// Invalidating cache
		if (cache_operation(TEE_CACHECLEAN, mbox_io_buf_base, (HSM_OUT_RET_OFFSET + HSM_OUT_RET_LEN)) != TEE_SUCCESS)
		{
			HSM_PTA_ERROR("\x1b[31m == CACHE CLEAN FAIL!! == \033[0m\n");
			return -1;
		}
		if (cache_operation(TEE_CACHEINVALIDATE, mbox_io_buf_base, (HSM_OUT_RET_OFFSET + HSM_OUT_RET_LEN)) != TEE_SUCCESS)
		{
			HSM_PTA_ERROR("\x1b[31m == CACHE INVALIDATE FAIL!! == \033[0m\n");
			return -1;
		}

		hsm_feedback_result = read8(mbox_io_buf_base + HSM_OUT_RET_OFFSET);
		// HSM_PTA_ERROR("hsm_feedback_result  %d \033[0m\n", hsm_feedback_result);
		if (hsm_feedback_result != MBOX_SERV_FEEDBACK_INIT_VAL)
		{
			HSM_PTA_DEBUG("\x1b[33m Got HSM feedback: 0x%x \033[0m\n", hsm_feedback_result);
			break;
		}
		mdelay(1);
	}

	if (hsm_feedback_result == MBOX_SERV_FEEDBACK_VERIFY_FAIL) {
		HSM_PTA_ERROR("\x1b[31m == Verification FAIL!! == \033[0m\n");
		return TEE_ERROR_SIGNATURE_INVALID;
	}


	if (out_len != 0) {
		memcpy(out_buf, mbox_io_buf_base + HSM_OUT_BUF_OFFSET, out_len);
	}

	if (out_len2 != 0) {
		memcpy(out_buf2, mbox_io_buf_base + HSM_OUT_BUF2_OFFSET, out_len2);
	}

	return 0;
}
