/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <kernel/interrupt.h>
#include <mm/core_memprot.h>

#include "reg_op.h"
#include "mbox_hw.h"
#include "mbox_sw.h"
#include "mbox_drv.h"

// extern void mtk_mbox_irq_start_thread(int irq);
// extern void mtk_mbox_irq_stop_thread(int irq);


struct mbox_info {
	const uint32_t base;
	struct mbox_reg *vbase;
	uint32_t irq;
	uint32_t err_irq_num;
};

struct erm_info {
	const uint32_t base;
	struct erm_reg *vbase;
	uint32_t irq;
};


register_phys_mem(MEM_AREA_IO_SEC, HSM_MBOX_BASE, MBOX_REGISTER_RANGE);
register_phys_mem(MEM_AREA_IO_SEC, HSM_ERMST_BASE, ERMST_REGISTER_RANGE);

/* MBOX base mapping */
static struct mbox_info mbox[CPU_SERIAL_CNT] = {
	{(const uint32_t)HSM_MBOX_BASE, (struct mbox_reg*)NULL, IVALID_IRQ_ID,  INVALID_MBOX_E_IRQ},
	{(const uint32_t)HSM_MBOX_BASE, (struct mbox_reg*)NULL, IVALID_IRQ_ID,  INVALID_MBOX_E_IRQ},
};

static struct erm_info erm = {
	.base = HSM_ERMST_BASE,
	.vbase = NULL,
	.irq = ERM_IRQ_ID,
};

/* pointer to self */
static const struct mbox_info *p_mbox;

/* callback function */
static void (*mbox_user_isr)(uint8_t msg_id);

static uint8_t ca53_itr_msg_id = -1;


void mbox_reset(void)
{
	uint8_t i = 0;
	write32(&p_mbox->vbase->msg_irq_en, 0UL);
	write32(&p_mbox->vbase->region_irq_en, 0UL);

	for (i = MBOX_MSG_ID_MIN; i <= MBOX_MSG_ID_MAX; i++) {
		write32(&p_mbox->vbase->msg[i], 0UL);
	}

	write32(&p_mbox->vbase->msg_pending, 0xFFFFFFFFUL);
}

/* MBOX_REQ_MSG_PENDING */
static inline uint32_t mbox_get_msg_bit_offset(uint8_t msg_id)
{
	return BIT(31UL - (uint32_t)(msg_id));
}

/* MBOX_NS_DOMAIN_CFG */
static inline uint32_t mbox_get_domain_offset(enum mbox_regions region)
{
	return (28UL - ((uint32_t)(region) * 4UL));
}

static inline uint32_t mbox_get_sec_offset(enum mbox_regions region)
{
	return (31UL - ((uint32_t)(region) * 4UL));
}

static inline uint32_t mbox_get_boundary_offset(uint8_t boundary)
{
	return (24UL - ((uint32_t)(boundary) * 8UL));
}

static inline int8_t mbox_is_pending(uint8_t msg_id, uint32_t pending)
{
	int8_t result = MBOX_FAIL;

	if ((pending & mbox_get_msg_bit_offset(msg_id)) > 0UL) {
		result = MBOX_SUCC;
	}

	return result;
}

static void __mbox_register_interrupt(void)
{
	uint32_t irq = erm.irq;

	// mtk_mbox_irq_start_thread(irq);
}

static void __mbox_unregister_interrupt(void)
{
	uint32_t irq = erm.irq;

	// mtk_mbox_irq_stop_thread(irq);
}

static enum itr_return mbox_isr(struct itr_handler *h)
{
	EMSG("[%s: %d]\n", __func__, __LINE__);
	uint32_t pending = read32(&erm.vbase->pending);

	/*
	 * clear pending.
	 * Please clear it before callback to avoid racing condition in the
	 * MBOX service.
	 */
	write32(&erm.vbase->pending, pending);

	if ((mbox_user_isr != NULL) && ((pending & BIT(ERM_LBIST_BIT) ) != 0UL)) {
		mbox_user_isr(ca53_itr_msg_id);
		mbox_irq_done();
	}
	return ITRR_HANDLED;
}
KEEP_PAGER(mbox_isr);

uint8_t mbox_itr_data = 0;
static struct itr_handler mbox_itr = {
	.it = ERM_IRQ_ID,
	.flags = ITRF_TRIGGER_LEVEL,
	.handler = mbox_isr,
	.data = &mbox_itr_data,
};
KEEP_PAGER(mbox_itr);

extern bool itr_set_polarity(unsigned int irq, unsigned int polarity);
static int init_mbox_itr(void)
{
	*((uint8_t*)(mbox_itr.data)) = 321;
	itr_add(&mbox_itr);
	itr_enable(ERM_IRQ_ID);
	return 0;
}

void mbox_init(enum CPU_SERIAL cpu_serial,
               void (*p_mbox_msg_isr)(uint8_t msg_id))
{
	// uint8_t *RegBase_virtual;

	/* point to self */
	p_mbox = &mbox[cpu_serial];

	for (int i = 0; i < CPU_SERIAL_CNT; ++i)
	{
		if (mbox[i].base != (const uint32_t)NULL)
		{
			mbox[i].vbase = (vaddr_t)phys_to_virt_io(mbox[i].base);
			IMSG("[%s] mbox[i].vbase : 0x%x\n", __FUNCTION__, mbox[i].vbase);
		}
	}
	erm.vbase = (vaddr_t)phys_to_virt_io(erm.base);
	IMSG("[%s] erm.vbase : 0x%x\n", __FUNCTION__, erm.vbase);
	// mbox_reset();

	/* store callback */
	mbox_user_isr = p_mbox_msg_isr;

	// init_mbox_itr();

}

void mbox_deinit(void)
{

}

void mbox_irq_done(void)
{
	uint32_t pending = read32(&erm.vbase->pending);
	write32(&erm.vbase->pending, pending);

	clrbits_le32(&erm.vbase->irq_set, BIT(ERM_LBIST_BIT));
}

void mbox_msg_tx(enum CPU_SERIAL send_to, uint8_t msg_id, uint32_t msg)
{
	if (msg_id <= MBOX_MSG_ID_MAX) {
		write32(&mbox[send_to].vbase->msg[msg_id], msg);
	}
}

void mbox_msg_rx(enum CPU_SERIAL send_to, uint8_t msg_id, uint32_t *pmsg)
{
	if ((msg_id <= MBOX_MSG_ID_MAX) && (pmsg != NULL)) {
		*pmsg = read32(&mbox[send_to].vbase->msg[msg_id]);
	}
}

void mbox_set_ca53_intr_msg_id(uint8_t msg_id)
{
	ca53_itr_msg_id = msg_id;
}
