/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein
 * is confidential and proprietary to MediaTek Inc. and/or its licensors.
 * Without the prior written permission of MediaTek inc. and/or its licensors,
 * any reproduction, modification, use or disclosure of MediaTek Software,
 * and information contained herein, in whole or in part, shall be strictly prohibited.
 *
 * MediaTek Inc. (C) 2019. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER ON
 * AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL WARRANTIES,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.
 * NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH RESPECT TO THE
 * SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY, INCORPORATED IN, OR
 * SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES TO LOOK ONLY TO SUCH
 * THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO. RECEIVER EXPRESSLY ACKNOWLEDGES
 * THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES
 * CONTAINED IN MEDIATEK SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK
 * SOFTWARE RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S ENTIRE AND
 * CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE RELEASED HEREUNDER WILL BE,
 * AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE MEDIATEK SOFTWARE AT ISSUE,
 * OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE CHARGE PAID BY RECEIVER TO
 * MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek Software")
 * have been modified by MediaTek Inc. All revisions are subject to any receiver's
 * applicable license agreements with MediaTek Inc.
 */
#include <stdio.h>
#include <stdbool.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>

#include "mtkexpat.h"
#include "expat.h"
//#include "utility.h"

#ifndef UNUSED
#define UNUSED(x) (x)=(x)
#endif

#define MTK_EXPAT_DEBUG_ENABLE 0

void mtk_expat_init(mtk_expat_context* ctx) {
    if(ctx == NULL) return;
    ctx->element = NULL;
    ctx->last_element = &(ctx->element);
    ctx->last_attr = NULL;
    sprintf(ctx->error_string, "none");
}

void mtk_expat_cleanup(mtk_expat_context* ctx) {
    if(ctx == NULL) return;
    mtk_expat_element* element = ctx->element;
    while(element) {
        if(element->name) {
            free(element->name);
            element->name = NULL;
        }
        mtk_expat_attribute* attr = element->attr;
        while(attr) {
            if(attr->key) {
               free(attr->key);
               attr->key = NULL;
            }
            if(attr->value) {
                free(attr->value);
                attr->value = NULL;
            }
            mtk_expat_attribute* tmp_attr = attr;
            attr = attr->next;
            free(tmp_attr);
        }
        mtk_expat_element* tmp_element = element;
        element = element->next;
        free(tmp_element);
    }
    mtk_expat_init(ctx);
}

void mtk_expat_dump(mtk_expat_context* ctx) {
    printf("mtk_expat_dump()\n");
    if(ctx == NULL) {
        printf("  null\n");
        return;
    }
    mtk_expat_element* element = ctx->element;
    while(element) {
        printf("  name=[%s]", element->name);
        mtk_expat_attribute* attr = element->attr;
        while(attr) {
            printf(" [%s]=[%s]", attr->key, attr->value);
            attr = attr->next;
        }
        printf("\n");
        element = element->next;
    }
}

void mtk_expat_dump_element(mtk_expat_element* element) {
    printf("mtk_expat_dump_element\n");
    if(element == NULL) {
        printf("  null\n");
        return;
    }
    printf("  name=[%s]", element->name);
    mtk_expat_attribute* attr = element->attr;
    while(attr) {
        printf(" [%s]=[%s]", attr->key, attr->value);
        attr = attr->next;
    }
    printf("\n");
}

mtk_expat_element* mtk_expat_add_element(mtk_expat_context* ctx, const char* name) {
    mtk_expat_element* element = NULL;
    (*ctx->last_element) = malloc(sizeof(mtk_expat_element));
    element = (*ctx->last_element);
    memset(element, 0, sizeof(mtk_expat_element));
    element->name = malloc(strlen(name) + 1);
    safe_strncpy(element->name, name, strlen(name) + 1);
    ctx->last_attr = &(element->attr);
    ctx->last_element = &element->next;
    return element;
}

void mtk_expat_add_attr_string(mtk_expat_context* ctx, const char* key, const char* value) {
    if(ctx->last_attr == NULL) return;
    (*ctx->last_attr) = malloc(sizeof(mtk_expat_attribute));
    memset((*ctx->last_attr), 0, sizeof(mtk_expat_attribute));
    (*ctx->last_attr)->key = malloc(strlen(key) + 1);
    safe_strncpy((*ctx->last_attr)->key, key, strlen(key) + 1);
    (*ctx->last_attr)->value = malloc(strlen(value) + 1);
    safe_strncpy((*ctx->last_attr)->value, value, strlen(value) + 1);
    ctx->last_attr = &((*ctx->last_attr)->next);
}

void mtk_expat_add_attr_bool(mtk_expat_context* ctx, const char* key, bool value) {
    mtk_expat_add_attr_string(ctx, key, value? "true" : "false");
}

void mtk_expat_add_attr_int(mtk_expat_context* ctx, const char* key, int value) {
    char tmp[16] = {0};
    sprintf(tmp, "%d", value);
    mtk_expat_add_attr_string(ctx, key, tmp);
}

int mtk_expat_get_element_num(mtk_expat_context* ctx, const char* name) {
    int count = 0;
    mtk_expat_element* element = ctx->element;
    while(element) {
        if(strcmp(element->name, name) == 0) {
            count++;
        }
        element = element->next;
    }
    return count;
}

mtk_expat_element* mtk_expat_get_element(mtk_expat_context* ctx, const char* name) {
    mtk_expat_element* element = ctx->element;
    while(element) {
        if(strcmp(element->name, name) == 0) {
            return element;
        }
        element = element->next;
    }
    return NULL;
}

mtk_expat_element* mtk_expat_get_element_index(mtk_expat_context* ctx, const char* name, int index) {
    int count = 0;
    mtk_expat_element* element = ctx->element;
    while(element) {
        if(strcmp(element->name, name) == 0) {
            if(count == index) {
                return element;
            }
            count++;
        }
        element = element->next;
    }
    return NULL;
}

const char* mtk_expat_get_attr_string(mtk_expat_element* element, const char* key, const char* default_value) {
    if(element == NULL) return default_value;
    mtk_expat_attribute* attr = element->attr;
    while(attr) {
        if(strcmp(attr->key, key) == 0) {
            return attr->value;
        }
        attr = attr->next;
    }
    return default_value;
}

bool mtk_expat_get_attr_bool(mtk_expat_element* element, const char* key, bool default_value) {
    const char* value = mtk_expat_get_attr_string(element, key, "");
    if(strcmp(value, "true") == 0) {
        return true;
    }
    if(strcmp(value, "false") == 0) {
        return false;
    }
    return default_value;
}

int mtk_expat_get_attr_int(mtk_expat_element* element, const char* key, int default_value) {
    int ret = 0;
    const char* value = mtk_expat_get_attr_string(element, key, "");
    if(sscanf((char*)value, "%d", &ret) <= 0) {
        return default_value;
    }
    return ret;
}

const char* mtk_expat_strerror(mtk_expat_context* ctx) {
    return ctx->error_string;
}

static void mtk_expat_xml_save_int(FILE* fp, mtk_expat_context* ctx) {
    mtk_expat_element* element = ctx->element;
    mtk_expat_attribute* attr;
    const char* tag = NULL;
    if(element) {
        tag = element->name;
        fprintf(fp, "<%s", tag);
        attr = element->attr;
        while(attr) {
            fprintf(fp, " %s=\"%s\"", attr->key, attr->value);
            attr = attr->next;
        }
        fprintf(fp, ">\n");
        element = element->next;
    }

    while(element) {
        fprintf(fp, "  <%s", element->name);
        attr = element->attr;
        while(attr) {
            fprintf(fp, " %s=\"%s\"", attr->key, attr->value);
            attr = attr->next;
        }
        fprintf(fp, "/>\n");
        element = element->next;
    }

    if(tag) {
        fprintf(fp, "</%s>\n", tag);
    }
}

// return false means error and can use mtk_expat_strerror(ctx) to get the error info in the string format
bool mtk_expat_xml_save(mtk_expat_context* ctx, const char* file) {
    bool ret = false;
    FILE* fp = NULL;

    do {
        fp = fopen(file, "w");
        if(fp == NULL) {

            sprintf(ctx->error_string, "fopen() failed reason=[%s]", strerror(errno));
            break;
        }
        fprintf(fp, "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n");
        mtk_expat_xml_save_int(fp, ctx);

        ret = true;
        sprintf(ctx->error_string, "none");
    } while(0);

    if(fp) {
        fclose(fp);
    }
    return ret;
}

static int mtk_expat_get_file_size(const char* file) {
    struct stat s;
    if(stat(file, &s) == -1) {
        return -1;
    }
    return s.st_size;
}

//======================= non-reentrant ==================================
static mtk_expat_context g_expat_ctx;
static int g_depth = 0;

static void mtk_expat_start_element(void *data, const char *element, const char **attribute) {
    int i;
    UNUSED(data);
    for(i = 0; i < g_depth; i++) {
        if(MTK_EXPAT_DEBUG_ENABLE) printf(" ");
    }
    if(MTK_EXPAT_DEBUG_ENABLE) printf("element=[%s]", element);
    mtk_expat_add_element(&g_expat_ctx, element);

    for(i = 0; attribute[i]; i += 2) {
        if(MTK_EXPAT_DEBUG_ENABLE) printf(" [%s]=[%s]", attribute[i], attribute[i + 1]);
        mtk_expat_add_attr_string(&g_expat_ctx, attribute[i], attribute[i + 1]);
    }

    if(MTK_EXPAT_DEBUG_ENABLE) printf("\n");
    g_depth++;
}

static void mtk_expat_end_element(void *data, const char *el) {
    UNUSED(data);
    UNUSED(el);
    g_depth--;
}

// this is non-reentrant API, do not call this API from different thread in the same time
// return false means error and can use mtk_expat_strerror(ctx) to get the error info in the string format
bool mtk_expat_xml_load(mtk_expat_context* ctx, const char* file) {
    bool ret = false;
    XML_Parser parser = NULL;
    char* buff = NULL;
    FILE *fp = NULL;

    g_expat_ctx = *ctx;

    do {
        fp = fopen(file, "r");
        if(fp == NULL) {
            sprintf(ctx->error_string, "fopen() failed reason=[%s]", strerror(errno));
            break;
        }

        int file_size = mtk_expat_get_file_size(file);
        if(file_size == -1) {
            sprintf(ctx->error_string, "stat() failed reason=[%s]", strerror(errno));
            break;
        }

        buff = (char*)malloc(file_size);
        if(buff == NULL) {
            sprintf(ctx->error_string, "malloc() failed");
            break;
        }
        memset(buff, 0, file_size);

        size_t read_file_size = 0;
        read_file_size = fread(buff, sizeof(char), file_size, fp);
        if(read_file_size != (size_t)file_size) {
            sprintf(ctx->error_string, "fread() failed");
            break;
        }

        parser = XML_ParserCreate(NULL);
        XML_SetElementHandler(parser, mtk_expat_start_element, mtk_expat_end_element);
        if(XML_Parse(parser, buff, file_size, XML_TRUE) == XML_STATUS_ERROR) {
            sprintf(ctx->error_string, "XML_Parse() failed reason=[%s]",
                XML_ErrorString(XML_GetErrorCode(parser)));
            break;
        }

        ret = true;
        sprintf(ctx->error_string, "none");
    } while(0);

    if(parser) {
        XML_ParserFree(parser);
        parser = NULL;
    }
    if(buff) {
        free(buff);
        buff = NULL;
    }
    if(fp) {
        fclose(fp);
        fp = NULL;
    }
    return ret;
}

