#include "mtk_gnss_at_command.h"
#include "mtk_gnss_at_server.h"
#include "mtk_gnss_at_log.h"

bool mtk_at_send_ok(int fd) {
    char buff[] = "\r\nOK\r\n";
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_error(int fd) {
    char buff[] = "\r\nERROR\r\n";
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_nmea_notify(int fd, const char* nmea) {
    char buff[512] = {0};
    int len = strlen(nmea);
    if(len > 500) {
        LOGE("nmea len=[%d] > 500", len);
        return false;
    }
    mtk_at_gen_gnss_nmea_notify(buff, sizeof(buff), nmea);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_nmea_end_notify(int fd) {
    char buff[64] = {0};
    mtk_at_gen_gnss_nmea_end_notify(buff, sizeof(buff));
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_satellite_notify(int fd, mtk_gnss_satellite_list* list) {
    char buff[3000] = {0};
    if(list->num > 64) {
        LOGE("list num=[%d] > 64", list->num);
        return false;
    }
    mtk_at_gen_gnss_satellite_notify(buff, sizeof(buff), list);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_location_notify(int fd, mtk_gnss_location* location) {
    char buff[256] = {0};
    mtk_at_gen_gnss_location_notify(buff, sizeof(buff), location);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_agnss_location_notify(int fd, mtk_agnss_location* location) {
    char buff[256] = {0};
    mtk_at_gen_agnss_location_notify(buff, sizeof(buff), location);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}


bool mtk_at_send_gnss_device_reset_notify(int fd, const char* reason) {
    char buff[128] = {0};
    mtk_at_gen_gnss_device_reset_notify(buff, sizeof(buff), reason);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_status_response(int fd, mtk_gnss_status* status) {
    char buff[256] = {0};
    mtk_at_gen_gnss_status_response(buff, sizeof(buff), status);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_ni_notify_request(int fd, mtk_gnss_ni_notify* notify) {
    char buff[3072] = {0};
    mtk_at_gen_gnss_ni_notify_request(buff, sizeof(buff), notify);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_ref_location_request(int fd) {
    char buff[128] = {0};
    mtk_at_gen_gnss_ref_location_request(buff, sizeof(buff));
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_ref_time_request(int fd) {
    char buff[128] = {0};
    mtk_at_gen_gnss_ref_time_request(buff, sizeof(buff));
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_loopback_response(int fd, const char* msg) {
    char buff[5000] = {0};
    mtk_at_gen_gnss_loopback_response(buff, sizeof(buff), msg);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_gnss_cert_name_response(int fd, mtk_gnss_cert_name_list* list) {
    char buff[3000] = {0};
    mtk_at_gen_gnss_cert_name_response(buff, sizeof(buff), list);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_geo_max_num_response(int fd, int num) {
    char buff[64] = {0};
    mtk_at_gen_geo_max_num_response(buff, sizeof(buff), num);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_geo_add_circle_response(int fd, mtk_geo_fence_create_state state, int id) {
    char buff[64] = {0};
    if(state > MTK_GEO_FENCE_CREATE_STATE_SUCCESS) {
        LOGE("unexpected state=[%d] (<= 0)", state);
    }
    mtk_at_gen_geo_add_circle_response(buff, sizeof(buff), state, id);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_geo_alert_notify(int fd, mtk_geo_alert_notify* notify) {
    char buff[256] = {0};
    if(notify->state < MTK_GEO_STATE_UNKNOWN || notify->state > MTK_GEO_STATE_EXITED) {
        LOGE("unexpected state=[%d] (0~2)", notify->state);
        return false;
    }
    mtk_at_gen_geo_alert_notify(buff, sizeof(buff), notify);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_at_send_geo_track_notify(int fd, mtk_geo_track_state state, const char* date_time) {
    char buff[128] = {0};
    if(state != MTK_GEO_TRACK_STATE_OK && state != MTK_GEO_TRACK_STATE_FAILURE) {
        LOGE("unexpected state=[%d] (0, -1)", state);
        return false;
    }
    int len = strlen(date_time);
    if(len > 120) {
        LOGE("unexpected date_time len=[%d] (< 120)", len);
        return false;
    }
    mtk_at_gen_geo_track_notify(buff, sizeof(buff), state, date_time);
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

static bool mtk_at_gnss_start_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_gnss_start_request) {
        return callbacks->_mtk_at_gnss_start_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_stop_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_gnss_stop_request) {
        return callbacks->_mtk_at_gnss_stop_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_host_reset_notify_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_gnss_host_reset_notify) {
        return callbacks->_mtk_at_gnss_host_reset_notify(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_status_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_gnss_status_request) {
        return callbacks->_mtk_at_gnss_status_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_enable_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    bool enabled = 0;
    mtk_at_parse_bool(&at_cmd, &enabled);
    if(callbacks->_mtk_at_gnss_enable_set) {
        return callbacks->_mtk_at_gnss_enable_set(fd, enabled);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_ni_enable_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    bool enabled = 0;
    mtk_at_parse_bool(&at_cmd, &enabled);
    if(callbacks->_mtk_at_gnss_ni_enable_set) {
        return callbacks->_mtk_at_gnss_ni_enable_set(fd, enabled);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_agps_mode_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    mtk_gnss_agps_mode mode;
    memset(&mode, 0, sizeof(mode));
    mode.msa_valid = mtk_at_parse_bool(&at_cmd, &mode.msa);
    mode.msb_valid = mtk_at_parse_bool(&at_cmd, &mode.msb);
    mode.mss_valid = mtk_at_parse_bool(&at_cmd, &mode.mss);
    mode.cid_valid = mtk_at_parse_bool(&at_cmd, &mode.cid);
    mode.aflt_valid = mtk_at_parse_bool(&at_cmd, &mode.aflt);
    mode.otdoa_valid = mtk_at_parse_bool(&at_cmd, &mode.otdoa);
    mode.supl_pref_method_valid = mtk_at_parse_int32(&at_cmd, &mode.supl_pref_method);
    mode.supl_valid = mtk_at_parse_bool(&at_cmd, &mode.supl);
    mode.epo_valid = mtk_at_parse_bool(&at_cmd, &mode.epo);
    if(mode.supl_pref_method < MTK_GNSS_SUPL_PREF_METHOD_MSA || mode.supl_pref_method > MTK_GNSS_SUPL_PREF_METHOD_NO_PREF) {
        LOGE("unexpected supl_pref_method=[%d] (0~2)", mode.supl_pref_method);
        return false;
    }
    if(callbacks->_mtk_at_gnss_agps_mode_set) {
        return callbacks->_mtk_at_gnss_agps_mode_set(fd, &mode);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_supl_version_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    int maj = 0;
    int min = 0;
    int ser_ind = 0;
    mtk_at_parse_int32(&at_cmd, &maj);
    if(maj < 0) {
        LOGE("unexpected SUPL major version=[%d] (> 0)", maj);
        return false;
    }
    mtk_at_parse_int32(&at_cmd, &min);
    if(min < 0) {
        LOGE("unexpected SUPL minor version=[%d] (> 0)", min);
        return false;
    }
    mtk_at_parse_int32(&at_cmd, &ser_ind);
    if(ser_ind < 0) {
        LOGE("unexpected SUPL minor version=[%d] (> 0)", ser_ind);
        return false;
    }
    if(callbacks->_mtk_at_gnss_supl_version_set) {
        return callbacks->_mtk_at_gnss_supl_version_set(fd, maj, min, ser_ind);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_supl_addr_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    char addr[128] = {0};
    int port = 0;
    bool tls_enabled = 0;
    if(!mtk_at_parse_string(&at_cmd, addr, sizeof(addr))) {
        LOGE("parse address failed");
        return false;
    }
    mtk_at_parse_int32(&at_cmd, &port);
    if(port < 0 || port > 65535) {
        LOGE("unexpected port=[%d] (0~65535)", port);
        return false;
    }
    mtk_at_parse_bool(&at_cmd, &tls_enabled);
    if(callbacks->_mtk_at_gnss_supl_addr_set) {
        return callbacks->_mtk_at_gnss_supl_addr_set(fd, addr, port, tls_enabled);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_delete_aiding_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    mtk_gnss_aiding_data flags;
    memset(&flags, 0, sizeof(flags));

    if(mtk_at_parse_comma_num(at_cmd) > 0) {
        mtk_at_parse_bool(&at_cmd, &flags.eph);
        mtk_at_parse_bool(&at_cmd, &flags.alm);
        mtk_at_parse_bool(&at_cmd, &flags.pos);
        mtk_at_parse_bool(&at_cmd, &flags.time);
        mtk_at_parse_bool(&at_cmd, &flags.iono);
        mtk_at_parse_bool(&at_cmd, &flags.utc);
        mtk_at_parse_bool(&at_cmd, &flags.svdir);
        mtk_at_parse_bool(&at_cmd, &flags.rti);
        mtk_at_parse_bool(&at_cmd, &flags.celldb);
    } else {
        flags.all = true;
    }

    if(callbacks->_mtk_at_gnss_delete_aiding_data_request) {
        return callbacks->_mtk_at_gnss_delete_aiding_data_request(fd, &flags);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_ni_notify_response_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    int id = 0;
    mtk_gnss_ni_response_type type = MTK_GNSS_NI_RESPONSE_ACCEPT;
    mtk_at_parse_int32(&at_cmd, &id);
    mtk_at_parse_int32(&at_cmd, &type);
    if(type < MTK_GNSS_NI_RESPONSE_ACCEPT || type > MTK_GNSS_NI_RESPONSE_NORESP) {
        LOGE("unexpected type=[%d] (1~3)", type);
        return false;
    }
    if(callbacks->_mtk_at_gnss_ni_notify_response) {
        return callbacks->_mtk_at_gnss_ni_notify_response(fd, id, type);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_ref_location_response_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    int age = 0;
    double lat = 0;
    double lng = 0;
    float acc = 0;
    mtk_at_parse_int32(&at_cmd, &age);
    if(age < 0) {
        LOGE("unexpected age=[%d] (>0)", age);
        return false;
    }
    mtk_at_parse_double(&at_cmd, &lat);
    mtk_at_parse_double(&at_cmd, &lng);
    mtk_at_parse_float(&at_cmd, &acc);
    if(acc < 0) {
        LOGE(" unexpected acc=[%f] (>0)", acc);
        return false;
    }
    if(callbacks->_mtk_at_gnss_ref_location_response) {
        return callbacks->_mtk_at_gnss_ref_location_response(fd, age, lat, lng, acc);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_ref_time_response_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    long long time = 0;
    int uncertainty = 0;
    mtk_at_parse_int64(&at_cmd, &time);
    if(time < 0L) {
        LOGE("unexpected time=[%lld] (> 0)", time);
        return false;
    }
    mtk_at_parse_int32(&at_cmd, &uncertainty);
    if(uncertainty < 0) {
        LOGE("unexpected uncertainty=[%d] (> 0)", uncertainty);
        return false;
    }
    if(callbacks->_mtk_at_gnss_ref_time_response) {
        return callbacks->_mtk_at_gnss_ref_time_response(fd, time, uncertainty);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_nmea_config_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    bool enabled = 0;
    mtk_at_parse_bool(&at_cmd, &enabled);
    if(callbacks->_mtk_at_gnss_nmea_config_set) {
        return callbacks->_mtk_at_gnss_nmea_config_set(fd, enabled);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_ttff_qop_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    int mode = 0;
    mtk_at_parse_int32(&at_cmd, &mode);
    if(callbacks->_mkt_at_gnss_ttff_qop_set) {
        return callbacks->_mkt_at_gnss_ttff_qop_set(fd, mode);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_loopback_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    char msg[5000] = {0};
    if(!mtk_at_parse_string(&at_cmd, msg, sizeof(msg))) {
        LOGE("parse msg fail");
        return false;
    }
    if(callbacks->_mtk_at_gnss_loopback_request) {
        return callbacks->_mtk_at_gnss_loopback_request(fd, msg);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_cert_set_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    mtk_gnss_cert cert;
    memset(&cert, 0, sizeof(cert));

    if(!mtk_at_parse_int32(&at_cmd, &cert.total_msg_len)) {
        LOGE("parse total_msg_len failed");
        return false;
    }
    if(!mtk_at_parse_int32(&at_cmd, &cert.seq_no)) {
        LOGE("parse seq_no failed");
        return false;
    }
    if(!mtk_at_parse_string(&at_cmd, cert.name, sizeof(cert.name))) {
        LOGE("parse name failed");
        return false;
    }
    if(!mtk_at_parse_string(&at_cmd, cert.data, sizeof(cert.data))) {
        LOGE("parse data failed");
        return false;
    }
    if(callbacks->_mtk_at_gnss_cert_set) {
        return callbacks->_mtk_at_gnss_cert_set(fd, &cert);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_cert_name_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_gnss_cert_name_request) {
        return callbacks->_mtk_at_gnss_cert_name_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_cert_delete_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    char name[129] = {0};

    if(!mtk_at_parse_string(&at_cmd, name, sizeof(name))) {
        LOGE("mtk_at_gnss_cert_delete_request_parser() parse name failed");
        return false;
    }
    if(callbacks->_mtk_at_gnss_cert_delete_request) {
        return callbacks->_mtk_at_gnss_cert_delete_request(fd, name);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_gnss_cert_delete_all_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_gnss_cert_delete_all_request) {
        return callbacks->_mtk_at_gnss_cert_delete_all_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_geo_max_num_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_geo_max_num_request) {
        return callbacks->_mtk_at_geo_max_num_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_geo_add_circle_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    mtk_geo_add_circle c;
    memset(&c, 0, sizeof(c));
    mtk_at_parse_int32(&at_cmd, &c.alert_type);
    if(c.alert_type < 1 || c.alert_type > 3) {
        LOGE("unexpected alert_type=[%d] (1~3)", c.alert_type);
        return false;
    }
    mtk_at_parse_int32(&at_cmd, &c.initial_state);
    if(c.initial_state < 0 || c.initial_state > 2) {
        LOGE("unexpected initial_state=[%d] (0~2)", c.initial_state);
        return false;
    }
    mtk_at_parse_double(&at_cmd, &c.lat);
    mtk_at_parse_double(&at_cmd, &c.lng);
    mtk_at_parse_double(&at_cmd, &c.radius);
    if(c.radius < 0) {
        LOGE("unexpected radius=[%f] (> 0)", c.radius);
        return false;
    }
    mtk_at_parse_int32(&at_cmd, &c.unknowntimerms);
    if(c.unknowntimerms < 0) {
        LOGE("unexpected unknowntimerms=[%d] (> 0)", c.unknowntimerms);
        return false;
    }
    if(callbacks->_mtk_at_geo_add_circle_request) {
        return callbacks->_mtk_at_geo_add_circle_request(fd, &c);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_geo_delete_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    int id = 0;
    mtk_at_parse_int32(&at_cmd, &id);
    if(callbacks->_mtk_at_geo_delete_request) {
        return callbacks->_mtk_at_geo_delete_request(fd, id);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_geo_delete_all_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    if(callbacks->_mtk_at_geo_delete_all_request) {
        return callbacks->_mtk_at_geo_delete_all_request(fd);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

static bool mtk_at_test_request_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    int num1 = 0, num2 = 0;
    double d1 = 0, d2 = 0;
    char str1[1024] = {0};
    char str2[1024] = {0};
    mtk_at_parse_int32(&at_cmd, &num1);
    mtk_at_parse_int32(&at_cmd, &num2);
    mtk_at_parse_double(&at_cmd, &d1);
    mtk_at_parse_double(&at_cmd, &d2);
    mtk_at_parse_string(&at_cmd, str1, sizeof(str1));
    mtk_at_parse_string(&at_cmd, str2, sizeof(str2));
    if(callbacks->_mtk_at_test_request) {
        return callbacks->_mtk_at_test_request(fd, num1, num2, d1, d2, str1, str2);
    } else {
        LOGE("callbacks is NULL");
    }
    return false;
}

bool mtk_at_server_parser(int fd, const char* at_cmd, mtk_gnss_at_server_callbacks* callbacks) {
    assert(at_cmd != NULL);
    assert(callbacks != NULL);
    const char* at_cmd_bkup = at_cmd;

    //LOGD("mtk_at_server_parser() fd=[%d] at_cmd=[%s]", fd, at_cmd);

    bool ret = true;
    if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_START_REQ)) {
        ret = mtk_at_gnss_start_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_STOP_REQ)) {
        ret = mtk_at_gnss_stop_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_HOST_RESET_NTF)) {
        ret = mtk_at_gnss_host_reset_notify_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_STATUS_REQ)) {
        ret = mtk_at_gnss_status_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_GNSS_ENABLE_SET)) {
        ret = mtk_at_gnss_enable_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_NI_ENABLE_SET)) {
        ret = mtk_at_gnss_ni_enable_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_AGPS_MODE_SET)) {
        ret = mtk_at_gnss_agps_mode_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_SUPL_VERSION_SET)) {
        ret = mtk_at_gnss_supl_version_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_SUPL_ADDRESS_SET)) {
        ret = mtk_at_gnss_supl_addr_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_DELETE_AIDING_REQ)) {
        ret = mtk_at_gnss_delete_aiding_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_NI_NOTIFY_RSP)) {
        ret = mtk_at_gnss_ni_notify_response_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_REF_LOC_RSP)) {
        ret = mtk_at_gnss_ref_location_response_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_REF_TIME_RSP)) {
        ret = mtk_at_gnss_ref_time_response_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_NMEA_CFG_SET)) {
        ret = mtk_at_gnss_nmea_config_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_TTFF_QOP_SET)) {
        ret = mtk_at_gnss_ttff_qop_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_LOOPBACK_REQ)) {
        ret = mtk_at_gnss_loopback_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_CERT_SET)) {
        ret = mtk_at_gnss_cert_set_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_CERT_NAME_REQ)) {
        ret = mtk_at_gnss_cert_name_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_CERT_DEL_REQ)) {
        ret = mtk_at_gnss_cert_delete_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GNSS_CERT_DEL_ALL_REQ)) {
        ret = mtk_at_gnss_cert_delete_all_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GEOFENCE_MAX_NUM_REQ)) {
        ret = mtk_at_geo_max_num_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GEOFENCE_ADD_CIRCLE_REQ)) {
        ret = mtk_at_geo_add_circle_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GEOFENCE_DEL_REQ)) {
        ret = mtk_at_geo_delete_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_GEOFENCE_DEL_ALL_REQ)) {
        ret = mtk_at_geo_delete_all_request_parser(fd, at_cmd, callbacks);
    } else if(mtk_at_parse_cmd(&at_cmd, MTK_AT_TEST_REQ)) {
        ret = mtk_at_test_request_parser(fd, at_cmd, callbacks);
    } else {
        LOGW("unknown at_cmd=[%s]", at_cmd_bkup);
        return false;
    }

    if(!ret) {
        LOGW("wrong at_cmd=[%s]", at_cmd_bkup);
    }

    return ret;
}

