#include <string.h> //strstr
#include <dirent.h>
#include <sys/stat.h> //mkdir
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <math.h>
#include <stddef.h>  // offsetof
#include <stdarg.h>
#include <unistd.h>  // usleep
#include <sys/socket.h>
#include <fcntl.h>
#include <arpa/inet.h>  // inet_addr
#include <sys/un.h>  // struct sockaddr_un
#include <pthread.h>
#include <sys/epoll.h>
#include <signal.h>
#include <semaphore.h>
#include <signal.h> //struct sigevent
#include <string.h> //memset, strncpy
#include <netdb.h> //struct addrinfo
#include <sys/types.h> //gettid
#include <netinet/in.h> //struct sockaddr_in
#include <netinet/tcp.h> //TCP_KEEPIDLE, TCP_KEEPINTVL, TCP_KEEPCNT
#include <netinet/ip.h> //IPTOS_LOWDELAY
#include <sys/epoll.h> //epoll_create
#include <semaphore.h> //sem_t
#include <inttypes.h> //PRId64
#include <stdbool.h>
#include "geofence.h"
#include "mtk_mnld_log.h"
#include "mtk_geofence_main.h"
#include "mtk_geofence_controller.h"
#include "mtk_lbs_utility.h"

//----------------------------------------------------------------------------------
static char geofence_raw_data[GEOFENCE_DATAPATH_SIZE] = {0};
static cyclical_buffer_t g_cyclical_buffer;    // cyclic buffer
static mtk_geofence_ctx geofence_ctx; /*used to keep every geofence property*/
static int tracking_stat;/*GNSS tracking status, 0:tracking OK, -1:tracking failure*/
static int continue_trk_points;/*continue tracking points*/
#ifdef MTK_GNSS_GEOFENCE_UT
timer_t g_geofence_start_timer;
#endif
//static mtk_geofence_client_capability geofence_client_cap;
static mtk_geofence_server_capability geofence_server_cap;

extern client_list g_geofence_client_list;
extern client_list g_geofence_client_control_list;

static int state_lookup_table[3][3] =
{
    {MTK_GEOFENCE_TRANSITION_UNCERTAIN, MTK_GEOFENCE_TRANSITION_ENTERED, MTK_GEOFENCE_TRANSITION_EXITED},
    {MTK_GEOFENCE_TRANSITION_UNCERTAIN, MTK_GEOFENCE_TRANSITION_NO_CHANGE, MTK_GEOFENCE_TRANSITION_EXITED},
    {MTK_GEOFENCE_TRANSITION_UNCERTAIN, MTK_GEOFENCE_TRANSITION_ENTERED, MTK_GEOFENCE_TRANSITION_NO_CHANGE}
};

int mtk_geofence2mnl_geofence_alert(int fd, int fence_id, int alet_stat, mtk_geofence_position_info* pos_data);

#ifdef MTK_GNSS_GEOFENCE_UT
static void geofence_timer_fence_alert(int id)
{
    int i, j;
    int fd;
    int fd_pool[MAX_GOEFENCE] = {0};
    LOGD("Timer timeout");
    for (i = 0; i < MAX_GOEFENCE; i++)
    {
        if ((fd = geofence_ctx.geofence_property[i].fd) > 0){
            mtk_geofence2mnl_geofence_alert(geofence_ctx.geofence_property[i].fd, (i+1), GEOFENCE_ENTERED, NULL);
            for (j = 0; j < MAX_GOEFENCE; j++){
                if (fd_pool[j] == fd){
                    break;
                } else if (fd_pool[j] == 0){
                    fd_pool[j] = fd;
                    mtk_geofence2mnl_geofence_tracking_status(fd, 0, NULL);                    
                    break;
                }
            }
        }
    }
    start_timer(g_geofence_start_timer, 10000);
}
#endif
static void buffer_initialize
                ( cyclical_buffer_t *buffer,           // buffer to initialize
                  unsigned int buffer_size )     // size of buffer to create
{
   // Set up buffer manipulation pointers
   // end_buffer points to the next byte after the buffer
   buffer->end_buffer   = buffer->start_buffer + buffer_size;
   buffer->next_read    = buffer->start_buffer;
   buffer->next_write   = buffer->start_buffer;
   buffer->buffer_size = buffer_size;
   return;
}

//*****************************************************************************
//
// Horiz_Diff_Sqd_ofl : Compute the Horizontal position Difference Squared [m^2]
// between a pair of Latitude and Longitudes [rads].
//
// Ideally the cosine Latitude input should be for the mean Latitude of the
// two input positions. However, in practice it will most likely be for
// one of them (either the first computed or the best determined) as only
// a crude estimate of "Horiz_Diff_Sqd" is required for gross error checks.
//
//*****************************************************************************
static float Horiz_Diff_Sqd_ofl(
    const double LLpos1[], // i - 1st position Latitude, Longitude [radians]
    const double LLpos2[], // i - 2nd position Latitude, Longitude [radians]
    const float cosLat ) // i - cosine of Latitude
{
    // Return argument definition.
    float Horiz_Diff_Sqd; // Horizontal position Difference Squared [m^2]
    // Local data definitions
    float dN; // Delta Northing [m]
    float dE; // Delta Easting [m]

    // Compute the Northing difference from the Latitudes [m]
    dN = (float)( LLpos1[0] - LLpos2[0] ) * (float)WGS84_MAJA;
    // Compute the Easting difference from the Longitudes [m]
    dE = (float)( LLpos1[1] - LLpos2[1] ) * (float)WGS84_MAJA * cosLat;
    // Horizontal position Difference Squared [m^2]
    Horiz_Diff_Sqd = dN*dN + dE*dE;
    return( Horiz_Diff_Sqd );
}

static double calculate_two_circle_intersection_area(double distance,double radius1,double radius2)
{
    double r1 = radius1,r2 = radius2;

    if(r1 > r2){
        double temp = r1;

        r1 = r2;
        r2 = temp;
    }

    if(r1+r2 <= distance) {
        return 0;
    }
    else if(r2-r1>=distance) {
        return PI*r1*r1;
    }
    else {
        double a1=acos((r1*r1+distance*distance-r2*r2)/(2.0*r1*distance));
        double a2=acos((r2*r2+distance*distance-r1*r1)/(2.0*r2*distance));

        return (a1*r1*r1+a2*r2*r2-r1*distance*sin(a1));
    }
}

/* Not used
static double get_fence_distance(const double latitude,const double longitude,
    const gpslocation set_position){
    float horiz_diff_sqd,cosLat;
    double distance;
    const double llpos1[2]={latitude*PI/180 ,longitude*PI/180};
    const double llpos2[2]={set_position.latitude*PI/180,set_position.longitude*PI/180};

    cosLat = cos(latitude*PI/180);
    horiz_diff_sqd = Horiz_Diff_Sqd_ofl(llpos1,llpos2,cosLat);
    distance = sqrt(horiz_diff_sqd);
    return distance;
}
*/

static mtk_geofence_status fence_state(const double latitude,const double longitude,
    const double radius_meters, const mtk_geofence_position_info set_position) {
    float horiz_diff_sqd,cosLat;
    double distance,insec_area,circle_area;
    const double llpos1[2]={latitude*PI/180 ,longitude*PI/180};
    const double llpos2[2]={set_position.latitude*PI/180,set_position.longitude*PI/180};
    double hacc_2sigma;

    if(set_position.fix_type != MTK_GPS_FIX_TYPE_3D)// 3: means 3D fix. 2: means 2D fix. 1:means no fix
    {
        //LOGD("not 3D fix, it's uncertain\n");
        return GEOFENCE_UNKNOWN;
    }

    hacc_2sigma = (1.7*set_position.h_acc);//convert accuracy from 67% to 95%
    cosLat = cos(latitude*PI/180);
    horiz_diff_sqd = Horiz_Diff_Sqd_ofl(llpos1,llpos2,cosLat);
    distance = sqrt(horiz_diff_sqd);

    //LOGD("fence ll: %lf, %lf, loc ll: %lf %lf",latitude,longitude,set_position.latitude,set_position.longitude);
    //LOGD("distance =%lf, radius_meters=%lf,accuracy=%f\n",distance,radius_meters,set_position.h_acc);
    insec_area = calculate_two_circle_intersection_area(distance,radius_meters,hacc_2sigma);

    //restrict accuracy to make sure both low accuracy & no fix case can enter "uncertain" state
    circle_area = PI * (hacc_2sigma) * (hacc_2sigma);
    LOGD("insec_are:%lf, circle area: %lf",insec_area,circle_area);

    if(insec_area > ( circle_area* 0.95)) //intersection area >95% of the fence area or intersection area >95% of the location accuracy area
    {
        if( circle_area >= (PI * radius_meters * radius_meters)) //location accuracy wrapped up the fence
        {
            //LOGD("big accuracy, it's uncertain\n");
            return GEOFENCE_UNKNOWN;
        }
        else
        {
            //LOGD("circle_area <= insec_area, it's inside\n");
            return GEOFENCE_ENTERED;
        }
    }
    else if(insec_area < ( circle_area* 0.05))//intersection area <95% of the fence area
    {
        //LOGD("circle_area >  insec_area, it's outside\n");
        return GEOFENCE_EXITED;
    }
    else
    {
        return GEOFENCE_UNKNOWN;
    }
}

static int mtk_geofence_get_avalibale_item(){
    int i = 0;

    while(i < MAX_GOEFENCE && geofence_ctx.geofence_property[i].is_used == true){
        i++;
    }

    if(i == MAX_GOEFENCE){
        return -1;
    }

    return i;
}

static bool mtk_geofence_check_avalibale_item(int fd, int item){
    if (item >= MAX_GOEFENCE || item < 0){
        return false;
    }

    if (geofence_ctx.geofence_property[item].is_used && (geofence_ctx.geofence_property[item].fd == fd)){
        return true;
    }

    return false;
}

static int mtk_geofence_check_fence_vadility(mtk_geofence_property *fence){
    return 0;
}

static bool mtk_geofence_check_data_fd_exist(int fd){
    return client_list_contains(&g_geofence_client_list, fd);
}

static int mtk_geofence_set_one_geofence_unknown_alarm(int idx){
    if(geofence_ctx.geofence_property[idx].is_used) {
        geofence_ctx.geofence_new_timer[idx].unknown_init_ttick = 0;
        geofence_ctx.geofence_new_timer[idx].unknown_elapsed_ttick = 0;
    }
    return 0;
}

static void mtk_geofence_tracking_status_changed_hdlr(int new_trk_stat, mtk_geofence_position_info *pos){
    int i, j;
    int fd_pool[MAX_GOEFENCE] = {0};
    int fd;

    for (i = 0; i < MAX_GOEFENCE; i++)
    {
        if ((fd = geofence_ctx.geofence_property[i].fd) > 0){
            for (j = 0; j < MAX_GOEFENCE; j++){
                if (fd_pool[j] == fd){
                    break;
                } else if (fd_pool[j] == 0){
                    fd_pool[j] = fd;
                    mtk_geofence2mnl_geofence_tracking_status(fd, new_trk_stat, pos);                    
                    break;
                }
            }
        }
    }
}

static void mtk_geofence_check_tracking_status(mtk_geofence_position_info *pos){
    int new_trk_stat;

    if (pos->fix_type == MTK_GPS_FIX_TYPE_3D){
        new_trk_stat = 0;
    } else {
        new_trk_stat = -1;
    }

    if (new_trk_stat != tracking_stat){
        continue_trk_points++;
        if (continue_trk_points >= TRACKING_STATUS_CHANGE_THREASHOLD){
            //Notify user tracking status changed
            tracking_stat = new_trk_stat;
            continue_trk_points = 0;
            mtk_geofence_tracking_status_changed_hdlr(new_trk_stat, pos);
        }
    } else {
        continue_trk_points = 0;
    }
}

static int mtk_geofence_add_new_fence(int fd, mtk_geofence_property * fence){
    int item;
    int fence_id;
    mtk_geofence_property_db fence_data;

    item = mtk_geofence_get_avalibale_item();

    if(item < 0)
    {
        return -1;
    }

    fence_data.latitude = fence->latitude;
    fence_data.longitude = fence->longitude;
    fence_data.radius = fence->radius;
    fence_data.latest_state = fence->latest_state;
    fence_data.monitor_transition = fence->monitor_transition;
    fence_data.unknown_timer = fence->unknown_timer;
    if (fence->latest_state == GEOFENCE_UNKNOWN){
        fence_data.last_transition = MTK_GEOFENCE_TRANSITION_UNCERTAIN;
    } else if (fence->latest_state == GEOFENCE_ENTERED) {
        fence_data.last_transition = MTK_GEOFENCE_TRANSITION_ENTERED;
    } else if (fence->latest_state == GEOFENCE_EXITED) {
        fence_data.last_transition = MTK_GEOFENCE_TRANSITION_EXITED;
    } else {
        fence_data.last_transition = MTK_GEOFENCE_TRANSITION_UNCERTAIN;
    }
    
    memcpy(&geofence_ctx.geofence_property[item].geofence_prop, &fence_data, sizeof(mtk_geofence_property_db));
    geofence_ctx.geofence_property[item].is_used = true;
    geofence_ctx.geofence_property[item].fd = fd;
    geofence_ctx.fence_number++;
    //set unknown timer
    mtk_geofence_set_one_geofence_unknown_alarm(item);

    fence_id = item + 1;
    return fence_id;
}

static int mtk_geofence_remove_fence(int fd, int fence_id) {
    int item;

    item = fence_id - 1;
    if (!mtk_geofence_check_avalibale_item(fd, item)){
        return -1;
    }

    memset(&geofence_ctx.geofence_property[item].geofence_prop, 0, sizeof(mtk_geofence_property_db));
    geofence_ctx.geofence_property[item].is_used = false;
    geofence_ctx.geofence_property[item].fd = -1;
    geofence_ctx.fence_number--;

    return 0;
}

void mtk_geofence_clear_fence_db(int fd) {
    int item;

    for (item = 0; item < MAX_GOEFENCE; item++) {
        if (geofence_ctx.geofence_property[item].is_used && (geofence_ctx.geofence_property[item].fd == fd)) {
            memset(&geofence_ctx.geofence_property[item].geofence_prop, 0, sizeof(mtk_geofence_property_db));
            geofence_ctx.geofence_property[item].is_used = false;
            geofence_ctx.geofence_property[item].fd = -1;
            geofence_ctx.fence_number--;
        }
    }
}

int mtk_geofence_clear_geofences_by_msg(int fd) {
    mtk_geofence_msg *prmsg=NULL;
    unsigned int msg_len;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    LOGD("geofence_clear_geofences,fd:%d\r\n", fd);
    //construct remove fence cmd
    msg_len = sizeof(mtk_geofence_msg);
    prmsg = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    prmsg->type = CLEAR_GEOFENCE;
    prmsg->length = msg_len;
    mtk_geofence_clear_fence_db(fd);
    mtk_geofence_controller_process(prmsg);
    return MTK_GFC_SUCCESS;

}

int mtk_geofence_add_rawdata_to_buffer(char *data, int length) {
    int i;

    for (i = 0; i < length; i++) {
        *(g_cyclical_buffer.next_write++) = data[i];
        if (g_cyclical_buffer.next_write == g_cyclical_buffer.end_buffer){
            g_cyclical_buffer.next_write = g_cyclical_buffer.start_buffer;
        }

        if (g_cyclical_buffer.next_write == g_cyclical_buffer.next_read){
            LOGE("geofence ring_buffer overflow");
            return -1;
        }
    }

    return 0;
}

int mtk_geofence_get_one_message(char *data, unsigned int len) {
    char *next_write, *next_read;
    unsigned int data_size, i, header_len;
    char buffer[GEOFENCE_BUFF_SIZE] = {0};
    unsigned int return_len = 0;
    mtk_geofence_msg geo_header;

    next_write = g_cyclical_buffer.next_write;
    next_read = g_cyclical_buffer.next_read;

    if (g_cyclical_buffer.next_read == next_write){
        //buffer empty
        return -1;
    }

    header_len = sizeof(mtk_geofence_msg);
    /*Compute data length*/
    if (g_cyclical_buffer.next_read < next_write){
        data_size = (unsigned long)next_write - (unsigned long)g_cyclical_buffer.next_read;
    }
    else{
        data_size = (unsigned long)g_cyclical_buffer.end_buffer - (unsigned long)g_cyclical_buffer.next_read +
                     (unsigned long)next_write - (unsigned long)g_cyclical_buffer.start_buffer;
    }

    /*Copy data header to buffer*/
    if (data_size >= header_len){
        for (i = 0; i < header_len; i++){
            buffer[i] = *(next_read++);
            if (next_read == g_cyclical_buffer.end_buffer){
                next_read = g_cyclical_buffer.start_buffer;
            }
        }

        memset(&geo_header, 0, sizeof(mtk_geofence_msg));
        memcpy(&geo_header, buffer, sizeof(mtk_geofence_msg));
        if (geo_header.length <= data_size){
            for (i = 0; (i < geo_header.length)&&(i < len); i++){
                data[i] = *(g_cyclical_buffer.next_read++);
                return_len++;
                if (g_cyclical_buffer.next_read == g_cyclical_buffer.end_buffer){
                    g_cyclical_buffer.next_read = g_cyclical_buffer.start_buffer;
                }
            }
        }
        else {
            //no enough data
            return -2;
        }
    }
    else{
        //no enough data
        return -2;
    }

    return return_len;

}

int create_geofence2mnl_data_fd() {
    int fd = socket_tcp_server_bind_local(true, GEOFENCEADP_MNL_TCP_DATA);
    memset(&g_geofence_client_list, 0, sizeof(g_geofence_client_list));
    return fd;
}

int create_geofence2mnl_control_fd() {
    int fd = socket_tcp_server_bind_local(true, GEOFENCEADP_MNL_TCP_CONTROL);
    memset(&g_geofence_client_control_list, 0, sizeof(g_geofence_client_control_list));
    return fd;
}

int mtk_geofence2mnl_get_valid_fence_num()
{
    return geofence_ctx.fence_number;
}

int mtk_geofence_send2adpator(int fd, const char* buff, int len) {
    int ret = write(fd, buff, len);
    if(ret == -1) {
        LOGE("geofence_send2adpator() write() failed, reason=[%s]%d", strerror(errno), errno);
    }
    return ret;
}

int mtk_geofence_server_capability_send(int fd) {
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    mtk_geofence_server_init_msg init_msg;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    LOGD("mtk_geofence_server_capability_send fd:%d %lu", fd, sizeof(mtk_geofence_server_init_msg));
    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(mtk_geofence_server_init_msg);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = GEOFENCE_SERVER_CAP;
    geo_header->length = msg_len;
    init_msg.geofence_support = geofence_server_cap.geofence_support;
    init_msg.server_data_fd = fd;

    memcpy( ((char*)geo_header) + sizeof(mtk_geofence_msg),&init_msg, sizeof(mtk_geofence_server_init_msg));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

void mtk_geofence_location_expire(mtk_geofence_position_info coordinate_loc)
{
    int i;
    mtk_geofence_status state;
    int transiton = MTK_GEOFENCE_TRANSITION_NO_CHANGE;
    LOGD("Fix_type:%d lat:%.6f lon:%.6f hacc:%.1f UTC:%02d%02d%02d", 
        coordinate_loc.fix_type, coordinate_loc.latitude, coordinate_loc.longitude, coordinate_loc.h_acc,
        coordinate_loc.utc_time.hour, coordinate_loc.utc_time.min, coordinate_loc.utc_time.sec);
    //Check tracking status
    mtk_geofence_check_tracking_status(&coordinate_loc);

    for ( i = 0; i < MAX_GOEFENCE; i++ )
    {
        if(geofence_ctx.geofence_property[i].is_used)
        {
            state = fence_state(geofence_ctx.geofence_property[i].geofence_prop.latitude,
                                geofence_ctx.geofence_property[i].geofence_prop.longitude,
                                geofence_ctx.geofence_property[i].geofence_prop.radius,
                                coordinate_loc);
            transiton = state_lookup_table[geofence_ctx.geofence_property[i].geofence_prop.latest_state][state];
            LOGD("fence_id:%d latest_state:%d state:%d transition:%d", 
                (i+1), geofence_ctx.geofence_property[i].geofence_prop.latest_state, state, transiton);
            geofence_ctx.geofence_property[i].geofence_prop.latest_state = state;
            if((geofence_ctx.geofence_property[i].geofence_prop.last_transition != transiton)
                && (transiton != MTK_GEOFENCE_TRANSITION_NO_CHANGE)
                && (state != GEOFENCE_UNKNOWN))
            {
                if (geofence_ctx.geofence_property[i].geofence_prop.monitor_transition & transiton) {
                    mtk_geofence2mnl_geofence_alert(geofence_ctx.geofence_property[i].fd, (i+1), state, &coordinate_loc);
                    LOGD("fence id %d transition update from %x -> %x\n", (i + 1), geofence_ctx.geofence_property[i].geofence_prop.monitor_transition, transiton);
                } else {
                    LOGD("fence id %d transition update from %x -> %x bypass\n",(i + 1), geofence_ctx.geofence_property[i].geofence_prop.monitor_transition, transiton);
                }
                geofence_ctx.geofence_new_timer[i].unknown_init_ttick = 0; //reset unknown timer init ttick
                geofence_ctx.geofence_new_timer[i].unknown_elapsed_ttick =0; //reset unknown timer elapsed time
                geofence_ctx.geofence_property[i].geofence_prop.latest_state = state;/*update latest state*/
                geofence_ctx.geofence_property[i].geofence_prop.last_transition = transiton; /*update latest transition*/
            }
            else
            {
                if(state == GEOFENCE_UNKNOWN)
                {
                    //FLP_TRC("fence id %d,current state == uncertain",i);
                    if(geofence_ctx.geofence_new_timer[i].unknown_init_ttick == 0) //check if it is the first time entering uncertain state
                    {
                        //FLP_TRC("unknown ttick =0; elapsed = %d, init ttick=%d",geofence_new_timer[i].unknown_elapsed_ttick,geofence_new_timer[i].unknown_init_ttick );
                        if(geofence_ctx.geofence_new_timer[i].unknown_elapsed_ttick == 0 ) //check if already inform the state transition NTF to AP
                        {
                            geofence_ctx.geofence_new_timer[i].unknown_init_ttick = get_time_in_millisecond(); //record the initial time of unknown timer
                            //FLP_TRC("unknown init ttick ==%d",geofence_new_timer[i].unknown_init_ttick);
                        }
                        else
                        {
                            //FLP_TRC("no update");
                            //do nothing, the transition was dated to AP already
                        }
                    }
                    else  //update unknown timer elapsed time
                    {
                        geofence_ctx.geofence_new_timer[i].unknown_elapsed_ttick = get_time_in_millisecond() - geofence_ctx.geofence_new_timer[i].unknown_init_ttick; //update unknown counter
                        LOGD("unknown timer elapsed: %ld", geofence_ctx.geofence_new_timer[i].unknown_elapsed_ttick);
                        if( (geofence_ctx.geofence_new_timer[i].unknown_elapsed_ttick 
                            >= geofence_ctx.geofence_property[i].geofence_prop.unknown_timer)
                            && (geofence_ctx.geofence_property[i].geofence_prop.unknown_timer > 0)) //Check if unknown counter reach user defined value
                        {
                            //Notify transition state: from in/out to uncertain
                            mtk_geofence2mnl_geofence_alert(geofence_ctx.geofence_property[i].fd, (i+1), GEOFENCE_UNKNOWN, &coordinate_loc);
                            LOGD("fence id %d transition update from %x -> %x uncertain\n",(i+1), geofence_ctx.geofence_property[i].geofence_prop.last_transition, transiton);
                            geofence_ctx.geofence_new_timer[i].unknown_init_ttick = 0; //only reset init ttick, elapsed time will not reset to keep a record of the update of transition state
                            geofence_ctx.geofence_property[i].geofence_prop.latest_state = GEOFENCE_UNKNOWN;/*update latest state*/
                            geofence_ctx.geofence_property[i].geofence_prop.last_transition = MTK_GEOFENCE_TRANSITION_UNCERTAIN; /*update latest transition*/
                        }
                    }
                } else {
                    geofence_ctx.geofence_new_timer[i].unknown_init_ttick = 0; //reset unknown timer init ttick
                    geofence_ctx.geofence_new_timer[i].unknown_elapsed_ttick =0; //reset unknown timer elapsed time
                }
            }
        }
    }
}

int mtk_geofence2mnl_geofence_alert(int fd, int fence_id, int alet_stat, mtk_geofence_position_info* pos_data) 
{
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    mtk_geofence_alert fence_alert;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(mtk_geofence_alert);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = GEOFENCE_ALERT_INFO;
    geo_header->length = msg_len;
    if (pos_data == NULL){//For mock UT test if pos_data is NULL
        //+EGEORESP: 1,1,24.125465,121.365496,100.23,1.23,3.54,3,6,2,6,3,2.52,1.32,2.12\r\n
        fence_alert.id = fence_id;
        fence_alert.alert_state = 1;
        fence_alert.latitude = 24.125465;
        fence_alert.longitude = 121.365496;
        fence_alert.altitude = 100.23;
        fence_alert.speed = 1.23;
        fence_alert.heading = 3.54;
        fence_alert.h_acc = 3;
        fence_alert.h_err_majoraxis = 6;
        fence_alert.h_err_minoraxis = 2;
        fence_alert.h_err_angle = 6;
        fence_alert.hor_conf = 3;
        fence_alert.pdop = 2.52;
        fence_alert.hdop = 1.32;
        fence_alert.vdop = 2.12;
    } else {
        fence_alert.id = fence_id;
        fence_alert.alert_state = alet_stat;
        fence_alert.latitude = pos_data->latitude;
        fence_alert.longitude = pos_data->longitude;
        fence_alert.altitude = pos_data->altitude;
        fence_alert.speed = pos_data->speed;
        fence_alert.heading = pos_data->heading;
        fence_alert.h_acc = (int)pos_data->h_acc;
        fence_alert.h_err_majoraxis = (int)pos_data->h_err_majoraxis;
        fence_alert.h_err_minoraxis = (int)pos_data->h_err_minoraxis;
        fence_alert.h_err_angle = (int)pos_data->h_err_angle;
        fence_alert.hor_conf = (int)pos_data->h_acc;
        fence_alert.pdop = (int)pos_data->pdop;
        fence_alert.hdop = (int)pos_data->hdop;
        fence_alert.vdop = (int)pos_data->vdop;
    }
    memcpy(((char*)geo_header) + sizeof(mtk_geofence_msg),&fence_alert, sizeof(mtk_geofence_alert));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

int mtk_geofence2mnl_geofence_tracking_status(int fd, int tracking_stat, mtk_geofence_position_info *pos) {
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    mtk_gnss_tracking_status tracking_status;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(mtk_gnss_tracking_status);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = GNSS_TRACKING_STATUS;
    geo_header->length = msg_len;
    if (pos == NULL){//UT test case if pos == null
        tracking_status.status = tracking_stat;
        tracking_status.year =2020;
        tracking_status.month = 5;
        tracking_status.day = 10;
        tracking_status.hour = 21;
        tracking_status.minute = 6;
        tracking_status.second = 23;
    } else {
        tracking_status.status = tracking_stat;
        tracking_status.year = (int)(pos->utc_time.year + 1900);
        tracking_status.month = (int)(pos->utc_time.month + 1);
        tracking_status.day = (int)pos->utc_time.mday;
        tracking_status.hour = (int)pos->utc_time.hour;
        tracking_status.minute = (int)pos->utc_time.min;
        tracking_status.second = (int)pos->utc_time.sec;
    }
    LOGD("geofence tracking status change:stat:%d time:%d %d %d %d %d %d", 
        tracking_stat, tracking_status.year, tracking_status.month, tracking_status.day, tracking_status.hour, tracking_status.minute,  tracking_status.second);
    memcpy( ((char*)geo_header) + sizeof(mtk_geofence_msg),&tracking_status, sizeof(mtk_gnss_tracking_status));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

int mtk_geofence2mnl_add_geofence_result(int fd, int fence_id, int result) {
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    mtk_geofence_create_status fence_ret;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(mtk_geofence_create_status);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = GEOFENCE_RESPONSE_INFO;
    geo_header->length = msg_len;
    fence_ret.createstat = result;
    fence_ret.id = fence_id;
    memcpy( ((char*)geo_header) + sizeof(mtk_geofence_msg),&fence_ret, sizeof(mtk_geofence_create_status));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

int mtk_geofence2mnl_geofence_remove_result(int fd, int result) {
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    mtk_geofence_result rlt;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(mtk_geofence_result);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = EXCUTE_RESULT;
    geo_header->length = msg_len;
    rlt.cmdtype = REMOVE_GEOFENCE;
    rlt.result = result;
    memcpy( ((char*)geo_header) + sizeof(mtk_geofence_msg),&rlt, sizeof(mtk_geofence_result));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

int mtk_geofence2mnl_geofence_clear_result(int fd, int result) {
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    mtk_geofence_result rlt;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(mtk_geofence_result);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = EXCUTE_RESULT;
    geo_header->length = msg_len;
    rlt.cmdtype = CLEAR_GEOFENCE;
    rlt.result = result;
    memcpy( ((char*)geo_header) + sizeof(mtk_geofence_msg),&rlt, sizeof(mtk_geofence_result));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

int mtk_geofence2mnl_geofence_query_result(int fd, int result) {
    int ret;
    mtk_geofence_msg *geo_header=NULL;
    unsigned int msg_len;
    int fence_num;
    char geo_data[GEOFENCE_BUFF_SIZE] = {0};

    //construct add fence cmd
    msg_len = sizeof(mtk_geofence_msg) + sizeof(int);
    geo_header = (mtk_geofence_msg *)&geo_data[0];
    if (msg_len > GEOFENCE_BUFF_SIZE){
        LOGE("geofence main message length too long:%d", msg_len);
        return MTK_GFC_ERROR;
    }

    geo_header->type = GEOFENCE_NUM;
    geo_header->length = msg_len;
    if (result == 0){
        fence_num = MAX_GOEFENCE;
    } else {
        fence_num = -1;
    }
    memcpy( ((char*)geo_header) + sizeof(mtk_geofence_msg),&fence_num, sizeof(int));
    ret = mtk_geofence_send2adpator(fd, (char*)geo_header, msg_len);
    if(ret < 0) {
        LOGE("MTK mnl to adaptor send error return error");
        return MTK_GFC_ERROR;
    }
    return MTK_GFC_SUCCESS;
}

void mtk_geofence2mnl_geofence_add_area(int fd, mtk_geofence_msg *prmsg)
{
    mtk_geofence_property fence;
    unsigned int msg_len;
    char data[GEOFENCE_BUFF_SIZE] = {0};
    int ret;
    int fence_id;
    int server_data_id;

    msg_len = sizeof(mtk_geofence_msg) + sizeof(int) + sizeof(mtk_geofence_property);
    if (prmsg->length == msg_len){
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
        memcpy(&fence, (((char*)prmsg)+sizeof(mtk_geofence_msg)+sizeof(int)), sizeof(mtk_geofence_property));
    } else if (prmsg->length > msg_len) {
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
        memcpy(&fence, (((char*)prmsg)+sizeof(mtk_geofence_msg)+sizeof(int)), sizeof(mtk_geofence_property));
        LOGD("mnl msg len:%d > struct len:%d", prmsg->length, msg_len);
    } else {
        memcpy(data, (char*)prmsg, prmsg->length);
        memset((char *)(data + prmsg->length), 0, (msg_len - prmsg->length));
        memcpy(&server_data_id, (((char*)data)+sizeof(mtk_geofence_msg)), sizeof(int));
        memcpy(&fence, (((char*)data)+sizeof(mtk_geofence_msg)+sizeof(int)), sizeof(mtk_geofence_property));
        LOGD("mnl msg len:%d < struct len:%d", prmsg->length, msg_len);
    }

    if (!mtk_geofence_check_data_fd_exist(server_data_id)){
        mtk_geofence2mnl_add_geofence_result(fd, 0, MTK_ADD_GEOFENCE_ERROR);
        LOGE("mtk_geofence2mnl_geofence_add_area:server data fd don't exist %d", server_data_id);
        return;
    }

    ret = mtk_geofence_check_fence_vadility(&fence);
    if (ret == 0) {
        if ((fence_id = mtk_geofence_add_new_fence(server_data_id, &fence)) > 0){
            mtk_geofence2mnl_add_geofence_result(fd, fence_id, MTK_ADD_GEOFENCE_SUCCESS);
        } else if (fence_id == -1){
            mtk_geofence2mnl_add_geofence_result(fd, 0, MTK_ADD_GEOFENCE_TOO_MANY);
        }
    } else {
        mtk_geofence2mnl_add_geofence_result(fd, 0, MTK_ADD_GEOFENCE_ERROR);
    }

    LOGD("geofence_add_area,fd:%d lat:%.6f lon%.6f rad:%.2f InitStat:%d AlertType:%d UnknownTime:%d\r\n",
        fd,
        fence.latitude,
        fence.longitude,
        fence.radius,
        fence.latest_state,
        fence.monitor_transition,
        fence.unknown_timer);
}

void mtk_geofence2mnl_geofence_remove_fence(int fd, mtk_geofence_msg *prmsg)
{
    int id;
    unsigned int msg_len;
    char data[GEOFENCE_BUFF_SIZE] = {0};
    int server_data_id;

    msg_len = sizeof(mtk_geofence_msg)+ sizeof(int) + sizeof(int);
    if (prmsg->length == msg_len){
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
        memcpy(&id, (((char*)prmsg)+sizeof(mtk_geofence_msg)+ sizeof(int)), sizeof(int));
    } else if (prmsg->length > msg_len) {
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
        memcpy(&id, (((char*)prmsg)+sizeof(mtk_geofence_msg)+ sizeof(int)), sizeof(int));
        LOGD("mnl msg len:%d > struct len:%d", prmsg->length, msg_len);
    } else {
        memcpy(data, (char*)prmsg, prmsg->length);
        memset((char *)(data + prmsg->length), 0, (msg_len - prmsg->length));
        memcpy(&server_data_id, (((char*)data)+sizeof(mtk_geofence_msg)), sizeof(int));
        memcpy(&id, (((char*)data)+sizeof(mtk_geofence_msg)+ sizeof(int)), sizeof(int));
        LOGD("mnl msg len:%d < struct len:%d", prmsg->length, msg_len);
    }

    if (!mtk_geofence_check_data_fd_exist(server_data_id)){
        mtk_geofence2mnl_geofence_remove_result(fd, -1);
        LOGE("mtk_geofence2mnl_geofence_remove_fence:server data fd don't exist %d", server_data_id);
        return;
    }

    LOGD("geofence_remove_fence,fd:%d %d\r\n",fd, id);
    if (mtk_geofence_remove_fence(server_data_id, id) >= 0){
        mtk_geofence2mnl_geofence_remove_result(fd, 0);
    } else {
        mtk_geofence2mnl_geofence_remove_result(fd, -1);
    }
}

void mtk_geofence2mnl_geofence_clear_fence(int fd, mtk_geofence_msg *prmsg)
{
    int server_data_id;
    unsigned int msg_len;
    char data[GEOFENCE_BUFF_SIZE] = {0};

    msg_len = sizeof(mtk_geofence_msg) + sizeof(int);
    if (prmsg->length == msg_len){
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
    } else if (prmsg->length > msg_len) {
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
        LOGD("mnl msg len:%d > struct len:%d", prmsg->length, msg_len);
    } else {
        memcpy(data, (char*)prmsg, prmsg->length);
        memset((char *)(data + prmsg->length), 0, (msg_len - prmsg->length));
        memcpy(&server_data_id, (((char*)data)+sizeof(mtk_geofence_msg)), sizeof(int));
        LOGD("mnl msg len:%d < struct len:%d", prmsg->length, msg_len);
    }

    if (!mtk_geofence_check_data_fd_exist(server_data_id)){
        mtk_geofence2mnl_geofence_clear_result(fd, -1);
        LOGE("mtk_geofence2mnl_geofence_clear_fence:server data fd don't exist %d", server_data_id);
        return;
    }

    LOGD("geofence_clear_fence,fd:%d\r\n",fd);
    mtk_geofence_clear_fence_db(server_data_id);
    mtk_geofence2mnl_geofence_clear_result(fd, 0);
}

void mtk_geofence2mnl_geofence_query_num(int fd, mtk_geofence_msg *prmsg)
{
    int server_data_id;
    unsigned int msg_len;
    char data[GEOFENCE_BUFF_SIZE] = {0};

    LOGD("geofence_query_num,fd:%d\r\n",fd);
    msg_len = sizeof(mtk_geofence_msg) + sizeof(int);
    if (prmsg->length == msg_len){
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
    } else if (prmsg->length > msg_len) {
        memcpy(&server_data_id, (((char*)prmsg)+sizeof(mtk_geofence_msg)), sizeof(int));
        LOGD("mnl msg len:%d > struct len:%d", prmsg->length, msg_len);
    } else {
        memcpy(data, (char*)prmsg, prmsg->length);
        memset((char *)(data + prmsg->length), 0, (msg_len - prmsg->length));
        memcpy(&server_data_id, (((char*)data)+sizeof(mtk_geofence_msg)), sizeof(int));
        LOGD("mnl msg len:%d < struct len:%d", prmsg->length, msg_len);
    }

    if (!mtk_geofence_check_data_fd_exist(server_data_id)){
        mtk_geofence2mnl_geofence_query_result(fd, -1);
        LOGE("mtk_geofence2mnl_geofence_query_num:server data fd don't exist %d", server_data_id);
        return;
    }

    mtk_geofence2mnl_geofence_query_result(fd, 0);
}

int mtk_geofence2mnl_hdlr (int fd, char *buff, int length) {
    char data[GEOFENCE_BUFF_SIZE] = {0};
    int ret = MTK_GFC_ERROR, len;
    unsigned int msg_len;
    mtk_geofence_command cmd;
    mtk_geofence_msg *prmsg = NULL;

    if(buff == NULL) {
        LOGE("mnl2gfchal_hdlr, recv prmsg is null pointer\r\n");
        return MTK_GFC_ERROR;
    }

    if (mtk_geofence_add_rawdata_to_buffer(buff, length) < 0){
        //error handle
    }

    while ( (len = mtk_geofence_get_one_message(data, GEOFENCE_BUFF_SIZE)) >= 0)
    {
        if((len > 0) && (len <= GEOFENCE_BUFF_SIZE)) {
            prmsg = (mtk_geofence_msg *)&data[0];
        } else {
            LOGE("len err:%d",len);
            return ret;
        }
        cmd = prmsg->type;
        msg_len = prmsg->length;

        LOGD("cmd %d, len %d\r\n", cmd, msg_len);

        switch (cmd) {
            case ADD_GEOFENCE_AREA:
                mtk_geofence_controller_process(prmsg);
                mtk_geofence2mnl_geofence_add_area(fd, prmsg);
                break;
            case REMOVE_GEOFENCE:
                mtk_geofence2mnl_geofence_remove_fence(fd,prmsg);
                mtk_geofence_controller_process(prmsg);
                break;
            case CLEAR_GEOFENCE:
                mtk_geofence2mnl_geofence_clear_fence(fd, prmsg);
                mtk_geofence_controller_process(prmsg);
                break;
            case QUERY_GEOFENCE_NUM:
                mtk_geofence2mnl_geofence_query_num(fd, prmsg);
                break;
            case GEOFENCE_CLIENT_CAP:
                //currenttly do nothing
                break;
            default:
                LOGE("invalid geofence cmd:%d",cmd);
                break;
        }

        memset(data, 0, GEOFENCE_BUFF_SIZE);
        len = 0;
    }
    return MTK_GFC_SUCCESS;
}

void mtk_geofence_init(void)
{
    g_cyclical_buffer.start_buffer = &geofence_raw_data[0];
    buffer_initialize(&g_cyclical_buffer, GEOFENCE_DATAPATH_SIZE);
    tracking_stat = -1;
    continue_trk_points = 0;
    #ifdef MTK_GNSS_GEOFENCE_UT
    g_geofence_start_timer = init_timer(geofence_timer_fence_alert);
    start_timer(g_geofence_start_timer, 10000);
    #endif
}
