#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <stddef.h>  // offsetof
#include <stdarg.h>
#include <sys/stat.h>
#include <unistd.h>  // usleep
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <arpa/inet.h>  // inet_addr
#include <sys/un.h>  // struct sockaddr_un
#include <pthread.h>
#include <sys/epoll.h>
#include <signal.h>
#include <semaphore.h>
#include <sys/syscall.h>

#if defined(__ANDROID_OS__)
#include <cutils/log.h>     // Android log
#endif
#include "mtk_mnld_log.h"

#include "mtk_lbs_utility.h"
#include "data_coder.h"

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "mtk_lbs_utility"

int asc_str_to_usc2_str(char* output, const char* input) {
    int len = 2;

    output[0] = 0xfe;
    output[1] = 0xff;

    while (*input != 0) {
        output[len++] = 0;
        output[len++] = *input;
        input++;
    }

    output[len] = 0;
    output[len+1] = 0;
    return len;
}

void raw_data_to_hex_string(char* output, const char* input, int input_len) {
    int i = 0;
    for (i = 0; i < input_len; i++) {
        int tmp = (input[i] >> 4) & 0xf;

        if (tmp >= 0 && tmp <= 9) {
            output[i*2] = tmp + '0';
        } else {
            output[i*2] = (tmp - 10) + 'A';
        }

        tmp = input[i] & 0xf;
        if (tmp >= 0 && tmp <= 9) {
            output[(i*2)+1] = tmp + '0';
        } else {
            output[(i*2)+1] = (tmp - 10) + 'A';
        }
    }
    output[i*2] = 0;
}

void msleep(int interval) {
    usleep(interval * 1000);
}

// in millisecond
time_t get_tick() {
    struct timespec ts;
    if (clock_gettime(CLOCK_MONOTONIC, &ts) == -1) {
        LOGE("clock_gettime failed reason=[%s]", strerror(errno));
        return -1;
    }
    return (ts.tv_sec*1000) + (ts.tv_nsec/1000000);
}

// in millisecond
UINT64 get_time_in_millisecond() {
    struct timespec ts;
    if (clock_gettime(CLOCK_BOOTTIME, &ts) == -1) {
        LOGE("get_time_in_millisecond  failed reason=[%s]", strerror(errno));
        return 0;
    }
    return (UINT64)((UINT64)(ts.tv_sec*1000) + (UINT64)(ts.tv_nsec/1000000));
}

sem_t g_mnld_exit_sem;
extern bool mnld_exiting;

// -1 means failure
int block_here() {
    if (sem_init(&g_mnld_exit_sem, 0, 0) == -1) {
        LOGE("block_here() sem_init failure reason=%s\n", strerror(errno));
        return -1;
    }
    sem_wait(&g_mnld_exit_sem);
    if (sem_destroy(&g_mnld_exit_sem) == -1) {
        LOGE("block_here() sem_destroy reason=%s\n", strerror(errno));
    }
    return 0;
}

void mnld_block_exit(void) {
    mnld_exiting = true;
    LOGW("mnld exiting....");
    if(sem_post(&g_mnld_exit_sem) == -1) {
        LOGE("sem_post failed");
        _exit(0);
    }
}

/*************************************************
* Timer
**************************************************/
// -1 means failure
timer_t init_timer_id(timer_callback cb, int id) {
    struct sigevent sevp;
    timer_t timerid;

    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_value.sival_int = id;
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = cb;

    if (timer_create(CLOCK_MONOTONIC, &sevp, &timerid) == -1) {
        LOGE("timer_create  failed reason=[%s]", strerror(errno));
        return INVALID_TIMERID;
    }
    LOGD("timerid init:%lu", (unsigned long)timerid);
    return timerid;
}

// -1 means failure
timer_t init_timer(timer_callback cb) {
    return init_timer_id(cb, 0);
}

// -1 means failure
int start_timer(timer_t timerid, int milliseconds) {
    if(timerid != INVALID_TIMERID) {
        struct itimerspec expire;
        expire.it_interval.tv_sec = 0;
        expire.it_interval.tv_nsec = 0;
        expire.it_value.tv_sec = milliseconds/1000;
        expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
        return timer_settime(timerid, 0, &expire, NULL);
    } else {
        LOGE("invalid timerid:%p", timerid);
        return -1;
    }
}

// -1 means failure
int stop_timer(timer_t timerid) {
    return start_timer(timerid, 0);
}

// -1 means failure
int deinit_timer(timer_t timerid) {
    LOGD("timerid deinit:%lu", (unsigned long)timerid);

    if(timerid != INVALID_TIMERID) {
        if (timer_delete(timerid) == -1) {
            LOGE("timer_delete error:%s", strerror(errno));
            return -1;
        }
    } else {
        LOGE("invalid timerid:%p", timerid);
        return -1;
    }

    return 0;
}
/*************************************************
* Timer Alarm
**************************************************/
// -1 means failure
int init_timer_id_alarm(int id) {
    int wakealarm_fd;

    LOGE("timerfd_create  CLOCK_BOOTTIME_ALARM \n");
    //CLOCK_BOOTTIME_ALARM can work when AP sleep
    wakealarm_fd = timerfd_create(CLOCK_BOOTTIME_ALARM, id);
    if (wakealarm_fd == -1) {
        LOGE("timerfd_create CLOCK_BOOTTIME_ALARM failed reason=[%s]\n", strerror(errno));
        return -1;
    }
    return wakealarm_fd;
}

// -1 means failure
int  init_timer_alarm() {
    return init_timer_id_alarm(0);
}

// -1 means failure
int start_timer_alarm(int timerfd, int milliseconds) {
    struct itimerspec expire;
    expire.it_interval.tv_sec = 0;
    expire.it_interval.tv_nsec = 0;
    expire.it_value.tv_sec = milliseconds/1000;
    expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
    return timerfd_settime(timerfd, 0, &expire, NULL);
}

// -1 means failure
int stop_timer_alarm(int timerid) {
    return start_timer_alarm(timerid, 0);
}
/*************************************************
* Epoll
**************************************************/
// -1 means failure
int epoll_add_fd(int epfd, int fd) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = EPOLLIN;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_add_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

// -1 failed
int epoll_add_fd2(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_add_fd2() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

int epoll_del_fd(int epfd, int fd) {
    struct epoll_event  ev;
    int                 ret;

    if (epfd == -1)
        return -1;

    ev.events  = EPOLLIN;
    ev.data.fd = fd;
    do {
        ret = epoll_ctl(epfd, EPOLL_CTL_DEL, fd, &ev);
    } while (ret < 0 && errno == EINTR);
    return ret;
}

// -1 failed
int epoll_mod_fd(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    if (epoll_ctl(epfd, EPOLL_CTL_MOD, fd, &ev) == -1) {
        LOGE("epoll_mod_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

/*************************************************
* Local UDP Socket
**************************************************/
// -1 means failure
int socket_set_reuse_addr(int fd, bool reuse) {
    int val = (int)reuse;
    int newval;
    socklen_t intlen = sizeof(newval);

    if (0 != setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val))) {
        LOGE("setsockopt(SO_REUSEADDR) fail %s(%d)", strerror(errno), errno);
        return -1;
    }

    if (0 != getsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &newval, &intlen)) {
        LOGE("getsockopt(SO_REUSEADDR) fail %s(%d)", strerror(errno), errno);
        return -1;
    }

    if (newval != val) {
        LOGE("Failed to set SO_REUSEADDR");
        return -1;
    }

    return 0;
}

// -1 means failure
int socket_bind_udp(const char* path) {
    int fd;
    struct sockaddr_un addr;

    fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("socket_bind_udp() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    LOGD("fd=%d,path=%s\n", fd, path);

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    MNLD_STRNCPY(addr.sun_path + 1, path,sizeof(addr.sun_path) - 1);
    addr.sun_family = AF_UNIX;
    unlink(addr.sun_path);
    if(socket_set_reuse_addr(fd, true) == -1) {
        LOGE("socket_bind_udp() socket_set_reuse_addr() failed path=[%s] reason=[%s]%d",
            path, strerror(errno), errno);
    }
    if( bind(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        LOGE("socket_bind_udp() bind() failed path=[%s] reason=[%s]%d",
            path, strerror(errno), errno);
        close(fd);
        return -1;
    }

    LOGD("path=%s\n", path);

    return fd;
}

// -1 means failure
int socket_bind_udp_force(const char* path) {
    int ret = 0;
    int retry_cnt = 0;

    do {
        ret = socket_bind_udp(path);
        if(ret < 0) {
            retry_cnt++;
            LOGE("socket_bind_udp_force() bind() failed path=[%s] reason=[%s]%d, retry_cnt:%d",
                path, strerror(errno), errno, retry_cnt);
            msleep(1000);
            if(retry_cnt > 10) {
                LOGE("socket bind fail, mnld exit...");
                _exit(-1);
            }
        }
    } while(ret < 0);

    LOGD("path=%s\n", path);

    return ret;
}

int mnld_flp_to_mnld_fd_init(const char* path) {
    int flp_fd = -1;
    struct sockaddr_un cmd_local;

    if ((flp_fd = socket(AF_LOCAL, SOCK_DGRAM, 0)) == -1) {
        LOGE("flp2mnl socket create failed\n");
        return flp_fd;
    }

    unlink(path);
    memset(&cmd_local, 0, sizeof(cmd_local));
    cmd_local.sun_family = AF_LOCAL;
    MNLD_STRNCPY(cmd_local.sun_path, path,sizeof(cmd_local.sun_path));

    if (bind(flp_fd, (struct sockaddr *)&cmd_local, sizeof(cmd_local)) < 0) {
        LOGE("flp2mnl socket bind failed\n");
        close(flp_fd);
        flp_fd = -1;
        return flp_fd;
    }

    int res = chmod(path, 0660);  // (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
    LOGD("chmod res = %d, %s\n", res, strerror(errno));
    return flp_fd;
}

// -1 means failure
int socket_set_blocking(int fd, int blocking) {
    if (fd < 0) {
        LOGE("socket_set_blocking() invalid fd=%d", fd);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (flags == -1) {
        LOGE("socket_set_blocking() fcntl() failed invalid flags=%d reason=[%s]%d",
            flags, strerror(errno), errno);
        return -1;
    }

    flags = blocking ? ((unsigned int)flags&~O_NONBLOCK) : ((unsigned int)flags|O_NONBLOCK);
    return (fcntl(fd, F_SETFL, flags) == 0) ? 0 : -1;
}

// -1 means failure
int safe_sendto(const char* path, const char* buff, int len) {
    int ret = 0;
    struct sockaddr_un addr;
    int retry = 10;
    int fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("safe_sendto() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1){
        LOGE("fcntl failed reason=[%s]%d",
                    strerror(errno), errno);

        close(fd);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    MNLD_STRNCPY(addr.sun_path + 1, path,sizeof(addr.sun_path) - 1);
    addr.sun_family = AF_UNIX;

    while ((ret = sendto(fd, buff, len, 0,
        (const struct sockaddr *)&addr, sizeof(addr))) == -1) {
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("safe_sendto() sendto() failed path=[%s] ret=%d reason=[%s]%d",
            path, ret, strerror(errno), errno);
        break;
    }

    close(fd);
    return ret;
}

int mnld_sendto_flp(const void* dest, const char* buf, int size) {
    // dest: MTK_MNLD2HAL
    int ret = 0;
    int len = 0;
    struct sockaddr_un soc_addr;
    socklen_t addr_len;
    int retry = 10;

    int fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("safe_sendto() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }

    MNLD_STRNCPY(soc_addr.sun_path, dest,sizeof(soc_addr.sun_path));
    soc_addr.sun_family = AF_UNIX;
    addr_len = (offsetof(struct sockaddr_un, sun_path) + strlen(soc_addr.sun_path) + 1);

    //LOGD("mnld2flp fd: %d\n", fd);
    while ((len = sendto(fd, buf, size, 0,
        (const struct sockaddr *)&soc_addr, (socklen_t)addr_len)) == -1) {
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("[mnld2hal] ERR: sendto dest=[%s] len=%d reason =[%s]\n",
            (const char*)dest, size, strerror(errno));
        ret = -1;
        break;
    }
    close(fd);
    return ret;
}

// -1 means failure
int safe_recvfrom(int fd, char* buff, int len) {
    int ret = 0;
    int retry = 10;

    int flags = fcntl(fd, F_GETFL, 0);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1){
        LOGE("fcntl failed reason=[%s]%d",
                    strerror(errno), errno);
    }

    while ((ret = recvfrom(fd, buff, len, 0,
         NULL, NULL)) == -1) {
        LOGW("safe_recvfrom() ret=%d len=%d", ret, len);
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("safe_recvfrom() recvfrom() failed reason=[%s]%d",
            strerror(errno), errno);
        break;
    }
    return ret;
}

/*************************************************
* File
**************************************************/
// 0 not exist, 1 exist
int is_file_exist(const char* path) {
    FILE* fp = NULL;
    fp = fopen(path, "r");
    if (fp == NULL) {
        return 0;
    } else {
        fclose(fp);
        return 1;
    }
    return 0;
}

// -1 means failure
int get_file_size(const char* path) {
    struct stat s;
    if (stat(path, &s) == -1) {
        LOGD("get_file_size() stat() fail for [%s] reason=[%s]",
            path, strerror(errno));
        return -1;
    }
    return s.st_size;
}

// -1 failure
int delete_file(const char* file_path) {
    return remove(file_path);
}

// -1 means failure
int write_msg2file(const char* dest, const char* msg, ...) {
    FILE* fp = NULL;
    char buf[1024] = {0};
    va_list ap;

    if (dest == NULL || msg == NULL) {
        return -1;
    }

    va_start(ap, msg);
    if (vsnprintf(buf, sizeof(buf), msg, ap) < 0){
        LOGE("write_msg2file vsnprintf error occurred");\
    }
    va_end(ap);

    fp = fopen(dest, "a");
    if (fp == NULL) {
        LOGE("write_msg2file() fopen() fail reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    MNLD_FRINTF(fp, "%s", buf);
    MNLD_FCLOSE(fp);
    return 0;
}

void init_condition(SYNC_LOCK_T *lock) {
    int ret = 0;

    pthread_mutex_lock(&(lock->mutx));
    lock->condtion = 0;
    ret = pthread_mutex_unlock(&(lock->mutx));
    LOGD_ENG("ret mutex unlock = %d\n", ret);

    return;
}

void get_condition(SYNC_LOCK_T *lock) {
    int ret = 0;

    ret = pthread_mutex_lock(&(lock->mutx));
    //LOGD("ret mutex lock = %d\n", ret);

    while (!lock->condtion) {
        ret = pthread_cond_wait(&(lock->con), &(lock->mutx));
        //LOGD("ret cond wait = %d\n" , ret);
    }

    lock->condtion = 0;
    ret = pthread_mutex_unlock(&(lock->mutx));
    LOGD_ENG("ret mutex unlock = %d\n", ret);

    return;
}

void release_condition(SYNC_LOCK_T *lock) {
    int ret = 0;

    ret = pthread_mutex_lock(&(lock->mutx));
    //LOGD("ret mutex lock = %d\n", ret);

    lock->condtion = 1;
    ret = pthread_cond_signal(&(lock->con));
    //LOGD("ret cond_signal = %d\n", ret);

    ret = pthread_mutex_unlock(&(lock->mutx));
    LOGD_ENG("ret unlock= %d\n", ret);

    return;
}

#ifdef __LINUX_OS__

#define LISTEN_BACKLOG 4
/* Only the bottom bits are really the socket type; there are flags too. */
#define SOCK_TYPE_MASK 0xf
/* Documented in header file. */
int socket_make_sockaddr_un(const char *name, struct sockaddr_un *p_addr, socklen_t *alen)
{
    memset (p_addr, 0, sizeof (*p_addr));
    size_t namelen;

    namelen  = strlen(name);

    // Test with length +1 for the *initial* '\0'.
    if ((namelen + 1) > sizeof(p_addr->sun_path)) {
        goto error;
    }

    /*
     * Note: The path in this case is *not* supposed to be
     * '\0'-terminated. ("man 7 unix" for the gory details.)
     */

    p_addr->sun_path[0] = 0;
    memcpy(p_addr->sun_path + 1, name, namelen);

    p_addr->sun_family = AF_LOCAL;
    *alen = namelen + offsetof(struct sockaddr_un, sun_path) + 1;
    return 0;
error:
    return -1;
}

/**
 * connect to peer named "name" on fd
 * returns same fd or -1 on error.
 * fd is not closed on error. that's your job.
 *
 * Used by AndroidSocketImpl
 */
int socket_local_client_connect(int fd, const char *name)
{
    struct sockaddr_un addr;
    socklen_t alen;
    int err;

    err = socket_make_sockaddr_un(name, &addr, &alen);

    if (err < 0) {
        goto error;
    }

    if(connect(fd, (struct sockaddr *) &addr, alen) < 0) {
        goto error;
    }

    return fd;

error:
    return -1;
}

/**
 * connect to peer named "name"
 * returns fd or -1 on error
 */
int socket_local_client(const char *name, int type)
{
    int s;

    s = socket(AF_LOCAL, type, 0);
    if(s < 0) return -1;

    if ( 0 > socket_local_client_connect(s, name)) {
        close(s);
        return -1;
    }

    return s;
}



/**
 * Binds a pre-created socket(AF_LOCAL) 's' to 'name'
 * returns 's' on success, -1 on fail
 *
 * Does not call listen()
 */
int socket_local_server_bind(int s, const char *name)
{
    struct sockaddr_un addr;
    socklen_t alen;
    int n;
    int err;

    err = socket_make_sockaddr_un(name, &addr, &alen);

    if (err < 0) {
        return -1;
    }

    /* basically: if this is a filesystem path, unlink first */
    /*ignore ENOENT*/
    unlink(addr.sun_path);

    n = 1;
    setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &n, sizeof(n));

    if(bind(s, (struct sockaddr *) &addr, alen) < 0) {
        return -1;
    }

    return s;

}


/** Open a server-side UNIX domain datagram socket in the Linux non-filesystem
 *  namespace
 *
 *  Returns fd on success, -1 on fail
 */

int socket_local_server(const char *name, int type)
{
    int err;
    int s;

    s = socket(AF_LOCAL, type, 0);
    if (s < 0) return -1;

    err = socket_local_server_bind(s, name);

    if (err < 0) {
        close(s);
        return -1;
    }

    if ((type & SOCK_TYPE_MASK) == SOCK_STREAM) {
        int ret;

        ret = listen(s, LISTEN_BACKLOG);

        if (ret < 0) {
            close(s);
            return -1;
        }
    }

    return s;
}
/*
------------------------
| Length |  Message Body  |
------------------------
*/

#define MNLD_WAKELOCK_BUFF_SIZE           (10 * 1000)
#define HAL_MNL_BUFF_SIZE_SND           (32 * 1000)

int socket_tcp_send(int fd, const char* buff, int len) {
    char buff_send[HAL_MNL_BUFF_SIZE_SND] = {0};
    int offset = 0;
    int retry = 10;

    if(fd < 0) {
        LOGE("socket_tcp_send() invalid fd:%d", fd);
        return -1;
    }
    put_binary(buff_send, &offset, buff, len);  //Put length to the head

    int ret = -1;
    while((ret = write(fd, buff_send, offset)) == -1) {
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("socket_tcp_send() write() failed fd=[%d] ret=%d reason=[%s]%d",
            fd, ret, strerror(errno), errno);
        break;
    }
#ifdef MNLD_DUMP_CMD_RAW
    LOGD("send %d, ret %d, fd:%d", len, ret, fd);
    for(int i=0;i<len+4;i++) {
        LOGD("0x%x", buff_send[i]);
    }
#endif
    return ret;
}

/******************************************************************************
* Socket
******************************************************************************/

//-1 means fail or serverfd is returned
int socket_tcp_server_bind_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("socket_tcp_server_bind_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("socket_tcp_server_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(socket_set_reuse_addr(fd, true) == -1) {
        LOGE("socket_tcp_server_bind_local() socket_set_reuse_addr() failed path=[%s] reason=[%s]%d",
            name, strerror(errno), errno);
    }
    if(bind(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_tcp_server_bind_local() bind() failed, abstruct=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    if(listen(fd, 5) == -1) {
        LOGW("socket_tcp_server_bind_local() listen() failed, reason=[%s]%d",
            strerror(errno), errno);
    }
    return fd;
}

// -1 means failure
int socket_tcp_server_bind_local_force(bool abstract, const char* name) {
    int ret = 0;
    int retry_cnt = 0;

    do {
        ret = socket_tcp_server_bind_local(abstract, name);
        if(ret < 0) {
            retry_cnt++;
            LOGE("socket_tcp_server_bind_local_force() bind() failed path=[%s] reason=[%s]%d, retry_cnt:%d",
                name, strerror(errno), errno, retry_cnt);
            msleep(1000);
            if(retry_cnt > 10) {
                LOGE("socket bind fail, mnld exit...");
                _exit(-1);
            }
        }
    } while(ret < 0);

    LOGD("name=%s\n", name);

    return ret;
}

//-1 means fail or new clientfd is returned
int socket_tcp_server_new_connect(int serverfd) {
    int newfd;
    struct sockaddr_in addr;
    socklen_t len = sizeof(addr);

    newfd = accept(serverfd, (struct sockaddr*)&addr, &len);
    if(newfd == -1) {
        LOGE("socket_tcp_server_new_connect() accept() failed, serverfd=%d reason=[%s]%d",
            serverfd, strerror(errno), errno);
        return -1;
    }
    return newfd;
}

//-1 means fail or serverfd is returned
int socket_tcp_client_connect_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("socket_tcp_client_connect_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("socket_tcp_client_connect_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_tcp_client_connect_local() connect() failed, abstruct=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }

    return fd;
}

void mnld_buffer_initialize(cyclical_buffer_t *buffer, char *buffer_body, unsigned int buffer_size) {
   // Set up buffer manipulation pointers
   // end_buffer points to the next byte after the buffer
   buffer->start_buffer = buffer_body;
   buffer->end_buffer   = buffer->start_buffer + buffer_size;
   buffer->next_read    = buffer->start_buffer;
   buffer->next_write   = buffer->start_buffer;
   buffer->buffer_size = buffer_size;
   return;
}
#if 0
void mnld_dump_cycle(cyclical_buffer_t *cyc_buffer, char *buff, int len) {
    char* print_buf[HAL_MNL_BUFF_SIZE_SND] = {0};
    LOGD("=====Cycle Dump, len:%d=======", len);
    for(int i=0;i<len;i++) {
        char ch = 0;
        ch = *(cyc_buffer->next_read++);
        if(cyc_buffer->next_read == cyc_buffer->end_buffer) {
            cyc_buffer->next_read = cyc_buffer->start_bufferr;
        }
        sprintf(print_buf, " 0x%x", ch);
        LOGD("0x%x 0x%x 0x%x 0x%x", );
    }
}
#endif

int mnld_put_msg_to_cycle(cyclical_buffer_t *cyc_buffer, char *buff, int len) {
    int i;

#ifdef MNLD_DUMP_CMD_RAW
    LOGD("mnld_inf_put_msg_to_cycle:%d", len);
#endif
    for (i = 0; i < len; i++)
    {
        *(cyc_buffer->next_write++) = buff[i];
#ifdef MNLD_DUMP_CMD_RAW
        LOGD("0x%x", *(cyc_buffer->next_write-1));
#endif
        if (cyc_buffer->next_write == cyc_buffer->end_buffer){
            cyc_buffer->next_write = cyc_buffer->start_buffer;
        }

        if (cyc_buffer->next_write == cyc_buffer->next_read){
            LOGE("mnld_inf_put_msg_to_cycle buffer full!\r\n");
            return -1;
        }
    }

    return 0;
}
/******************************************
Get one message from cycle buffer
******************************************/
int mnld_get_one_msg(cyclical_buffer_t *cyc_buffer, char *buff) {
    char *next_write, *next_read;
    int data_size, i, header_len;
    char buffer[HAL_MNL_BUFF_SIZE_SND] = {0};
    int data_len = 0;
    int return_len = 0;

    next_write = cyc_buffer->next_write;
    next_read = cyc_buffer->next_read;

    if (cyc_buffer->next_read == next_write)
    {
        //buffer empty
        return -1;
    }

    header_len = sizeof(int);
    /*Compute data length*/
    if (cyc_buffer->next_read < next_write)
    {
        data_size = (unsigned long)next_write - (unsigned long)cyc_buffer->next_read;
    }
    else
    {
        data_size = (unsigned long)cyc_buffer->end_buffer - (unsigned long)cyc_buffer->next_read +
                     (unsigned long)next_write - (unsigned long)cyc_buffer->start_buffer;
    }

    /*Copy data header to buffer*/
    if (data_size >= header_len)
    {
        for (i = 0; i < header_len; i++)
        {
            buffer[i] = *(next_read++);
            if (next_read == cyc_buffer->end_buffer){
                next_read = cyc_buffer->start_buffer;
            }
        }

        memcpy(&data_len, buffer, sizeof(int));
        if (data_len <= data_size) {
        #ifdef MNLD_DUMP_CMD_RAW
            LOGD("data len:%d, header len:%d", data_len, header_len);
        #endif
            for (i = 0; i < (data_len + header_len); i++)
            {
                buff[i] = *(cyc_buffer->next_read++);
            #ifdef MNLD_DUMP_CMD_RAW
                LOGD("0x%x", buff[i]);
            #endif
                return_len++;
                if (cyc_buffer->next_read == cyc_buffer->end_buffer){
                    cyc_buffer->next_read = cyc_buffer->start_buffer;
                }
            }
        }
        else {
            //no enough data
            return -2;
        }

    }
    else
    {
        //no enough data
        return -2;
    }

    return return_len;
}

void client_list_dump(client_list* list) {
    LOGD("client_list_dump() num=[%d]", list->num);
    int i = 0;
    for(i = 0; i < list->num; i++) {
        LOGD("  i=[%d] fd=[%d]", i, list->clients[i].fd);
    }
}

bool client_list_add(client_list* list, int fd) {
    if(list->num >= MAX_CLIENT_NUM) {
        LOGE("client_list_add() fail due to MAX_CLIENT_NUM=%d", MAX_CLIENT_NUM);
        return false;
    }

    list->clients[list->num].fd = fd;
    list->num++;
    return true;
}

bool client_list_remove(client_list* list, int fd) {
    int i = 0;
    for(i = 0; i < list->num; i++) {
        if(list->clients[i].fd == fd) {
            for(; i < list->num - 1; i++) {
                list->clients[i] = list->clients[i + 1];
            }
            memset(&list->clients[i], 0, sizeof(client_ctx));
            list->num--;
            return true;
        }
    }
    return false;
}

bool client_list_contains(client_list* list, int fd) {
    int i = 0;
    LOGD("to find fd:%d", fd);
    for(i = 0; i < list->num; i++) {
        if(list->clients[i].fd == fd) {
            return true;
        }
    }
    return false;
}

client_ctx* client_list_get_ctx_by_fd(client_list* list, int fd) {
    unsigned int i = 0;
    for(i = 0; i < list->num; i++) {
        if(list->clients[i].fd == fd) {
            return &list->clients[i];
        }
    }
    return NULL;
}

bool client_list_is_all_gnss_stop(client_list* list) {
    int i = 0;
    for(i = 0; i < list->num; i++) {
        if(list->clients[i].gnss_started) {
            return false;
        }
    }
    return true;
}

bool client_list_is_all_gnss_cleanup(client_list* list) {
    int i = 0;
    for(i = 0; i < list->num; i++) {
        if(list->clients[i].gnss_inited) {
            return false;
        }
    }
    return true;
}

bool client_list_is_all_meas_stop(client_list* list) {
    int i = 0;
    for(i = 0; i < list->num; i++) {
        if(list->clients[i].meas_started) {
            return false;
        }
    }
    return true;
}

int client_send_to_all(client_list* list, const char* buff, int len) {
    int i = 0;
    int ret = -1;

    for(i = 0; i < list->num; i++) {
        int fd = list->clients[i].fd;
        ret = socket_tcp_send(fd, buff, len);
    }
    return ret;
}

extern client_list g_hal_basic_client_list;
extern client_list g_hal_ext_client_list;

int mnl2hal_basic(const char* buff, int len) {
    int i = 0;
    int ret = -1;

    for(i = 0; i < g_hal_basic_client_list.num; i++) {
        if(g_hal_basic_client_list.clients[i].gnss_started || g_hal_basic_client_list.clients[i].meas_started) {
            int fd = g_hal_basic_client_list.clients[i].fd;
            ret = socket_tcp_send(fd, buff, len);
        }
    }
    return ret;
}

int mnl2hal_basic_all(const char* buff, int len) {
    int i = 0;
    int ret = -1;

    for(i = 0; i < g_hal_basic_client_list.num; i++) {
        int fd = g_hal_basic_client_list.clients[i].fd;
        ret = socket_tcp_send(fd, buff, len);
    }
    return ret;
}

int mnl2hal_ext(const char* buff, int len) {
    return client_send_to_all(&g_hal_ext_client_list, buff, len);
}

wake_lock_ctx wlock_ctx;

void mnld_wake_lock() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    int fd = open("/sys/power/wake_lock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    int ret = write(fd, MNLD_WAKE_LOCK_ID, strlen(MNLD_WAKE_LOCK_ID));
    if(ret == -1) {
        LOGE("write() failed id=[%s], reason=[%s]%d", MNLD_WAKE_LOCK_ID, strerror(errno), errno);
        close(fd);
        return;
    }

    close(fd);
#endif
    return;
}

void mnld_wake_unlock() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_unlock on Linux
    int fd = open("/sys/power/wake_unlock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    int ret = write(fd, MNLD_WAKE_LOCK_ID, strlen(MNLD_WAKE_LOCK_ID));
    if(ret == -1) {
        LOGE("write() failed id=[%s], reason=[%s]%d", MNLD_WAKE_LOCK_ID, strerror(errno), errno);
        close(fd);
        return;
    }

    close(fd);
#endif
    return;
}

int mnld_wakeup_mutex_lock() {
    int ret = pthread_mutex_lock(&wlock_ctx.mutex);
    if(ret != 0) {
        LOGE("pthread_mutex_lock() failed, reason=[%s]%d", strerror(ret), ret);
    }
    return ret;
}

int mnld_wakeup_mutex_unlock() {
    int ret = pthread_mutex_unlock(&wlock_ctx.mutex);
    if(ret != 0) {
        LOGE("pthread_mutex_unlock() failed, reason=[%s]%d", strerror(ret), ret);
    }

    return ret;
}

void mnld_wake_lock_take() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    UINT64 time_cur = 0;
    int timer_ret;
    int mutex_ret = 0;

    mutex_ret = mnld_wakeup_mutex_lock();
    time_cur = get_time_in_millisecond();

    if(time_cur > (wlock_ctx.time_last_refresh + MNLD_WAKE_LOCK_LATENCY)) {
        timer_ret = stop_timer(wlock_ctx.unlock_timer);
        LOGD("stop unlock_timer, ret=%d", timer_ret);
        if(wlock_ctx.wake_lock_acquired == false) {
            wlock_ctx.wake_lock_acquired = true;
            mnld_wake_lock();
            LOGD("mnld_wake_lock");
        }
    }

    if(mutex_ret == 0) {
        mnld_wakeup_mutex_unlock();
    }
#endif
}

void mnld_wake_lock_give() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    UINT64 time_cur = 0;
    int timer_ret;
    int mutex_ret = 0;

    mutex_ret = mnld_wakeup_mutex_lock();
    //delay to do wake_unlock to ensure the msg can be deliveried to other process
    time_cur = get_time_in_millisecond();
    if(time_cur > (wlock_ctx.time_last_refresh + MNLD_WAKE_LOCK_LATENCY)) {  //Only refresh timer when over latency time
        timer_ret = start_timer(wlock_ctx.unlock_timer, MNLD_WAKE_LOCK_TIMEOUT);
        LOGD("start unlock_timer, ret=%d", timer_ret);
        wlock_ctx.time_last_refresh = get_time_in_millisecond();
    }

    if(mutex_ret == 0) {
        mnld_wakeup_mutex_unlock();
    }
#endif
}

#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
static void mnld_timer_wake_unlock_routine(int id) {
    int mutex_ret = 0;
    //do not use the internal msg or it will cause infinite loop in epoll_wait
    mutex_ret = mnld_wakeup_mutex_lock();
    if(wlock_ctx.wake_lock_acquired == true) {
        wlock_ctx.wake_lock_acquired = false;
        mnld_wake_unlock();
        LOGD("mnld_wake_unlock");
    }
    if(mutex_ret == 0) {
        mnld_wakeup_mutex_unlock();
    }
}
#endif

void mnld_wake_lock_init() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    int ret = 0;
    int read_len = 0;
    char buff_read[MNLD_WAKELOCK_BUFF_SIZE] = {0};

    int fd = open("/sys/power/wake_lock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    read_len = read(fd, buff_read, sizeof(buff_read)-1);
    buff_read[MNLD_WAKELOCK_BUFF_SIZE -1] = 0;
    if(read_len >= strlen(MNLD_WAKE_LOCK_ID) && strstr(buff_read, MNLD_WAKE_LOCK_ID) != NULL) {
        mnld_wake_unlock();
    }
    LOGD("wake_lock:[%s]", buff_read);
    close(fd);

    memset(&wlock_ctx, 0, sizeof(wlock_ctx));
    wlock_ctx.wake_lock_acquired = false;
    wlock_ctx.unlock_timer = init_timer(mnld_timer_wake_unlock_routine);
    if(wlock_ctx.unlock_timer == INVALID_TIMERID) {
        LOGE("init_timer(mnld_timer_wake_unlock_routine, 0) failed");
        exit(1);
    }

    ret = pthread_mutex_init(&wlock_ctx.mutex, NULL);
    if(ret != 0) {
        LOGE("pthread_mutex_init() failed, reason=[%s]%d", strerror(errno), errno);
        exit(1);
    }
#endif
}

void mnld_wake_lock_deinit() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    deinit_timer(wlock_ctx.unlock_timer);
    wlock_ctx.unlock_timer = INVALID_TIMERID;
    pthread_mutex_destroy(&wlock_ctx.mutex);
#endif
}

#endif //__LINUX_OS__

void set_signal_ignore(int signo) {
    sigset_t signal_mask;

    LOGD("set_signal_ignore, signo:%d", signo);
    if(sigemptyset(&signal_mask) < 0) {
        LOGW("sigemptyset fail, %s(%d)", strerror(errno), errno);
    }
    if(sigaddset(&signal_mask, signo) < 0) {
        LOGW("sigaddset fail, %s(%d)", strerror(errno), errno);
    }
    if(sigprocmask(SIG_BLOCK, &signal_mask, NULL) < 0) {
        LOGW("sigprocmask fail, %s(%d)", strerror(errno), errno);
    }
}
