#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h> /* usleep */
#include <sys/time.h>
#include <string.h>
#include <sys/epoll.h>
#include <errno.h>
#include <semaphore.h>
#include <math.h>
#include <dirent.h>
#include <sys/stat.h>

#include "mtk_tree.h"
#include "agps_interface.h"
#include "agps_debug_interface.h"
#include "mnldinf_basic.h"
#include "mnldinf_ext.h"
#include "mtk_lbs_utility.h"
#include "mtk_gnss_at_client.h"
#include "MtkNlp.h"

// ------------- history-----------------
//  1.00 MTK LBS EM implementation done
//  1.01 Add MTK GNSS AT Command
//  1.02 porting to OpenWrt
//  1.03 Change MNLD interface to TCP mnldinf
// ------------- definition -----------------
#define MTK_LBS_EM_VERSION      "1.03"

//---------------------------------- porting configurations -------------------------------------------
#define MTK_LBS_EM_TEST_CERT_PATH   "certs/UbuntuOne-Go_Daddy_CA.pem"
#define MTK_LBS_EM_TEST_CERT_NAME   "UbuntuOne-Go_Daddy_CA.pem"

//-------------------------------------------------------------------------------------------------
#define INVALID_VALUE   -1
#define LBS_EM_INT_CMD_RETRY_CONNECT 3
#define LBS_EM_MAX_RETRY_CONNECT_TIMES 10
#define LBS_EM_RETRY_CONNECT_INTERVAL 3000
#define LBS_EM_WAKE_LOCK_TIMEOUT (5 * 1000)
#define LBS_EM_WAKE_LOCK_ID "lbs_em_wakelock"

typedef enum {
    UI_MAIN_TYPE_EXIT = 0,
    UI_MAIN_TYPE_GPS_CONTROL = 1,
    UI_MAIN_TYPE_AGPS_SETTINGS,
    UI_MAIN_TYPE_DEBUG_SETTINGS,
    UI_MAIN_TYPE_GNSS_AT_SETTINGS,
    UI_MAIN_TYPE_NLP_TEST,
    UI_MAIN_TYPE_DEBUG_1 = 3646633,
} ui_main_type;

typedef enum {
    UI_GPS_CONTROL_TYPE_BACK = 0,
    UI_GPS_CONTROL_TYPE_GPS_INIT,
    UI_GPS_CONTROL_TYPE_GPS_START,
    UI_GPS_CONTROL_TYPE_GPS_START_WITH_WAKE_LOCK_RELEASE,
    UI_GPS_CONTROL_TYPE_GPS_STOP,
    UI_GPS_CONTROL_TYPE_GPS_CLEANUP,
    UI_GPS_CONTROL_TYPE_GPS_DELETE_AIDING_DATA,
    UI_GPS_CONTROL_TYPE_GPS_NI_NOTIRY_RESPONSE,
} ui_gps_control_type;

typedef enum {
    UI_GPS_DELETE_AIDING_TYPE_BACK = 0,
    UI_GPS_DELETE_AIDING_TYPE_HOT_START,
    UI_GPS_DELETE_AIDING_TYPE_WARM_START,
    UI_GPS_DELETE_AIDING_TYPE_COLD_START,
    UI_GPS_DELETE_AIDING_TYPE_FULL_START,
} ui_gps_delete_aiding_type;

typedef enum {
    UI_AGPS_SETTINGS_TYPE_BACK = 0,

    UI_AGPS_SETTINGS_TYPE_GET_AGPS_SETTINGS = 1,
    UI_AGPS_SETTINGS_TYPE_SET_AGPS_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_AGPS_PROTOCOL,
    UI_AGPS_SETTINGS_TYPE_SET_CDMA_PREFERENCE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_PREFER_METHOD,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_MSA_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_MSB_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_ECID_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_AUTONOMOUS_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_AFLT_ENABLE = 10,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_LPP_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VERSION,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_TLS_VERSION,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_CA_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_UDP_ENABLE =15,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_UT2,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_UT3,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VER_MINOR,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VER_SER_IND,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_PROFILE = 20,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_PERIODIC_SETTINGS,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_AREA_SETTINGS,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SI_QOP_SETTINGS,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_ALLOW_NI_REQUEST,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_ALLOW_ROAMING = 25,
    UI_AGPS_SETTINGS_TYPE_SET_CP_AUTO_RESET,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_LOG_TO_FILE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SENSITIVE_LOG,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_CERT_FROM_SDCARD,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_AUTO_PROFILE = 30,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_DEDICATE_APN,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_SYNC_TO_SLP,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_IMSI_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_E911_GPS_ICON,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_LPPE_NETWORK_LOCATION_DISABLE = 35,
    UI_AGPS_SETTINGS_TYPE_SET_CP_LPPE_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_UP_LPPE_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_AOSP_PROFILE_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_BIND_NLP_SETTING_TO_SUPL,
    UI_AGPS_SETTINGS_TYPE_SET_SUPL_LBS_LOG_ENABLE = 40,

    UI_AGPS_SETTINGS_TYPE_SET_INGORE_SI_FOR_E911,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_WLAN,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_SRN,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_SENSOR,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_DBH = 45,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_WLAN,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_SRN,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_SENSOR,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_DBH,
    UI_AGPS_SETTINGS_TYPE_SET_UP_LPP_IN_2G_3G_DISABLE = 50,
    UI_AGPS_SETTINGS_TYPE_SET_UP_RRLP_IN_4G_DISABLE,
    UI_AGPS_SETTINGS_TYPE_SET_UP_SI_DISABLE,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_DEFAULT_NLP_LOCATION_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_FORCE_OTDOA_ASSIST_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_AOSP_POSITION_MODE_ENABLE = 55,
    UI_AGPS_SETTINGS_TYPE_SET_PRIVACY_OVERRIDE_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_ALLOW_NI_FOR_GPS_OFF_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_IP_VERSION_PREFER,

    UI_AGPS_SETTINGS_TYPE_SET_SUPL_TCP_KEEPALIVE,
    UI_AGPS_SETTINGS_TYPE_SET_LPPE_CROWD_SOURCE_CONFIDENT = 60,
    UI_AGPS_SETTINGS_TYPE_SET_GNSS_SIB8_SIB16_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_GLONASS_SATELLITE_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_BEIDOU_SATELLITE_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_GALILEO_SATELLITE_ENABLE,

    UI_AGPS_SETTINGS_TYPE_SET_CP_MOLR_POS_METHOD = 100,
    UI_AGPS_SETTINGS_TYPE_SET_CP_EXTERNAL_ADDR_ENABLE,
    UI_AGPS_SETTINGS_TYPE_SET_CP_MLC_NUMBER,
    UI_AGPS_SETTINGS_TYPE_SET_CP_EPC_MOLR_LPP_PAYLOAD_ENABLE,

    UI_AGPS_SETTINGS_TYPE_RESET_TO_DEFAULT = 999,
} ui_agps_settings_type;

typedef enum {
    UI_AT_SETTINGS_TYPE_BACK = 0,

    UI_AT_SETTINGS_GNSS_START_REQUEST,
    UI_AT_SETTINGS_GNSS_START_WITH_WAKE_LOCK_RELEASE,
    UI_AT_SETTINGS_GNSS_STOP_REQUEST,
    UI_AT_SETTINGS_HOST_REBOOT_NOTIFY,
    UI_AT_SETTINGS_GNSS_STATUS_REQUEST = 5,
    UI_AT_SETTINGS_GNSS_ENABLE_SET,
    UI_AT_SETTINGS_NI_ENABLE_SET,
    UI_AT_SETTINGS_AGPS_MODE_SET,
    UI_AT_SETTINGS_SUPL_VERSION_SET,
    UI_AT_SETTINGS_SUPL_ADDRESS_SET = 10,
    UI_AT_SETTINGS_DELETE_AIDING_DATA_REQUEST,
    UI_AT_SETTINGS_NI_NOTIFY_RESPONSE,
    UI_AT_SETTINGS_REFERENCE_LOCATION_RESPONSE,
    UI_AT_SETTINGS_REFERENCE_TIME_RESPONSE,
    UI_AT_SETTINGS_NMEA_CONFIG_SET = 15,
    UI_AT_SETTINGS_LOOPBACK_REQUEST,

    UI_AT_SETTINGS_GEOFENCE_MAX_NUMBER_REQUEST = 31,
    UI_AT_SETTINGS_GEOFENCE_ADD_CIRCLE_REQUEST,
    UI_AT_SETTINGS_GEOFENCE_ADD_CIRCLE_WITH_WAKE_LOCK_RELEASE,
    UI_AT_SETTINGS_GEOFENCE_DELETE_REQUEST,
    UI_AT_SETTINGS_GEOFENCE_DELETE_ALL_REQUEST,

    UI_AT_SETTINGS_CERTIFICATE_INJECT = 51,
    UI_AT_SETTINGS_CERTIFICATE_NAME_LIST_REQUEST,
    UI_AT_SETTINGS_CERTIFICATE_DELETE,
    UI_AT_SETTINGS_CERTIFICATE_DELETE_ALL,

    UI_AT_SETTINGS_INPUT_A_GNSS_AT_COMMAND_MANUALLY = 99,
} ui_at_test_type;

typedef enum {
    UI_NLP_TEST_TYPE_BACK = 0,
    UI_NLP_TEST_TYPE_START_RQUEST,
    UI_NLP_TEST_TYPE_STOP_RQUEST,
    UI_NLP_TEST_TYPE_GET_LAST_LOC,
    UI_NLP_TEST_TYPE_GET_SINGLE_UPDATE,
    UI_NLP_TEST_TYPE_GET_PRESSURE_SENSOR,
} ui_nlp_test_type;

typedef enum {
    UI_DEBUG_1_TYPE_BACK = 0,
    UI_DEBUG_1_TYPE_RESET_AGPSD,
    UI_DEBUG_1_TYPE_TEST_BUTTON,
    UI_DEBUG_1_TYPE_EMULATOR_MODE_ON,
    UI_DEBUG_1_TYPE_EMULATOR_MODE_OFF,
    UI_DEBUG_1_TYPE_START_TEST_CASE,
} ui_debug_1_type;

#define MAX_GEOFENCE_NUM 20

typedef struct {
    int id;
} at_geofence_ctx;

typedef struct {
    int num;
    at_geofence_ctx elem[MAX_GEOFENCE_NUM];
} at_geofence_list;


// LBS EM (client) <-> GNSS ADP (server)
#define GNSSADP_AT_SOCKET     "mtk_gnssadp_at"

// LBS EM (client) <-> LBS EM (server)
#define LBSEM_INTERNAL_SOCKET "mtk_lbsem_internal"
int g_epfd;
int g_agps_debug_fd = -1;
int g_gnssadp_fd = -1;
int g_mnld_basic_fd = -1;
int g_mnld_ext_fd = -1;
int g_internal_fd = -1;
mtk_socket_fd g_nlpservice_fd; // nlp client
timer_t g_connect_retry_timer;
int g_connect_retry_count = 0;
agps_intf_agps_config g_config;
at_geofence_list g_geo_list;

// wake lock
pthread_mutex_t g_mutex;
timer_t g_wake_unlock_timer;
bool g_wake_lock_acquired = false;
bool g_at_geofence_add_with_wake_lock_release = false;


void mnldinf_location_hdlr(gps_location location);
void mnldinf_nmea_hdlr(int64_t timestamp, const char* nmea, int length);
void mnldinf_gnss_sv_hdlr(gnss_sv_info sv);
static void lbs_em_acquire_wake_lock_mutex();

bool geofence_list_add(at_geofence_list* list, int id) {
    if(list->num >= MAX_GEOFENCE_NUM) {
        LOGE("[Geofence] Fail to add due to MAX_GEOFENCE_NUM=%d", MAX_GEOFENCE_NUM);
        return false;
    }

    list->elem[list->num].id = id;
    list->num++;
    return true;
}

bool geofence_list_remove(at_geofence_list* list, int id) {
    int i = 0;
    for(i = 0; i < list->num; i++) {
        if(list->elem[i].id == id) {
            for(; i < list->num - 1; i++) {
                list->elem[i] = list->elem[i + 1];
            }
            memset(&list->elem[i], 0, sizeof(at_geofence_ctx));
            list->num--;
            return true;
        }
    }
    return false;
}

// ------------- utils -----------------

static void clear_stdin_buff() {
    char c;
    while((c = (char)getchar()) != '\n' && c != EOF);
}

static int scanf_int(int default_value, bool clear_screen) {
    int ret;
    int input;
    ret = scanf("%d", &input);
    clear_stdin_buff();
    if(clear_screen) {
        system("clear");
    }
    if(ret <= 0) {
        return default_value;
    } else {
        return input;
    }
}

static long long scanf_long(long long default_value, bool clear_screen) {
    int ret;
    long long input;
    ret = scanf("%lld", &input);
    clear_stdin_buff();
    if(clear_screen) {
        system("clear");
    }
    if(ret <= 0) {
        return default_value;
    } else {
        return input;
    }
}

static double scanf_double(double default_value, bool clear_screen) {
    int ret;
    double input;
    ret = scanf("%lf", &input);
    clear_stdin_buff();
    if(clear_screen) {
        system("clear");
    }
    if(ret <= 0) {
        return default_value;
    } else {
        return input;
    }
}


//-1 means failure
static int get_time_str(char* buf, int len) {
    struct timeval  tv;
    struct timezone tz;
    struct tm      *tm;

    gettimeofday(&tv, &tz);
    tm = localtime(&tv.tv_sec);
    if (tm == NULL) {
        return 0;
    }

    memset(buf, 0, len);
    sprintf(buf, "%04d/%02d/%02d %02d:%02d:%02d.%03d",
        tm->tm_year + 1900, 1 + tm->tm_mon, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec,
        (int)(tv.tv_usec / 1000));

    return 0;
}

#define msleep(usec) {\
    int ret = usleep((usec)*1000);\
    if(ret == -1) {\
        LOGE("usleep() failed, usec=%lu reason=[%s]%d", strerror(errno), errno);\
    }\
}

// ------------- global_variable -----------------
traversal_path g_traversal;
tree_node g_tree_root;

int g_agps_debug_enable = 0;
int g_gps_info_enable = 1;
int g_nmea_enable = 1;
int g_at_cmd_enable = 1;

bool g_at_cert_injecting = false;
sem_t g_cert_sem;
bool g_cert_result = false;
int g_notify_id = 0;

static bool cert_inject_full_test(mtk_gnss_cert *cert) {
    //test 41 full cert testing
    memset(cert, 0, sizeof(*cert));
    FILE* fp = fopen(MTK_LBS_EM_TEST_CERT_PATH, "r");
    if(fp == NULL) {
        LOGE("cert_inject_procedure() fopen() failed");
        return false;
    }

    //get file size
    fseek(fp, 0L, SEEK_END);
    cert->total_msg_len = ftell(fp);
    fseek(fp, 0L, SEEK_SET);

    cert->seq_no = 0;
    LBS_STRNCPY(cert->name, MTK_LBS_EM_TEST_CERT_NAME, sizeof(cert->name));

    int ret;
    do {
        memset(cert->data, 0, sizeof(cert->data));
        ret = fread(cert->data, 1, 1000, fp);
        if(ret > 0) {
            //LOGD("fread() ret=[%d] buff=[%s]", ret, buff);
            cert->seq_no++;
            mtk_at_send_gnss_cert_set(g_gnssadp_fd, cert);
            sem_wait(&g_cert_sem);
            if(g_cert_result != true) {
                LOGE("cert_inject_procedure() test 41 failed!!");
                fclose(fp);
                return false;
            }
        }
    } while(ret > 0);

    fclose(fp);
    LOGW("cert_inject_procedure() all test cases are passed!!");
    return true;
}

static void cert_inject_procedure() {
    g_at_cert_injecting = true;
    mtk_gnss_cert cert;
    memset(&cert, 0, sizeof(cert));

    do {
        //test 1 invalid total_msg_len 0
        cert.total_msg_len = 0;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 1 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 2 invalid seq_no 0
        cert.total_msg_len = 1;
        cert.seq_no = 0;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 3 empy name
        cert.total_msg_len = 1;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 3 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 4 empy data
        cert.total_msg_len = 1;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 4 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 5 total_msg_len < data size
        cert.total_msg_len = 1;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "data", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 5 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 6 invalid total_msg_len -1
        cert.total_msg_len = -1;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 6 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 7 invalid total_msg_len 64001
        cert.total_msg_len = 64001;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 7 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 8 invalid seq_no -1
        cert.total_msg_len = 1;
        cert.seq_no = -1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 8 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 9 invalid seq_no 65
        cert.total_msg_len = 1;
        cert.seq_no = 65;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 9 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 10 invlid name
        cert.total_msg_len = 1;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "/", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 10 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 11 invlid name
        cert.total_msg_len = 1;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d2", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 11 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 21 invalid total_msg_len
        cert.total_msg_len = 2;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 21.1 failed!!");
            g_at_cert_injecting = false;
            break;
        }
        cert.total_msg_len = 1;
        cert.seq_no = 2;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "e", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 21.2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 22 invalid seq_no
        cert.total_msg_len = 2;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 22.1 failed!!");
            g_at_cert_injecting = false;
            break;
        }
        cert.total_msg_len = 2;
        cert.seq_no = 3;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "e", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 22.2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 23 invalid name
        cert.total_msg_len = 2;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 23.1 failed!!");
            g_at_cert_injecting = false;
            break;
        }
        cert.total_msg_len = 2;
        cert.seq_no = 2;
        LBS_STRNCPY(cert.name, "d2", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "e", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 23.2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 24 data over total size
        cert.total_msg_len = 2;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "d", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 24.1 failed!!");
            g_at_cert_injecting = false;
            break;
        }
        cert.total_msg_len = 2;
        cert.seq_no = 2;
        LBS_STRNCPY(cert.name, "d", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "e2", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != false) {
            LOGE("cert_inject_procedure() test 24.2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 31 simple test
        cert.total_msg_len = 4;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "test", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "data", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 31 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 32 valid name
        cert.total_msg_len = 5;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 []()-._", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "data2", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 32 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 33 simple sequence test
        cert.total_msg_len = 3;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "test2", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "a", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 33.1 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        cert.total_msg_len = 3;
        cert.seq_no = 2;
        LBS_STRNCPY(cert.name, "test2", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "b", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 33.2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        cert.total_msg_len = 3;
        cert.seq_no = 3;
        LBS_STRNCPY(cert.name, "test2", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "c", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 33.3 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 34 test seq_no reset
        cert.total_msg_len = 2;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "test3", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "a", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 34.1 failed!!");
            g_at_cert_injecting = false;
            break;
        }
        cert.total_msg_len = 2;
        cert.seq_no = 1;
        LBS_STRNCPY(cert.name, "test4", sizeof(cert.name));
        LBS_STRNCPY(cert.data, "ab", sizeof(cert.data));
        mtk_at_send_gnss_cert_set(g_gnssadp_fd, &cert);
        sem_wait(&g_cert_sem);
        if(g_cert_result != true) {
            LOGE("cert_inject_procedure() test 34.2 failed!!");
            g_at_cert_injecting = false;
            break;
        }

        //test 41 full cert testing
        if (!cert_inject_full_test(&cert)) {
            g_at_cert_injecting = false;
            break;
        }
    } while(false);

}

// ------------- implementation -----------------

void ui_main() {
    LOGD("=== Main Menu, Version=[%s] ===", MTK_LBS_EM_VERSION);
    LOGD("0.  Exit");
    LOGD(" ");
    LOGD("1.  GPS Control (ex: Start, Stop, Delete Aiding Data, ...etc.)");  // more
    LOGD("2.  AGPS Settings (ex: SUPL profile, ...etc.)");   // more
    LOGD("3.  Debug Settings");   // more
    LOGD("4.  GNSS AT Command Test");   // more
    LOGD("5.  NLP Service Test");   // more

    LOGD("input: ");
    ui_main_type input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case UI_MAIN_TYPE_EXIT:
            exit(0);
            break;
        case UI_MAIN_TYPE_GPS_CONTROL:
        case UI_MAIN_TYPE_AGPS_SETTINGS:
        case UI_MAIN_TYPE_DEBUG_SETTINGS:
        case UI_MAIN_TYPE_GNSS_AT_SETTINGS:
        case UI_MAIN_TYPE_NLP_TEST:
        case UI_MAIN_TYPE_DEBUG_1:
            traversal_path_go(&g_traversal, input);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_gps_control() {
    LOGD("=== GPS Control ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  GPS Init");
    LOGD("2.  GPS Start");
    LOGD("3.  GPS Start with Wake Lock Release");
    LOGD("4.  GPS Stop");
    LOGD("5.  GPS Cleanup");
    LOGD("6.  GPS Delete Aiding Data");  // more
    LOGD("7.  GPS Ni Notify Response");

    LOGD("input: ");
    ui_gps_control_type input = scanf_int(INVALID_VALUE, false);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case UI_GPS_CONTROL_TYPE_BACK:
            traversal_path_back(&g_traversal);
            break;
        case UI_GPS_CONTROL_TYPE_GPS_INIT:
            LOGD("GPS Init");
            mnldinf_basic_gnss_init(g_mnld_basic_fd);
            break;
        case UI_GPS_CONTROL_TYPE_GPS_START:
            LOGD("GPS Start");
            mnldinf_basic_gnss_start(g_mnld_basic_fd);
            break;
        case UI_GPS_CONTROL_TYPE_GPS_START_WITH_WAKE_LOCK_RELEASE:
            LOGD("GPS Start With wake lock release");
            mnldinf_basic_gnss_start(g_mnld_basic_fd);
            //delay to do wake_unlock to ensure the msg can be deliveried to other process
            timer_start(g_wake_unlock_timer, LBS_EM_WAKE_LOCK_TIMEOUT);
            break;
        case UI_GPS_CONTROL_TYPE_GPS_STOP:
            LOGD("GPS Stop");
            mnldinf_basic_gnss_stop(g_mnld_basic_fd);
            lbs_em_acquire_wake_lock_mutex();
            break;
        case UI_GPS_CONTROL_TYPE_GPS_CLEANUP:
            LOGD("GPS Cleanup");
            mnldinf_basic_gnss_cleanup(g_mnld_basic_fd);
            break;
        case UI_GPS_CONTROL_TYPE_GPS_DELETE_AIDING_DATA:
            traversal_path_go(&g_traversal, input);
            break;
        case UI_GPS_CONTROL_TYPE_GPS_NI_NOTIRY_RESPONSE: {
                LOGD("NI Notify Response: Response Type (1=Accept, 2=Deny, 3=NO Response) input:");
                int user_response = scanf_int(INVALID_VALUE, false);
                if(user_response == INVALID_VALUE) {
                    LOGE("ERR: read invalid resp_type");
                    return;
                }

                system("clear");
                mnldinf_ext_ni_respond(g_mnld_ext_fd, g_notify_id, user_response);
                LOGD("NI Notify Response id=[%d] resp_type=[%d] Sent", g_notify_id, user_response);
            }
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_gps_control_delete_aiding_data() {
    LOGD("=== GPS Delete Aiding Data ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Hot Start");
    LOGD("2.  Warm Start");
    LOGD("3.  Cold Start");
    LOGD("4.  Full Start");

    LOGD("input: ");
    ui_gps_delete_aiding_type input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

#define GPS_DELETE_EPHEMERIS        0x0001
#define GPS_DELETE_ALMANAC          0x0002
#define GPS_DELETE_POSITION         0x0004
#define GPS_DELETE_TIME             0x0008
#define GPS_DELETE_IONO             0x0010
#define GPS_DELETE_UTC              0x0020
#define GPS_DELETE_HEALTH           0x0040
#define GPS_DELETE_SVDIR            0x0080
#define GPS_DELETE_SVSTEER          0x0100
#define GPS_DELETE_SADATA           0x0200
#define GPS_DELETE_RTI              0x0400
#define GPS_DELETE_CELLDB_INFO      0x8000
#define GPS_DELETE_ALL              0xFFFF

    switch(input) {
        case UI_GPS_DELETE_AIDING_TYPE_BACK:
            traversal_path_back(&g_traversal);
            break;
        case UI_GPS_DELETE_AIDING_TYPE_HOT_START:
            LOGD("Hot Start");
            mnldinf_ext_gnss_delete_aiding_data(g_mnld_ext_fd, GPS_DELETE_RTI);
            traversal_path_back(&g_traversal);
            break;
        case UI_GPS_DELETE_AIDING_TYPE_WARM_START:
            LOGD("Warm Start");
            mnldinf_ext_gnss_delete_aiding_data(g_mnld_ext_fd, GPS_DELETE_EPHEMERIS);
            traversal_path_back(&g_traversal);
            break;
        case UI_GPS_DELETE_AIDING_TYPE_COLD_START:
            LOGD("Cold Start");
            mnldinf_ext_gnss_delete_aiding_data(g_mnld_ext_fd, GPS_DELETE_EPHEMERIS |
                GPS_DELETE_POSITION | GPS_DELETE_TIME | GPS_DELETE_IONO |
                GPS_DELETE_UTC | GPS_DELETE_HEALTH);
            traversal_path_back(&g_traversal);
            break;
        case UI_GPS_DELETE_AIDING_TYPE_FULL_START:
            LOGD("Full Start");
            mnldinf_ext_gnss_delete_aiding_data(g_mnld_ext_fd, GPS_DELETE_ALL);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

agps_intf_agps_config ui_agps_settings_read() {
    agps_intf_agps_config config = get_agps_config_v24();
    LOGD("Read AGPS Settings (V24): %s", config.valid? "Success": "Failure");
    dump_agps_intf_agps_config(config);
    return config;
}

char* get_agps_protocol() {
    switch(g_config.agps_setting.agps_protocol) {
        case 0:
            return "User Plane";
        case 1:
            return "Control Plane";
        default:
            return "Unknown value";
    }
}

char* get_cdma_preferred() {
    switch(g_config.up_setting.cdma_preferred) {
        case 0:
            return "WCDMA prefer";
        case 2:
            return "CDMA Force";
        default:
            return "Unknown value";
    }
}

char* get_pref_method() {
    switch(g_config.up_setting.pref_method) {
        case 0:
            return "MSA";
        case 1:
            return "MSB";
        case 2:
            return "No prefer";
        default:
            return "Unknown value";
    }
}

char* get_supl_version() {
    switch(g_config.up_setting.supl_version) {
        case 1:
            return "SUPL1.0";
        case 2:
            return "SUPL2.0";
        default:
            return "Unknown value";
    }
}

char* get_tls_version() {
    switch(g_config.up_setting.tls_version) {
        case 0:
            return "1_0";
        case 1:
            return "1_1";
        case 2:
            return "2_0";
        default:
            return "Unknown value";
    }
}

char* get_ip_version_prefer() {
    switch(g_config.up_setting.ip_version_prefer) {
        case 0:
            return "IP V6";
        case 1:
            return "IP V4";
        default:
            return "Unknown value";
    }
}

char* get_molr_pos_method() {
    switch(g_config.cp_setting.molr_pos_method) {
        case 0:
            return "Location Estimate";
        case 1:
            return "Assistance Data";
        default:
            return "Unknown value";
    }
}

char* get_enabled_status(int value) {
    switch(value) {
        case 0:
            return "Disabled";
        case 1:
            return "Enabled";
        default:
            return "Unknown value";
    }
}

char* get_disabled_status(int value) {
    switch(value) {
        case 0:
            return "Enabled";
        case 1:
            return "Disabled";
        default:
            return "Unknown value";
    }
}
void ui_agps_settings() {
    g_config = ui_agps_settings_read();
    LOGD("=== AGPS Settings ====");
    LOGD("0.   Back");
    LOGD(" ");
    LOGD("1.   Read AGPS Settings");
    LOGD(" ");
    LOGD("2.   Set AGPS Enable: [%d]", g_config.agps_setting.agps_enable);
    LOGD("3.   UP/CP switching (SI only): [%s]", get_agps_protocol());
    LOGD("4.   CDMA Settings: [%s]", get_cdma_preferred());
    LOGD("5.   Set SUPL Prefer Settings: [%s]", get_pref_method());
    LOGD("6.   UP Pos Technology MSA Enable: [%d]", g_config.up_setting.msa_enable);
    LOGD("7.   UP Pos Technology MSB Enable: [%d]", g_config.up_setting.msb_enable);
    LOGD("8.   UP Pos Technology ECID Enable: [%d]", g_config.up_setting.ecid_enable);
    LOGD("9.   UP Pos Technology Autonomus Enable: [%d]", g_config.up_setting.autonomous_enable);
    LOGD("10.  UP Pos Technology AFLT Enable: [%d]", g_config.up_setting.aflt_enable);
    LOGD("11.  UP Pos Protocol LPP Enable: [%d]", g_config.up_setting.lpp_enable);
    LOGD("12.  SUPL Version: [%s]", get_supl_version());
    LOGD("13.  Set SUPL TLS Version: [%s]", get_tls_version());
    LOGD("14.  Certification Verification Enable: [%d]", g_config.up_setting.ca_enable);
    LOGD("15.  UDP Enable: [%d]", g_config.up_setting.udp_enable);
    LOGD("16.  Set SUPL UT2 Timer: [%d]", g_config.up_setting.ut2);
    LOGD("17.  Set SUPL UT3 Timer: [%d]", g_config.up_setting.ut3);
    LOGD("18.  Set SUPL VER Miner: [%d]", g_config.up_setting.supl_ver_minor);
    LOGD("19.  Set SUPL VER Serind: [%d]", g_config.up_setting.supl_ver_ser_ind);
    LOGD("20.  Set SUPL Profile: SLP Address: [%s] SLP Port: [%d] TLS Enable: [%d]",
            g_config.cur_supl_profile.addr,
            g_config.cur_supl_profile.port,
            g_config.cur_supl_profile.tls);
    LOGD("21.  Periodic Settings");
    LOGD("22.  Area Settings");
    LOGD("23.  SUPL SI QOP Settings");
    LOGD("24.  Allow Network Initiated Request: [%d]", g_config.up_setting.ni_request);
    LOGD("25.  Allow Roaming: [%d]", g_config.up_setting.roaming);
    LOGD("26.  CP Auto Reset: [%d]", g_config.cp_setting.cp_auto_reset);
    LOGD("27.  Log SUPL to File: [%d]", g_config.up_setting.supl_log);
    LOGD("28.  Log Sensitive Data: [%d]", !g_config.up_setting.no_sensitive_log);
    LOGD("29.  Allow AGPS Certificates for Lab Test: [%d]", g_config.up_setting.cert_from_sdcard);
    LOGD("30.  Allow Auto Configuring SUPL Profile Base on Current Plmn: [%d]", g_config.up_setting.auto_profile_enable);
    LOGD("31.  Allow SUPL dedicated APN: [%d]", g_config.up_setting.apn_enable);
    LOGD("32.  Sync AGPS Settings to SLPD: [%d]", g_config.up_setting.sync_to_slp);
    LOGD("33.  Allow valid IMSI in SUPL messages: [%d]", g_config.up_setting.imsi_enable);
    LOGD("34.  Show GPS Icon during Network Initiated Session: [%d]", g_config.agps_setting.e911_gps_icon_enable);
    LOGD("35.  LPPe Disable Crowd-Source Location: [%d]", g_config.agps_setting.lppe_network_location_disable);
    LOGD("36.  LPPe CP Enable: [%d]", g_config.cp_setting.cp_lppe_enable);
    LOGD("37.  LPPe UP Enable: [%d]", g_config.up_setting.up_lppe_enable);
    LOGD("38.  AOSP Profiling Enable: [%d]", g_config.up_setting.aosp_profile_enable);
    LOGD("39.  Use NLP Settings to Supl: [%d]", g_config.up_setting.bind_nlp_setting_to_supl);
    LOGD("40.  LBS Log Enable: [%d]", g_config.agps_setting.lbs_log_enable);
    LOGD("41.  Ingore SI for E911: [%d]", g_config.agps_setting.ignore_si_for_e911);
    LOGD("42.  LPPe CP Wlan Enable: [%d]", g_config.cp_setting.cp_lppe_wlan_enable);
    LOGD("43.  LPPe CP SRN Enable: [%d]", g_config.cp_setting.cp_lppe_srn_enable);
    LOGD("44.  LPPe CP Sensor Enable: [%d]", g_config.cp_setting.cp_lppe_sensor_enable);
    LOGD("45.  LPPe CP DBH Enable: [%d]", g_config.cp_setting.cp_lppe_dbh_enable);
    LOGD("46.  LPPe UP Wlan Enable: [%d]", g_config.up_setting.up_lppe_wlan_enable);
    LOGD("47.  LPPe UP SRN Enable: [%d]", g_config.up_setting.up_lppe_srn_enable);
    LOGD("48.  LPPe UP Sensor Enable: [%d]", g_config.up_setting.up_lppe_sensor_enable);
    LOGD("49.  LPPe UP DBH Enable: [%d]", g_config.up_setting.up_lppe_dbh_enable);
    LOGD("50.  UP LPP in 2G/3G Disable: [%d]", g_config.up_setting.up_lppe_in_2g3g_disable);
    LOGD("51.  UP RRLP in 4G Disable: [%d]", g_config.up_setting.up_rrlp_in_4g_disable);
    LOGD("52.  UP SI Disable: [%d]", g_config.up_setting.up_si_disable);
    LOGD("53.  LPPe Default NLP Location Enable: [%d]", g_config.agps_setting.lppe_def_nlp_enable);
    LOGD("54.  Force OTDOA Assist Enable: [%d]", g_config.up_setting.force_otdoa_assist_req);
    LOGD("55.  AOSP Position Mode Enable: [%d]", g_config.up_setting.aosp_pos_mode_enable);
    LOGD("56.  Privacy Override Enable: [%d]", g_config.up_setting.privacy_override_mode);
    LOGD("57.  Allow Ni For GPS OFF Enable: [%d]", g_config.up_setting.allow_ni_for_gps_off);
    LOGD("58.  IP Version Prefer: [%s]", get_ip_version_prefer());
    LOGD("59.  TCP Keep Alive Settings: [%d]", g_config.up_setting.tcp_keepalive);
    LOGD("60.  LPPe Crowd Source Confident: [%d]", g_config.agps_setting.lppe_crowd_source_confident);
    LOGD("61.  SIB8 SIB16 Enable: [%d]", g_config.gnss_setting.sib8_sib16_enable);
    LOGD("62.  A-GLONASS Enable: [%d]", g_config.gnss_setting.a_glonass_satellite_enable);
    LOGD("63.  A-Beidou Enable: [%d]", g_config.gnss_setting.a_beidou_satellite_enable);
    LOGD("64.  A-Galileo Enable: [%d]", g_config.gnss_setting.a_galileo_satellite_enable);
    LOGD(" ");
    LOGD(" ");
    LOGD("100. MOLR PosMethod: [%s]", get_molr_pos_method());
    LOGD("101. Set CP External Address: [%s] [%s]", get_enabled_status(g_config.cp_setting.external_addr_enable), g_config.cp_setting.external_addr);
    LOGD("102. Set CP MLC number: [%s] [%s]", get_enabled_status(g_config.cp_setting.mlc_number_enable), g_config.cp_setting.mlc_number);
    LOGD("103. Set CP EPC MOLR PDU Enable: [%d]", g_config.cp_setting.epc_molr_lpp_payload_enable);
    LOGD(" ");
    LOGD("999.  Reset To Default");

    LOGD("input: ");
    ui_agps_settings_type input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case UI_AGPS_SETTINGS_TYPE_BACK:
            traversal_path_back(&g_traversal);
            break;
        case UI_AGPS_SETTINGS_TYPE_GET_AGPS_SETTINGS:
            ui_agps_settings_read();
            break;
        case UI_AGPS_SETTINGS_TYPE_SET_AGPS_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_AGPS_PROTOCOL:
        case UI_AGPS_SETTINGS_TYPE_SET_CDMA_PREFERENCE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_PREFER_METHOD:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_MSA_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_MSB_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_ECID_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_AUTONOMOUS_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_AFLT_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_LPP_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VERSION:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_TLS_VERSION:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_CA_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_UDP_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_UT2:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_UT3:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VER_MINOR:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VER_SER_IND:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_PROFILE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_PERIODIC_SETTINGS:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_AREA_SETTINGS:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SI_QOP_SETTINGS:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_ALLOW_NI_REQUEST:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_ALLOW_ROAMING:
        case UI_AGPS_SETTINGS_TYPE_SET_CP_AUTO_RESET:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_LOG_TO_FILE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SENSITIVE_LOG:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_CERT_FROM_SDCARD:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_AUTO_PROFILE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_DEDICATE_APN:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_SYNC_TO_SLP:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_IMSI_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_E911_GPS_ICON:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_LPPE_NETWORK_LOCATION_DISABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_CP_LPPE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_UP_LPPE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_AOSP_PROFILE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_BIND_NLP_SETTING_TO_SUPL:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_LBS_LOG_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_INGORE_SI_FOR_E911:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_WLAN:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_SRN:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_SENSOR:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_DBH:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_WLAN:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_SRN:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_SENSOR:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_DBH:
        case UI_AGPS_SETTINGS_TYPE_SET_UP_LPP_IN_2G_3G_DISABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_UP_RRLP_IN_4G_DISABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_UP_SI_DISABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_DEFAULT_NLP_LOCATION_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_FORCE_OTDOA_ASSIST_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_AOSP_POSITION_MODE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_PRIVACY_OVERRIDE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_ALLOW_NI_FOR_GPS_OFF_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_IP_VERSION_PREFER:
        case UI_AGPS_SETTINGS_TYPE_SET_SUPL_TCP_KEEPALIVE:
        case UI_AGPS_SETTINGS_TYPE_SET_LPPE_CROWD_SOURCE_CONFIDENT:
        case UI_AGPS_SETTINGS_TYPE_SET_GNSS_SIB8_SIB16_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_GLONASS_SATELLITE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_BEIDOU_SATELLITE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_GALILEO_SATELLITE_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_CP_MOLR_POS_METHOD:
        case UI_AGPS_SETTINGS_TYPE_SET_CP_EXTERNAL_ADDR_ENABLE:
        case UI_AGPS_SETTINGS_TYPE_SET_CP_MLC_NUMBER:
        case UI_AGPS_SETTINGS_TYPE_SET_CP_EPC_MOLR_LPP_PAYLOAD_ENABLE:
            traversal_path_go(&g_traversal, input);
            break;
        case UI_AGPS_SETTINGS_TYPE_RESET_TO_DEFAULT:
            LOGD("Reset To Default");
            set_reset_to_default();
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_agps_enable() {
    LOGD("=== Set AGPS Enable ====");
    LOGD("Current value: [%d]", g_config.agps_setting.agps_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Set AGPS Enable");
    LOGD("2.  Set AGPS Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Set AGPS Enable");
            set_agps_enabled(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Set AGPS Disable");
            set_agps_enabled(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_agps_protocol() {
    LOGD("=== UP/CP switching (SI only) ====");
    LOGD("Current Value: [%s]", get_agps_protocol());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Set AGPS Protocol to User Plane");
    LOGD("2.  Set AGPS Protocol to Control Plane");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Set AGPS Protocol to User Plane");
            set_protocol(AGPS_INTF_AGPS_PROTOCOL_UP);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Set AGPS Protocol to Control Plane");
            set_protocol(AGPS_INTF_AGPS_PROTOCOL_CP);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_cdma_preference() {
    LOGD("=== CDMA Settings ====");
    LOGD("Current Value: [%s]", get_cdma_preferred());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Set WCDMA Prefer");
    LOGD("2.  Set CDMA Force");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Set WCDMA Prefer");
            set_cdma_pref(AGPS_INTF_CDMA_PREFERRED_WCDMA);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Set CDMA Force");
            set_cdma_pref(AGPS_INTF_CDMA_PREFERRED_CDMA_FORCE);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_prefer_method() {
    LOGD("=== Set SUPL Prefer Settings ====");
    LOGD("Current Value: [%s]", get_pref_method());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Set MSA Prefer");
    LOGD("2.  Set MSB Prefer");
    LOGD("3.  Set NO Prefer");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Set MSA Prefer");
            set_up_pref_method(AGPS_INTF_PREF_METHOD_MSA);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Set MSB Prefer");
            set_up_pref_method(AGPS_INTF_PREF_METHOD_MSB);
            traversal_path_back(&g_traversal);
            break;
        case 3:
            LOGD("Set NO Prefer");
            set_up_pref_method(AGPS_INTF_PREF_METHOD_NO_PREF);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_pos_tech_msa_enable() {
    LOGD("=== UP Pos Technology MSA Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.msa_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP Pos Technology MSA Enable");
    LOGD("2.  UP Pos Technology MSA Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP Pos Technology MSA Enable");
            set_pos_technology_msa(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP Pos Technology MSA Disable");
            set_pos_technology_msa(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_pos_tech_msb_enable() {
    LOGD("=== UP Pos Technology MSB Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.msb_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP Pos Technology MSB Enable");
    LOGD("2.  UP Pos Technology MSB Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP Pos Technology MSB Enable");
            set_pos_technology_msb(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP Pos Technology MSB Disable");
            set_pos_technology_msb(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_pos_tech_ecid_enable() {
    LOGD("=== UP Pos Technology ECID Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.ecid_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP Pos Technology ECID Enable");
    LOGD("2.  UP Pos Technology ECID Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP Pos Technology ECID Enable");
            set_pos_technology_ecid(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP Pos Technology ECID Disable");
            set_pos_technology_ecid(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_pos_tech_autonomous_enable() {
    LOGD("=== UP Pos Technology AUTONOMOUS Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.autonomous_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP Pos Technology AUTONOMOUS Enable");
    LOGD("2.  UP Pos Technology AUTONOMOUS Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP Pos Technology AUTONOMOUS Enable");
            set_pos_technology_autonomous(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP Pos Technology AUTONOMOUS Disable");
            set_pos_technology_autonomous(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_pos_tech_aflt_enable() {
    LOGD("=== UP Pos Technology AFLT Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.aflt_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP Pos Technology AFLT Enable");
    LOGD("2.  UP Pos Technology AFLT Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP Pos Technology AFLT Enable");
            set_pos_technology_aflt(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP Pos Technology AFLT Disable");
            set_pos_technology_aflt(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_lpp_enable() {
    LOGD("=== UP Pos Protocol LPP Settings ====");
    LOGD("Current Value: [%d]", g_config.up_setting.lpp_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP Pos Protocol LPP Enable");
    LOGD("2.  UP Pos Protocol LPP Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP Pos Protocol LPP Enable");
            set_lpp_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP Pos Protocol LPP Disable");
            set_lpp_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_version() {
    LOGD("=== SUPL Version ====");
    LOGD("Current Value: [%s]", get_supl_version());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  SUPL Version 1.0");
    LOGD("2.  SUPL Version 2.0");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("SUPL Version 1.0");
            set_supl_version(AGPS_INTF_SUPL_VERSION_1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("SUPL Version 2.0");
            set_supl_version(AGPS_INTF_SUPL_VERSION_2);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_tls_version() {
    LOGD("=== TLS Version ====");
    LOGD("Current Value: [%s]", get_tls_version());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  TLS Version - TLS 1.0");
    LOGD("2.  TLS Version - TLS 1.1");
    LOGD("3.  TLS Version - TLS 1.2");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("TLS Version - TLS 1.0");
            set_supl_tls_version(0);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("TLS Version - TLS 1.1");
            set_supl_tls_version(1);
            traversal_path_back(&g_traversal);
            break;
        case 3:
            LOGD("TLS Version - TLS 1.2");
            set_supl_tls_version(2);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_profile() {
    int ret;
    char supl_host[128] = {0};
    int supl_port = 0;
    int supl_tls = 0;
    LOGD("=== Set SUPL Profile ====");

    LOGD("SUPL Host: ");
    ret = scanf("%s", supl_host);
    clear_stdin_buff();
    if(ret <= 0) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("SUPL Port: ");
    ret = scanf("%d", &supl_port);
    clear_stdin_buff();
    if(ret <= 0) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("SUPL TLS (0 is disable, otherwise is enable): ");
    ret = scanf("%d", &supl_tls);
    clear_stdin_buff();
    if(ret <= 0) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    if(supl_tls) {
        supl_tls = 1;
    }

    system("clear");
    traversal_path_back(&g_traversal);
    LOGD("SUPL Host=[%s]", supl_host);
    LOGD("SUPL Port=[%d]", supl_port);
    LOGD("SUPL TLS=[%d]", supl_tls);
    LOGD("Set SUPL Profile");
    set_supl_profile(supl_host, supl_port, supl_tls);
}

void ui_agps_settings_supl_periodic_settings() {
    LOGD("=== PERIODIC SETTINGS ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Start Periodic Session");
    LOGD("2.  Stop Periodic Session");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Start Periodic Session");
            start_supl_2_periodic_session();
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Stop Periodic Session");
            stop_supl_2_periodic_session();
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_area_settings() {
    LOGD("=== AREA EVENT SETTINGS ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Start Area Entering Event");
    LOGD("2.  Start Area Inside Event");
    LOGD("3.  Start Area Outside Event");
    LOGD("4.  Start Area Leaving Event");
    LOGD("5.  Stop Area Event");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Start Area Entering Event");
            start_supl_2_area_event_session(agps_intf_area_event_type_entering);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Start Area Inside Event");
            start_supl_2_area_event_session(agps_intf_area_event_type_inside);
            traversal_path_back(&g_traversal);
            break;
        case 3:
            LOGD("Start Area Outside Event");
            start_supl_2_area_event_session(agps_intf_area_event_type_outside);
            traversal_path_back(&g_traversal);
            break;
        case 4:
            LOGD("Start Area Leaving Event");
            start_supl_2_area_event_session(agps_intf_area_event_type_leaving);
            traversal_path_back(&g_traversal);
            break;
        case 5:
            LOGD("Stop Area Event");
            stop_supl_2_area_event_session();
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

int calc_meter_from_k_value(int k) {
    double meter;
    if (k == 1)
        return 1;
    meter = (pow(1.1, k) - 1) *10;
    return (int) ceil(meter);
}

int calc_k_value_from_meter(int meter) {
       double k;
       k = log((double) meter / 10 + 1) / log(1.1);
       return (int) floor(k);
   }

void ui_agps_settings_supl_qop() {
    int horizontal_acc = 0;
    int vertical_acc = 0;
    int location_age = 0;
    int delay = 0;
    LOGD("=== Set SUPL QOP ====");

    LOGD("Current K-Value: Horizontal Accuracy:[%d] Vertical Accuracy:[%d] Location Age:[%d] Delay:[%d]",
            g_config.up_setting.qop_hacc,
            g_config.up_setting.qop_vacc,
            g_config.up_setting.qop_loc_age,
            g_config.up_setting.qop_delay);
    LOGD("(Meter-Value: Horizontal Accuracy:[%d] Vertical Accuracy:[%d] Location Age:[%d] Delay:[%d])",
            calc_meter_from_k_value(g_config.up_setting.qop_hacc),
            calc_meter_from_k_value(g_config.up_setting.qop_vacc),
            g_config.up_setting.qop_loc_age,
            g_config.up_setting.qop_delay);

    LOGD("Select input format (1: K-value 2: Meter): ");
    int format = scanf_int(INVALID_VALUE, false);
    if(format != 1 && format != 2) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    char* sFormat = NULL;
    if (format == 1) {
        sFormat = "K-value";
    } else {
        sFormat = "Meter";
    }

    LOGD("Horizontal Accuracy (%s): ", sFormat);
    horizontal_acc = scanf_int(INVALID_VALUE, false);
    if(horizontal_acc == INVALID_VALUE) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Vertical Accuracy (%s): ", sFormat);
    vertical_acc = scanf_int(INVALID_VALUE, false);
    if(vertical_acc == INVALID_VALUE) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Location Age: ");
    location_age = scanf_int(INVALID_VALUE, false);
    if(location_age == INVALID_VALUE) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Delay: ");
    delay = scanf_int(INVALID_VALUE, false);
    if(delay == INVALID_VALUE) {
        system("clear");
        traversal_path_back(&g_traversal);
        LOGE("ERR: read invalid input");
        return;
    }

    system("clear");
    traversal_path_back(&g_traversal);
    LOGD("Horizontal Accuracy (%s)=[%d]", sFormat, horizontal_acc);
    LOGD("Vertical Accuracy (%s)=[%d]", sFormat, vertical_acc);
    LOGD("Location Age=[%d]", location_age);
    LOGD("Delay=[%d]", delay);
    LOGD("Set SUPL QOP");
    if (format == 2) {
        horizontal_acc = calc_k_value_from_meter(horizontal_acc);
        vertical_acc = calc_k_value_from_meter(vertical_acc);
    }
    set_qop(horizontal_acc, vertical_acc, location_age, delay);
}

void ui_agps_settings_supl_ca_enable() {
    LOGD("=== Certification Verification Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.ca_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Certification Verification Enable");
    LOGD("2.  Certification Verification Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Certification Verification Enable");
            set_cert_verify(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Certification Verification Disable");
            set_cert_verify(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_udp_enable() {
    LOGD("=== UDP Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.udp_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UDP Enable");
    LOGD("2.  UDP Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UDP Enable");
            set_udp_enable_v2(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UDP Disable");
            set_udp_enable_v2(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_ut2() {
    LOGD("=== Change SUPL UT2 Timer ====");
    LOGD("Current Value: [%d]", g_config.up_setting.ut2);
    LOGD("input integer: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Change SUPL UT2 Timer=[%d]", input);
    traversal_path_back(&g_traversal);
    set_supl_ut2(input);
}

void ui_agps_settings_supl_ut3() {
    LOGD("=== Change SUPL UT3 Timer ====");
    LOGD("Current Value: [%d]", g_config.up_setting.ut3);
    LOGD("input integer: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Change SUPL UT3 Timer=[%d]", input);
    traversal_path_back(&g_traversal);
    set_supl_ut3(input);
}

void ui_agps_settings_supl_ver_minor() {
    LOGD("=== Change SUPL VER MINOR ====");
    LOGD("Current Value: [%d]", g_config.up_setting.supl_ver_minor);
    LOGD("input integer: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Change SUPL VER MINOR=[%d]", input);
    traversal_path_back(&g_traversal);
    set_supl_ver_minor(input);
}

void ui_agps_settings_supl_ver_ser_ind() {
    LOGD("=== Change SUPL SERIND ====");
    LOGD("Current Value: [%d]", g_config.up_setting.supl_ver_ser_ind);
    LOGD("input integer: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("Change SUPL VER SERIND=[%d]", input);
    traversal_path_back(&g_traversal);
    set_supl_ver_ser_ind(input);
}

void ui_agps_settings_supl_ni_request() {
    LOGD("=== Allow Network Initiated Request ====");
    LOGD("Current Value: [%d]", g_config.up_setting.ni_request);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow Network Initiated Request Enable");
    LOGD("2.  Allow Network Initiated Request Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow Network Initiated Request Enable");
            set_allow_ni(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow Network Initiated Request Disable");
            set_allow_ni(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_allow_roaming() {
    LOGD("=== Allow Roaming ====");
    LOGD("Current Value: [%d]", g_config.up_setting.roaming);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow Roaming Enable");
    LOGD("2.  Allow Roaming Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow Roaming Enable");
            set_allow_roaming(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow Roaming Disable");
            set_allow_roaming(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_cp_molr_pos_method() {
    LOGD("=== CP MOLR PosMethod ====");
    LOGD("Current Value: [%s]", get_molr_pos_method());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Set Location Estimate");
    LOGD("2.  Set Assistance Data");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Set Location Estimate");
            set_molr_pos_methd(AGPS_INTF_MOLR_POS_METHOD_LOC_EST);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Set Assistance Data");
            set_molr_pos_methd(AGPS_INTF_MOLR_POS_METHOD_ASSIST_DATA);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_cp_external_addr_enable() {
    LOGD("=== External Addr ====");
    LOGD("Current Value: [%s] [%s]", get_enabled_status(g_config.cp_setting.external_addr_enable), g_config.cp_setting.external_addr);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  External Addr Enable");
    LOGD("2.  External Addr Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, false);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            {
                char external_addr[128] = {0};
                LOGD("External Addr: ");
                int ret = scanf("%s", external_addr);
                clear_stdin_buff();
                if(ret <= 0) {
                    system("clear");
                    traversal_path_back(&g_traversal);
                    LOGE("ERR: read invalid input");
                    return;
                }
                set_external_addr(1,external_addr);
                traversal_path_back(&g_traversal);
            }
            break;
        case 2:
            LOGD("External Addr Disable");
            set_external_addr(0, g_config.cp_setting.external_addr);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_cp_mlc_number() {
    LOGD("=== MLC Number ====");
    LOGD("Current Value: [%s] [%s]", get_enabled_status(g_config.cp_setting.mlc_number_enable), g_config.cp_setting.mlc_number);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  MLC Number Enable");
    LOGD("2.  MLC Number Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, false);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            {
                char mlc_number[128] = {0};
                LOGD("MLC Number: ");
                int ret = scanf("%s", mlc_number);
                clear_stdin_buff();
                if(ret <= 0) {
                    system("clear");
                    traversal_path_back(&g_traversal);
                    LOGE("ERR: read invalid input");
                    return;
                }
                set_mlc_number(1,mlc_number);
                traversal_path_back(&g_traversal);
            }
            break;
        case 2:
            LOGD("MLC Number Disable");
            set_mlc_number(0, g_config.cp_setting.mlc_number);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_cp_auto_reset_enable() {
    LOGD("=== CP Auto Reset ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.cp_auto_reset);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  CP Auto Reset Enable");
    LOGD("2.  CP Auto Reset Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("CP Auto Reset Enable");
            set_cp_auto_reset(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("CP Auto Reset Disable");
            set_cp_auto_reset(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_epc_molr_pdu_enable() {
    LOGD("=== EPC MOLR PDU Enable ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.epc_molr_lpp_payload_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  EPC MOLR PDU Enable");
    LOGD("2.  EPC MOLR PDU Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("EPC MOLR PDU Enable");
            set_epc_molr_pdu_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("EPC MOLR PDU Disable");
            set_epc_molr_pdu_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_log_to_file() {
    LOGD("=== Log SUPL To File ====");
    LOGD("Current Value: [%d]", g_config.up_setting.supl_log);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Log SUPL To File Enable");
    LOGD("2.  Log SUPL To File Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Log SUPL To File Enable");
            set_supl_to_file(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Log SUPL To File Disable");
            set_supl_to_file(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_sensitive_log() {
    LOGD("=== Log Sensitive Data ====");
    LOGD("Current Value: [%d]", !g_config.up_setting.no_sensitive_log);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Log Sensitive Data Enable");
    LOGD("2.  Log Sensitive Data Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Log Sensitive Data Enable");
            set_no_sensitive_log(0);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Log Sensitive Data Disable");
            set_no_sensitive_log(1);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}
void ui_agps_settings_supl_cert_from_sdcard() {
    LOGD("=== Allow AGPS Certificates for Lab Test ====");
    LOGD("Current Value: [%d]", g_config.up_setting.cert_from_sdcard);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow AGPS Certificates for Lab Test Enable");
    LOGD("2.  Allow AGPS Certificates for Lab Test Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow AGPS Certificates for Lab Test Enable");
            set_cert_from_sdcard(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow AGPS Certificates for Lab Test Disable");
            set_cert_from_sdcard(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_auto_profile() {
    LOGD("=== Allow Auto Configuring SUPL Profile Base on Current Plmn ====");
    LOGD("Current Value: [%d]", g_config.up_setting.auto_profile_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow Auto Configuring SUPL Profile Base on Current Plmn Enable");
    LOGD("2.  Allow Auto Configuring SUPL Profile Base on Current Plmn Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow Auto Configuring SUPL Profile Base on Current Plmn Enable");
            set_auto_profile_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow Auto Configuring SUPL Profile Base on Current Plmn Disable");
            set_auto_profile_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_dedicate_apn() {
    LOGD("=== Set SUPL Dedicate APN ====");
    LOGD("Current Value: [%d]", g_config.up_setting.apn_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow SUPL Dedicate APN Enable");
    LOGD("2.  Allow SUPL Dedicate APN Didable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow SUPL Dedicate APN Enable");
            set_supl_apn_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow SUPL Dedicate APN Disable");
            set_supl_apn_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_sync_to_slp() {
    LOGD("=== Sync AGPS Settings to SLPD ====");
    LOGD("Current Value: [%d]", g_config.up_setting.sync_to_slp);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Sync AGPS Settings to SLPD Enable");
    LOGD("2.  Sync AGPS Settings to SLPD Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Sync AGPS Settings to SLPD Enable");
            set_sync_to_slp_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Sync AGPS Settings to SLPD Disable");
            set_sync_to_slp_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_imsi_enable() {
    LOGD("=== Allow valid IMSI in SUPL message ====");
    LOGD("Current Value: [%d]", g_config.up_setting.imsi_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow valid IMSI in SUPL message Enable");
    LOGD("2.  Allow valid IMSI in SUPL message Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow valid IMSI in SUPL message Enable");
            set_imsi_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow valid IMSI in SUPL message Disable");
            set_imsi_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_e911_gps_icon() {
    LOGD("=== Show GPS Icon during Network Initial Session ====");
    LOGD("Current Value: [%d]", g_config.agps_setting.e911_gps_icon_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Show GPS Icon during Network Initial Session Enable");
    LOGD("2.  Show GPS Icon during Network Initial Session Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Show GPS Icon during Network Initial Session Enable");
            set_e911_gps_icon_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Show GPS Icon during Network Initial Session Disable");
            set_e911_gps_icon_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_lppe_nlp_disable() {
    LOGD("=== Disable LPPe Crowd-Source Location ====");
    LOGD("Current Value: [%d]", g_config.agps_setting.lppe_network_location_disable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Disable LPPe Crowd-Source Location");
    LOGD("2.  Enable LPPe Crowd-Source Location");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Disable LPPe Crowd-Source Location");
            set_lppe_network_location_disable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Enable LPPe Crowd-Source Location");
            set_lppe_network_location_disable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_cp_lppe_enable() {
    LOGD("=== LPPe CP Enable ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.cp_lppe_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe CP Enable");
    LOGD("2.  LPPe CP Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe CP Enable");
            set_lppe_cp_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe CP Disable");
            set_lppe_cp_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_up_lppe_enable() {
    LOGD("=== LPPe UP Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.up_lppe_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe UP Enable");
    LOGD("2.  LPPe UP Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe UP Enable");
            set_lppe_up_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe UP Disable");
            set_lppe_up_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_aosp_profile_enable() {
    LOGD("=== AOSP Profiling Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.aosp_profile_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  AOSP Profiling Enable");
    LOGD("2.  AOSP Profiling Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("AOSP Profiling Enable");
            set_aosp_profile_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("AOSP Profiling Disable");
            set_aosp_profile_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_bind_nlp_settings_to_supl() {
    LOGD("=== Use NLP Settings to Supl ====");
    LOGD("Current Value: [%d]", g_config.up_setting.bind_nlp_setting_to_supl);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Use NLP Settings to Supl Enable");
    LOGD("2.  Use NLP Settings to Supl Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Use NLP Settings to Supl Enable");
            set_bind_nlp_setting_to_supl(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Use NLP Settings to Supl Disable");
            set_bind_nlp_setting_to_supl(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_supl_lbs_log_enable() {
    LOGD("=== LBS log Enable ====");
    LOGD("Current Value: [%d]", g_config.agps_setting.lbs_log_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LBS log Enable");
    LOGD("2.  LBS log Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LBS log Enable");
            set_lbs_log_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LBS log Disable");
            set_lbs_log_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_ignore_si_for_e911() {
    LOGD("=== Ingore SI for E911 ====");
    LOGD("Current Value: [%d]", g_config.agps_setting.ignore_si_for_e911);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Ingore SI for E911 Enable");
    LOGD("2.  Ingore SI for E911 Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Ingore SI for E911 Enable");
            set_ingore_si_for_e911(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Ingore SI for E911 Disable");
            set_ingore_si_for_e911(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_cp_wlan() {
    LOGD("=== LPPe CP Wlan Enable ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.cp_lppe_wlan_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe CP Wlan Enable");
    LOGD("2.  LPPe CP Wlan Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe CP Wlan Enable");
            set_lppe_cp_wlan_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe CP Wlan Disable");
            set_lppe_cp_wlan_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_cp_srn() {
    LOGD("=== LPPe CP SRN Enable ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.cp_lppe_srn_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe CP SRN Enable");
    LOGD("2.  LPPe CP SRN Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe CP SRN Enable");
            set_lppe_cp_srn_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe CP SRN Disable");
            set_lppe_cp_srn_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_cp_sensor() {
    LOGD("=== LPPe CP Sensor Enable ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.cp_lppe_sensor_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe CP Sensor Enable");
    LOGD("2.  LPPe CP Sensor Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe CP Sensor Enable");
            set_lppe_cp_sensor_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe CP Sensor Disable");
            set_lppe_cp_sensor_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_cp_dbh() {
    LOGD("=== LPPe CP DBH Enable ====");
    LOGD("Current Value: [%d]", g_config.cp_setting.cp_lppe_dbh_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe CP DBH Enable");
    LOGD("2.  LPPe CP DBH Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe CP DBH Enable");
            set_lppe_cp_dbh_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe CP DBH Disable");
            set_lppe_cp_dbh_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_up_wlan() {
    LOGD("=== LPPe UP Wlan Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.up_lppe_wlan_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe UP Wlan Enable");
    LOGD("2.  LPPe UP Wlan Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe UP Wlan Enable");
            set_lppe_up_wlan_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe UP Wlan Disable");
            set_lppe_up_wlan_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_up_srn() {
    LOGD("=== LPPe UP SRN Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.up_lppe_srn_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe UP SRN Enable");
    LOGD("2.  LPPe UP SRN Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe UP SRN Enable");
            set_lppe_up_srn_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe UP SRN Disable");
            set_lppe_up_srn_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_up_sensor() {
    LOGD("=== LPPe UP Sensor Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.up_lppe_sensor_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe UP Sensor Enable");
    LOGD("2.  LPPe UP Sensor Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe UP Sensor Enable");
            set_lppe_up_sensor_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe UP Sensor Disable");
            set_lppe_up_sensor_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}


void ui_agps_settings_lppe_up_dbh() {
    LOGD("=== LPPe UP DBH Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.up_lppe_dbh_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe UP DBH Enable");
    LOGD("2.  LPPe UP DBH Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe UP DBH Enable");
            set_lppe_up_dbh_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe UP DBH Disable");
            set_lppe_up_dbh_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_up_lppe_2g_3g_disable() {
    LOGD("=== UP LPP in 2G/3G Disable ====");
    LOGD("Current Value: [%s]", get_disabled_status(g_config.up_setting.up_lppe_in_2g3g_disable));
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP LPP in 2G/3G Disable");
    LOGD("2.  UP LPP in 2G/3G Enable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP LPP in 2G/3G Disable");
            set_up_lpp_in_2g3g_disable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP LPP in 2G/3G Enable");
            set_up_lpp_in_2g3g_disable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_up_rrlp_in_4g_disable() {
    LOGD("=== UP RRLP in 4G Disable ====");
    LOGD("Current Value: [%s]", get_disabled_status(g_config.up_setting.up_rrlp_in_4g_disable));
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP RRLP in 4G Disable");
    LOGD("2.  UP RRLP in 4G Enable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP RRLP in 4G Disable");
            set_up_rrlp_in_4g_disable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP RRLP in 4G Enable");
            set_up_rrlp_in_4g_disable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_up_si_disable() {
    LOGD("=== UP SI Disable ====");
    LOGD("Current Value: [%s]", get_disabled_status(g_config.up_setting.up_si_disable));
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  UP SI Disable");
    LOGD("2.  UP SI Enable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("UP SI Disable");
            set_up_si_disable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("UP SI Enable");
            set_up_si_disable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_lppe_default_nlp_location() {
    LOGD("=== LPPe Default NLP Location Enable ====");
    LOGD("Current Value: [%d]", g_config.agps_setting.lppe_def_nlp_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  LPPe Default NLP Location Enable");
    LOGD("2.  LPPe Default NLP Location Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("LPPe Default NLP Location Enable");
            set_lppe_def_nlp_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("LPPe Default NLP Location Disable");
            set_lppe_def_nlp_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_force_otdoa_assist() {
    LOGD("=== Force OTDOA Assist Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.force_otdoa_assist_req);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Force OTDOA Assist Enable");
    LOGD("2.  Force OTDOA Assist Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Force OTDOA Assist Enable");
            set_force_otdoa_assist_req(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Force OTDOA Assist Disable");
            set_force_otdoa_assist_req(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_up_aosp_pos_mode() {
    LOGD("=== AOSP Position Mode Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.aosp_pos_mode_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  AOSP Position Mode Enable");
    LOGD("2.  AOSP Position Mode Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("AOSP Position Mode Enable");
            set_up_aosp_pos_mode_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("AOSP Position Mode Disable");
            set_up_aosp_pos_mode_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_privacy_override() {
    LOGD("=== Privacy Override Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.privacy_override_mode);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Privacy Override Enable");
    LOGD("2.  Privacy Override Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Privacy Override Enable");
            set_privacy_override_mode(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Privacy Override Disable");
            set_privacy_override_mode(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_allow_ni_for_gps_off() {
    LOGD("=== Allow Ni For GPS OFF Enable ====");
    LOGD("Current Value: [%d]", g_config.up_setting.allow_ni_for_gps_off);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Allow Ni For GPS OFF Enable");
    LOGD("2.  Allow Ni For GPS OFF Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("Allow Ni For GPS OFF Enable");
            set_allow_ni_for_gps_off(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("Allow Ni For GPS OFF Disable");
            set_allow_ni_for_gps_off(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_ip_version_prefer() {
    LOGD("=== IP Version Prefer ====");
    LOGD("Current Value: [%s]", get_ip_version_prefer());
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  IP V6");
    LOGD("2.  IP V4");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("IP V6");
            set_ip_version_prefer(AGPS_INTF_SUPL_IP_VERSION_IPV6);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("IP V4");
            set_ip_version_prefer(AGPS_INTF_SUPL_IP_VERSION_IPV4);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}



void ui_agps_settings_supl_tcp_keepalive() {
    LOGD("=== TCP Keep Alive ====");
    LOGD("Current Value: [%d]", g_config.up_setting.tcp_keepalive);
    LOGD("input integer: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("TCP Keep Alive=[%d]", input);
    traversal_path_back(&g_traversal);
    set_tcp_keep_alive(input);
}

void ui_agps_settings_lppe_crowd_source_confident() {
    LOGD("=== LPPe Crowd Source Confident ====");
    LOGD("Current Value: [%d]", g_config.agps_setting.lppe_crowd_source_confident);
    LOGD("input integer: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    LOGD("LPPe Crowd Source Confident=[%d]", input);
    traversal_path_back(&g_traversal);
    set_lppe_crowd_source_confident(input);
}


void ui_agps_settings_gnss_sib8_sib16_enable() {
    LOGD("=== SIB8/SIB16 Enable ====");
    LOGD("Current Value: [%d]", g_config.gnss_setting.sib8_sib16_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  SIB8/SIB16 Enable");
    LOGD("2.  SIB8/SIB16 Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("SIB8/SIB16 Enable");
            set_sib8_16_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("SIB8/SIB16 Disable");
            set_sib8_16_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_gnss_a_glonass_enable() {
    LOGD("=== A-GLONASS Enable ====");
    LOGD("Current Value: [%d]", g_config.gnss_setting.a_glonass_satellite_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  A-GLONASS Enable");
    LOGD("2.  A-GLONASS Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("A-GLONASS Enable");
            set_a_glonass_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("A-GLONASS Disable");
            set_a_glonass_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_gnss_a_beidou_enable() {
    LOGD("=== A-Beidou Enable ====");
    LOGD("Current Value: [%d]", g_config.gnss_setting.a_beidou_satellite_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  A-Beidou Enable");
    LOGD("2.  A-Beidou Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("A-Beidou Enable");
            set_a_beidou_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("A-Beidou Disable");
            set_a_beidou_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_agps_settings_gnss_a_galileo_enable() {
    LOGD("=== A-Galileo Enable ====");
    LOGD("Current Value: [%d]", g_config.gnss_setting.a_galileo_satellite_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  A-Galileo Enable");
    LOGD("2.  A-Galileo Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("A-Galileo Enable");
            set_a_galileo_enable(1);
            traversal_path_back(&g_traversal);
            break;
        case 2:
            LOGD("A-Galileo Disable");
            set_a_galileo_enable(0);
            traversal_path_back(&g_traversal);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_debug_settings() {
    LOGD("=== Debug Settings ====");
    LOGD(" AGPS Debug Interface: [%d]", g_agps_debug_enable);
    LOGD(" GPS Information: [%d]", g_gps_info_enable);
    LOGD(" NMEA Information: [%d]", g_nmea_enable);
    LOGD(" GNSS AT Command Information: [%d]", g_at_cmd_enable);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  AGPS Debug Interface Enable");
    LOGD("2.  AGPS Debug Interface Disable");
    LOGD("3.  GPS Information Enable");
    LOGD("4.  GPS Information Disable");
    LOGD("5.  NMEA Information Enable");
    LOGD("6.  NMEA Information Disable");
    LOGD("7.  GNSS AT Command Information Enable");
    LOGD("8.  GNSS AT Command Information Disable");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
            LOGD("AGPS Debug Interface Enable");
            g_agps_debug_enable = 1;
            break;
        case 2:
            LOGD("AGPS Debug Interface Disable");
            g_agps_debug_enable = 0;
            break;
        case 3:
            LOGD("GPS Information Enable");
            g_gps_info_enable = 1;
            break;
        case 4:
            LOGD("GPS Information Disable");
            g_gps_info_enable = 0;
            break;
        case 5:
            LOGD("NMEA Information Enable");
            g_nmea_enable = 1;
            break;
        case 6:
            LOGD("NMEA Information Disable");
            g_nmea_enable = 0;
            break;
        case 7:
            LOGD("GNSS AT Command Information Enable");
            g_at_cmd_enable = 1;
            break;
        case 8:
            LOGD("GNSS AT Command Information Disable");
            g_at_cmd_enable = 0;
            break;
        //TODO test only
        case 11:
            // phased out
            //mnl2hal_mnld_reboot();
            break;
        case 12: {
            gps_location location;
            memset(&location, 0, sizeof(location));
            location.flags = 0xffff;
            location.lat = 1.1;
            location.lng = 2.2;
            location.alt = 3.3;
            location.speed = 4.4;
            location.bearing = 5.5;
            location.h_accuracy = 6.6;
            location.timestamp = 1234567890123456L;
            mnldinf_location_hdlr(location);
            break;
        }
        case 13:
            mnldinf_nmea_hdlr(10000, "$GPGSV,2,1,08,01,40,083,46,02,17,308,41,12,07,344,39,14,22,228,45*75", 250);
            mnldinf_nmea_hdlr(10001, "$GPRMC,123519,A,4807.038,N,01131.000,E,022.4,084.4,230394,003.1,W*6A", 251);
            mnldinf_nmea_hdlr(10002, "$GPGLL,4916.45,N,12311.12,W,225444,A,*1D", 252);
            mnldinf_nmea_hdlr(10003, "$GPVTG,054.7,T,034.4,M,005.5,N,010.2,K*48", 253);
            msleep2(500);
            break;
        case 14: {
            gnss_sv_info sv;
            memset(&sv, 0, sizeof(sv));
            sv.num_svs = 3;

            sv.sv_list[0].svid = 1;
            sv.sv_list[0].constellation = 11;
            sv.sv_list[0].c_n0_dbhz = 31;
            sv.sv_list[0].elevation = 1.1;
            sv.sv_list[0].azimuth = 1.11;
            sv.sv_list[0].flags = 11;

            sv.sv_list[1].svid = 2;
            sv.sv_list[1].constellation = 22;
            sv.sv_list[1].c_n0_dbhz = 32;
            sv.sv_list[1].elevation = 2.2;
            sv.sv_list[1].azimuth = 2.22;
            sv.sv_list[1].flags = 22;

            sv.sv_list[2].svid = 3;
            sv.sv_list[2].constellation = 33;
            sv.sv_list[2].c_n0_dbhz = 33;
            sv.sv_list[2].elevation = 3.3;
            sv.sv_list[2].azimuth = 3.33;
            sv.sv_list[2].flags = 33;

            mnldinf_gnss_sv_hdlr(sv);
            break;
        }
        case 15:
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_debug_1() {
    int emulator_mode = get_emulator_mode();
    LOGD("=== Debug 1 , emulator_mode=[%d] ====", emulator_mode);
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  Reset AGPSD");
    LOGD("2.  Test Button");
    LOGD("3.  Emulator Mode ON");
    LOGD("4.  Emulator Mode OFF");
    LOGD("5.  Start Test Case");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case UI_DEBUG_1_TYPE_BACK:
            traversal_path_back(&g_traversal);
            break;
        case UI_DEBUG_1_TYPE_RESET_AGPSD:
            LOGD("Reset AGPSD");
            do_reset_agpsd();
            msleep2(1000); // give the time for system to re-start AGPSD
            break;
        case UI_DEBUG_1_TYPE_TEST_BUTTON:
            traversal_path_go(&g_traversal, input);
            break;
        case UI_DEBUG_1_TYPE_EMULATOR_MODE_ON:
            LOGD("Emulator Mode ON");
            start_emualator_mode(1);
            msleep2(1000); // give the time for system to re-start AGPSD
            break;
        case UI_DEBUG_1_TYPE_EMULATOR_MODE_OFF:
            LOGD("Emulator Mode OFF");
            start_emualator_mode(0);
            msleep2(1000); // give the time for system to re-start AGPSD
            break;
        case UI_DEBUG_1_TYPE_START_TEST_CASE:
            traversal_path_go(&g_traversal, input);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_debug_1_test_case() {
    LOGD("=== Test Case ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1:  All TC Loop 1");
    LOGD("2:  All TC Loop 2");
    LOGD("3:  All TC Loop 1000");
    LOGD("4:  All Common TC");
    LOGD("5:  All UP TC");
    LOGD("6:  All CP TC");
    LOGD("7:  All EVDO TC");
    LOGD("8:  Partial TC 1");
    LOGD("9:  Partial TC 2");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case 0:
            traversal_path_back(&g_traversal);
            break;
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
            start_test_case(input - 1);
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void ui_debug_1_test_button() {
    LOGD("=== Test Button ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1:  agps_open_gps_req");
    LOGD("2:  agps_close_gps_req");
    LOGD("3:  agps_reset_gps_req");
    LOGD("4:  ni_notify1 notify");
    LOGD("5:  ni_notify1 verify");
    LOGD("6:  ni_notify2 notify");
    LOGD("7:  ni_notify2 verify");
    LOGD("8:  data_conn_req");
    LOGD("9:  data_conn_release");
    LOGD("10:  md_sim_info_req");
    LOGD("11:  md_data_conn_state_req");
    LOGD("12:  lppe_capability");
    LOGD("13:  lppe_start_meas");
    LOGD("14:  lppe_stop_meas");
    LOGD("15:  lppe_query_meas");
    LOGD("16:  agps_md_huge_data");
    LOGD("17:  agps_open_gps_rejected");
    LOGD("18:  ni_notify3 notify");
    LOGD("19:  Test 19");
    LOGD("20:  Test 20");
    LOGD("21:  Test 21");
    LOGD("22:  Test 22");
    LOGD("23:  Test 23");
    LOGD("24:  Test 24");
    LOGD("25:  Test 25");
    LOGD("26:  500 lppe meas stress test");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    if (input == 0) {
        traversal_path_back(&g_traversal);
    } else if (input >0 && input <= 25) {
        start_test_button(input - 1);
    } else if (input == 26) {
        LOGE("Stress test for 500 lppe meas requests");
        int i =0;
        for (i = 0; i< 500; i++) {
            start_test_button(13 - 1); // start meas
            msleep2(5500);
            start_test_button(14 - 1); // stop meas
            msleep2(500);
            start_test_button(15 - 1); // query meas
        }
    } else {
        LOGE("ERR: read unhandled value=[%d]", input);
        return;
    }
}

void at_geofence_add_circle_request() {
    mtk_geo_add_circle c;
    memset(&c, 0, sizeof(c));
    LOGD("Geofence Add Circle Request: Alert Type (1=Enter 2=Exit) input:");
    c.alert_type = scanf_int(INVALID_VALUE, false);
    if(c.alert_type == INVALID_VALUE) {
        LOGE("ERR: read invalid alert_type");
        return;
    }
    LOGD("Geofence Add Circle Request: Initial State (0=Unkown 1=Entered 2=Exited) input:");
    c.initial_state = scanf_int(INVALID_VALUE, false);
    if(c.initial_state == INVALID_VALUE) {
        LOGE("ERR: read invalid initial_state");
        return;
    }
    LOGD("Geofence Add Circle Request: Latitude (xx.xxxxxx in degrees, WGS84 coordinate system) input:");
    c.lat = scanf_double(0, false);
    if(c.lat == 0) {
        LOGE("ERR: read invalid lat");
        return;
    }
    LOGD("Geofence Add Circle Request: Longitude (xx.xxxxxx in degrees, WGS84 coordinate system) input:");
    c.lng = scanf_double(0, false);
    if(c.lng == 0) {
        LOGE("ERR: read invalid lng");
        return;
    }
    LOGD("Geofence Add Circle Request: Radius (in meter) input:");
    c.radius = scanf_double(0, false);
    if(c.radius == 0) {
        LOGE("ERR: read invalid radius");
        return;
    }
    LOGD("Geofence Add Circle Request: Unknown Timer (in millisecond) input:");
    c.unknowntimerms = scanf_int(INVALID_VALUE, false);
    if(c.unknowntimerms == INVALID_VALUE) {
        LOGE("ERR: read invalid unknowntimerms");
        return;
    }

    system("clear");
    mtk_at_send_geo_add_circle_request(g_gnssadp_fd, &c);
    LOGD("Geofence Add Circle Request alert=[%d] initial=[%d] lat=[%.6f] lng=[%.6f] radius=[%.2f] unknowntimerms=[%d] Sent",
        c.alert_type, c.initial_state, c.lat, c.lng, c.radius, c.unknowntimerms);
}
void ui_gnss_at_settings() {
    LOGD("=== GNSS AT Command ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1:  GNSS Start Request");
    LOGD("2:  GNSS Start Request With Wake Lock Release");
    LOGD("3:  GNSS Stop Request");
    LOGD("4:  Host Reboot Notify");
    LOGD("5:  GNSS Status Request");
    LOGD("6:  GNSS Enable Set");
    LOGD("7:  NI Enable Set");
    LOGD("8:  AGPS Mode Set");
    LOGD("9:  SUPL Version Set");
    LOGD("10:  SUPL Address Set");
    LOGD("11: Delete Aiding Data Request");
    LOGD("12: NI Notify Response");
    LOGD("13: Reference Location Response");
    LOGD("14: Reference Time Response");
    LOGD("15: NMEA Config Set");
    LOGD("16: Loopback Request");
    LOGD(" ");
    LOGD("31: Geofence Max Number Request");
    LOGD("32: Geofence Add Circle Request");
    LOGD("33: Geofence Add Circle Request With Wake Lock Release");
    LOGD("34: Geofence Delete Request");
    LOGD("35: Geofence Delete All Request");
    LOGD(" ");
    LOGD("51: Certificate Inject");
    LOGD("52: Certificate Name List Request");
    LOGD("53: Certificate Delete");
    LOGD("54: Certificate Delete All");
    LOGD(" ");
    LOGD("99: Input a GNSS AT Command manually");

    LOGD("input: ");
    int input = scanf_int(INVALID_VALUE, false);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }
    switch(input) {
        case UI_AT_SETTINGS_TYPE_BACK: {
            system("clear");
            traversal_path_back(&g_traversal);
            break;
        }
        case UI_AT_SETTINGS_GNSS_START_REQUEST: {
            system("clear");
            mtk_at_send_gnss_start_request(g_gnssadp_fd);
            LOGD("GNSS Start Request Sent");
            break;
        }
        case UI_AT_SETTINGS_GNSS_START_WITH_WAKE_LOCK_RELEASE: {
            system("clear");
            mtk_at_send_gnss_start_request(g_gnssadp_fd);
            LOGD("GNSS Start Request Sent with wake lock release");
            //delay to do wake_unlock to ensure the msg can be deliveried to other process
            timer_start(g_wake_unlock_timer, LBS_EM_WAKE_LOCK_TIMEOUT);
            break;
        }
        case UI_AT_SETTINGS_GNSS_STOP_REQUEST: {
            system("clear");
            mtk_at_send_gnss_stop_request(g_gnssadp_fd);
            LOGD("GNSS Stop Request Sent");
            lbs_em_acquire_wake_lock_mutex();
            break;
        }
        case UI_AT_SETTINGS_HOST_REBOOT_NOTIFY: {
            system("clear");
            mtk_at_send_gnss_host_reset_notify(g_gnssadp_fd);
            LOGD("Host Reboot Notify Sent");
            break;
        }
        case UI_AT_SETTINGS_GNSS_STATUS_REQUEST: {
            system("clear");
            mtk_at_send_gnss_status_request(g_gnssadp_fd);
            LOGD("GNSS Status Request Sent");
            break;
        }
        case UI_AT_SETTINGS_GNSS_ENABLE_SET: {
            LOGD("GNSS Enable Set (0=disable, 1=enable) input:");
            int value = scanf_int(INVALID_VALUE, false);
            if(value == INVALID_VALUE) {
                LOGE("ERR: read invalid value");
                return;
            }

            system("clear");
            mtk_at_send_gnss_enable_set(g_gnssadp_fd, value);
            LOGD("GNSS Enable Set [%d] Sent", value);
            break;
        }
        case UI_AT_SETTINGS_NI_ENABLE_SET: {
            LOGD("NI Enable Set (0=disable, 1=enable) input:");
            int value = scanf_int(INVALID_VALUE, false);
            if(value == INVALID_VALUE) {
                LOGE("ERR: read invalid value");
                return;
            }

            system("clear");
            mtk_at_send_gnss_ni_enable_set(g_gnssadp_fd, value);
            LOGD("NI Enable Set [%d] Sent", value);
            break;
        }
        case UI_AT_SETTINGS_AGPS_MODE_SET: {
            int ret = 0;
            mtk_gnss_agps_mode mode;
            memset(&mode, 0, sizeof(mode));
            LOGD("AGPS Mode Set: MSA (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid msa");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.msa_valid = true;
                mode.msa = ret;
            }

            LOGD("AGPS Mode Set: MSB (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid msb");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.msb_valid = true;
                mode.msb = ret;
            }

            LOGD("AGPS Mode Set: MSS (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid mss");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.mss_valid = true;
                mode.mss = ret;
            }

            LOGD("AGPS Mode Set: CID (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid cid");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.cid_valid = true;
                mode.cid = ret;
            }

            LOGD("AGPS Mode Set: AFLT (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid aflt");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.aflt_valid = true;
                mode.aflt = ret;
            }

            LOGD("AGPS Mode Set: OTDOA (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid otdoa");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.otdoa_valid = true;
                mode.otdoa = ret;
            }

            LOGD("AGPS Mode Set: supl_pref_method (0=MSA, 1=MSB, 2=NO Preference, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid supl_pref_method");
                return;
            }
            if(ret == 0 || ret == 1 || ret == 2) {
                mode.supl_pref_method_valid = true;
                mode.supl_pref_method = ret;
            }

            LOGD("SUPL Enable/Disable (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid supl");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.supl_valid = true;
                mode.supl = ret;
            }

            LOGD("EPO Enable/Disable (0=disable, 1=enable, others=ignore) input:");
            ret = scanf_int(INVALID_VALUE, false);
            if(ret == INVALID_VALUE) {
                LOGE("ERR: read invalid epo");
                return;
            }
            if(ret == 0 || ret == 1) {
                mode.epo_valid = true;
                mode.epo = ret;
            }

            system("clear");
            mtk_at_send_gnss_agps_mode_set(g_gnssadp_fd, &mode);
            LOGD("AGPS Mode Set msa=[%d] msb=[%d] mss=[%d] cid=[%d] aflt=[%d] otdoa=[%d] supl_pref_method=[%d] Sent",
                mode.msa, mode.msb, mode.mss, mode.cid, mode.aflt, mode.otdoa, mode.supl_pref_method);
            break;
        }
        case UI_AT_SETTINGS_SUPL_VERSION_SET: {
            LOGD("SUPL Version Set: Major Version input:");
            int maj = scanf_int(INVALID_VALUE, false);
            if(maj == INVALID_VALUE) {
                LOGE("ERR: read invalid maj");
                return;
            }
            LOGD("SUPL Version Set: Minor Version input:");
            int min = scanf_int(INVALID_VALUE, false);
            if(min == INVALID_VALUE) {
                LOGE("ERR: read invalid min");
                return;
            }
            LOGD("SUPL Version Set: Service Indicator Version input:");
            int ser_ind = scanf_int(INVALID_VALUE, false);
            if(min == INVALID_VALUE) {
                LOGE("ERR: read invalid ser_ind");
                return;
            }

            system("clear");
            mtk_at_send_gnss_supl_version_set(g_gnssadp_fd, maj, min, ser_ind);
            LOGD("SUPL Version Set maj=[%d] min=[%d] ser_ind=[%d] Sent", maj, min, ser_ind);
            break;
        }
        case UI_AT_SETTINGS_SUPL_ADDRESS_SET: {
            LOGD("SUPL Address Set: SUPL Address (FQDN/IPv4/IPv6) input:");
            char addr[128] = {0};
            int ret = scanf("%[^\n]", addr); //get a whole line
            clear_stdin_buff();
            if(ret <= 0) {
                LOGE("ERR: read invalid input");
                return;
            }
            LOGD("SUPL Address Set: Port (ex: 7275) input:");
            int port = scanf_int(INVALID_VALUE, false);
            if(port == INVALID_VALUE) {
                LOGE("ERR: read invalid port");
                return;
            }
            LOGD("SUPL Address Set: TLS enable (0=disable, 1=enable) input:");
            int tls_enable = scanf_int(INVALID_VALUE, false);
            if(tls_enable == INVALID_VALUE) {
                LOGE("ERR: read invalid tls_enable");
                return;
            }

            system("clear");
            mtk_at_send_gnss_supl_addr_set(g_gnssadp_fd, addr, port, tls_enable);
            LOGD("SUPL Address Set addr=[%s] port=[%d] tls_enable=[%d] Sent", addr, port, tls_enable);
            break;
        }
        case UI_AT_SETTINGS_DELETE_AIDING_DATA_REQUEST: {
            mtk_gnss_aiding_data flags;
            memset(&flags, 0, sizeof(flags));

            LOGD("Delete Aiding Data Request: All (0=no, 1=yes) input:");
            int input = scanf_int(INVALID_VALUE, false);
            if(input == INVALID_VALUE) {
                LOGE("ERR: read invalid flags.all");
                return;
            } else {
                flags.all = (input == 1);
            }

            if(!flags.all) {
                LOGD("Delete Aiding Data Request: EPH (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.eph");
                    return;
                } else {
                    flags.eph = (input == 1);
                }
                LOGD("Delete Aiding Data Request: ALM (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.alm");
                    return;
                } else {
                    flags.alm = (input == 1);
                }
                LOGD("Delete Aiding Data Request: Position (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.pos");
                    return;
                } else {
                    flags.pos = (input == 1);
                }
                LOGD("Delete Aiding Data Request: Time (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.time");
                    return;
                } else {
                    flags.time = (input == 1);
                }
                LOGD("Delete Aiding Data Request: IONO (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.iono");
                    return;
                } else {
                    flags.iono = (input == 1);
                }
                LOGD("Delete Aiding Data Request: UTC (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.utc");
                    return;
                } else {
                    flags.utc = (input == 1);
                }
                LOGD("Delete Aiding Data Request: SVDIR (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.svdir");
                    return;
                } else {
                    flags.svdir = (input == 1);
                }
                LOGD("Delete Aiding Data Request: RTI (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.rti");
                    return;
                } else {
                    flags.rti = (input == 1);
                }
                LOGD("Delete Aiding Data Request: CellID (0=no, 1=yes) input:");
                input = scanf_int(INVALID_VALUE, false);
                if(input == INVALID_VALUE) {
                    LOGE("ERR: read invalid flags.celldb");
                    return;
                } else {
                    flags.celldb = (input == 1);
                }

            }

            system("clear");
            mtk_at_send_gnss_delete_aiding_data_request(g_gnssadp_fd, &flags);
            LOGD("Delete Aiding Data Request all=[%d] eph=[%d] alm=[%d] pos=[%d] time=[%d] iono=[%d] utc=[%d] svdir=[%d] rti=[%d] celldb=[%d] Sent",
                flags.all, flags.eph, flags.alm, flags.pos, flags.time, flags.iono, flags.utc, flags.svdir, flags.rti, flags.celldb);
            break;
        }
        case UI_AT_SETTINGS_NI_NOTIFY_RESPONSE: {
            LOGD("NI Notify Response: Response Type (1=Accept, 2=Deny, 3=NO Response) input:");
            int resp_type = scanf_int(INVALID_VALUE, false);
            if(resp_type == INVALID_VALUE) {
                LOGE("ERR: read invalid resp_type");
                return;
            }

            system("clear");
            mtk_at_send_gnss_ni_notify_response(g_gnssadp_fd, g_notify_id, resp_type);
            LOGD("NI Notify Response id=[%d] resp_type=[%d] Sent", g_notify_id, resp_type);
            break;
        }
        case UI_AT_SETTINGS_REFERENCE_LOCATION_RESPONSE: {
            LOGD("Reference Location Response: Age (seconds) input:");
            int age = scanf_int(INVALID_VALUE, false);
            if(age == INVALID_VALUE) {
                LOGE("ERR: read invalid age");
                return;
            }
            LOGD("Reference Location Response: Latitude (xx.xxxxxx in degrees, WGS84 coordinate system) input:");
            double lat = scanf_double(0, false);
            if(lat == 0) {
                LOGE("ERR: read invalid lat");
                return;
            }
            LOGD("Reference Location Response: Longitude (xx.xxxxxx in degrees, WGS84 coordinate system) input:");
            double lng = scanf_double(0, false);
            if(lng == 0) {
                LOGE("ERR: read invalid lng");
                return;
            }
            LOGD("Reference Location Response: Accuracy (horizontal position accuracy, radial, in meters, 68%% confidence) input:");
            float acc = scanf_double(0, false);
            if(acc == 0) {
                LOGE("ERR: read invalid acc");
                return;
            }

            system("clear");
            mtk_at_send_gnss_ref_location_response(g_gnssadp_fd, age, lat, lng, acc);
            LOGD("Reference Location Response age=[%d] lat=[%.6f] lng=[%.6f] acc=[%.2f] Sent", age, lat, lng, acc);
            break;
        }
        case UI_AT_SETTINGS_REFERENCE_TIME_RESPONSE: {
            LOGD("Reference Time Response: Time (Milliseconds since January 1, 1970) input:");
            long long time = scanf_long(INVALID_VALUE, false);
            if(time == INVALID_VALUE) {
                LOGE("ERR: read invalid time");
                return;
            }
            LOGD("Reference Time Response: Uncertainty (milliseconds) input:");
            int uncertainty = scanf_int(INVALID_VALUE, false);
            if(uncertainty == INVALID_VALUE) {
                LOGE("ERR: read invalid uncertainty");
                return;
            }

            system("clear");
            mtk_at_send_gnss_ref_time_response(g_gnssadp_fd, time, uncertainty);
            LOGD("Reference Time Response time=[%lld] uncertainty=[%d] Sent", time, uncertainty);
            break;
        }
        case UI_AT_SETTINGS_NMEA_CONFIG_SET: {
            LOGD("NMEA Config Set (0=disable, 1=enable) input:");
            int value = scanf_int(INVALID_VALUE, false);
            if(value == INVALID_VALUE) {
                LOGE("ERR: read invalid value");
                return;
            }

            system("clear");
            mtk_at_send_gnss_nmea_config_set(g_gnssadp_fd, value);
            LOGD("NMEA Config Set [%d] Sent", value);
            break;
        }
        case UI_AT_SETTINGS_LOOPBACK_REQUEST: {
            LOGD("Loopback Request: Test Message input:");
            char msg[128] = {0};
            int ret = scanf("%[^\n]", msg); //get a whole line
            clear_stdin_buff();
            if(ret <= 0) {
                LOGE("ERR: read invalid input");
                return;
            }

            system("clear");
            mtk_at_send_gnss_loopback_request(g_gnssadp_fd, msg);
            LOGD("Loopback Request [%s] Sent", msg);
            break;
        }
        case UI_AT_SETTINGS_GEOFENCE_MAX_NUMBER_REQUEST: {
            system("clear");
            mtk_at_send_geo_max_num_request(g_gnssadp_fd);
            LOGD("Geofence Max Number Request Sent");
            break;
        }
        case UI_AT_SETTINGS_GEOFENCE_ADD_CIRCLE_REQUEST: {
            g_at_geofence_add_with_wake_lock_release = false;
            at_geofence_add_circle_request();
            break;
        }
        case UI_AT_SETTINGS_GEOFENCE_ADD_CIRCLE_WITH_WAKE_LOCK_RELEASE: {
            g_at_geofence_add_with_wake_lock_release = true;
            at_geofence_add_circle_request();
            break;
        }
        case UI_AT_SETTINGS_GEOFENCE_DELETE_REQUEST: {
            LOGD("Geofence Delete Request: id (integer) input:");
            int id = scanf_int(INVALID_VALUE, false);
            if(id == INVALID_VALUE) {
                LOGE("ERR: read invalid id");
                return;
            }

            system("clear");
            mtk_at_send_geo_delete_request(g_gnssadp_fd, id);
            LOGD("Geofence Delete Request id=[%d] Sent", id);

            geofence_list_remove(&g_geo_list,id);
            if (g_geo_list.num == 0) {
                LOGD("Geofence all elements are removed, acquire wake lock");
                g_at_geofence_add_with_wake_lock_release = false;
                lbs_em_acquire_wake_lock_mutex();
            }

            break;
        }
        case UI_AT_SETTINGS_GEOFENCE_DELETE_ALL_REQUEST: {
            system("clear");
            mtk_at_send_geo_delete_all_request(g_gnssadp_fd);
            LOGD("Geofence Delete All Request is sent, acquire wake lock");
            g_at_geofence_add_with_wake_lock_release = false;
            lbs_em_acquire_wake_lock_mutex();
            break;
        }
        case UI_AT_SETTINGS_CERTIFICATE_INJECT: {
            system("clear");
            cert_inject_procedure();
            break;
        }
        case UI_AT_SETTINGS_CERTIFICATE_NAME_LIST_REQUEST: {
            system("clear");
            mtk_at_send_gnss_cert_name_request(g_gnssadp_fd);
            LOGD("Certificate Name List Request Sent");
            break;
        }
        case UI_AT_SETTINGS_CERTIFICATE_DELETE: {
            LOGD("Certificate Delete: file name:");
            char name[128] = {0};
            int ret = scanf("%[^\n]", name); //get a whole line
            clear_stdin_buff();
            if(ret <= 0) {
                LOGE("ERR: read invalid input");
                return;
            }

            system("clear");
            mtk_at_send_gnss_cert_delete_request(g_gnssadp_fd, name);
            LOGD("Certificate Delete Sent name=[%s]", name);
            break;
        }
        case UI_AT_SETTINGS_CERTIFICATE_DELETE_ALL: {
            system("clear");
            mtk_at_send_gnss_cert_delete_all_request(g_gnssadp_fd);
            LOGD("Certificate Delete All Sent");
            break;
        }
        case 99: {
            char buff[1024] = {0};
            LOGD("Input a GNSS AT Command manually input:");
            int ret = scanf("%[^\n]", buff); //get a whole line
            clear_stdin_buff();
            if(ret <= 0) {
                LOGE("ERR: read invalid input");
                return;
            }

            //append \r\n to the string
            int i = 0;
            while(buff[i] != 0) {
                i++;
            }
            buff[i] = '\r';
            buff[i + 1] = '\n';

            system("clear");
            ret = write(g_gnssadp_fd, buff, strlen(buff));
            if(ret == -1) {
                LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
            }
            LOGD("GNSS AT Command Sent: %s", buff);
            break;
        }
        default: {
            system("clear");
            LOGE("ERR: read unhandled value=[%d]", input);
            break;
        }
    }
}

bool mtk_nlp_send_start_request(int fd) {
    char buff[128] = {0};
    mtk_at_gen_gnss_cert_delete_all_request(buff, sizeof(buff));
    int ret = write(fd, buff, strlen(buff));
    if(ret == -1) {
        LOGE("write() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool mtk_nlp_find_file_in_dir(char *dir, char *target_dir, char *target_file, char *target_path) {
    bool target_found = false;
    DIR *dp;
    struct dirent *entry;
    struct stat statbuf;
    char new_dir[256];

    if((dp = opendir(dir)) == NULL) {
        fprintf(stderr,"cannot open directory: %s\n", dir);
        return false;
    }
    //LOGD("switch to dir: %s\n", dir);
    chdir(dir);

    while ((entry = readdir(dp)) != NULL) {
        lstat(entry->d_name, &statbuf);

        if(strcmp(".",entry->d_name) == 0 || strcmp("..",entry->d_name) == 0)
            continue;
        if (strncmp(entry->d_name, target_file, strlen(target_file)) == 0) {
            target_found = true;
            snprintf(target_path, 255, "%s/%s", dir, entry->d_name);
            break;
        }

        if (strncmp(entry->d_name, target_dir, strlen(target_dir)) ==0) {
            snprintf(new_dir, 255, "%s/%s", dir, entry->d_name);
            target_found = mtk_nlp_find_file_in_dir(new_dir, target_dir, target_file, target_path);
            if (target_found) {
                break;
            }
        }
    }
    closedir(dp);
    return target_found;
}

void mtk_nlp_request_pressure_sensor() {
    char iio_barometer_path[256] = "";
    char pressure_value[32];
    int pressure_value_pa;
    bool found = false;
    char *IIO_SENSOR_PATH = "/sys/bus/iio/devices";
    char *IIO_SENSOR_DEVICE_PREFIX = "iio:";
    char *IIO_SENSOR_PRESSURE_RAW_ATTRIBUTE = "in_pressure_raw";
    int fd;
    int count;

    //printf("Directory scan events dir:\n");
    found = mtk_nlp_find_file_in_dir(IIO_SENSOR_PATH, IIO_SENSOR_DEVICE_PREFIX,
            IIO_SENSOR_PRESSURE_RAW_ATTRIBUTE, iio_barometer_path);
    if (!found || iio_barometer_path[0] == '\0') {
        LOGD("Unable to find barometer device with raw attribute in %s", IIO_SENSOR_PATH);
    }

    fd = open(iio_barometer_path, O_RDWR);
    if (fd < 0) {
        LOGD("Failed to open %s\n", iio_barometer_path);
        return;
    }
    count = read(fd, pressure_value, 32);
    pressure_value[31] = '\0';
    if (count > 0) {
        pressure_value_pa = atoi(pressure_value);
        LOGD("Pressure sensor value :%d\n", pressure_value_pa);
    } else {
        LOGD("Failed to get pressure ensor value!\n");
    }
    close(fd);
}

void ui_nlp_test() {
    LOGD("=== NLP Test ====");
    LOGD("0.  Back");
    LOGD(" ");
    LOGD("1.  NLP Start Request");
    LOGD("2.  NLP Stop Request");
    LOGD("3.  Get Last Location");
    LOGD("4.  Single Location Request");
    LOGD("5.  Pressure Sensor Request");

    LOGD("input: ");
    ui_gps_control_type input = scanf_int(INVALID_VALUE, true);
    if(input == INVALID_VALUE) {
        LOGE("ERR: read invalid input");
        return;
    }

    switch(input) {
        case UI_NLP_TEST_TYPE_BACK:
            traversal_path_back(&g_traversal);
            break;
        case UI_NLP_TEST_TYPE_START_RQUEST:
            LOGD("NLP start request");
            MtkNlp_start_location_updates(&g_nlpservice_fd, 1000);
            break;
        case UI_NLP_TEST_TYPE_STOP_RQUEST:
            LOGD("NLP stop request");
            MtkNlp_stop_location_updates(&g_nlpservice_fd);
            break;
        case UI_NLP_TEST_TYPE_GET_LAST_LOC:
            LOGD("Get last location");
            MtkNlp_get_last_location(&g_nlpservice_fd);
            break;
        case UI_NLP_TEST_TYPE_GET_SINGLE_UPDATE:
            LOGD("Get single location update");
            MtkNlp_get_single_update(&g_nlpservice_fd);
            break;
        case UI_NLP_TEST_TYPE_GET_PRESSURE_SENSOR:
            LOGD("Get Pressure sensor update");
            mtk_nlp_request_pressure_sensor();
            break;
        default:
            LOGE("ERR: read unhandled value=[%d]", input);
            return;
    }
}

void mtk_lbs_em_tree_construction(tree_node* header) {
    tree_node* n = NULL;   // for manipulate

    tree_init(header, ui_main);

    n = header;
    n = tree_add_child(n, UI_MAIN_TYPE_GPS_CONTROL, ui_gps_control);
        tree_add_child(n, UI_GPS_CONTROL_TYPE_GPS_DELETE_AIDING_DATA, ui_gps_control_delete_aiding_data);
        n = n->parent;
    n = tree_add_child(n, UI_MAIN_TYPE_AGPS_SETTINGS, ui_agps_settings);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_AGPS_ENABLE, ui_agps_settings_agps_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_AGPS_PROTOCOL, ui_agps_settings_agps_protocol);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CDMA_PREFERENCE, ui_agps_settings_cdma_preference);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_PREFER_METHOD, ui_agps_settings_supl_prefer_method);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_MSA_ENABLE, ui_agps_settings_supl_pos_tech_msa_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_MSB_ENABLE, ui_agps_settings_supl_pos_tech_msb_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_ECID_ENABLE, ui_agps_settings_supl_pos_tech_ecid_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_AUTONOMOUS_ENABLE, ui_agps_settings_supl_pos_tech_autonomous_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_POS_TECT_AFLT_ENABLE, ui_agps_settings_supl_pos_tech_aflt_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_LPP_ENABLE, ui_agps_settings_supl_lpp_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VERSION, ui_agps_settings_supl_version);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_TLS_VERSION, ui_agps_settings_supl_tls_version);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_CA_ENABLE, ui_agps_settings_supl_ca_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_UDP_ENABLE, ui_agps_settings_supl_udp_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_UT2, ui_agps_settings_supl_ut2);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_UT3, ui_agps_settings_supl_ut3);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VER_MINOR, ui_agps_settings_supl_ver_minor);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_VER_SER_IND, ui_agps_settings_supl_ver_ser_ind);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_PROFILE, ui_agps_settings_supl_profile);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_PERIODIC_SETTINGS, ui_agps_settings_supl_periodic_settings);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_AREA_SETTINGS, ui_agps_settings_supl_area_settings);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SI_QOP_SETTINGS, ui_agps_settings_supl_qop);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_ALLOW_NI_REQUEST, ui_agps_settings_supl_ni_request);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_ALLOW_ROAMING, ui_agps_settings_supl_allow_roaming);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CP_AUTO_RESET, ui_agps_settings_cp_auto_reset_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SUPL_LOG_TO_FILE, ui_agps_settings_supl_log_to_file);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SENSITIVE_LOG, ui_agps_settings_supl_sensitive_log);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_CERT_FROM_SDCARD, ui_agps_settings_supl_cert_from_sdcard);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_AUTO_PROFILE, ui_agps_settings_supl_auto_profile);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_DEDICATE_APN, ui_agps_settings_supl_dedicate_apn);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_SYNC_TO_SLP, ui_agps_settings_supl_sync_to_slp);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_IMSI_ENABLE, ui_agps_settings_supl_imsi_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_E911_GPS_ICON, ui_agps_settings_supl_e911_gps_icon);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_LPPE_NETWORK_LOCATION_DISABLE, ui_agps_settings_supl_lppe_nlp_disable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CP_LPPE_ENABLE, ui_agps_settings_cp_lppe_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_UP_LPPE_ENABLE, ui_agps_settings_supl_up_lppe_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_AOSP_PROFILE_ENABLE, ui_agps_settings_supl_aosp_profile_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_BIND_NLP_SETTING_TO_SUPL, ui_agps_settings_supl_bind_nlp_settings_to_supl);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_LBS_LOG_ENABLE, ui_agps_settings_supl_lbs_log_enable);

        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_INGORE_SI_FOR_E911, ui_agps_settings_ignore_si_for_e911);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_WLAN, ui_agps_settings_lppe_cp_wlan);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_SRN, ui_agps_settings_lppe_cp_srn);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_SENSOR, ui_agps_settings_lppe_cp_sensor);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_CP_DBH, ui_agps_settings_lppe_cp_dbh);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_WLAN, ui_agps_settings_lppe_up_wlan);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_SRN, ui_agps_settings_lppe_up_srn);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_SENSOR, ui_agps_settings_lppe_up_sensor);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_UP_DBH, ui_agps_settings_lppe_up_dbh);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_UP_LPP_IN_2G_3G_DISABLE, ui_agps_settings_up_lppe_2g_3g_disable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_UP_RRLP_IN_4G_DISABLE, ui_agps_settings_up_rrlp_in_4g_disable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_UP_SI_DISABLE, ui_agps_settings_up_si_disable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_DEFAULT_NLP_LOCATION_ENABLE, ui_agps_settings_lppe_default_nlp_location);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_FORCE_OTDOA_ASSIST_ENABLE, ui_agps_settings_force_otdoa_assist);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_AOSP_POSITION_MODE_ENABLE, ui_agps_settings_up_aosp_pos_mode);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_PRIVACY_OVERRIDE_ENABLE, ui_agps_settings_privacy_override);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_ALLOW_NI_FOR_GPS_OFF_ENABLE, ui_agps_settings_allow_ni_for_gps_off);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_IP_VERSION_PREFER, ui_agps_settings_ip_version_prefer);

        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_SUPL_TCP_KEEPALIVE, ui_agps_settings_supl_tcp_keepalive);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_LPPE_CROWD_SOURCE_CONFIDENT, ui_agps_settings_lppe_crowd_source_confident);

        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_GNSS_SIB8_SIB16_ENABLE, ui_agps_settings_gnss_sib8_sib16_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_GLONASS_SATELLITE_ENABLE, ui_agps_settings_gnss_a_glonass_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_BEIDOU_SATELLITE_ENABLE, ui_agps_settings_gnss_a_beidou_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_GNSS_A_GALILEO_SATELLITE_ENABLE, ui_agps_settings_gnss_a_galileo_enable);

        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CP_MOLR_POS_METHOD, ui_agps_settings_cp_molr_pos_method);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CP_EXTERNAL_ADDR_ENABLE, ui_agps_settings_cp_external_addr_enable);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CP_MLC_NUMBER, ui_agps_settings_cp_mlc_number);
        tree_add_child(n, UI_AGPS_SETTINGS_TYPE_SET_CP_EPC_MOLR_LPP_PAYLOAD_ENABLE, ui_agps_settings_epc_molr_pdu_enable);

        n = n->parent;
    n = tree_add_child(n, UI_MAIN_TYPE_DEBUG_SETTINGS, ui_debug_settings);
        n = n->parent;
    n = tree_add_child(n, UI_MAIN_TYPE_DEBUG_1, ui_debug_1);
        tree_add_child(n, UI_DEBUG_1_TYPE_START_TEST_CASE, ui_debug_1_test_case);
        tree_add_child(n, UI_DEBUG_1_TYPE_TEST_BUTTON, ui_debug_1_test_button);
        n = n->parent;
    n = tree_add_child(n, UI_MAIN_TYPE_GNSS_AT_SETTINGS, ui_gnss_at_settings);
        n = n->parent;
    n = tree_add_child(n, UI_MAIN_TYPE_NLP_TEST, ui_nlp_test);
        n = n->parent;
}

int mtk_lbs_em_start() {

    traversal_path_init(&g_traversal);

    mtk_lbs_em_tree_construction(&g_tree_root);

    //TODO need to remove
    //traversal_path_go(&g_traversal, UI_MAIN_TYPE_GNSS_AT_SETTINGS);

    system("clear");
    while(1) {
        int ret = tree_traversal(&g_tree_root, &g_traversal);
        if(ret) {
            LOGE("ERR: tree_traversal() line=[%d]", __LINE__);
            return -1;
        }
    }
    return 0;
}

// ------------- AGPS Debug Interface handlers  -----------------

void toast_message (const char* msg) {
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    LOGD("\E[1;32;40m");
    LOGD("%s [AGPS Debug][Toast] msg=[%s]", time_buff, msg);
    LOGD("\E[0m");
}

void view_message (int color, const char* msg) {
    if(g_agps_debug_enable) {
        int is_red = (color == 0xffff0000)? 1 : 0;
        char time_buff[64] = {0};
        get_time_str(time_buff, sizeof(time_buff));
        if(is_red) {
            LOGD("\E[1;31;40m");
        }
        LOGD("%s [AGPS Debug][Info] color=[%x] msg=[%s]", time_buff, color, msg);
        if(is_red) {
            LOGD("\E[0m");
        }
    }
}

void dialog_message (const char* title, const char* msg) {
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    LOGD("\E[1;33;40m");
    LOGD("%s [AGPS Debug][Dialog] title=[%s] msg=[%s]", time_buff, title, msg);
    LOGD("\E[0m");
}

agps_debug_interface g_agps_debug_interface = {
    toast_message,
    view_message,
    dialog_message,
};

static void agps_debug_message_hdlr(int epfd) {
    int ret = agps_debug_interface_msg_handler(g_agps_debug_fd, &g_agps_debug_interface);
    if(ret) {
        LOGE("ERR: agps_message_thread()  re-connect AGPS Debug Interface");
        while(1) {
            if(g_agps_debug_fd >= 0) {
                close(g_agps_debug_fd);
            }
            msleep2(1000);
            g_agps_debug_fd = agps_debug_interface_get_fd();
            if(g_agps_debug_fd >= 0) {
                epoll_add_fd3(epfd, g_agps_debug_fd);
                break;
            } else {
                LOGE("ERR: agps_message_thread()  re-connect fail, retry it in next second");
            }
        }
    }
}

static void mtk_lbs_em_close_fd(int fd) {
    close(fd);
    if(g_epfd != -1 && fd == g_epfd) {
        g_epfd = -1;
        LOGW("epfd is closed");
    } else if(g_mnld_basic_fd != -1 && fd == g_mnld_basic_fd) {
        g_mnld_basic_fd = -1;
        LOGW("MNL Basic Interface is closed");
    } else if(g_mnld_ext_fd != -1 && fd == g_mnld_ext_fd) {
        g_mnld_ext_fd = -1;
        LOGW("MNL Ext Interface is closed");
    } else if(g_nlpservice_fd.fd != -1 && fd == g_nlpservice_fd.fd) {
        g_nlpservice_fd.fd = -1;
        LOGW("NLP Service Interface is closed");
    } else {
        LOGE("unexpected fd=[%d] to be close", fd);
    }
}

static void mtk_lbs_em_timer_retry_port_routine(int id) {
    char cmd = LBS_EM_INT_CMD_RETRY_CONNECT;
    int fd = socket_udp_client_create_local(true, LBSEM_INTERNAL_SOCKET);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    int ret = write(fd, &cmd, sizeof(cmd));
    if(ret == -1) {
        LOGE("write() failed id=[%d], reason=[%s]%d", LBS_EM_INT_CMD_RETRY_CONNECT, strerror(errno), errno);
    }
    close(fd);
}

static void mtk_lbs_em_timer_retry_connect_hdlr() {
    g_connect_retry_count++;
    LOGW("try times=(%d/%d)", g_connect_retry_count, LBS_EM_MAX_RETRY_CONNECT_TIMES);

    bool all_success = true;

    if(g_mnld_basic_fd < 0) {
        g_mnld_basic_fd = mnldinf_basic_register("LBS EM");
        if(g_mnld_basic_fd < 0) {
            all_success = false;
        } else {
            LOGW("mnldinf_basic_register() success fd=[%d]", g_mnld_basic_fd);
            epoll_add_fd3(g_epfd, g_mnld_basic_fd);
            mnldinf_basic_client_cap cap;
            cap.support_gnss = true;
            mnldinf_basic_capability_config(g_mnld_basic_fd, &cap);
        }
    }

    if(g_mnld_ext_fd < 0) {
        g_mnld_ext_fd = mnldinf_ext_register("LBS EM");
        if(g_mnld_ext_fd < 0) {
            all_success = false;
        } else {
            LOGW("mnldinf_ext_register() success fd=[%d]", g_mnld_basic_fd);
            epoll_add_fd3(g_epfd, g_mnld_ext_fd);
            mnldinf_ext_client_cap cap;
            cap.support_gnss = true;
            mnldinf_ext_capability_config(g_mnld_ext_fd, &cap);
        }
    }

    if(g_nlpservice_fd.fd < 0) {
        g_nlpservice_fd.fd = MtkNlp_get_fd();
        if(g_nlpservice_fd.fd < 0) {
            all_success = false;
        } else {
            LOGW("MtkNlp_get_fd() success fd=[%d]", g_nlpservice_fd.fd);
            epoll_add_fd3(g_epfd, g_nlpservice_fd.fd);
        }
    }

    if(all_success) {
        LOGW("all scuess");
        g_connect_retry_count = 0; //reset count for next time use
    } else {
        if(g_connect_retry_count < LBS_EM_MAX_RETRY_CONNECT_TIMES) {
            timer_start(g_connect_retry_timer, LBS_EM_RETRY_CONNECT_INTERVAL);
        } else {
            g_connect_retry_count = 0; //reset count for next time use
            LOGW("port retry count reach to max, no retry anymore");
        }
    }
}

static void mtk_lbs_em_epoll_internal_hdlr(int fd) {
    char buff[1024] = {0};
    int ret = read(fd, buff, sizeof(buff));
    if(ret == -1) {
        LOGE("read() failed, fd=[%d] reason=[%s]%d", fd, strerror(errno), errno);
        crash_with_log();
    }
    int cmd = buff[0];
    switch(cmd) {
    case LBS_EM_INT_CMD_RETRY_CONNECT:
        mtk_lbs_em_timer_retry_connect_hdlr();
        break;
    default:
        LOGE("unknown cmd=[%d]", cmd);
        break;
    }
}
/**
   * Decodes GSM 7-bit septets in 8-bit octets.
   *
   * @param ud The 8-bit octets representing 7-bit data
   * @param fillBits The number of fill bits, if any
   * @param udl The message length, if known
   * @return The converted message
   */
//public static int[] decode7bit(int[] ud, int udl, int fillBits)
static void decodeGSMPackedString(const char* input, char* output, int length)
{
    int udl = -1;
    const char* ud = input;

    const char upperBits[] = {
        0xFE, // 0 = B7|B6|B5|B4|B3|B2|B1
        0xFC, // 1 = B7|B6|B5|B4|B3|B2
        0xF8, // 2 = B7|B6|B5|B4|B3
        0xF0, // 3 = B7|B6|B5|B4
        0xE0, // 4 = B7|B6|B5
        0xC0, // 5 = B7|B6
        0x80  // 6 = B7
    };

    const char lowerBits[] = {
        0x01, // 0 =                   B0
        0x03, // 1 =                B1|B0
        0x07, // 2 =             B2|B1|B0
        0x0F, // 3 =          B3|B2|B1|B0
        0x1F, // 4 =       B4|B3|B2|B1|B0
        0x3F, // 5 =    B5|B4|B3|B2|B1|B0
        0x7F  // 6 = B6|B5|B4|B3|B2|B1|B0
    };

    if (udl < 0) {
        udl = length * 8 / 7;
    }

    int b = 6, p = 0;
    int i = 0;
    for (i = 0; i < udl; i++) {
        switch (b) {
            case 7: // U0
                output[i] = (ud[p] & upperBits[0]) >> 1;
            break;

            case 6: // L6
              output[i] = ud[p] & lowerBits[b];
            break;

            default: // The rest
              output[i] = ((ud[p] & lowerBits[b]) << (6 - b))
                      + ((ud[p-1] & upperBits[b+1]) >> (b+2));
            break;
        }

        if (--b == -1) {
            b = 7;
        } else {
            p++;
        }
    }
}

// Converts a string (or Hex string) to a char array
static int stringToByteArray(const char* original, char* output)
{
    int final_len = strlen(original)/2;
    size_t i=0, j=0;

    for (i=0, j=0; j<final_len; i+=2, j++) {
        output[j] = (original[i] % 32 + 9) % 25 * 16 + (original[i+1] % 32 + 9) % 25;
    }
    output[final_len] = '\0';
    return final_len;
}

static void decodeUCS2ASCII(const char* input, char* output, int len)
{
    char* asc_input = (char*) input+2;
    int asc_len = len/2 - 1;
    int i = 0;

    for (i = 0; i < asc_len; i++) {
        output[i] = (asc_input[i*2 + 1]);
    }
    output[i] = '\0';
}

static void decodeString(const char* original, char* output, ni_encoding_type coding)
{
    char inputByte[1024] = {};
    int len = stringToByteArray(original, inputByte);

    switch (coding) {
        case NI_ENCODING_TYPE_NONE:
            strncpy(output, inputByte, 1024);
            break;
            
        case NI_ENCODING_TYPE_GSM7:
            decodeGSMPackedString(inputByte, output, len);
            break;

        case NI_ENCODING_TYPE_UTF8:
            strncpy(output, inputByte, 1024);
            break;

        case NI_ENCODING_TYPE_UCS2:
            decodeUCS2ASCII(inputByte, output, len);
            break;

        default:
            LOGE("Unknown encoding:%d for NI text: %s", coding, original);
            break;
    }
}


#define MNL_LOG(...)   { printf("\E[0;30;43m"); printf(__VA_ARGS__); printf("\E[0m"); fflush(stdout); }
// ------------- mnldinf basic handlers -----------------
void mnldinf_basic_connection_broken_hdlr() {
    LOGW("MNLD is crash, start to re-connection procedure for Basic");
    mtk_lbs_em_close_fd(g_mnld_basic_fd);
    timer_start(g_connect_retry_timer, LBS_EM_RETRY_CONNECT_INTERVAL);
}

void mnldinf_location_hdlr(gps_location location) {
//    LOGD("flags=[0x%x] time=[%"PRId64"] hacc=[%.1f] alt=[%.1f] vacc=[%.1f]",
//        location.flags, location.timestamp, location.h_accuracy, location.alt, location.v_accuracy);

    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] location  flags=0x%x timestamp=%lld",
        time_buff, location.flags, (long long)location.timestamp);
    if(location.flags & MTK_GPS_LOCATION_HAS_LAT_LONG) {
        //MNL_LOG("    lat,lng = %f,%f", location.lat, location.lng);
    }
    if(location.flags & MTK_GPS_LOCATION_HAS_ALT) {
        MNL_LOG("    alt=%f", location.alt);
    }
    if(location.flags & MTK_GPS_LOCATION_HAS_SPEED) {
        MNL_LOG("    speed=%f", location.speed);
    }
    if(location.flags & MTK_GPS_LOCATION_HAS_BEARING) {
        MNL_LOG("    bearing=%f", location.bearing);
    }
    if(location.flags & MTK_GPS_LOCATION_HAS_HORIZONTAL_ACCURACY) {
        MNL_LOG("    accuracy=%f", location.h_accuracy);
    }
}

void mnldinf_gnss_status_hdlr(gps_status status) {
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    if(!g_gps_info_enable) return;
    MNL_LOG("%s [MNL] gps_status  status=[%d] (BEGIN=1 END=2 ENGINE_ON=3 ENGINE_OFF=4)",
        time_buff, status);
}

void mnldinf_gnss_sv_hdlr(gnss_sv_info sv) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] gps_sv  num=[%d]", time_buff, sv.num_svs);
    int i = 0;
    for(i = 0; i < sv.num_svs; i++) {
        gnss_sv* s = &sv.sv_list[i];
        MNL_LOG("    i=[%d] svid=[%02d] constellation=[%02d] cn=[%f] ele=[%f] azi=[%f] flags=[%02d]",
            i, s->svid, s->constellation, s->c_n0_dbhz, s->elevation, s->azimuth, s->flags);
    }
}

void mnldinf_nmea_hdlr(int64_t timestamp, const char* nmea, int length) {
    if(!g_nmea_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] nmea  timestamp=[%lld] len=[%d] %s", time_buff, (long long)timestamp, length, nmea);
}

void mnldinf_nmea_done_hdlr() {
    if(!g_nmea_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] nmea done", time_buff);
}

void mnldinf_gnss_measurements_hdlr(gnss_data *data) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] gps_measurements", time_buff);
}

void mnldinf_basic_capability_update_hdlr(mnldinf_basic_server_cap *cap) {
    //LOGD("support_gnss=[%d]", cap->support_gnss);
}

//MNLD send the message to GNSS adaptor
static mnldinf_basic g_mnld_basic_callbacks = {
    mnldinf_basic_connection_broken_hdlr,
    mnldinf_location_hdlr,
    mnldinf_gnss_status_hdlr,
    mnldinf_gnss_sv_hdlr,
    mnldinf_nmea_hdlr,
    mnldinf_nmea_done_hdlr,
    mnldinf_gnss_measurements_hdlr,
    mnldinf_basic_capability_update_hdlr,
};

// ------------- mnldinf ext handlers -----------------
void mnldinf_ext_connection_broken_hdlr() {
    LOGW("MNLD is crash, start to re-connection procedure for EXT");
    mtk_lbs_em_close_fd(g_mnld_ext_fd);
    timer_start(g_connect_retry_timer, LBS_EM_RETRY_CONNECT_INTERVAL);
}

void mnldinf_request_wakelock_hdlr() {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] request_wakelock", time_buff);
}

void mnldinf_release_wakelock_hdlr() {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] release_wakelock", time_buff);
}

void mnldinf_request_utc_time_hdlr() {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] request_utc_time", time_buff);
}

void mnldinf_request_nlp_hdlr(bool independentFromGnss, bool isUserEmergency) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] request_nlp_loc independentFromGnss=%d isUserEmergency=%d",
            time_buff, independentFromGnss, isUserEmergency);
}

void mnldinf_request_ni_notify_hdlr(int notify_id, agps_ni_type ni_type,
        agps_notify_type notify_type,
        const char* requestor_id, const char* client_name,
        ni_encoding_type requestor_id_encoding, ni_encoding_type client_name_encoding) {
    char output[1024] = {};
    g_notify_id = notify_id;
    LOGD("ni notify requested notify_id=[%d]", notify_id);
    LOGD("ni notify requested ni_type=[%d]"
            " (1=VOICE 2=UMTS_SUPL 3=UMTS_CTRL_PLANE 4=EMERGENCY_SUPL)", ni_type);
    LOGD("ni notify requested notify_type=[%d]"
            " (0=NONE 1=NOTIFY_ONLY 2=NOTIFY_ALLOW_NO_ANSWER 3=NOTIFY_DENY_NO_ANSWER 4=PRIVACY)"
            , notify_type);

    decodeString(requestor_id, output, requestor_id_encoding);
    LOGD("ni notify requestor_id_encoding=[%d] (0=NONE 1=GSM7 2=UTF8 3=UCS2), requstor_id: [%s]",
            requestor_id_encoding, output);

    decodeString(client_name, output, client_name_encoding);
    LOGD("ni notify client_name_encoding=[%d] (0=NONE 1=GSM7 2=UTF8 3=UCS2), client_name: [%s]",
            client_name_encoding, output);

}

void mnldinf_request_data_conn_hdlr(struct sockaddr_storage* addr, agps_type type) {
    //the host is not allowed the device to setup the data connection, only the host can setup APN by itself, do nothing
}

void mnldinf_release_data_conn_hdlr() {
    //the host is not allowed the device to setup the data connection, only the host can setup APN by itself, do nothing
}

void mnldinf_request_set_id_hdlr(request_setid flags) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] request_set_id", time_buff);
}

void mnldinf_request_ref_loc_hdlr(request_refloc flags) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] request_ref_loc", time_buff);
}

void mnldinf_output_vzw_debug_hdlr(const char* str) {
    //no need to support this, do nothing
}

void mnldinf_update_gnss_name_hdlr(const char* name, int length) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    //MNL_LOG("%s [MNL] gnss_name len=[%d] %s", time_buff, length, name);
}

void mnldinf_gnss_navigation_msg_hdlr(gnss_nav_msg *msg) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    MNL_LOG("%s [MNL] gps_navigation", time_buff);
}


void mnldinf_nfw_access_notify_hdlr(mnldinf_nfw_notification *nfw_notify) {
    //no AT command to forward AOSP non-framework notification, do nothing
}

void mnldinf_ext_capability_update_hdlr(mnldinf_ext_server_cap *cap) {
    if(!g_gps_info_enable) return;
    char time_buff[64] = {0};
    get_time_str(time_buff, sizeof(time_buff));
    //MNL_LOG("%s [MNL] mnld_capabilities support_gnss=[%d]", time_buff, cap->support_gnss);
}

void mnldinf_agps_location_update(mnldinf_agps_location location) {
    MNL_LOG("[MNL] mnldinf_agps_location_update");
}

static mnldinf_ext g_mnld_ext_callbacks = {
    mnldinf_ext_connection_broken_hdlr,
    mnldinf_request_wakelock_hdlr,
    mnldinf_release_wakelock_hdlr,

    mnldinf_request_utc_time_hdlr,
    mnldinf_request_nlp_hdlr,
    mnldinf_request_ni_notify_hdlr,

    mnldinf_request_data_conn_hdlr,
    mnldinf_release_data_conn_hdlr,

    mnldinf_request_set_id_hdlr,
    mnldinf_request_ref_loc_hdlr,
    mnldinf_output_vzw_debug_hdlr,

    mnldinf_update_gnss_name_hdlr,

    mnldinf_gnss_navigation_msg_hdlr,
    mnldinf_nfw_access_notify_hdlr,
    mnldinf_ext_capability_update_hdlr,
    mnldinf_agps_location_update,
};


// ------------- mk at  handlers -----------------
static void mtk_at_result_hdlr(int fd, bool is_ok, const char* msg) {
    UNUSED(msg);
    if(is_ok) {
        LOGW("GNSS Adapter reply: OK");
    } else {
        LOGE("GNSS Adapter reply: ERROR");
    }
    if(g_at_cert_injecting) {
        g_cert_result = is_ok;
        sem_post(&g_cert_sem);
    }
}

static void mtk_at_gnss_nmea_notify_hdlr(int fd, const char* nmea) {
    if (g_at_cmd_enable) {
        LOGD("mtk_at_gnss_nmea_notify_hdlr() fd=[%d] nmea=[%s]", fd, nmea);
    }
}

static void mtk_at_gnss_nmea_end_notify_hdlr(int fd) {
    if (g_at_cmd_enable) {
        LOGD("mtk_at_gnss_nmea_end_notify_hdlr() fd=[%d]", fd);
    }
}

static void mtk_at_gnss_satellite_notify_hdlr(int fd, mtk_gnss_satellite_list* list) {
    if (g_at_cmd_enable) {
        LOGD("mtk_at_gnss_satellite_notify_hdlr() fd=[%d]", fd);
        mtk_at_dump_mtk_gnss_satellite_list(list);
    }
}

static void mtk_at_gnss_location_notify_hdlr(int fd, mtk_gnss_location* location) {
    if (g_at_cmd_enable) {
        LOGD("mtk_at_gnss_location_notify_hdlr() fd=[%d]", fd);
        mtk_at_dump_mtk_gnss_location(location);
    }
}

static void mtk_at_gnss_agnss_location_notify_hdlr(int fd, mtk_agnss_location* location) {
    LOGW("mtk_at_gnss_agnss_location_notify_hdlr() fd=[%d]", fd);
    mtk_at_dump_mtk_agnss_location(location);
}

static void mtk_at_gnss_device_reset_notify_hdlr(int fd, const char* reason) {
    LOGW("mtk_at_gnss_device_reset_notify_hdlr() fd=[%d] reason=[%s]", fd, reason);
}

static void mtk_at_gnss_status_response_hdlr(int fd, mtk_gnss_status* status) {
    LOGW("mtk_at_gnss_status_response_hdlr() fd=[%d]", fd);
    mtk_at_dump_mtk_gnss_status(status);
}

static void mtk_at_gnss_ni_notify_request_hdlr(int fd, mtk_gnss_ni_notify* notify) {
    LOGW("mtk_at_gnss_ni_notify_request_hdlr() fd=[%d]", fd);
    mtk_at_dump_mtk_gnss_ni_notify(notify);
    mnldinf_request_ni_notify_hdlr(notify->id, notify->type, notify->notify_type, notify->request_id,
            notify->text, notify->request_id_encode_type, notify->text_encode_type);
}

static void mtk_at_gnss_ref_location_request_hdlr(int fd) {
    LOGW("mtk_at_gnss_ref_location_request_hdlr() fd=[%d]", fd);
}

static void mtk_at_gnss_ref_time_request_hdlr(int fd) {
    LOGW("mtk_at_gnss_ref_time_request_hdlr() fd=[%d]", fd);
}

static void mtk_at_gnss_loopback_response_hdlr(int fd, const char* msg) {
    LOGW("mtk_at_gnss_loopback_response_hdlr() fd=[%d] msg=[%s]", fd, msg);
}

static void mtk_at_gnss_cert_name_response_hdlr(int fd, mtk_gnss_cert_name_list* list) {
    LOGW("mtk_at_gnss_cert_name_response_hdlr() fd=[%d]", fd);
    mtk_at_dump_mtk_gnss_cert_name_list(list);
}

static void mtk_at_geo_max_num_response_hdlr(int fd, int num) {
    LOGW("mtk_at_geo_max_num_response_hdlr() fd=[%d] num=[%d]", fd, num);
}

static void mtk_at_geo_add_circle_response_hdlr(int fd, mtk_geo_fence_create_state state, int id) {
    LOGW("mtk_at_geo_add_circle_response_hdlr() fd=[%d] state=[%d] (0=success) id=[%d]", fd, state, id);
    if (g_at_geofence_add_with_wake_lock_release) {
        LOGD("Geofence Add Circle Request with wake lock release");
        //delay to do wake_unlock to ensure the msg can be deliveried to other process
        timer_start(g_wake_unlock_timer, LBS_EM_WAKE_LOCK_TIMEOUT);
    }
}

static void mtk_at_geo_alert_notify_hdlr(int fd, mtk_geo_alert_notify* notify) {
    LOGD("mtk_at_geo_alert_notify_hdlr() fd=[%d]", fd);
    mtk_at_dump_mtk_geo_alert_notify(notify);
}

static void mtk_at_geo_track_notify_hdlr(int fd, mtk_geo_track_state state, const char* date_time) {
    LOGD("mtk_at_geo_track_notify_hdlr() fd=[%d] state=[%d] (0=success) date_time=[%s]", fd, state, date_time);
}


static mtk_gnss_at_client_callbacks g_gnss_at_client_callbacks = {
    mtk_at_result_hdlr,
    mtk_at_gnss_nmea_notify_hdlr,
    mtk_at_gnss_nmea_end_notify_hdlr,
    mtk_at_gnss_satellite_notify_hdlr,
    mtk_at_gnss_location_notify_hdlr,
    mtk_at_gnss_agnss_location_notify_hdlr,
    mtk_at_gnss_device_reset_notify_hdlr,
    mtk_at_gnss_status_response_hdlr,
    mtk_at_gnss_ni_notify_request_hdlr,
    mtk_at_gnss_ref_location_request_hdlr,
    mtk_at_gnss_ref_time_request_hdlr,
    mtk_at_gnss_loopback_response_hdlr,
    mtk_at_gnss_cert_name_response_hdlr,
    mtk_at_geo_max_num_response_hdlr,
    mtk_at_geo_add_circle_response_hdlr,
    mtk_at_geo_alert_notify_hdlr,
    mtk_at_geo_track_notify_hdlr,
};

static void gnssadp_hdlr(int fd) {
    char buff[8192] = {0};
    int len = read(fd, buff, sizeof(buff));
    if(len <= 0) {
        LOGE("gnssadp_hdlr() the connection between EM and GNSS Adaptor is broken");
        close(fd);
        return;
    }

    buff[sizeof(buff)-1] = '\0';
    //TCP packets are combined together, need to extract them
    const char* p1 = buff;
    while(p1) {
        char buff2[3456] = {0};
        const char* p2 = strstr(p1, "\n");
        if(p2 != NULL) {
            len = (p2 - p1) + 1;
            //to support \r\n at beginning of the string
            if(len == 2 && p1[0] == '\r' && p1[1] == '\n') {
                p2++;
                p2 = strstr(p2, "\n");
                if(p2 == NULL) {
                    LOGE("unexpected data");
                    dump_buff(buff, len);
                    break;
                }
            }
            if (p2-p1 < sizeof(buff2)) {
                memcpy(buff2, p1, (p2 - p1) + 1);
            }
            mtk_at_client_parser(fd, buff2, &g_gnss_at_client_callbacks);
            p2++;
        }
        p1 = p2;
    }
}

static void nlp_client_read_and_decode(int fd) {
    char buff[8192] = {0};
    int len = read(fd, buff, sizeof(buff));
    if(len <= 0) {
        LOGE("nlp_client_read_and_decode() the connection between EM and NLP Service is broken");
        mtk_lbs_em_close_fd(fd);
        timer_start(g_connect_retry_timer, LBS_EM_RETRY_CONNECT_INTERVAL);
        return;
    }
    LOGW(" MtkNlp_client_read_and_decode fd=%d size=%d", fd, len);

    MtkNlp_decoded_location decLoc = MtkNlp_receiver_decode_location(fd, buff);
    LOGD("MtkNlp_Location_dump()");
    if (decLoc.nlp_location_valid) {
        SOCK_LOGD("  valid=[%d]", decLoc.nlp_location_valid);
        MtkNlp_location* data = &decLoc.nlp_location;
        SOCK_LOGD("  time=[%lld]", data->time);
        SOCK_LOGD("  elapsed_real_time_nanos=[%lld]", data->elapsed_real_time_nanos);
        SOCK_LOGD("  latitude=[%f]", data->latitude);
        SOCK_LOGD("  longitude=[%f]", data->longitude);
        SOCK_LOGD("  altitude_valid=[%d]", data->altitude_valid);
        SOCK_LOGD("  altitude=[%f]", data->altitude);
        SOCK_LOGD("  horizontal_accuracy_valid=[%d]", data->horizontal_accuracy_valid);
        SOCK_LOGD("  horizontal_accuracy=[%f]", data->horizontal_accuracy);
        SOCK_LOGD("  vertical_accuracy_valid=[%d]", data->vertical_accuracy_valid);
        SOCK_LOGD("  vertical_accuracy=[%f]", data->vertical_accuracy);
        SOCK_LOGD("  location_source_valid=[%d]", data->location_source_valid);
        SOCK_LOGD("  location_source=[%d]", data->location_source);
    } else {
        SOCK_LOGE("  No valid location reported");
    }
}


bool lbs_em_wake_lock() {
#if !defined(__LBS_EM_OS_LINUX__)    //no permission to open wake_lock on Linux
    int fd = open("/sys/power/wake_lock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }

    int ret = write(fd, LBS_EM_WAKE_LOCK_ID, strlen(LBS_EM_WAKE_LOCK_ID));
    if(ret == -1) {
        LOGE("write() failed id=[%s], reason=[%s]%d", LBS_EM_WAKE_LOCK_ID, strerror(errno), errno);
        close(fd);
        return false;
    }

    close(fd);
#endif
    return true;
}

bool lbs_em_wake_unlock() {
#if !defined(__LBS_EM_OS_LINUX__)    //no permission to open wake_unlock on Linux
    int fd = open("/sys/power/wake_unlock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }

    int ret = write(fd, LBS_EM_WAKE_LOCK_ID, strlen(LBS_EM_WAKE_LOCK_ID));
    if(ret == -1) {
        LOGE("write() failed id=[%s], reason=[%s]%d", LBS_EM_WAKE_LOCK_ID, strerror(errno), errno);
        close(fd);
        return false;
    }

    close(fd);
#endif
    return true;
}

void lbs_em_mutex_lock() {
    int ret = pthread_mutex_lock(&g_mutex);
    if(ret != 0) {
        LOGE("pthread_mutex_lock() failed, reason=[%s]%d", strerror(ret), ret);
    }
}

void lbs_em_mutex_unlock() {
    int ret = pthread_mutex_unlock(&g_mutex);
    if(ret != 0) {
        LOGE("pthread_mutex_unlock() failed, reason=[%s]%d", strerror(ret), ret);
    }
}

static void lbs_em_timer_wake_unlock_routine(int id) {
    //do not use the internal msg or it will cause infinite loop in epoll_wait
    lbs_em_mutex_lock();
    if(g_wake_lock_acquired) {
        if(lbs_em_wake_unlock()) {
            g_wake_lock_acquired = false;
        }
    }
    LOGD("lbs_em_timer_wake_unlock_routine g_wake_lock_acquired=%d", g_wake_lock_acquired);
    lbs_em_mutex_unlock();
}

static void lbs_em_acquire_wake_lock_mutex() {
    lbs_em_mutex_lock();
    timer_stop(g_wake_unlock_timer);
    if(!g_wake_lock_acquired) {
        g_wake_lock_acquired = lbs_em_wake_lock();
    }
    LOGD("lbs_em_acquire_wake_lock g_wake_lock_acquired=%d", g_wake_lock_acquired);
    lbs_em_mutex_unlock();
}

// ------------- message receiving thread -----------------
static void* event_thread(void *arg) {
    #define MAX_EPOLL_EVENT 50
    struct epoll_event events[MAX_EPOLL_EVENT];

    msleep2(300);
    LOGD(" ");
    g_epfd = epoll_create(MAX_EPOLL_EVENT);
    if(g_epfd == -1) {
        LOGE("ERR: event_thread()  epoll_create() fail reason=[%s]", strerror(errno));
        return NULL;
    }

    g_agps_debug_fd = agps_debug_interface_get_fd();
    if(g_agps_debug_fd < 0) {
        LOGE("ERR: event_thread()  agps_debug_interface_get_fd() fail");
    } else {
        epoll_add_fd3(g_epfd, g_agps_debug_fd);
    }

    g_mnld_basic_fd = mnldinf_basic_register("LBS EM");
    if(g_mnld_basic_fd < 0) {
        LOGE("ERR: event_thread()  mnldinf_basic_register fail");
    } else {
        epoll_add_fd3(g_epfd, g_mnld_basic_fd);
        mnldinf_basic_client_cap cap;
        cap.support_gnss = true;
        mnldinf_basic_capability_config(g_mnld_basic_fd, &cap);
    }

    g_mnld_ext_fd = mnldinf_ext_register("LBS EM");
    if(g_mnld_ext_fd < 0) {
        LOGE("ERR: event_thread()  mnldinf_basic_register fail");
    } else {
        epoll_add_fd3(g_epfd, g_mnld_ext_fd);
        mnldinf_ext_client_cap cap;
        cap.support_gnss = true;
        mnldinf_ext_capability_config(g_mnld_ext_fd, &cap);
    }


    g_gnssadp_fd = socket_tcp_client_connect_local(true, GNSSADP_AT_SOCKET);
    if(g_gnssadp_fd < 0) {
        LOGE("ERR: event_thread()  socket_tcp_client_connect_local(true, GNSSADP_AT_SOCKET) fail");
    } else {
        epoll_add_fd3(g_epfd, g_gnssadp_fd);
    }


    g_internal_fd = socket_udp_server_bind_local(true, LBSEM_INTERNAL_SOCKET);
    if(g_internal_fd < 0) {
        LOGE("socket_udp_server_bind_local(true, LBSEM_INTERNAL_SOCKET) failed");
        exit(1);
    } else {
        epoll_add_fd3(g_epfd, g_internal_fd);
    }


    g_connect_retry_timer = timer_init(mtk_lbs_em_timer_retry_port_routine, 0);
    if(g_connect_retry_timer == NULL) {
        LOGE("timer_init(mtk_lbs_em_timer_retry_port_routine, 0) failed");
        exit(1);
    }

    g_wake_unlock_timer = timer_init(lbs_em_timer_wake_unlock_routine, 0);
    if(g_wake_unlock_timer == NULL) {
        LOGE("timer_init(lbs_em_timer_wake_unlock_routine, 0) failed");
        exit(1);
    }

    mtk_socket_client_init_local(&g_nlpservice_fd, NLP_SERVICE_CHANNEL, SOCK_NS_ABSTRACT);// nlp client
    g_nlpservice_fd.fd = MtkNlp_get_fd();
    if(g_nlpservice_fd.fd < 0) {
        LOGE("ERR: event_thread()  socket_tcp_client_connect_local(true, NLP_SERVICE_CHANNEL) fail");
    } else {
        epoll_add_fd3(g_epfd, g_nlpservice_fd.fd);
    }

    lbs_em_acquire_wake_lock_mutex();


    while(1) {
        int i;
        int n;

        n = epoll_wait(g_epfd, events, MAX_EPOLL_EVENT , -1);
        if(n == -1) {
            if(errno == EINTR) {
                continue;
            } else {
                LOGE("ERR: event_thread()  epoll_wait failure reason=[%s]", strerror(errno));
                return NULL;
            }
        }

        for(i = 0; i < n; i++) {
            // agps debug interface
            if(events[i].data.fd == g_agps_debug_fd) {
                if(events[i].events & EPOLLIN) {
                    agps_debug_message_hdlr(g_epfd);
                }
            }
            // mnldinf basic interface
            if(events[i].data.fd == g_mnld_basic_fd) {
                if(events[i].events & EPOLLIN) {
                    mnldinf_basic_cmd_hdlr(g_mnld_basic_fd, &g_mnld_basic_callbacks);                }
            }
            // mnldinf ext interface
            if(events[i].data.fd == g_mnld_ext_fd) {
                if(events[i].events & EPOLLIN) {
                    mnldinf_ext_cmd_hdlr(g_mnld_ext_fd, &g_mnld_ext_callbacks);
                }
            }

            if(events[i].data.fd == g_gnssadp_fd) {
                if(events[i].events & EPOLLIN) {
                    gnssadp_hdlr(g_gnssadp_fd);
                }
            }

            if(events[i].data.fd == g_nlpservice_fd.fd) {
                if(events[i].events & EPOLLIN) {
                    nlp_client_read_and_decode(g_nlpservice_fd.fd);
                }
            }
            // internal event interface
            if(events[i].data.fd == g_internal_fd) {
                if(events[i].events & EPOLLIN) {
                    mtk_lbs_em_epoll_internal_hdlr(g_internal_fd);
                }
            }
        }
    }

    g_wake_lock_acquired = false;
    lbs_em_timer_wake_unlock_routine(0);
    //delay to do wake_unlock to ensure the msg can be deliveried to other process
    //timer_start(g_wake_unlock_timer, LBS_EM_WAKE_LOCK_TIMEOUT);

    return NULL;
}

// ------------- main() -----------------

int main() {
    sem_init(&g_cert_sem, 0 /*pshared*/, 0 /*value*/);

    pthread_t pthread1;
    pthread_create(&pthread1, NULL, event_thread, NULL);

    g_geo_list.num = 0;

    return mtk_lbs_em_start();
}

/*
printf("\E[0;30;47m");
printf("\E[0m");

the type of word:
0 normal
1 bold
4 underline
5 gray in background
7 switch the color of word and the color of background

the color of word:
30 black
31 red
32 green
33 yellow
34 blue
35 purple
36 light green
37 white

the color of background:
40 black
41 red
42 green
43 yellow
44 blue
45 purple
46 light green
47 white
*/

