#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <stddef.h>  // offsetof
#include <stdarg.h>
#include <sys/stat.h>
#include <unistd.h>  // usleep
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <arpa/inet.h>  // inet_addr
#include <sys/un.h>  // struct sockaddr_un
#include <pthread.h>
#include <sys/epoll.h>
#include <signal.h>
#include <semaphore.h>
#include <stdbool.h>

#include "mnldinf_utility.h"
#include "mnldinf_log.h"
#include "mnldinf_common.h"
#include "mnldinf_data_coder.h"

#ifdef LOG_TAG
#undef LOG_TAG
#define LOG_TAG "mtk_lbs_utility"
#endif
#if 0
#define LOGD(...)   { printf(__VA_ARGS__); printf("\n"); fflush(stdout); }
#define LOGW(...)   { printf("\E[1;35;40m"); printf(__VA_ARGS__); printf("\E[0m"); printf("\n"); fflush(stdout); }
#define LOGE(...)   { printf("\E[1;31;40m"); printf(__VA_ARGS__); printf("\E[0m"); printf("\n"); fflush(stdout); }
#define LOGI LOGD
#endif
void mnldinf_msleep(int interval) {
    usleep(interval * 1000);
}

// in millisecond
time_t mnldinf_get_tick() {
    struct timespec ts;
    if (clock_gettime(CLOCK_MONOTONIC, &ts) == -1) {
        LOGE("clock_gettime failed reason=[%s]", strerror(errno));
        return -1;
    }
    return (ts.tv_sec*1000) + (ts.tv_nsec/1000000);
}

// in millisecond
time_t mnldinf_get_time_in_millisecond() {
    struct timespec ts;
    if (clock_gettime(CLOCK_REALTIME, &ts) == -1) {
        LOGE("mnldinf_get_time_in_millisecond  failed reason=[%s]", strerror(errno));
        return -1;
    }
    return ((long long)ts.tv_sec*1000) + ((long long)ts.tv_nsec/1000000);
}

//#ifdef __LINUX_OS__
#if !defined(__ANDROID_OS__)
/*************************************************
* Timer
**************************************************/
// -1 means failure
timer_t mnldinf_init_timer_id(timer_callback cb, int id) {
    struct sigevent sevp;
    timer_t timerid;

    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_value.sival_int = id;
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = cb;

    if (timer_create(CLOCK_MONOTONIC, &sevp, &timerid) == -1) {
        LOGE("timer_create  failed reason=[%s]", strerror(errno));
        return INVALID_TIMERID;
    }
    //LOGD("timerid init:%lu", (unsigned long)timerid);
    return timerid;
}

// -1 means failure
timer_t mnldinf_init_timer(timer_callback cb) {
    return mnldinf_init_timer_id(cb, 0);
}

// -1 means failure
int mnldinf_start_timer(timer_t timerid, int milliseconds) {
    if(timerid != INVALID_TIMERID) {
        struct itimerspec expire;
        expire.it_interval.tv_sec = 0;
        expire.it_interval.tv_nsec = 0;
        expire.it_value.tv_sec = milliseconds/1000;
        expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
        return timer_settime(timerid, 0, &expire, NULL);
    } else {
        LOGE("invalid timerid:%lu", (unsigned long)timerid);
        return -1;
    }
}

// -1 means failure
int mnldinf_stop_timer(timer_t timerid) {
    return mnldinf_start_timer(timerid, 0);
}

// -1 means failure
int mnldinf_deinit_timer(timer_t timerid) {
    LOGD("timerid deinit:%lu", (unsigned long)timerid);
    if(timerid != INVALID_TIMERID) {
        if (timer_delete(timerid) == -1) {
            LOGE("timer_delete error:%s", strerror(errno));
            return -1;
        }
    } else {
        LOGE("invalid timerid:%lu", (unsigned long)timerid);
        return -1;
    }

    return 0;
}

#endif
/*************************************************
* Epoll
**************************************************/
// -1 means failure
int mnldinf_epoll_add_fd(int epfd, int fd) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = EPOLLIN;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("mnldinf_epoll_add_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

// -1 failed
int mnldinf_epoll_add_fd2(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("mnldinf_epoll_add_fd2() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

int mnldinf_epoll_del_fd(int epfd, int fd) {
    struct epoll_event  ev;
    int                 ret;

    if (epfd == -1)
        return -1;

    ev.events  = EPOLLIN;
    ev.data.fd = fd;
    do {
        ret = epoll_ctl(epfd, EPOLL_CTL_DEL, fd, &ev);
    } while (ret < 0 && errno == EINTR);
    return ret;
}

// -1 failed
int mnldinf_epoll_mod_fd(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    if (epoll_ctl(epfd, EPOLL_CTL_MOD, fd, &ev) == -1) {
        LOGE("mnldinf_epoll_mod_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

/*************************************************
* Local UDP Socket
**************************************************/
// -1 means failure
int mnldinf_socket_bind_udp(const char* path) {
    int fd;
    struct sockaddr_un addr;

    fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("mnldinf_socket_bind_udp() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    LOGD("fd=%d,path=%s\n", fd, path);

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    MNLD_STRNCPY(addr.sun_path + 1, path,sizeof(addr.sun_path) - 1);
    addr.sun_family = AF_UNIX;
    unlink(path);

    if (bind(fd, (const struct sockaddr *)&addr, sizeof(addr)) < 0) {
        LOGE("mnldinf_socket_bind_udp() bind() failed path=[%s] reason=[%s]%d",
            addr.sun_path+1, strerror(errno), errno);
        close(fd);
        return -1;
    }else
        LOGI("bind ok path=[%s]", addr.sun_path+1);
    return fd;
}

// -1 means failure
int mnldinf_socket_set_blocking(int fd, int blocking) {
    if (fd < 0) {
        LOGE("mnldinf_socket_set_blocking() invalid fd=%d", fd);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (flags == -1) {
        LOGE("mnldinf_socket_set_blocking() fcntl() failed invalid flags=%d reason=[%s]%d",
            flags, strerror(errno), errno);
        return -1;
    }

    flags = blocking ? (flags&~O_NONBLOCK) : (flags|O_NONBLOCK);
    return (fcntl(fd, F_SETFL, flags) == 0) ? 0 : -1;
}

// -1 means failure
int mnldinf_safe_sendto(const char* path, const char* buff, int len) {
    int ret = 0;
    struct sockaddr_un addr;
    int retry = 10;
    int fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("mnldinf_safe_sendto() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1){
        LOGE("fcntl failed reason=[%s]%d",
                    strerror(errno), errno);

        close(fd);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    MNLD_STRNCPY(addr.sun_path + 1, path,sizeof(addr.sun_path) - 1);
    addr.sun_family = AF_UNIX;

    while ((ret = sendto(fd, buff, len, 0,
        (const struct sockaddr *)&addr, sizeof(addr))) == -1) {
        if (errno == EINTR) {
            LOGE("errno==EINTR\n");
            continue;
        }
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                LOGE("errno==EAGAIN\n");
                continue;
            }
        }
        LOGE("mnldinf_safe_sendto() sendto() failed path=[%s] ret=%d reason=[%s]%d",
            path, ret, strerror(errno), errno);
        break;
    }

    close(fd);
    return ret;
}

// -1 means failure
int mnldinf_safe_recvfrom(int fd, char* buff, int len) {
    int ret = 0;
    int retry = 10;
    int flags = fcntl(fd, F_GETFL, 0);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1){
        LOGE("fcntl failed reason=[%s]%d",
                    strerror(errno), errno);

        close(fd);
        return -1;
    }

    while ((ret = recvfrom(fd, buff, len, 0,
         NULL, NULL)) == -1) {
        LOGW("mnldinf_safe_recvfrom() ret=%d len=%d", ret, len);
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("mnldinf_safe_recvfrom() recvfrom() failed reason=[%s]%d",
            strerror(errno), errno);
        break;
    }
    return ret;
}

/*
------------------------
| Length |  Message Body  |
------------------------
*/
// -1 means failure
int mnldinf_tcp_send(int fd, const char* buff, int len) {
    char buff_send[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;

    mnldinf_put_binary(buff_send, &offset, buff, len);  //Put length to the head

    int ret = write(fd, buff_send, offset);
    if(ret == -1) {
        LOGE(" write() failed, reason=[%s]%d", strerror(errno), errno);
    }

    //LOGD("send %d, ret %d", len, ret);
    return ret;
}

int mnldinf_safe_recv(int fd, char* buff, int len) {
    int ret = 0;
    int retry = 10;

    while ((ret = read(fd, buff, len)) == -1) {
        LOGW("mnldinf_safe_recvfrom() ret=%d len=%d", ret, len);
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("mnldinf_safe_recvfrom() recvfrom() failed reason=[%s]%d",
            strerror(errno), errno);
        break;
    }
    return ret;
}


/******************************************************************************
* Socket
******************************************************************************/

//-1 means fail or serverfd is returned
int mnldinf_socket_tcp_server_bind_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("mnldinf_socket_tcp_server_bind_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("mnldinf_socket_tcp_server_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(bind(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("mnldinf_socket_tcp_server_bind_local() bind() failed, abstruct=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    if(listen(fd, 5) == -1) {
        LOGW("mnldinf_socket_tcp_server_bind_local() listen() failed, reason=[%s]%d",
            strerror(errno), errno);
    }
    return fd;
}

//-1 means fail or new clientfd is returned
int mnldinf_socket_tcp_server_new_connect(int serverfd) {
    int newfd;
    struct sockaddr_in addr;
    socklen_t len = sizeof(addr);

    newfd = accept(serverfd, (struct sockaddr*)&addr, &len);
    if(newfd == -1) {
        LOGE("mnldinf_socket_tcp_server_new_connect() accept() failed, serverfd=%d reason=[%s]%d",
            serverfd, strerror(errno), errno);
        return -1;
    }
    return newfd;
}

//-1 means fail or serverfd is returned
int mnldinf_socket_tcp_client_connect_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("mnldinf_socket_tcp_client_connect_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("mnldinf_socket_tcp_client_connect_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("mnldinf_socket_tcp_client_connect_local() connect() failed, abstruct=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }

    return fd;
}


//-1 means fail or serverfd is returned
int mnldinf_socket_udp_server_bind_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("mnldinf_socket_udp_server_bind_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("mnldinf_socket_udp_server_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(bind(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("mnldinf_socket_udp_server_bind_local() bind() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

//-1 means fail or clientfd is returned
int mnldinf_socket_udp_client_create_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("mnldinf_socket_udp_client_create_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("mnldinf_socket_udp_client_create_local() connect() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

bool mnldinf_socket_udp_exist_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        MNLD_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_is_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return false;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        close(fd);
        return false;
    }
    close(fd);
    return true;

}

void mnldinf_buffer_initialize(cyclical_buffer_t *buffer, char *buffer_body, unsigned int buffer_size) {
   // Set up buffer manipulation pointers
   // end_buffer points to the next byte after the buffer
   buffer->start_buffer = buffer_body;
   buffer->end_buffer   = buffer->start_buffer + buffer_size;
   buffer->next_read    = buffer->start_buffer;
   buffer->next_write   = buffer->start_buffer;
   buffer->buffer_size = buffer_size;
   return;
}

int mnldinf_put_msg_to_cycle(cyclical_buffer_t *cyc_buffer, char *buff, int len) {
    int i;
#ifdef MNLDINF_DUMP_CMD_RAW
    LOGD("mnldinf_put_msg_to_cycle:%d", len);
#endif
    for (i = 0; i < len; i++)
    {
        *(cyc_buffer->next_write++) = buff[i];
        if (cyc_buffer->next_write == cyc_buffer->end_buffer){
            cyc_buffer->next_write = cyc_buffer->start_buffer;
        }

        if (cyc_buffer->next_write == cyc_buffer->next_read){
        #ifdef MNLDINF_DUMP_CMD_RAW
            LOGE("mnldinf_put_msg_to_cycle buffer full!\r\n");
        #endif
            return -1;
        }
    }

    return 0;
}
/******************************************
Get one message from cycle buffer
******************************************/
int mnldinf_get_one_msg(cyclical_buffer_t *cyc_buffer, char *buff) {
    char *next_write, *next_read;
    int data_size, i, header_len;
    char buffer[HAL_MNL_BUFF_SIZE_SND] = {0};
    int data_len = 0;
    int return_len = 0;

    next_write = cyc_buffer->next_write;
    next_read = cyc_buffer->next_read;
    if (cyc_buffer->next_read == next_write)
    {
        //buffer empty
        return -1;
    }

    header_len = sizeof(int);
    /*Compute data length*/
    if (cyc_buffer->next_read < next_write)
    {
        data_size = (unsigned long)next_write - (unsigned long)cyc_buffer->next_read;
    }
    else
    {
        data_size = (unsigned long)cyc_buffer->end_buffer - (unsigned long)cyc_buffer->next_read +
                     (unsigned long)next_write - (unsigned long)cyc_buffer->start_buffer;
    }

    /*Copy data header to buffer*/
    if (data_size >= header_len)
    {
        for (i = 0; i < header_len; i++)
        {
            buffer[i] = *(next_read++);
            if (next_read == cyc_buffer->end_buffer){
                next_read = cyc_buffer->start_buffer;
            }
        }

        memcpy(&data_len, buffer, sizeof(int));
    #ifdef MNLDINF_DUMP_CMD_RAW
        LOGD("data len:%d, header len:%d", data_len, header_len);
    #endif
        if (data_len <= data_size) {
            for (i = 0; i < (data_len + header_len); i++)
            {
                buff[i] = *(cyc_buffer->next_read++);
            #ifdef MNLDINF_DUMP_CMD_RAW
                LOGD("0x%x", buff[i]);
            #endif
                return_len++;
                if (cyc_buffer->next_read == cyc_buffer->end_buffer){
                    cyc_buffer->next_read = cyc_buffer->start_buffer;
                }
            }
        }
        else {
            LOGD("func:%s, line:%d, no enough data", __func__, __LINE__);
            //no enough data
            return -2;
        }

    }
    else
    {
        LOGD("func:%s, line:%d, no enough data", __func__, __LINE__);
        //no enough data
        return -2;
    }

    return return_len;
}

wake_lock_ctx wlock_ctx;

void mnldinf_wake_lock() {
    int fd = open("/sys/power/wake_lock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    int ret = write(fd, MNLDINF_WAKE_LOCK_ID, strlen(MNLDINF_WAKE_LOCK_ID));
    if(ret == -1) {
        LOGE("write() failed id=[%s], reason=[%s]%d", MNLDINF_WAKE_LOCK_ID, strerror(errno), errno);
        close(fd);
        return;
    }

    close(fd);

    return;
}

void mnldinf_wake_unlock() {
    int fd = open("/sys/power/wake_unlock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    int ret = write(fd, MNLDINF_WAKE_LOCK_ID, strlen(MNLDINF_WAKE_LOCK_ID));
    if(ret == -1) {
        LOGE("write() failed id=[%s], reason=[%s]%d", MNLDINF_WAKE_LOCK_ID, strerror(errno), errno);
        close(fd);
        return;
    }

    close(fd);

    return;
}

int mnldinf_wakeup_mutex_lock() {
    int ret = pthread_mutex_lock(&wlock_ctx.mutex);
    if(ret != 0) {
        LOGE("pthread_mutex_lock() failed, reason=[%s]%d", strerror(ret), ret);
    }

    return ret;
}

int mnldinf_wakeup_mutex_unlock() {
    int ret = pthread_mutex_unlock(&wlock_ctx.mutex);
    if(ret != 0) {
        LOGE("pthread_mutex_unlock() failed, reason=[%s]%d", strerror(ret), ret);
    }

    return ret;
}

void mnldinf_wake_lock_take() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    time_t time_cur = 0;
    time_cur = mnldinf_get_time_in_millisecond();

    if(time_cur > (wlock_ctx.time_last_refresh + MNLDINF_WAKE_LOCK_LATENCY)) {
        int mutex_ret = 0;
        mnldinf_stop_timer(wlock_ctx.unlock_timer);
        mutex_ret = mnldinf_wakeup_mutex_lock();
        if(wlock_ctx.wake_lock_acquired == false) {
            wlock_ctx.wake_lock_acquired = true;
            mnldinf_wake_lock();
        }
        if(mutex_ret == 0) {
            mnldinf_wakeup_mutex_unlock();
        }
    }
#endif
}

void mnldinf_wake_lock_give() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    time_t time_cur = 0;
    //delay to do wake_unlock to ensure the msg can be deliveried to other process
    time_cur = mnldinf_get_time_in_millisecond();
    if(time_cur > (wlock_ctx.time_last_refresh + MNLDINF_WAKE_LOCK_LATENCY)) {  //Only refresh timer when over latency time
        mnldinf_start_timer(wlock_ctx.unlock_timer, MNLDINF_WAKE_LOCK_TIMEOUT);
        wlock_ctx.time_last_refresh = mnldinf_get_time_in_millisecond();
    }
#endif
}

#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
static void mnldinf_timer_wake_unlock_routine(int id) {
    int mutex_ret = 0;
    //do not use the internal msg or it will cause infinite loop in epoll_wait
    mutex_ret = mnldinf_wakeup_mutex_lock();
    if(wlock_ctx.wake_lock_acquired == true) {
        wlock_ctx.wake_lock_acquired = false;
        mnldinf_wake_unlock();
    }
    if(mutex_ret == 0) {
        mnldinf_wakeup_mutex_unlock();
    }
}
#endif

void mnldinf_wake_lock_init() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    int ret = 0;
    int read_len = 0;
    char buff_read[HAL_MNL_BUFF_SIZE] = {0};

    int fd = open("/sys/power/wake_lock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    read_len = read(fd, buff_read, sizeof(buff_read)-1);
    buff_read[HAL_MNL_BUFF_SIZE -1] = 0;
    if(read_len >= strlen(MNLDINF_WAKE_LOCK_ID) && strstr(buff_read, MNLDINF_WAKE_LOCK_ID) != NULL) {
        mnldinf_wake_unlock();
    }
    LOGD("wake_lock:[%s]", buff_read);
    close(fd);

    memset(&wlock_ctx, 0, sizeof(wlock_ctx));
    wlock_ctx.wake_lock_acquired = false;
    wlock_ctx.unlock_timer = mnldinf_init_timer(mnldinf_timer_wake_unlock_routine);
    if(wlock_ctx.unlock_timer == INVALID_TIMERID) {
        LOGE("init_timer(mnldinf_timer_wake_unlock_routine, 0) failed");
        exit(1);
    }

    ret = pthread_mutex_init(&wlock_ctx.mutex, NULL);
    if(ret != 0) {
        LOGE("pthread_mutex_init() failed, reason=[%s]%d", strerror(errno), errno);
        exit(1);
    }
#endif
}

void mnldinf_wake_lock_deinit() {
#if !defined(__LIBMNL_SIMULATOR__)    //no permission to open wake_lock on Linux
    int read_len = 0;
    char buff_read[HAL_MNL_BUFF_SIZE] = {0};

    int fd = open("/sys/power/wake_lock", O_RDWR | O_NONBLOCK);
    if(fd == -1) {
        LOGE("open() failed, reason=[%s]%d", strerror(errno), errno);
        return;
    }

    read_len = read(fd, buff_read, sizeof(buff_read)-1);
    buff_read[HAL_MNL_BUFF_SIZE -1] = 0;
    if(read_len >= strlen(MNLDINF_WAKE_LOCK_ID) && strstr(buff_read, MNLDINF_WAKE_LOCK_ID) != NULL) {
        mnldinf_wake_unlock();
    }
    LOGD("wake_lock:[%s]", buff_read);
    close(fd);
    mnldinf_deinit_timer(wlock_ctx.unlock_timer);
    wlock_ctx.unlock_timer = INVALID_TIMERID;
    pthread_mutex_destroy(&wlock_ctx.mutex);
#endif
}

