#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>
#include <dlfcn.h>
#include <stdlib.h>
#include <sys/epoll.h> //epoll

#include "mnldinf_ext.h"
#include "mnldinf_basic.h"

#include "mnldinf_utility.h"
#include "mnldinf_log.h"

#ifdef LOG_TAG
#undef LOG_TAG
#define LOG_TAG "mnldinftest"
#endif
#define MNLD_TEST "mnldinf_test"

#define MAX_EPOLL_EVENT 10

int mnldinf_basic_fd = -1;
int mnldinf_ext_fd = -1;

//=========================================================

void dump_gps_location(gps_location location) {
    LOGD("[dump_gps_location]flags:%d, lat:%f, lng:%f, alt:%f, speed:%f, bearing:%f, h_accuracy:%f, v_acc:%f, s_acc:%f, b_acc:%f, timestamp:%ld, pdop:%.2f, hdop:%.2f, vdop:%.2f"
        , location.flags, location.lat, location.lng, location.alt, location.speed, location.bearing
        , location.h_accuracy, location.v_accuracy, location.s_accuracy, location.b_accuracy, location.timestamp
        , location.pdop, location.hdop, location.vdop);
}

void dump_gnss_sv_info(gnss_sv_info sv) {
    unsigned int i = 0;
    LOGD("[dump_gnss_sv_info], sv_num:%d", sv.num_svs);
    for(i = 0; i < sv.num_svs; i++) {
        LOGD("SV[%d], cons:%d, Cn0dBHz:%f, elev:%f, azim:%f, flags:%d, cf:%f"
            , sv.sv_list[i].svid, sv.sv_list[i].constellation, sv.sv_list[i].c_n0_dbhz, sv.sv_list[i].elevation
            , sv.sv_list[i].azimuth, sv.sv_list[i].flags, sv.sv_list[i].carrier_frequency);
    }
}

static void update_location(gps_location location) {
    LOGD("");
    dump_gps_location(location);
}

static void update_gps_status(gps_status status) {
    LOGD("status=%d", status);
}

static void update_gps_sv(gnss_sv_info sv) {
    LOGD("");
    dump_gnss_sv_info(sv);
}

#define MNLD_TEST_CMD_CNT_MAX 5
#define GPS_DELETE_EPHEMERIS        0x0001
#define GPS_DELETE_ALMANAC          0x0002
#define GPS_DELETE_POSITION         0x0004
#define GPS_DELETE_TIME             0x0008
#define GPS_DELETE_IONO             0x0010
#define GPS_DELETE_UTC              0x0020
#define GPS_DELETE_HEALTH           0x0040
#define GPS_DELETE_SVDIR            0x0080
#define GPS_DELETE_SVSTEER          0x0100
#define GPS_DELETE_SADATA           0x0200
#define GPS_DELETE_RTI              0x0400
#define GPS_DELETE_CELLDB_INFO      0x8000
#define GPS_DELETE_ALL              0xFFFF

#define DUMP_BASIC_CAP(cap) do {\
    LOGD("[%s]:gnss_support=%d", __func__, (cap)->support_gnss);\
} while(0)

#define DUMP_EXT_CAP(cap) do {\
    LOGD("[%s]:gnss_support=%d", __func__, (cap)->support_gnss);\
} while(0)

static void update_nmea(int64_t timestamp, const char* nmea, int length) {
    static int nmea_cnt = 100;
    static int test_cnt = 1024;
    /*LOGD("timestamp=%ld nmea=[%s] length=%d",
        timestamp, nmea, length);*/
    if(test_cnt > 0 && nmea_cnt-- == 0 && mnldinf_basic_fd > 0) {
        if(test_cnt-- >= 0) {
            mnldinf_basic_gnss_stop(mnldinf_basic_fd);
            mnldinf_basic_gnss_cleanup(mnldinf_basic_fd);
            LOGD("GPS restart cnt(%d)", 1024-test_cnt);

            switch(test_cnt%4) {
                case 0:  //Hot
                    mnldinf_ext_gnss_delete_aiding_data(mnldinf_ext_fd, GPS_DELETE_RTI);
                    break;
                case 1:  //Warm
                    mnldinf_ext_gnss_delete_aiding_data(mnldinf_ext_fd, GPS_DELETE_EPHEMERIS);
                    break;
                case 2:  //Cold
                    mnldinf_ext_gnss_delete_aiding_data(mnldinf_ext_fd, GPS_DELETE_EPHEMERIS |
                GPS_DELETE_POSITION | GPS_DELETE_TIME | GPS_DELETE_IONO |
                GPS_DELETE_UTC | GPS_DELETE_HEALTH);
                    break;
                case 3:  //Full
                    mnldinf_ext_gnss_delete_aiding_data(mnldinf_ext_fd, GPS_DELETE_ALL);
                    break;
                default:
                    mnldinf_ext_gnss_delete_aiding_data(mnldinf_ext_fd, GPS_DELETE_RTI);
                    break;
            }
            mnldinf_basic_gnss_init(mnldinf_basic_fd);
            mnldinf_basic_gnss_start(mnldinf_basic_fd);

            mnldinf_ext_set_ttff_acc(mnldinf_ext_fd, test_cnt%3);
            nmea_cnt = 100;  //Reset nmea_cnt
        }
    }
}

static void update_nmea_done(void) {
    LOGD("NMEA Done!!!");
}

static void update_gnss_measurements(gnss_data *data) {
    LOGD("");
    //dump_gnss_data(data);
}

static void basic_capability_update(mnldinf_basic_server_cap *cap) {
    mnldinf_basic_server_cap basic_server_cap;
    mnldinf_basic_client_cap basic_client_cap = {
        .support_gnss = true,
    };

    memcpy(&basic_server_cap, cap, sizeof(mnldinf_basic_server_cap));
    mnldinf_basic_capability_config(mnldinf_basic_fd, &basic_client_cap);
    DUMP_BASIC_CAP(&basic_client_cap);
}

static void ext_capability_update(mnldinf_ext_server_cap *cap) {
    mnldinf_ext_server_cap ext_server_cap;
    mnldinf_ext_client_cap ext_client_cap = {
        .support_gnss = true,
    };

    memcpy(&ext_server_cap, cap, sizeof(mnldinf_ext_server_cap));
    mnldinf_ext_capability_config(mnldinf_ext_fd, &ext_client_cap);
    DUMP_EXT_CAP(&ext_client_cap);
}

static void update_gnss_navigation(gnss_nav_msg *msg) {
    LOGD("");
    //dump_gnss_nav_msg(msg);
}

static void request_wakelock() {
    LOGD("");
}
static void release_wakelock() {
    LOGD("");
}
static void request_utc_time() {
    LOGD("");
}
static void request_data_conn(struct sockaddr_storage* addr, agps_type type) {
    LOGD("");
}
static void release_data_conn() {
    LOGD("");
}
static void request_ni_notify(int session_id, agps_ni_type ni_type, agps_notify_type type, const char* requestor_id,
            const char* client_name, ni_encoding_type requestor_id_encoding,
            ni_encoding_type client_name_encoding) {
    /*LOGD("  session_id=%d type=%d requestor_id=[%s] client_name=[%s] requestor_id_encoding=%d client_name_encoding=%d",
        session_id, type, requestor_id, client_name, requestor_id_encoding, client_name_encoding);*/
        LOGD("");
}
static void request_set_id(request_setid flags) {
    //LOGD("  flags=0x%x", flags);
    LOGD("");
}
static void request_ref_loc(request_refloc flags) {
    //LOGD("  flags=0x%x", flags);
    LOGD("");
}
static void output_vzw_debug(const char* str) {
    LOGD("");
}

static void update_gnss_name(const char* name, int length) {
    LOGD("length=%d, name=%s", length, name);
    LOGD("");
}

static void request_nlp(bool independent_from_gnss, bool is_user_emergency) {
    LOGD("request nlp[%d], emergency:%d", independent_from_gnss, is_user_emergency);
}

void mnldtest_nfw_access_notify(mnldinf_nfw_notification nfw_notify) {
    LOGD("");
}

void mnldtest_basic_connection_broken() {
    LOGD("");
}

void mnldtest_ext_connection_broken() {
    LOGD("");
}

void mnldtest_ext_agps_location_update(mnldinf_agps_location location){
    LOGD("lat:%f, lng:%f, %lld, type:%d", location.latitude, location.longitude, location.timestamp, location.type);
}

static mnldinf_ext mnldinf_ext_cbs = {
    mnldtest_ext_connection_broken,
    request_wakelock,
    release_wakelock,

    request_utc_time,
    request_nlp,
    request_ni_notify,
    request_data_conn,
    release_data_conn,

    request_set_id,
    request_ref_loc,
    output_vzw_debug,

    update_gnss_name,
    update_gnss_navigation,
    mnldtest_nfw_access_notify,
    ext_capability_update,
    mnldtest_ext_agps_location_update,
};

static mnldinf_basic mnldinf_basic_cbs = {
    mnldtest_basic_connection_broken,
    update_location,
    update_gps_status,
    update_gps_sv,
    update_nmea,
    update_nmea_done,
    update_gnss_measurements,
    basic_capability_update,
};

static void mnld_client_stdin_hdlr(int fd) {

}

int main() {
    LOGD("mnld interface start, version 1.00");

    mnldinf_basic_fd = mnldinf_basic_register(MNLD_TEST);
    if(mnldinf_basic_fd == -1) {
        LOGE("mnldinf_basic_register failed: %s", strerror(errno));
        exit(1);
    }

    mnldinf_ext_fd = mnldinf_ext_register(MNLD_TEST);
    if(mnldinf_ext_fd == -1) {
        LOGE("mnldinf_ext_register failed: %s", strerror(errno));
        exit(1);
    }

    struct epoll_event events[MAX_EPOLL_EVENT];
    int epfd = epoll_create(MAX_EPOLL_EVENT);
    if(epfd == -1) {
        LOGE("epoll_create() failed");
        exit(1);
    }

    mnldinf_epoll_add_fd(epfd, mnldinf_basic_fd);
    mnldinf_epoll_add_fd(epfd, mnldinf_ext_fd);

    mnldinf_epoll_add_fd(epfd, STDIN_FILENO);

    mnldinf_basic_gnss_init(mnldinf_basic_fd);

    mnldinf_basic_gnss_start(mnldinf_basic_fd);
    mnldinf_basic_gnss_measurement_enable(mnldinf_basic_fd, 1);
    mnldinf_ext_epo_enable(mnldinf_ext_fd, 0x1);
    long long sv_bl = 0xFAFA;
    mnldinf_ext_set_sv_blacklist(mnldinf_ext_fd, &sv_bl, 16);

    while(true) {
        int i, n;
        LOGD("wait event");
        n = epoll_wait(epfd, events, MAX_EPOLL_EVENT, -1);
        if(n == -1) {
            if(errno == EINTR) {
                continue;
            } else {
                LOGE("epoll_wait() failed reason=[%s]%d", strerror(errno), errno);
                crash_with_log();
            }
        }

        for(i = 0; i < n; i++) {
            int fd = events[i].data.fd;
            if(fd == mnldinf_basic_fd) {
                if(events[i].events & EPOLLIN) {
                    LOGD("basic hdlr");
                    mnldinf_basic_cmd_hdlr(fd, &mnldinf_basic_cbs);
                }
            } else if(fd == mnldinf_ext_fd) {
                if(events[i].events & EPOLLIN) {
                    LOGD("ext hdlr");
                    mnldinf_ext_cmd_hdlr(fd, &mnldinf_ext_cbs);
                }
            } else if(fd == STDIN_FILENO) {
                    if(events[i].events & EPOLLIN) {
                        mnld_client_stdin_hdlr(fd);
                        goto exit;
                    }
            } else {
                LOGE("unexpected fd=%d coming", fd);
                crash_with_log();
            }
        }

    }
exit:
    close(epfd);
    close(mnldinf_basic_fd);
    close(mnldinf_ext_fd);
    return 0;
}
