#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <stddef.h> // offsetof
#include <stdarg.h>
#include <sys/stat.h>
#include <unistd.h> //usleep
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <arpa/inet.h> //inet_addr
#include <sys/un.h> //struct sockaddr_un
#include <sys/epoll.h>
#include <poll.h>
#include <sys/types.h>
#include <netdb.h>
#include <openssl/ssl.h>
#include <pthread.h>
#include <semaphore.h>

#include "data_coder.h"
#include "agps_interface.h"
//#include "hal2mnl_interface.h"
//#include "mnl2hal_interface.h"
#include "mtk_lbs_utility.h"

#ifdef LOGD
#undef LOGD
#endif
#ifdef LOGE
#undef LOGE
#endif
#define LOGD(...)   { printf(__VA_ARGS__); fflush(stdout); }
#define LOGE(...)   { printf("\E[1;31;40m"); printf(__VA_ARGS__); printf("\E[0m"); fflush(stdout); }

//------------------------- Utils --------------------------------
// 2 file, 1 folder, 0 not exist
static int is_path_exist(const char* path) {
    struct stat s;
    int err = stat(path, &s);
    if(-1 == err) {
        if(ENOENT == errno) {
            // does not exist
            return 0;
        } else {
            // perror("stat");
            return 0;
        }
    } else {
        if(S_ISDIR(s.st_mode)) {
            // it's a dir
            return 1;
        } else {
            // it's a file
            return 2;
        }
    }
    return 0;
}

//-1 failure
int create_file(char* path) {
    FILE* fp = NULL;
    fp = fopen(path, "w");
    if(fp == NULL) {
        LOGE("create_file fopen failure reason=[%s]\n", strerror(errno));
        return -1;
    }
    fclose(fp);
    return 0;
}

//-1 failure
int delete_file(char* path) {
    return remove(path);
}

// 0 timed out, -1 failure, others there is a input message
static int poll_wait(int fd, int timeout) {
    struct pollfd poll_fd[1];
    int ret;

    memset(poll_fd, 0, sizeof(poll_fd));
    poll_fd[0].fd      = fd;
    poll_fd[0].events  = POLLIN;
    poll_fd[0].revents = 0;

    ret = poll(poll_fd, 1, timeout);
    if(ret > 0) {
        if(poll_fd[0].revents & POLLIN ||
            poll_fd[0].revents & POLLERR) {
            ret = 1 << 0;
        }
    }
    return ret;
}

static const char* get_ssl_info_where_string(int where) {
    switch(where) {
    case SSL_CB_LOOP:
        return "LOOP";
    case SSL_CB_EXIT:
        return "EXIT";
    case SSL_CB_READ:
        return "READ";
    case SSL_CB_WRITE:
        return "WRITE";
    case SSL_CB_ALERT:
        return "ALERT";
    case SSL_CB_READ_ALERT:
        return "READ_ALERT";
    case SSL_CB_WRITE_ALERT:
        return "WRITE_ALERT";
    case SSL_CB_ACCEPT_LOOP:
        return "ACCEPT_LOOP";
    case SSL_CB_ACCEPT_EXIT:
        return "ACCEPT_EXIT";
    case SSL_CB_CONNECT_LOOP:
        return "CONNECT_LOOP";
    case SSL_CB_CONNECT_EXIT:
        return "CONNECT_EXIT";
    case SSL_CB_HANDSHAKE_START:
        return "HANDSHAKE_START";
    case SSL_CB_HANDSHAKE_DONE:
        return "HANDSHAKE_DONE";
    default:
        return "UNKNOWN";
    }
}

static const char* get_ssl_error_string(int ret) {
    switch(ret) {
    case SSL_ERROR_NONE:
        return "WANT_NONE";
    case SSL_ERROR_ZERO_RETURN:
        return "WANT_ZERO_RETURN";
    case SSL_ERROR_WANT_READ:
        return "WANT_READ";
    case SSL_ERROR_WANT_WRITE:
        return "WANT_WRITE";
    case SSL_ERROR_WANT_CONNECT:
        return "WANT_CONNECT";
    case SSL_ERROR_WANT_ACCEPT:
        return "WANT_ACCEPT";
    case SSL_ERROR_WANT_X509_LOOKUP:
        return "WANT_X509_LOOKUP";
    case SSL_ERROR_SYSCALL:
        return "SYSCALL";
    case SSL_ERROR_SSL:
        return "SSL";
    default:
        return "UNKNOWN";
    }
}

static const char* get_ssl_error_string2(const SSL* ssl, int ret) {
    int err = SSL_get_error(ssl, ret);
    return get_ssl_error_string(err);
}

static int mtk_socket_connect_network_ipv4(struct sockaddr_in* addr) {
    int port = ntohs(addr->sin_port);
    char ip_str[64] = {0};
    inet_ntop(AF_INET, &(addr->sin_addr), ip_str, sizeof(ip_str));
    int fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (fd < 0) {
        LOGE("mtk_socket_connect_network_ipv4() socket() failed reason=[%s]%d\n",
            strerror(errno), errno);
        return -1;
    }
    //SOCK_LOGD("mtk_socket_connect_network_ipv4() IPv4=[%s] port=[%d]", ip_str, port);
    if (connect(fd, (struct sockaddr *)addr, sizeof(*addr)) < 0) {
        LOGE("mtk_socket_connect_network_ipv4() connect() failed reason=[%s]%d for host=[%s] port=[%d]\n",
            strerror(errno), errno, ip_str, port);
        close(fd);
        return -1;
    }
    return fd;
}

//-1 means failure
static int mtk_socket_connect_network_ipv6(struct sockaddr_in6* addr) {
    int port = ntohs(addr->sin6_port);
    int fd = socket(AF_INET6, SOCK_STREAM, IPPROTO_TCP);
    if (fd < 0) {
        LOGE("mtk_socket_connect_network_ipv6() socket() failed reason=[%s]%d\n",
            strerror(errno), errno);
        return -1;
    }
    //unsigned char* ipv6_addr = addr->sin6_addr.s6_addr;
    //SOCK_LOGD("mtk_socket_connect_network_ipv6() IPv6=[%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x] port=[%d]",
    //    ipv6_addr[0], ipv6_addr[1], ipv6_addr[2], ipv6_addr[3], ipv6_addr[4], ipv6_addr[5], ipv6_addr[6], ipv6_addr[7],
    //    ipv6_addr[8], ipv6_addr[9], ipv6_addr[10], ipv6_addr[11], ipv6_addr[12], ipv6_addr[13], ipv6_addr[14], ipv6_addr[15],
    //    port);
    if (connect(fd, (struct sockaddr *)addr, sizeof(*addr)) < 0) {
        LOGE("mtk_socket_connect_network_ipv6() connect() failed reason=[%s]%d for port=[%d]\n",
            strerror(errno), errno, port);
        close(fd);
        return -1;
    }
    return fd;
}

// return fd, -1 means failure
static int mtk_socket_connect_network(const char* host, int port) {
    int fd = 0;
    struct addrinfo hints;
    struct addrinfo *result;
    struct addrinfo *rp;
    int ret;
    char port_str[11] = {0};
    sprintf(port_str, "%u", port);
    memset(&hints, 0, sizeof(struct addrinfo));
    hints.ai_family = AF_UNSPEC;    // Allow IPv4 or IPv6, AF_UNSPEC, AF_INET, AF_INET6
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_ADDRCONFIG; // check local system
    hints.ai_protocol = IPPROTO_TCP;
    ret = getaddrinfo(host, port_str, &hints, &result);
    if(ret != 0) {
        LOGE("mtk_socket_connect_network() getaddrinfo() failure reason=[%s] %d\n",
            gai_strerror(ret), ret);
        return -1;
    }
    for (rp = result; rp != NULL; rp = rp->ai_next) {
        if(rp->ai_family == AF_INET) {
            ret = mtk_socket_connect_network_ipv4((struct sockaddr_in*)rp->ai_addr);
            if(ret >= 0) {
                fd = ret;
                break;
            }
        }
        if(rp->ai_family == AF_INET6) {
            ret = mtk_socket_connect_network_ipv6((struct sockaddr_in6*)rp->ai_addr);
            if(ret >= 0) {
                fd = ret;
                break;
            }
        }
    }
    freeaddrinfo(result);
    if(ret >= 0) {
        return fd;
    } else {
        return -1;
    }
}

//------------------------- Test Case Implementation --------------------------------
#define AGPS_TEST_CASE_VERSION  "1.00"
#define SUPL_RAW_ENABLED        "/data/agps_supl/supl_raw_enabled"
#define MNLD_PATH               "/usr/bin/mnld"   //for MT2735 change, mnld0
#define AGPSD_PATH              "/usr/bin/mtk_agpsd"
#define LBS_EM_PATH             "/usr/bin/lbs_em_tool"  //for MT2735 change, mtk_lbs_em_tool
#define SUPL_SERVER_FQDN        "supl.google.com"
#define SUPL_SERVER_PORT        7275
#define SUPL_SERVER_TLS         1
#define GPS_DELETE_ALL          0xFFFF
#define MNLD_POLL_TIMEOUT       5000
#define AGPS_PROCESS_TIMEOUT    (20 * 1000)
#define AGPS_TLS_TIMEOUT        (10 * 1000)

#define SHOW_RESULT(is_continue_on_fail, error_count) {\
    if(error_count > 0) {\
        LOGD("Result [");\
        LOGE("FAIL");\
        LOGD("]\n");\
        if(is_continue_on_fail) {\
            error_count = 0; \
        } else {\
            goto exit;\
        }\
    } else {\
        LOGD("Result [");\
        printf("\E[1;32;40m");\
        LOGD("SUCCESS");\
        printf("\E[0m");\
        LOGD("]\n");\
    }\
}
#if 0
// 1 is success, 0 need to wait the next message, -1 read process is failure
static int read_and_check_sv_status(int fd) {
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;
    int ver;
    mnl2hal_cmd cmd;
    int read_len;
    int ret = 0;

    read_len = safe_recvfrom(fd, buff, sizeof(buff));
    if (read_len <= 0) {
        LOGE("    mnl2hal_hdlr() safe_recvfrom() failed read_len=%d\n", read_len);
        return -1;
    }
    ver = get_int(buff, &offset);
    UNUSED(ver);
    cmd = get_int(buff, &offset);

    if(cmd == MNL2HAL_GPS_SV) {
        gnss_sv_info sv;
        get_binary(buff, &offset, (char*)&sv);
        LOGD("    read sv num=[%d]\n", sv.num_svs);
        if(sv.num_svs >= 3) {
            ret = 1;
        }
    }
    return ret;
}

// return error_count
static int test_case_gps_started(int is_continue_on_fail) {
    int ret = 0;
    int error_count = 0;

    //---------------------------------------------------------
    LOGD("5. Create MNL 2 HAL Interface\n");
    int mnl_fd = create_mnl2hal_fd();
    if(mnl_fd < 0) {
        LOGE("    MNL 2 HAL Interface is not available, please check this issue with GPS team\n");
        error_count++;
    }

    SHOW_RESULT(is_continue_on_fail, error_count);

    //---------------------------------------------------------
    LOGD("6. Monitor GPS and AGPS flow\n");
    long long start_time = get_tick();
    while(1) {
        ret = poll_wait(mnl_fd, MNLD_POLL_TIMEOUT);
        if(ret <= 0) {
            LOGE("    No message sent from MNLD after starting GPS, please check this issue with GPS team\n");
            error_count++;
            break;
        }
        ret = read_and_check_sv_status(mnl_fd);
        if(ret == 1) {
            break;
        } else if(ret < 0) {
            LOGE("    The message sent from MNLD is incorrect, please check this issue with GPS team\n");
            error_count++;
            break;
        }
        long long time_used = get_tick() - start_time;
        if(time_used >= AGPS_PROCESS_TIMEOUT) {
            LOGE("    Cannot detect AGPS status with %d seconds, please check this issue with GPS/AGPS team\n",
                AGPS_PROCESS_TIMEOUT / 1000);
            error_count++;
            break;
        }
    }

    SHOW_RESULT(is_continue_on_fail, error_count);
exit:
    if(mnl_fd >= 0) {
        close(mnl_fd);
    }
    return error_count;
}
#endif
static void ssl_info_callback(const SSL *ssl, int where, int ret) {
    if(ret == -1) {
        int err = SSL_get_error(ssl, ret);
        if(err != SSL_ERROR_WANT_READ && err != SSL_ERROR_WANT_WRITE) {
            LOGE("    ssl_info_callback() where=[%s] ssl_error=[%s] SSL_state=[%s]\n",
            get_ssl_info_where_string(where), get_ssl_error_string2(ssl, ret),
            SSL_state_string_long(ssl));
        }
    }
}

// 1 means success, 0 means failure
static int ssl_cert_verify(X509_STORE_CTX *x509_ctx, void *arg) {
    return 1;
}

static sem_t g_sem;
static SSL* g_ssl;
static int g_tls_result = 0;    // 1 success, 0 fail

static void* tls_event_thread(void *arg) {
    #define MAX_EPOLL_EVENT 50
    struct epoll_event events[MAX_EPOLL_EVENT];
    long long start_time = get_tick();
    int fd = *((int*)arg);
    int epfd = epoll_create(MAX_EPOLL_EVENT);
    if(epfd == -1) {
        LOGE("    tls_event_thread() epoll_create() fail, reason=[%s]%d\n",
            strerror(errno), errno);
        goto exit;
    }
    epoll_add_fd3(epfd, fd);
    g_tls_result = 0;

    while(1) {
        int i;
        int n;

        n = epoll_wait(epfd, events, MAX_EPOLL_EVENT , 1000);
        if(n == -1) {
            if(errno == EINTR) {
                continue;
            } else {
                LOGE("    tls_event_thread() epoll_wait() fail, reason=[%s]%d\n",
                    strerror(errno), errno);
                break;
            }
        }

        for(i = 0; i < n; i++) {
            if(events[i].data.fd == fd && events[i].events & EPOLLIN) {
                int ret = SSL_connect(g_ssl);
                if(ret == 1) {
                    g_tls_result = 1;
                    goto exit;
                } else if(ret == 0) {
                    goto exit;
                }
            }
        }

        long long time_used = get_tick() - start_time;
        if(time_used >= AGPS_TLS_TIMEOUT) {
            break;
        }

    }

exit:
    sem_post(&g_sem);
    return 0;
}

// return < 0 means failure
static int test_case_tls_handshaking(int fd) {
    socket_set_blocking2(fd, 0);
    SSL_library_init();

    const SSL_METHOD* method = SSLv23_client_method();
    SSL_CTX* ctx = SSL_CTX_new(method);
    long option = SSL_CTX_get_options(ctx);
    option = option & ~(SSL_OP_NO_SSLv2 | SSL_OP_NO_TLSv1_1 | SSL_OP_NO_TLSv1_2);
    option |= (SSL_OP_NO_SSLv2 | SSL_OP_NO_TLSv1_2);
    SSL_CTX_set_options(ctx, option);
    SSL_CTX_set_session_cache_mode(ctx, SSL_SESS_CACHE_CLIENT | SSL_SESS_CACHE_NO_INTERNAL);
    SSL_CTX_set_info_callback(ctx, ssl_info_callback);
    SSL_CTX_set_cert_verify_callback(ctx, ssl_cert_verify, NULL);

    g_ssl = SSL_new(ctx);
    BIO *sbio = BIO_new_socket(fd, BIO_NOCLOSE);
    SSL_set_bio(g_ssl, sbio, sbio);

    pthread_t pthread1;
    pthread_create(&pthread1, NULL, tls_event_thread, &fd);
    msleep2(500);

    int ret = SSL_connect(g_ssl);
    if(ret == 1) {
        //success
        ret = 0;
        goto exit;
    } else if(ret < 0) {
        int err = SSL_get_error(g_ssl, ret);
        if(err == SSL_ERROR_WANT_READ || err == SSL_ERROR_WANT_WRITE) {
            //create a thread for TLS handshaking
            if(sem_init(&g_sem, 0, 0) == -1) {
                LOGE("    sem_init() fail, reason=[%s], please check this issue with System team\n",
                    strerror(errno));
                ret = -1;
                goto exit;
            }
            sem_wait(&g_sem);
            if(sem_destroy(&g_sem) == -1) {
                LOGE("    sem_destroy() fail, reason=[%s], please check this issue with System team\n",
                    strerror(errno));
                ret = -2;
                goto exit;
            }
            if(g_tls_result == 1) {
                ret = 0;
            } else {
                ret = -3;
            }
            goto exit;
        } else {
            //error 1
            LOGE("    SSL_connect() failed reason=[%s]%d\n",
                get_ssl_error_string(err), err);
                ret = -4;
                goto exit;
        }
    } else {
        //error 2
        LOGE("    SSL_connect() fail, reason=[%s], please check whether SUPL server is available by Tester\n",
            get_ssl_error_string2(g_ssl, ret));
        ret = -5;
        goto exit;
    }
exit:
    SSL_free(g_ssl);
    SSL_CTX_free(ctx);
    return ret;
}

// return error_count
static int agps_test_case(int is_continue_on_fail, const char* host, int port, int tls) {
    int ret = 0;
    int error_count = 0;

    LOGD("agps_test_case version=[%s] is_continue_on_fail=[%d] host=[%s] port=[%d] tls=[%d]\n",
        AGPS_TEST_CASE_VERSION, is_continue_on_fail, host, port, tls);
    create_file(SUPL_RAW_ENABLED);
    //---------------------------------------------------------
    LOGD("1. Check existance of [GPS Service], [AGPS Service] and [LBS EM Tool]\n");
    ret = is_path_exist(MNLD_PATH);
    if(ret != 2) {
        LOGE("    [MNLD] does not exist under=[%s], please check this issue with GPS team\n",
            MNLD_PATH);
        error_count++;
    }

    ret = is_path_exist(AGPSD_PATH);
    if(ret != 2) {
        LOGE("    [AGPSD] does not exist under=[%s], please check this issue with AGPS team\n",
            AGPSD_PATH);
        error_count++;
    }

    ret = is_path_exist(LBS_EM_PATH);
    if(ret != 2) {
        LOGE("    [LBS EM Tool] does not exist under=[%s], please check this issue with AGPS team\n",
            LBS_EM_PATH);
        //error_count++;  // optional checking, no need to set error_count
    }

    SHOW_RESULT(is_continue_on_fail, error_count);

    //---------------------------------------------------------
    // change AGPS Settings
    set_agps_enabled(1);
    set_protocol(AGPS_INTF_AGPS_PROTOCOL_UP);
    set_cdma_pref(AGPS_INTF_CDMA_PREFERRED_WCDMA);
    set_up_pref_method(AGPS_INTF_PREF_METHOD_MSB);
    set_pos_technology_msb(1);
    set_supl_version(2);
    set_tls_version(AGPS_INTF_TLS_VERSION_1_1);
    set_supl_profile(host, port, tls);

    //---------------------------------------------------------
    LOGD("2. Check Network Connection State with Server FQDN=[%s] port=[%d]\n", host, port);

    ret = mtk_socket_connect_network(host, port);
    if(ret < 0) {
        LOGE("    Cannot connect to SUPL Server=[%s] with port=[%d], please check whether Network connection is available by Tester\n",
            host, port);
        error_count++;
    } else {
        if(tls) {
            LOGD("2.1 Start TLS Handshaking\n");
            if(test_case_tls_handshaking(ret) < 0) {
                LOGE("    TLS Handshaking fail, please check SUPL Server is available by Tester\n");
                error_count++;
            }
        }
    }
    close(ret);

    SHOW_RESULT(is_continue_on_fail, error_count);

#if 0 //mt2735 agps test not support below step.
    //---------------------------------------------------------
    LOGD("3. Do GPS Delete Aiding Data\n");
    ret = hal2mnl_gps_delete_aiding_data(GPS_DELETE_ALL);
    if(ret < 0) {
        LOGE("    hal2mnl_gps_delete_aiding_data() Fail, please check this issue with GPS team\n");
        error_count++;
    }

    SHOW_RESULT(is_continue_on_fail, error_count);

    //---------------------------------------------------------
    LOGD("4. Start GPS\n");
    ret = hal2mnl_gps_init();
    if(ret < 0) {
        LOGE("    hal2mnl_gps_init() Fail, please check this issue with GPS team\n");
        error_count++;
    }
    ret = hal2mnl_gps_start();
    if(ret < 0) {
        LOGE("    hal2mnl_gps_start() Fail, please check this issue with GPS team\n");
        error_count++;
    }

    SHOW_RESULT(is_continue_on_fail, error_count);

    //---------------------------------------------------------
    test_case_gps_started(is_continue_on_fail);

    //---------------------------------------------------------
    LOGD("7. Stop GPS\n");
    ret = hal2mnl_gps_stop();
    if(ret < 0) {
        LOGE("    hal2mnl_gps_stop() Fail, please check this issue with GPS team\n");
        error_count++;
    }

    SHOW_RESULT(is_continue_on_fail, error_count);

    //---------------------------------------------------------
    LOGD("8. Cleanup GPS\n");
    ret = hal2mnl_gps_cleanup();
    if(ret < 0) {
        LOGE("    hal2mnl_gps_cleanup() Fail, please check this issue with GPS team\n");
        error_count++;
    }

    SHOW_RESULT(is_continue_on_fail, error_count);
#endif

exit:
    // release all resouce here
    delete_file(SUPL_RAW_ENABLED);
    return error_count;
}

int main(int argc, char** argv) {
    char host[128] = SUPL_SERVER_FQDN;
    int port = SUPL_SERVER_PORT;
    int tls = SUPL_SERVER_TLS;
    int is_continue_on_fail = 0;

    if(argc >= 2) {
        is_continue_on_fail = atoi(argv[1]);
        if(is_continue_on_fail) {
            is_continue_on_fail = 1;
        }
    }

    if(argc >= 3) {
        strcpy(host, argv[2]);
    }
    if(argc >= 4) {
        int tmp = atoi(argv[3]);
        if(tmp != 0) {
            port = tmp;
        }
    }
    if(argc >= 5) {
        tls = atoi(argv[4]);
        if(tls) {
            tls = 1;
        }
    }
    return agps_test_case(is_continue_on_fail, host, port, tls);
}

