#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <stddef.h>  // offsetof
#include <stdarg.h>
#include <sys/stat.h>
#include <unistd.h>  // usleep
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <arpa/inet.h>  // inet_addr
#include <sys/un.h>  // struct sockaddr_un
#include <pthread.h>
#include <sys/epoll.h>
#include <signal.h>
#include <semaphore.h>

#if defined(__ANDROID_OS__)
#include <cutils/log.h>     // Android log
#endif

#include "mtk_lbs_utility.h"


// -1 means failure
/*static int get_time_str(char* buf, int len) {
    struct timeval  tv;
    struct timezone tz;
    struct tm      *tm;

    gettimeofday(&tv, &tz);
    tm = localtime(&tv.tv_sec);

    memset(buf, 0, len);
    sprintf(buf, "%04d/%02d/%02d %02d:%02d:%02d.%03d",
        tm->tm_year + 1900, 1 + tm->tm_mon, tm->tm_mday, tm->tm_hour, tm->tm_min,
        tm->tm_sec, (int)(tv.tv_usec / 1000));

    return 0;
}*/

/*************************************************
* Basic Utilities
**************************************************/
void tag_log(int type, const char* tag, const char *fmt, ...) {
    char out_buf[1100] = {0};
    char buf[1024] = {0};
    va_list ap;
#if defined(__ANDROID_OS__)
    int prio = 0;
#endif
    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    sprintf(out_buf, "%s %s", tag, buf);

#if defined(__ANDROID_OS__)
    if (type == 0) {
        prio = ANDROID_LOG_DEBUG;
    } else {
        prio = ANDROID_LOG_ERROR;
    }
        __android_log_print(prio, "lbs", "%s", out_buf);
#else
/* ignore time buffer
    char time_buf[64] = {0};
    UNUSED(type);
    UNUSED(prio);
    get_time_str(time_buf, sizeof(time_buf));

    printf("%s %s\n", time_buf, out_buf);
    */
    printf("%s\n", out_buf);
#endif
}

void msleep2(int interval) {
    usleep(interval * 1000);
}

// in millisecond
time_t get_tick() {
    struct timespec ts;
    if (clock_gettime(CLOCK_MONOTONIC, &ts) == -1) {
        LOGE("clock_gettime failed reason=[%s]", strerror(errno));
        return -1;
    }
    return (ts.tv_sec*1000) + (ts.tv_nsec/1000000);
}

// in millisecond
time_t get_time_in_millisecond() {
    struct timespec ts;
    if (clock_gettime(CLOCK_REALTIME, &ts) == -1) {
        LOGE("get_time_in_millisecond  failed reason=[%s]", strerror(errno));
        return -1;
    }
    return ((long long)ts.tv_sec*1000) + ((long long)ts.tv_nsec/1000000);
}

/*************************************************
* Epoll
**************************************************/
// -1 means failure
int epoll_add_fd3(int epfd, int fd) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = EPOLLIN;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_add_fd3() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

// -1 failed
int epoll_add_fd2(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_add_fd2() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

bool epoll_add_fd(int epfd, int fd, int events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    //don't set fd to edge trigger due to event lost when multiple events coming together
    //level trigger can avoid above issue
    //do not set EPOLLOUT due to it will always trigger when write is available
    if(epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_ctl() failed, epfd=%d fd=%d events=0x%x reason=[%s]%d",
            epfd, fd, events, strerror(errno), errno);
        return false;
    }
    return true;
}

int epoll_del_fd(int epfd, int fd) {
    struct epoll_event  ev;
    int                 ret;

    if (epfd == -1)
        return -1;

    ev.events  = EPOLLIN;
    ev.data.fd = fd;
    do {
        ret = epoll_ctl(epfd, EPOLL_CTL_DEL, fd, &ev);
    } while (ret < 0 && errno == EINTR);
    return ret;
}

// -1 failed
int epoll_mod_fd(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    if (epoll_ctl(epfd, EPOLL_CTL_MOD, fd, &ev) == -1) {
        LOGE("epoll_mod_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

/******************************************************************************
* Timer
******************************************************************************/

//NULL means fail or timerid is returned
timer_t timer_init(timer_routine cb, int id) {
    struct sigevent sevp;
    timer_t timerid;

    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_value.sival_int = id;
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = (void*)cb;

    if(timer_create(CLOCK_BOOTTIME, &sevp, &timerid) == -1) {
        LOGE("timer_init() timer_create() failed, reason=[%s]%d",
            strerror(errno), errno);
        return NULL;
    }
    return timerid;
}

bool timer_deinit(timer_t timerid) {
    if(timer_delete(timerid) == -1) {
        LOGE("timer_deinit() timer_delete() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool timer_start(timer_t timerid, int milliseconds) {
    struct itimerspec expire;
    expire.it_interval.tv_sec = 0;
    expire.it_interval.tv_nsec = 0;
    expire.it_value.tv_sec = milliseconds/1000;
    expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
    if(timer_settime(timerid, 0, &expire, NULL) == -1) {
        LOGE("timer_start() timer_settime() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool timer_stop(timer_t timerid) {
    return timer_start(timerid, 0);
}

//-1 means fail or the remaining time is returned
int64_t timer_get_remaining_time(timer_t timerid) {
    struct itimerspec ts;
    if(timer_gettime(timerid, &ts) == -1) {
        LOGE("timer_get_remaining_time() timer_gettime() failed, reason=[%s]%d", strerror(errno), errno);
        return -1;
    }
    return ((int64_t)ts.it_value.tv_sec * 1000 + ts.it_value.tv_nsec / 1000000);
}

/*************************************************
* Local UDP Socket
**************************************************/
// -1 means failure
int socket_bind_udp(const char* path) {
    int fd;
    struct sockaddr_un addr;

    fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("socket_bind_udp() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    memcpy(addr.sun_path + 1, path, strlen(path));
    addr.sun_family = AF_UNIX;
    unlink(addr.sun_path);
    if (bind(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        LOGE("socket_bind_udp() bind() failed path=[%s] reason=[%s]%d",
            path, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

// -1 means failure
int socket_set_blocking2(int fd, int blocking) {
    if (fd < 0) {
        LOGE("socket_set_blocking2() invalid fd=%d", fd);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (flags == -1) {
        LOGE("socket_set_blocking2() fcntl() failed invalid flags=%d reason=[%s]%d",
            flags, strerror(errno), errno);
        return -1;
    }

    flags = blocking ? (flags&~O_NONBLOCK) : (flags|O_NONBLOCK);
    return (fcntl(fd, F_SETFL, flags) == 0) ? 0 : -1;
}

// -1 means failure
int safe_sendto(const char* path, const char* buff, int len) {
    int ret = 0;
    struct sockaddr_un addr;
    int retry = 10;
    int fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("safe_sendto() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    memcpy(addr.sun_path + 1, path, strlen(path));
    addr.sun_family = AF_UNIX;

    while ((ret = sendto(fd, buff, len, 0,
        (const struct sockaddr *)&addr, sizeof(addr))) == -1) {
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("safe_sendto() sendto() failed path=[%s] ret=%d reason=[%s]%d",
            path, ret, strerror(errno), errno);
        break;
    }

    close(fd);
    return ret;
}

// -1 means failure
int safe_recvfrom(int fd, char* buff, int len) {
    int ret = 0;
    int retry = 10;

    while ((ret = recvfrom(fd, buff, len, 0,
         NULL, NULL)) == -1) {
        LOGW("safe_recvfrom() ret=%d len=%d", ret, len);
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("safe_recvfrom() recvfrom() failed reason=[%s]%d",
            strerror(errno), errno);
        break;
    }
    return ret;
}


//-1 means fail or clientfd is returned
int socket_tcp_client_connect_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        LBS_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("socket_tcp_client_connect_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_tcp_client_connect_local() connect() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

//-1 means fail or serverfd is returned
int socket_udp_server_bind_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        LBS_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("socket_udp_server_bind_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_server_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(bind(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_udp_server_bind_local() bind() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

//-1 means fail or clientfd is returned
int socket_udp_client_create_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        LBS_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_client_create_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_udp_client_create_local() connect() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

bool socket_udp_exist_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        LBS_STRNCPY(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_is_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return false;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        close(fd);
        return false;
    }
    close(fd);
    return true;

}

void dump_buff(const char* data, int len) {
    int i = 0;
    int offset = 0;
    char buff[1024] = {0};

    LOGD("len=%d", len);

    for(i = 0; i < len; i++) {

        if(i%16 == 0) {
            sprintf(buff + offset, "%04x0 ", i/16);
            offset += 6;
        }

        sprintf(buff + offset, "%02x ", data[i]&0xff);
        offset += 3;

        if(i%16 == 15) {
            LOGD("  %s ", buff);

            memset(buff, 0, sizeof(buff));
            offset = 0;
        }
    }
    if(offset > 0) {
        LOGD("  %s ", buff);
    }
}

