#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h> // for open
#include <unistd.h> // for close
#include <errno.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <arpa/inet.h> //inet_addr
#include <sys/un.h> //struct sockaddr_un
#if defined(__ANDROID_OS__)
#include <cutils/sockets.h>
#include <android/log.h>
#endif
#include <string.h>

#include "data_coder.h"
#include "agps_debug_interface.h"

#define LOGD(...)   { printf(__VA_ARGS__); fflush(stdout); }
#define LOGE(...)   { printf(__VA_ARGS__); fflush(stdout); }
#define AGPS_DEBUG_INTERFACE_PATH  "agpsd3"
#ifndef UNUSED
#define UNUSED(x) (x)=(x)
#endif

typedef enum {
    DEBUG_MGR_MESSAGE = 0,
    DEBUG_MGR_STATE,
} debug_mgr_type_enum;

#define DEBUG_MGR_MESSAGE_TOAST (1<<0)
#define DEBUG_MGR_MESSAGE_VIEW  (1<<1)
#define DEBUG_MGR_MESSAGE_DIALOG (1<<2)

// -1 means failure
static int local_socket_stream_connect(const char* path) {
    struct sockaddr_un addr;
    int fd = socket(PF_LOCAL, SOCK_STREAM, 0);
    if(fd < 0) {
        LOGE("local_socket_stream_connect()  socket() failed fd=%d\n", fd);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    memcpy(addr.sun_path + 1, path, strlen(path));
    addr.sun_family = AF_UNIX;

    if (connect(fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        LOGE("local_socket_stream_connect()  connect() failed reason=[%s] path=[%s]\n", strerror(errno), path);
        close(fd);
        return -1;
    }

    return fd;
}

//-1 means failure
static int safe_read(int fd, void* buf, int len) {
    int n, retry = 10;

    if(fd < 0 || buf == NULL || len < 0) {
        LOGE("safe_read()  fd=%d buf=%p len=%d\n", fd, buf, len);
        return -1;
    }

    if(len == 0) {
        return 0;
    }

    while((n = read(fd, buf, len)) < 0) {
        if(errno == EINTR) continue;
        if(errno == EAGAIN) {
            if(retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
            goto exit;
        }
        goto exit;
    }
    return n;

exit:
    if(errno != EAGAIN) {
        LOGE("safe_read()  reason=[%s] fd=%d len=%d buf=%p\n",
            strerror(errno), fd, len, buf);
    }
    return -1;
}

// -1 means failure
int agps_debug_interface_msg_handler(int fd, agps_debug_interface* handlers) {
    char tmp[1024] = {0};
    int ret;
    int offset;
    int len;

    debug_mgr_type_enum type = 0;
    int data1 = 0;
    int data2 = 0;
    int data3 = 0;
    char msg1[1024] = {0};
    char msg2[1024] = {0};
    char binary[1024] = {0};
    int binary_size = 0;

    UNUSED(type);
    UNUSED(data1);
    UNUSED(data2);
    UNUSED(data3);
    UNUSED(binary_size);

    // read type
    memset(tmp, 0, sizeof(tmp));
    ret = safe_read(fd, tmp, 4);
    if(ret < 4) {
        LOGE("agps_debug_interface_msg_handler()  read type failure, ret=%d\n", ret);
        return -1;
    }
    offset = 0;
    type = get_int(tmp, &offset);

    // read data1
    memset(tmp, 0, sizeof(tmp));
    ret = safe_read(fd, tmp, 4);
    if(ret < 4) {
        LOGE("agps_debug_interface_msg_handler()  read data1 failure, ret=%d\n", ret);
        return -1;
    }
    offset = 0;
    data1 = get_int(tmp, &offset);

    // read data2
    ret = safe_read(fd, tmp, 4);
    if(ret < 4) {
        LOGE("agps_debug_interface_msg_handler()  read data2 failure, ret=%d\n", ret);
        return -1;
    }
    offset = 0;
    data2 = get_int(tmp, &offset);

    // read data3
    ret = safe_read(fd, tmp, 4);
    if(ret < 4) {
        LOGE("agps_debug_interface_msg_handler()  read data3 failure, ret=%d\n", ret);
        return -1;
    }
    offset = 0;
    data3 = get_int(tmp, &offset);

    // read msg1
    ret = safe_read(fd, tmp, 1);
    if(ret < 1) {
        LOGE("agps_debug_interface_msg_handler()  read msg1 failure, ret=%d\n", ret);
        return -1;
    }
    if(tmp[0]) {
        // read msg1 length
        ret = safe_read(fd, tmp, 4);
        if(ret < 4) {
            LOGE("agps_debug_interface_msg_handler()  read msg1 failure in length, ret=%d\n", ret);
            return -1;
        }
        offset = 0;
        len = get_int(tmp, &offset);
        // read msg1 pdu
        ret = safe_read(fd, msg1, len);
        if(ret < len) {
            LOGE("agps_debug_interface_msg_handler()  read msg1 failure in pdu, ret=%d\n", ret);
            return -1;
        }
    }

    // read msg2
    ret = safe_read(fd, tmp, 1);
    if(ret < 1) {
        LOGE("agps_debug_interface_msg_handler()  read msg2 failure, ret=%d\n", ret);
        return -1;
    }
    if(tmp[0]) {
        // read msg2 length
        ret = safe_read(fd, tmp, 4);
        if(ret < 4) {
            LOGE("agps_debug_interface_msg_handler()  read msg2 failure in length, ret=%d\n", ret);
            return -1;
        }
        offset = 0;
        len = get_int(tmp, &offset);
        // read msg2 pdu
        ret = safe_read(fd, msg2, len);
        if(ret < len) {
            LOGE("agps_debug_interface_msg_handler()  read msg2 failure in pdu, ret=%d\n", ret);
            return -1;
        }
    }

    // read binary
    ret = safe_read(fd, tmp, 4);
    if(ret < 4) {
        LOGE("agps_debug_interface_msg_handler()  read binary failure in length, ret=%d\n", ret);
        return -1;
    }
    offset = 0;
    binary_size = get_int(tmp, &offset);
    if(binary_size > 0) {
        ret = safe_read(fd, binary, binary_size);
        if(ret < binary_size) {
            LOGE("agps_debug_interface_msg_handler()  read binary failure in pdu, ret=%d\n", ret);
            return -1;
        }
    }

    // handling
    if(type == DEBUG_MGR_MESSAGE) {
        if(data1 & DEBUG_MGR_MESSAGE_TOAST) {
            handlers->toast_message(msg2);
        }
        if(data1 & DEBUG_MGR_MESSAGE_VIEW) {
            handlers->view_message(data2, msg2);
        }
        if(data1 & DEBUG_MGR_MESSAGE_DIALOG) {
            handlers->dialog_message(msg1, msg2);
        }
    }

    return 0;
}

// -1 means failure
int agps_debug_interface_get_fd() {
#if defined(__ANDROID_OS__)
    int fd = local_socket_stream_connect(AGPS_DEBUG_INTERFACE_PATH);
#else
    int fd = local_socket_stream_connect(AGPS_DEBUG_INTERFACE_PATH);
#endif
    return fd;
}

