#include <ctype.h>  // isspace
#include <time.h>
#include <sys/time.h>
#include "mtk_gnss_at_command.h"
#include "mtk_gnss_at_struct.h"
#include "mtk_gnss_at_log.h"

static const char* at_lstrip(const char* at_cmd) {
    assert(at_cmd != NULL);

    char* start = (char*)at_cmd;
    while(*start && isspace(*start)) {
        start++;
    }
    return start;
}

bool mtk_at_parse_cmd(const char** at_cmd, const char* target_cmd) {
    assert(at_cmd != NULL);
    assert(target_cmd != NULL);

    int len = strlen(target_cmd);
    if(strncmp(*at_cmd, target_cmd, len) == 0) {
        char prefix_end = (*at_cmd + len)[0];
        if(prefix_end == ':' || prefix_end == '=' || prefix_end == '\r' || prefix_end == '\n' ||
            prefix_end == ' ' || prefix_end == '\t' || prefix_end == '\0') {
            *at_cmd = at_lstrip(*at_cmd + len + 1);
            return true;
        }
    }
    return false;
}

static const char* at_next_param(const char* at_cmd) {
    assert(at_cmd != NULL);

    char* start = (char*)at_cmd;
    while(*start && *start != ',') {
        start++;
    }
    if(*start) {
        start++;
    }
    return at_lstrip(start);
}

int mtk_at_parse_comma_num(const char* at_cmd) {
    int count = 0;
    while(*at_cmd) {
        if(*at_cmd == ',') {
            count++;
        }
        at_cmd++;
    }
    return count;
}

bool mtk_at_parse_string(const char** at_cmd, char* output, int output_size) {
    assert(at_cmd != NULL);
    assert(output != NULL);
    assert(output_size >= 0);

    char* start = (char*)(*at_cmd);
    char* end = NULL;

    if(*start == ',') {
        *at_cmd = at_next_param(*at_cmd);
        return false;
    }

    while(*start && *start != '"' && *start != ',') {
        start++;
    }

    if(*start == 0) {
        return false;
    } else if(*start == ',') {
        *at_cmd = at_next_param(start);
        return false;
    } else {
        start++;
        end = start;
        while(*end && *end != '"') {
            end++;
        }
        if(*end == 0) {
            *at_cmd = end;
            return false;
        } else {
            int len = end - start;
            if((len + 1) >= output_size) {
                LOGE("mtk_at_parse_string() len=%d > output_size=%d", len, output_size);
                return false;
            } else {
                strncpy(output, start, len);
                output[len] = 0;
                *at_cmd = at_next_param(end);
                return true;
            }
        }
    }
    LOGE("mtk_at_parse_string() unexpected flow");
    return false;
}

bool mtk_at_parse_bool(const char** at_cmd, bool* output) {
    int value = 0;
    int ret = mtk_at_parse_int32(at_cmd, &value);
    if(ret) {
        *output = value? true:false;
    }
    return ret;
}

bool mtk_at_parse_int32(const char** at_cmd, int* output) {
    assert(at_cmd != NULL);
    assert(output != NULL);

    int ret = sscanf(*at_cmd, "%d", output);
    if(ret > 0) {
        *at_cmd = at_next_param((*at_cmd) + 1);
        return true;
    }
    if(**at_cmd) {
        *at_cmd = at_next_param((*at_cmd));
    }
    return false;
}

bool mtk_at_parse_int64(const char** at_cmd, long long* output) {
    assert(at_cmd != NULL);
    assert(output != NULL);

    int ret = sscanf(*at_cmd, "%lld", output);
    if(ret > 0) {
        *at_cmd = at_next_param((*at_cmd) + 1);
        return true;
    }
    if(**at_cmd) {
        *at_cmd = at_next_param((*at_cmd));
    }
    return false;
}

bool mtk_at_parse_float(const char** at_cmd, float* output) {
    assert(at_cmd != NULL);
    assert(output != NULL);

    int ret = sscanf(*at_cmd, "%f", output);
    if(ret > 0) {
        *at_cmd = at_next_param((*at_cmd) + 1);
        return true;
    }
    if(**at_cmd) {
        *at_cmd = at_next_param((*at_cmd));
    }
    return false;
}

bool mtk_at_parse_double(const char** at_cmd, double* output) {
    assert(at_cmd != NULL);
    assert(output != NULL);

    int ret = sscanf(*at_cmd, "%lf", output);
    if(ret > 0) {
        *at_cmd = at_next_param((*at_cmd) + 1);
        return true;
    }
    if(**at_cmd) {
        *at_cmd = at_next_param((*at_cmd));
    }
    return false;
}

void mtk_at_dump_mtk_gnss_satellite_list(mtk_gnss_satellite_list* list) {
    assert(list != NULL);
    LOGD("mtk_at_dump_mtk_gnss_satellite_list() num=%d", list->num);
    int i = 0;
    for(i = 0; i < list->num; i++) {
        LOGD("  i=[%d] svid=[%02d] constellation=[%d] cn0=[%.02f] elevation=[%.02f] azimuth=[%.02f] flags=[0x%x] carrier_frequency=[%.02f]",
            i, list->satellites[i].svid, list->satellites[i].constellation, list->satellites[i].cn0,
            list->satellites[i].elevation, list->satellites[i].azimuth, list->satellites[i].flags,
            list->satellites[i].carrier_frequency);
    }
}

void mtk_at_dump_mtk_gnss_location(mtk_gnss_location* l) {
    assert(l != NULL);
    LOGD("mtk_at_dump_mtk_gnss_location() lat=[%lf]%d lng=[%lf]%d alt=[%.2f]%d speed=[%.2f]%d bearing=[%.2f]%d h_acc=[%.2f]%d v_acc=[%.2f]%d s_acc=[%.2f]%d "\
        "b_acc=[%.2f]%d timestamp=[%lld]%d date_time=[%s]%d pdop=[%.2f]%d hdop=[%.2f]%d vdop=[%.2f]%d",
        l->lat, l->lat_valid, l->lng, l->lng_valid, l->alt, l->alt_valid, l->speed, l->speed_valid, l->bearing, l->bearing_valid,
        l->h_acc, l->h_acc_valid, l->v_acc, l->v_acc_valid, l->s_acc, l->s_acc_valid, l->b_acc, l->b_acc_valid, l->timestamp, l->timestamp_valid,
        l->date_time, l->date_time_valid, l->pdop, l->pdop_valid, l->hdop, l->hdop_valid, l->vdop, l->vdop_valid);
}

void mtk_at_dump_mtk_agnss_location(mtk_agnss_location* l) {
    assert(l != NULL);
    LOGD("mtk_at_dump_mtk_agnss_location() lat=[%lf] lng=[%lf] alt[%.2f]%d speed=[%.2f]%d bearing=[%.2f]%d h_acc=[%.2f]%d"\
        " timestamp=[%lld]%d date_time=[%s]%d type=[%d]%d",
        l->lat, l->lng, l->alt, l->alt_valid, l->speed, l->speed_valid, l->bearing, l->bearing_valid,
        l->h_acc, l->h_acc_valid, l->timestamp, l->timestamp_valid, l->date_time, l->date_time_valid, l->type, l->type_valid);
}


void mtk_at_dump_mtk_gnss_status(mtk_gnss_status* status) {
    assert(status != NULL);
    LOGD("mtk_at_dump_mtk_gnss_status() gnss_status=[%d] ni_status=[%d] supl_maj_ver=[%d] supl_min_ver=[%d] gnss_version=[%s] agps_version=[%s]"\
        " supl_ser_ind=[%d] supl_addr=[%s] supl_port=[%d] supl_tls_enabled=[%d] imsi_valid=[%d] imsi=[%s] num_digital_mnc_in_imsi_valid=[%d]"\
        " num_digital_mnc_in_imsi=[%d]",
        status->gnss_status, status->ni_status, status->supl_maj_ver, status->supl_min_ver, status->gnss_version, status->agps_version,
        status->supl_ser_ind, status->supl_addr, status->supl_port, status->supl_tls_enabled, status->imsi_valid, status->imsi,
        status->num_digital_mnc_in_imsi_valid, status->num_digital_mnc_in_imsi);
}

void mtk_at_dump_mtk_gnss_agps_mode(mtk_gnss_agps_mode* mode) {
    assert(mode != NULL);
    LOGD("mtk_at_dump_mtk_gnss_agps_mode() msa=[%d]%d msb=[%d]%d mss=[%d]%d cid=[%d]%d aflt=[%d]%d otdoa=[%d]%d supl_pref_method=[%d]%d(0=MSA, 1=MSB, 2=NO Pref)"\
        " supl=[%d]%d epo=[%d]%d",
        mode->msa, mode->msa_valid, mode->msb, mode->msb_valid, mode->mss, mode->mss_valid, mode->cid, mode->cid_valid, mode->aflt, mode->aflt_valid,
        mode->otdoa, mode->otdoa_valid, mode->supl_pref_method, mode->supl_pref_method_valid, mode->supl, mode->supl_valid, mode->epo, mode->epo_valid);
}

void mtk_at_dump_mtk_gnss_aiding_data(mtk_gnss_aiding_data* data) {
    assert(data != NULL);
    LOGD("mtk_at_dump_mtk_gnss_aiding_data() all=[%d] eph=[%d] alm=[%d] pos=[%d] time=[%d] iono=[%d] utc=[%d] svdir=[%d]  rti=[%d] celldb=[%d]",
        data->all, data->eph, data->alm, data->pos, data->time, data->iono, data->utc, data->svdir, data->rti, data->celldb);
}

void mtk_at_dump_mtk_gnss_ni_notify(mtk_gnss_ni_notify* notify) {
    assert(notify != NULL);
    LOGD("mtk_at_dump_mtk_gnss_ni_notify() id=[%d] type=[%d] notify_type=[%d] requestor_id=[%s] encode=[%d] text=[%s] encode=[%d]",
        notify->id, notify->type, notify->notify_type, notify->request_id, notify->request_id_encode_type, notify->text, notify->text_encode_type);
}

void mtk_at_dump_mtk_gnss_cert(mtk_gnss_cert* cert) {
    assert(cert != NULL);
    LOGD("mtk_at_dump_mtk_gnss_cert() total_msg_len=[%d] seq_no=[%d] name=[%s] data=[%s]",
        cert->total_msg_len, cert->seq_no, cert->name, cert->data);
}

void mtk_at_dump_mtk_gnss_cert_name_list(mtk_gnss_cert_name_list* list) {
    assert(list != NULL);
    int i = 0;
    LOGD("mtk_at_dump_mtk_gnss_cert_name_list() num=[%d]", list->num);
    for(i = 0; i < list->num; i++) {
        LOGD("  i=[%d] name=[%s]", i, list->names[i]);
    }
}

void mtk_at_dump_mtk_geo_add_circle(mtk_geo_add_circle* circle) {
    assert(circle != NULL);
    LOGD("mtk_at_dump_mtk_geo_add_circle() alert_type=[%d] (1=entry 2=exit) initial_state=[%d] (0=unknown 1=entered 2=exited) lat=[%.6f] lng=[%.6f] radius=[%.2f] unknowntimerms=[%d]",
        circle->alert_type, circle->initial_state, circle->lat, circle->lng, circle->radius, circle->unknowntimerms);
}

void mtk_at_dump_mtk_geo_alert_notify(mtk_geo_alert_notify* alert) {
    assert(alert != NULL);
    LOGD("mtk_at_dump_mtk_geo_alert_notify() id=[%d] state=[%d] (0=unknown 1=entered 2=exited) lat=[%.6f] lng=[%.6f] alt=[%.2f] speed=[%.2f] "
        "heading=[%.2f] hacc=[%d] h_err_maj=[%d] h_err_min=[%d] h_err_angle=[%d] h_confidence=[%d] pdop=[%.2f] hdop=[%.2f] vdop=[%.2f]",
        alert->id, alert->state, alert->lat, alert->lng, alert->alt, alert->speed, alert->heading, alert->hacc,
        alert->h_err_maj, alert->h_err_min, alert->h_err_angle, alert->h_confidence, alert->pdop, alert->hdop, alert->vdop);
}

void mtk_at_gen_gnss_start_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=\r\n", MTK_AT_GNSS_START_REQ);
}

void mtk_at_gen_gnss_stop_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=\r\n", MTK_AT_GNSS_STOP_REQ);
}

void mtk_at_gen_gnss_nmea_notify(char* at_cmd, int at_cmd_size, const char* nmea) {
    assert(nmea != NULL);
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s: \"%s\"\r\n", MTK_AT_GNSS_NMEA_NTF, nmea);
}

void mtk_at_gen_gnss_nmea_end_notify(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_NMEA_END_NTF);
}

void mtk_at_gen_gnss_satellite_notify(char* at_cmd, int at_cmd_size, mtk_gnss_satellite_list* list) {
    assert(list != NULL);
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    int i = 0;

    at_cmd += sprintf(at_cmd, "%s: %d", MTK_AT_GNSS_SATELLITE_NTF, list->num);
    for(i = 0; i < list->num; i++) {
        at_cmd += sprintf(at_cmd, ",%d,%d,%.2f,%.2f,%.2f,%d,%.2f",
            list->satellites[i].svid, list->satellites[i].constellation, list->satellites[i].cn0,
            list->satellites[i].elevation, list->satellites[i].azimuth, list->satellites[i].flags,
            list->satellites[i].carrier_frequency);
    }
    at_cmd += sprintf(at_cmd, "\r\n");
}
void mtk_at_gen_agnss_location_notify(char* at_cmd, int at_cmd_size, mtk_agnss_location* location) {
    assert(location != NULL);
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    at_cmd += sprintf(at_cmd, "%s: ", MTK_AT_GNSS_AGPS_LOCATION_NTF);
    at_cmd += sprintf(at_cmd, "%.6f,", location->lat);
    at_cmd += sprintf(at_cmd, "%.6f,", location->lng);

    if(location->alt_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->alt);
    }
    at_cmd += sprintf(at_cmd, ",");
    
    if(location->speed_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->speed);
    }
    at_cmd += sprintf(at_cmd, ",");
    
    if(location->bearing_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->bearing);
    }
    at_cmd += sprintf(at_cmd, ",");
    
    if(location->h_acc_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->h_acc);
    }
    at_cmd += sprintf(at_cmd, ",");
    
    if(location->timestamp_valid) {
        at_cmd += sprintf(at_cmd, "%lld", location->timestamp);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->timestamp_valid) {
        struct timeval tv;
        struct tm      *tm;
        tv.tv_sec = location->timestamp / 1000;
        tm = gmtime(&tv.tv_sec);

        location->date_time_valid = true;
        sprintf(location->date_time, "%04d/%02d/%02d,%02d:%02d:%02d",
            tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
        at_cmd += sprintf(at_cmd, "\"%s\"", location->date_time);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->type_valid) {
        at_cmd += sprintf(at_cmd, "%d", location->type);
    }
    at_cmd += sprintf(at_cmd, "\r\n");
}

void mtk_at_gen_gnss_location_notify(char* at_cmd, int at_cmd_size, mtk_gnss_location* location) {
    assert(location != NULL);
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    at_cmd += sprintf(at_cmd, "%s: ", MTK_AT_GNSS_LOCATION_NTF);

    if(location->lat_valid) {
        at_cmd += sprintf(at_cmd, "%.6f", location->lat);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->lng_valid) {
        at_cmd += sprintf(at_cmd, "%.6f", location->lng);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->alt_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->alt);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->speed_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->speed);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->bearing_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->bearing);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->h_acc_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->h_acc);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->v_acc_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->v_acc);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->s_acc_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->s_acc);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->b_acc_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->b_acc);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->timestamp_valid) {
        at_cmd += sprintf(at_cmd, "%lld", location->timestamp);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->timestamp_valid) {
        struct timeval tv;
        struct tm      *tm;
        tv.tv_sec = location->timestamp / 1000;
        tm = gmtime(&tv.tv_sec);

        location->date_time_valid = true;
        sprintf(location->date_time, "%04d/%02d/%02d,%02d:%02d:%02d",
            tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
        at_cmd += sprintf(at_cmd, "\"%s\"", location->date_time);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->pdop_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->pdop);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->hdop_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->hdop);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(location->vdop_valid) {
        at_cmd += sprintf(at_cmd, "%.2f", location->vdop);
    }
    at_cmd += sprintf(at_cmd, "\r\n");
}

void mtk_at_gen_gnss_host_reset_notify(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_HOST_RESET_NTF);
}

void mtk_at_gen_gnss_device_reset_notify(char* at_cmd, int at_cmd_size, const char* reason) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s: \"%s\"\r\n", MTK_AT_GNSS_GNSS_RESET_NTF, reason);
}

void mtk_at_gen_gnss_status_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_STATUS_REQ);
}

void mtk_at_gen_gnss_status_response(char* at_cmd, int at_cmd_size, mtk_gnss_status* status) {
    assert(status != NULL);
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    at_cmd += sprintf(at_cmd, "%s: %d,%d,%d,%d,\"%s\",\"%s\",%d,\"%s\",%d,%d", MTK_AT_GNSS_STATUS_RSP,
        status->gnss_status, status->ni_status, status->supl_maj_ver, status->supl_min_ver,
        status->gnss_version, status->agps_version, status->supl_ser_ind, status->supl_addr,
        status->supl_port, status->supl_tls_enabled);

    at_cmd += sprintf(at_cmd, ",");
    if(status->imsi_valid) {
        at_cmd += sprintf(at_cmd, "\"%s\"", status->imsi);
    }

    at_cmd += sprintf(at_cmd, ",");
    if(status->num_digital_mnc_in_imsi_valid) {
        at_cmd += sprintf(at_cmd, "%d", status->num_digital_mnc_in_imsi);
    }
    
    at_cmd += sprintf(at_cmd, "\r\n");
}

void mtk_at_gen_gnss_enable_set(char* at_cmd, int at_cmd_size, bool enabled) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d\r\n", MTK_AT_GNSS_GNSS_ENABLE_SET, enabled);
}

void mtk_at_gen_gnss_ni_enable_set(char* at_cmd, int at_cmd_size, bool enabled) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d\r\n", MTK_AT_GNSS_NI_ENABLE_SET, enabled);
}

void mtk_at_gen_gnss_agps_mode_set(char* at_cmd, int at_cmd_size, mtk_gnss_agps_mode* mode) {
    assert(mode != NULL);
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    at_cmd += sprintf(at_cmd, "%s=", MTK_AT_GNSS_AGPS_MODE_SET);
    if(mode->msa_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->msa);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->msb_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->msb);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->mss_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->mss);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->cid_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->cid);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->aflt_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->aflt);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->otdoa_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->otdoa);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->supl_pref_method_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->supl_pref_method);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->supl_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->supl);
    }
    at_cmd += sprintf(at_cmd, ",");

    if(mode->epo_valid) {
        at_cmd += sprintf(at_cmd, "%d", mode->epo);
    }

    at_cmd += sprintf(at_cmd, "\r\n");
}

void mtk_at_gen_gnss_supl_version_set(char* at_cmd, int at_cmd_size, int maj, int min, int ser_ind) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d,%d,%d\r\n", MTK_AT_GNSS_SUPL_VERSION_SET, maj, min, ser_ind);
}

void mtk_at_gen_gnss_supl_addr_set(char* at_cmd, int at_cmd_size, const char* addr, int port, bool tls_enabled) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(addr != NULL);

    sprintf(at_cmd, "%s=\"%s\",%d,%d\r\n", MTK_AT_GNSS_SUPL_ADDRESS_SET, addr, port, tls_enabled);
}

void mtk_at_gen_gnss_delete_aiding_data_request(char* at_cmd, int at_cmd_size, mtk_gnss_aiding_data* flags) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(flags != NULL);

    if(flags->all) {
        sprintf(at_cmd, "%s=\r\n", MTK_AT_GNSS_DELETE_AIDING_REQ);
    } else {
        sprintf(at_cmd, "%s=%d,%d,%d,%d,%d,%d,%d,%d,%d\r\n", MTK_AT_GNSS_DELETE_AIDING_REQ,
            flags->eph, flags->alm, flags->pos, flags->time, flags->iono, flags->utc, flags->svdir,
            flags->rti, flags->celldb);
    }
}

void mtk_at_gen_gnss_ni_notify_request(char* at_cmd, int at_cmd_size, mtk_gnss_ni_notify* notify) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(notify != NULL);

    sprintf(at_cmd, "%s: %d,%d,%d,\"%s\",%d,\"%s\",%d\r\n", MTK_AT_GNSS_NI_NOTIFY_REQ,
        notify->id, notify->type, notify->notify_type, notify->request_id, notify->request_id_encode_type,
        notify->text, notify->text_encode_type);
}

void mtk_at_gen_gnss_ni_notify_response(char* at_cmd, int at_cmd_size, int id, mtk_gnss_ni_response_type response) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d,%d\r\n", MTK_AT_GNSS_NI_NOTIFY_RSP, id, response);
}

void mtk_at_gen_gnss_ref_location_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_REF_LOC_REQ);
}

void mtk_at_gen_gnss_ref_location_response(char* at_cmd, int at_cmd_size, int age, double lat, double lng, float acc) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d,%.6f,%.6f,%.2f\r\n", MTK_AT_GNSS_REF_LOC_RSP, age, lat, lng, acc);
}

void mtk_at_gen_gnss_ref_time_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_REF_TIME_REQ);
}

void mtk_at_gen_gnss_ref_time_response(char* at_cmd, int at_cmd_size, long long time, int uncertainty) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%lld,%d\r\n", MTK_AT_GNSS_REF_TIME_RSP, time, uncertainty);
}

void mtk_at_gen_gnss_nmea_config_set(char* at_cmd, int at_cmd_size, bool enabled) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d\r\n", MTK_AT_GNSS_NMEA_CFG_SET, enabled);
}

void mtk_at_gen_gnss_ttff_qop_set(char* at_cmd, int at_cmd_size, int mode) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d\r\n", MTK_AT_GNSS_TTFF_QOP_SET, mode);
}

void mtk_at_gen_gnss_loopback_request(char* at_cmd, int at_cmd_size, const char* msg) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(msg != NULL);

    sprintf(at_cmd, "%s=\"%s\"\r\n", MTK_AT_GNSS_LOOPBACK_REQ, msg);
}

void mtk_at_gen_gnss_loopback_response(char* at_cmd, int at_cmd_size, const char* msg) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(msg != NULL);

    sprintf(at_cmd, "%s=\"%s\"\r\n", MTK_AT_GNSS_LOOPBACK_RSP, msg);
}

void mtk_at_gen_gnss_cert_set(char* at_cmd, int at_cmd_size, mtk_gnss_cert* cert) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(cert != NULL);

    sprintf(at_cmd, "%s=%d,%d,\"%s\",\"%s\"\r\n", MTK_AT_GNSS_CERT_SET, cert->total_msg_len, cert->seq_no, cert->name, cert->data);
}

void mtk_at_gen_gnss_cert_name_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_CERT_NAME_REQ);
}

void mtk_at_gen_gnss_cert_name_response(char* at_cmd, int at_cmd_size, mtk_gnss_cert_name_list* list) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(list != NULL);

    int i = 0;
    at_cmd += sprintf(at_cmd, "%s: ", MTK_AT_GNSS_CERT_NAME_RSP);
    for(i = 0; i < list->num; i++) {
        at_cmd += sprintf(at_cmd, "\"%s\",", list->names[i]);
    }
    at_cmd--;
    at_cmd += sprintf(at_cmd, "\r\n");
}

void mtk_at_gen_gnss_cert_delete_request(char* at_cmd, int at_cmd_size, const char* name) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(name != NULL);

    sprintf(at_cmd, "%s=\"%s\"\r\n", MTK_AT_GNSS_CERT_DEL_REQ, name);
}

void mtk_at_gen_gnss_cert_delete_all_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GNSS_CERT_DEL_ALL_REQ);
}

void mtk_at_gen_geo_max_num_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s\r\n", MTK_AT_GEOFENCE_MAX_NUM_REQ);
}

void mtk_at_gen_geo_max_num_response(char* at_cmd, int at_cmd_size, int num) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s: %d\r\n", MTK_AT_GEOFENCE_MAX_NUM_RSP, num);
}

void mtk_at_gen_geo_add_circle_request(char* at_cmd, int at_cmd_size, mtk_geo_add_circle* circle) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(circle != NULL);

    sprintf(at_cmd, "%s=%d,%d,%.6f,%.6f,%.6f,%d\r\n", MTK_AT_GEOFENCE_ADD_CIRCLE_REQ,
        circle->alert_type, circle->initial_state, circle->lat, circle->lng, circle->radius, circle->unknowntimerms);
}

void mtk_at_gen_geo_add_circle_response(char* at_cmd, int at_cmd_size, mtk_geo_fence_create_state state, int id) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    if(state == MTK_GEO_FENCE_CREATE_STATE_SUCCESS) {
        sprintf(at_cmd, "%s: %d,%d\r\n", MTK_AT_GEOFENCE_ADD_CIRCLE_RSP, state, id);
    } else {
        sprintf(at_cmd, "%s: %d\r\n", MTK_AT_GEOFENCE_ADD_CIRCLE_RSP, state);
    }
}

void mtk_at_gen_geo_delete_request(char* at_cmd, int at_cmd_size, int id) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=%d\r\n", MTK_AT_GEOFENCE_DEL_REQ, id);
}

void mtk_at_gen_geo_delete_all_request(char* at_cmd, int at_cmd_size) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);

    sprintf(at_cmd, "%s=\r\n", MTK_AT_GEOFENCE_DEL_ALL_REQ);
}

void mtk_at_gen_geo_alert_notify(char* at_cmd, int at_cmd_size, mtk_geo_alert_notify* notify) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(notify != NULL);

    sprintf(at_cmd, "%s: %d,%d,%.6f,%.6f,%.2f,%.2f,%.2f,%d,%d,%d,%d,%d,%.2f,%.2f,%.2f\r\n", MTK_AT_GEOFENCE_ALERT_NTF,
        notify->id, notify->state, notify->lat, notify->lng, notify->alt, notify->speed, notify->heading,
        notify->hacc, notify->h_err_maj, notify->h_err_min, notify->h_err_angle, notify->h_confidence, notify->pdop, notify->hdop, notify->vdop);
}

void mtk_at_gen_geo_track_notify(char* at_cmd, int at_cmd_size, mtk_geo_track_state state, const char* date_time) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(date_time != NULL);

    sprintf(at_cmd, "%s: %d,\"%s\"\r\n", MTK_AT_GEOFENCE_TRACK_NTF, state, date_time);
}

void mtk_at_gen_test_request(char* at_cmd, int at_cmd_size, int num1, int num2, double d1, double d2, const char* str1, const char* str2) {
    assert(at_cmd != NULL);
    assert(at_cmd_size > 0);
    assert(str1 != NULL);
    assert(str2 != NULL);

    sprintf(at_cmd, "%s=%d,%d,%.6f,%.6f,\"%s\",\"%s\"\r\n", MTK_AT_TEST_REQ, num1, num2, d1, d2, str1, str2);
}

