#include <string.h> //strstr
#include <dirent.h>
#include <sys/stat.h> //mkdir
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <stddef.h>  // offsetof
#include <stdarg.h>
#include <unistd.h>  // usleep
#include <sys/socket.h>
#include <fcntl.h>
#include <arpa/inet.h>  // inet_addr
#include <sys/un.h>  // struct sockaddr_un
#include <pthread.h>
#include <sys/epoll.h>
#include <signal.h>
#include <semaphore.h>
#include <signal.h> //struct sigevent
#include <string.h> //memset, strncpy
#include <netdb.h> //struct addrinfo
#include <sys/types.h> //gettid
#include <netinet/in.h> //struct sockaddr_in
#include <netinet/tcp.h> //TCP_KEEPIDLE, TCP_KEEPINTVL, TCP_KEEPCNT
#include <netinet/ip.h> //IPTOS_LOWDELAY
#include <sys/epoll.h> //epoll_create
#include <semaphore.h> //sem_t
#include <inttypes.h> //PRId64
#include <stdbool.h>
#include "geofence.h"

#define GEOFENCEADP_VERSION "2.02"

#define GEOFENCE_USER_NAME  "mtk_geofence_adaptor"
//---------------------------------- channel -------------------------------------------
// LBS EM (client) <-> GNSS ADP (server)
    
#define LOGD(...)   { printf(__VA_ARGS__); printf("\n"); fflush(stdout); }
#define LOGW(...)   { printf("\E[1;35;40m"); printf(__VA_ARGS__); printf("\E[0m"); printf("\n"); fflush(stdout); }
#define LOGE(...)   { printf("\E[1;31;40m"); printf(__VA_ARGS__); printf("\E[0m"); printf("\n"); fflush(stdout); }
#define LOGI(...)   { printf("\E[1;35;40m"); printf(__VA_ARGS__); printf("\E[0m"); printf("\n"); fflush(stdout); }

//--------------------------------------------------------------------------------------------------------
#define MNLD_STRNCPY(dst,src,size) do{\
                                       strncpy((char *)(dst), (src), (size - 1));\
                                      (dst)[size - 1] = '\0';\
                                     }while(0)

timer_t g_handler_timer;

int tcp_client_fd;
int epfd;
char std_in_buff[1024];

typedef void (*timer_routine) (int id);

void msleep(int interval) {
    usleep(interval * 1000);
}

// in millisecond
time_t get_tick() {
    struct timespec ts;
    if (clock_gettime(CLOCK_MONOTONIC, &ts) == -1) {
        LOGE("clock_gettime failed reason=[%s]", strerror(errno));
        return -1;
    }
    return (ts.tv_sec*1000) + (ts.tv_nsec/1000000);
}

// in millisecond
time_t get_time_in_millisecond() {
    struct timespec ts;
    if (clock_gettime(CLOCK_REALTIME, &ts) == -1) {
        LOGE("get_time_in_millisecond  failed reason=[%s]", strerror(errno));
        return -1;
    }
    return ((long long)ts.tv_sec*1000) + ((long long)ts.tv_nsec/1000000);
}

/*************************************************
* Timer
**************************************************/
// -1 means failure
//NULL means fail or timerid is returned
timer_t timer_init(timer_routine cb, int id) {
    struct sigevent sevp;
    timer_t timerid;

    memset(&sevp, 0, sizeof(sevp));
    sevp.sigev_value.sival_int = id;
    sevp.sigev_notify = SIGEV_THREAD;
    sevp.sigev_notify_function = (void*)cb;

    if(timer_create(CLOCK_BOOTTIME, &sevp, &timerid) == -1) {
        LOGE("timer_init() timer_create() failed, reason=[%s]%d",
            strerror(errno), errno);
        return NULL;
    }
    return timerid;
}

bool timer_deinit(timer_t timerid) {
    if(timer_delete(timerid) == -1) {
        LOGE("timer_deinit() timer_delete() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool timer_start(timer_t timerid, int milliseconds) {
    struct itimerspec expire;
    expire.it_interval.tv_sec = 0;
    expire.it_interval.tv_nsec = 0;
    expire.it_value.tv_sec = milliseconds/1000;
    expire.it_value.tv_nsec = (milliseconds%1000)*1000000;
    if(timer_settime(timerid, 0, &expire, NULL) == -1) {
        LOGE("timer_start() timer_settime() failed, reason=[%s]%d", strerror(errno), errno);
        return false;
    }
    return true;
}

bool timer_stop(timer_t timerid) {
    return timer_start(timerid, 0);
}

//-1 means fail or the remaining time is returned
int64_t timer_get_remaining_time(timer_t timerid) {
    struct itimerspec ts;
    if(timer_gettime(timerid, &ts) == -1) {
        LOGE("timer_get_remaining_time() timer_gettime() failed, reason=[%s]%d", strerror(errno), errno);
        return -1;
    }
    return ((int64_t)ts.it_value.tv_sec * 1000 + ts.it_value.tv_nsec / 1000000);
}

/*************************************************
* Epoll
**************************************************/
// -1 means failure
int epoll_add_fd(int epfd, int fd) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = EPOLLIN;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_add_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

// -1 failed
int epoll_add_fd2(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    // don't set the fd to edge trigger
    // the some event like accept may be lost if two or more clients are connecting to server at the same time
    // level trigger is preferred to avoid event lost
    // do not set EPOLLOUT due to it will always trigger when write is available
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev) == -1) {
        LOGE("epoll_add_fd2() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

int epoll_del_fd(int epfd, int fd) {
    struct epoll_event  ev;
    int                 ret;

    if (epfd == -1)
        return -1;

    ev.events  = EPOLLIN;
    ev.data.fd = fd;
    do {
        ret = epoll_ctl(epfd, EPOLL_CTL_DEL, fd, &ev);
    } while (ret < 0 && errno == EINTR);
    return ret;
}

// -1 failed
int epoll_mod_fd(int epfd, int fd, uint32_t events) {
    struct epoll_event ev;
    memset(&ev, 0, sizeof(ev));
    ev.data.fd = fd;
    ev.events = events;
    if (epoll_ctl(epfd, EPOLL_CTL_MOD, fd, &ev) == -1) {
        LOGE("epoll_mod_fd() epoll_ctl() failed reason=[%s]%d epfd=%d fd=%d",
            strerror(errno), errno, epfd, fd);
        return -1;
    }
    return 0;
}

/*************************************************
* Local UDP Socket
**************************************************/
// -1 means failure
int socket_bind_udp(const char* path) {
    int fd;
    struct sockaddr_un addr;

    fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("socket_bind_udp() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    LOGD("fd=%d,path=%s\n", fd, path);

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    MNLD_STRNCPY(addr.sun_path + 1, path,sizeof(addr.sun_path) - 1);
    addr.sun_family = AF_UNIX;
    unlink(path);

    if (bind(fd, (const struct sockaddr *)&addr, sizeof(addr)) < 0) {
        LOGE("socket_bind_udp() bind() failed path=[%s] reason=[%s]%d",
            addr.sun_path+1, strerror(errno), errno);
        close(fd);
        return -1;
    }else
        LOGI("bind ok path=[%s]", addr.sun_path+1);
    return fd;
}

// -1 means failure
int socket_set_blocking(int fd, int blocking) {
    if (fd < 0) {
        LOGE("socket_set_blocking() invalid fd=%d", fd);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (flags == -1) {
        LOGE("socket_set_blocking() fcntl() failed invalid flags=%d reason=[%s]%d",
            flags, strerror(errno), errno);
        return -1;
    }

    flags = blocking ? (flags&~O_NONBLOCK) : (flags|O_NONBLOCK);
    return (fcntl(fd, F_SETFL, flags) == 0) ? 0 : -1;
}

// -1 means failure
int safe_sendto(const char* path, const char* buff, int len) {
    int ret = 0;
    struct sockaddr_un addr;
    int retry = 10;
    int fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
    if (fd < 0) {
        LOGE("safe_sendto() socket() failed reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }

    int flags = fcntl(fd, F_GETFL, 0);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1){
        LOGE("fcntl failed reason=[%s]%d",
                    strerror(errno), errno);

        close(fd);
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_path[0] = 0;
    MNLD_STRNCPY(addr.sun_path + 1, path,sizeof(addr.sun_path) - 1);
    addr.sun_family = AF_UNIX;

    while ((ret = sendto(fd, buff, len, 0,
        (const struct sockaddr *)&addr, sizeof(addr))) == -1) {
        if (errno == EINTR) {
            LOGE("errno==EINTR\n");
            continue;
        }
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                LOGE("errno==EAGAIN\n");
                continue;
            }
        }
        LOGE("safe_sendto() sendto() failed path=[%s] ret=%d reason=[%s]%d",
            path, ret, strerror(errno), errno);
        break;
    }

    close(fd);
    return ret;
}

// -1 means failure
int safe_recvfrom(int fd, char* buff, int len) {
    int ret = 0;
    int retry = 10;
    int flags = fcntl(fd, F_GETFL, 0);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1){
        LOGE("fcntl failed reason=[%s]%d",
                    strerror(errno), errno);

        close(fd);
        return -1;
    }

    while ((ret = recvfrom(fd, buff, len, 0,
         NULL, NULL)) == -1) {
        LOGW("safe_recvfrom() ret=%d len=%d", ret, len);
        if (errno == EINTR) continue;
        if (errno == EAGAIN) {
            if (retry-- > 0) {
                usleep(100 * 1000);
                continue;
            }
        }
        LOGE("safe_recvfrom() recvfrom() failed reason=[%s]%d",
            strerror(errno), errno);
        break;
    }
    return ret;
}

/******************************************************************************
* Socket
******************************************************************************/

//-1 means fail or serverfd is returned
int socket_tcp_server_bind_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        strncpy(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("socket_tcp_server_bind_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("socket_tcp_server_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(bind(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_tcp_server_bind_local() bind() failed, abstruct=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    if(listen(fd, 5) == -1) {
        LOGW("socket_tcp_server_bind_local() listen() failed, reason=[%s]%d",
            strerror(errno), errno);
    }
    return fd;
}

//-1 means fail or new clientfd is returned
int socket_tcp_server_new_connect(int serverfd) {
    int newfd;
    struct sockaddr_in addr;
    socklen_t len = sizeof(addr);

    newfd = accept(serverfd, (struct sockaddr*)&addr, &len);
    if(newfd == -1) {
        LOGE("socket_tcp_server_new_connect() accept() failed, serverfd=%d reason=[%s]%d",
            serverfd, strerror(errno), errno);
        return -1;
    }
    return newfd;
}

//-1 means fail or serverfd is returned
int socket_tcp_client_connect_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        strncpy(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("socket_tcp_client_connect_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(fd == -1) {
        LOGE("socket_tcp_client_connect_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_tcp_client_connect_local() connect() failed, abstruct=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }

    return fd;
}


//-1 means fail or serverfd is returned
int socket_udp_server_bind_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        strncpy(addr.sun_path, name, sizeof(addr.sun_path));
        if(unlink(addr.sun_path) == -1) {
            LOGW("socket_udp_server_bind_local() unlink() failed, reason=[%s]%d",
                strerror(errno), errno);
        }
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_server_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(bind(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_udp_server_bind_local() bind() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

//-1 means fail or clientfd is returned
int socket_udp_client_create_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        strncpy(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_client_create_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return -1;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        LOGE("socket_udp_client_create_local() connect() failed, abstract=%d name=[%s] reason=[%s]%d",
            abstract, name, strerror(errno), errno);
        close(fd);
        return -1;
    }
    return fd;
}

bool socket_udp_exist_local(bool abstract, const char* name) {
    int fd;
    int size;
    struct sockaddr_un addr;

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    size = strlen(name) + offsetof(struct sockaddr_un, sun_path) + 1;
    if(abstract) {
        addr.sun_path[0] = 0;
        memcpy(addr.sun_path + 1, name, strlen(name));
    } else {
        strncpy(addr.sun_path, name, sizeof(addr.sun_path));
    }
    fd = socket(AF_UNIX, SOCK_DGRAM, 0);
    if(fd == -1) {
        LOGE("socket_udp_is_bind_local() socket() failed, reason=[%s]%d",
            strerror(errno), errno);
        return false;
    }
    if(connect(fd, (struct sockaddr*)&addr, size) == -1) {
        close(fd);
        return false;
    }
    close(fd);
    return true;

}

void geofenceadp_timer_handler_protection(int id) {
    LOGE("geofenceadp_timer_handler_protection() timeout");
}

void geofenceadp_add_geofence(void)
{
    mtk_geofence_property fence;
    int fence_id;
    memset(&fence, 0, sizeof(fence));
    fence.latitude = 24.123456;
    fence.longitude = 121.012345;
    fence.radius = 200.0;
    fence.latest_state = GEOFENCE_UNKNOWN;
    fence.monitor_transition = MTK_GEOFENCE_ENTER;
    fence.unknown_timer = 100000;
    fence_id = geofenceinf_add_geofence(&fence);
    LOGD("geofenceadp_add_geofence success id=%d", fence_id);
}

void geofenceadp_delete_geofence(int id)
{
    if (geofenceinf_remove_geofence(id) < 0){
        LOGE("geofenceadp_delete_geofence fail:%d", id);
    } else {
        LOGD("geofenceadp_delete_geofence success:%d", id);
    }
}

void geofenceadp_clear_geofence(void)
{
    if (geofenceinf_clear_geofences() < 0){
        LOGE("geofenceadp_clear_geofence fail");
    } else {
        LOGD("geofenceadp_clear_geofence success");
    }
}

void geofenceadp_query_geofence_num(void)
{
    int fence_num;
    fence_num = geofenceinf_query_geofences_num();
    LOGD("geofenceadp_query_geofence_num success number=%d", fence_num);
}

void geofenceadp_std_handler_process(char *buffer) {
    char buff[1024];

    MNLD_STRNCPY(buff, buffer, sizeof(buff));
    if (strncmp(buff, "addfence", strlen("addfence")) == 0){
        geofenceadp_add_geofence();
    } else if (strncmp(buff, "clearfence", strlen("clearfence")) == 0){
        geofenceadp_clear_geofence();
    } else if (strncmp(buff, "queryfence", strlen("queryfence")) == 0){
        geofenceadp_query_geofence_num();
    } else if (strncmp(buff, "delfence", strlen("delfence")) == 0){
        int id;
        id = atoi(&buff[9]);
        geofenceadp_delete_geofence(id);
    } 
}

void geofenceadp_connection_broken()
{
    LOGD("geofenceadp_connection_broken");
}

void geofenceadp_fence_alert(mtk_geofence_alert *alert)
{
    LOGD("geofenceadp_fence_alert");
}

void geofenceadp_tracking_status(mtk_gnss_tracking_status *status)
{
    LOGD("geofenceadp_tracking_status");
}

void geofenceadp_capability_update(mtk_geofence_server_capability *cap)
{
    LOGD("geofenceadp_capability_update server capability update");
}

static mtk_geofence_callback geofence_callbacks = {
    geofenceadp_connection_broken,
    geofenceadp_fence_alert,
    geofenceadp_tracking_status,
    geofenceadp_capability_update
};


//--------------------------------------------------------------------------------------------------------
    
void geofenceadp_epoll_client_disconnect_server(int fd) {
    LOGD("geofenceadp_epoll_client_disconnect_server fd=%d", fd);
    epoll_del_fd(epfd, fd);
}

int geofenceadp_epoll_stdin_hdlr(int fd) {
    char buff[2048] = {0};
    int ret = read(fd, buff, sizeof(buff));
    if(ret == -1) {
        LOGE("STDIN read() failed, fd=%d reason=[%s]%d", fd, strerror(errno), errno);
        return -1;
    }
    LOGD("stdin has data [%s]", buff);
    geofenceadp_std_handler_process(buff);

    return 0;
}

void geofencedp_mock_start() {
    LOGD("gnssadp_mock_start() ver=%s", GEOFENCEADP_VERSION);

    //memset(&client_ctx, 0, sizeof(geofence_client_ctx));

    tcp_client_fd = geofenceinf_client_register(GEOFENCE_USER_NAME, &geofence_callbacks);
    if(tcp_client_fd == -1) {
        LOGE("socket_tcp_client_connect_local(true, GEOFENCEADP_MNL_SOCKET) failed");
        exit(1);
    } else {
        mtk_geofence_client_capability client_cap;
        client_cap.geofence_support = true;
        geofenceinf_client_capability_config(&client_cap);
    }
    g_handler_timer = timer_init(geofenceadp_timer_handler_protection, 0);

    #define MAX_EPOLL_EVENT 10
    struct epoll_event events[MAX_EPOLL_EVENT];
    epfd = epoll_create(MAX_EPOLL_EVENT);
    if(epfd == -1) {
        LOGE("epoll_create() failed");
        exit(1);
    }

    epoll_add_fd2(epfd, STDIN_FILENO, EPOLLIN);
    epoll_add_fd2(epfd, tcp_client_fd, EPOLLIN);

    while(true) {
        int i, n;
        n = epoll_wait(epfd, events, MAX_EPOLL_EVENT, -1);
        if(n == -1) {
            if(errno == EINTR) {
                continue;
            } else {
                LOGE("epoll_wait() failed reason=[%s]%d", strerror(errno), errno);
                exit(1);
            }
        }

        for(i = 0; i < n; i++) {
            int fd = events[i].data.fd;
            timer_start(g_handler_timer, 30 * 1000);
            if(fd == STDIN_FILENO) {
                if(events[i].events & EPOLLIN) {
                    if (geofenceadp_epoll_stdin_hdlr(fd) == -1){
                        goto exit;
                    }
                }
            } else if(fd == tcp_client_fd) {
                if(events[i].events & EPOLLIN) {
                    mnl2geofence_hdlr(fd, &geofence_callbacks);
                }
            } else {
                LOGE("unexpected fd=%d coming", fd);
            }
            timer_stop(g_handler_timer);
        }
    }

exit:
    close(epfd);
    close(tcp_client_fd);
    timer_deinit(g_handler_timer);
}

int main() {
    geofencedp_mock_start();
    return 0;
}

