#include <stdio.h>   /* Standard input/output definitions */
#include <stdbool.h>   /* Standard bool true/false definitions */
#include <string.h>  /* String function definitions */
#include <unistd.h>  /* UNIX standard function definitions */
#include <fcntl.h>   /* File control definitions */
#include <errno.h>   /* Error number definitions */
#include <termios.h> /* POSIX terminal control definitions */
#include <time.h>
#include <pthread.h>
#include <stdlib.h>
#include <signal.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/epoll.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <strings.h>
#include <semaphore.h>

#include "gnss_atci_log.h"
#include "gnss_atci_utility.h"
#include "gnss_atci_socket_utils.h"
#include "gnss_atci_socket_data_coder.h"

#define GNSS_ATCI_UDP_ADDR "/data/server"

#define PRINT_TITLE(...)   { printf("\E[0;30;43m"); printf(__VA_ARGS__); printf("\E[0m"); fflush(stdout); }
#define PRINT_RAW(...)   { printf(__VA_ARGS__); fflush(stdout);}

#define GNSS_ATCI_INTERFACE_BUFF_SIZE (80*1024)
#define GNSS_ATCI_PATH_SIZE (256)

typedef enum {
    GNSS_ATCI_OPEN_LOG = 0,
    GNSS_ATCI_CLOSE_LOG = 1,
    GNSS_ATCI_WRITE_LOG = 2,
} gnss_atci_msg_id;

typedef struct {
    char*           sock_addr;
}log_db;

log_db gnss_atci_db;

typedef struct {
    bool     (*create_logfile)(const char* path);
    bool     (*write_logdata)(char* data, int len);
    bool     (*close_logfile)();
} gnss_atci_inf;

static bool gnss_atci_handle_open_logfile(const char* file_name) {
    time_t tm;
    struct tm *p = NULL;
    char full_file_name[256] = {0};

    if (time(&tm)==((time_t)-1)) {
        LOGE("time()  fail(%s)!!\r\n", strerror(errno));
    }
    p = localtime(&tm);

    if(p == NULL) {
        LOGE("Get localtime fail:[%s]%d", strerror(errno), errno);
        return false;
    }

    snprintf(full_file_name, sizeof(full_file_name), "%s_%04d_%02d%02d_%02d%02d%02d", file_name,
    1900 + p->tm_year, 1 + p->tm_mon, p->tm_mday,
    p->tm_hour, p->tm_min, p->tm_sec);

    PRINT_RAW("\r\n==============GPSLog start, %s=============\r\n", full_file_name);
    return true;
}

static bool gnss_atci_handle_write_logdata(char* data, int len) {
    PRINT_RAW("%s", data);
    return true;
}

static bool gnss_atci_handle_close_logfile() {
    PRINT_RAW("\r\n==============GPSLog end=============\r\n");
    return true;
}

static gnss_atci_inf gnss_atci_handler_interface = {
    gnss_atci_handle_open_logfile,
    gnss_atci_handle_write_logdata,
    gnss_atci_handle_close_logfile
};

char gnss_atci_buff[GNSS_ATCI_INTERFACE_BUFF_SIZE] = {0};

static bool gnss_atci_hdlr(int fd, gnss_atci_inf* hdlr) {
    int offset = 0;

    gnss_atci_msg_id cmd;
    int ptype;
    int log_type;
    int read_len = 0;
    bool ret = true;

    if (NULL == hdlr) {
       LOGE("gnss_atci_hdlr, hdlr not valid");
       return false;
    }

    if (fd < 0) {
       LOGE("gnss_atci_hdlr, fd not valid");
       return false;
    }
    memset(gnss_atci_buff, 0, sizeof(gnss_atci_buff));
    read_len = gnss_atci_safe_recvfrom(fd, gnss_atci_buff, sizeof(gnss_atci_buff)-1);
    if (read_len <= 0) {
        LOGE("gnss_atci_hdlr() gnss_atci_safe_recv() failed read_len=%d", read_len);
        return false;
    }

    hdlr->write_logdata(gnss_atci_buff, strlen(gnss_atci_buff));

    return ret;
}

#define GNSS_AT_CMD "AT%GNSS=fm"

int main(int argc, char** argv) {
    #define EPOLL_MAX_NUM 4
    int sock_fd = -1;
    int i = 0;
    int n = 0;
    int epfd = epoll_create(EPOLL_MAX_NUM);
    struct epoll_event events[EPOLL_MAX_NUM];

    sock_fd = gnss_atci_socket_create(GNSS_ATCI_UDP_ADDR, SOCK_NS_ABSTRACT);

    if (sock_fd < 0) {
        LOGE("gnss_atci_write_thread create sock_fd failed, reason=[%s]", strerror(errno));
        return 0;
    }

    if(argc >= 2) {
        gnss_atci_send(sock_fd, argv[1], strlen(argv[1]));
        //safe_sendto(GNSS_ATCI_UDP_ADDR, argv[1], strlen(argv[1]));
    } else {
        gnss_atci_send(sock_fd, GNSS_AT_CMD, strlen(GNSS_AT_CMD));
       // safe_sendto(GNSS_ATCI_UDP_ADDR, GNSS_AT_CMD, strlen(GNSS_AT_CMD));
    }

    if (gnss_atci_epoll_add_fd(epfd, sock_fd) == -1) {
        LOGE("gnss_atci_write_thread failed for fd_hal failed, reason=[%s]", strerror(errno));
        return 0;
    }

    while (1) {
        n = epoll_wait(epfd, events, EPOLL_MAX_NUM , -1);
        if (-1 == n) {
            if (errno == EINTR) {
                continue;
            } else {
                LOGE("gnss_atci_write_thread epoll_wait failure reason=[%s]", strerror(errno));
                return 0;
            }
        }
        for (i = 0; i < n; i++) {
            if (events[i].data.fd == sock_fd) {
                if (events[i].events & EPOLLIN) {
                    if (!gnss_atci_hdlr(sock_fd, &gnss_atci_handler_interface)) {
                        LOGE("gnss_atci_hdlr failure reason=[%s]", strerror(errno));
                    }
                }
            } else {
                LOGE("mnld_main_thread() unknown sock_fd=%d", events[i].data.fd);
            }
        }
    }
    LOGE("gnss_atci_write_thread exit");
    return 0;
}
