#ifndef __MNLDINF_UTILITY_H__
#define __MNLDINF_UTILITY_H__

#include <time.h>
#include <stdint.h>
#include <pthread.h>

#include "mnldinf_log.h"
#include "mnldinf_common.h"

#ifdef __cplusplus
extern "C" {
#endif

/*************************************************
* Basic Utilities
**************************************************/
#define  MTK_GPS_NVRAM  0

#define INVALID_TIMERID ((timer_t)-1)

#define DUMP_BYTES_PER_LINE 16
#define DUMP_MAX_PRINT_LINE 10
#define SOCKET_RETRY_MAX_COUNT 20

#define MNLDINF_DUMP_MEM(addr, len) do{\
    int i = 0, j = 0;\
    char print_buf[DUMP_BYTES_PER_LINE*5+1] = {0};\
    unsigned int print_len = len;\
    if(len > DUMP_MAX_PRINT_LINE*DUMP_BYTES_PER_LINE) {\
        print_len = DUMP_MAX_PRINT_LINE*DUMP_BYTES_PER_LINE;\
    }\
    LOGD(">>>>>DUMP_START(addr:%p, len:%d, print_len:%ud)<<<<<", addr, len,print_len);\
    for(i = 0; i < print_len; i+=DUMP_BYTES_PER_LINE) {\
        memset(print_buf, 0, sizeof(print_buf));\
        for(j=0;j<DUMP_BYTES_PER_LINE;j++) {\
            sprintf(&print_buf[j*5], "0x%02x ", ((char *)addr)[i*DUMP_BYTES_PER_LINE+j]);\
            print_buf[DUMP_BYTES_PER_LINE*5] = '\0';\
        }\
        LOGD("[%3d]:%s", i/DUMP_BYTES_PER_LINE, print_buf);\
    }\
    LOGD("<<<<<DUMP_STOP>>>>>");\
}while(0)

void mnldinf_msleep(int interval);

#define MNLD_STRNCPY(dst,src,size) do{\
                                       strncpy((char *)(dst), (char *)(src), (size - 1));\
                                      (dst)[size - 1] = '\0';\
                                     }while(0)

#define MNLD_SPRINTF(buf,fmt,...)  do{\
                                       if(sprintf((char *)(buf), fmt,##__VA_ARGS__) < 0){\
                                           LOGE("sprintf error occurred");\
                                       }\
                                     }while(0)

#define MNLD_SNPRINTF(buf,len,fmt,...) do{\
                                       if(snprintf((char *)(buf), len, fmt,##__VA_ARGS__) < 0){\
                                           LOGE("snprintf error occurred");\
                                       }\
                                     }while(0)

#define MNLD_VSNPRINTF(buf,len,fmt,...) do{\
                                       if(vsnprintf((char *)(buf), len, fmt,##__VA_ARGS__) < 0){\
                                           LOGE("vsnprintf error occurred");\
                                       }\
                                     }while(0)

#define MNLD_FRINTF(buf,fmt,...) do{\
                                       if(fprintf(buf, fmt,##__VA_ARGS__) < 0){\
                                           LOGE("fprintf error occurred");\
                                       }\
                                     }while(0)

#define MNLDINF_MSG_HEAD_LEN (3*sizeof(int))

//Cycle buffer
typedef struct cyclical_buffer     // Cyclical buffer
{
    char *next_write;     // next position to write to
    char *next_read;      // next position to read from
    char *start_buffer;   // start of buffer
    char *end_buffer;     // end of buffer + 1
    int buffer_size;
} cyclical_buffer_t;

//Cmd record and dump
#define GPS_HAL_TIME_STR_LEN 25
#define GPS_HAL_CMD_RECORD_NUM 10
#define GPS_HAL_CMD_MONITER_TIMEOUT (3*1000)

typedef struct {
    char enter_time[GPS_HAL_TIME_STR_LEN];
    char exit_time[GPS_HAL_TIME_STR_LEN];
    time_t exec_time;
    mnl2hal_cmd cmd;
} gps_cmd_record;

/*************************************************
* Timer
**************************************************/
typedef void (* timer_callback)();


// in millisecond
time_t mnldinf_get_tick();

// in millisecond
time_t mnldinf_get_time_in_millisecond();

/*************************************************
* Epoll
**************************************************/
// -1 means failure
int mnldinf_epoll_add_fd(int epfd, int fd);

// -1 failed
int mnldinf_epoll_add_fd2(int epfd, int fd, uint32_t events);

// -1 failed
int mnldinf_epoll_del_fd(int epfd, int fd);

int mnldinf_epoll_mod_fd(int epfd, int fd, uint32_t events);

/*************************************************
* Local UDP Socket
**************************************************/
// -1 means failure
int mnldinf_socket_bind_udp(const char* path);

// -1 means failure
int mnldinf_socket_set_blocking(int fd, int blocking);

// -1 means failure
int mnldinf_safe_sendto(const char* path, const char* buff, int len);

// -1 means failure
int mnldinf_safe_recvfrom(int fd, char* buff, int len);

// -1 means failure
int mnldinf_start_timer(timer_t timerid, int milliseconds);

// -1 means failure
int mnldinf_stop_timer(timer_t timerid);

// -1 means failure
timer_t mnldinf_init_timer(timer_callback cb);

// -1 means failure
int mnldinf_deinit_timer(timer_t timerid);

/*************************************************
* Cycle buffer
**************************************************/
void mnldinf_buffer_initialize(cyclical_buffer_t *buffer, char *buffer_body, unsigned int buffer_size);
int mnldinf_put_msg_to_cycle(cyclical_buffer_t *cyc_buffer, char *buff, int len);
int mnldinf_get_one_msg(cyclical_buffer_t *cyc_buffer, char *buff);


/*************************************************
* Socket
**************************************************/
int mnldinf_tcp_send(int fd, const char* buff, int len);
int mnldinf_socket_tcp_server_bind_local(bool abstract, const char* name);
int mnldinf_socket_tcp_server_new_connect(int serverfd);
int mnldinf_socket_tcp_client_connect_local(bool abstract, const char* name);
int mnldinf_socket_udp_server_bind_local(bool abstract, const char* name);
int mnldinf_socket_udp_client_create_local(bool abstract, const char* name);
bool mnldinf_socket_udp_exist_local(bool abstract, const char* name);
int mnldinf_safe_recv(int fd, char* buff, int len);

//------------------------------------------------------
//Linux wake lock

#define MNLDINF_WAKE_LOCK_ID "mnldinf_wakelock"
//delay to do wake_unlock to ensure the msg can be deliveried to other process
#define MNLDINF_WAKE_LOCK_TIMEOUT (5 * 1000)
//Timer refresh latency, to protect timer update too often in short time
#define MNLDINF_WAKE_LOCK_LATENCY (1 * 1000)

typedef struct {
    bool wake_lock_acquired;
    timer_t unlock_timer;
    pthread_mutex_t mutex;
    time_t time_last_refresh;
} wake_lock_ctx;

void mnldinf_wake_lock_init();
void mnldinf_wake_lock_deinit();
void mnldinf_wake_lock_take();
void mnldinf_wake_lock_give();

#ifdef __cplusplus
}
#endif

#endif
