#include "mnldinf_basic.h"
#include "mnldinf_utility.h"
#include "mnldinf_data_coder.h"
#include "mnldinf_log.h"
#include "errno.h"

//int log_dbg_level = mnldinf_LOG_LEVEL;

#ifdef LOG_TAG
#undef LOG_TAG
#define LOG_TAG "mnldinf_basic"
#endif
#ifndef MIN
#define MIN(A,B) ((A)<(B)?(A):(B))
#endif

char g_mnldinf_basic_msg_buf[HAL_MNL_BUFF_SIZE_SND*2] = {0};
cyclical_buffer_t g_mnldinf_basic_cbuffer;    // io - cyclic buffer
mnldinf_basic_client_cap g_mnldinf_basic_client_cap;
volatile bool g_mnldinf_basic_cap_setted = false;
gnss_data meas_data;

#define DUMP_BASIC_CLIENT_CAP(cap) do {\
    LOGD("[%s]:mnldinf_basic_client_cap size:%lu, gnss_support=%d", __func__, sizeof(mnldinf_basic_client_cap), (cap)->support_gnss);\
} while(0)

#define DUMP_BASIC_SERVER_CAP(cap) do {\
    LOGD("[%s]:mnldinf_basic_server_cap size:%lu, gnss_support=%d", __func__, sizeof(mnldinf_basic_server_cap), (cap)->support_gnss);\
} while(0)

static gps_cmd_record basic_cmd_list[GPS_HAL_CMD_RECORD_NUM];
static int basic_cmd_rcd_idx = 0;
static time_t basic_enter_time_ms = 0;
static timer_t basic_cmd_moniter_timer = INVALID_TIMERID;
static int basic_last_record_idx = 0xFF;

void mnldinf_basic_cmd_enter_record(mnl2hal_cmd cmd) {
    int ret = 0;
    /*Clear parameters before setting*/
    memset(basic_cmd_list[basic_cmd_rcd_idx].enter_time, 0, GPS_HAL_TIME_STR_LEN);
    memset(basic_cmd_list[basic_cmd_rcd_idx].exit_time, 0, GPS_HAL_TIME_STR_LEN);
    basic_cmd_list[basic_cmd_rcd_idx].cmd = 0;
    basic_cmd_list[basic_cmd_rcd_idx].exec_time = 0;

    basic_last_record_idx = 0xFF;  //Print all history command
    basic_enter_time_ms = mnldinf_get_time_in_millisecond();
    ret = mnldinf_get_time_str(basic_cmd_list[basic_cmd_rcd_idx].enter_time, GPS_HAL_TIME_STR_LEN);
    if(ret == -1) {
        LOGW("mnldinf_get_time_str fail(%s)", strerror(errno));
    }
    basic_cmd_list[basic_cmd_rcd_idx].cmd = cmd;

    ret = mnldinf_start_timer(basic_cmd_moniter_timer, GPS_HAL_CMD_MONITER_TIMEOUT);
    if(ret == -1) {
        LOGW("start_timer fail(%s)", strerror(errno));
    }
}

void mnldinf_basic_cmd_exit_record(mnl2hal_cmd cmd) {
    if(cmd == basic_cmd_list[basic_cmd_rcd_idx].cmd) {
        int ret = 0;
        ret = mnldinf_get_time_str(basic_cmd_list[basic_cmd_rcd_idx].exit_time, GPS_HAL_TIME_STR_LEN);
        if(ret == -1) {
            LOGW("mnldinf_get_time_str fail(%s)", strerror(errno));
        }
        basic_cmd_list[basic_cmd_rcd_idx].exec_time = mnldinf_get_time_in_millisecond() - basic_enter_time_ms;
        basic_cmd_rcd_idx++;
        basic_cmd_rcd_idx = basic_cmd_rcd_idx%GPS_HAL_CMD_RECORD_NUM;  //Over-write oldest recording
        ret = mnldinf_stop_timer(basic_cmd_moniter_timer);
        if(ret == -1) {
            LOGW("mnldinf_stop_timer fail(%s)", strerror(errno));
        }
    } else {
        LOGW("cmd not matched:enter->%d, exit->%d", basic_cmd_list[basic_cmd_rcd_idx].cmd, cmd);
    }
}

void mnldinf_basic_cmd_list_dump(void) {
    int print_idx = 0;
    int cur_rcd = basic_cmd_rcd_idx;
    int print_more = 0;
    int empty_cnt = 0;

    if(basic_last_record_idx != cur_rcd) {
        print_more = 1;
        basic_last_record_idx = cur_rcd;
    }

    LOGW("Dump GPS HAL command record:");
    for(print_idx = 0; print_idx < GPS_HAL_CMD_RECORD_NUM; print_idx++){
        if(basic_cmd_list[print_idx].cmd != 0) {  //Valid cmd
            if(print_idx == cur_rcd) {
                LOGW("[%2d]%s:%3d << Current command(%lld)", print_idx, basic_cmd_list[print_idx].enter_time, basic_cmd_list[print_idx].cmd, (long long)(mnldinf_get_time_in_millisecond() - basic_enter_time_ms ));
            } else {
                if(print_more) {
                    LOGW("[%2d]%s~%s(%lld):%3d", print_idx, basic_cmd_list[print_idx].enter_time, basic_cmd_list[print_idx].exit_time, (long long)basic_cmd_list[print_idx].exec_time, basic_cmd_list[print_idx].cmd );
                }
            }
        } else {
            empty_cnt++;
        }
    }
    if(empty_cnt) {
        LOGW("empty_cnt:%d", empty_cnt);
    }
}

void mnldinf_basic_cmd_list_init(void) {
    memset(basic_cmd_list, 0, sizeof(basic_cmd_list));
    basic_cmd_rcd_idx = 0;
    basic_enter_time_ms = 0;
    basic_last_record_idx = 0xFF;

    basic_cmd_moniter_timer = mnldinf_init_timer(mnldinf_basic_cmd_list_dump);
}

//Tell server the support capability of mnld basic interface
int mnldinf_basic_client_capability_update(int fd) {
    LOGI("mnldinf_basic_client_capability_update, fd:%d", fd);
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;

    if(g_mnldinf_basic_cap_setted == false) {
        LOGD("mnldinf_basic_client_capability_update, set as default.");
        memset(&g_mnldinf_basic_client_cap, 0, sizeof(mnldinf_basic_client_cap));
        g_mnldinf_basic_client_cap.support_gnss = 1;
    }
    mnldinf_put_int(buff, &offset, HAL_MNL_INTERFACE_VERSION);

    mnldinf_put_int(buff, &offset, HAL2MNL_BASIC_CAP_SYNC);
    mnldinf_put_binary(buff, &offset, (char *)&g_mnldinf_basic_client_cap, (int)sizeof(mnldinf_basic_client_cap));

    return mnldinf_tcp_send(fd, buff, offset);
}

void mnldinf_basic_capability_config(int fd, mnldinf_basic_client_cap *cap) {
    if(cap == NULL) {
        LOGE("NULL cap !!!!fd:%d", fd);
        return;
    }
    DUMP_BASIC_CLIENT_CAP(cap);
    memset(&g_mnldinf_basic_client_cap, 0, sizeof(mnldinf_basic_client_cap));
    memcpy(&g_mnldinf_basic_client_cap, cap, sizeof(mnldinf_basic_client_cap));
    g_mnldinf_basic_cap_setted = true;
    if(-1 == mnldinf_basic_client_capability_update(fd)) {
        LOGE("mnldinf_basic_capability_config fail!!!");
    }
}

int mnldinf_basic_gnss_init(int fd) {
    LOGI("mnldinf_basic_gnss_init, fd:%d", fd);
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;
    mnldinf_put_int(buff, &offset, HAL_MNL_INTERFACE_VERSION);

    mnldinf_put_int(buff, &offset, HAL2MNL_GPS_INIT);

    return mnldinf_tcp_send(fd, buff, offset);
}

int mnldinf_basic_gnss_start(int fd) {
    LOGI("mnldinf_basic_gnss_start, fd:%d", fd);
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;
    mnldinf_put_int(buff, &offset, HAL_MNL_INTERFACE_VERSION);

    mnldinf_put_int(buff, &offset, HAL2MNL_GPS_START);

    return mnldinf_tcp_send(fd, buff, offset);
}
int mnldinf_basic_gnss_stop(int fd) {
    LOGI("mnldinf_basic_gnss_stop, fd:%d", fd);
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;
    mnldinf_put_int(buff, &offset, HAL_MNL_INTERFACE_VERSION);

    mnldinf_put_int(buff, &offset, HAL2MNL_GPS_STOP);

    return mnldinf_tcp_send(fd, buff, offset);
}

int mnldinf_basic_gnss_cleanup(int fd) {
    LOGI("mnldinf_basic_gnss_cleanup, fd:%d", fd);
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;
    mnldinf_put_int(buff, &offset, HAL_MNL_INTERFACE_VERSION);

    mnldinf_put_int(buff, &offset, HAL2MNL_GPS_CLEANUP);

    return mnldinf_tcp_send(fd, buff, offset);
}

int mnldinf_basic_gnss_measurement_enable(int fd, bool enabled) {
    LOGD("enabled=%d, fd:%d", enabled, fd);
    char buff[HAL_MNL_BUFF_SIZE] = {0};
    int offset = 0;
    mnldinf_put_int(buff, &offset, HAL_MNL_INTERFACE_VERSION);

    mnldinf_put_int(buff, &offset, HAL2MNL_GPS_MEASUREMENT);
    mnldinf_put_int(buff, &offset, enabled);

    return mnldinf_tcp_send(fd, buff, offset);
}

void mnldinf_dump_gnss_sv_info(gnss_sv_info sv) {
    unsigned int i = 0;
    LOGD("[dump_gnss_sv_info], sv_num:%d", sv.num_svs);
    for(i = 0; i < sv.num_svs; i++) {
        LOGD("SV[%d], cons:%d, Cn0dBHz:%f, elev:%f, azim:%f, flags:%d, cf:%f"
            , sv.sv_list[i].svid, sv.sv_list[i].constellation, sv.sv_list[i].c_n0_dbhz, sv.sv_list[i].elevation
            , sv.sv_list[i].azimuth, sv.sv_list[i].flags, sv.sv_list[i].carrier_frequency);
    }
}

// -1 means failure
int mnldinf_basic_cmd_hdlr(int fd, mnldinf_basic* hdlr) {
    char buff[HAL_MNL_BUFF_SIZE_SND] = {0};
    char buff_read[HAL_MNL_BUFF_SIZE_SND] = {0};
    int offset = 0;
    int ver;
    mnl2hal_cmd cmd;
    int read_len;
    int ret = 0;
    int msg_len = 0;

    read_len = mnldinf_safe_recv(fd, buff_read, sizeof(buff_read));
    if (read_len <= 0) {
        close(fd);
        if(hdlr->mnldinf_connection_broken) {
            LOGW("Connection broken...");
            hdlr->mnldinf_connection_broken();
        }
        LOGE("mnldinf_basic_cmd_hdlr() mnldinf_safe_recvfrom() failed read_len=%d, %s", read_len, strerror(errno));
        return -1;
    }

    mnldinf_put_msg_to_cycle(&g_mnldinf_basic_cbuffer, buff_read, read_len);

    while(mnldinf_get_one_msg(&g_mnldinf_basic_cbuffer, buff) > 0) {
        offset = 0;
        msg_len = mnldinf_get_int(buff, &offset, sizeof(buff));  //Get message length
        UNUSED(msg_len);
        ver = mnldinf_get_int(buff, &offset, sizeof(buff));
        UNUSED(ver);
        cmd = mnldinf_get_int(buff, &offset, sizeof(buff));
        mnldinf_basic_cmd_enter_record(cmd);
        //LOGD("cmd:%d, msg_len:%d, ver:%d", cmd, msg_len, ver);
        switch (cmd) {
        case MNL2HAL_LOCATION: {
            if (hdlr->mnldinf_location) {
                gps_location location;
                memset(&location, 0, sizeof(location));
                mnldinf_get_binary(buff, &offset, (char*)&location, sizeof(buff), sizeof(gps_location));
                hdlr->mnldinf_location(location);
            } else {
                LOGE("mnl2hal_hdlr() location is NULL");
                ret = -1;
            }
            break;
        }
        case MNL2HAL_GPS_STATUS: {
            if (hdlr->mnldinf_gnss_status) {
                gps_status status;
                memset(&status, 0, sizeof(status));
                mnldinf_get_binary(buff, &offset, (char*)&status, sizeof(buff), sizeof(gps_status));
                hdlr->mnldinf_gnss_status(status);
            } else {
                LOGE("mnl2hal_hdlr() gps_status is NULL");
                ret = -1;
            }
            break;
        }
        case MNL2HAL_GPS_SV: {
            if (hdlr->mnldinf_gnss_sv) {
                gnss_sv_info sv;
                memset(&sv, 0, sizeof(sv));
                mnldinf_get_binary(buff, &offset, (char*)&sv, sizeof(buff), sizeof(gnss_sv_info));
                //mnldinf_dump_gnss_sv_info(sv);
                hdlr->mnldinf_gnss_sv(sv);
            } else {
                LOGE("mnl2hal_hdlr() gps_sv is NULL");
                ret = -1;
            }
            break;
        }
        case MNL2HAL_NMEA: {
            if (hdlr->mnldinf_nmea) {
                int64_t timestamp = mnldinf_get_long(buff, &offset, sizeof(buff));
                char* nmea = mnldinf_get_string(buff, &offset, sizeof(buff));
                int used_length = nmea - buff;
                int max_length = sizeof(buff) - used_length;
                int length = mnldinf_get_int(buff, &offset, sizeof(buff));
                length = MIN(length,max_length);
                hdlr->mnldinf_nmea(timestamp, nmea, length);
            } else {
                LOGE("mnl2hal_hdlr() nmea is NULL");
                ret = -1;
            }
            break;
        }
        case MNL2HAL_GNSS_MEASUREMENTS: {
            if (hdlr->mnldinf_gnss_measurements) {
                memset(&meas_data, 0, sizeof(meas_data));
                mnldinf_get_binary(buff, &offset, (char*)&meas_data, sizeof(buff), sizeof(gnss_data));
                hdlr->mnldinf_gnss_measurements(&meas_data);
            } else {
                LOGE("mnl2hal_hdlr() gnss_measurements is NULL");
                ret = -1;
            }
            break;
        }
        case MNL2HAL_BASIC_CAP_SYNC: {
            if(hdlr->mnldinf_capability_update) {
                mnldinf_basic_server_cap cap;
                memset(&cap, 0, sizeof(cap));
                mnldinf_get_binary(buff, &offset, (char *)&cap, sizeof(buff), sizeof(cap));
                DUMP_BASIC_SERVER_CAP(&cap);
                hdlr->mnldinf_capability_update(&cap);
            } else {
                LOGE("mnl2hal_hdlr() mnldinf_capability_update is NULL");
                ret = -1;
            }
            break;
        }
        case MNL2HAL_NMEA_DONE: {
            if(hdlr->mnldinf_nmea_done) {
                hdlr->mnldinf_nmea_done();
            } else {
                LOGE("mnl2hal_hdlr() MNL2HAL_BASIC_CAP_SYNC is NULL");
                ret = -1;
            }
            break;
        }
        default: {
            LOGE("mnl2hal_hdlr() unknown cmd=%d", cmd);
            ret = -1;
            break;
        }
        }

        mnldinf_basic_cmd_exit_record(cmd);
    }

    return ret;
}

void mnldinf_basic_structure_size_dump(void) {
    PRINT_SIZE(mnldinf_basic_client_cap);
    PRINT_SIZE(gps_location);
    PRINT_SIZE(gps_status);
    PRINT_SIZE(gnss_sv_info);
    PRINT_SIZE(gnss_data);
    PRINT_SIZE(mnldinf_basic_server_cap);
}

int mnldinf_basic_register(char *client_name) {
    int fd = -1;

    fd = mnldinf_socket_tcp_client_connect_local(true, MNLDINF_BASIC_TCP);
    LOGD("New client:[%s], fd:%d", client_name, fd);
    if(-1 != fd) {
        mnldinf_basic_structure_size_dump();

        mnldinf_basic_cmd_list_init();

        mnldinf_socket_set_blocking(fd, 0);  //Set as none-blocking
        mnldinf_buffer_initialize(&g_mnldinf_basic_cbuffer, g_mnldinf_basic_msg_buf, sizeof(g_mnldinf_basic_msg_buf));
    }

    return fd;
}
