/*
 * Copyright (c) 2018 Qianxun SI Inc. All rights reserved.
 *
 * Revision: 1.6
 */

#ifndef _RTKSDK_SDK_H_
#define _RTKSDK_SDK_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>
#include <stdint.h>
#include "mtk_gps_type.h"

/**
 * Definition for SDK status.
 */
typedef enum {
    RTKSDK_SDK_STATUS_UNKNOWN = 0,
    RTKSDK_SDK_STATUS_SYSTEM_FAILURE = 1001,
    RTKSDK_SDK_STATUS_NETWORK_UNAVAILABLE = 1002,
    RTKSDK_SDK_STATUS_NETWORK_NO_PERMISSION = 1003,
    RTKSDK_SDK_STATUS_CONFIG_ERROR = 1004,
    RTKSDK_SDK_STATUS_INVALID_PARAM = 1005,
    RTKSDK_SDK_STATUS_UNKNOWN_APPKEY = 1006,
    RTKSDK_SDK_STATUS_NO_ACCESS = 1007,
    RTKSDK_SDK_STATUS_SERVER_ABORT = 1008,

    RTKSDK_SDK_STATUS_INIT_SUCCESS = 2000,
    RTKSDK_SDK_STATUS_INIT_FAILED = 2001,
    RTKSDK_SDK_STATUS_START_SUCCESS = 2002,
    RTKSDK_SDK_STATUS_START_FAILED = 2003,
    RTKSDK_SDK_STATUS_STOP_SUCCESS = 2004,
    RTKSDK_SDK_STATUS_STOP_FAILED = 2005,
    RTKSDK_SDK_STATUS_CLEANUP_SUCCESS = 2006,
    RTKSDK_SDK_STATUS_CLEANUP_FAILED = 2007,

    RTKSDK_SDK_STATUS_GNSS_RUNTIME_ERROR = 3000,
    RTKSDK_SDK_STATUS_GNSS_LOST_POSITION = 3001,
    RTKSDK_SDK_STATUS_GNSS_HAVE_POSITION = 3002,

    RTKSDK_SDK_STATUS_FLP_IMU_FAULT_DETECT = 3500,
    RTKSDK_SDK_STATUS_FLP_IMU_NOMOUNTED = 3501,
    RTKSDK_SDK_STATUS_FLP_ODOMETER_INVALID = 3502,
    RTKSDK_SDK_STATUS_FLP_PPS_INVLIAD = 3503,
    RTKSDK_SDK_STATUS_FLP_GNSS_MEAS_DELAY = 3504,
    RTKSDK_SDK_STATUS_FLP_IMU_BREAK = 3505,
    RTKSDK_SDK_STATUS_FLP_GNSS_BREAK = 3506,

    RTKSDK_SDK_STATUS_AUTH_PARAM_ERROR = 4000,
    RTKSDK_SDK_STATUS_AUTH_ACCOUNT_NOT_EXIST = 4001,
    RTKSDK_SDK_STATUS_AUTH_UNACTIVATED_ACCOUNT = 4002,
    RTKSDK_SDK_STATUS_AUTH_NO_AVAILABLE_ACCOUNT = 4003,
    RTKSDK_SDK_STATUS_AUTH_ACCOUNT_EXPIRED = 4004,
    RTKSDK_SDK_STATUS_AUTH_ACCOUNT_TOEXPIRE = 4005,
    RTKSDK_SDK_STATUS_AUTH_ACCOUNT_NEED_BIND = 4006,
    RTKSDK_SDK_STATUS_AUTH_NO_VDR_PERMISSION = 4007,
    RTKSDK_SDK_STATUS_AUTH_NO_PDR_PERMISSION = 4008,
    RTKSDK_SDK_STATUS_AUTH_NO_AGNSS_PERMISSION = 4009,
    RTKSDK_SDK_STATUS_AUTH_NO_RTK_PERMISSION = 4010,

    RTKSDK_SDK_STATUS_AUTH_SUCCESS = 4011,
    RTKSDK_SDK_STATUS_AUTH_NEED_MANUAL_ACTIVATION = 4012,
    RTKSDK_SDK_STATUS_AUTH_NEED_TERMINAL_ACTIVATION = 4013,
    RTKSDK_SDK_STATUS_AUTH_ACCOUNT_BEING_PROCESSED = 4014,
    RTKSDK_SDK_STATUS_AUTH_ACCOUNT_FORBIDDEN = 4015,
    RTKSDK_SDK_STATUS_AUTH_PARTIAL_CAPS_UNAVAILABLE = 4016,
    RTKSDK_SDK_STATUS_AUTH_NO_CAPS_AVAILABLE = 4017,

    RTKSDK_SDK_STATUS_OSS_CONNECT_SUCCESS = 5000,
    RTKSDK_SDK_STATUS_OSS_CONNECT_FAIL = 5001,
    RTKSDK_SDK_STATUS_OSS_CONNECT_TIMEOUT = 5002,
    RTKSDK_SDK_STATUS_OSS_RECONNECTING = 5003,
    RTKSDK_SDK_STATUS_OSS_DISCONNECT_SUCCESS = 5004,
    RTKSDK_SDK_STATUS_OSS_DISCONNECT_FAIL = 5005,
    RTKSDK_SDK_STATUS_OSS_CONNECTION_KICKOFF = 5006,

    RTKSDK_SDK_STATUS_NOSR_REQUEST_RTK_SUCCESS = 6000,
    RTKSDK_SDK_STATUS_NOSR_REQUEST_RTK_FAIL = 6001,
    RTKSDK_SDK_STATUS_NOSR_CLOSE_RTK_SUCCESS = 6002,
    RTKSDK_SDK_STATUS_NOSR_CLOSE_RTK_FAIL = 6003,
    RTKSDK_SDK_STATUS_NOSR_RTK_UNAVAILABLE = 6004,
    RTKSDK_SDK_STATUS_NOSR_GGA_OUT_OF_SERVICE_AREA = 6005,
    RTKSDK_SDK_STATUS_NOSR_INVALID_GGA = 6006,
    RTKSDK_SDK_STATUS_NOSR_KICKOFF = 6007,
    RTKWX_SDK_STATUS_IAP_SUCCESS     = 7000,
    RTKWX_SDK_STATUS_IAP_FAIL        = 7001,
    RTKWX_SDK_STATUS_IAP_UPDATING     = 7002,
    RTKWX_SDK_STATUS_IAP_TIMEOUT     = 7003
} RTKSDKSdkStatus;

/**
 * Definition for network status.
 */
typedef enum {
    /**
     * unknown network, the initial value.
     */
    RTKSDK_NET_TYPE_UNKNOWN,
    /**
     * none network.
     */
    RTKSDK_NET_TYPE_NONETWORK,
    /**
     * wifi network.
     */
    RTKSDK_NET_TYPE_WIFI,
    /**
     * gsm network.
     */
    RTKSDK_NET_TYPE_GSM
} RTKSDKNetworkStatus;

/**
 * Definition for position mode.
 */
typedef enum {
    /**
     * Positioning with RTK service.
     */
    RTKSDK_POS_MODE_RTK = 1,
    /**
     * Positioning with RTK + DR services.
     */
    RTKSDK_POS_MODE_RTK_DR = 2,
    RTKSDK_POS_MODE_CHIP = 3
} RTKSDKPosMode;

/**
 * Aiding data type(s) to delete
 */
typedef unsigned short                 RTKGNSSAidingData;
#define RTKSDK_GNSS_DEL_EPHEMERIS        0x0001
#define RTKSDK_GNSS_DEL_ALMANAC          0x0002
#define RTKSDK_GNSS_DEL_POSITION         0x0004
#define RTKSDK_GNSS_DEL_TIME             0x0008
#define RTKSDK_GNSS_DEL_IONO             0x0010
#define RTKSDK_GNSS_DEL_UTC              0x0020
#define RTKSDK_GNSS_DEL_HEALTH           0x0040
#define RTKSDK_GNSS_DEL_SVDIR            0x0080
#define RTKSDK_GNSS_DEL_SVSTEER          0x0100
#define RTKSDK_GNSS_DEL_SADATA           0x0200
#define RTKSDK_GNSS_DEL_RTI              0x0400
#define RTKSDK_GNSS_DEL_CELLDB_INFO      0x8000
#define RTKSDK_GNSS_DEL_ALL              0xFFFF


/* Application scenarios */
typedef unsigned char  RTKGnssApplyScene;
#define RTKSDK_GNSS_APPLY_SCENE_ANYWHERE     0   /* for anywhere */
#define RTKSDK_GNSS_APPLY_SCENE_AUTOROOF     1   /* on the roof of auto */
#define RTKSDK_GNSS_APPLY_SCENE_AUTOINSIDE   2   /* inside the auto */
#define RTKSDK_GNSS_APPLY_SCENE_BIKE         3   /* share bike */
#define RTKSDK_GNSS_APPLY_SCENE_CAMERA       4   /* monitor camera */
#define RTKSDK_GNSS_APPLY_SCENE_WATCH        5   /* Watch */


/*Flags to indicate logmask.*/
typedef unsigned int RTKSDKLogMask;
/*LOG enable flag. */
/*Disable all kinds of logs*/
#define RTKSDK_LOG_MASK_OFF  0
/*NOTE: This is Special, it means enable all kinds of logs*/
#define RTKSDK_LOG_MASK_ON   1
/*Enable basic debug log*/
#define RTKSDK_LOG_MASK_DEBUG 0x04
/*Enable Pedatain for rtk playback, it needs more space*/
#define RTKSDK_LOG_MASK_ALGO 0x08
/*Enable NMEA Files for statistical accuracy*/
#define RTKSDK_LOG_MASK_NMEA 0x10
/*Enable Raw Measurement Data*/
#define RTKSDK_LOG_MASK_RAWX 0x20
/*Enable algo_nmea for DR playback, it needs more space*/
#define RTKSDK_LOG_MASK_ALGO_DR 0x40

/* Authentication mode */
typedef unsigned char            RTKSDKAuthMode;
#define RTKSDK_AUTH_MODE_APPKEY    0    /* authenticate by application key */
#define RTKSDK_AUTH_MODE_DSK       1    /* authenticate by device service key */

/**
 * Configuration of SDK server info.
 */
typedef struct {
    size_t          size;
    char            auth_server[64];
    unsigned short  auth_port;
} RTKSDKServerConfig;

/**
 * Represent the SDK configuration.
 */
typedef struct {
    size_t  size;
    /**
     * Authentication mode.
     */
    RTKSDKAuthMode auth_mode;
    /**
     * Refer to authencation mode.
     * Application unique identity for RTK_AUTH_MODE_APPKEY.
     * Device service key for RTK_AUTH_MODE_DSK.
     */
    char    app_key[32];
    /**
     * Refer to authencation mode.
     * Secret for corresponding key.
     */
    char    app_secret[128];
    /**
     * Device id, UUID.
     */
    char    device_id[64];
    /**
     * Device type.
     */
    char    device_type[64];
    /**
     * SDK working directoy.
     */
    char    root_dir[256];

    /**
     * LOG enable flag, please refer to RTKSDKLogMask
     */
    RTKSDKLogMask    log_enable;

    /**
     * Separate log directoy if needed, otherwise logs also stored in the root_dir.
     */
    char    log_dir[256];

    /**
     * The dir to create AF_UNIX socket used by RTK, the dir must have all permissions(RWX).
     */
    char    socket_dir[256];

    /**
     * The path & filename of RTK's configurate file as RTK supply for,
     *  since many of customers wanna to change the path.
     *  Notice, the path MUST contain the filename,etc: "/etc/testCfg.ini".
     */
    char    cfg_filename[256];

    /**
     *  see 'Application scenarios'.
     */
    RTKGnssApplyScene  apply_scenario;

    /**
     * Configuration of SDK server info.
     */
    RTKSDKServerConfig  server_config;
} RTKSDKSdkConfig;


/* Fusion position flag */
typedef unsigned char                 RTKGnssPosFlag;
#define POS_FUSION_FLAG_UNAVAILABLE         0
#define POS_FUSION_FLAG_GNSSONLY            1
#define POS_FUSION_FLAG_DGNSS               2
#define POS_FUSION_FLAG_RTK_FIX             4
#define POS_FUSION_FLAG_RTK_FLOAT           5
#define POS_FUSION_FLAG_FUSION              6
#define POS_FUSION_FLAG_DR                  7
#define POS_FUSION_FLAG_CHIP                8
#define POS_FUSION_FLAG_CHIP_FUSION         9
#define POS_FUSION_FLAG_PPP                 10
#define POS_FUSION_FLAG_BT                  11
#define POS_FUSION_FLAG_WIFI                12
#define POS_FUSION_FLAG_GNSSONLY_FUSION     13
#define POS_FUSION_FLAG_RTK_FUSION          14

typedef unsigned long long            RTKGnssUtcTime;

/* Pos confidence flag*/
typedef unsigned int                  RTKGnssConfidenceFlag;
#define POS_CONFIDENCE_FLAG_INVALID           0
#define POS_CONFIDENCE_FLAG_GNSSONLY          1
#define POS_CONFIDENCE_FLAG_DGNSS             2
#define POS_CONFIDENCE_FLAG_RTK_FIX           4
#define POS_CONFIDENCE_FLAG_RTK_FLOAT         5
#define POS_CONFIDENCE_FLAG_DR                60
#define POS_CONFIDENCE_FLAG_GNSS_FUSION       61
#define POS_CONFIDENCE_FLAG_RTK_FIX_FUSION    64
#define POS_CONFIDENCE_FLAG_RTK_FLOAT_FUSION  65

typedef struct {
    RTKGnssConfidenceFlag  conf_type;
    float             conf_param_68[3];   /* 68% N/E/U confidence */
    float             conf_param_95[3];   /* 95% N/E/U confidence */
    float             conf_param_99[3];   /* 99.9% N/E/U confidence */
} RTKSDKGnssConfidenceParams;


typedef unsigned int RTKSDKReportFreq;
#define RTKSDK_REPORT_FREQ_0_1_HZ   1
#define RTKSDK_REPORT_FREQ_0_2_HZ   2
#define RTKSDK_REPORT_FREQ_1_HZ     3
#define RTKSDK_REPORT_FREQ_5_HZ     4
#define RTKSDK_REPORT_FREQ_10_HZ    5
#define RTKSDK_IOCTL_CMD_IMU_RATE     1
typedef int RTKSDKIMURate;
#define RTKSDK_IMU_RATE_0_HZ          0
#define RTKSDK_IMU_RATE_1_HZ          1
#define RTKSDK_IMU_RATE_2_HZ          2
#define RTKSDK_IMU_RATE_5_HZ          5
#define RTKSDK_IMU_RATE_10_HZ         10
#define RTKSDK_IMU_RATE_25_HZ         25
#define RTKSDK_IMU_RATE_50_HZ         50

/**
 *Set the extra device type
 */
#define RTKSDK_IOCTL_CMD_SET_EXT_DEVTYPE   2

/**
 *  IOCTL TABLE
 *
 *  CMD                             DATA_TYPE           COMMENTS
 *  RTKSDK_IOCTL_CMD_IMU_RATE         RTKSDKIMURate         configure the output rate(0/1/2/5/10/25/50 HZ) of WZIMU info.
 *  RTKSDKIMURate rate = RTKSDK_IMU_RATE_1_HZ;
 *  int len = sizeof(RTKSDKIMURate);
 *  ioctl(RTKSDK_IOCTL_CMD_IMU_RATE, (void *)&rate, (int *)&len));
 *
 */


/**
 * Represent the GNSS location.
 */
typedef struct {
    /**
     * Set to sizeof(RTKSDKGnssLocation)
     */
    size_t          size;
    /**
     * Position fusion flag
     */
    RTKGnssPosFlag   posflag;
    /**
     * Represents latitude in degrees.
     */
    double          latitude;
    /**
     * Represents longitude in degrees.
     */
    double          longitude;
    /**
     * Represents altitude in meters above the WGS 84 reference ellipsoid.
     */
    double          altitude;
    /**
     * Represents speed in meters per second.
     */
    float           speed;
    /**
     * Represents heading in degrees.
     */
    float           bearing;
    /**
     * Represents expected accuracy in meters.
     */
    float           accuracy;
    /**
     * Timestamp for the location fix.
     */
    RTKGnssUtcTime   timestamp;
    /**
     * Average CN0
     */
    float           avg_cno;
    /**
     * HDOP
     */
    float           hdop;
    /**
     * Number of satellites used for positioning.
     */
    unsigned char         sat_used;
    /**
     * Confidence coefficient.
     */
    RTKSDKGnssConfidenceParams confidence_param;
} RTKSDKGnssLocation;

#define RTK_SENSOR_TYPE_LEN         64
/**
 * Represent DR configuration.
 */
typedef struct {
    /**
     * Set to sizeof(RTKSDKSensorConfig).
     */
    size_t  size;
    /**
     * Gyro type
     */
    char    gyro_type[RTK_SENSOR_TYPE_LEN];
    /**
     * Accelerometer type
     */
    char    acce_type[RTK_SENSOR_TYPE_LEN];
    /**
     * Magnetometer type
     */
    char    magn_type[RTK_SENSOR_TYPE_LEN];
    /**
     * Barometer model
     */
    char    press_type[RTK_SENSOR_TYPE_LEN];
    /**
     * Odometer type
     */
    char    odom_type[RTK_SENSOR_TYPE_LEN];
    /**
     * Thermometer type
     */
    char    temp_type[RTK_SENSOR_TYPE_LEN];
    /**
     * Four wheel speed type
     */
    char    ospd_type[RTK_SENSOR_TYPE_LEN];
} RTKSDKSensorConfig;

typedef struct {
    unsigned long long       TimeStamp;
    unsigned short int       CheckSum;
    unsigned short int       MatchConfidence;
    unsigned short int       RoadID;
    unsigned short int       GridID;
    unsigned char            RoadType;         // 0:normal 1:channel  2:tunnel, 3:bridge 4:near IC
    unsigned char            RoadClass;
    signed short int         NorthDis;         //m
    signed short int         EastDis;          //m
    unsigned short int       LinkLength1;      //m
    unsigned short int       LinkLength2;      //m
    unsigned short int       LinkDirection;    // degree
    double                   MatchLat;         // degree
    double                   MatchLon;         // degree;
    unsigned char            MatchType;
    signed char              State;
    unsigned long long       Reserved;         // 0:unknown, 1:from server 2:from local 3:local ramp entry 4: local ramp exit
    float                    MatchDistance;    //m
    signed short int         Fence;            // 0:unsure,  2: ele road
    unsigned  int            ADCode;           //000000
    float                    RoadSpeedLimit;   //meters/s
    float                    RoadHeading;      // road heading, from -180~180
    unsigned short int       MapMatchedRamp;
    int                      NearMap;          // 0-NO, !0-YES
} RTKSDKMapMatchingPara;

typedef enum {
    RTK_SENSOR_TYPE_ACCEL = 0X0002,
    RTK_SENSOR_TYPE_MAGN  = 0X0004,
    RTK_SENSOR_TYPE_PRESS = 0X0008,
    RTK_SENSOR_TYPE_GYRO  = 0X0010,
    RTK_SENSOR_TYPE_TEMP  = 0X0020,
    RTK_SENSOR_TYPE_ODOM  = 0X0040,
    RTK_SENSOR_TYPE_4SPD  = 0X0080
} RTK_SENSOR_TYPE;

typedef struct {
    unsigned int             m_Mask;       /* valid sensor mask */
    unsigned long long       m_Time;       /* time */
    float                    m_ARate[3];   /* angular rate in radius per second */
    float                    m_SForce[3];  /* specific force in m/s2 */
    float                    m_Mag[3];     /* mag field in gauss */
    float                    m_Baro;       /* baro in Pa */
    float                    m_Temp;       /* temperature in celsius */
    float                    m_Speed;      /* auto speed in m/s */
    float                    m_4Spd[5];    /* steering and 4 speed of front-left, front-right, rear-left, rear-right respectively */
} RTKSDKSensorData;

typedef struct {
    int                 Valid;      // 0 means invalid; Non-zero means valid
    unsigned long long  TimeStamp;
} RTKSDKGNSSPPS;

typedef struct {
    int                 Valid;      // 0 means invalid; Non-zero means valid
    unsigned long long  TimeStamp;
    float               Orientation;
    float               BaseLen;
} RTKSDKGNSSOrt;

/**
 * Represent Lever-arm configuration.
 */
typedef struct {
    /**
     * Set to sizeof(RTKSDKLeverArmParams).
     */
    size_t   size;
    /**
     * Lever-arm of GNSS antenna and IMU
     */
    float    gnss2imu[3];
    /**
     * Lever-arm of IMU and the middle of vehicle's rear wheels
     */
    float    imu2rearmiddle[3];
    /**
     * misalignment angle of IMU and vehicle
     */
    float    imu2vehicle[3];
    /**
     * REAR Wheel Track
     */
    float    wheeltrack;
} RTKSDKLeverArmParams;

/** Maximum number of Measurements in gnss_measurement_callback(). */
#define RTKSDK_GNSS_MAX_MEASUREMENT   128
/** Maximum number of frequency in each GNSS system that is used in position calculation. */
#define RTKSDK_MAX_FREQ_NUM           1

/**
 * Flags to indicate what fields in RTKSDKGnssMeasurement are valid.
 */
typedef uint32_t RTKSDKGnssMeasurementFlags;
/** A valid 'snr' is stored in the data structure.                      */
#define RTKSDK_GNSS_MEASUREMENT_HAS_SNR                               (1<<0)
/** A valid 'elevation' is stored in the data structure.                */
#define RTKSDK_GNSS_MEASUREMENT_HAS_ELEVATION                         (1<<1)
/** A valid 'elevation uncertainty' is stored in the data structure.    */
#define RTKSDK_GNSS_MEASUREMENT_HAS_ELEVATION_UNCERTAINTY             (1<<2)
/** A valid 'azimuth' is stored in the data structure.                  */
#define RTKSDK_GNSS_MEASUREMENT_HAS_AZIMUTH                           (1<<3)
/** A valid 'azimuth uncertainty' is stored in the data structure.      */
#define RTKSDK_GNSS_MEASUREMENT_HAS_AZIMUTH_UNCERTAINTY               (1<<4)
/** A valid 'pseudorange' is stored in the data structure.              */
#define RTKSDK_GNSS_MEASUREMENT_HAS_PSEUDORANGE                       (1<<5)
/** A valid 'pseudorange uncertainty' is stored in the data structure.  */
#define RTKSDK_GNSS_MEASUREMENT_HAS_PSEUDORANGE_UNCERTAINTY           (1<<6)
/** A valid 'carrier frequency' is stored in the data structure.        */
#define RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_FREQUENCY                 (1<<9)
/** A valid 'carrier cycles' is stored in the data structure.           */
#define RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_CYCLES                    (1<<10)
/** A valid 'carrier phase' is stored in the data structure.            */
#define RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_PHASE                     (1<<11)
/** A valid 'carrier phase uncertainty' is stored in the data structure. */
#define RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_PHASE_UNCERTAINTY         (1<<12)
/** A valid 'doppler shift' is stored in the data structure.            */
#define RTKSDK_GNSS_MEASUREMENT_HAS_DOPPLER_SHIFT                     (1<<15)
/** A valid 'doppler shift uncertainty' is stored in the data structure. */
#define RTKSDK_GNSS_MEASUREMENT_HAS_DOPPLER_SHIFT_UNCERTAINTY         (1<<16)
/** A valid 'used in fix' flag is stored in the data structure.         */
#define RTKSDK_GNSS_MEASUREMENT_HAS_USED_IN_FIX                       (1<<17)


/**
 * Constellation type of GnssSvInfo
 */
typedef uint8_t RTKSDKGnssConstellationType;
#define RTKSDK_GNSS_CONSTELLATION_UNKNOWN      (0)
#define RTKSDK_GNSS_CONSTELLATION_GPS          (1)
#define RTKSDK_GNSS_CONSTELLATION_SBAS         (2)
#define RTKSDK_GNSS_CONSTELLATION_GLONASS      (3)
#define RTKSDK_GNSS_CONSTELLATION_QZSS         (4)
#define RTKSDK_GNSS_CONSTELLATION_BEIDOU       (5)
#define RTKSDK_GNSS_CONSTELLATION_GALILEO      (6)

typedef uint16_t RTKSDKGnssNavigationMessageType;
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_UNKNOWN        0
/** GPS L1 C/A message contained in the structure.      */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GPS_L1CA       0x0101
/** GPS L2-CNAV message contained in the structure.     */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GPS_L2CNAV     0x0102
/** GPS L5-CNAV message contained in the structure.     */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GPS_L5CNAV     0x0103
/** GPS CNAV-2 message contained in the structure.      */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GPS_CNAV2      0x0104
/** Glonass L1 CA message contained in the structure.   */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GLO_L1CA       0x0301
/** Beidou D1 message contained in the structure.       */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_BDS_D1         0x0501
/** Beidou D2 message contained in the structure.       */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_BDS_D2         0x0502
/** Galileo I/NAV message contained in the structure.   */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GAL_I          0x0601
/** Galileo F/NAV message contained in the structure.   */
#define RTKSDK_GNSS_NAVIGATION_MESSAGE_TYPE_GAL_F          0x0602

/**
 * Flags indicating the GNSS measurement state.
 *
 * The expected behavior here is for GPS HAL to set all the flags that applies.
 * For example, if the state for a satellite is only C/A code locked and bit
 * synchronized, and there is still millisecond ambiguity, the state should be
 * set as:
 *
 * RTKSDK_GNSS_MEASUREMENT_STATE_CODE_LOCK | RTKSDK_GNSS_MEASUREMENT_STATE_BIT_SYNC |
 *         RTKSDK_GNSS_MEASUREMENT_STATE_MSEC_AMBIGUOUS
 *
 * If GNSS is still searching for a satellite, the corresponding state should be
 * set to RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN(0).
 */
typedef uint32_t RTKSDKGnssMeasurementState;
#define RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN                  (0)
#define RTKSDK_GNSS_MEASUREMENT_STATE_CODE_LOCK                (1<<0)
#define RTKSDK_GNSS_MEASUREMENT_STATE_BIT_SYNC                 (1<<1)
#define RTKSDK_GNSS_MEASUREMENT_STATE_SUBFRAME_SYNC            (1<<2)
#define RTKSDK_GNSS_MEASUREMENT_STATE_TOW_DECODED              (1<<3)
#define RTKSDK_GNSS_MEASUREMENT_STATE_MSEC_AMBIGUOUS           (1<<4)
#define RTKSDK_GNSS_MEASUREMENT_STATE_SYMBOL_SYNC              (1<<5)
#define RTKSDK_GNSS_MEASUREMENT_STATE_GLO_STRING_SYNC          (1<<6)
#define RTKSDK_GNSS_MEASUREMENT_STATE_GLO_TOD_DECODED          (1<<7)
#define RTKSDK_GNSS_MEASUREMENT_STATE_BDS_D2_BIT_SYNC          (1<<8)
#define RTKSDK_GNSS_MEASUREMENT_STATE_BDS_D2_SUBFRAME_SYNC     (1<<9)
#define RTKSDK_GNSS_MEASUREMENT_STATE_GAL_E1BC_CODE_LOCK       (1<<10)
#define RTKSDK_GNSS_MEASUREMENT_STATE_GAL_E1C_2ND_CODE_LOCK    (1<<11)
#define RTKSDK_GNSS_MEASUREMENT_STATE_GAL_E1B_PAGE_SYNC        (1<<12)
#define RTKSDK_GNSS_MEASUREMENT_STATE_SBAS_SYNC                (1<<13)
#define RTKSDK_GNSS_MEASUREMENT_STATE_TOW_KNOWN                (1<<14)
#define RTKSDK_GNSS_MEASUREMENT_STATE_GLO_TOD_KNOWN            (1<<15)


/**
 * Flags indicating the Accumulated Delta Range's states.
 */
typedef uint16_t RTKSDKGnssAccumulatedDeltaRangeState;
#define RTKSDK_GNSS_ADR_STATE_UNKNOWN                      (0)
#define RTKSDK_GNSS_ADR_STATE_VALID                        (1<<0)
#define RTKSDK_GNSS_ADR_STATE_RESET                        (1<<1)
#define RTKSDK_GNSS_ADR_STATE_CYCLE_SLIP                   (1<<2)
#define RTKSDK_GNSS_ADR_STATE_HALF_CYCLE_RESOLVED          (1<<3)
#define RTKSDK_GNSS_ADR_STATE_HALF_CYCLE_REPORTED          (1<<4)


/**
 * Enumeration of available values for the GNSS Measurement's multipath
 * indicator.
 */
typedef uint8_t RTKSDKGnssMultipathIndicator;
/** The indicator is not available or unknown.                  */
#define RTKSDK_GNSS_MULTIPATH_INDICATOR_UNKNOWN                (0)
/** The measurement is indicated to be affected by multipath.   */
#define RTKSDK_GNSS_MULTIPATH_INDICATOR_PRESENT                (1)
/** The measurement is indicated to be not affected by multipath. */
#define RTKSDK_GNSS_MULTIPATH_INDICATOR_NOT_PRESENT            (2)

/**
 * Flags to indicate what fields in RTKSDKGnssClock are valid.
 */
typedef uint16_t RTKSDKGnssClockFlags;
/** A valid 'leap second' is stored in the data structure.      */
#define RTKSDK_GNSS_CLOCK_HAS_LEAP_SECOND               (1<<0)
/** A valid 'time uncertainty' is stored in the data structure. */
#define RTKSDK_GNSS_CLOCK_HAS_TIME_UNCERTAINTY          (1<<1)
/** A valid 'full bias' is stored in the data structure.        */
#define RTKSDK_GNSS_CLOCK_HAS_FULL_BIAS                 (1<<2)
/** A valid 'bias' is stored in the data structure. */
#define RTKSDK_GNSS_CLOCK_HAS_BIAS                      (1<<3)
/** A valid 'bias uncertainty' is stored in the data structure. */
#define RTKSDK_GNSS_CLOCK_HAS_BIAS_UNCERTAINTY          (1<<4)
/** A valid 'drift' is stored in the data structure. */
#define RTKSDK_GNSS_CLOCK_HAS_DRIFT                     (1<<5)
/** A valid 'drift uncertainty' is stored in the data structure. */
#define RTKSDK_GNSS_CLOCK_HAS_DRIFT_UNCERTAINTY         (1<<6)


/**
 * Status of Navigation Message
 * When a message is received properly without any parity error in its navigation words, the
 * status should be set to NAV_MESSAGE_STATUS_PARITY_PASSED. But if a message is received
 * with words that failed parity check, but GPS is able to correct those words, the status
 * should be set to NAV_MESSAGE_STATUS_PARITY_REBUILT.
 * No need to send any navigation message that contains words with parity error and cannot be
 * corrected.
 */
typedef uint16_t RTKSDKNavigationMessageStatus;
#define RTKSDK_NAV_MESSAGE_STATUS_UNKONW           (0)
#define RTKSDK_NAV_MESSAGE_STATUS_PARITY_PASSED    (1<<0)
#define RTKSDK_NAV_MESSAGE_STATUS_PARITY_REBUILT   (1<<1)

/**
 * Represents an estimate of the GNSS clock time.
 */
typedef struct {
    /** set to sizeof(RTKSDKGnssClock) */
    size_t size;

    /**
     * A set of flags indicating the validity of the fields in this data
     * structure.
     */
    RTKSDKGnssClockFlags flags;

    /**
     * Leap second data.
     * The sign of the value is defined by the following equation:
     *      utc_time_ns = time_ns - (full_bias_ns + bias_ns) - leap_second *
     *      1,000,000,000
     *
     * If the data is available 'flags' must contain RTKSDK_GNSS_CLOCK_HAS_LEAP_SECOND.
     */
    int16_t leap_second;

    /**
     * The GNSS receiver internal clock value. This is the local hardware clock
     * value.
     *
     * For local hardware clock, this value is expected to be monotonically
     * increasing while the hardware clock remains power on. (For the case of a
     * HW clock that is not continuously on, see the
     * hw_clock_discontinuity_count field). The receiver's estimate of GPS time
     * can be derived by substracting the sum of full_bias_ns and bias_ns (when
     * available) from this value.
     *
     * This GPS time is expected to be the best estimate of current GPS time
     * that GNSS receiver can achieve.
     *
     * Sub-nanosecond accuracy can be provided by means of the 'bias_ns' field.
     * The value contains the 'time uncertainty' in it.
     *
     * This field is mandatory.
     */
    int64_t time_ns;

    /**
     * 1-Sigma uncertainty associated with the clock's time in nanoseconds.
     * The uncertainty is represented as an absolute (single sided) value.
     *
     * If the data is available, 'flags' must contain
     * RTKSDK_GNSS_CLOCK_HAS_TIME_UNCERTAINTY. This value is effectively zero (it is
     * the reference local clock, by which all other times and time
     * uncertainties are measured.)  (And thus this field can be not provided,
     * per RTKSDK_GNSS_CLOCK_HAS_TIME_UNCERTAINTY flag, or provided & set to 0.)
     */
    double time_uncertainty_ns;

    /**
     * The difference between hardware clock ('time' field) inside GPS receiver
     * and the true GPS time since 0000Z, January 6, 1980, in nanoseconds.
     *
     * The sign of the value is defined by the following equation:
     *      local estimate of GPS time = time_ns - (full_bias_ns + bias_ns)
     *
     * This value is mandatory if the receiver has estimated GPS time. If the
     * computed time is for a non-GPS constellation, the time offset of that
     * constellation to GPS has to be applied to fill this value. The error
     * estimate for the sum of this and the bias_ns is the bias_uncertainty_ns,
     * and the caller is responsible for using this uncertainty (it can be very
     * large before the GPS time has been solved for.) If the data is available
     * 'flags' must contain RTKSDK_GNSS_CLOCK_HAS_FULL_BIAS.
     */
    int64_t full_bias_ns;

    /**
     * Sub-nanosecond bias.
     * The error estimate for the sum of this and the full_bias_ns is the
     * bias_uncertainty_ns
     *
     * If the data is available 'flags' must contain RTKSDK_GNSS_CLOCK_HAS_BIAS. If GPS
     * has computed a position fix. This value is mandatory if the receiver has
     * estimated GPS time.
     */
    double bias_ns;

    /**
     * 1-Sigma uncertainty associated with the local estimate of GPS time (clock
     * bias) in nanoseconds. The uncertainty is represented as an absolute
     * (single sided) value.
     *
     * If the data is available 'flags' must contain
     * RTKSDK_GNSS_CLOCK_HAS_BIAS_UNCERTAINTY. This value is mandatory if the receiver
     * has estimated GPS time.
     */
    double bias_uncertainty_ns;

    /**
     * The clock's drift in nanoseconds (per second).
     *
     * A positive value means that the frequency is higher than the nominal
     * frequency, and that the (full_bias_ns + bias_ns) is growing more positive
     * over time.
     *
     * The value contains the 'drift uncertainty' in it.
     * If the data is available 'flags' must contain RTKSDK_GNSS_CLOCK_HAS_DRIFT.
     *
     * This value is mandatory if the receiver has estimated GNSS time
     */
    double drift_nsps;

    /**
     * 1-Sigma uncertainty associated with the clock's drift in nanoseconds (per second).
     * The uncertainty is represented as an absolute (single sided) value.
     *
     * If the data is available 'flags' must contain
     * RTKSDK_GNSS_CLOCK_HAS_DRIFT_UNCERTAINTY. If GPS has computed a position fix this
     * field is mandatory and must be populated.
     */
    double drift_uncertainty_nsps;

    /**
     * When there are any discontinuities in the HW clock, this field is
     * mandatory.
     *
     * A "discontinuity" is meant to cover the case of a switch from one source
     * of clock to another.  A single free-running crystal oscillator (XO)
     * should generally not have any discontinuities, and this can be set and
     * left at 0.
     *
     * If, however, the time_ns value (HW clock) is derived from a composite of
     * sources, that is not as smooth as a typical XO, or is otherwise stopped &
     * restarted, then this value shall be incremented each time a discontinuity
     * occurs.  (E.g. this value may start at zero at device boot-up and
     * increment each time there is a change in clock continuity. In the
     * unlikely event that this value reaches full scale, rollover (not
     * clamping) is required, such that this value continues to change, during
     * subsequent discontinuity events.)
     *
     * While this number stays the same, between RTKGnssClock reports, it can be
     * safely assumed that the time_ns value has been running continuously, e.g.
     * derived from a single, high quality clock (XO like, or better, that's
     * typically used during continuous GNSS signal sampling.)
     *
     * It is expected, esp. during periods where there are few GNSS signals
     * available, that the HW clock be discontinuity-free as long as possible,
     * as this avoids the need to use (waste) a GNSS measurement to fully
     * re-solve for the GPS clock bias and drift, when using the accompanying
     * measurements, from consecutive RTKSDKGnssData reports.
     */
    uint32_t hw_clock_discontinuity_count;

    /**
     * system time tag, millisecond since 1970/1/1 locally.
     */
    int64_t time_stamp;

} RTKSDKGnssClock;

typedef struct {
    /**
     * Time offset at which the measurement was taken in nanoseconds.
     * The reference receiver's time is specified by RTKSDKGnssData::clock::time_ns and should be
     * interpreted in the same way as indicated by GpsClock::type.
     *
     * The sign of time_offset_ns is given by the following equation:
     *      measurement time = RTKSDKGnssClock::time_ns + time_offset_ns
     *
     * It provides an individual time-stamp for the measurement, and allows sub-nanosecond accuracy.
     * This is a mandatory value.
     */
    double time_offset_ns;

    /**
     * Per satellite sync state. It represents the current sync state for the associated satellite.
     * Based on the sync state, the 'received GPS tow' field should be interpreted accordingly.
     *
     * This is a mandatory value.
     */
    RTKSDKGnssMeasurementState state;

    /**
     * The received GNSS Time-of-Week at the measurement time, in nanoseconds.
     * Ensure that this field is independent (see comment at top of
     * RTKGnssMeasurement struct.)
     *
     * For GPS & QZSS, this is:
     *   Received GPS Time-of-Week at the measurement time, in nanoseconds.
     *   The value is relative to the beginning of the current GPS week.
     *
     *   Given the highest sync state that can be achieved, per each satellite, valid range
     *   for this field can be:
     *     Searching       : [ 0       ]   : RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN
     *     C/A code lock   : [ 0   1ms ]   : RTKSDK_GNSS_MEASUREMENT_STATE_CODE_LOCK is set
     *     Bit sync        : [ 0  20ms ]   : RTKSDK_GNSS_MEASUREMENT_STATE_BIT_SYNC is set
     *     Subframe sync   : [ 0    6s ]   : RTKSDK_GNSS_MEASUREMENT_STATE_SUBFRAME_SYNC is set
     *     TOW decoded     : [ 0 1week ]   : RTKSDK_GNSS_MEASUREMENT_STATE_TOW_DECODED is set
     *
     *   Note well: if there is any ambiguity in integer millisecond,
     *   RTKSDK_GNSS_MEASUREMENT_STATE_MSEC_AMBIGUOUS should be set accordingly, in the 'state' field.
     *
     *   This value must be populated if 'state' != RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN.
     *
     * For Glonass, this is:
     *   Received Glonass time of day, at the measurement time in nanoseconds.
     *
     *   Given the highest sync state that can be achieved, per each satellite, valid range for
     *   this field can be:
     *     Searching       : [ 0       ]   : RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN
     *     C/A code lock   : [ 0   1ms ]   : RTKSDK_GNSS_MEASUREMENT_STATE_CODE_LOCK is set
     *     Symbol sync     : [ 0  10ms ]   : RTKSDK_GNSS_MEASUREMENT_STATE_SYMBOL_SYNC is set
     *     Bit sync        : [ 0  20ms ]   : RTKSDK_GNSS_MEASUREMENT_STATE_BIT_SYNC is set
     *     String sync     : [ 0    2s ]   : RTKSDK_GNSS_MEASUREMENT_STATE_GLO_STRING_SYNC is set
     *     Time of day     : [ 0  1day ]   : RTKSDK_GNSS_MEASUREMENT_STATE_GLO_TOD_DECODED is set
     *
     * For Beidou, this is:
     *   Received Beidou time of week, at the measurement time in nanoseconds.
     *
     *   Given the highest sync state that can be achieved, per each satellite, valid range for
     *   this field can be:
     *     Searching    : [ 0       ] : RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN
     *     C/A code lock: [ 0   1ms ] : RTKSDK_GNSS_MEASUREMENT_STATE_CODE_LOCK is set
     *     Bit sync (D2): [ 0   2ms ] : RTKSDK_GNSS_MEASUREMENT_STATE_BDS_D2_BIT_SYNC is set
     *     Bit sync (D1): [ 0  20ms ] : RTKSDK_GNSS_MEASUREMENT_STATE_BIT_SYNC is set
     *     Subframe (D2): [ 0  0.6s ] : RTKSDK_GNSS_MEASUREMENT_STATE_BDS_D2_SUBFRAME_SYNC is set
     *     Subframe (D1): [ 0    6s ] : RTKSDK_GNSS_MEASUREMENT_STATE_SUBFRAME_SYNC is set
     *     Time of week : [ 0 1week ] : RTKSDK_GNSS_MEASUREMENT_STATE_TOW_DECODED is set
     *
     * For Galileo, this is:
     *   Received Galileo time of week, at the measurement time in nanoseconds.
     *
     *     E1BC code lock   : [ 0   4ms ]   : RTKSDK_GNSS_MEASUREMENT_STATE_GAL_E1BC_CODE_LOCK is set
     *     E1C 2nd code lock: [ 0 100ms ]   :
     *     RTKSDK_GNSS_MEASUREMENT_STATE_GAL_E1C_2ND_CODE_LOCK is set
     *
     *     E1B page    : [ 0    2s ] : RTKSDK_GNSS_MEASUREMENT_STATE_GAL_E1B_PAGE_SYNC is set
     *     Time of week: [ 0 1week ] : RTKSDK_GNSS_MEASUREMENT_STATE_TOW_DECODED is set
     *
     * For SBAS, this is:
     *   Received SBAS time, at the measurement time in nanoseconds.
     *
     *   Given the highest sync state that can be achieved, per each satellite,
     *   valid range for this field can be:
     *     Searching    : [ 0     ] : RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN
     *     C/A code lock: [ 0 1ms ] : RTKSDK_GNSS_MEASUREMENT_STATE_CODE_LOCK is set
     *     Symbol sync  : [ 0 2ms ] : RTKSDK_GNSS_MEASUREMENT_STATE_SYMBOL_SYNC is set
     *     Message      : [ 0  1s ] : RTKSDK_GNSS_MEASUREMENT_STATE_SBAS_SYNC is set
     */
    int64_t received_sv_time_in_ns;

    /**
     * 1-Sigma uncertainty of the Received GPS Time-of-Week in nanoseconds.
     *
     * This value must be populated if 'state' != RTKSDK_GNSS_MEASUREMENT_STATE_UNKNOWN.
     */
    int64_t received_sv_time_uncertainty_in_ns;

    /**
     * Carrier-to-noise density in dB-Hz, typically in the range [0, 63].
     * It contains the measured C/N0 value for the signal at the antenna port.
     *
     * This is a mandatory value.
     */
    double c_n0_dbhz;

    /**
     * Pseudorange rate at the timestamp in m/s. The correction of a given
     * Pseudorange Rate value includes corrections for receiver and satellite
     * clock frequency errors. Ensure that this field is independent (see
     * comment at top of RTKGnssMeasurement struct.)
     *
     * It is mandatory to provide the 'uncorrected' 'pseudorange rate', and provide GpsClock's
     * 'drift' field as well (When providing the uncorrected pseudorange rate, do not apply the
     * corrections described above.)
     *
     * The value includes the 'pseudorange rate uncertainty' in it.
     * A positive 'uncorrected' value indicates that the SV is moving away from the receiver.
     *
     * The sign of the 'uncorrected' 'pseudorange rate' and its relation to the sign of 'doppler
     * shift' is given by the equation:
     *      pseudorange rate = -k * doppler shift   (where k is a constant)
     *
     * This should be the most accurate pseudorange rate available, based on
     * fresh signal measurements from this channel.
     *
     * It is mandatory that this value be provided at typical carrier phase PRR
     * quality (few cm/sec per second of uncertainty, or better) - when signals
     * are sufficiently strong & stable, e.g. signals from a GPS simulator at >=
     * 35 dB-Hz.
     */
    double pseudorange_rate_mps;

    /**
     * 1-Sigma uncertainty of the pseudorange_rate_mps.
     * The uncertainty is represented as an absolute (single sided) value.
     *
     * This is a mandatory value.
     */
    double pseudorange_rate_uncertainty_mps;

    /**
     * Accumulated delta range's state. It indicates whether ADR is reset or there is a cycle slip
     * (indicating loss of lock).
     *
     * This is a mandatory value.
     */
    RTKSDKGnssAccumulatedDeltaRangeState accumulated_delta_range_state;

    /**
     * Accumulated delta range since the last channel reset in meters.
     * A positive value indicates that the SV is moving away from the receiver.
     *
     * The sign of the 'accumulated delta range' and its relation to the sign of 'carrier phase'
     * is given by the equation:
     *          accumulated delta range = -k * carrier phase    (where k is a constant)
     *
     * This value must be populated if 'accumulated delta range state' != RTKSDK_GNSS_ADR_STATE_UNKNOWN.
     * However, it is expected that the data is only accurate when:
     *      'accumulated delta range state' == GPS_ADR_STATE_VALID.
     */
    double accumulated_delta_range_m;

    /**
     * 1-Sigma uncertainty of the accumulated delta range in meters.
     * This value must be populated if 'accumulated delta range state' != RTKSDK_GNSS_ADR_STATE_UNKNOWN.
     */
    double accumulated_delta_range_uncertainty_m;

    /**
     * Best derived Pseudorange by the chip-set, in meters.
     * The value contains the 'pseudorange uncertainty' in it.
     *
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_PSEUDORANGE.
     */
    double pseudorange_m;

    /**
     * 1-Sigma uncertainty of the pseudorange in meters.
     * The value contains the 'pseudorange' and 'clock' uncertainty in it.
     * The uncertainty is represented as an absolute (single sided) value.
     *
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_PSEUDORANGE_UNCERTAINTY.
     */
    double pseudorange_uncertainty_m;

    /**
     * Carrier frequency at which codes and messages are modulated, it can be L1 or L2.
     * If the field is not set, the carrier frequency is assumed to be L1.
     *
     * If the data is available, 'flags' must contain
     * RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_FREQUENCY.
     */
    float carrier_frequency_hz;

    /**
     * The number of full carrier cycles between the satellite and the receiver.
     * The reference frequency is given by the field 'carrier_frequency_hz'.
     * Indications of possible cycle slips and resets in the accumulation of
     * this value can be inferred from the accumulated_delta_range_state flags.
     *
     * If the data is available, 'flags' must contain
     * RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_CYCLES.
     */
    int64_t carrier_cycles;

    /**
     * The RF phase detected by the receiver, in the range [0.0, 1.0].
     * This is usually the fractional part of the complete carrier phase measurement.
     *
     * The reference frequency is given by the field 'carrier_frequency_hz'.
     * The value contains the 'carrier-phase uncertainty' in it.
     *
     * If the data is available, 'flags' must contain
     * RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_PHASE.
     */
    double carrier_phase;

    /**
     * 1-Sigma uncertainty of the carrier-phase.
     * If the data is available, 'flags' must contain
     * RTKSDK_GNSS_MEASUREMENT_HAS_CARRIER_PHASE_UNCERTAINTY.
     */
    double carrier_phase_uncertainty;

    /**
     * Doppler shift in Hz.
     * A positive value indicates that the SV is moving toward the receiver.
     *
     * The reference frequency is given by the field 'carrier_frequency_hz'.
     * The value contains the 'doppler shift uncertainty' in it.
     *
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_DOPPLER_SHIFT.
     */
    double doppler_shift_hz;

    /**
     * 1-Sigma uncertainty of the doppler shift in Hz.
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_DOPPLER_SHIFT_UNCERTAINTY.
     */
    double doppler_shift_uncertainty_hz;

    /**
     * An enumeration that indicates the 'multipath' state of the event.
     *
     * The multipath Indicator is intended to report the presence of overlapping
     * signals that manifest as distorted correlation peaks.
     *
     * - if there is a distorted correlation peak shape, report that multipath
     *   is RTKSDK_GNSS_MULTIPATH_INDICATOR_PRESENT.
     * - if there is not a distorted correlation peak shape, report
     *   RTKSDK_GNSS_MULTIPATH_INDICATOR_NOT_PRESENT
     * - if signals are too weak to discern this information, report
     *   RTKSDK_GNSS_MULTIPATH_INDICATOR_UNKNOWN
     *
     * Example: when doing the standardized overlapping Multipath Performance
     * test (3GPP TS 34.171) the Multipath indicator should report
     * RTKSDK_GNSS_MULTIPATH_INDICATOR_PRESENT for those signals that are tracked, and
     * contain multipath, and RTKSDK_GNSS_MULTIPATH_INDICATOR_NOT_PRESENT for those
     * signals that are tracked and do not contain multipath.
     */
    RTKSDKGnssMultipathIndicator multipath_indicator;

    /**
     * Signal-to-noise ratio at correlator output in dB.
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_SNR.
     * This is the power ratio of the "correlation peak height above the
     * observed noise floor" to "the noise RMS".
     */
    double snr_db;

    /**
     * Loss of lock indicator.
     */
    int32_t LLI;

} QZWZGnssChannelMeas;

/**
 * Represents a GNSS Measurement, it contains raw and computed information.
 *
 * Independence - All signal measurement information (e.g. sv_time,
 * pseudorange_rate, multipath_indicator) reported in this struct should be
 * based on GNSS signal measurements only. You may not synthesize measurements
 * by calculating or reporting expected measurements based on known or estimated
 * position, velocity, or time.
 */
typedef struct {
    /** set to sizeof(RTKSDKGnssMeasurement) */
    size_t size;

    /** A set of flags indicating the validity of the fields in this data structure. */
    RTKSDKGnssMeasurementFlags flags;

    /**
     * Pseudo-random number for the SV, or FCN/OSN number for Glonass. The
     * distinction is made by looking at constellation field. Values should be
     * in the range of:
     *
     * - GPS:     1-32
     * - SBAS:    120-151, 183-192
     * - GLONASS: 1-24, the orbital slot number (OSN), if known.  Or, if not:
     *            93-106, the frequency channel number (FCN) (-7 to +6) offset by + 100
     *            i.e. report an FCN of -7 as 93, FCN of 0 as 100, and FCN of +6 as 106.
     * - QZSS:    193-200
     * - Galileo: 1-36
     * - Beidou:  1-37
     */
    int16_t svid;

    /**
     * Defines the constellation of the given SV. Value should be one of those
     * GNSS_CONSTELLATION_* constants
     */
    RTKSDKGnssConstellationType constellation;

    /**
     * A set of bit-mask indicating the validity of parameter group of each
     * frequency point. From low bit to high, corresponding to different GNSS system,
     * each bit relates to L1, L2, L5 from GPS, or L1, L2, L3 from GLONASS, and so on.
     */
    uint8_t gnss_frequency_mask;


    /**
     * Elevation in degrees, the valid range is [-90, 90].
     * The value contains the 'elevation uncertainty' in it.
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_ELEVATION.
     */
    double elevation_deg;

    /**
     * 1-Sigma uncertainty of the elevation in degrees, the valid range is [0, 90].
     * The uncertainty is represented as the absolute (single sided) value.
     *
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_ELEVATION_UNCERTAINTY.
     */
    double elevation_uncertainty_deg;

    /**
     * Azimuth in degrees, in the range [0, 360).
     * The value contains the 'azimuth uncertainty' in it.
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_AZIMUTH.
     *  */
    double azimuth_deg;

    /**
     * 1-Sigma uncertainty of the azimuth in degrees, the valid range is [0, 180].
     * The uncertainty is represented as an absolute (single sided) value.
     *
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_AZIMUTH_UNCERTAINTY.
     */
    double azimuth_uncertainty_deg;

    /**
     * Whether the GNSS represented by the measurement was used for computing the most recent fix.
     * If the data is available, 'flags' must contain RTKSDK_GNSS_MEASUREMENT_HAS_USED_IN_FIX.
     */
    unsigned char   used_in_fix;


    /**
     * 3-element array containing groups of paramters, each of which is specific to a GNSS
     * frequency point. The validity of each group is masked by bit field in
     * gnss_frequency_mask.
     */
    QZWZGnssChannelMeas channel_meas[RTKSDK_MAX_FREQ_NUM];

    int32_t reserved1[4];

    double  reserved2[4];
} RTKSDKGnssMeasurement;

/**
 * Represents a reading of GNSS measurements. For devices where GnssSystemInfo's
 * year_of_hw is set to 2016+, it is mandatory that these be provided, on
 * request, when the GNSS receiver is searching/tracking signals.
 *
 * - Reporting of GPS constellation measurements is mandatory.
 * - Reporting of all tracked constellations are encouraged.
 */
typedef struct {
    /** set to sizeof(RTKSDKGnssData) */
    size_t size;

    /** Number of measurements. */
    size_t measurement_count;

    /** The array of measurements. */
    RTKSDKGnssMeasurement measurements[RTKSDK_GNSS_MAX_MEASUREMENT];

    /** The GPS clock time reading. */
    RTKSDKGnssClock clock;
} RTKSDKGnssData;

/** Represents a GPS navigation message (or a fragment of it). */
typedef struct {
    /** set to sizeof(RTKGnssNavigationMessage) */
    size_t size;

    /**
     * Pseudo-random number for the SV, or FCN/OSN number for Glonass. The
     * distinction is made by looking at constellation field. Values should be
     * in the range of:
     *
     * - GPS:     1-32
     * - SBAS:    120-151, 183-192
     * - GLONASS: 1-24, the orbital slot number (OSN), if known.  Or, if not:
     *            93-106, the frequency channel number (FCN) (-7 to +6) offset by + 100
     *            i.e. report an FCN of -7 as 93, FCN of 0 as 100, and FCN of +6 as 106.
     * - QZSS:    193-200
     * - Galileo: 1-36
     * - Beidou:  1-37
     */
    int16_t svid;

    /**
     * The type of message contained in the structure.
     * This is a Mandatory value.
     */
    RTKSDKGnssNavigationMessageType type;

    /**
     * The status of the received navigation message.
     * No need to send any navigation message that contains words with parity error and cannot be
     * corrected.
     */
    RTKSDKNavigationMessageStatus status;

    /**
     * Message identifier.
     * It provides an index so the complete Navigation Message can be assembled. i.e. fo L1 C/A
     * subframe 4 and 5, this value corresponds to the 'frame id' of the navigation message.
     * Subframe 1, 2, 3 does not contain a 'frame id' and this value can be set to -1.
     */
    int16_t message_id;

    /**
     * Sub-message identifier.
     * If required by the message 'type', this value contains a sub-index within the current
     * message (or frame) that is being transmitted.
     * i.e. for L1 C/A the submessage id corresponds to the sub-frame id of the navigation message.
     */
    int16_t submessage_id;

    /**
     * The length of the data (in bytes) contained in the current message.
     * If this value is different from zero, 'data' must point to an array of the same size.
     * e.g. for L1 C/A the size of the sub-frame will be 40 bytes (10 words, 30 bits/word).
     *
     * This is a Mandatory value.
     */
    size_t data_length;

    /**
     * The data of the reported GPS message.
     * The bytes (or words) specified using big endian format (MSB first).
     *
     * For L1 C/A, each subframe contains 10 30-bit GPS words. Each GPS word (30 bits) should be
     * fitted into the last 30 bits in a 4-byte word (skip B31 and B32), with MSB first.
     */
    uint8_t data[64];

} RTKSDKGnssNavigationMessage;

/**
 * Represents GPS Navigation model info
 * Refer to ICD-200 for the description of each element.
 */
typedef struct {
    uint8_t     codeL2;            /* Code on L2 flag                        */
    uint8_t     URA;               /* SF1 raw accuracy factor                */
    uint8_t     SV_health;         /* SV health byte                         */
    uint8_t     L2Pdata;           /* L2-P data flag                         */
    uint8_t     fit_interval;      /* As in ICD-200                          */
    uint8_t     IODE;              /* As in ICD-200                          */
    uint16_t    weeknum;           /* GPS week number                        */
    uint16_t    IODC;              /* IODC -- 10 LSBs                        */
    float       SVacc;             /* SV accuracy; meters                    */
    float       T_GD;              /* Group delay time factor; seconds       */
    float       t_oc;              /* Time of this block's applicability;    */
    float       a_f2;              /* SV clock coef2; sec/sec^2              */
    float       a_f1;              /* SV clock coef1; sec/sec                */
    float       a_f0;              /* SV clock coef0; sec                    */
    float       C_uc;              /* Radians                                */
    float       C_us;              /* Radians                                */
    float       t_oe;              /* Seconds                                */
    float       C_ic;              /* Radians                                */
    float       C_is;              /* Radians                                */
    float       C_rc;              /* Meters                                 */
    float       C_rs;              /* Meters                                 */
    double      delta_n;           /* Radians/sec                            */
    double      M_0;               /* Radians                                */
    double      e;                 /* Dimensionless                          */
    double      sqrt_A;            /* Meters**-1/2                           */
    double      OMEGA_0;           /* Radians                                */
    double      i_0;               /* Radians                                */
    double      omega;             /* Radians                                */
    double      OMEGADOT;          /* Radians                                */
    double      IDOT;              /* Radians                                */
} RTKSDKGpsNaviModel;


/**
 * Represents Glonass Navigation model info
 *
 */
typedef struct {
    uint8_t     FT;                /*User Range Accuracy index.  P32 ICD Glonass for value of Ft.*/
    uint8_t     M;                 /*Glonass vehicle type. M=1 means type M*/
    uint8_t     Bn;                /*Bn SV health see p30 ICD Glonass. */
    uint8_t     utc_offset;        /*Current GPS-UTC leap seconds [sec]; 0 if unknown. */
    uint8_t     En;                /*Age of current information in days */
    uint8_t     P1;                /*Time interval between adjacent values of tb in minutes */
    uint8_t     P2;                /*1 if tb is odd and 0 if tb is even */
    double      deltaTau;          /*time difference between transmission in G2 and G1*/
    double      gamma;             /*SV clock frequency error ratio  */
    double      freqno;            /*Freq slot: -7 to +13 incl. */
    double      lsx;               /* x luni solar accel  */
    double      lsy;               /* y luni solar accel  */
    double      lsz;               /*z luni solar accel  */
    double      tauN;              /*SV clock bias scale factor 2^-30 [seconds]. */
    double      gloSec;            /*gloSec=[(N4-1)*1461 + (NT-1)]*86400 + tb*900 */
    double      x;                 /*x position at toe  */
    double      y;                 /*y position at toe  */
    double      z;                 /*z position at toe */
    double      vx;                /* x velocity at toe  */
    double      vy;                /* y velocity at toe  */
    double      vz;                /*z velocity at toe  */
    double      tauGPS;
} RTKSDKGlonassNaviModel;


/**
 * Represents Beidou Navigation model info
 *
 */
typedef struct {
    uint8_t     AODE;              /*Age of Data, Ephemeris*/
    uint8_t     AODC;              /*Age of Data, Clock (AODC)*/
    uint8_t     URAI;              /*URA Index, URA is used to describe the signal-in-space accuracy in meters*/
    uint32_t    weeknum;           /* BDS week number */
    float       t_oc;              /* Time of this block's applicability;    */
    float       t_oe;              /* Ephemeris reference time  */
    float       C_uc;              /* Radians                                */
    float       C_us;              /* Radians                                */
    float       C_ic;              /* Radians                                */
    float       C_is;              /* Radians                                */
    float       C_rc;              /* Meters                                 */
    float       C_rs;              /* Meters                                 */
    double      sqrt_A;            /* Meters**-1/2                           */
    double      e;                 /* Dimensionless                          */
    double      omega;             /* Radians                                */
    double      delta_n;           /* Radians/sec                            */
    double      M_0;               /* Radians                                */
    double      OMEGA_0;           /* Radians                                */
    double      OMEGADOT;          /* Radians                                */
    double      i_0;               /* Radians                                */
    double      IDOT;              /* Radians                                */
    double      a_0;               /*Clock correction polynomial coefficient(seconds) */
    double      a_1;               /*Clock correction polynomial coefficient (sec/sec) */
    double      a_2;               /*Clock correction polynomial coefficient (sec/sec2) */
    double      T_GD_1;            /*Equipment group delay differential TGD1 */
    double      T_GD_2;            /*Equipment group delay differential TGD2 */
    uint8_t     SV_health;         /*health status of this Ephemeris*/
} RTKSDKBdsNaviModel;

/** Represents a GNSS navigation model message. */
typedef struct {
    /** set to sizeof(GnssNavigationModelMessage) */
    size_t size;

    /**
     * Pseudo-random number for the SV, or FCN/OSN number for Glonass. The
     * distinction is made by looking at constellation field. Values should be
     * in the range of:
     *
     * - GPS:     1-32
     * - GLONASS: 1-24, the orbital slot number (OSN), if known.  Or, if not:
     *            93-106, the frequency channel number (FCN) (-7 to +6) offset by + 100
     *            i.e. report an FCN of -7 as 93, FCN of 0 as 100, and FCN of +6 as 106.
     * - Beidou:  1-37
     */
    int16_t svid;

    /**
     * Defines the constellation of the given SV. Value should be one of those
     * GNSS_CONSTELLATION_* constants
     */
    RTKSDKGnssConstellationType constellation;
    union {
        RTKSDKGpsNaviModel       gps_eph;
        RTKSDKGlonassNaviModel   gln_eph;
        RTKSDKBdsNaviModel       bds_eph;
    } choice;

} RTKSDKGnssNavigationModelMessage;


typedef struct {
    /** set to sizeof(GnssNavigationModelData) */
    size_t size;

    /** Number of satellite navigation models. */
    size_t navigation_model_count;

    /** The array of navigation models. */
    RTKSDKGnssNavigationModelMessage navigation_models[RTKSDK_GNSS_MAX_MEASUREMENT];
} RTKSDKGnssNavigationModelData;


typedef struct {
    RTKSDKGnssConstellationType  constellation;
    float                  alfa0;
    float                  alfa1;
    float                  alfa2;
    float                  alfa3;
    float                  beta0;
    float                  beta1;
    float                  beta2;
    float                  beta3;
} RTKSDKGnssIonosphericModelMessage;

/**
 * Represent SDK callbacks.
 */
typedef struct {
    /**
     * Set to sizeof(RTKSDKSdkCallbacks).
     */
    size_t  size;

    /**
     * Callback utility for acquiring the system wakelock. It needs to be
     * passed in if the application layer does not manage the wakelock.
     * 1 for acquiring the system wakelock; 0 for releasing the system wakelock.
     */
    void    (*acquire_wakeLock)(int flag);
    /*
     * Notify sdk integrator to push the raw data cached in buf back to
     * QianXun rubik module for controlling.
     */
    void    (*fill_raw_data)(unsigned char *buf, int len);
    /*
     * Callback to report positioning results.
     */
    void    (*fill_position)(RTKSDKGnssLocation *pos);
    /*
     * Callback to report NMEA.
     */
    void    (*fill_nmea_info)(RTKGnssUtcTime time, const char* nmea, int len);
    /*
     * Callback to report sdk status code.
     */
    void    (*status_response)(RTKSDKSdkStatus status);
} RTKSDKSdkCallbacks;

/**
 * Represent the SDK interface.
 */
typedef struct {
    /**
     * Set to sizeof(RTKSDKSdkInterface).
     */
    size_t       size;
    /**
     * Initialize the SDK, to provide sdk callbacks and configuration.
     */
    int          (*init)(RTKSDKSdkCallbacks *callbacks, RTKSDKSdkConfig *config);
    /**
     * Start the sdk.
     */
    int          (*start)(void);
    /**
     * Stop the sdk.
     */
    int          (*stop)(void);
    /**
     * Cleanup the sdk.
     */
    int          (*cleanup)(void);
    /**
     * Get sdk version.
     */
    char*        (*get_ver_info)(void);
    /**
     * Algo control.
     */
    int          (*ioctl)(int command, void* params, int* paramsLen);
    /**
     * Notify sdk when network status changes.
     */
    void         (*update_conn_status)(RTKSDKNetworkStatus networkType);
    /**
     * Configure positioning mode. RTKSDKPosMode defines the positioning mode, including rtk
     * and rtk + dr mode. frequency presents the time gap between position fix reports.
     * It will take effect immediately.
     */
    void         (*set_position_mode)(RTKSDKPosMode mode, RTKSDKReportFreq frequency);
    /**
     * Delete specific types of aiding data.
     */
    void         (*delete_aiding_data)(RTKGNSSAidingData flags);

    /**
     * Inject ephemeris of GPS/GLONASS/Beidou, etc.
     */
    int          (*inject_eph)(RTKSDKGnssNavigationModelData *data);
    /**
     * Inject measurements
     */
    int          (*inject_meas)(RTKSDKGnssData *data);
    /**
     * Inject raw navigation message
     */
    int          (*inject_navMsg)(RTKSDKGnssNavigationMessage *message);
    /**
     * Inject inospheric model data
     */
    int          (*inject_iono)(RTKSDKGnssIonosphericModelMessage *message);

    /**
     * Injects current location from another location provider
     */
    int          (*inject_position)(RTKSDKGnssLocation *pos);

    /**
     * Inject gnss data which must include RXM-RAWX(Multi-GNSS Raw Measurement Data).
     */
    int          (*inject_gnss_data)(unsigned char *buffer, size_t len);
    /**
     * Inject pulse per second which is acquired at the time of GNSS positioned.
     */
    int          (*inject_gnss_pps)(RTKSDKGNSSPPS *pps);
    /**
     * Inject dual antenna GNSS orientation.
     */
    int          (*inject_gnss_ort)(RTKSDKGNSSOrt *ort);
    /**
     * Inject sensor configuration.
     */
    int          (*inject_sensor_config)(RTKSDKSensorConfig *config);
    /**
     * Inject senor data.
     */
    int          (*inject_sensor_data)(RTKSDKSensorData *sensorData);
    /**
     * Use third-party offline map matching. Insert external map match
     * to SDK instead of using the offline map inside the SDK.
     */
    int          (*inject_map_matching_para)(RTKSDKMapMatchingPara *mapParams);
    /**
     * Set lever arm configurations.
     */
    int          (*set_lever_arm_para)(RTKSDKLeverArmParams *lam);
} RTKSDKSdkInterface;
typedef struct {
    RTKSDKSdkCallbacks *callbacks;
    RTKSDKSdkConfig *config;
}RTKSdkInit;
typedef struct {
    int command;
    void* params;
    int* paramsLen;
}RTKSdkIoctl;
typedef struct {
    RTKSDKPosMode mode;
    RTKSDKReportFreq frequency;
}RTKSdkSetPositionMode;
typedef struct {
    unsigned char *buffer;
    size_t len;
}RTKSdkInjectGnssData;
typedef struct {
    RTKGnssUtcTime time;
    const char* nmea;
    int len;
}RTKSdkCbFillNmeaInfo;

UINT8 mnld_rtk_interface(MNLDRtkInterfaceType rtk_message_type, void* rtk_message);
unsigned char  MNLD_RTK_loading(void);
unsigned char  MNLD_RTK_unloading();
int mnld_rtk_init();
int mnld_rtk_cleanup();
int mnld_rtk_start();
int mnld_rtk_stop();

/**
 * Get the SDK interfaces.

 * @param void
 * @return a pointer of RTKSDKSdkInterface type that contains the SDK's interfaces
 */
typedef const RTKSDKSdkInterface* (*GetRTKSDKSdkInterface)(void);

#ifdef __cplusplus
}
#endif
#endif /* _RTKSDK_SDK_H_ */
