#ifndef __GEOFENCE_H__
#define __GEOFENCE_H__

#pragma pack(push)
#pragma pack(4)

typedef struct {
    bool geofence_support;
}mtk_geofence_client_capability;

typedef struct {
    bool geofence_support;
}mtk_geofence_server_capability;

typedef enum {
    GEOFENCE_CLIENT_CAP,//geofence client capability
    INIT_GEOFENCE,//Reserved
    ADD_GEOFENCE_AREA,
    PAUSE_GEOFENCE,//Reserved
    RESUME_GEOFENCE,//Reserved
    REMOVE_GEOFENCE,
    RECOVER_GEOFENCE,//Reserved
    CLEAR_GEOFENCE,
    QUERY_GEOFENCE_NUM
} mtk_geofence_command;

typedef enum {
    GEOFENCE_UNKNOWN = 0,
    GEOFENCE_ENTERED = 1,
    GEOFENCE_EXITED = 2,
} mtk_geofence_status;

typedef int mtk_monitor_transition;
#define MTK_GEOFENCE_ENTER       0x01
#define MTK_GEOFENCE_EXIT        0x02
#define MTK_GEOFENCE_UNCERTAIN   0x04//Reserved

typedef struct mtk_geofence_area {
    double latitude;
    double longitude;
    double radius;
    int latest_state; /*current state(mtk_geofence_status), most cases is GEOFENCE_UNKNOWN*/
    mtk_monitor_transition monitor_transition; /*bitwise , MTK_GEOFENCE_EXIT/MTK_GEOFENCE_ENTER/MTK_GEOFENCE_UNCERTAIN*/
    int unknown_timer;/*The time limit after which the UNCERTAIN transition should be triggered. This paramter is defined in milliseconds.*/
} mtk_geofence_property;

typedef struct mtk_geofence_alert {
    int id;
    int alert_state;//mtk_geofence_status
    double latitude;
    double longitude;
    double altitude;
    double speed;
    double heading;
    int h_acc;
    int h_err_majoraxis;
    int h_err_minoraxis;
    int h_err_angle;
    int hor_conf;
    double pdop;
    double hdop;
    double vdop;
}mtk_geofence_alert;

typedef struct mtk_gnss_tracking_status {
    int status; //GNSS service tracking OK = 0,GNSS service tracking failure  = -1
    int year;
    int month;
    int day;
    int hour;
    int minute;
    int second;
}mtk_gnss_tracking_status;

typedef struct {
    void (*geofence_connection_broken)();
    void (*geofence_fence_alert_callback)(mtk_geofence_alert *ptr);
    void (*geofence_tracking_status_callback)(mtk_gnss_tracking_status *ptr);
    void (*geofence_capability_update)(mtk_geofence_server_capability *cap);
}mtk_geofence_callback;

#define MTK_GFC_SUCCESS     (0)
#define MTK_GFC_ERROR       (-1)

#pragma pack(pop)

/*****************************************************************************
 * FUNCTION
 *  geofenceinf_add_geofence
 * DESCRIPTION
 * add one geofence
 * PARAMETERS
 *  fence: fence detail information
 * RETURNS
 *  Success: geofence id, >=1
 *  Fail: -1, create fence fail. -2, insufficient_memory. -3,too many fences
 *****************************************************************************/
int geofenceinf_add_geofence(mtk_geofence_property *fence);

/*****************************************************************************
 * FUNCTION
 *  geofenceinf_remove_geofence
 * DESCRIPTION
 * romve one geofence
 * PARAMETERS
 *  geofence_id: geofence id
 * RETURNS
 *  MTK_GFC_ERROR
 *  MTK_GFC_SUCCESS
 *****************************************************************************/
int geofenceinf_remove_geofence(const int geofence_id) ;

/*****************************************************************************
 * FUNCTION
 *  geofenceinf_clear_geofences
 * DESCRIPTION
 * clear all geofences
 * PARAMETERS
 * RETURNS
 *  MTK_GFC_ERROR
 *  MTK_GFC_SUCCESS
 *****************************************************************************/
int geofenceinf_clear_geofences(void) ;

/*****************************************************************************
 * FUNCTION
 *  geofenceinf_query_geofences_num
 * DESCRIPTION
 * query geofence numbers that geofence feature support
 * PARAMETERS
 * RETURNS
 *  MTK_GFC_ERROR(-1)
 *  number: geofence numbers, >= 1
 *****************************************************************************/
int geofenceinf_query_geofences_num(void);

/*****************************************************************************
 * FUNCTION
 *  geofenceinf_client_register
 * DESCRIPTION
 * register geofence client
 * PARAMETERS
 *  name: client name
 *  mtk_geofence_callback *callback: data messsage callback function
 * RETURNS
 *  MTK_GFC_ERROR(-1), means error
 *  fd:file descriptor
 *****************************************************************************/
int geofenceinf_client_register(const char* name, mtk_geofence_callback *callback);

/*****************************************************************************
 * FUNCTION
 *  geofenceinf_client_capability_config
 * DESCRIPTION
 * config client capability, and send to geofence server
 * PARAMETERS
 *  cap: client capability
 * RETURNS
 *  MTK_GFC_ERROR
 *  MTK_GFC_SUCCESS
 *****************************************************************************/
int geofenceinf_client_capability_config(mtk_geofence_client_capability *cap);

/*****************************************************************************
 * FUNCTION
 *  mnl2geofence_hdlr
 * DESCRIPTION
 * mnl to geofence adaptor handle function
 * PARAMETERS
 *  fd:file descriptor
 *  mtk_geofence_callback *callback: data messsage callback function
 * RETURNS
 *  MTK_GFC_ERROR
 *  MTK_GFC_SUCCESS
 *****************************************************************************/
int mnl2geofence_hdlr (int fd, mtk_geofence_callback *callback);

#endif
