// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2022 MediaTek Inc.
 */
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/regmap.h>
#include <linux/mfd/mt6330/registers.h>
#include <linux/mfd/mt6330/core.h>

#include "chrdet_notifier.h"

SRCU_NOTIFIER_HEAD(chrdet_notifier_list);

#define MT6330_TOPSTATUS_CHRDET_SHIFT		2

struct mtk_pmic_chrdet_regs {
	u32 status_reg;
	u32 status_mask;
	u32 intsel_reg;
	u32 intsel_mask;
};

#define MTK_PMIC_CHRDET_REGS(_status_reg, _status_mask,	\
	_intsel_reg, _intsel_mask)			\
{							\
	.status_reg		= _status_reg,		\
	.status_mask		= _status_mask,		\
	.intsel_reg		= _intsel_reg,		\
	.intsel_mask		= _intsel_mask,		\
}

struct mtk_pmic_regs {
	const struct mtk_pmic_chrdet_regs chrdet_regs;
};

static const struct mtk_pmic_regs mt6330_regs = {
	.chrdet_regs =
		MTK_PMIC_CHRDET_REGS(MT6330_TOPSTATUS, 0x04,
		MT6330_PSC_TOP_INT_CON0, 0x20),
};

struct mtk_pmic_chrdet_info {
	struct mtk_pmic_chrdet *chrdet;
	const struct mtk_pmic_chrdet_regs *regs;
	int irq;
	bool enable;
	bool wakeup;
};

struct mtk_pmic_chrdet {
	struct device *dev;
	struct regmap *regmap;
	struct mtk_pmic_chrdet_info info;
};

int register_chrdet_notifier(struct notifier_block *nb)
{
	int ret;
	struct device *dev;
	struct platform_device *pdev;
	struct mtk_pmic_chrdet *chrdet;
	u32 top_status = 0, detected = 0;

	dev = bus_find_device_by_name(
		&platform_bus_type, NULL, "mtk-pmic-chrdet");
	if (!dev) {
		pr_err("[%s] No pmic chrdet device!\n", __func__);
		return 0;
	}
	pdev = to_platform_device(dev);
	if (!pdev) {
		pr_err("[%s] No pmic chrdet platform device!\n", __func__);
		return 0;
	}
	chrdet = platform_get_drvdata(pdev);
	if (!chrdet) {
		pr_err("[%s] No chrdet struct\n", __func__);
		return 0;
	}

	regmap_read(chrdet->regmap, chrdet->info.regs->status_reg, &top_status);
	detected = top_status & (1 << MT6330_TOPSTATUS_CHRDET_SHIFT);
	pr_info("[%s] chrdet = %s using PMIC\n", __func__,
		 detected ? "detected" : "undetected");

	ret = srcu_notifier_chain_register(&chrdet_notifier_list, nb);

	srcu_notifier_call_chain(&chrdet_notifier_list, detected, NULL);
	return ret;
}
EXPORT_SYMBOL(register_chrdet_notifier);

int unregister_chrdet_notifier(struct notifier_block *nb)
{
	return srcu_notifier_chain_unregister(&chrdet_notifier_list, nb);
}
EXPORT_SYMBOL(unregister_chrdet_notifier);

static irqreturn_t mtk_pmic_chrdet_irq_handler_thread(int irq, void *data)
{
	struct mtk_pmic_chrdet_info *info = data;
	u32 top_status = 0, detected = 0;

	regmap_read(info->chrdet->regmap, info->regs->status_reg, &top_status);

	detected = top_status & (1 << MT6330_TOPSTATUS_CHRDET_SHIFT);

	pr_info("chrdet = %s using PMIC\n",
		 detected ? "detected" : "undetected");

	srcu_notifier_call_chain(&chrdet_notifier_list, detected, NULL);

	return IRQ_HANDLED;
}

static int mtk_pmic_chrdet_setup(struct mtk_pmic_chrdet *chrdet,
		struct mtk_pmic_chrdet_info *info)
{
	int ret;

	info->chrdet = chrdet;

	ret = regmap_update_bits(chrdet->regmap, info->regs->intsel_reg,
				 info->regs->intsel_mask,
				 info->regs->intsel_mask);
	if (ret < 0)
		return ret;

	ret = devm_request_threaded_irq(chrdet->dev, info->irq, NULL,
					mtk_pmic_chrdet_irq_handler_thread,
					IRQF_ONESHOT | IRQF_TRIGGER_HIGH,
					"mtk-pmic-chrdet", info);
	if (ret) {
		pr_err("Failed to request IRQ: %d: %d\n",
			info->irq, ret);
		return ret;
	}

	return 0;
}

static int __maybe_unused mtk_pmic_chrdet_suspend(struct device *dev)
{
	struct mtk_pmic_chrdet *chrdet = dev_get_drvdata(dev);

	if (chrdet->info.wakeup)
		enable_irq_wake(chrdet->info.irq);

	return 0;
}

static int __maybe_unused mtk_pmic_chrdet_resume(struct device *dev)
{
	struct mtk_pmic_chrdet *chrdet = dev_get_drvdata(dev);

	if (chrdet->info.wakeup)
		disable_irq_wake(chrdet->info.irq);

	return 0;
}

static SIMPLE_DEV_PM_OPS(mtk_pmic_chrdet_pm_ops, mtk_pmic_chrdet_suspend,
			mtk_pmic_chrdet_resume);

static const struct of_device_id of_mtk_pmic_chrdet_match_tbl[] = {
	{
		.compatible = "mediatek,mt6330-chrdet",
		.data = &mt6330_regs,
	}, {
		/* sentinel */
	}
};
MODULE_DEVICE_TABLE(of, of_mtk_pmic_chrdet_match_tbl);

static int mtk_pmic_chrdet_probe(struct platform_device *pdev)
{
	int error, index = 0;
	struct mt6330_chip *pmic_chip = dev_get_drvdata(pdev->dev.parent);
	struct device_node *node = pdev->dev.of_node, *child;
	struct mtk_pmic_chrdet *chrdet;
	const struct mtk_pmic_regs *mtk_pmic_regs;
	const struct of_device_id *of_id =
		of_match_device(of_mtk_pmic_chrdet_match_tbl, &pdev->dev);

	pr_info("%s\n", __func__);

	if (!of_id)
		return -ENOMEM;

	chrdet = devm_kzalloc(&pdev->dev, sizeof(*chrdet), GFP_KERNEL);
	if (!chrdet)
		return -ENOMEM;

	chrdet->dev = &pdev->dev;
	chrdet->regmap = pmic_chip->regmap;
	mtk_pmic_regs = of_id->data;

	for_each_child_of_node(node, child) {
		chrdet->info.regs = &mtk_pmic_regs->chrdet_regs;

		chrdet->info.irq = platform_get_irq(pdev, index);
		if (chrdet->info.irq < 0)
			return chrdet->info.irq;

		if (of_property_read_bool(child, "enable")) {
			pr_info("chrdet enabled\n");
			chrdet->info.enable = true;

			if (of_property_read_bool(child, "wakeup-source"))
				chrdet->info.wakeup = true;

			error = mtk_pmic_chrdet_setup(chrdet, &chrdet->info);
			if (error) {
				pr_err("Set chrdet error(%d).\n", error);
				return error;
			}
		} else {
			pr_info("chrdet disabled\n");
		}
	}

	platform_set_drvdata(pdev, chrdet);

	return 0;
}

static struct platform_driver pmic_chrdet_pdrv = {
	.probe = mtk_pmic_chrdet_probe,
	.driver = {
		   .name = "mtk-pmic-chrdet",
		   .of_match_table = of_mtk_pmic_chrdet_match_tbl,
		   .pm = &mtk_pmic_chrdet_pm_ops,
	},
};

module_platform_driver(pmic_chrdet_pdrv);

MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Jericho Lee <jericho.lee@mediatek.com>");
MODULE_DESCRIPTION("MTK pmic-chrdet driver v0.1");
