// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2020 MediaTek Inc.
 */

#include <linux/cdev.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/iopoll.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/slab.h>
#include <linux/uaccess.h>

#define IOCTL_DEV_IOCTLID	'P'
#define PCIE_SMT_TEST_SLOT      _IOW(IOCTL_DEV_IOCTLID, 0, int)
#define PCIE_SMT_DEV_NAME	"pcie_smt"

struct mtk_pcie_smt {
	char *name;
	void __iomem *sif_base;
	int err_count;
	/* char device related */
	struct cdev smt_cdev;
	struct class *f_class;
};

static int mtk_pcie_test_open(struct inode *inode, struct file *file)
{
	struct mtk_pcie_smt *pcie_smt;

	pcie_smt = container_of(inode->i_cdev, struct mtk_pcie_smt, smt_cdev);
	file->private_data = pcie_smt;
	pr_info("mtk_pcie_smt open: successful\n");

	return 0;
}

static int mtk_pcie_test_release(struct inode *inode, struct file *file)
{

	pr_info("mtk_pcie_smt release: successful\n");
	return 0;
}

static ssize_t mtk_pcie_test_read(struct file *file, char *buf,
		size_t count, loff_t *ptr)
{

	pr_info("mtk_pcie_smt read: returning zero bytes\n");
	return 0;
}

static ssize_t mtk_pcie_test_write(struct file *file, const char *buf,
		size_t count, loff_t *ppos)
{

	pr_info("mtk_pcie_smt write: accepting zero bytes\n");
	return 0;
}

/* Loopback test for PCIe port
 * Port 0 -> 2 lane
 */
static int pcie_loopback_test(void __iomem *phy_base)
{
	int val = 0, ret = 0;
	int err_count = 0;

	pr_info("pcie loopback test start\n");

	if (!phy_base) {
		pr_info("phy_base is not initialed!\n");
		return -1;
	}

	/* L1ss = enable */
	val = readl(phy_base + 0x28);
	val |= (0x01 << 5);
	writel(val, phy_base + 0x28);

	val = readl(phy_base + 0x28);
	val |= (0x01 << 4);
	writel(val, phy_base + 0x28);

	val = readl(phy_base + 0x28);
	val &= (~0x200);
	writel(val, phy_base + 0x28);

	val = readl(phy_base + 0x28);
	val |= (0x01 << 8);
	writel(val, phy_base + 0x28);

	val = readl(phy_base + 0x28);
	val &= (~0x800);
	writel(val, phy_base + 0x28);

	val = readl(phy_base + 0x28);
	val |= (0x01 << 10);
	writel(val, phy_base + 0x28);

	/* Set Rate=Gen1 */
	usleep_range(1, 2);
	val = readl(phy_base + 0x70);
	val |= (0x01);
	writel(val, phy_base + 0x70);

	val = readl(phy_base + 0x70);
	val &= (~0x30000);
	writel(val, phy_base + 0x70);

	val = readl(phy_base + 0x70);
	val |= (0x01 << 4);
	writel(val, phy_base + 0x70);

	usleep_range(1, 2);
	val = readl(phy_base + 0x70);
	val &= (~0x10);
	writel(val, phy_base + 0x70);

	/* Force PIPE (P0) */
	val = readl(phy_base + 0x70);
	val |= (0x01);
	writel(val, phy_base + 0x70);

	val = readl(phy_base + 0x70);
	val &= (~0xc00);
	writel(val, phy_base + 0x70);

	val = readl(phy_base + 0x70);
	val &= (~0x3000);
	writel(val, phy_base + 0x70);

	val = readl(phy_base + 0x70);
	val |= (0x01 << 8);
	writel(val, phy_base + 0x70);

	val = readl(phy_base + 0x70);
	val |= (0x01 << 4);
	writel(val, phy_base + 0x70);

	usleep_range(1, 2);
	val = readl(phy_base + 0x70);
	val &= (~0x10);
	writel(val, phy_base + 0x70);

	/* Set TX output Pattern */
	usleep_range(1, 2);
	val = readl(phy_base + 0x4010);
	val |= (0x0d);
	writel(val, phy_base + 0x4010);


	/* Set TX PTG Enable */
	val = readl(phy_base + 0x28);
	val |= (0x01 << 30);
	writel(val, phy_base + 0x28);

	/* Set RX Pattern Checker (Type & Enable) */
	val = readl(phy_base + 0x501c);
	val |= (0x01 << 1);
	writel(val, phy_base + 0x501c);

	val = readl(phy_base + 0x501c);
	val |= (0x0d << 4);
	writel(val, phy_base + 0x501c);

	/* toggle ptc_en for status counter clear */
	val = readl(phy_base + 0x501c);
	val &= (~0x2);
	writel(val, phy_base + 0x501c);

	val = readl(phy_base + 0x501c);
	val |= (0x01 << 1);
	writel(val, phy_base + 0x501c);

	/* Check status */
	msleep(50);
	val = readl(phy_base + 0x50c8);
	if ((val & 0x3) != 0x3) {
		err_count = val >> 12;
		pr_info("PCIe test failed: %#x!\n", val);
		pr_info("lane0 error count: %d\n", err_count);
		ret = -1;
	}

	return ret;
}

static int pcie_smt_test_slot(struct mtk_pcie_smt *pcie_smt)
{
	if (!pcie_smt) {
		pr_info("pcie_smt not found\n");
		return -ENODEV;
	}

	return pcie_loopback_test(pcie_smt->sif_base);
}

static long mtk_pcie_test_ioctl(struct file *file, unsigned int cmd,
				unsigned long arg)
{
	struct mtk_pcie_smt *pcie_smt = file->private_data;

	switch (cmd) {
	case PCIE_SMT_TEST_SLOT:
		pr_info("pcie_smt slot: %d\r\n", (unsigned int)arg);

		return pcie_smt_test_slot(pcie_smt);
	default:
		return -ENOTTY;
	}

	return 0;
}

static const struct file_operations pcie_smt_fops = {
	.owner = THIS_MODULE,
	.read = mtk_pcie_test_read,
	.write = mtk_pcie_test_write,
	.unlocked_ioctl = mtk_pcie_test_ioctl,
	.compat_ioctl = mtk_pcie_test_ioctl,
	.open = mtk_pcie_test_open,
	.release = mtk_pcie_test_release,
};

static int mtk_pcie_smt_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	dev_t dev_r;
	int ret;
	struct cdev *dev_ctx;
	struct resource *reg_res;
	struct mtk_pcie_smt *pcie_smt;

	pcie_smt = kzalloc(sizeof(*pcie_smt), GFP_KERNEL);
	if (!pcie_smt) {
		dev_warn(dev, "pcie_smt alloc failed\n");
		return -ENOMEM;
	}

	reg_res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "phy-sif");
	pcie_smt->sif_base = devm_ioremap_resource(dev, reg_res);
	if (IS_ERR(pcie_smt->sif_base)) {
		dev_warn(dev, "failed to map pcie phy base: sif\n");
		ret = PTR_ERR(pcie_smt->sif_base);
		goto release;
	}

	/* char device related */
	pcie_smt->name = PCIE_SMT_DEV_NAME;
	dev_ctx = &pcie_smt->smt_cdev;

	ret = alloc_chrdev_region(&dev_r, 0, 1, pcie_smt->name);
	if (ret) {
		dev_warn(dev, "pcie_smt_init alloc_chrdev_region failed -\n");
		goto release;
	}

	cdev_init(dev_ctx, &pcie_smt_fops);
	dev_ctx->owner = THIS_MODULE;

	ret = cdev_add(dev_ctx, dev_r, 1);
	if (ret) {
		dev_warn(dev, "failed to add cdev\n");
		goto release_cdev;
	}

	pcie_smt->f_class = class_create(THIS_MODULE, pcie_smt->name);
	device_create(pcie_smt->f_class, NULL, dev_ctx->dev, NULL, pcie_smt->name);
	platform_set_drvdata(pdev, pcie_smt);

	pm_runtime_enable(dev);
	pm_runtime_get_sync(dev);

	return 0;

release_cdev:
	unregister_chrdev_region(dev_r, MINORMASK);
release:
	kfree(pcie_smt);

	return ret;
}

static int mtk_pcie_smt_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct mtk_pcie_smt *pcie_smt = platform_get_drvdata(pdev);
	struct cdev *dev_ctx = &pcie_smt->smt_cdev;
	struct class *f_class = pcie_smt->f_class;

	/* remove char device */
	device_destroy(f_class, dev_ctx->dev);
	class_destroy(f_class);
	unregister_chrdev_region(dev_ctx->dev, MINORMASK);

	kfree(pcie_smt);

	pm_runtime_put_sync(dev);
	pm_runtime_disable(dev);

	return 0;
}

static const struct of_device_id mtk_pcie_smt_id_table[] = {
	{ .compatible = "mediatek,mt2735-pcie-phy" },
	{ },
};

static struct platform_driver mtk_pcie_smt_driver = {
	.probe = mtk_pcie_smt_probe,
	.remove = mtk_pcie_smt_remove,
	.driver = {
		.name = "mtk-pcie-smt",
		.owner = THIS_MODULE,
		.of_match_table = mtk_pcie_smt_id_table,
	},
};

module_platform_driver(mtk_pcie_smt_driver);
