//SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2017 MediaTek Inc.
 */

#include "nandx_util.h"
#include "nandx_core.h"
#include "../nand_device.h"
#include "device_slc.h"

/* onfi nand timing mode */
static struct slc_timing_mode timing_mode[] = {
	{
		.addr = 0x01,
		.mode = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05}
	},
};

/* tREA, tREH, tCR, tRP, tWP, tWH, tWHR, tCLS, tALS, tCLH,tALH, tWC, tRC */
static struct nand_sdr_timing sdr_timing[SLC_TIMING_NUM] = {
	{40, 30, 0, 50, 50, 30, 120, 50, 50, 20, 20, 100, 100},
	{30, 15, 0, 25, 25, 15,  80, 25, 25, 10, 10,  45,  50},
	{25, 15, 0, 20, 17, 15,  80, 15, 15, 10, 10,  35,  35},
	{20, 10, 0, 15, 15, 10,  60, 10, 10,  5,  5,  30,  30},
	{20, 10, 0, 12, 12, 10,  60, 12, 12,  5,  5,  25,  25},
	{16,  7, 0, 10, 10,  7,  60, 10, 10,  5,  5,  20,  20},
};

/* Not all SLC devices entirely match ONFI-1.0 spec. So customize them here.
 * MT29F8G08ABBCAH4 & MT29F4G08ABBFAH4 use the following settings
 * and support timing mode 0~3
 */
static struct nand_sdr_timing sdr_timing_micron_slc[SLC_TIMING_NUM] = {
	{40, 30, 0, 50, 50, 30, 120, 50, 50, 20, 20, 100, 100},
	{30, 15, 0, 25, 25, 15,  80, 25, 25, 10, 10,  45,  50},
	{25, 15, 0, 20, 17, 15,  80, 15, 15, 10, 10,  35,  35},
	{25, 10, 0, 15, 15, 10,  80, 10, 10,  5,  5,  30,  30},
};

static struct nand_sdr_timing sdr_timing_winbond_slc[SLC_TIMING_NUM] = {
	{40, 30, 0, 50, 50, 30, 120, 50, 50, 20, 20, 100, 100},
	{30, 15, 0, 25, 25, 15,  80, 25, 25, 10, 10,  45,  50},
	{25, 15, 0, 20, 17, 15,  80, 15, 15, 10, 10,  35,  35},
};

/* tREA, tREH, tCR, tRP, tWP, tWH, tWHR, tCLS, tALS, tCLH,tALH, tWC, tRC */
static struct nand_sdr_timing sdr_timing_hynix_slc[] = {
	{30, 15, 10, 25, 25, 15,  60, 25, 25, 10, 10,  45,  45},
};

/* tREA, tREH, tCR, tRP, tWP, tWH, tWHR, tCLS, tALS, tCLH,tALH, tWC, tRC */
static struct nand_sdr_timing sdr_timing_NM4888KMPAXAI[] = {
	{22, 10, 0, 12, 12, 10,  80, 10, 10, 5, 5, 25, 25},
};

/* onfi nand basic commands */
static struct nand_cmds onfi_cmds = {
	.reset = 0xff,
	.read_id = 0x90,
	.read_status = 0x70,
	.read_param_page = 0xec,
	.set_feature = 0xef,
	.get_feature = 0xee,
	.read_1st = 0x00,
	.read_2nd = 0x30,
	.random_out_1st = 0x05,
	.random_out_2nd = 0xe0,
	.program_1st = 0x80,
	.program_2nd = 0x10,
	.erase_1st = 0x60,
	.erase_2nd = 0xd0,
	.read_cache = 0x31,
	.read_cache_last = 0x3f,
	.program_cache = 0x15,
};

/* onfi nand extend commands */
static struct slc_extend_cmds onfi_extend_cmds = {
	.read_multi_1st = 0x00,
	.read_multi_2nd = 0x32,
	.program_multi_1st = 0x80,
	.program_multi_2nd = 0x11,
	.erase_multi_1st = 0x60,
	.erase_multi_2nd = 0xd1,
	.read_status_enhanced = 0x78
};

/* toggle nand basic commands */
static struct nand_cmds toggle_cmds = {
	.reset = 0xff,
	.read_id = 0x90,
	.read_status = 0x70,
	.read_param_page = 0x03,
	.set_feature = 0xef,
	.get_feature = 0xee,
	.read_1st = 0x00,
	.read_2nd = 0x30,
	.random_out_1st = 0x05,
	.random_out_2nd = 0xe0,
	.program_1st = 0x80,
	.program_2nd = 0x10,
	.erase_1st = 0x60,
	.erase_2nd = 0xd0,
	.read_cache = 0x31,
	.read_cache_last = 0x3f,
	.program_cache = 0x15,
};

/* toggle nand extend commands */
static struct slc_extend_cmds toggle_extend_cmds = {
	.read_multi_1st = 0x60,
	.read_multi_2nd = 0x30,
	.program_multi_1st = 0x80,
	.program_multi_2nd = 0x11,
	.erase_multi_1st = 0x60,
	.erase_multi_2nd = -1,
	.read_status_enhanced = 0xf1
};

/* means the start bit of addressing type */
static struct nand_addressing slc_addressing = {
	.row_bit_start = 0,
	.block_bit_start = 6,
	.plane_bit_start = 6,
	.lun_bit_start = 18,
};

/* means the status of read status value by bit location */
static struct nand_status slc_status = {
	.array_busy = BIT(5),
	.write_protect = BIT(7),
	.erase_fail = BIT(0),
	.program_fail = BIT(0)
};

/* measure cycle by the times */
static struct nand_endurance slc_endurance = {
	.pe_cycle = 60000,
	.ecc_req = 1,
	.max_bitflips = 8
};

/*
 * measure time by the us.
 * tRST:
 * According to the datasheet of NAND MT29F8G08ABBCAH4:
 * the first time the RESET(FFh) command is issued while the device is idle,
 * the device will be busy for a maximum of 1ms.
 *
 * tR:
 * JSFDDQ5QHAxGD max is 30us
 *
 * tPROG:
 * JSFDDQ5QHAxGD max is 700us
 *
 * tPCBSY:
 * JSFDDQ5QHAxGD max is 700us
 *
 * tFEAT:
 * According to the datasheet of NAND MT29F8G08ABBCAH4:
 * After send the get feature command, R/B# need time tWB + tFEAT + tRR to be ready
 * After send the set feature command, R/B# need time tWB + tFEAT to be ready
 * tWB max is 100ns, tFEAT max is 1us, tRR min is 20ns
 * Consider sytem timer deviation, relax the tFEAT as 5us
 */
static struct nand_array_timing slc_array_timing = {
	.tRST = 1000,
	.tWHR = 1,
	.tR = 100,
	.tRCBSY = 25,
	.tFEAT = 5,
	.tPROG = 700,
	.tPCBSY = 700,
	.tBERS = 10000,
	.tDBSY = 1
};

/*
 * NAND -nand basic information
 * NAND_DEVICE
 *   _name,
 *   _id,
 *   _id_len, _io_width, _row_cycle, _col_cycle,
 *   _target_num, _lun_num, _plane_num, _block_num,
 *   _block_size, _page_size, _spare_size, _min_program_pages,
 *   _cmds, _addressing, _status,
 *   _endurance, _array_timing
 */
static struct device_slc slc_nand[] = {
	/* MCP */
	/* JSC */
	{
		NAND_DEVICE("JSFDDQ5QHAxGD-405x",
				NAND_PACK_ID(0xad, 0xa3, 0x81, 0x16, 0x20, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 1, 4096,
				KB(256), KB(4), 256, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE0,
		NULL,
		sdr_timing_hynix_slc
	},
	/* NANYA */
	{
		NAND_DEVICE("NM4484NSPAXAE-3E",
				NAND_PACK_ID(0x90, 0xac, 0x90, 0x26, 0x76, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 1, 2048,
				KB(256), KB(4), 256, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE0,
		&timing_mode[0],
		sdr_timing
	},
	{
		NAND_DEVICE("NM4888KMPAXAI",
				NAND_PACK_ID(0xc2, 0xa3, 0xd1, 0x15, 0x5a, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 1, 8192,
				KB(128), KB(2), 64, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE0,
		&timing_mode[0],
		sdr_timing_NM4888KMPAXAI
	},
	/* Parallel-NAND*/
	/* Micron */
	{
		NAND_DEVICE("MT29F4G08ABA",
				NAND_PACK_ID(0x2c, 0xdc, 0x80, 0xa6, 0x62, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 1, 4096,
				KB(256), KB(4), 256, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE0,
		&timing_mode[0],
		sdr_timing
	},
	{
		/* This device is used on:
		 * mt6880's socket board
		 * m.2 datacard
                 */
		NAND_DEVICE("MT29F4G08ABBFAH4-IT:F",
				NAND_PACK_ID(0x2c, 0xac, 0x80, 0x26, 0x62, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 1, 2048,
				KB(256), KB(4), 256, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE3,
		&timing_mode[0],
		sdr_timing_micron_slc
	},
	{
		/* This device is used on:
		 * mt6890's MP board
                 */
		NAND_DEVICE("MT29F8G08ABBCAH4-IT:C",
				NAND_PACK_ID(0x2c, 0xa3, 0x90, 0x26, 0x64, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 2, 2048,
				KB(256), KB(4), 224, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE3,
		&timing_mode[0],
		sdr_timing_micron_slc
	},
	{
		/*
		 * MT29F8G08ADBFA or
		 * MT29GZ6A6BPIET-53AIT.112 or MT29GZ6A6BPIET-53AAT.112
                 */
		NAND_DEVICE("MT29F8G08ADBFA",
				NAND_PACK_ID(0x2c, 0xa3, 0xd0, 0x26, 0x66, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 1, 4096,
				KB(256), KB(4), 256, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE2,
		&timing_mode[0],
		sdr_timing_micron_slc
	},
	/* Winbond */
	{
		NAND_DEVICE("W29N04GZBIBA",
				NAND_PACK_ID(0xef, 0xac, 0x90, 0x15, 0x54, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 1, 2, 2048,
				KB(128), KB(2), 64, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE2,
		&timing_mode[0],
		sdr_timing_winbond_slc
	},
	{
		NAND_DEVICE("W29N08GZBIBA",
				NAND_PACK_ID(0xef, 0xa3, 0x91, 0x15, 0x58, 0, 0, 0),
				5, NAND_IO8, 3, 2,
				1, 2, 2, 2048,
				KB(128), KB(2), 64, 1,
				&onfi_cmds, &slc_addressing, &slc_status,
				&slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE2,
		&timing_mode[0],
		sdr_timing_winbond_slc
	},
	/* Toshiba */
	{
		NAND_DEVICE("TC58NVG2S0HTA00",
			    NAND_PACK_ID(0x98, 0xdc, 0x90, 0x26, 0x76,
					 0x16, 0, 0),
			    6, NAND_IO8, 3, 2,
			    1, 1, 1, 2048,
			    KB(256), KB(4), 256, 1,
			    &toggle_cmds, &slc_addressing, &slc_status,
			    &slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x10, 0x04, 0x06, 0x04, 0x02),
		&toggle_extend_cmds,
		CHIP_TIMING_MODE0,
		NULL,
		sdr_timing
	},
	{
		NAND_DEVICE("MX30UF4G18AC",
			    NAND_PACK_ID(0xc2, 0xac, 0x90, 0x15, 0x56,
					 0, 0, 0),
			    5, NAND_IO8, 3, 2,
			    1, 1, 1, 4096,
			    KB(128), KB(2), 64, 1,
			    &onfi_cmds, &slc_addressing, &slc_status,
			    &slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x10, 0x04, 0x06, 0x04, 0x02),
		&onfi_extend_cmds,
		CHIP_TIMING_MODE0,
		&timing_mode[0],
		sdr_timing
	},
	{
		NAND_DEVICE("NO-DEVICE",
			    NAND_PACK_ID(0, 0, 0, 0, 0, 0, 0, 0),
			    0, 0, 0, 0,
			    0, 0, 0, 0,
			    0, 0, 0, 1,
			    &onfi_cmds, &slc_addressing, &slc_status,
			    &slc_endurance, &slc_array_timing),
		SLC_DRIVE_STRENGTH(0x80, 0x00, 0x01, 0x02, 0x03),
		NULL,
		CHIP_TIMING_MODE0,
		NULL,
		sdr_timing
	}
};

struct nand_device *nand_get_device(int index)
{
	return &slc_nand[index].dev;
}
