// SPDX-License-Identifier: GPL-2.0
/*
 * Mediatek watchdog common driver.
 *
 *
 * Copyright (C) 2020 MediaTek Inc.
 *
 * Author: Anan Sun <anan.sun@mediatek.com>
 */

#define pr_fmt(fmt)	"mtk-wdt: " fmt

#include <linux/bitops.h>
#include <linux/err.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/watchdog.h>
#include <linux/delay.h>
#include <asm/system_misc.h>
#include <linux/of_address.h>
#include <dbgtop.h>

#define WDT_MODE			(0x0000)
#define WDT_EN				BIT(0)
#define EXTPOL				BIT(1)
#define EXTEN				BIT(2)
#define WDT_IRQ				BIT(3)
#define WDT_2STAGE_TMO_EN		BIT(4)	/* 6885: HW setting */
#define WDT_MODE_AUTO_START		BIT(4)	/* 6589: reserved */
#define IRQ_LVL_EN			BIT(5)
#define DUAL_MODE			BIT(6)
#define WDT_CNT_RESET_SEL		BIT(8)
#define WDT_MODE_UNLOCK_KEY		(0x22000000)
/* 6885: clear WDT_STA by WDT_MODE */
#define WDT_STA_CLR_MASK		(0x01FF)
#define WDT_STA_CLEAR_UNLOCK_KEY	(0x23000000)

#define WDT_LENGTH			(0x0004)
#define WDT_LENGTH_UNLOCK_KEY		(0x0008)
/* ((sec * 32768) / 512 - 1) = 1 length unit */
#define WDT_LENGTH_SEC2LENGTH(sec)	(((sec) << 6) - 1)
#define WDT_TIMEOUT_MAX_SEC		(31)
#define WDT_TIMEOUT_DEFAULT_SEC		(20)
#define WDT_TIMEOUT_MIN_SEC		(1)
#define WDT_LENGTH_SHIFT		(5)
#define WDT_LENGTH_MASK			GENMASK(15, 5)

#define WDT_RESTART			(0x0008)
#define WDT_RESTART_KEY			(0x1971)

#define WDT_SWRST			(0x0014)
#define WDT_SWRST_KEY			(0x1209)

#define WDT_NONRST_REG2			(0x0024)
#define WDT_REBOOT_RPMBPK		BIT(0)
#define WDT_REBOOT_RECOVERY		BIT(1)
#define WDT_REBOOT_BOOTLOADER		BIT(2)
#define WDT_REBOOT_META			BIT(3)
#define WDT_REBOOT_DDR_RESERVE		BIT(4)
#define WDT_REBOOT_MODE_MSK		(0x1F)
#define WDT_REBOOT_STAGE_CUR_SHIFT	(28)
#define WDT_REBOOT_STAGE_MASK		(0xfUL)
#define WDT_REBOOT_STAGE_KERNEL		(0x3UL)

#define WDT_REQ_MODE			(0x0030)
#define WDT_REQ_MODE_UNLOCK_KEY		(0x33000000)
#define WDT_REQ_IRQ_EN			(0x0034)
#define WDT_REQ_IRQ_EN_UNLOCK_KEY	(0x44000000)
#define SPM_THERMAL_RST			BIT(0)
#define SPM_RST				BIT(1)
#define EINT_RST			BIT(2)
#define SYSRST_RST			BIT(3)
#define DVFSP_RST			BIT(4)
#define MCUPM_RST			BIT(5)
#define DEBUG_TOP_RST			BIT(6)
#define PCIE_PERST_B_RST		BIT(7)
#define PMCU_RSTREQ			BIT(16)	/* 2731: IOC_RST */
#define MDDBG_RST			BIT(17)
#define THERMAL_CTL_RST			BIT(18)
#define DEBUG_RST			BIT(19)

#define WDT_EXT_REQ_CON			(0x0038)
#define EINT_DGRST_EN			BIT(0)
#define EINT_SEL_MASK			(0xf)
#define EINT_SEL_SHIFT			(4)
#define EINT_SEL			GENMASK(7, 4)

#define WDT_LATCH_CTL2			(0x0048)
#define DFD_TIMEOUT_MASK		GENMASK(16, 0)
#define DFD_EN				BIT(17)
#define DFD_THERM1_DIS			BIT(18)
#define DFD_THERM2_DIS			BIT(19)
#define WDT_LATCH_CTL2_UNLOCK_KEY	(0x95000000)

#define WDT_SYSDBG_EN1			(0x0088)
#define SYSDBG_EN_KEY1			(0x1B2A)
#define WDT_SYSDBG_EN2			(0x008C)
#define SYSDBG_EN_KEY2			(0x4F59)

#define DRV_NAME			"mtk-wdt"
#define DRV_VERSION			"1.0"
#define DEV_VERSION_MSK			(0xffUL)
#define DEV_VERSION(chip, base)		\
		((((chip) & DEV_VERSION_MSK) << 8) | ((base) & DEV_VERSION_MSK))
#define DEV_VERSION_BASE(vers)		((vers) & DEV_VERSION_MSK)
#define DEV_VERSION_CHIP(vers)		(((vers) >> 8) & DEV_VERSION_MSK)

enum wdt_version_id {
	WDT_MT6589		= 1UL,
	WDT_MT6885,
	WDT_MT2731,
	WDT_MT2735,
};

enum wdt_req_mode {
	WDT_REQ_DIS		= 0UL,
	WDT_REQ_EN_IRQ_MODE,
	WDT_REQ_EN_RST_MODE,
};

struct mtk_wdt_dev;

struct mtk_wdt_plat {
	unsigned int		version;
	void (*request_default)(struct mtk_wdt_dev *mtk_wdt);
};

struct mtk_wdt_dev {
	struct watchdog_device	wdt_dev;
	void __iomem		*wdt_base;
	u32			eint;
	u32			dfd_timeout;
	const struct mtk_wdt_plat *wdt_plat;
};

static bool nowayout = WATCHDOG_NOWAYOUT;
static unsigned int timeout = WDT_TIMEOUT_MAX_SEC;
static void __iomem *toprgu_base;

static void mtk_wdt_update_current_stage(struct mtk_wdt_dev *mtk_wdt,
					 uint32_t stage)
{
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	uint32_t reg_val = readl(wdt_base + WDT_NONRST_REG2);

	reg_val &= (~(WDT_REBOOT_STAGE_MASK << WDT_REBOOT_STAGE_CUR_SHIFT));
	reg_val |= (stage << WDT_REBOOT_STAGE_CUR_SHIFT);

	writel(reg_val, wdt_base + WDT_NONRST_REG2);
}

static void mtk_wdt_request_config(struct mtk_wdt_dev *mtk_wdt,
				   unsigned int mark_bit,
				   enum wdt_req_mode mode)
{
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	u32 reg_val_en, reg_val_mode;

	if ((mark_bit == EINT_RST) && (mode != WDT_REQ_DIS)) {
		if (mtk_wdt->eint > EINT_SEL_MASK) {
			pr_err("invalid eint id\n");
			return;
		}
		writel((((mtk_wdt->eint) << EINT_SEL_SHIFT) | EINT_DGRST_EN),
		       wdt_base + WDT_EXT_REQ_CON);

	} else if (mark_bit == SYSRST_RST) {
		if (mode == WDT_REQ_DIS) {
			writel(0, wdt_base + WDT_SYSDBG_EN1);
			writel(0, wdt_base + WDT_SYSDBG_EN2);
		} else {
			writel(SYSDBG_EN_KEY1, wdt_base + WDT_SYSDBG_EN1);
			writel(SYSDBG_EN_KEY2, wdt_base + WDT_SYSDBG_EN2);
		}
	}

	reg_val_en = readl(wdt_base + WDT_REQ_MODE);
	if (mode == WDT_REQ_DIS) {
		reg_val_en &= ~mark_bit;
	} else {
		reg_val_en |= mark_bit;
	}
	writel((WDT_REQ_MODE_UNLOCK_KEY | reg_val_en), wdt_base + WDT_REQ_MODE);

	reg_val_mode = readl(wdt_base + WDT_REQ_IRQ_EN);
	if (mode != WDT_REQ_EN_IRQ_MODE) {
		reg_val_mode &= ~mark_bit;
	} else {
		reg_val_mode |= mark_bit;
	}
	writel((WDT_REQ_IRQ_EN_UNLOCK_KEY | reg_val_mode),
	       wdt_base + WDT_REQ_IRQ_EN);
}

static void mtk_wdt_parse_dt(struct device_node *np,
			     struct mtk_wdt_dev *mtk_wdt)
{
	int err;
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	unsigned int reg_val;

	err = of_property_read_u32(np, "mediatek,rg_dfd_timeout",
				   &mtk_wdt->dfd_timeout);

	if (!err && mtk_wdt->dfd_timeout) {
		/* enable dfd_en and setup timeout */
		reg_val = readl(wdt_base + WDT_LATCH_CTL2);
		reg_val &= ~(DFD_THERM2_DIS | DFD_TIMEOUT_MASK);
		reg_val |= (DFD_EN | DFD_THERM1_DIS |
			   (mtk_wdt->dfd_timeout & DFD_TIMEOUT_MASK));

		writel((WDT_LATCH_CTL2_UNLOCK_KEY | reg_val),
		       wdt_base + WDT_LATCH_CTL2);
	}
}

static void mtk_wdt_parse_eint(struct device_node *np,
			       struct mtk_wdt_dev *mtk_wdt)
{
	int err = of_property_read_u32(np, "eint", &mtk_wdt->eint);

	if (err)
		mtk_wdt->eint = (EINT_SEL_MASK + 1);
}

static int mtk_wdt_restart(struct watchdog_device *wdt_dev,
			   unsigned long action, void *cmd)
{
	struct mtk_wdt_dev *mtk_wdt = watchdog_get_drvdata(wdt_dev);
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	u32 reg_val = readl(wdt_base + WDT_NONRST_REG2);

	pr_info("[WDT]mtk_wdt_restart\n");
	reg_val &= ~WDT_REBOOT_MODE_MSK;
	if (cmd && (strcmp(cmd, "rpmbpk") == 0)) {
		reg_val |= WDT_REBOOT_RPMBPK;

	} else if (cmd && (strcmp(cmd, "recovery") == 0)) {
		reg_val |= WDT_REBOOT_RECOVERY;

	} else if (cmd && (strcmp(cmd, "bootloader") == 0)) {
		reg_val |= WDT_REBOOT_BOOTLOADER;

	} else if (cmd && !strcmp(cmd, "meta")) {
		reg_val |= WDT_REBOOT_META;
		pr_info("arch_reset, reboot meta mode\n");

	} else if (cmd && !strcmp(cmd, "ddr-reserve")) {
		reg_val |= WDT_REBOOT_DDR_RESERVE;
		pr_info("arch_reset, reboot ddr-reserve mode\n");
	}
	writel(reg_val, wdt_base + WDT_NONRST_REG2);

	reg_val = (WDT_LENGTH_SEC2LENGTH(WDT_TIMEOUT_DEFAULT_SEC) <<
		  WDT_LENGTH_SHIFT);
	reg_val &= WDT_LENGTH_MASK;
	writel((WDT_LENGTH_UNLOCK_KEY | reg_val), wdt_base + WDT_LENGTH);

	writel(WDT_RESTART_KEY, wdt_base + WDT_RESTART);

	if (!(readl(wdt_base + WDT_NONRST_REG2) & WDT_REBOOT_DDR_RESERVE))
		mtk_dbgtop_dram_reserved(0);

	reg_val = readl(wdt_base + WDT_MODE);
	// reg_val |= (WDT_EN | EXTEN);
	reg_val |= EXTEN;
	reg_val &= ~(WDT_IRQ | DUAL_MODE);
	writel((WDT_MODE_UNLOCK_KEY | reg_val), wdt_base + WDT_MODE);

	if (!arm_pm_restart) {
		while (1) {
			writel(WDT_SWRST_KEY, wdt_base + WDT_SWRST);
			mdelay(5);
		}
	}

	return NOTIFY_DONE;
}

static int mtk_wdt_ping(struct watchdog_device *wdt_dev)
{
	struct mtk_wdt_dev *mtk_wdt = watchdog_get_drvdata(wdt_dev);
	void __iomem *wdt_base = mtk_wdt->wdt_base;

	writel(WDT_RESTART_KEY, wdt_base + WDT_RESTART);
	return 0;
}

static int mtk_wdt_set_timeout(struct watchdog_device *wdt_dev,
			       unsigned int timeout)
{
	struct mtk_wdt_dev *mtk_wdt = watchdog_get_drvdata(wdt_dev);
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	u32 reg_val;

	wdt_dev->timeout = timeout;
	reg_val = (WDT_LENGTH_SEC2LENGTH(timeout) << WDT_LENGTH_SHIFT);
	reg_val &= WDT_LENGTH_MASK;
	writel((WDT_LENGTH_UNLOCK_KEY | reg_val), wdt_base + WDT_LENGTH);

	mtk_wdt_ping(wdt_dev);

	return 0;
}

static int mtk_wdt_stop(struct watchdog_device *wdt_dev)
{
	struct mtk_wdt_dev *mtk_wdt = watchdog_get_drvdata(wdt_dev);
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	u32 reg_val;

	reg_val = readl(wdt_base + WDT_MODE);
	reg_val &= (~WDT_EN);
	writel((WDT_MODE_UNLOCK_KEY | reg_val), wdt_base + WDT_MODE);

	clear_bit(WDOG_HW_RUNNING, &mtk_wdt->wdt_dev.status);
	return 0;
}

static int mtk_wdt_start(struct watchdog_device *wdt_dev)
{
	u32 reg_val;
	struct mtk_wdt_dev *mtk_wdt = watchdog_get_drvdata(wdt_dev);
	void __iomem *wdt_base = mtk_wdt->wdt_base;
	int ret;
	ret = mtk_wdt_set_timeout(wdt_dev, wdt_dev->timeout);
	if (ret < 0)
		return ret;

	reg_val = readl(wdt_base + WDT_MODE);
	reg_val |= (WDT_EN | EXTEN | WDT_IRQ | DUAL_MODE);
	writel((WDT_MODE_UNLOCK_KEY | reg_val), wdt_base + WDT_MODE);

	set_bit(WDOG_HW_RUNNING, &mtk_wdt->wdt_dev.status);

	return 0;
}

static const struct watchdog_info mtk_wdt_info = {
	.identity	= DRV_NAME,
	.options	= WDIOF_SETTIMEOUT |
			  WDIOF_KEEPALIVEPING |
			  WDIOF_MAGICCLOSE,
};

static const struct watchdog_ops mtk_wdt_ops = {
	.owner		= THIS_MODULE,
	.start		= mtk_wdt_start,
	.stop		= mtk_wdt_stop,
	.ping		= mtk_wdt_ping,
	.set_timeout	= mtk_wdt_set_timeout,
	.restart	= mtk_wdt_restart,
};

static int mtk_wdt_probe(struct platform_device *pdev)
{
	struct mtk_wdt_dev *mtk_wdt;
	struct device *dev = &pdev->dev;
	struct resource *res;
	int err;
	u32 mode;

	mtk_wdt = devm_kzalloc(dev, sizeof(*mtk_wdt), GFP_KERNEL);
	if (!mtk_wdt)
		return -ENOMEM;

	platform_set_drvdata(pdev, mtk_wdt);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	mtk_wdt->wdt_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(mtk_wdt->wdt_base))
		return PTR_ERR(mtk_wdt->wdt_base);
	toprgu_base = mtk_wdt->wdt_base;

	arm_pm_restart = NULL;
	mtk_wdt->wdt_dev.info = &mtk_wdt_info;
	mtk_wdt->wdt_dev.ops = &mtk_wdt_ops;
	mtk_wdt->wdt_dev.timeout = WDT_TIMEOUT_MAX_SEC;
	mtk_wdt->wdt_dev.max_timeout = WDT_TIMEOUT_MAX_SEC;
	mtk_wdt->wdt_dev.min_timeout = WDT_TIMEOUT_MIN_SEC;
	mtk_wdt->wdt_dev.parent = dev;

	mtk_wdt->wdt_plat = of_device_get_match_data(dev);
	mtk_wdt_update_current_stage(mtk_wdt, WDT_REBOOT_STAGE_KERNEL);

	mtk_wdt_parse_dt(dev->of_node, mtk_wdt);
	mtk_wdt_parse_eint(dev->of_node, mtk_wdt);
	if (mtk_wdt->wdt_plat && mtk_wdt->wdt_plat->request_default)
		mtk_wdt->wdt_plat->request_default(mtk_wdt);

	watchdog_init_timeout(&mtk_wdt->wdt_dev, timeout, dev);
	watchdog_set_nowayout(&mtk_wdt->wdt_dev, nowayout);
	watchdog_set_restart_priority(&mtk_wdt->wdt_dev, 128);

	watchdog_set_drvdata(&mtk_wdt->wdt_dev, mtk_wdt);
	writel((WDT_MODE_UNLOCK_KEY | WDT_EN), mtk_wdt->wdt_base + WDT_MODE);
	if (readl(mtk_wdt->wdt_base + WDT_MODE) & WDT_EN) {
    	set_bit(WDOG_HW_RUNNING, &mtk_wdt->wdt_dev.status);
		mtk_wdt_start(&mtk_wdt->wdt_dev);
	}
	else
		mtk_wdt_stop(&mtk_wdt->wdt_dev);
	err = watchdog_register_device(&mtk_wdt->wdt_dev);
	if (unlikely(err))
		return err;
	mode = readl(mtk_wdt->wdt_base + WDT_MODE);
	pr_info("[mode]0x%x\n", mode);
	dev_info(dev, "Watchdog enabled (timeout=%d sec, nowayout=%d)\n",
		 mtk_wdt->wdt_dev.timeout, nowayout);	
	return 0;
}

static void mtk_wdt_shutdown(struct platform_device *pdev)
{
	struct mtk_wdt_dev *mtk_wdt = platform_get_drvdata(pdev);

	if (watchdog_active(&mtk_wdt->wdt_dev))
		mtk_wdt_stop(&mtk_wdt->wdt_dev);
}

static int mtk_wdt_remove(struct platform_device *pdev)
{
	struct mtk_wdt_dev *mtk_wdt = platform_get_drvdata(pdev);

	mtk_wdt_stop(&mtk_wdt->wdt_dev);
	watchdog_unregister_device(&mtk_wdt->wdt_dev);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int mtk_wdt_suspend(struct device *dev)
{
	struct mtk_wdt_dev *mtk_wdt = dev_get_drvdata(dev);

	if (watchdog_active(&mtk_wdt->wdt_dev))
		mtk_wdt_stop(&mtk_wdt->wdt_dev);

	return 0;
}

static int mtk_wdt_resume(struct device *dev)
{
	struct mtk_wdt_dev *mtk_wdt = dev_get_drvdata(dev);

	if (watchdog_active(&mtk_wdt->wdt_dev))
		mtk_wdt_start(&mtk_wdt->wdt_dev);

	return 0;
}

static const struct dev_pm_ops mtk_wdt_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(mtk_wdt_suspend, mtk_wdt_resume)
};
#endif

static void mtk_wdt_request_default_mt2731(struct mtk_wdt_dev *mtk_wdt)
{
	mtk_wdt_request_config(mtk_wdt, THERMAL_CTL_RST, WDT_REQ_EN_RST_MODE);
	/* spm: scpsys */
	mtk_wdt_request_config(mtk_wdt, SPM_RST, WDT_REQ_EN_RST_MODE);
}

static struct mtk_wdt_plat mt2731_wdt = {
	.version = DEV_VERSION(WDT_MT2731, WDT_MT6589),
	.request_default = mtk_wdt_request_default_mt2731,
};

static struct mtk_wdt_plat mt2735_wdt = {
	.version = DEV_VERSION(WDT_MT2735, WDT_MT6885),
	.request_default = NULL,
};

static const struct of_device_id mtk_wdt_dt_ids[] = {
	{ .compatible = "mediatek,mt2731-wdt", .data = &mt2731_wdt },
	{ .compatible = "mediatek,mt2735-wdt", .data = &mt2735_wdt },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, mtk_wdt_dt_ids);

static struct platform_driver mtk_wdt_driver = {
	.probe		= mtk_wdt_probe,
	.remove		= mtk_wdt_remove,
	.shutdown	= mtk_wdt_shutdown,
	.driver		= {
		.name		= DRV_NAME,
#ifdef CONFIG_PM_SLEEP
		.pm		= &mtk_wdt_pm_ops,
#endif
		.of_match_table	= mtk_wdt_dt_ids,
	},
};
module_platform_driver(mtk_wdt_driver);

void __iomem *mtk_wd_Gettoprgu(void)
{
	struct device_node *np_rgu;
	unsigned int i, cnt = ARRAY_SIZE(mtk_wdt_dt_ids) - 1;

	if (!toprgu_base) {
		for (i = 0; i < cnt; i++) {
			np_rgu = of_find_compatible_node(NULL, NULL,
					mtk_wdt_dt_ids[i].compatible);
			if (!np_rgu)
				continue;

			toprgu_base = of_iomap(np_rgu, 0);
			if (!toprgu_base)
				pr_debug("RGU iomap failed\n");

			break;
		}
	}

	pr_info("get toprgu is %p\n", toprgu_base);

	return toprgu_base;
}
EXPORT_SYMBOL_GPL(mtk_wd_Gettoprgu);

module_param(timeout, uint, 0);
MODULE_PARM_DESC(timeout, "Watchdog heartbeat in seconds");

module_param(nowayout, bool, 0);
MODULE_PARM_DESC(nowayout, "Watchdog cannot be stopped once started (default="
		 __MODULE_STRING(WATCHDOG_NOWAYOUT) ")");

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Matthias Brugger <matthias.bgg@gmail.com>");
MODULE_DESCRIPTION("Mediatek WatchDog Timer Driver");
MODULE_VERSION(DRV_VERSION);
