// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2019 MediaTek Inc.
 * Author: Eason Yen <eason.yen@mediatek.com>
 */

#include <linux/module.h>
#include <linux/pm_runtime.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>

#include "../common/mtk-afe-platform-driver.h"
#include "../common/mtk-sp-spk-amp.h"

#include "mt6880-afe-common.h"
#include "mt6880-afe-clk.h"
#include "mt6880-afe-gpio.h"
#include <linux/cdev.h>
#include <linux/regmap.h>
#include "nau8810.h"
#include "tlv320aic3x.h"




/* #include "../../codecs/mt6359.h" */

/*
 * if need additional control for the ext spk amp that is connected
 * after Lineout Buffer / HP Buffer on the codec, put the control in
 * mt6880_mt6359_spk_amp_event()
 */
#define EXT_SPK_AMP_W_NAME "Ext_Speaker_Amp"

static const char *const mt6880_spk_type_str[] = {MTK_SPK_NOT_SMARTPA_STR,
						  MTK_SPK_RICHTEK_RT5509_STR,
						  MTK_SPK_MEDIATEK_MT6660_STR
						  };
static const char *const mt6880_spk_i2s_type_str[] = {MTK_SPK_I2S_0_STR,
						      MTK_SPK_I2S_1_STR,
						      MTK_SPK_I2S_2_STR,
						      MTK_SPK_I2S_3_STR,
						      MTK_SPK_I2S_5_STR};

static const struct soc_enum mt6880_spk_type_enum[] = {
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(mt6880_spk_type_str),
			    mt6880_spk_type_str),
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(mt6880_spk_i2s_type_str),
			    mt6880_spk_i2s_type_str),
};

/* Use speaker : MTK_SPK_MEDIATEK_MT6660 on i2s0 and i2s3 */
static int mt6880_spk_type_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	int idx = mtk_spk_get_type();

	pr_debug("%s() = %d\n", __func__, idx);
	ucontrol->value.integer.value[0] = idx;
	return 0;
}

static int mt6880_spk_i2s_out_type_get(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *ucontrol)
{
	int idx = mtk_spk_get_i2s_out_type();

	pr_debug("%s() = %d\n", __func__, idx);
	ucontrol->value.integer.value[0] = idx;
	return 0;
}

static int mt6880_spk_i2s_in_type_get(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_value *ucontrol)
{
	int idx = mtk_spk_get_i2s_in_type();

	pr_debug("%s() = %d\n", __func__, idx);
	ucontrol->value.integer.value[0] = idx;
	return 0;
}

static int mt6880_mt6359_spk_amp_event(struct snd_soc_dapm_widget *w,
				       struct snd_kcontrol *kcontrol,
				       int event)
{
	struct snd_soc_dapm_context *dapm = w->dapm;
	struct snd_soc_card *card = dapm->card;

	dev_info(card->dev, "%s(), event %d\n", __func__, event);

	switch (event) {
	case SND_SOC_DAPM_POST_PMU:
		/* spk amp on control */
		break;
	case SND_SOC_DAPM_PRE_PMD:
		/* spk amp off control */
		break;
	default:
		break;
	}

	return 0;
};

static const struct snd_soc_dapm_widget mt6880_mt6359_widgets[] = {
	SND_SOC_DAPM_SPK(EXT_SPK_AMP_W_NAME, mt6880_mt6359_spk_amp_event),
};

static const struct snd_soc_dapm_route mt6880_mt6359_routes[] = {
};

static const struct snd_kcontrol_new mt6880_mt6359_controls[] = {
	SOC_DAPM_PIN_SWITCH(EXT_SPK_AMP_W_NAME),
	SOC_ENUM_EXT("MTK_SPK_TYPE_GET", mt6880_spk_type_enum[0],
		     mt6880_spk_type_get, NULL),
	SOC_ENUM_EXT("MTK_SPK_I2S_OUT_TYPE_GET", mt6880_spk_type_enum[1],
		     mt6880_spk_i2s_out_type_get, NULL),
	SOC_ENUM_EXT("MTK_SPK_I2S_IN_TYPE_GET", mt6880_spk_type_enum[1],
		     mt6880_spk_i2s_in_type_get, NULL),
};

/*
 * define mtk_spk_i2s_mck node in dts when need mclk,
 * BE i2s need assign snd_soc_ops = mt6880_mt6359_i2s_ops
 */
static int mt6880_mt6359_i2s_hw_params(struct snd_pcm_substream *substream,
				       struct snd_pcm_hw_params *params)
{
    struct snd_soc_pcm_runtime *rtd = substream->private_data;
    unsigned int rate = params_rate(params);
    unsigned int mclk_fs_ratio = 256;	/* zhengzhou 0326 */
    unsigned int mclk_fs = rate * mclk_fs_ratio;
    unsigned int cpu_dai_fmt, codec_dai_fmt;
    int ret = 0;

	cpu_dai_fmt = SND_SOC_DAIFMT_I2S;
	codec_dai_fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_CBS_CFS;

	ret = snd_soc_dai_set_fmt(rtd->cpu_dai, cpu_dai_fmt);
	if (ret != 0 && ret != -ENOTSUPP)
		return ret;

	ret = snd_soc_dai_set_fmt(rtd->codec_dai, codec_dai_fmt);
    if (ret != 0 && ret != -ENOTSUPP)
        return ret;
	/*	zhengzhou 0316 +++*/
    ret = snd_soc_dai_set_sysclk(rtd->cpu_dai, 0,
                      mclk_fs, SND_SOC_CLOCK_OUT);
	printk("snd_soc_dai_set_sysclk ret =%d",ret);
    if (ret != 0 && ret != -ENOTSUPP)
        return ret;
    
    return snd_soc_dai_set_sysclk(rtd->codec_dai, 0,
                      mclk_fs, SND_SOC_CLOCK_IN); 
					  	/*	zhengzhou 0316 ---*/
}

static const struct snd_soc_ops mt6880_mt6359_i2s_ops = {
	.hw_params = mt6880_mt6359_i2s_hw_params,
};
#if 0
static int mt6880_mt6359_mtkaif_calibration(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_component *component =
		snd_soc_rtdcom_lookup(rtd, AFE_PCM_NAME);
	struct mtk_base_afe *afe = snd_soc_component_get_drvdata(component);
	struct mt6880_afe_private *afe_priv = afe->platform_priv;
	struct snd_soc_component *codec_component =
		snd_soc_rtdcom_lookup(rtd, CODEC_MT6359_NAME);

	int phase;
	unsigned int monitor;
	int test_done_1, test_done_2, test_done_3;
	int cycle_1, cycle_2, cycle_3;
	int prev_cycle_1, prev_cycle_2, prev_cycle_3;
	int counter;

	dev_info(afe->dev, "%s(), start\n", __func__);

	pm_runtime_get_sync(afe->dev);
	mt6880_afe_gpio_request(afe, true, MT6880_DAI_ADDA, 1);
	mt6880_afe_gpio_request(afe, true, MT6880_DAI_ADDA, 0);
	mt6880_afe_gpio_request(afe, true, MT6880_DAI_ADDA_CH34, 1);
	mt6880_afe_gpio_request(afe, true, MT6880_DAI_ADDA_CH34, 0);

	mt6359_mtkaif_calibration_enable(codec_component);

	/* set clock protocol 2 */
	regmap_update_bits(afe->regmap, AFE_AUD_PAD_TOP, 0xff, 0x38);
	regmap_update_bits(afe->regmap, AFE_AUD_PAD_TOP, 0xff, 0x39);

	/* set test type to synchronizer pulse */
	regmap_update_bits(afe_priv->topckgen, CKSYS_AUD_TOP_CFG,
			   0xffff, 0x4);

	afe_priv->mtkaif_calibration_num_phase = 42;	/* mt6359: 0 ~ 42 */
	afe_priv->mtkaif_calibration_ok = true;
	afe_priv->mtkaif_chosen_phase[0] = -1;
	afe_priv->mtkaif_chosen_phase[1] = -1;
	afe_priv->mtkaif_chosen_phase[2] = -1;

	for (phase = 0;
	     phase <= afe_priv->mtkaif_calibration_num_phase &&
	     afe_priv->mtkaif_calibration_ok;
	     phase++) {
		mt6359_set_mtkaif_calibration_phase(codec_component,
						    phase, phase, phase);

		regmap_update_bits(afe_priv->topckgen, CKSYS_AUD_TOP_CFG,
				   0x1, 0x1);

		test_done_1 = 0;
		test_done_2 = 0;
		test_done_3 = 0;
		cycle_1 = -1;
		cycle_2 = -1;
		cycle_3 = -1;
		counter = 0;
		while (test_done_1 == 0 ||
		       test_done_2 == 0 ||
		       test_done_3 == 0) {
			regmap_read(afe_priv->topckgen, CKSYS_AUD_TOP_MON,
				    &monitor);

			test_done_1 = (monitor >> 28) & 0x1;
			test_done_2 = (monitor >> 29) & 0x1;
			test_done_3 = (monitor >> 30) & 0x1;
			if (test_done_1 == 1)
				cycle_1 = monitor & 0xf;

			if (test_done_2 == 1)
				cycle_2 = (monitor >> 4) & 0xf;

			if (test_done_3 == 1)
				cycle_3 = (monitor >> 8) & 0xf;

			/* handle if never test done */
			if (++counter > 10000) {
				dev_err(afe->dev, "%s(), test fail, cycle_1 %d, cycle_2 %d, cycle_3 %d, monitor 0x%x\n",
					__func__,
					cycle_1, cycle_2, cycle_3, monitor);
				afe_priv->mtkaif_calibration_ok = false;
				break;
			}
		}

		if (phase == 0) {
			prev_cycle_1 = cycle_1;
			prev_cycle_2 = cycle_2;
			prev_cycle_3 = cycle_3;
		}

		if (cycle_1 != prev_cycle_1 &&
		    afe_priv->mtkaif_chosen_phase[0] < 0) {
			afe_priv->mtkaif_chosen_phase[0] = phase - 1;
			afe_priv->mtkaif_phase_cycle[0] = prev_cycle_1;
		}

		if (cycle_2 != prev_cycle_2 &&
		    afe_priv->mtkaif_chosen_phase[1] < 0) {
			afe_priv->mtkaif_chosen_phase[1] = phase - 1;
			afe_priv->mtkaif_phase_cycle[1] = prev_cycle_2;
		}

		if (cycle_3 != prev_cycle_3 &&
		    afe_priv->mtkaif_chosen_phase[2] < 0) {
			afe_priv->mtkaif_chosen_phase[2] = phase - 1;
			afe_priv->mtkaif_phase_cycle[2] = prev_cycle_3;
		}

		regmap_update_bits(afe_priv->topckgen, CKSYS_AUD_TOP_CFG,
				   0x1, 0x0);

		if (afe_priv->mtkaif_chosen_phase[0] >= 0 &&
		    afe_priv->mtkaif_chosen_phase[1] >= 0 &&
		    afe_priv->mtkaif_chosen_phase[2] >= 0)
			break;
	}

	if (afe_priv->mtkaif_chosen_phase[0] < 0 ||
	    afe_priv->mtkaif_chosen_phase[1] < 0 ||
	    afe_priv->mtkaif_chosen_phase[2] < 0)
		afe_priv->mtkaif_calibration_ok = false;

	if (!afe_priv->mtkaif_calibration_ok)
		mt6359_set_mtkaif_calibration_phase(codec_component,
						    0, 0, 0);
	else
		mt6359_set_mtkaif_calibration_phase(codec_component,
			afe_priv->mtkaif_chosen_phase[0],
			afe_priv->mtkaif_chosen_phase[1],
			afe_priv->mtkaif_chosen_phase[2]);

	/* disable rx fifo */
	regmap_update_bits(afe->regmap, AFE_AUD_PAD_TOP, 0xff, 0x38);

	mt6359_mtkaif_calibration_disable(codec_component);

	mt6880_afe_gpio_request(afe, false, MT6880_DAI_ADDA, 1);
	mt6880_afe_gpio_request(afe, false, MT6880_DAI_ADDA, 0);
	mt6880_afe_gpio_request(afe, false, MT6880_DAI_ADDA_CH34, 1);
	mt6880_afe_gpio_request(afe, false, MT6880_DAI_ADDA_CH34, 0);
	pm_runtime_put(afe->dev);

	dev_info(afe->dev, "%s(), end, calibration ok %d\n",
		 __func__,
		 afe_priv->mtkaif_calibration_ok);

	return 0;
}
#endif
#if 0
static int mt6880_mt6359_init(struct snd_soc_pcm_runtime *rtd)
{

	struct snd_soc_component *component =
		snd_soc_rtdcom_lookup(rtd, AFE_PCM_NAME);
	struct mtk_base_afe *afe = snd_soc_component_get_drvdata(component);
	struct mt6880_afe_private *afe_priv = afe->platform_priv;
	struct snd_soc_component *codec_component =
		snd_soc_rtdcom_lookup(rtd, CODEC_MT6359_NAME);
	struct snd_soc_dapm_context *dapm = &rtd->card->dapm;
	/* set mtkaif protocol */
	mt6359_set_mtkaif_protocol(codec_component,
				   MT6359_MTKAIF_PROTOCOL_2_CLK_P2);
	afe_priv->mtkaif_protocol = MTKAIF_PROTOCOL_2_CLK_P2;

	/* mtkaif calibration */
	if (afe_priv->mtkaif_protocol == MTKAIF_PROTOCOL_2_CLK_P2)
		mt6880_mt6359_mtkaif_calibration(rtd);

	/* disable ext amp connection */
	snd_soc_dapm_disable_pin(dapm, EXT_SPK_AMP_W_NAME);

	return 0;
}
#endif
/*
static int mt6880_i2s_hw_params_fixup(struct snd_soc_pcm_runtime *rtd,
				      struct snd_pcm_hw_params *params)
{
	dev_info(rtd->dev, "%s(), fix format to 32bit\n", __func__);

	
	snd_mask_reset_range(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT),
			     0, SNDRV_PCM_FORMAT_LAST);

	params_set_format(params, SNDRV_PCM_FORMAT_S32_LE);
	return 0;
}
*/
static struct snd_soc_dai_link mt6880_mt6359_dai_links[] = {
	/* Front End DAI links */
	{
		.name = "Playback_1",
		.stream_name = "Playback_1",
		.cpu_dai_name = "DL1",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
	},
	{
		.name = "Playback_12",
		.stream_name = "Playback_12",
		.cpu_dai_name = "DL12",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
	},
	{
		.name = "Playback_2",
		.stream_name = "Playback_2",
		.cpu_dai_name = "DL2",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
	},
	{
		.name = "Playback_3",
		.stream_name = "Playback_3",
		.cpu_dai_name = "DL3",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
	},
	{
		.name = "Playback_4",
		.stream_name = "Playback_4",
		.cpu_dai_name = "DL4",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
	},
	{
		.name = "Capture_1",
		.stream_name = "Capture_1",
		.cpu_dai_name = "UL1",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_2",
		.stream_name = "Capture_2",
		.cpu_dai_name = "UL2",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_3",
		.stream_name = "Capture_3",
		.cpu_dai_name = "UL3",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_4",
		.stream_name = "Capture_4",
		.cpu_dai_name = "UL4",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_5",
		.stream_name = "Capture_5",
		.cpu_dai_name = "UL5",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_6",
		.stream_name = "Capture_6",
		.cpu_dai_name = "UL6",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_7",
		.stream_name = "Capture_7",
		.cpu_dai_name = "UL7",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_Mono_1",
		.stream_name = "Capture_Mono_1",
		.cpu_dai_name = "UL_MONO_1",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_Mono_2",
		.stream_name = "Capture_Mono_2",
		.cpu_dai_name = "UL_MONO_2",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Capture_Mono_3",
		.stream_name = "Capture_Mono_3",
		.cpu_dai_name = "UL_MONO_3",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Playback_HDMI",
		.stream_name = "Playback_HDMI",
		.cpu_dai_name = "HDMI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
	},
	{
		.name = "Hostless_LPBK",
		.stream_name = "Hostless_LPBK",
		.cpu_dai_name = "Hostless LPBK DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_FM",
		.stream_name = "Hostless_FM",
		.cpu_dai_name = "Hostless FM DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_Speech",
		.stream_name = "Hostless_Speech",
		.cpu_dai_name = "Hostless Speech DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_Sph_Echo_Ref",
		.stream_name = "Hostless_Sph_Echo_Ref",
		.cpu_dai_name = "Hostless_Sph_Echo_Ref_DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_Spk_Init",
		.stream_name = "Hostless_Spk_Init",
		.cpu_dai_name = "Hostless_Spk_Init_DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_ADDA_DL_I2S_OUT",
		.stream_name = "Hostless_ADDA_DL_I2S_OUT",
		.cpu_dai_name = "Hostless_ADDA_DL_I2S_OUT DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_SRC_1",
		.stream_name = "Hostless_SRC_1",
		.cpu_dai_name = "Hostless_SRC_1_DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
			    SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_Speech_BT",
		.stream_name = "Hostless_Speech_BT",
		.cpu_dai_name = "Hostless Speech_BT DAI",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.trigger = {SND_SOC_DPCM_TRIGGER_PRE,
				SND_SOC_DPCM_TRIGGER_PRE},
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	/* Back End DAI links */
#if 0
	{
		.name = "Primary Codec",
		.cpu_dai_name = "ADDA",
		.codec_dai_name = "mt6359-snd-codec-aif1",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
		.init = mt6880_mt6359_init,
	},
	{
		.name = "Primary Codec CH34",
		.cpu_dai_name = "ADDA_CH34",
		.codec_dai_name = "mt6359-snd-codec-aif2",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
#endif
	{
		.name = "AP_DMIC",
		.cpu_dai_name = "AP_DMIC",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "AP_DMIC_CH34",
		.cpu_dai_name = "AP_DMIC_CH34",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
			
#if 0
	{
		.name = "Speaker Codec",
		.cpu_dai_name = "I2S3",
		.codec_dai_name = "mt6660-aif",
		.dai_fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_CBS_CFS
			| SND_SOC_DAIFMT_GATED,
		.ops = &mt6880_mt6359_i2s_ops,
		.no_pcm = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.ignore_pmdown_time = 1,
		.be_hw_params_fixup = mt6880_i2s_hw_params_fixup,
	},
	{
		.name = "Speaker Codec Ref",
		.cpu_dai_name = "I2S0",
		.codec_dai_name = "mt6660-aif",
		.dai_fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_CBS_CFS
			| SND_SOC_DAIFMT_GATED,
		.ops = &mt6880_mt6359_i2s_ops,
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
		.ignore_pmdown_time = 1,
		.be_hw_params_fixup = mt6880_i2s_hw_params_fixup,
	},
	{
		.name = "External Codec DL",
		.cpu_dai_name = "External Codec DL",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.ignore_pmdown_time = 1,
	},
	{
		.name = "External Codec UL",
		.cpu_dai_name = "External Codec UL",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
		.ignore_pmdown_time = 1,
	},
#endif
	{
		.name = "I2S0",
		.cpu_dai_name = "I2S0",
		.codec_dai_name = "tlv320aic3x-hifi",	//tianyan@2021.7.16 modify for compatible codec tlv320aic3104 start
		.codec_name = "tlv320aic3x-codec.0-0018",//tianyan@2021.7.16 modify for compatible codec tlv320aic3104 end
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
		.dai_fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_CBS_CFS
			| SND_SOC_DAIFMT_GATED,
		.ops = &mt6880_mt6359_i2s_ops,
	},
	{
		.name = "I2S1",
		.cpu_dai_name = "I2S1",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "I2S2",
		.cpu_dai_name = "I2S2",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "I2S3",
		.cpu_dai_name = "I2S3",
		.codec_dai_name = "tlv320aic3x-hifi",	//tianyan@2021.7.16 modify for compatible codec tlv320aic3104 start
		.codec_name = "tlv320aic3x-codec.0-0018",//tianyan@2021.7.16 modify for compatible codec tlv320aic3104 end
		.no_pcm = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.dai_fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_CBS_CFS
			| SND_SOC_DAIFMT_GATED,
		.ops = &mt6880_mt6359_i2s_ops,
	},
	{
		.name = "I2S4",
		.cpu_dai_name = "I2S4",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "I2S5",
		.cpu_dai_name = "I2S5",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
	},	
	{
		.name = "I2S6",
		.cpu_dai_name = "I2S6",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "HW Gain 1",
		.cpu_dai_name = "HW Gain 1",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "HW Gain 2",
		.cpu_dai_name = "HW Gain 2",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "HW_SRC_1",
		.cpu_dai_name = "HW_SRC_1",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "HW_SRC_2",
		.cpu_dai_name = "HW_SRC_2",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "CONNSYS_I2S",
		.cpu_dai_name = "CONNSYS_I2S",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "PCM 0",
		.cpu_dai_name = "PCM 0",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "PCM 1",
		.cpu_dai_name = "PCM 1",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "PCM 2",
		.cpu_dai_name = "PCM 2",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "TDM",
		.cpu_dai_name = "TDM",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "ETDM",
		.cpu_dai_name = "ETDM",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},

	/* dummy BE for ul memif to record from dl memif */
	{
		.name = "Hostless_UL1",
		.cpu_dai_name = "Hostless_UL1 DAI",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_UL2",
		.cpu_dai_name = "Hostless_UL2 DAI",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_UL3",
		.cpu_dai_name = "Hostless_UL3 DAI",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	{
		.name = "Hostless_UL6",
		.cpu_dai_name = "Hostless_UL6 DAI",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.no_pcm = 1,
		.dpcm_capture = 1,
		.ignore_suspend = 1,
	},
	/* BTCVSD */
#ifdef CONFIG_SND_SOC_MTK_BTCVSD
	{
		.name = "BTCVSD",
		.stream_name = "BTCVSD",
		.cpu_dai_name   = "snd-soc-dummy-dai",
		.platform_name  = "18050000.mtk-btcvsd-snd",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
	},
#endif
};

static struct snd_soc_card mt6880_mt6359_soc_card = {
	.name = "mt6880-mt6359",
	.owner = THIS_MODULE,
	.dai_link = mt6880_mt6359_dai_links,
	.num_links = ARRAY_SIZE(mt6880_mt6359_dai_links),

	.controls = mt6880_mt6359_controls,
	.num_controls = ARRAY_SIZE(mt6880_mt6359_controls),
	.dapm_widgets = mt6880_mt6359_widgets,
	.num_dapm_widgets = ARRAY_SIZE(mt6880_mt6359_widgets),
	.dapm_routes = mt6880_mt6359_routes,
	.num_dapm_routes = ARRAY_SIZE(mt6880_mt6359_routes),
};

//tianyan@2021.8.2 modify for compatible codec tlv320aic3104 and add loopback test start
#if defined(CONFIG_SND_SOC_NAU8810) || defined(CONFIG_SND_SOC_TLV320AIC3X)

struct mt6880_codec_cmd{
	bool read;//true is read
	int reg;	
	int value;
};

enum mt6880_codec_type{
	MT6880_CODEC_TYPE_TLV320AIC3104,
	MT6880_CODEC_TYPE_NAU8810,
};

static struct regmap *mt6880_regmap = NULL;
static enum mt6880_codec_type g_mt6880_codec_type = MT6880_CODEC_TYPE_TLV320AIC3104;

#define MT6880_CODEC_CMD_GENERAL 		(5000+1)
#define MT6880_CODEC_CMD_LOOPBACK_TEST 	(5000+2)


static long mt6880_codec_ioctl(struct file *file, unsigned int cmd, unsigned long args)
{
	struct mt6880_codec_cmd mt6880_cmd;
	int ret=0;

	if(!mt6880_regmap)
		return -1;

	switch(cmd)
	{
		case MT6880_CODEC_CMD_GENERAL:
			ret = copy_from_user(&mt6880_cmd, (void *)args,sizeof(mt6880_cmd));
			printk("mt6880 cmd:read=%d,reg=%x,value=%x\n",mt6880_cmd.read,mt6880_cmd.reg,mt6880_cmd.value);
			if(mt6880_cmd.read)
			{
				ret =regmap_read(mt6880_regmap,mt6880_cmd.reg, &mt6880_cmd.value);
				printk("mt6880 read:read=%d,reg=%x,value=%x\n",mt6880_cmd.read,mt6880_cmd.reg,mt6880_cmd.value);
				ret = copy_to_user((void*)args,&mt6880_cmd,sizeof(mt6880_cmd));
			}
			else
			{	
				ret=regmap_write(mt6880_regmap,mt6880_cmd.reg, mt6880_cmd.value);
			}

			break;
		case MT6880_CODEC_CMD_LOOPBACK_TEST:
#ifdef CONFIG_SND_SOC_NAU8810
			if( MT6880_CODEC_TYPE_NAU8810 == g_mt6880_codec_type)
				nau8810_loopback_test();
#endif

#ifdef CONFIG_SND_SOC_TLV320AIC3X
			if( MT6880_CODEC_TYPE_TLV320AIC3104 == g_mt6880_codec_type)
				tlv320aic3104_loopback_test();
#endif

			break;
		
		default:
			break;
	}

	return ret;
}


static struct cdev mt6880_codec_cdev;
static struct class *mt6880_codec_cls;

static struct file_operations  mt6880_chdev_ops = {
	.unlocked_ioctl = mt6880_codec_ioctl,
};
#endif
//tianyan@2021.8.2 modify for compatible codec tlv320aic3104 and add loopback test end

static int mt6880_mt6359_dev_probe(struct platform_device *pdev)
{
	struct snd_soc_card *card = &mt6880_mt6359_soc_card;
	struct device_node *platform_node;
#if 0
	struct device_node *codec_node, *spk_codec_node;
	struct device_node *spk_codec_node;
	struct device_node *ext_codec_node;
#endif


	int ret;
	int i;

//tianyan@2021.8.2 modify for compatible codec tlv320aic3104 and add loopback test start
#if defined(CONFIG_SND_SOC_NAU8810) || defined(CONFIG_SND_SOC_TLV320AIC3X)
	dev_t devno;
#endif
//tianyan@2021.8.2 modify for compatible codec tlv320aic3104 and add loopback test end

	dev_info(&pdev->dev, "%s(), start\n", __func__);
#if 0
	ret = mtk_spk_update_dai_link(card, pdev, &mt6880_mt6359_i2s_ops);
		if (ret) {
			dev_err(&pdev->dev, "%s(), mtk_spk_update_dai_link error\n",
				__func__);
			return -EINVAL;
	}
#endif
	platform_node = of_parse_phandle(pdev->dev.of_node,
					 "mediatek,platform", 0);
	if (!platform_node) {
		dev_err(&pdev->dev, "Property 'platform' missing or invalid\n");
		return -EINVAL;
	}

	for (i = 0; i < card->num_links; i++) {
		if (mt6880_mt6359_dai_links[i].platform_name)
			continue;
		mt6880_mt6359_dai_links[i].platform_of_node = platform_node;
	}

	card->dev = &pdev->dev;

#if 0
	ext_codec_node = of_parse_phandle(pdev->dev.of_node,
				      "mediatek,ext-codec", 0);
	if (!ext_codec_node) {
		dev_err(&pdev->dev,
			"Property 'ext-codec' missing or invalid\n");
		return -EINVAL;
	}

	codec_node = of_parse_phandle(pdev->dev.of_node,
				      "mediatek,audio-codec", 0);
	if (!codec_node) {
		dev_err(&pdev->dev,
			"Property 'audio-codec' missing or invalid\n");
		return -EINVAL;
	}

	spk_codec_node = of_parse_phandle(pdev->dev.of_node,
					  "mediatek,speaker-codec", 0);
	if (!spk_codec_node) {
		dev_err(&pdev->dev,
			"Property 'speaker-codec' missing or invalid\n");
		return -EINVAL;
	}
	
#endif

	for (i = 0; i < card->num_links; i++) {
		if (mt6880_mt6359_dai_links[i].codec_name)
			continue;
#if defined(CONFIG_SND_SOC_PROSLIC)
		if (!strcmp(mt6880_mt6359_dai_links[i].name, "ETDM"))
			mt6880_mt6359_dai_links[i].codec_of_node =
				ext_codec_node;
#endif
	}
	
//tianyan@2021.8.2 modify for compatible codec tlv320aic3104 and add loopback test start
#if defined(CONFIG_SND_SOC_NAU8810) || defined(CONFIG_SND_SOC_TLV320AIC3X)

#ifdef CONFIG_SND_SOC_TLV320AIC3X
	if(!mt6880_regmap)
	{
		mt6880_regmap = aic3x_get_aic3x_config();
	}
#endif

#ifdef CONFIG_SND_SOC_NAU8810
//tianyan@2021.7.16 modify for compatible codec tlv320aic3104 start
	if(!mt6880_regmap){
		mt6880_regmap = nau8810_get_nau8810_config();
		g_mt6880_codec_type = MT6880_CODEC_TYPE_NAU8810;
		printk("change to codec nau8810\n");
		for (i = 0; i < card->num_links; i++) {
			if(!strcmp(mt6880_mt6359_dai_links[i].name, "I2S0")){
				mt6880_mt6359_dai_links[i].codec_dai_name = "nau8810-hifi";
				mt6880_mt6359_dai_links[i].codec_name = "nau8810.0-001a";
			}
			if(!strcmp(mt6880_mt6359_dai_links[i].name, "I2S3")){
				mt6880_mt6359_dai_links[i].codec_dai_name = "nau8810-hifi";
				mt6880_mt6359_dai_links[i].codec_name = "nau8810.0-001a";
			}
		}	
	}
//tianyan@2021.7.16 modify for compatible codec tlv320aic3104 end
#endif

	if(mt6880_regmap){
		ret = alloc_chrdev_region(&devno,0,1,"mt6880_chdev_devno");
		if(ret <0){
			dev_info(&pdev->dev,"%s alloc_chrdev_region err\n",__func__);
			return ret;
		}

		cdev_init(&mt6880_codec_cdev,&mt6880_chdev_ops);
		ret = cdev_add(&mt6880_codec_cdev,devno,1);
		if(ret <0){
			dev_info(&pdev->dev,"%s cdev_add err\n",__func__);
			return ret;
		}

		mt6880_codec_cls = class_create(THIS_MODULE,"mt6880_codec_class");
		if(!mt6880_codec_cls){
			dev_info(&pdev->dev,"%s class_create err\n",__func__);
			return -EINVAL;
		}

		device_create(mt6880_codec_cls,NULL,devno,NULL,"mt6880_codec_dev");
	}
#endif
//tianyan@2021.8.2 modify for compatible codec tlv320aic3104 and add loopback test end

	ret = devm_snd_soc_register_card(&pdev->dev, card);
	if (ret)
		dev_err(&pdev->dev, "%s snd_soc_register_card fail %d\n",
			__func__, ret);

	dev_info(&pdev->dev, "%s(), ret %d\n", __func__, ret);
	return ret;
}

#ifdef CONFIG_OF
static const struct of_device_id mt6880_mt6359_dt_match[] = {
	{.compatible = "mediatek,mt6880-mt6359-sound",},
	{}
};
#endif

static const struct dev_pm_ops mt6880_mt6359_pm_ops = {
	.poweroff = snd_soc_poweroff,
	.restore = snd_soc_resume,
};

static struct platform_driver mt6880_mt6359_driver = {
	.driver = {
		.name = "mt6880-mt6359",
#ifdef CONFIG_OF
		.of_match_table = mt6880_mt6359_dt_match,
#endif
		.pm = &mt6880_mt6359_pm_ops,
	},
	.probe = mt6880_mt6359_dev_probe,
};

module_platform_driver(mt6880_mt6359_driver);

/* Module information */
MODULE_DESCRIPTION("MT6880 MT6359 ALSA SoC machine driver");
MODULE_AUTHOR("Eason Yen <eason.yen@mediatek.com>");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("mt6880 mt6359 soc card");
