/*
 * Copyright (C) 2016 MediaTek Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-2.0.html for more details.
 */
/*! \file
*    \brief  Declaration of library functions
*
*    Any definitions in this file will be shared among GLUE Layer and internal Driver Stack.
*/

#define pr_fmt(fmt) KBUILD_MODNAME "@(%s:%d) " fmt, __func__, __LINE__

#include <linux/delay.h>
#include <linux/memblock.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/thermal.h>
#include <mtk-clkbuf-bridge.h>

#include <connectivity_build_in_adapter.h>

#include "osal.h"
#include "conninfra.h"
#include "conninfra_conf.h"
#include "consys_hw.h"
#include "consys_reg_mng.h"
#include "consys_reg_util.h"
#include "mt6880.h"
#include "emi_mng.h"
#include "mt6880_consys_reg.h"
#include "mt6880_consys_reg_offset.h"
#include "mt6880_pos.h"

/*******************************************************************************
*                         C O M P I L E R   F L A G S
********************************************************************************
*/

/*******************************************************************************
*                                 M A C R O S
********************************************************************************
*/
#define PLATFORM_SOC_CHIP 0x6880

/*******************************************************************************
*                    E X T E R N A L   R E F E R E N C E S
********************************************************************************
*/

/*******************************************************************************
*                              C O N S T A N T S
********************************************************************************
*/

/*******************************************************************************
*                             D A T A   T Y P E S
********************************************************************************
*/

/*******************************************************************************
*                  F U N C T I O N   D E C L A R A T I O N S
********************************************************************************
*/

static int consys_clk_get_from_dts(struct platform_device *pdev);
static int consys_clk_detach(void);
static int consys_clock_buffer_ctrl(unsigned int enable);
static unsigned int consys_soc_chipid_get(void);
static unsigned int consys_get_hw_ver(void);
static void consys_clock_fail_dump(void);
static int consys_thermal_query(void);
static int consys_power_state(void);
static int consys_thermal_register(struct platform_device *pdev, struct conninfra_dev_cb* dev_cb);
static int consys_thermal_get_temp_cb(void*, int*);

/*******************************************************************************
*                            P U B L I C   D A T A
********************************************************************************
*/
struct consys_hw_ops_struct g_consys_hw_ops_mt6880 = {
	/* load from dts */
	/* TODO: mtcmos should move to a independent module */
	.consys_plt_clk_get_from_dts = consys_clk_get_from_dts,
	.consys_plt_clk_detach = consys_clk_detach,

	/* clock */
	.consys_plt_clock_buffer_ctrl = consys_clock_buffer_ctrl,
	.consys_plt_co_clock_type = consys_co_clock_type,

	/* POS */
	.consys_plt_conninfra_on_power_ctrl = consys_conninfra_on_power_ctrl,
	.consys_plt_set_if_pinmux = consys_set_if_pinmux,

	.consys_plt_polling_consys_chipid = consys_polling_chipid,
	.consys_plt_d_die_cfg = connsys_d_die_cfg,
	.consys_plt_spi_master_cfg = connsys_spi_master_cfg,
	.consys_plt_a_die_cfg = connsys_a_die_cfg,
	.consys_plt_afe_wbg_cal = connsys_afe_wbg_cal,
	.consys_plt_subsys_pll_initial = connsys_subsys_pll_initial,
	.consys_plt_low_power_setting = connsys_low_power_setting,
	.consys_plt_soc_chipid_get = consys_soc_chipid_get,
	.consys_plt_conninfra_wakeup = consys_conninfra_wakeup,
	.consys_plt_conninfra_sleep = consys_conninfra_sleep,
	.consys_plt_is_rc_mode_enable = consys_is_rc_mode_enable,

	/* debug */
	.consys_plt_clock_fail_dump = consys_clock_fail_dump,
	.consys_plt_get_hw_ver = consys_get_hw_ver,

	.consys_plt_spi_read = consys_spi_read,
	.consys_plt_spi_write = consys_spi_write,
	.consys_plt_adie_top_ck_en_on = NULL,
	.consys_plt_adie_top_ck_en_off = NULL,
	.consys_plt_spi_clock_switch = NULL,
	.consys_plt_subsys_status_update = NULL,

	.consys_plt_thermal_query = consys_thermal_query,
	.consys_plt_thermal_register = consys_thermal_register,
	.consys_plt_power_state = consys_power_state,
	.consys_plt_config_setup = NULL,
	.consys_plt_bus_clock_ctrl = NULL,
};


struct consys_plat_thermal_data g_consys_plat_therm_data;

/* For mt6880 */
extern struct consys_hw_ops_struct g_consys_hw_ops_mt6880;
extern struct consys_reg_mng_ops g_dev_consys_reg_ops_mt6880;
extern struct consys_platform_emi_ops g_consys_platform_emi_ops_mt6880;
extern struct consys_platform_pmic_ops g_consys_platform_pmic_ops_mt6880;

const struct conninfra_plat_data mt6880_plat_data = {
	.chip_id = PLATFORM_SOC_CHIP,
	.hw_ops = &g_consys_hw_ops_mt6880,
	.reg_ops = &g_dev_consys_reg_ops_mt6880,
	.platform_emi_ops = &g_consys_platform_emi_ops_mt6880,
	.platform_pmic_ops = &g_consys_platform_pmic_ops_mt6880,
};

/*******************************************************************************
*                           P R I V A T E   D A T A
********************************************************************************
*/
#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
static struct platform_device *connsys_pdev;
#endif

static const struct thermal_zone_of_device_ops tz_conninfra_thermal_ops = {
	.get_temp = consys_thermal_get_temp_cb,
};

/*******************************************************************************
*                              F U N C T I O N S
********************************************************************************
*/

/* mtcmos contorl */
int consys_clk_get_from_dts(struct platform_device *pdev)
{
#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
	if (pdev == NULL) {
		pr_err("%s pdev is null", __func__);
		return -1;
	}
	connsys_pdev = pdev;
	pm_runtime_enable(&connsys_pdev->dev);
#else
	pr_info("MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE not support");

#endif /* #if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE */
	return 0;
}

int consys_clk_detach(void)
{
#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
	if (connsys_pdev == NULL)
		return 0;
	pm_runtime_disable(&connsys_pdev->dev);
#endif
	return 0;
}

int consys_platform_spm_conn_ctrl(unsigned int enable)
{
	int ret = 0;

#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
	if (enable) {
		ret = pm_runtime_get_sync(&connsys_pdev->dev);
		if (ret) {
			pr_err("Turn on oonn_infra power fail. Ret=%d", ret);
			return -1;
		}
	} else {
		ret = pm_runtime_put_sync(&connsys_pdev->dev);
		if (ret) {
			pr_err("Turn off conn_infra power fail. Ret=%d", ret);
			return -1;
		}

	}
#else
	pr_err("Function: %s is not support", __func__);
#endif
	return ret;
}

int consys_clock_buffer_ctrl(unsigned int enable)
{
	/* This function call didn't work now.
	 * clock buffer is HW controlled, not SW controlled.
	 * Keep this function call to update status.
	 */
	if (enable)
		clk_buf_ctrl(CLK_BUF_CONN, true);	/*open XO_WCN*/
	else
		clk_buf_ctrl(CLK_BUF_CONN, false);	/*close XO_WCN*/
	return 0;
}

int consys_co_clock_type(void)
{
	const struct conninfra_conf *conf;
	bool conf_tcxo = false;

	/* Default solution */
	conf = conninfra_conf_get_cfg();
	if (conf != NULL && conf->tcxo_gpio != 0) {
		conf_tcxo = true;
	}

	/* TODO: for co-clock mode, there are two case: 26M and 52M. Need something to distinguish it. */
	if (conf_tcxo || conn_hw_env.tcxo_support)
		return CONNSYS_CLOCK_SCHEMATIC_26M_EXTCXO;
	else
		return CONNSYS_CLOCK_SCHEMATIC_26M_COTMS;
}

unsigned int consys_soc_chipid_get(void)
{
	return PLATFORM_SOC_CHIP;
}

unsigned int consys_get_hw_ver(void)
{
	return CONN_HW_VER;
}

void consys_clock_fail_dump(void)
{
	pr_info("[%s]", __func__);
}


void update_thermal_data(struct consys_plat_thermal_data* input)
{
	memcpy(&g_consys_plat_therm_data, input, sizeof(struct consys_plat_thermal_data));
#if CFG_CONNINFRA_THERMAL_SUPPORT
	/* Special factor, not in POS */
	/* THERMCR1 [16:17]*/
	CONSYS_REG_WRITE(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERMCR1,
			(CONSYS_REG_READ(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERMCR1) |
				(0x3 << 16)));
#endif
}

#if CFG_CONNINFRA_THERMAL_SUPPORT
int calculate_thermal_temperature(int y)
{
	struct consys_plat_thermal_data *data = &g_consys_plat_therm_data;
	int t;
	int const_offset = 25;

	/*
	 *    MT6635 E1 : read 0x02C = 0x66358A00
	 *    MT6635 E2 : read 0x02C = 0x66358A10
	 *    MT6635 E3 : read 0x02C = 0x66358A11
	 */
	if (conn_hw_env.adie_hw_version == 0x66358A10 ||
		conn_hw_env.adie_hw_version == 0x66358A11)
		const_offset = 28;

	/* temperature = (y-b)*slope + (offset) */
	/* TODO: offset + 25 : this is only for E1, E2 is 28 */
	t = (y - (data->thermal_b == 0 ? 0x36 : data->thermal_b)) *
			((data->slop_molecule + 209) / 100) + (data->offset + const_offset);

	pr_info("y=[%d] b=[%d] constOffset=[%d] [%d] [%d] => t=[%d]\n",
			y, data->thermal_b, const_offset, data->slop_molecule, data->offset,
			t);

	return t;
}
#endif /* CFG_CONNINFRA_THERMAL_SUPPORT */

int consys_thermal_query(void)
{
#if CFG_CONNINFRA_THERMAL_SUPPORT

#define THERMAL_DUMP_NUM	11
#define LOG_TMP_BUF_SZ		256
#define TEMP_SIZE		13
	void __iomem *addr = NULL;
	int cal_val, res = 0;
	/* Base: 0x1800_2000, CONN_TOP_THERM_CTL */
	const unsigned int thermal_dump_crs[THERMAL_DUMP_NUM] = {
		0x00, 0x04, 0x08, 0x0c,
		0x10, 0x14, 0x18, 0x1c,
		0x20, 0x24, 0x28,
	};
	char tmp[TEMP_SIZE] = {'\0'};
	char tmp_buf[LOG_TMP_BUF_SZ] = {'\0'};
	unsigned int i;

	addr = ioremap_nocache(CONN_INFRA_GPT_BASE, 0x100);
	if (addr == NULL) {
		pr_err("GPT2_CTRL_BASE remap fail");
		return -1;
	}

	/* Hold Semaphore, TODO: may not need this, because
		thermal cr seperate for different  */
	if (consys_sema_acquire_timeout(CONN_SEMA_THERMAL_INDEX, CONN_SEMA_TIMEOUT) == CONN_SEMA_GET_FAIL) {
		pr_err("[THERM QRY] Require semaphore fail\n");
		iounmap(addr);
		return -1;
	}

	/* therm cal en */
	CONSYS_SET_BIT(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERM_CAL_EN, (0x1 << 19));
	/* GPT2 En */
	CONSYS_SET_BIT(addr + GPT2_AP_ENABLE, 0x1);

	/* thermal trigger */
	CONSYS_SET_BIT(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERM_CAL_EN, (0x1 << 18));
	udelay(500);
	/* get thermal value */
	cal_val = CONSYS_REG_READ(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERM_CAL_EN);
	cal_val = (cal_val >> 8) & 0x7f;

	/* thermal debug dump */
	for (i = 0; i < THERMAL_DUMP_NUM; i++) {
		snprintf(
			tmp, TEMP_SIZE, "[0x%08x]",
			CONSYS_REG_READ(REG_CONN_TOP_THERM_CTL_ADDR + thermal_dump_crs[i]));
		strncat(tmp_buf, tmp, strlen(tmp));
	}
	res = calculate_thermal_temperature(cal_val);
	pr_info("[%s] temp=%dC (%dmC) efuse:[0x%08x][0x%08x][0x%08x][0x%08x] thermal dump: %s",
		__func__, res, res*1000,
		g_consys_plat_therm_data.efuse0, g_consys_plat_therm_data.efuse1,
		g_consys_plat_therm_data.efuse2, g_consys_plat_therm_data.efuse3,
		tmp_buf);

	/* GPT2 disable, no effect on 6880 */
	CONSYS_CLR_BIT(addr + GPT2_AP_ENABLE, 0x1);

	/* disable */
	CONSYS_CLR_BIT(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERM_CAL_EN, (0x1 << 19));

	consys_sema_release(CONN_SEMA_THERMAL_INDEX);

	iounmap(addr);

	return (res*1000);
#else
	return THERMAL_TEMP_INVALID;
#endif /* CFG_CONNINFRA_THERMAL_SUPPORT */

}

int consys_thermal_get_temp_cb(void* data, int* temp)
{
	struct conninfra_dev_cb *dev_cb = data;

	if (dev_cb && dev_cb->conninfra_thermal_query_cb) {
		dev_cb->conninfra_thermal_query_cb(NULL, temp);
	} else {
		pr_info("Thermal callback is not support");
		if (temp)
			*temp = THERMAL_TEMP_INVALID;
	}

	return 0;
}


int consys_thermal_register(struct platform_device *pdev, struct conninfra_dev_cb* dev_cb)
{
	struct thermal_zone_device *tzdev;

	if (dev_cb->conninfra_thermal_query_cb == NULL)
		pr_info("Thermal callback is not support");

	/* register thermal zone */
	tzdev = devm_thermal_zone_of_sensor_register(
		&pdev->dev, 0, dev_cb, &tz_conninfra_thermal_ops);

	return 0;
}

int consys_power_state(void)
{
#if CFG_CONNINFRA_POWER_STATUS_SUPPORT
	const char* osc_str[] = {
		"fm ", "gps ", "bgf ", "wf ", "ap2conn ", "conn_thm ", "conn_pta ", "conn_infra_bus "
	};
	char buf[256] = {'\0'};
	int r;
	int i;

	/* Setup debug select to power state
	 * debug sel: 0x1806015C[2:0]=3'b000 (default value)
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_CFG_DBG_SEL,
		0x0, 0x7);

	r = CONSYS_REG_READ(REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_2);
	for (i = 0; i < 8; i++) {
		if ((r & (0x1 << (18 + i))) > 0)
			strncat(buf, osc_str[i], strlen(osc_str[i]));
	}
	pr_info("[%s] [0x%x] %s", __func__, r, buf);
#endif
	return 0;
}

