/*
 * Copyright (C) 2016 MediaTek Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-2.0.html for more details.
 */
/*! \file
*    \brief  Declaration of library functions
*
*    Any definitions in this file will be shared among GLUE Layer and internal Driver Stack.
*/

#include <linux/memblock.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/random.h>

#include <connectivity_build_in_adapter.h>

#include "consys_reg_mng.h"
#include "mt6880_consys_reg.h"
#include "mt6880_consys_reg_offset.h"
#include "consys_hw.h"
#include "consys_reg_util.h"

#define LOG_TMP_BUF_SZ 256
#define CONSYS_POWER_MODE_LEGACY "Legacy"
#define CONSYS_POWER_MODE_RC "RC"

static int consys_reg_init(struct platform_device *pdev);
static int consys_reg_deinit(void);
static int consys_check_reg_readable(void);
static int consys_is_consys_reg(unsigned int addr);
static int consys_is_bus_hang(void);
static int consys_dump_bus_status(void);
static int consys_dump_conninfra_status(void);
static int consys_is_host_csr(unsigned long addr);

struct consys_reg_mng_ops g_dev_consys_reg_ops_mt6880 = {
	.consys_reg_mng_init = consys_reg_init,
	.consys_reg_mng_deinit = consys_reg_deinit,

	.consys_reg_mng_check_reable = consys_check_reg_readable,
	.consys_reg_mng_is_consys_reg = consys_is_consys_reg,
	.consys_reg_mng_is_bus_hang = consys_is_bus_hang,
	.consys_reg_mng_dump_bus_status = consys_dump_bus_status,
	.consys_reg_mng_dump_conninfra_status = consys_dump_conninfra_status,
	.consys_reg_mng_dump_cpupcr = NULL,
	.consys_reg_mng_is_host_csr = consys_is_host_csr,

	/* step */
	.consys_reg_mng_validate_idx_n_offset = NULL,
	.consys_reg_mng_find_can_write_reg = NULL,
	.consys_reg_mng_get_phy_addr_by_idx = NULL,
	.consys_reg_mng_get_virt_addr_by_idx = NULL,
	.consys_reg_mng_get_chip_id_idx_offset = NULL,
	.consys_reg_mng_get_reg_symbol_num = NULL
};


const char* consys_base_addr_index_to_str[CONSYS_BASE_ADDR_MAX] = {
	"CONN_INFRA_RGU_BASE",
	"CONN_INFRA_CFG_BASE",
	"CONN_HOST_CSR_TOP_BASE",
	"INFRACFG_AO_BASE",
	"TOPRGU_BASE",
	"SPM_BASE",
	"INFRACFG_BASE",
	"CONN_WT_SLP_CTL_REG",
	"CONN_AFE_CTL_REG",
	"GPIO",
	"CONN_RF_SPI_MST_REG",
	"CONN_SEMAPHORE",
	"CONN_TOP_THERM_CTL",
	"IOCFG_BM",
	"CONN_DEBUG_CTL",
	"CONN_INFRA_CLKGEN_ON_TOP",
	"CONN_INFRA_BUS_CR",
	"CONN_INFRA_DEBUG_CTRL_AO",
};

struct consys_base_addr conn_reg;

struct consys_base_addr* get_conn_reg_base_addr()
{
	return &conn_reg;
}
#if CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT
static void consys_bus_hang_dump_c_host_csr(void)
{
	unsigned int r1, r2, r3, r4, r5, r6, r7, r8;
	char tmp[LOG_TMP_BUF_SZ] = {'\0'};
	char tmp_buf[LOG_TMP_BUF_SZ] = {'\0'};
	unsigned int i;

	/* CONN_INFRA_DBG No.5: gps2conn gals
	 * Read gals status for debug
	 * Dump 1800_E2B0 for gals slave
	 */
	r1 = CONSYS_REG_READ(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_CR_GALS_GPS2CONN_GALS_DBG);

	/* CONN_INFRA_DBG No.6: conn2gps gals
	 * Read gals status for debug
	 * 1. Read 1800_E2B4 (GALS_CONN2GPS_GALS_DBG)
	 * 2. Read 1800_E2CC[7:6] for gals idle signal
	 */
	r2 = CONSYS_REG_READ(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_CR_GALS_CONN2GPS_GALS_DBG);
	r3 = CONSYS_REG_READ(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_CR_GALS_CONN2GPS_CTRL_0);

	/* CONN_INFRA_DBG No.7: conn_infra timeout mechanism
	 * 1. off domain AHB slaves hang result to timeout
	 * 	- Read 1806_0414, 1806_0418
	 * 2. on domain APB slaves hang result to timeout
	 * 	- Read 1806_042c, 1806_0430
	 */
	r4 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_BUS_OFF_DBG_1);
	r5 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_BUS_OFF_DBG_2);
	r6 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_BUS_ON_TOP_DBG_APB_1);
	r7 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_BUS_ON_TOP_DBG_APB_2);

	/* CONN_INFRA_DBG No.8: bus_debug_monitor
	 * 1. Read 1802_F000[8] to confirm if the irq is resulted by bus hang
	 *    timeout mechanisim
	 * 2. Read 0x 1802_F408 ~ 0x1802_F424 to see the command information
	 *    which resulting bus hang
	 */
	r8 = CONSYS_REG_READ(
		REG_CONN_INFRA_DEBUG_CTRL_AO + CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0);
	for (i = 0x408; i <= 0x424; i += 4) {
		snprintf(tmp, LOG_TMP_BUF_SZ, "[%x]",
			CONSYS_REG_READ(REG_CONN_INFRA_DEBUG_CTRL_AO + i));
		strncat(tmp_buf, tmp, strlen(tmp));
	}

	pr_info("[CONN_BUS_C_1][%x][%x][%x][%x][%x][%x][%x][%x]",
		r1, r2, r3, r4, r5, r6, r7, r8);
	pr_info("[CONN_BUS_C_2]%s", tmp_buf);
}
#endif

static void consys_bus_hang_dump_c(void)
{
#if CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT
	char tmp[LOG_TMP_BUF_SZ] = {'\0'};
	char tmp_buf[LOG_TMP_BUF_SZ] = {'\0'};
	int i;
	const static unsigned int conninfra_cfg_dump_list[] =
		{0x504, 0x514, 0x524, 0x534, 0x574, 0x584};
	const static unsigned int conninfra_bus_cr[] =
		{0x2a0, 0x2a4, 0x2b0, 0x2b4, 0x2a0, 0x2a4};
	void __iomem *addr = NULL;
	unsigned int ary12, ary13;

	consys_bus_hang_dump_c_host_csr();

	/* ary12: 0x1020_E810	CONN2AP_GALS_SLV_DBG
	 * ary13: 0x1020_E814	AP2CONN_GALS_MST_DBG
	 */
	addr = ioremap_nocache(0x1020E810, 0x20);
	if (addr != NULL) {
		ary12 = CONSYS_REG_READ(addr);
		ary13 = CONSYS_REG_READ(addr + 0x4);
		iounmap(addr);
	} else {
		ary12 = 0xdeaddead;
		ary13 = 0xdeaddead;
		pr_err("[%s] ioremap error", __func__);
	}

	/* CONN_INFRA_DBG No.2: sleep protect
	 * ary0: 0x1800_1504
	 * ary1: 0x1800_1514
	 * ary2: 0x1800_1524
	 * ary3: 0x1800_1534
	 * ary4: 0x1800_1574
	 * ary5: 0x1800_1584
	 * ary6: 0x1800_E2A0
	 * ary7: 0x1800_E2A4
	 * ary8: 0x1800_E2B0
	 * ary9: 0x1800_E2B4
	 */
	/* CONN_INFRA_DBG No.3: ap2conn gals
	 * Read gals status for debug
	 * ary10: Read 1800_E2A0 for gals slave
	 */

	/* CONN_INFRA_DBG No.4: conn2ap gals
	 * Read gals status for debug
	 * ary11: Read 1800_E2A4 for gals master
	 */

	for (i = 0; i < sizeof(conninfra_cfg_dump_list)/sizeof(unsigned int); i++)
	{
		snprintf(tmp, LOG_TMP_BUF_SZ, "[%x]",
			CONSYS_REG_READ(REG_CONN_INFRA_CFG_ADDR + conninfra_cfg_dump_list[i]));
		strncat(tmp_buf, tmp, strlen(tmp));
	}

	for (i = 0; i < sizeof(conninfra_bus_cr)/sizeof(unsigned int); i++)
	{
		snprintf(tmp, LOG_TMP_BUF_SZ, "[%x]",
			CONSYS_REG_READ(REG_CONN_INFRA_BUS_CR_ADDR + conninfra_bus_cr[i]));
		strncat(tmp_buf, tmp, strlen(tmp));
	}
	pr_info("[CONN_BUS_C]%s[%x][%x]", tmp_buf, ary12, ary13);
#endif
}

static void consys_bus_hang_dump_a(void)
{
#if CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT
	unsigned int r1, r2, r7, r8, r9, r10, r11, r12, r13;
	void __iomem *addr = NULL;

	/* SPM CR */
	/* r1: PCM_REG13_DATA 0x1000_6110
	 * [4]: conn_ddr_en
	 * [13]: conn_state
	 * [14]: conn_srcclkena
	 * [15]: conn_apsrc_req
	 */
	r1 = CONSYS_REG_READ(
		REG_SPM_BASE_ADDR + SPM_PCM_REG13_DATA);

	/* r2: SRC_REQ_STA_0 0x1000_6114
	 * [12]: CONN_SRCCLKENA
	 * [14]: CONN_INFRA_REQ
	 * [15]: CONN_APSRC_REQ
	 * [16]: CONN_VRF18_REQ
	 * [17]: CONN_DDR_EN
	 */
	r2 = CONSYS_REG_READ(
		REG_SPM_BASE_ADDR + SPM_SRC_REQ_STA_0);

	/* r7: SPM debug CR 0x1000_6100
	 * 0x10006100[0] sc_26m_ck_off
	 * 	1'b0: 26M on; 1'b1
	 * 0x10006100[3] sc_axi_ck_off
	 * 	1'b0: bus ck on; 1'b1 bus ck off
	 * 0x10006100[5] sc_md26m_ck_off
	 * 	1'b0: MD 26M on; 1'b1 MD 26M off
	 * 0x10006100[20] sc_cksq0_off
	 * 	1'b0: clock square0 on; 1'b1 clock square off
	 */
	r7 = CONSYS_REG_READ(
		REG_SPM_BASE_ADDR + SPM_PCM_REG0_DATA);

	/*
	 *  0x1000_6304[2] : pwr_on
	 *  0x1000_616c[1] : pwr_ack
	 *  0x1000_6304[3] : pwr_on_s
	 *  0x1000_6170[1] : pwr_ack_s
	 *  0x1000_6304[1] : iso_en
	 *  0x1000_6304[0] : ap_sw_rst
	 * r8:  0x1000_616C
	 * r9:  0x1000_6170
	 * r10: 0x1000_6304
	 */
	r8 = CONSYS_REG_READ(
		REG_SPM_BASE_ADDR + SPM_PWR_STATUS);
	r9 = CONSYS_REG_READ(
		REG_SPM_BASE_ADDR + SPM_PWR_STATUS_2ND);
	r10 = CONSYS_REG_READ(
		REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON);

	/* r11: TOPCKGEN infra bus clock 0x10000010[2:0]
	 * 0: tck_26m_mx9_ck => 26M
	 * 1: mainpll_d4_d4 => 136.5M
	 * 2: mainpll_d7_d2 => 156M
	 * 3: mainpll_d4_d2 => 273M
	 * 4: mainpll_d5_d2 => 218.4M
	 * 5: mainpll_d6_d2 => 182M
	 * 6: osc_d4 => 65M
	 */
	addr = ioremap_nocache(0x10000000, 0x20);
	if (addr != NULL) {
		r11 = CONSYS_REG_READ(addr + 0x10);
		iounmap(addr);
	} else {
		r11 = 0xdeaddead;
		pr_err("[%s] ioremap error", __func__);
	}

	/* ap2conn gals sleep protect status
	 * - 0x1000_1220 [13] / 0x1000_1228 [13] (rx/tx) (sleep protect enable ready)
	 *   both of them should be 1'b0  (CR at ap side)
	 *  r12 : 0x1000_1220 (rx)
	 *  r13 : 0x1000_1228 (tx)
	 */
	r12 = CONSYS_REG_READ(
		REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN);
	r13 = CONSYS_REG_READ(
		REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1);

	pr_info("[CONN_BUS_A] %s [%x][%x] [%x][%x][%x][%x] [%x] [%x][%x]",
		CONSYS_POWER_MODE_LEGACY, r1, r2,
		r7, r8, r9, r10,
		r11,
		r12, r13);
#endif
}

static void consys_bus_hang_dump_b(void)
{
#if CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT
	unsigned int r1, r2, r3, r4, r5, r6, r7;

	/* r1: 0x180602C0
	 * 	[4]	conn_srcclkena_ack
	 * 	[5]	conn_ap_bus_ack
	 * 	[6]	conn_apsrc_ack
	 * 	[7]	conn_ddr_en_ack
	 */
	r1 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_0);

	/*****************
	 *  Debug sel
	 *****************/
	/* r2: 0x1806015C[2:0] = 3'b000 (default)
	 * Dump 0x180602C8
	 * [25]: conn_infra_bus_osc_en
	 * [24]: conn_pta_osc_en
	 * [23]: conn_thm_osc_en
	 * [22]: ap2conn_osc_en
	 * [21]: wfsys_osc_en
	 * [20]: bgfsys_osc_en
	 * [19]: gpssys_osc_en
	 * [18]: fmsys_osc_en
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_CFG_DBG_SEL,
		0x0, 0x7);
	r2 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_2);

	/* r3: 0x1806015C[2:0] = 3'b010
	 * Dump 0x180602C8
	 * [12]: conn_srcclkena_0_bblpm_ack
	 * [13]: conn_srcclkena_0_fpm_ack
	 * [28]: conn_srcclkena_1_bblpm_ack
	 * [29]: conn_srcclkena_1_fpm_ack
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_CFG_DBG_SEL,
		0x2, 0x7);
	r3 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_2);

	/* r4: 0x1806015C[2:0] = 3'b011
	 * Dump 0x180602C8
	 * [12]: conn_srcclkena_2_bblpm_ack
	 * [13]: conn_srcclkena_2_fpm_ack
	 * [28]: conn_srcclkena_3_bblpm_ack
	 * [29]: conn_srcclkena_3_fpm_ack
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_CFG_DBG_SEL,
		0x3, 0x7);
	r4 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_2);

	/* r5: MTCMOS
	 * 0x1806_02CC[15]: power_enable
	 * 0x1806_02CC[14]: pwr_on
	 * 	1'b0: power off; 1'b1: power on
	 * 0x1806_02CC[13]: pwr_ack
	 * 	1'b0: power off; 1'b1: power on
	 * 0x1806_02CC[12]: pwr_on_s
	 * 	1'b0: power off; 1'b1: power on
	 * 0x1806_02CC[11]: pwr_ack_s
	 * 	1'b0: power off; 1'b1: power on
	 * 0x1806_02CC[10]: iso_en
	 * 	1'b0: power on; 1'b1: power off
	 * 0x1806_02CC[9]: conn_infra_off_xreset_b
	 * 	1'b0: power off; 1'b1: power on
	 * 0x1806_02CC[8]: conn_infra_off_hreset_b
	 * 1'b0: power off; 1'b1: power on
	 */
	r5 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_3);

	/* Check conn_infra off bus clock
	 * - write 0x1 to 0x1806_0000[0], reset clock detect
	 * - 0x1806_0000[2]  conn_infra off bus clock (should be 1'b1 if clock exist)
	 * - 0x1806_0000[1]  osc clock (should be 1'b1 if clock exist)
	 */
	CONSYS_SET_BIT(REG_CONN_HOST_CSR_ADDR, 0x1);
	udelay(1);
	r6 = CONSYS_REG_READ(REG_CONN_HOST_CSR_ADDR);

	/* Check conn_infra off domain bus hang irq status
	 * - 0x1806_02D4[0], should be 1'b1, or means conn_infra off bus might hang
	 */
	r7 = CONSYS_REG_READ(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_5);

	pr_info("[CONN_BUS_B] %s [%x] [%x][%x][%x] [%x] [%x][%x]",
		CONSYS_POWER_MODE_LEGACY, r1,
		r2, r3, r4, /* debug select */
		r5,
		r6, r7);
#endif
}

static int consys_dump_bus_status(void)
{
	consys_bus_hang_dump_c();
	return 0;
}

static int consys_dump_conninfra_status(void)
{
	consys_bus_hang_dump_a();
	consys_bus_hang_dump_b();
	return 0;
}

static int consys_is_bus_hang(void)
{
	int ret = 0;
#if CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT
	int r;

	/* dump SPM register */
	consys_bus_hang_dump_a();

	/* ap2conn gals sleep protect status
	 * - 0x1000_1220 [13] / 0x1000_1228 [13] (rx/tx) (sleep protect enable ready)
	 *   both of them should be 1'b0 ~(CR at ap side)
	 *  0x1000_1220 (rx)
	 *  0x1000_1228 (tx)
	 */
	r = CONSYS_REG_READ_BIT(
		REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN,
		(0x1 << 13));
	if (r != 0)
		return CONNINFRA_AP2CONN_RX_SLP_PROT_ERR;
	r = CONSYS_REG_READ_BIT(
		REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1,
		(0x1 << 13));
	if (r != 0)
		return CONNINFRA_AP2CONN_TX_SLP_PROT_ERR;

	consys_bus_hang_dump_b();

	/* AP2CONN_INFRA OFF
	 * 1. Check "AP2CONN_INFRA ON step is ok"
	 * 2. Check conn_infra off bus clock
	 *    - write 0x1 to 0x1806_0000[0], reset clock detect
	 *    - 0x1806_0000[2] conn_infra off bus clock (should be 1'b1 if clock exist)
	 *    - 0x1806_0000[1] osc clock (should be 1'b1 if clock exist)
	 * 3. Check conn_infra off domain bus hang irq status
	 *   - 0x1806_02d4[0], should be 1'b1, or means conn_infra off bus might hang
	 */
	CONSYS_SET_BIT(REG_CONN_HOST_CSR_ADDR, 0x1);
	udelay(1);
	r = CONSYS_REG_READ(REG_CONN_HOST_CSR_ADDR);
	if ((r & TOP_BUS_MUC_STAT_HCLK_FR_CK_DETECT_BIT) == 0 ||
	    (r & TOP_BUS_MUC_STAT_OSC_CLK_DETECT_BIT) == 0) {
		pr_info("[%s] readable fail = bus clock", __func__);
		consys_bus_hang_dump_c_host_csr();
		return 0;
	}

	r = CONSYS_REG_READ_BIT(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_5, 0x1);
	if (r == 0x0) {
		ret |= CONNINFRA_INFRA_BUS_HANG_IRQ;
	}

	consys_bus_hang_dump_c();
#endif /* CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT */
	return ret;

}

int consys_check_reg_readable(void)
{
#if CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT
	int r;
	unsigned int rnd;
	int delay_time = 1, spent = delay_time, max_wait = 16000; // 1.6 ms
	int retry = max_wait / 10;

	/* AP2CONN_INFRA ON */
	/* 1. Check ap2conn gals sleep protect status
	 * - 0x1000_1220 [13] / 0x1000_1228 [13](rx/tx) (sleep protect enable ready)
	 *   both of them should be 1'b0  (CR at ap side)
	 */
	r = CONSYS_REG_READ_BIT(
		REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN,
		(0x1 << 13));
	if (r != 0)
		return 0;

	r = CONSYS_REG_READ_BIT(
		REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1,
		(0x1 << 13));
	if (r != 0)
		return 0;

	/* AP2CONN_INFRA OFF
	 * 1. Check "AP2CONN_INFRA ON step is ok"
	 * 2. Check conn_infra off bus clock
	 *    - write 0x1 to 0x1806_0000[0], reset clock detect
	 *    - 0x1806_0000[2] conn_infra off bus clock (should be 1'b1 if clock exist)
	 *    - 0x1806_0000[1] osc clock (should be 1'b1 if clock exist)
	 * 3. Check conn_infra off domain bus hang irq status
	 *   - 0x1806_02d4[0], should be 1'b1, or means conn_infra off bus might hang
	 */
	while (retry > 0 && spent < max_wait) {
		CONSYS_SET_BIT(REG_CONN_HOST_CSR_ADDR, 0x1);
		udelay(delay_time);
		r = CONSYS_REG_READ(REG_CONN_HOST_CSR_ADDR);
		if ((r & TOP_BUS_MUC_STAT_HCLK_FR_CK_DETECT_BIT) == 0 ||
			(r & TOP_BUS_MUC_STAT_OSC_CLK_DETECT_BIT) == 0) {
			spent += delay_time;
			retry--;
			if (retry == 0)
				pr_info("[%s] retry=0 r=[%x]", __func__, r);
			else
				delay_time = 10;
			rnd = get_random_int() % 10;
			spent += rnd;
			udelay(rnd);
			continue;
		}
		break;
	}
	if (retry == 0 || spent >= max_wait) {
		pr_info("[%s] readable fail = bus clock retry=[%d] spent=[%d]", __func__,
			retry, spent);
		return 0;
	}

	r = CONSYS_REG_READ_BIT(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_DBG_DUMMY_5, 0x1);
	if (r == 0x0)
		return 0;

#endif /* CFG_CONNINFRA_BUS_HANG_DEBUG_SUPPORT */
	return 1;
}


int consys_is_consys_reg(unsigned int addr)
{
	return 0;
}


static int consys_is_host_csr(unsigned long addr)
{
	struct consys_reg_base_addr *host_csr_addr =
			&conn_reg.reg_base_addr[CONN_HOST_CSR_TOP_BASE];

	if (addr >= host_csr_addr->phy_addr &&
			addr < host_csr_addr->phy_addr + host_csr_addr->size)
		return 1;
	return 0;
}

int consys_reg_init(struct platform_device *pdev)
{
	int ret = -1;
	struct device_node *node = NULL;
	struct consys_reg_base_addr *base_addr = NULL;
	struct resource res;
	int flag, i = 0;

	node = pdev->dev.of_node;
	pr_info("[%s] node=[%p]\n", __func__, node);
	if (node) {
		for (i = 0; i < CONSYS_BASE_ADDR_MAX; i++) {
			base_addr = &conn_reg.reg_base_addr[i];

			ret = of_address_to_resource(node, i, &res);
			if (ret) {
				pr_err("Get Reg Index(%d-%s) failed",
						i, consys_base_addr_index_to_str[i]);
				continue;
			}

			base_addr->phy_addr = res.start;
			base_addr->vir_addr =
				(unsigned long) of_iomap(node, i);
			of_get_address(node, i, &(base_addr->size), &flag);

			pr_info("Get Index(%d-%s) phy(0x%zx) baseAddr=(0x%zx) size=(0x%zx)",
				i, consys_base_addr_index_to_str[i], base_addr->phy_addr,
				base_addr->vir_addr, base_addr->size);
		}

	} else {
		pr_err("[%s] can't find CONSYS compatible node\n", __func__);
		return ret;
	}
	return 0;

}

static int consys_reg_deinit(void)
{
	int i = 0;

	for (i = 0; i < CONSYS_BASE_ADDR_MAX; i++) {
		if (conn_reg.reg_base_addr[i].vir_addr) {
			pr_info("[%d] Unmap %s (0x%zx)",
				i, consys_base_addr_index_to_str[i],
				conn_reg.reg_base_addr[i].vir_addr);
			iounmap((void __iomem*)conn_reg.reg_base_addr[i].vir_addr);
			conn_reg.reg_base_addr[i].vir_addr = 0;
		}
	}

	return 0;
}

