// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2019 MediaTek Inc.
 */

/* platform dependent */
#include "plat_def.h"
/* macro for read/write cr */
#include "consys_reg_util.h"
#include "consys_reg_mng.h"
/* cr base address */
#include "mt6880_consys_reg.h"
/* cr offset */
#include "mt6880_consys_reg_offset.h"
/* For function declaration */
#include "mt6880_pos.h"
#include "mt6880.h"

#include <linux/ratelimit.h>

/*******************************************************************************
*                                 M A C R O S
********************************************************************************
*/

#define CONN_INFRA_SYSRAM__A_DIE_DIG_TOP_CK_EN_MASK   0x7f

/*******************************************************************************
*                             D A T A   T Y P E S
********************************************************************************
*/

/*******************************************************************************
*                              C O N S T A N T S
********************************************************************************
*/

const static char* g_spi_system_name[SYS_SPI_MAX] = {
	"SYS_SPI_WF1",
	"SYS_SPI_WF",
	"SYS_SPI_BT",
	"SYS_SPI_FM",
	"SYS_SPI_GPS",
	"SYS_SPI_TOP",
	"SYS_SPI_WF2",
	"SYS_SPI_WF3",
};


/*******************************************************************************
*                  F U N C T I O N   D E C L A R A T I O N S
********************************************************************************
*/
static int consys_spi_read_nolock(enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int *data);
static int consys_spi_write_nolock(enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int data);
static void consys_spi_write_offset_range_nolock(
	enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int value,
	unsigned int reg_offset, unsigned int value_offset, unsigned int size);
#if CFG_CONNINFRA_THERMAL_SUPPORT
static int connsys_a_die_thermal_cal(
	int efuse_valid,
	unsigned int efuse0, unsigned int efuse1, unsigned int efuse2, unsigned int efuse3);
#endif
static int consys_polling_chipid_int(unsigned int retry, unsigned int sleep_ms);
static int consys_adie_top_ck_en_top_ctrl(bool on);

static int consys_sema_acquire(enum conn_semaphore_type index);

unsigned int consys_emi_set_remapping_reg(
	phys_addr_t con_emi_base_addr,
	phys_addr_t md_shared_emi_base_addr)
{
	/* 0x1806_01DC[19:0], gps_ap_peri_base[19:0] = 0x0_1000 (un-related to emi)
	 * conn2ap peri ramappinng -> 0x1000_0000
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_HOST_CSR_ADDR + CONN2AP_REMAP_GPS_PERI_BASE_ADDR,
		0x01000, 0xFFFFF);

	return 0;
}

int consys_conninfra_on_power_ctrl(unsigned int enable)
{
#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
	int ret;
#else
#ifndef CONFIG_FPGA_EARLY_PORTING
	int check;
#endif
#endif

	if (enable) {
#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Turn on SPM clock (apply this for SPM's CONNSYS power control related CR accessing)
		 * address: 0x1000_6000[0]
		 *          0x1000_6000[31:16]
		 * Data: [0]=1'b1
		 *       [31:16]=16'h0b16 (key)
		 * Action: write
		 */
		CONSYS_REG_WRITE_MASK(
			REG_SPM_BASE_ADDR + SPM_POWERON_CONFIG_EN, 0x0b160001, 0xffff0001);
#endif

		/* Power on Connsys MTCMOS */
#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
		ret = consys_platform_spm_conn_ctrl(enable);
		if (ret) {
			pr_err("Turn on oonn_infra power fail.\n");
			return -1;
		}
#else /* MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE */
		pr_info("Turn on conn_infra power by POS steps\n");
		/* Assert "conn_infra_on" primary part power on, set "connsys_on_domain_pwr_on"=1
		 * Address: 0x1000_6304[2]
		 * Data: 1'b1
		 * Action: write
		 */
		CONSYS_SET_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<2));

#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Check "conn_infra_on" primary part power status, check "connsys_on_domain_pwr_ack"=1
		 * (polling "10 times" and each polling interval is "0.5ms")
		 * Address: 0x1000_616C[1]
		 * Data: 1'b1
		 * Action: polling
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(REG_SPM_BASE_ADDR + SPM_PWR_STATUS, 1, 1, 10, 500, check);
		if (check != 0)
			pr_err("Check conn_infra_on primary power fail. 0x1000_616C is 0x%08x. Expect [1] as 1.\n",
				CONSYS_REG_READ(REG_SPM_BASE_ADDR + SPM_PWR_STATUS));
#endif

		/* Assert "conn_infra_on" secondary part power on, set "connsys_on_domain_pwr_on_s"=1
		 * Address: 0x1000_6304[3]
		 * Data: 1'b1
		 * Action: write
		 */
		CONSYS_SET_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<3));

#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Check "conn_infra_on" secondary part power status,
		 * check "connsys_on_domain_pwr_ack_s"=1
		 * (polling "10 times" and each polling interval is "0.5ms")
		 * Address: 0x1000_6170[1]
		 * Data: 1'b1
		 * Action: polling
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(REG_SPM_BASE_ADDR + SPM_PWR_STATUS_2ND, 1, 1, 10, 500, check);
		if (check != 0)
			pr_err("Check conn_infra_on secondary power fail. 0x1000_6170 is 0x%08x. Expect [1] as 1.\n",
				CONSYS_REG_READ(REG_SPM_BASE_ADDR + SPM_PWR_STATUS_2ND));
#endif

		/* Turn on AP-to-CONNSYS bus clock, set "conn_clk_dis"=0
		 * (apply this for bus clock toggling)
		 * Address: 0x1000_6304[4]
		 * Data: 1'b0
		 * Action: write
		 */
		CONSYS_CLR_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<4));

		/* Wait 1 us */
		udelay(1);

		/* De-assert "conn_infra_on" isolation, set "connsys_iso_en"=0
		 * Address: 0x1000_6304[1]
		 * Data: 1'b0
		 * Action: write
		 */
		CONSYS_CLR_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<1));

		/* De-assert CONNSYS S/W reset (TOP RGU CR),
		 * set "ap_sw_rst_b"=1
		 * Address: WDT_SWSYSRST[9] (0x1000_7018[9])
		 *          WDT_SWSYSRST[31:24] (0x1000_7018[31:24])
		 * Data: [9]=1'b0
		 *       [31:24]=8'h88 (key)
		 * Action: Write
		 */
		CONSYS_REG_WRITE_MASK(
			REG_TOP_RGU_ADDR + TOP_RGU_WDT_SWSYSRST, 0x88000000, 0xff000200);

		/* De-assert CONNSYS S/W reset (SPM CR), set "ap_sw_rst_b"=1
		 * Address: CONN_PWR_CON[0] (0x1000_6304[0])
		 * Data: 1'b1
		 * Action: write
		 */
		CONSYS_SET_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, 0x1);

		/* Wait 0.5ms  */
		udelay(500);

		/* conn2ap/ap2conn slpprot disable */
		/* Turn off AHB RX bus sleep protect (AP2CONN AHB Bus protect)
		 * (apply this for INFRA AHB bus accessing when CONNSYS had been turned on)
		 * Address: 0x1000_12A4[31:0] (INFRA_TOPAXI_PROTECTEN_CLR)
		 * Data: 32'h1000_0000
		 * Action: write
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_CLR, 0x10000000);

#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Check AHB RX bus sleep protect turn off
		 * (polling "100 times" and each polling interval is "0.5ms")
		 * Address: 0x1000_1228[28] (INFRA_TOPAXI_PROTECTEN2_STA1[2])
		 * Data: 1'b0
		 * Action: polling
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1, 28, 0x0, 100, 500, check);
		if (check != 0)
			pr_err("Polling AHB RX bus sleep protect turn off fail. status=0x%08x\n",
				CONSYS_REG_READ(REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1));
#endif

		/* Turn off AXI Rx bus sleep protect (CONN2AP AXI Rx Bus protect)
		 * (disable sleep protection when CONNSYS had been turned on)
		 * Note : Should turn off AXI Rx sleep protection first.
		 * Address:
		 * 	INFRA_TOPAXI_PROTECTEN_CLR (0x1000_12A4[31:0])
		 * Value: 32'h0000_4000
		 * Action: write32
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_CLR, 0x00004000);

		/* Turn off AXI TX bus sleep protect (AP2CONN AXI Bus protect)
		 * (apply this for INFRA AXI bus accessing when CONNSYS had been turned on)
		 * 
		 * INFRA_TOPAXI_PROTECTEN_CLR (0x1000_12A4) = 32'h0800_0000
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_CLR, 0x08000000);

		/* Turn off AHB TX bus sleep protect (AP2CONN AXI Bus protect)
		 * (apply this for INFRA AXI bus accessing when CONNSYS had been turned on)
		 * 
		 * INFRA_TOPAXI_PROTECTEN_CLR (0x1000_12A4[31:0]) = 32'h0000_2000
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_CLR, 0x00002000);

#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Check AHB TX bus sleep protect turn off
		 * (polling ""100 times"" and each polling interval is ""0.5ms"")
		 * If AP2CONN (TX/RX) protect turn off fail, power on fail.
		 * (DRV access connsys CR will trigger bus hang,
		 *  because bus transaction wiil queue at GALS, )
		 *
		 * INFRA_TOPAXI_PROTECTEN_STA1[13] (0x1000_1228[13]) = 1'b0
		 * Action: polling
		 * Note: AP2CONN AHB bus related setting
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1, 13, 0x0, 100, 500, check);
		if (check != 0) {
			pr_err("Polling 0x1000_1228[13] fail, exp is 0 but get 0x%08x",
				CONSYS_REG_READ(REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1));
			return -1;
		}
#endif /* CONFIG_FPGA_EARLY_PORTING */
		/* Wait 6ms (apply this for CONNSYS XO clock ready) */
		msleep(6);
#endif /* MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE */
	} else {
		/* conn2ap/ap2conn slpprot enable */
#if MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE
		pr_info("Turn off conn_infra power by SPM API\n");
		ret = consys_platform_spm_conn_ctrl(enable);
		if (ret) {
			pr_err("Turn off conn_infra power fail, ret=%d\n", ret);
			return -1;
		}
#else
		/* Turn on AHB TX bus sleep protect (AP2CONN AXI Bus protect)
		 * (apply this for INFRA AXI bus protection to prevent bus hang when
		 * CONNSYS had been turned off)
		 * Address: 0x1000_12a0[31:0]
		 * Data: 0x0000_2000
		 * Action: write
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_SET,
			0x00002000);

#ifndef CONFIG_FPGA_EARLY_PORTING
		/* check AHB TX bus sleep protect turn on (polling "100 times")
		 * Address: 0x1000_1228[13]
		 * Data: 1'b1
		 * Action: polling
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1,
			13, 1, 100, 1000, check);
		if (check)
			pr_err("Polling AHB TX bus sleep protect turn on fail.\n");
#endif /* CONFIG_FPGA_EARLY_PORTING */

		/* Turn on AXI Tx bus sleep protect (CONN2AP AXI Tx Bus protect)
		 * (apply this for INFRA AXI bus protection to prevent bus hang
		 *  when CONNSYS had been turned off)
		 * Note:
		 *   Should turn on AXI Tx sleep protection first.
		 * Address:
		 * 	INFRA_TOPAXI_PROTECTEN_SET
		 * 	0x1000_12A0[31:0]=32'h0800_0000
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_SET,
			0x08000000);

#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Check AXI Tx bus sleep protect turn on
		 * (polling "100 times", polling interval is 1ms)
		 * Address: INFRA_TOPAXI_PROTECTEN_STA1[27] (0x1000_1228[27])
		 * Value: 1'b1
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1,
			27, 1, 100, 1000, check);
		if (check)
			pr_err("Polling AXI Tx bus sleep protect turn on fail.");
#endif

		/* Turn on AXI Rx bus sleep protect (CONN2AP AXI RX Bus protect)
		 * (apply this for INFRA AXI bus protection to prevent bus hang when
		 * CONNSYS had been turned off)
		 * Note:
		 *	Should turn on AXI Rx sleep protection after
		 *	AXI Tx sleep protection has been turn on.
		 * Address: 0x1000_12A0[31:0]
		 * Data: 0x0000_4000
		 * Action: write
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_SET,
			0x00004000);
#ifndef CONFIG_FPGA_EARLY_PORTING
		/* check AXI Rx bus sleep protect turn on
		 * (polling "100 times", polling interval is 1ms)
		 * Address: 0x1000_1228[14]
		 * Data: 1'b1
		 * Action: polling
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1,
			14, 1, 100, 1000, check);
		if (check)
			pr_err("Polling AXI Rx bus sleep protect turn on fail.\n");
#endif
		/* Turn on AHB RX bus sleep protect (AP2CONN AXI Bus protect)
		 * (apply this for INFRA AXI bus protection to prevent bus hang
		 *  when CONNSYS had been turned off)
		 * INFRA_TOPAXI_PROTECTEN_SET
		 * 	0x1000_12A0[31:0]=32'h1000_0000
		 */
		CONSYS_REG_WRITE(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_SET,
			0x10000000);
#ifndef CONFIG_FPGA_EARLY_PORTING
		/* Check AHB RX bus sleep protect turn on (polling "10 times")
		 * INFRA_TOPAXI_PROTECTEN_STA1[28] (0x1000_1228[28]), value: 1'b1
		 */
		check = 0;
		CONSYS_REG_BIT_POLLING(
			REG_INFRACFG_AO_ADDR + INFRA_TOPAXI_PROTECTEN_STA1,
			28, 1, 100, 1000, check);
		if (check)
			pr_err("Polling AHB RX bus (AP2CONN) sleep protect turn on fail.");
#endif
		/* Power off connsys MTCMOS */
		/* Assert "conn_infra_on" isolation, set "connsys_iso_en"=1
		 * Address: CONN_PWR_CON[1] (0x1000_6304[1])
		 * Value: 1'b1
		 * Action: write
		 */
		CONSYS_SET_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<1));

		/* Assert CONNSYS S/W reset (SPM CR), set "ap_sw_rst_b"=0
		 * Address: CONN_PWR_CON[0] (0x1000_6304[0])
		 * Value: 1'b0
		 * Action: write
		 */
		CONSYS_CLR_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, 0x1);

		/* Assert CONNSYS S/W reset(TOP RGU CR), set "ap_sw_rst_b"=0
		 * Address: WDT_SWSYSRST[9] (0x1000_7018[9])
		 *          WDT_SWSYSRST[31:24] (0x1000_7018[31:24])
		 * Value: [9]=1'b1
		 *        [31:24]=8'h88 (key)
		 * Action: write
		 * Note: this CR value for reset control is active high (0: not reset, 1: reset)
		 */
		CONSYS_REG_WRITE_MASK(
			REG_TOP_RGU_ADDR + TOP_RGU_WDT_SWSYSRST,
			0x88000200,
			0xff000200);

		/* Turn off AP-to-CONNSYS bus clock, set "conn_clk_dis"=1
		 * (apply this for bus clock gating)
		 * Address: CONN_PWR_CON[4] (0x1000_6304[4])
		 * Value: 1'b1
		 * Action: write
		 */
		CONSYS_SET_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<4));

		/* wait 1us (?) */
		udelay(1);

		/* De-assert "conn_infra_on" primary part power on,
		 * set "connsys_on_domain_pwr_on"=0
		 * Address: CONN_PWR_CON[2] (0x1000_6304[2])
		 * Value: 1'b0
		 * Action: write
		 */
		CONSYS_CLR_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<2));

		/* De-assert "conn_infra_on" secondary part power on,
		 * set "connsys_on_domain_pwr_on_s"=0
		 * Address: CONN_PWR_CON[3] (0x1000_6304[3])
		 * Value: 1'b0
		 * Action: write
		 */
		CONSYS_CLR_BIT(REG_SPM_BASE_ADDR + SPM_CONN_PWR_CON, (0x1<<3));
#endif /* MTK_CONNINFRA_CLOCK_BUFFER_API_AVAILABLE */
	}
	return 0;
}

int consys_conninfra_wakeup(void)
{
	/* Wake up conn_infra
	 * Address: CONN_HOST_CSR_TOP_CONN_INFRA_WAKEPU_TOP_CONN_INFRA_WAKEPU_TOP (0x180601a0)
	 * Data: 1'b1
	 * Action: write
	 */
	CONSYS_REG_WRITE(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_WAKEPU_TOP_CONN_INFRA_WAKEPU_TOP,
		0x1);

	/* Check CONNSYS version ID
	 * (polling "10 times" for specific project code and each polling interval is "1ms")
	 */
	if (consys_polling_chipid_int(10, 1) != 0) {
		pr_err("[%s] Polling chip id fail\n", __func__);
		return -1;
	}
	return 0;
}

int consys_conninfra_sleep(void)
{
	/* Release conn_infra force on
	 * Address: CONN_HOST_CSR_TOP_CONN_INFRA_WAKEPU_TOP_CONN_INFRA_WAKEPU_TOP (0x180601a0)
	 * Data: 1'b0
	 * Action: write
	 */
	CONSYS_REG_WRITE(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_WAKEPU_TOP_CONN_INFRA_WAKEPU_TOP,
		0x0);
	return 0;
}

void consys_set_if_pinmux(unsigned int enable)
{
	if (enable) {
		/* Set pinmux for the interface between D-die and A-die (Aux1)
		 * (CONN_HRST_B/CONN_TOP_CLK/CONN_TOP_DATA)
		 * Address:
		 *   0x1000_53F8 = 32'hF0000000 (GPIO_MODE15_CLR[30:28]: GPIO127
		 *   0x1000_53F4 = 32'h10000000 (GPIO_MODE15_SET[30:28]: GPIO127
		 *   0x1000_5408 = 32'h000000FF (GPIO_MODE16_CLR[2:0]: GPIO128, [6:4]:GPIO129)
		 *   0x1000_5404 = 32'h00000011 (GPIO_MODE16_SET[2:0]: GPIO128, [6:4]:GPIO129)
		 * Note: GPIO127/GPIO128/GPIO129
		 */
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + GPIO_MODE15_CLR, 0xf0000000);
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + GPIO_MODE15_SET, 0x10000000);
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + GPIO_MODE16_CLR, 0x000000ff);
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + GPIO_MODE16_SET, 0x00000011);

		/* Set pinmux driving to 2mA
		 * Address:
		 *     0x11D1_0008 = 32'h000001ff
		 *     (DRV_CFG0_CLR[2:0]: conn_hrst_b, [5:3]: conn_top_clk, [8:6]: conn_top_data)
		 * Action: write32
		 * CODA: IOCFG_BM
		 */
		CONSYS_REG_WRITE(REG_IOCFG_BM_ADDR + IOCFG_BM_DRV_CFG0_CLR, 0x000001ff);

		/* POS update: 20200325, fix typo */
		/* Set CONN_TOP_CLK/CONN_TOP_DATA driving to 4mA
		 * Address:
		 *   0x11D1_0004 = 32'h00000048 (00000000_00000000_00000000_01001000)
		 *   (DRV_CFG0_SET[2:0]: conn_hrst_b, [5:3]: conn_top_clk, [8:6]: conn_top_data)
		 * Action: write32
		 * CODA: IOCFG_BM
		 */
		CONSYS_REG_WRITE(REG_IOCFG_BM_ADDR + IOCFG_BM_DRV_CFG0_SET, 0x00000048);

		/* Set pinmux PUPD setting
		 * Clear CONN_TOP_DATA PD setting
		 * Address:
		 * 	0x11D1_00d8 = 32'h00000010
		 * 	(PD_CFG0_CLR[4]: conn_top_data, 1:CLEAR bit)
		 * CODA: IOCFG_BM
		 */
		CONSYS_REG_WRITE(REG_IOCFG_BM_ADDR + IOCFG_BM_PD_CFG0_CLR, 0x00000010);

		/* Set pinmux PUPD setting
		 * CONN_TOP_DATA as PU
		 * Address:
		 * 	0x11D1_0114 = 32'h00000010
		 * 	(PU_CFG0_SET[4]: conn_top_data, 1: SET bit)
		 * CODA: IOCFG_BM
		 */
		CONSYS_REG_WRITE(REG_IOCFG_BM_ADDR + IOCFG_BM_PU_CFG0_SET, 0x00000010);
	} else {
		/* Set pinmux for the interface between D-die and A-die (Aux0)
		 * 0x1000_53F8=32'hF0000000
		 * 0x1000_5408=32'h000000FF
		 */
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + GPIO_MODE15_CLR, 0xf0000000);
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + GPIO_MODE16_CLR, 0x000000ff);
		/* Set pinmux PUPD setting
		 * Clear CONN_TOP_DATA PU setting
		 * 0x11D1_0118=32'h00000010
		 * 	[2]: conn_hrst_b
		 * 	[3]: conn_top_clk
		 * 	[4]: conn_top_data
		 */
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + IOCFG_BM_PU_CFG0_CLR, 0x00000010);
		/* Set pinmux PUPD setting
		 * CONN_TOP_DATA as PD
		 * 0x11D1_00d4=32'h00000010
		 * 	[2]: conn_hrst_b
		 * 	[3]: conn_top_clk
		 * 	[4]: conn_top_data
		 */
		CONSYS_REG_WRITE(REG_GPIO_BASE_ADDR + IOCFG_BM_PD_CFG0_SET, 0x00000010);
	}
#ifdef CONFIG_FPGA_EARLY_PORTING
	pr_info("[%s] not for FPGA\n", __func__);
#endif
}

int consys_polling_chipid_int(unsigned int retry, unsigned int sleep_ms)
{
	unsigned int count = retry + 1;
	unsigned int consys_hw_ver = 0;
	unsigned int consys_configuration_id = 0;

	while (--count > 0) {
		consys_hw_ver = CONSYS_REG_READ(
					REG_CONN_INFRA_CFG_ADDR +
					CONN_HW_VER_OFFSET);
		if (consys_hw_ver == CONN_HW_VER) {
			consys_configuration_id = CONSYS_REG_READ(
				 REG_CONN_INFRA_CFG_ADDR + CONN_CFG_ID_OFFSET);
			pr_info("Consys HW version id(0x%x) cfg_id=(0x%x)\n",
				consys_hw_ver, consys_configuration_id);
			break;
		}
		msleep(sleep_ms);
	}

	if (count == 0) {
		pr_err("Read CONSYS version id fail. Expect 0x%x but get 0x%x\n",
			CONN_HW_VER, consys_hw_ver);
		return -1;
	}

	return 0;
}

int consys_polling_chipid(void)
{
	return consys_polling_chipid_int(10, 1);
}

int connsys_d_die_cfg(void)
{
	unsigned int efuse;
	/* Read D-die Efuse
	 * Address: AP2CONN_EFUSE_DATA (0x1800_1020)
	 * Data:
	 * Action: read
	 */
	efuse = CONSYS_REG_READ(REG_CONN_INFRA_CFG_ADDR + AP2CONN_EFUSE_DATA);
	pr_info("D-die efuse: 0x%08x", efuse);

	/* POS update: 20200325: Update conn_infra sysram hwctrl setting
         */
#if 0
	/* conn_infra sysram hw control setting -> disable hw power down
	 * Address: CONN_INFRA_RGU_SYSRAM_HWCTL_PDN_SYSRAM_HWCTL_PDN (0x1800_0050)
	 * Data: 32'h0
	 * Action: write
	 */
	CONSYS_REG_WRITE(
		REG_CONN_INFRA_RGU_ADDR + CONN_INFRA_RGU_SYSRAM_HWCTL_PDN_SYSRAM_HWCTL_PDN, 0x0);

	/* conn_infra sysram hw control setting -> enable hw sleep
	 * Address: CONN_INFRA_RGU_SYSRAM_HWCTL_SLP_SYSRAM_HWCTL_SLP (0x1800_0054)
	 * Data: 32'h0000_00FF
	 * Action: write
	 */
	CONSYS_REG_WRITE(
		REG_CONN_INFRA_RGU_ADDR + CONN_INFRA_RGU_SYSRAM_HWCTL_SLP_SYSRAM_HWCTL_SLP,
		0x000000ff);
#endif
	return 0;
}

int connsys_spi_master_cfg(unsigned int next_status)
{
	/* TOP_CK_ADDR		0x18005084[11:0]	0x02C
	 *
	 * GPS_CK_ADDR		0x18005088[11:0]	0xA0C
	 * GPS_L5_CK_ADDR	0x18005088[27:16]	0xAFC
	 * => 0x1800_5088=0x0afc_0a0c
	 *
	 * GPS_RFBUF_ADR	0x18005094[11:0]	0x0FC
	 * GPS_L5_EN_ADDR	0x18005098[11:0]	0x0F8
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_TOP_CK_ADDR,
		0x02c, 0xfff);
	CONSYS_REG_WRITE_MASK(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_GPS_CK_ADDR,
		0x0afc0a0c, 0x0fff0fff);
	CONSYS_REG_WRITE_MASK(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_GPS_RFBUF_ADDR,
		0x0fc, 0xfff);
	CONSYS_REG_WRITE_MASK(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_GPS_L5_EN_ADDR,
		0x0f8, 0xfff);

	/* CMD_LENGTH	0x18005004[4:0]	0x8
	 * WB_BG_ADDR1	0x18005010	0xA03C
	 * WB_BG_ADDR2	0x18005014	0xA03C
	 * WB_BG_ADDR3	0x18005018	0xAA18
	 * WB_BG_ADDR4	0x1800501C	0xAA18
	 * WB_BG_ADDR5	0x18005020	0xA0C8
	 * WB_BG_ADDR6	0x18005024	0xAA00
	 * WB_BG_ADDR7	0x18005028	0xA0B4
	 * WB_BG_ADDR8	0x1800502C	0xA34C
	 * WB_BG_ON1  	0x18005030	0x00000000
	 * WB_BG_ON2  	0x18005034	0x00000000
	 * WB_BG_ON3  	0x18005038	0x74E0FFF5
	 * WB_BG_ON4  	0x1800503C	0x76E8FFF5
	 * WB_BG_ON5  	0x18005040	0x00000000
	 * WB_BG_ON6  	0x18005044	0xFFFFFFFF
	 * WB_BG_ON7  	0x18005048	0x00000019
	 * WB_BG_ON8  	0x1800504C	0x00010400 
	 * WB_BG_OFF1 	0x18005050	0x57400000
	 * WB_BG_OFF2 	0x18005054	0x57400000
	 * WB_BG_OFF3 	0x18005058	0x44E0FFF5
	 * WB_BG_OFF4 	0x1800505C	0x44E0FFF5
	 * WB_BG_OFF5 	0x18005060	0x00000001
	 * WB_BG_OFF6 	0x18005064	0x00000000
	 * WB_BG_OFF7 	0x18005068	0x00040019
	 * WB_BG_OFF8 	0x1800506C	0x00410440 
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_SLP_CTL,
		0x8, 0x1f);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR1,
		0xa03c);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR2,
		0xa03c);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR3,
		0xaa18);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR4,
		0xaa18);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR5,
		0xa0c8);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR6,
		0xaa00);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR7,
		0xa0b4);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ADDR8,
		0xa34c);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON1,
		0x0);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON2,
		0x0);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON3,
		0x74E0FFF5);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON4,
		0x76E8FFF5);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON5,
		0x0);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON6,
		0xFFFFFFFF);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON7,
		0x00000019);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_ON8,
		0x00010400);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF1,
		0x57400000);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF2,
		0x57400000);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF3,
		0x44E0FFF5);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF4,
		0x44E0FFF5);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF5,
		0x00000001);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF6,
		0x00000000);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF7,
		0x00040019);
	CONSYS_REG_WRITE(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_CTL_REG_WB_BG_OFF8,
		0x00410440);

	return 0;
}

#ifndef CONFIG_FPGA_EARLY_PORTING
/*****************************************************************************
* FUNCTION
*  connsys_a_die_efuse_read
* DESCRIPTION
*  Read a-die efuse
* PARAMETERS
*  efuse_addr: read address
* RETURNS
*  int
*	0: fail, efuse is invalid
*	1: success, efuse is valid
*****************************************************************************/
static int connsys_a_die_efuse_read_nolock(
	unsigned int efuse_addr,
	unsigned int* data0, unsigned int* data1,
	unsigned int* data2, unsigned int* data3)
{
	int ret = 0;
	int retry = 0;
	int ret0, ret1, ret2, ret3;

	if (data0 == NULL || data1 == NULL || data2 == NULL || data3 == NULL) {
		pr_err("[%s] invalid parameter (%p, %p, %p, %p)",
			__func__, data0, data1, data2, data3);
		return 0;
	}
	/* Efuse control clear, clear Status /trigger
	 * Address: ATOP EFUSE_CTRL_write_efsrom_kick_and_read_kick_busy_flag (0x108[30])
	 * Data: 1'b0
	 * Action: TOPSPI_WR
	 */
	consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, &ret);
	ret &= ~(0x1 << 30);
	consys_spi_write_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, ret);

	/* Efuse Read 1st 16byte
	 * Address:
	 *    ATOP EFUSE_CTRL_efsrom_mode (0x108[7:6]) = 2'b00
	 *    ATOP EFUSE_CTRL_efsrom_ain (0x108[25:16]) = efuse_addr (0)
	 *    ATOP EFUSE_CTRL_write_efsrom_kick_and_read_kick_busy_flag (0x108[30]) = 1'b1
	 * Action: TOPSPI_WR
	 */
	consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, &ret);
	ret &= ~(0x43ff00c0);
	ret |= (0x1 << 30);
	ret |= ((efuse_addr << 16) & 0x3ff0000);
	consys_spi_write_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, ret);

	/* Polling EFUSE busy = low
	 * (each polling interval is "30us" and polling timeout is 2ms)
	 * Address:
	 *    ATOP EFUSE_CTRL_write_efsrom_kick_and_read_kick_busy_flag (0x108[30]) = 1'b0
	 * Action: TOPSPI_Polling
	 */
	consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, &ret);
	while ((ret & (0x1 << 30)) != 0 && retry < 70) {
		retry++;
		udelay(30);
		consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, &ret);
	}
	if ((ret & (0x1 << 30)) != 0) {
		pr_info("[%s] EFUSE busy, retry failed(%d)\n", __func__, retry);
	}

	/* Check efuse_valid & return
	 * Address: ATOP EFUSE_CTRL_csri_efsrom_dout_vld_sync_1_ (0x108[29])
	 * Action: TOPSPI_RD
	 */
	/* if (efuse_valid == 1'b1)
	 *     Read Efuse Data to global var
	 */
	consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_CTRL, &ret);
	if (((ret & (0x1 << 29)) >> 29) == 1) {
		ret0 = consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_RDATA0, data0);
		ret1 = consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_RDATA1, data1);
		ret2 = consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_RDATA2, data2);
		ret3 = consys_spi_read_nolock(SYS_SPI_TOP, ATOP_EFUSE_RDATA3, data3);

		pr_info("efuse = [0x%08x, 0x%08x, 0x%08x, 0x%08x]", *data0, *data1, *data2, *data3);
		if (ret0 || ret1 || ret2 || ret3)
			pr_err("efuse read error: [%d, %d, %d, %d]", ret0, ret1, ret2, ret3);
		ret = 1;
	} else {
		pr_err("EFUSE is invalid\n");
		ret = 0;
	}
	return ret;
}
#endif

#if CFG_CONNINFRA_THERMAL_SUPPORT
static int connsys_a_die_thermal_cal(
	int efuse_valid,
	unsigned int efuse0, unsigned int efuse1, unsigned int efuse2, unsigned int efuse3)
{
	struct consys_plat_thermal_data input;

	memset(&input, 0, sizeof(struct consys_plat_thermal_data));
	if (efuse_valid) {
		if (efuse1 & (0x1 << 7)) {
			consys_spi_write_offset_range_nolock(
				SYS_SPI_TOP, ATOP_RG_TOP_THADC_BG, efuse1, 12, 3, 4);
			consys_spi_write_offset_range_nolock(
				SYS_SPI_TOP, ATOP_RG_TOP_THADC, efuse1, 23, 0, 3);
		}
		if(efuse1 & (0x1 << 15)) {
			consys_spi_write_offset_range_nolock(
				SYS_SPI_TOP, ATOP_RG_TOP_THADC, efuse1, 26, 13, 2);
			input.slop_molecule = (efuse1 & 0x1f00) >> 8;
			pr_info("slop_molecule=[%d]", input.slop_molecule);
		}
		if (efuse1 & (0x1 << 23)) {
			/* [22:16] */
			input.thermal_b = (efuse1 & 0x7f0000) >> 16;
			pr_info("thermal_b =[%d]", input.thermal_b);
		}
		if (efuse1 & (0x1 << 31)) {
			input.offset = (efuse1 & 0x7f000000) >> 24;
			pr_info("offset=[%d]", input.offset);
		}
	}
	input.efuse0 = efuse0;
	input.efuse1 = efuse1;
	input.efuse2 = efuse2;
	input.efuse3 = efuse3;

	update_thermal_data(&input);
	return 0;
}
#endif /* CFG_CONNINFRA_THERMAL_SUPPORT */

int connsys_a_die_cfg(void)
{
#ifndef CONFIG_FPGA_EARLY_PORTING
	int check;
	unsigned int adie_chip_id = 0x0;
	int efuse_valid = 0;
	unsigned int efuse0 = 0, efuse1 = 0, efuse2 = 0, efuse3 = 0;

	/* De-assert A-die reset
	 * Address:
	 * 	CONN_INFRA_CFG_ADIE_CTL_ADIE_RSTB (0x18001030[0])
	 * Value: 1'b1
	 * Action: write
	 */
	CONSYS_SET_BIT(REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_ADIE_CTL, 0x1);

	/* Get semaphore once */
	if (consys_sema_acquire_timeout(CONN_SEMA_RFSPI_INDEX, CONN_SEMA_TIMEOUT) == CONN_SEMA_GET_FAIL) {
		pr_err("[%s] Require semaphore fail\n", __func__);
		return -1;
	}

	/* Read MT6635 ID
	 * Address:
	 * 	ATOP CHIP_ID
	 * 	0x02C[31:16]: hw_code
	 * 	0x02C[15:0]: hw_ver
	 * Action: TOPSPI_RD
	 * Note:
	 * 	MT6635 E1 : read 0x02C = 0x66358A00;
	 * 	MT6635 E2 : read 0x02C = 0x66358A10;
	 * 	MT6635 E3 : read 0x02C = 0x66358A11;
	 */
	check = consys_spi_read_nolock(SYS_SPI_TOP, ATOP_CHIP_ID, &adie_chip_id);
	if (check) {
		/* Release semaphore */
		consys_sema_release(CONN_SEMA_RFSPI_INDEX);
		pr_err("[%s] Get ATOP_CHIP_ID fail, check=%d", __func__, check);
		return -1;
	}
	pr_info("A-die CHIP ID is 0x%x", adie_chip_id);
	conn_hw_env.adie_hw_version = adie_chip_id;
	conn_hw_env.is_rc_mode = 0;

	/* Update spi fm read extra bit setting
	 * CONN_RF_SPI_MST_REG_FM_CTRL_FM_RD_EXT_EN	0x1800400C[15]=1'b0
	 * CONN_RF_SPI_MST_REG_FM_CTRL_FM_RD_EXT_CNT	0x1800400C[7:0]=8'h0
	 * Action: write
	 * Note:
	 * 	Need update this setting when fm is at power-off state,
	 * 	otherwise spi access FM CR will fail
	 */
	CONSYS_CLR_BIT(REG_CONN_RFSPI_ADDR + CONN_RF_SPI_MST_REG_FM_CTRL, (0x1 << 15));
	CONSYS_CLR_BIT(REG_CONN_RFSPI_ADDR + CONN_RF_SPI_MST_REG_FM_CTRL, (0xff));

	/* Update Thermal addr for 6635
	 * CONN_TOP_THERM_CTL_THERM_AADDR
	 * 	0x18002018=32'h50305A00
	 */
	CONSYS_REG_WRITE(REG_CONN_TOP_THERM_CTL_ADDR + CONN_TOP_THERM_CTL_THERM_AADDR, 0x50305A00);

	/* Rread efuse Data(000) */
	efuse_valid = connsys_a_die_efuse_read_nolock(0x0, &efuse0, &efuse1, &efuse2, &efuse3);
	/* Thermal Cal (TOP) */
#if CFG_CONNINFRA_THERMAL_SUPPORT
	connsys_a_die_thermal_cal(efuse_valid, efuse0, efuse1, efuse2, efuse3);
#endif
	/* Increase XOBUF supply-V
	 * ATOP RG_TOP_XTAL_01 (0XA18) = 32'hF6E8FFF5
	 * Action: TOPSPI_WR
	 */
	consys_spi_write_nolock(SYS_SPI_TOP, ATOP_RG_TOP_XTAL_01, 0xF6E8FFF5);

	/* Increase XOBUF supply-R for MT6635 E1
	 * ATOP RG_TOP_XTAL_02 (0XA1C)
	 * if(MT6635 E1) //rf_hw_ver = 0x8a00
	 * 	32'hD5555FFF
	 * else
	 * 	32'h0x55555FFF
	 * Action: TOPSPI_WR
	 */
	if (adie_chip_id == 0x66358a00)
		consys_spi_write_nolock(SYS_SPI_TOP, ATOP_RG_TOP_XTAL_02, 0xD5555FFF);
	else
		consys_spi_write_nolock(SYS_SPI_TOP, ATOP_RG_TOP_XTAL_02, 0x55555FFF);

	/* Release semaphore */
	consys_sema_release(CONN_SEMA_RFSPI_INDEX);
#endif

	return 0;
}

int connsys_afe_wbg_cal(void)
{
	/* 20200622:
	 * Confirmed with DE, all config are default value, now.
	 */
	return 0;
}

int connsys_subsys_pll_initial(void)
{
	/* Check with DE, only 26M on mobile phone */
	/* CONN_AFE_CTL_RG_PLL_STB_TIME_RG_WBG_BPLL_STB_TIME	0x180030F4[30:16]	0x314
	 * CONN_AFE_CTL_RG_PLL_STB_TIME_RG_WBG_WPLL_STB_TIME	0x180030F4[14:0]	0x521
	 * CONN_AFE_CTL_RG_DIG_EN_02_RG_WBG_EN_BT_PLL		0x18003004[7:6]		0x1
	 * CONN_AFE_CTL_RG_DIG_EN_02_RG_WBG_EN_WF_PLL		0x18003004[1:0]		0x3
	 * CONN_AFE_CTL_RG_DIG_EN_02_RG_WBG_EN_MCU_PLL		0x18003004[3:2]		0x1
	 * CONN_AFE_CTL_RG_DIG_EN_02_RG_WBG_EN_MCU_PLL_WF	0x18003004[17:16]	0x3
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_AFE_CTL_BASE_ADDR + CONN_AFE_CTL_RG_PLL_STB_TIME,
		0x03140521, 0x7fff7fff);
	CONSYS_REG_WRITE_MASK(
		REG_CONN_AFE_CTL_BASE_ADDR + CONN_AFE_CTL_RG_DIG_EN_02,
		0x30047, 0x300cf);

	return 0;
}

int connsys_low_power_setting(unsigned int curr_status, unsigned int next_status)
{
	/* For this 6880, only GPS is need. Other should be reject. */
	if (curr_status != 0 || next_status != (0x1 << CONNDRV_TYPE_GPS)) {
		pr_err("[%s] Unsupport status: curr(0x%08x), next(0x%08x)",
			__func__, curr_status, next_status);
		return -1;
	}
	/* Unmask on2off/off2on slpprot_rdy enable checker @conn_infra off power off
	 * => check slpprot_rdy = 1'b1 and go to sleep
	 *
	 * CONN_INFRA_CFG_PWRCTRL0_CONN_INFRA_CFG_SLP_RDY_MASK
	 * 0x18001200[15:12] = 4'h1
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_PWRCTRL0, (0x1 << 12), 0xf000);

	if (!consys_is_rc_mode_enable()) {
		/* Legacy mode */
		/* Disable conn_top rc osc_ctrl_top
		 * CONN_INFRA_CFG_RC_CTL_0_CONN_INFRA_OSC_RC_EN
		 * 0x18001380[7] = 1'b0
		 */
		CONSYS_CLR_BIT(REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_0, (0x1<<7));
		/* Legacy OSC control stable time
		 *
		 * CONN_INFRA_CFG_OSC_CTL_0_XO_VCORE_RDY_STABLE_TIME
		 * 	0x18001300[7:0]=8'd6
		 * CONN_INFRA_CFG_OSC_CTL_0_XO_INI_STABLE_TIME
		 * 	0x18001300[15:8]=8'd7
		 * CONN_INFRA_CFG_OSC_CTL_0_XO_BG_STABLE_TIME
		 * 	0x18001300[23:16]=8'd8
		 */
		CONSYS_REG_WRITE_MASK(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_OSC_CTL_0,
			0x080706, 0x00ffffff);
		/* Legacy OSC control unmask conn_srcclkena_ack
		 * CONN_INFRA_CFG_OSC_CTL_1_ACK_FOR_XO_STATE_MASK
		 * 	0x18001304[16] = 1'b0
		 */
		CONSYS_CLR_BIT(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_OSC_CTL_1, (0x1<<16));
	} else {
		/* RC mode */
		/* GPS RC OSC control stable time
		 * CONN_INFRA_CFG_RC_CTL_1_GPS_XO_VCORE_RDY_STABLE_TIME_0
		 * 	0x18001394[7:0]=8'd6
		 * CONN_INFRA_CFG_RC_CTL_1_GPS_XO_INI_STABLE_TIME_0
		 * 	0x18001394[15:8]=8'd7
		 * CONN_INFRA_CFG_RC_CTL_1_GPS_XO_BG_STABLE_TIME_0
		 * 	0x18001394[23:16]=8'd8
		 * CONN_INFRA_CFG_RC_CTL_1_GPS_XO_VCORE_OFF_STABLE_TIME_0
		 * 	0x18001394[31:24]=8'd2
		 * 0x18001394[7:0]
		 */
		CONSYS_REG_WRITE(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_1_GPS, 0x02080706);
		/* GPS RC OSC control unmask conn_srcclkena_ack
		 * CONN_INFRA_CFG_RC_CTL_0_GPS_ACK_FOR_XO_STATE_MASK_0
		 * 	0x18001390[15] = 1'b0
		 */
		CONSYS_CLR_BIT(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_0_GPS, (0x1<<15));
		/* TOP RC OSC control stable time
		 * CONN_INFRA_CFG_RC_CTL_1_TOP_XO_VCORE_RDY_STABLE_TIME_3
		 * 	0x180013C4[7:0]=8'd6
		 * CONN_INFRA_CFG_RC_CTL_1_TOP_XO_INI_STABLE_TIME_3
		 * 	0x180013C4[15:8]=8'd7
		 * CONN_INFRA_CFG_RC_CTL_1_TOP_XO_BG_STABLE_TIME_3
		 * 	0x180013C4[23:16]=8'd8
		 * CONN_INFRA_CFG_RC_CTL_1_TOP_XO_VCORE_OFF_STABLE_TIME_3
		 * 	0x180013C4[31:24]=8'd2"
		 */
		CONSYS_REG_WRITE(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_1_TOP,
			0x02080706);
		/* TOP RC OSC control unmask conn_srcclkena_ack
		 * CONN_INFRA_CFG_RC_CTL_0_TOP_ACK_FOR_XO_STATE_MASK_3
		 * 	0x180013C0[15]=1'b0
		 */
		CONSYS_CLR_BIT(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_0_TOP, (0x1<<15));

		/* Enable conn_top rc osc_ctrl_gps
		 * CONN_INFRA_CFG_RC_CTL_0_GPSSYS_OSC_RC_EN
		 * 	0x18001380[4]=1'b1
		 */
		CONSYS_SET_BIT(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_0, (0x1<<4));
		/* Set conn_srcclkena control by conn_infra_emi_ctl
		 * CONN_INFRA_CFG_EMI_CTL_0_CONN_EMI_RC_EN
		 * 	0x18001400[0]=1'b1
		 */
		CONSYS_SET_BIT(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_EMI_CTL_0, 0x1);

		/* Disable legacy osc control
		 * CONN_INFRA_CFG_RC_CTL_0_OSC_LEGACY_EN
		 * 	0x18001380[0]=1'b0
		 */
		CONSYS_CLR_BIT(
			REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_RC_CTL_0, 0x1);
	}
	/* conn2ap sleep protect release bypass ddr_en_ack check
	 * CONN_INFRA_CFG_EMI_CTL_0_DDR_EN_BP_PROT
	 * 	0x18001400[18]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_EMI_CTL_0, (0x1<<18));
	/* Enable ddr_en timeout, timeout value = 1023T (Bus clock)
	 * CONN_INFRA_CFG_EMI_CTL_0_DDR_CNT_LIMIT
	 * 	0x18001400[14:4]=11'd1023
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_EMI_CTL_0, 0x3ff0, 0x7ff0);
	/* Enable ddr_en timeout value update (rising edge)
	 * CONN_INFRA_CFG_EMI_CTL_0_DDR_EN_CNT_UPDATE
	 *	0x18001400[15]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_EMI_CTL_0, (0x1<<15));
	/* Disable ddr_en timeout value update (falling edge)
	 * CONN_INFRA_CFG_EMI_CTL_0_DDR_EN_CNT_UPDATE
	 * 	0x18001400[15]=1'b0
	 */
	CONSYS_CLR_BIT(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_EMI_CTL_0, (0x1<<15));

	/* Disable conn_infra off pwr_ack mask
	 * CONN_INFRA_RGU_CONN_INFRA_OFF_TOP_PWR_CTL_CONN_INFRA_OFF_TOP_PWR_ACK_S_MASK
	 * 	0x18000000[31:16]=16'h494E (key)
	 * 	0x18000000[6]=1'b0
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_INFRA_RGU_ADDR + CONN_INFRA_RGU_CONN_INFRA_OFF_TOP_PWR_CTL,
		0x494e0000, 0xffff0040);

	/* Disable A-die top_ck_en (for Bring-up keep a-die dig_top ck on)
	 * Call common API
	 * ATOP Clock default on, need to turn off manually
	 */
	consys_adie_top_ck_en_top_ctrl(0);

	/* Select conn_infra_cfg debug_sel to low power related
	 * CONN_HOST_CSR_TOP_CONN_INFRA_CFG_DBG_SEL_CONN_INFRA_CFG_DBG_SEL
	 * 0x1806015c[2:0]=3'b000
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_HOST_CSR_ADDR + CONN_HOST_CSR_TOP_CONN_INFRA_CFG_DBG_SEL,
		0x0, 0x7);

	/* OFF domain AHB slaves timeout limit setting
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_OFF_TIMEOUT_CTRL_CONN_INFRA_OFF_TIMEOUT_LIMIT
	 * 0x1800E300[14:7]=8'h02
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_OFF_TIMEOUT_CTRL,
		(0x2 << 7), 0x7f80);

	/* Enable off domain AHB slaves timeout
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_OFF_TIMEOUT_CTRL_CONN_INFRA_OFF_TIMEOUT_EN
	 * 0x1800E300[2]=1'b1
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_OFF_TIMEOUT_CTRL_CONN_INFRA_OFF_AHB_MUX_EN
	 * 0x1800E300[1]=1'b1
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_OFF_TIMEOUT_CTRL_CONN_INFRA_OFF_APB_MUX_EN
	 * 0x1800E300[0]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_OFF_TIMEOUT_CTRL,
		0x7);

	/* ON domain APB slaves timeout limit setting
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_ON_TIMEOUT_CTRL_CONN_INFRA_ON_TIMEOUT_LIMIT
	 * 0x1800E31C[14:7]=8'h02
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_ON_TIMEOUT_CTRL,
		(0x2 << 7), 0x7f80);

	/* Enable on domain APB slaves timeout
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_ON_TIMEOUT_CTRL_CONN_INFRA_ON_TIMEOUT_EN
	 * 0x1800E31C[2]=1'b1
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_ON_TIMEOUT_CTRL_CONN_INFRA_ON_AHB_MUX_EN
	 * 0x1800E31C[1]=1'b1
	 * CONN_INFRA_BUS_CR_CONN_INFRA_BUS_ON_TIMEOUT_CTRL_CONN_INFRA_ON_APB_MUX_EN
	 * 0x1800E31C[0]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_BUS_CR_ADDR + CONN_INFRA_BUS_ON_TIMEOUT_CTRL,
		0x7);

	/* Hold timeout reset
	 * CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_VDNR_GEN_ON_U_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0_reset_setting
	 * 0x1802F000[9] = 1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_DEBUG_CTRL_AO + CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0,
		(0x1 << 9));

	/* Set conn_infra bus timeout value
	 * CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_VDNR_GEN_ON_U_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0_timeout_thres
	 * 0x1802F000[31:16]=16'h23BC
	 */
	CONSYS_REG_WRITE_MASK(
		REG_CONN_INFRA_DEBUG_CTRL_AO + CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0,
		0x23BC0000, 0xFFFF0000);

	/* Enable conn_infra bus timeout
	 * CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_VDNR_GEN_ON_U_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0_debug_en
	 * 0x1802F000[2]=1'b1
	 * CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_VDNR_GEN_ON_U_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0_debug_cken
	 * 0x1802F000[3]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_DEBUG_CTRL_AO + CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0,
		0xc);

	/* Release timeout reset
	 * CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_VDNR_GEN_ON_U_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0_reset_setting
	 * 0x1802F000[9] = 1'b0
	 */
	CONSYS_CLR_BIT(
		REG_CONN_INFRA_DEBUG_CTRL_AO + CONN_INFRA_DEBUG_CTRL_AO_CONN_INFRA_ON_CTRL0,
		(0x1 << 9));

	/* Enable conn_infra_cfg setting
	 * (osc_en would pull low after conn_infra mtcmos off)
	 * CONN_INFRA_CFG_PWRCTRL0_HWCTL_OSC_ON_CHECK_TOP_PWR_EN
	 * 0x1800_1200[9]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_PWRCTRL0, (0x1 << 9));

	/* Enable bpll divder2 enable for conn_infra bus hclk
	 * CONN_INFRA_CLKGEN_ON_TOP_CKGEN_BUS_BPLL_DIV_2_BPLL_DIV_2_DIV_EN
	 * 0x18009004[0]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_CLKGEN_ON_TOP_ADDR + CONN_INFRA_CKGEN_BUS_BPLL_DIV_2, 0x1);

	/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
	/* !!!!!!!!!!!!!!!!!!!!!! CANNOT add code after HERE!!!!!!!!!!!!!!!!!!!!!!!!!! */
	/* !!!!!!!!!!!!!!!!!!!!!!! conninfra may go to sleep !!!!!!!!!!!!!!!!!!!!!!!!! */
	/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */

	/* Disable conn_infra bus clock sw control  ==> conn_infra bus clock hw control
	 * CONN_INFRA_CFG_CKGEN_BUS_HCLK_CKSEL_SWCTL
	 * 	0x1800_9A00[23]=1'b0
	 */
	CONSYS_CLR_BIT(
		REG_CONN_INFRA_CLKGEN_ON_TOP_ADDR + CONN_INFRA_CLKGEN_ON_TOP_CKGEN_BUS, (0x1<<23));
	/* Conn_infra HW_CONTROL => conn_infra enter dsleep mode
	 * CONN_INFRA_CFG_PWRCTRL0_HW_CONTROL
	 * 	0x1800_1200[0]=1'b1
	 */
	CONSYS_SET_BIT(
		REG_CONN_INFRA_CFG_ADDR + CONN_INFRA_CFG_PWRCTRL0, 0x1);
	return 0;
}

static int consys_sema_acquire(enum conn_semaphore_type index)
{
	if (CONSYS_REG_READ_BIT(
		REG_CONN_SEMAPHORE_ADDR + CONN_SEMAPHORE_M2_OWN_STA + index*4, 0x1) == 0x1) {
		return CONN_SEMA_GET_SUCCESS;
	} else {
		return CONN_SEMA_GET_FAIL;
	}
}

int consys_sema_acquire_timeout(unsigned int index, unsigned int usec)
{
	int i;

	if (index >= CONN_SEMA_NUM_MAX) {
		pr_err("[%s] wrong index: %d", __func__, index);
		return CONN_SEMA_GET_FAIL;
	}

	for (i = 0; i < usec; i++) {
		if (consys_sema_acquire(index) == CONN_SEMA_GET_SUCCESS) {
			return CONN_SEMA_GET_SUCCESS;
		}
		udelay(1);
	}
	pr_err("Get semaphore 0x%x timeout, dump status:\n", index);
	pr_err("M2:[0x%x] M3:[0x%x]\n",
		CONSYS_REG_READ(REG_CONN_SEMAPHORE_ADDR + CONN_SEMA_OWN_BY_M2_STA_REP),
		CONSYS_REG_READ(REG_CONN_SEMAPHORE_ADDR + CONN_SEMA_OWN_BY_M3_STA_REP));
	return CONN_SEMA_GET_FAIL;
}

void consys_sema_release(unsigned int index)
{
	if (index >= CONN_SEMA_NUM_MAX) {
		pr_err("[%s] wrong index: %d", __func__, index);
		return;
	}

	CONSYS_REG_WRITE(
		(REG_CONN_SEMAPHORE_ADDR + CONN_SEMAPHORE_M2_OWN_REL + index*4), 0x1);
}

struct spi_op {
	unsigned int busy_cr;
	unsigned int polling_bit;
	unsigned int addr_cr;
	unsigned int read_addr_format;
	unsigned int write_addr_format;
	unsigned int write_data_cr;
	unsigned int read_data_cr;
	unsigned int read_data_mask;
};

static const struct spi_op spi_op_array[SYS_SPI_MAX] = {
	/* SYS_SPI_WF1 */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 1,
		CONN_RF_SPI_MST_REG_SPI_WF_ADDR, 0x00001000, 0x00000000,
		CONN_RF_SPI_MST_REG_SPI_WF_WDAT,
		CONN_RF_SPI_MST_REG_SPI_WF_RDAT, 0xffffffff
	},
	/* SYS_SPI_WF */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 1,
		CONN_RF_SPI_MST_REG_SPI_WF_ADDR, 0x00003000, 0x00002000,
		CONN_RF_SPI_MST_REG_SPI_WF_WDAT,
		CONN_RF_SPI_MST_REG_SPI_WF_RDAT, 0xffffffff
	},
	/* SYS_SPI_BT */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 2,
		CONN_RF_SPI_MST_REG_SPI_BT_ADDR, 0x00005000, 0x00004000,
		CONN_RF_SPI_MST_REG_SPI_BT_WDAT,
		CONN_RF_SPI_MST_REG_SPI_BT_RDAT, 0x000000ff
	},
	/* SYS_SPI_FM */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 3,
		CONN_RF_SPI_MST_REG_SPI_FM_ADDR, 0x00007000, 0x00006000,
		CONN_RF_SPI_MST_REG_SPI_FM_WDAT,
		CONN_RF_SPI_MST_REG_SPI_FM_RDAT, 0x0000ffff
	},
	/* SYS_SPI_GPS */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 4,
		CONN_RF_SPI_MST_REG_SPI_GPS_GPS_ADDR, 0x00009000, 0x00008000,
		CONN_RF_SPI_MST_REG_SPI_GPS_GPS_WDAT,
		CONN_RF_SPI_MST_REG_SPI_GPS_GPS_RDAT, 0x0000ffff
	},
	/* SYS_SPI_TOP */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 5,
		CONN_RF_SPI_MST_REG_SPI_TOP_ADDR, 0x0000b000, 0x0000a000,
		CONN_RF_SPI_MST_REG_SPI_TOP_WDAT,
		CONN_RF_SPI_MST_REG_SPI_TOP_RDAT, 0xffffffff
	},
	/* SYS_SPI_WF2 */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 1,
		CONN_RF_SPI_MST_REG_SPI_WF_ADDR, 0x0000d000, 0x0000c000,
		CONN_RF_SPI_MST_REG_SPI_WF_WDAT,
		CONN_RF_SPI_MST_REG_SPI_WF_RDAT, 0xffffffff
	},
	/* SYS_SPI_WF3 */
	{
		CONN_RF_SPI_MST_REG_SPI_STA, 1,
		CONN_RF_SPI_MST_REG_SPI_WF_ADDR, 0x0000f000, 0x0000e000,
		CONN_RF_SPI_MST_REG_SPI_WF_WDAT,
		CONN_RF_SPI_MST_REG_SPI_WF_RDAT, 0xffffffff
	},
};

static int consys_spi_read_nolock(enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int *data)
{
#ifndef CONFIG_FPGA_EARLY_PORTING
	/* Read action:
	 * 1. Polling busy_cr[polling_bit] should be 0
	 * 2. Write addr_cr with data being {read_addr_format | addr[11:0]}
	 * 3. Trigger SPI by writing write_data_cr as 0
	 * 4. Polling busy_cr[polling_bit] as 0
	 * 5. Read data_cr[data_mask]
	 */
	int check = 0;
	const struct spi_op* op = &spi_op_array[subsystem];

	if (!data) {
		pr_err("[%s] invalid data ptr\n", __func__);
		return CONNINFRA_SPI_OP_FAIL;
	}

	CONSYS_REG_BIT_POLLING(
		REG_CONN_RFSPI_ADDR + op->busy_cr,
		op->polling_bit, 0, 100, 50, check);
	if (check != 0) {
		pr_err("[%s][%d][STEP1] polling 0x%08lx bit %d fail. Value=0x%08x\n",
			__func__, subsystem, REG_CONN_RFSPI_ADDR + op->busy_cr,
			op->polling_bit,
			CONSYS_REG_READ(REG_CONN_RFSPI_ADDR + op->busy_cr));
		return CONNINFRA_SPI_OP_FAIL;
	}

	CONSYS_REG_WRITE(
		REG_CONN_RFSPI_ADDR + op->addr_cr,
		(op->read_addr_format | addr));

	CONSYS_REG_WRITE(REG_CONN_RFSPI_ADDR + op->write_data_cr, 0);

	check = 0;
	CONSYS_REG_BIT_POLLING(
		REG_CONN_RFSPI_ADDR + op->busy_cr,
		op->polling_bit, 0, 100, 50, check);
	if (check != 0) {
		pr_err("[%s][%d][STEP4] polling 0x%08lx bit %d fail. Value=0x%08x\n",
			__func__, subsystem, REG_CONN_RFSPI_ADDR + op->busy_cr,
			op->polling_bit,
			CONSYS_REG_READ(REG_CONN_RFSPI_ADDR + op->busy_cr));
		return CONNINFRA_SPI_OP_FAIL;
	}

	check = CONSYS_REG_READ_BIT(REG_CONN_RFSPI_ADDR + op->read_data_cr, op->read_data_mask);
	*data = check;
#else
	*data = 0;
#endif /* CONFIG_FPGA_EARLY_PORTING */
	return 0;
}

int consys_spi_read(enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int *data)
{
	int ret;

	/* Get semaphore before read */
	if (consys_sema_acquire_timeout(CONN_SEMA_RFSPI_INDEX, CONN_SEMA_TIMEOUT) == CONN_SEMA_GET_FAIL) {
		pr_err("[SPI READ] Require semaphore fail\n");
		return CONNINFRA_SPI_OP_FAIL;
	}

	ret = consys_spi_read_nolock(subsystem, addr, data);

	consys_sema_release(CONN_SEMA_RFSPI_INDEX);

	return ret;
}

static int consys_spi_write_nolock(enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int data)
{
#ifndef CONFIG_FPGA_EARLY_PORTING
	int check = 0;
	const struct spi_op* op = &spi_op_array[subsystem];

	/* Write action:
	 * 1. Wait busy_cr[polling_bit] as 0
	 * 2. Write addr_cr with data being {write_addr_format | addr[11:0]
	 * 3. Write write_data_cr ad data
	 * 4. Wait busy_cr[polling_bit] as 0
	 */
	CONSYS_REG_BIT_POLLING(
		REG_CONN_RFSPI_ADDR + op->busy_cr,
		op->polling_bit, 0, 100, 50, check);
	if (check != 0) {
		pr_err("[%s][%d][STEP1] polling 0x%08lx bit %d fail. Value=0x%08x\n",
			__func__, subsystem, REG_CONN_RFSPI_ADDR + op->busy_cr,
			op->polling_bit,
			CONSYS_REG_READ(REG_CONN_RFSPI_ADDR + op->busy_cr));
		return CONNINFRA_SPI_OP_FAIL;
	}

	CONSYS_REG_WRITE(REG_CONN_RFSPI_ADDR + op->addr_cr, (op->write_addr_format | addr));

	CONSYS_REG_WRITE(REG_CONN_RFSPI_ADDR + op->write_data_cr, data);

	check = 0;
	CONSYS_REG_BIT_POLLING(
		REG_CONN_RFSPI_ADDR + op->busy_cr,
		op->polling_bit, 0, 100, 50, check);
	if (check != 0) {
		pr_err("[%s][%d][STEP4] polling 0x%08lx bit %d fail. Value=0x%08x\n",
			__func__, subsystem, REG_CONN_RFSPI_ADDR + op->busy_cr,
			op->polling_bit,
			CONSYS_REG_READ(REG_CONN_RFSPI_ADDR + op->busy_cr));
		return CONNINFRA_SPI_OP_FAIL;
	}
#endif /* CONFIG_FPGA_EARLY_PORTING */
	return 0;

}


int consys_spi_write(enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int data)
{
	int ret;

	/* Get semaphore before read */
	if (consys_sema_acquire_timeout(CONN_SEMA_RFSPI_INDEX, CONN_SEMA_TIMEOUT) == CONN_SEMA_GET_FAIL) {
		pr_err("[SPI WRITE] Require semaphore fail\n");
		return CONNINFRA_SPI_OP_FAIL;
	}

	ret = consys_spi_write_nolock(subsystem, addr, data);

	consys_sema_release(CONN_SEMA_RFSPI_INDEX);
	return ret;
}

int consys_spi_write_offset_range(
	enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int value,
	unsigned int reg_offset, unsigned int value_offset, unsigned int size)
{
	if (consys_sema_acquire_timeout(CONN_SEMA_RFSPI_INDEX, CONN_SEMA_TIMEOUT) == CONN_SEMA_GET_FAIL) {
		pr_err("[SPI READ] Require semaphore fail\n");
		return CONNINFRA_SPI_OP_FAIL;
	}
	consys_spi_write_offset_range_nolock(
		subsystem, addr, value, reg_offset, value_offset, size);

	consys_sema_release(CONN_SEMA_RFSPI_INDEX);
	return 0;
}

static void consys_spi_write_offset_range_nolock(
	enum sys_spi_subsystem subsystem, unsigned int addr, unsigned int value,
	unsigned int reg_offset, unsigned int value_offset, unsigned int size)
{
	unsigned int data = 0, data2;
	unsigned int reg_mask;
	int ret;

	pr_info("[%s][%s] addr=0x%04x value=0x%08x reg_offset=%d value_offset=%d size=%d",
		__func__, g_spi_system_name[subsystem], addr, value, reg_offset, value_offset, size);
	value = (value >> value_offset);
	value = GET_BIT_RANGE(value, size, 0);
	value = (value << reg_offset);
	ret = consys_spi_read_nolock(subsystem, addr, &data);
	if (ret) {
		pr_err("[%s][%s] Get 0x%08x error, ret=%d",
			__func__, g_spi_system_name[subsystem], addr, ret);
		return;
	}
	reg_mask = GENMASK(reg_offset + size - 1, reg_offset);
	data2 = data & (~reg_mask);
	data2 = (data2 | value);
	consys_spi_write_nolock(subsystem, addr, data2);
	pr_info("[%s][%s] Write CR:0x%08x from 0x%08x to 0x%08x",
		__func__, g_spi_system_name[subsystem],
		addr, data, data2);
}

static int consys_adie_top_ck_en_top_ctrl(bool on)
{
	int check = 0;
#ifndef CONFIG_FPGA_EARLY_PORTING
	/* 0x18005120[0] == 1'b1/1'b0
	 * 	1'b1/1'b0: Enable/Disable dig_top_ck in Adie
	 * Adress in Adie: 12'hA00
	 */
	if (on)
		CONSYS_SET_BIT(REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_WB_SLP_TOP_CK_0, (0x1 << 0));
	else
		CONSYS_CLR_BIT(REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_WB_SLP_TOP_CK_0, (0x1 << 0));

	/* POLLING 0x18005120[1] == 1'b0
	 * Wait 0x18005120[1] == 1'b0 (Wait finish status)
	 */
	CONSYS_REG_BIT_POLLING(
		REG_CONN_WT_SPL_CTL_ADDR + CONN_WT_SLP_WB_SLP_TOP_CK_0,
		1, 0, 10, 10, check);
	if (check == -1) {
		pr_err("[%s] %s fail\n", __func__, (on? "turn on" : "turn off"));
	}
#endif
	return check;
}

bool consys_is_rc_mode_enable(void)
{
	/* Not support RC mode */
	return 0;
}

