// SPDX-License-Identifier: MediaTekProprietary
#include "SpeechModemController.h"

#include "SpeechDriverInterface.h"
#include "SpeechDriverFactory.h"
#include <AudioVolumeFactory.h>
#include "SpeechEnhancementController.h"
#include "SpeechVMRecorder.h"
#include <utils/Log.h>

#define LOG_TAG "SpeechModemController"


namespace android {

SpeechModemController *SpeechModemController::mSpeechModemController = NULL;
SpeechModemController *SpeechModemController::getInstance() {
    static AudioLock mGetInstanceLock;
    AL_AUTOLOCK(mGetInstanceLock);

    if (mSpeechModemController == NULL) {
        mSpeechModemController = new SpeechModemController();
    }
    ASSERT(mSpeechModemController != NULL);
    return mSpeechModemController;
}

SpeechModemController::SpeechModemController() :
    mSpeechDriverFactory(SpeechDriverFactory::GetInstance()),
    mAudioALSAVolumeController(NULL),
    mUlMute(false),
    mDlMute(false),
    mBtMode(0),
    mBtNrec(true),
    mSpeechOn(false),
    mCurrentOutputDevice(AUDIO_DEVICE_OUT_EARPIECE),
    mCurrentInputDevice(AUDIO_DEVICE_IN_BUILTIN_MIC),
    mIsEcallIvsOn(false),
    mIsEcallPsapOn(false),
    mIsEcallCtrlSeqOn(false),
    mVolumeIndex(4),
    mSpkType(-1) {

    initialize();
}

SpeechModemController::~SpeechModemController() {

}

int SpeechModemController::initialize() {
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    mSpeechDriverFactory->SetActiveModemIndex(MODEM_1);
    mBGSPlayBuffer = NULL;

    return NO_ERROR;
}

int SpeechModemController::phoneCallOpen(const audio_devices_t input_device, const audio_devices_t output_devices) {
    AL_AUTOLOCK(mLock);
    AL_AUTOLOCK(mCheckOpenLock);

    if (mSpeechOn == true) {
        ALOGD("%s(), Modem SpeechOn is already open. return.", __FUNCTION__, output_devices, input_device);
        return NO_ERROR;
    }
    ALOGD("+%s(), output_devices = 0x%x, input_device = 0x%x", __FUNCTION__, output_devices, input_device);

    // get speech driver instance
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();

    // check BT device
    const bool bt_device_on = audio_is_bluetooth_sco_device(output_devices);
    const uint32_t sample_rate = calculateSampleRate(bt_device_on);
    ALOGD("%s(), bt_device_on = %d, sample_rate = %d", __FUNCTION__, bt_device_on, sample_rate);

    // Set MD side sampling rate
    speechDriver->SetModemSideSamplingRate(sample_rate);

    speechDriver->SetSpeechMode(input_device, output_devices);

    speechDriver->SpeechOn();
#if defined(MTK_SPEECH_MD_LPBK_TX2RX)
    speechDriver->SetModemLoopbackPoint(1);
#endif

    // for the case that customized app set mute before speech on
    AL_AUTOLOCK(mUlMuteLock);
    speechDriver->SetUplinkMute(mUlMute);

    // check VM need open
    SpeechVMRecorder *speechVMRecorder = SpeechVMRecorder::getInstance();
    if (speechVMRecorder->getVmConfig() == SPEECH_VM_SPEECH) {
        speechVMRecorder->open();
    }
    mCurrentOutputDevice = output_devices;
    mCurrentInputDevice = input_device;
    mSpeechOn = true;
    ALOGD("-%s(), mSpeechOn = %d", __FUNCTION__, mSpeechOn);

    return NO_ERROR;
}


int SpeechModemController::phoneCallClose() {
    AL_AUTOLOCK(mLock);

    if (mSpeechOn == false) {
        ALOGD("+%s(), Modem SpeechOn is not open. return.", __FUNCTION__);
        return NO_ERROR;
    }
    ALOGD("+%s(), mSpeechOn = %d", __FUNCTION__, mSpeechOn);
    const modem_index_t modem_index = mSpeechDriverFactory->GetActiveModemIndex();

    // check VM need close
    SpeechVMRecorder *speechVMRecorder = SpeechVMRecorder::getInstance();
    if (speechVMRecorder->getVMRecordStatus() == true) {
        speechVMRecorder->close();
    }

    // Get current active speech driver
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
#if defined(MTK_SPEECH_MD_LPBK_TX2RX)
    speechDriver->SetModemLoopbackPoint(0);
#endif

    eCallIvsSwitch(false);
    eCallPsapSwitch(false);
    eCallCtrlSeqSwitch(false);

    // Speech off
    if (speechDriver->GetApSideModemStatus(SPEECH_STATUS_MASK) == true) {
        speechDriver->SpeechOff();
    } else {
        ALOGE("%s(), Speech is already closed!!", __FUNCTION__);
    }

    mSpeechOn = false;
    ALOGD("-%s(), mSpeechOn = %d", __FUNCTION__, mSpeechOn);

    return NO_ERROR;
}

int SpeechModemController::phoneCallRouting(const audio_devices_t input_device,
                                                     const audio_devices_t output_devices) {

    AL_AUTOLOCK(mLock);
    AL_AUTOLOCK(mCheckOpenLock);

    if (output_devices == mCurrentOutputDevice) {
        ALOGD("%s(), the same output deivces(0x%x). return.", __FUNCTION__, output_devices, input_device);
        return NO_ERROR;
    }
    ALOGD("+%s(), output_devices = 0x%x, input_device = 0x%x", __FUNCTION__, output_devices, input_device);

    // get speech driver instance
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();

    // check BT device
    const bool bt_device_on = audio_is_bluetooth_sco_device(output_devices);
    const uint32_t sample_rate = calculateSampleRate(bt_device_on);
    ALOGD("%s(), bt_device_on = %d, sample_rate = %d", __FUNCTION__, bt_device_on, sample_rate);

    // Set MD side sampling rate
    speechDriver->SetModemSideSamplingRate(sample_rate);
    speechDriver->SetSpeechMode(input_device, output_devices);
    return NO_ERROR;
}

int SpeechModemController::configVm(const uint8_t vmConfig) {
    ALOGD("%s(), vmConfig=%d", __FUNCTION__, vmConfig);
    SpeechVMRecorder *speechVMRecorder = SpeechVMRecorder::getInstance();
    return speechVMRecorder->configVm(vmConfig);
}

int SpeechModemController::setBtMode(const int mode) {
    ALOGD("%s(), mBtMode: %d => %d", __FUNCTION__, mBtMode, mode);
    AL_AUTOLOCK(mLock);

    if (mBtMode != mode) {
        mBtMode = mode;

        if (mSpeechOn && audio_is_bluetooth_sco_device(mCurrentOutputDevice)) {
            // Get current active speech driver
            SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
            uint32_t sampleRate = calculateSampleRate(true);
            speechDriver->switchBTMode(sampleRate);
        }
    }
    return NO_ERROR;
}

int SpeechModemController::setBtNrec(const bool enable) {
    ALOGD("%s(), mBtNrec: %d => %d", __FUNCTION__, mBtNrec, enable);
    AL_AUTOLOCK(mLock);

    if (mBtNrec != enable) {
        if (mSpeechOn && audio_is_bluetooth_sco_device(mCurrentOutputDevice)) {
            // Get current active speech driver
            if (SpeechEnhancementController::GetInstance()->GetBtHeadsetNrecOn() != enable) {
                SpeechEnhancementController::GetInstance()->SetBtHeadsetNrecOnToAllModem(enable);
            }
        }
        mBtNrec = enable;
    }
    return NO_ERROR;
}

int SpeechModemController::setBtVgs(const bool support) {
    ALOGD("%s(), mBtVgs: %d => %d", __FUNCTION__, mBtVgs, support);
    AL_AUTOLOCK(mLock);

    if (mBtVgs != support) {
        AudioVolumeFactory::CreateAudioVolumeController()->setBtVolumeCapability(support);
        mBtVgs = support;
    }
    return NO_ERROR;
}

int SpeechModemController::setDlDigitalGain(const int16_t gain) {
    ALOGD("%s() gain=%d", __FUNCTION__, gain);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    return speechDriver->SetDownlinkGain(gain);
}

int SpeechModemController::setUlDigitalGain(const int16_t gain) {
    ALOGD("%s() gain=%d", __FUNCTION__, gain);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    return speechDriver->SetUplinkGain(gain);
}

int SpeechModemController::setSpeakerType(const int type) {
    if (mSpkType == type) {
        return NO_ERROR;
    } else {
        AudioVolumeFactory::CreateAudioVolumeController()->setSpeakerType(type);
        mSpkType = type;
        ALOGD("%s() type=%d, mSpkType=%d", __FUNCTION__, type, mSpkType);
        return NO_ERROR;
    }
}

int SpeechModemController::getUlAnalogGain() {
    uint16_t gain = 0;

    gain = AudioVolumeFactory::CreateAudioVolumeController()->getVoiceUlAnalogGain(mCurrentOutputDevice,
                                                                                 AUDIO_MODE_IN_CALL);
    ALOGD("%s(), gain=%d, mSpeechOn=%d, mCurrentOutputDevice=%d",
          __FUNCTION__, gain, mSpeechOn, mCurrentOutputDevice);
    return (int)gain;
}

int SpeechModemController::getDlAnalogGain() {
    int16_t degradeGain = 0;
    int gain = 0;

    gain = AudioVolumeFactory::CreateAudioVolumeController()->getVoiceDlAnalogGain(mVolumeIndex,
                                                                                   mCurrentOutputDevice,
                                                                                   AUDIO_MODE_IN_CALL);
    ALOGD("%s(), gain=%d, mSpeechOn=%d, mVolumeIndex=%d, mCurrentOutputDevice=%d",
          __FUNCTION__, gain, mSpeechOn, mVolumeIndex, mCurrentOutputDevice);
    return gain;
}

int SpeechModemController::setVolumeIndex(const int index) {
    //mVolumeIndex: valid range 0 to 7
    //range 1~7(map to xml 0~6), 0 for mute
    if (index < 0 || index > 7) {
        ALOGW("%s(), invalid index=%d, valid range is 0 to 7",
              __FUNCTION__, index);
    } else {
        ALOGD("%s(), mVolumeIndex=%d->%d, mCurrentOutputDevice=%d",
              __FUNCTION__, mVolumeIndex, index, mCurrentOutputDevice);
        mVolumeIndex = index;
        AudioVolumeFactory::CreateAudioVolumeController()->setVoiceVolume(index, mCurrentOutputDevice, AUDIO_MODE_IN_CALL);
        SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
        return speechDriver->setMDVolumeIndex(0, 0, mVolumeIndex);
    }
    return NO_ERROR;
}

int SpeechModemController::setUplinkMute(const bool mute) {
    AL_AUTOLOCK(mUlMuteLock);
    ALOGD("%s() mute(%d ->%d)", __FUNCTION__, mUlMute, mute);
    if (mUlMute != mute) {
        mUlMute = mute;
        SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
        speechDriver->SetUplinkMute(mute);
    }
    return NO_ERROR;
}

int SpeechModemController::setDownlinkMute(const bool mute) {
    AL_AUTOLOCK(mDlMuteLock);
    ALOGD("%s() mute(%d ->%d)", __FUNCTION__, mDlMute, mute);
    if (mDlMute != mute) {
        mDlMute = mute;
        SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
        speechDriver->SetDownlinkMute(mute);
    }
    return NO_ERROR;
}

bool SpeechModemController::getUplinkMute() {
    AL_AUTOLOCK(mUlMuteLock);
    ALOGD("%s() mUlMute=%d", __FUNCTION__, mUlMute);
    return mUlMute;
}

bool SpeechModemController::getDownlinkMute() {
    AL_AUTOLOCK(mDlMuteLock);
    ALOGD("%s() mDlMute=%d", __FUNCTION__, mDlMute);
    return mDlMute;
}

int SpeechModemController::eCallIvsSwitch(bool enable) {
    ALOGD("%s(), mIsEcallIvsOn(%d->%d)", __FUNCTION__, mIsEcallIvsOn, enable);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    if (false == speechDriver->GetApSideModemStatus(SPEECH_STATUS_MASK)) {
        ALOGW("%s(), mIsEcallIvsOn(%d->%d), speech is off, return.", __FUNCTION__, mIsEcallIvsOn, enable);
        return -EFAULT;
    }
    if (mIsEcallIvsOn != enable) {
        speechDriver->eCallIvsSwitch(enable);
        mIsEcallIvsOn = enable;
    }
    return NO_ERROR;
}

int SpeechModemController::eCallIvsSend() {
    ALOGD("%s()", __FUNCTION__);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    speechDriver->eCallIvsSend();

    return NO_ERROR;
}

int SpeechModemController::eCallPsapSwitch(bool enable) {
    ALOGD("%s(), mIsEcallPsapOn(%d->%d)", __FUNCTION__, mIsEcallPsapOn, enable);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    if (false == speechDriver->GetApSideModemStatus(SPEECH_STATUS_MASK)) {
        ALOGW("%s(), mIsEcallPsapOn(%d->%d), speech is off, return.", __FUNCTION__, mIsEcallPsapOn, enable);
        return -EFAULT;
    }
    if (mIsEcallPsapOn != enable) {
        speechDriver->eCallPsapSwitch(enable);
        mIsEcallPsapOn = enable;
    }
    return NO_ERROR;
}

int SpeechModemController::eCallPsapSend() {
    ALOGD("%s()", __FUNCTION__);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    speechDriver->eCallPsapSend();

    return NO_ERROR;
}

int SpeechModemController::eCallCtrlSeqSwitch(bool enable) {
    ALOGD("%s(), mIsEcallCtrlSeqOn(%d->%d)", __FUNCTION__, mIsEcallCtrlSeqOn, enable);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    if (false == speechDriver->GetApSideModemStatus(SPEECH_STATUS_MASK)) {
        ALOGW("%s(), mIsEcallCtrlSeqOn(%d->%d), speech is off, return.", __FUNCTION__, mIsEcallCtrlSeqOn, enable);
        return -EFAULT;
    }
    if (mIsEcallCtrlSeqOn != enable) {
        speechDriver->eCallCtrlSeqSwitch(enable);
        mIsEcallCtrlSeqOn = enable;
    }
    return NO_ERROR;
}

int SpeechModemController::eCallMsd(void *data, uint16_t len) {
    ALOGD("%s(), len = %d", __FUNCTION__, len);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    speechDriver->eCallMsd(data, len);

    return NO_ERROR;
}

int SpeechModemController::eCallTxCtrlParam(void *data, uint16_t len) {
    ALOGD("%s(), len = %d", __FUNCTION__, len);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    speechDriver->eCallTxCtrlParam(data, len);

    return NO_ERROR;
}

int SpeechModemController::bgsOpen(uint32_t sampleRate) {
    ALOGD("%s(), sampleRate = %d", __FUNCTION__, sampleRate);
    SpeechDriverInterface *speechDriver = mSpeechDriverFactory->GetSpeechDriver();
    mBGSPlayBuffer = BGSPlayer::GetInstance()->CreateBGSPlayBuffer(sampleRate, 1, AUDIO_FORMAT_PCM_16_BIT);
    BGSPlayer::GetInstance()->Open(speechDriver, 0x0, 0xFF);

    return NO_ERROR;
}

int SpeechModemController::bgsClose() {
    ALOGD("%s()", __FUNCTION__);
    BGSPlayer::GetInstance()->Close();
    BGSPlayer::GetInstance()->DestroyBGSPlayBuffer(mBGSPlayBuffer);
    mBGSPlayBuffer = NULL;

    return NO_ERROR;
}

uint32_t SpeechModemController::bgsWrite(void *buf, uint32_t bytes) {
    uint32_t bytesConsumed = 0;
    bytesConsumed = BGSPlayer::GetInstance()->Write(mBGSPlayBuffer, buf, bytes);
    ALOGD("%s(), bytes = %d, bytesConsumed = %d", __FUNCTION__, bytes, bytesConsumed);

    return bytesConsumed;
}

} // end of namespace android

