// SPDX-License-Identifier: MediaTekProprietary
#include "AudioALSAPlaybackHandlerVoiceMixer.h"

#include "SpeechDriverFactory.h"
#include "SpeechVoiceMixer.h"

#include <SpeechUtility.h>


#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "AudioALSAPlaybackHandlerVoiceMixer"

namespace android {
/*
 * =============================================================================
 *                     Property keys
 * =============================================================================
 */
const char *PROPERTY_KEY_VOICEMIXER_NO_SLEEP = "persist.vendor.audiohal.speech.voicemixer.no_sleep";


/*
 * =============================================================================
 *                     class
 * =============================================================================
 */
AudioALSAPlaybackHandlerVoiceMixer::AudioALSAPlaybackHandlerVoiceMixer(const stream_attribute_t *stream_attribute_source) :
    AudioALSAPlaybackHandlerBase(stream_attribute_source),
    mVoiceMixerPlayer(VoiceMixerPlayer::getInstance()) {
    mSpeechDriver = NULL;
    mPlaybackHandlerType = PLAYBACK_HANDLER_VOICE;
    mVoiceMixerPlayBuffer = NULL;

    memset((void *)&mOpenTime, 0, sizeof(mOpenTime));
    memset((void *)&mCurTime, 0, sizeof(mCurTime));
    mWriteCnt = 0;

    memset((void *)&mNewtimeLatency, 0, sizeof(mNewtimeLatency));
    memset((void *)&mOldtimeLatency, 0, sizeof(mOldtimeLatency));
    memset((void *)&mLatencyTimeMs, 0, sizeof(mLatencyTimeMs));
    mLatencyUs = 0;
    mBypassVoiceMixerSleep = 0;
}

status_t AudioALSAPlaybackHandlerVoiceMixer::open() {
    // debug pcm dump
    OpenPCMDump(LOG_TAG);

    // HW attribute config
    mStreamAttributeTarget.audio_format = AUDIO_FORMAT_PCM_16_BIT;
    mStreamAttributeTarget.audio_channel_mask = AUDIO_CHANNEL_OUT_MONO;
    mStreamAttributeTarget.num_channels = popcount(mStreamAttributeTarget.audio_channel_mask);
    mStreamAttributeTarget.sample_rate = VOICEMIXER_TARGET_SAMPLE_RATE; // same as source stream
    mStreamAttributeTarget.buffer_size = VOICEMIXER_PLAY_BUFFER_LEN;
    mStreamAttributeTarget.mPcmMixTypeDl = VOICE_MIXER_TYPE_REPLACE;

    mLatencyUs = getBufferLatencyUs(mStreamAttributeSource,
                                    mStreamAttributeSource->buffer_size);

    ALOGD("%s(), audio_mode: %d, audio_format: %d => %d, sample_rate: %u => %u, ch: %u => %u, "
          " buffer_size: (write)%u, (voicemixer)%u, flag: 0x%x, mLatencyUs: %u, mPcmMixTypeDl: %u",
          __FUNCTION__, mStreamAttributeSource->audio_mode,
          mStreamAttributeSource->audio_format,
          mStreamAttributeTarget.audio_format,
          mStreamAttributeSource->sample_rate,
          mStreamAttributeTarget.sample_rate,
          mStreamAttributeSource->num_channels,
          mStreamAttributeTarget.num_channels,
          mStreamAttributeSource->buffer_size,
          mStreamAttributeTarget.buffer_size,
          mStreamAttributeSource->mAudioOutputFlags,
          (uint32_t)(mLatencyUs & 0xFFFFFFFF),
          mStreamAttributeTarget.mPcmMixTypeDl);
    // bit conversion
    initBitConverter();
    // open VoiceMixer
    if (mStreamAttributeTarget.num_channels > 2) {
        mVoiceMixerPlayBuffer = mVoiceMixerPlayer->createPlayBuffer(
                                    mStreamAttributeSource->sample_rate,
                                    2,
                                    mStreamAttributeTarget.audio_format);

    } else {
        mVoiceMixerPlayBuffer = mVoiceMixerPlayer->createPlayBuffer(
                                    mStreamAttributeSource->sample_rate,
                                    mStreamAttributeSource->num_channels,
                                    mStreamAttributeTarget.audio_format);
    }

    mSpeechDriver = SpeechDriverFactory::GetInstance()->GetSpeechDriver();
    mVoiceMixerPlayer->open(mSpeechDriver, mStreamAttributeTarget.mPcmMixTypeDl);

    mBypassVoiceMixerSleep = get_uint32_from_property(PROPERTY_KEY_VOICEMIXER_NO_SLEEP);

    clock_gettime(CLOCK_MONOTONIC, &mOpenTime);
    mWriteCnt = 0;

    clock_gettime(CLOCK_MONOTONIC, &mNewtimeLatency);
    mOldtimeLatency = mNewtimeLatency;

    return NO_ERROR;
}

status_t AudioALSAPlaybackHandlerVoiceMixer::close() {
    ALOGD("%s(), flag: 0x%x", __FUNCTION__, mStreamAttributeSource->mAudioOutputFlags);
    // close VoiceMixer
    mVoiceMixerPlayer->close();
    mVoiceMixerPlayer->destroyPlayBuffer(mVoiceMixerPlayBuffer);
    // bit conversion
    deinitBitConverter();
    // debug pcm dump
    ClosePCMDump();
    return NO_ERROR;
}

status_t AudioALSAPlaybackHandlerVoiceMixer::routing(const audio_devices_t output_devices __unused) {
    return INVALID_OPERATION;
}

ssize_t AudioALSAPlaybackHandlerVoiceMixer::write(const void *buffer, size_t bytes) {
    ALOGV("%s()", __FUNCTION__);

    uint64_t spendTimeUs = 0;
    uint64_t writeTimeUs = 0;
    uint64_t sleepUs = 0;
    uint32_t sourceChannelNum = mStreamAttributeSource->num_channels;
    uint32_t targetChannelNum = 2;// set target channel = 2 for doBitConversion(2->1)

    mWriteCnt++;

    if (mSpeechDriver == NULL) {
        ALOGW("%s(), mSpeechDriver == NULL!!", __FUNCTION__);
        return bytes;
    }

    clock_gettime(CLOCK_MONOTONIC, &mNewtimeLatency);
    mLatencyTimeMs[0] = get_time_diff_ms(&mOldtimeLatency, &mNewtimeLatency);
    mOldtimeLatency = mNewtimeLatency;

    if (mSpeechDriver->CheckModemIsReady() == false) {
        uint32_t sleep_ms = getBufferLatencyMs(mStreamAttributeSource, bytes);
        if (sleep_ms != 0) {
            ALOGW("%s(), modem not ready, sleep %u ms", __FUNCTION__, sleep_ms);
            usleep(sleep_ms * 1000);
        }
        return bytes;
    }

    if (buffer == NULL) {
        ALOGE("%s(), sourceBuf == NULL!!, bytes = %zu", __FUNCTION__, bytes);
        return bytes;
    }

    // make data convert according to source and target channel
    float channelRatio = ((float)targetChannelNum / sourceChannelNum);
    void *targetBuf = NULL;
    size_t targetBytes = (size_t)(channelRatio * bytes + 1); // +1 avoid round off error
    uint32_t sampleBytes = (mStreamAttributeSource->audio_format == AUDIO_FORMAT_PCM_16_BIT) ? 2 : 4;

    AUDIO_ALLOC_BUFFER(targetBuf, targetBytes);
    if (targetBuf == NULL) {
        ALOGE("%s(), targetBuf == NULL!!, targetBytes = %zu, bytes = %zu", __FUNCTION__, targetBytes, bytes);
        return bytes;
    }

    uint32 targetIdx = 0;
    uint32 sourceIdx = 0;
    uint32 targetCopyBytes = sampleBytes * targetChannelNum;
    uint32 sourceCopyBytes = sampleBytes * sourceChannelNum;
    unsigned char *sourceBufBase = (unsigned char *) buffer;
    unsigned char *targetBufBase = (unsigned char *) targetBuf;

    // copy data with different index move
    for (targetIdx = 0; sourceIdx < bytes; targetIdx += targetCopyBytes) {
        memcpy(targetBufBase + sourceIdx, sourceBufBase + targetIdx, targetCopyBytes);
        sourceIdx += sourceCopyBytes;
    }
    ALOGV("%s() Source_ch:%d, bytes:%zu, Target_ch:%d, bytes:%zu, channel_ratio:%f",
          __FUNCTION__, sourceChannelNum, bytes, targetChannelNum, targetBytes, channelRatio);

    // bit conversion
    void *pBufferAfterBitConvertion = NULL;
    uint32_t bytesAfterBitConvertion = 0;
    doBitConversion(targetBuf, targetBytes, &pBufferAfterBitConvertion, &bytesAfterBitConvertion);

    // write data to background sound
    WritePcmDumpData(pBufferAfterBitConvertion, bytesAfterBitConvertion);

    uint32_t writtenBytes = VoiceMixerPlayer::getInstance()->write(mVoiceMixerPlayBuffer, pBufferAfterBitConvertion,
                                                                   bytesAfterBitConvertion);
    if (writtenBytes != bytesAfterBitConvertion) { // TODO: 16/32
        ALOGE("%s(), VoiceMixerPlayer::getInstance()->write() error, writtenBytes(%u) != bytesAfterBitConvertion(%u)",
              __FUNCTION__, writtenBytes, bytesAfterBitConvertion);
    }
    clock_gettime(CLOCK_MONOTONIC, &mNewtimeLatency);
    mLatencyTimeMs[1] = get_time_diff_ms(&mOldtimeLatency, &mNewtimeLatency);
    mOldtimeLatency = mNewtimeLatency;

    /* HAL sleep latency time to consume data smoothly */
    if (mBypassVoiceMixerSleep == false) {
        clock_gettime(CLOCK_MONOTONIC, &mCurTime);
        spendTimeUs = get_time_diff_ns(&mOpenTime, &mCurTime) / 1000;
        writeTimeUs = mWriteCnt * mLatencyUs;
        if (spendTimeUs < writeTimeUs) {
            sleepUs = writeTimeUs - spendTimeUs;
            if (mVoiceMixerPlayBuffer->isBufferEnough()) {
                usleep(sleepUs);
            } else {
                if (sleepUs > 1000) {
                    sleepUs -= 1000;
                    usleep(sleepUs);
                } else {
                    sleepUs = 0;
                }
            }
        } else if (spendTimeUs > (writeTimeUs + MODEM_FRAME_MS * 1000)) {
            if (getLogEnableByLevel(VOICEMIXER_LOG_LEVEL_PLAYBACK_HANDLER)) {
                ALOGW("%s(), spendTimeUs %u, writeTimeUs %u", __FUNCTION__,
                      (uint32_t)(spendTimeUs & 0xFFFFFFFF),
                      (uint32_t)(writeTimeUs & 0xFFFFFFFF));
            }
        }
    }

    clock_gettime(CLOCK_MONOTONIC, &mNewtimeLatency);
    mLatencyTimeMs[2] = get_time_diff_ms(&mOldtimeLatency, &mNewtimeLatency);
    mOldtimeLatency = mNewtimeLatency;

    uint64_t logThresholdMs = (mLatencyUs) / 1000;
    if (logThresholdMs < MODEM_FRAME_MS) {
        logThresholdMs = MODEM_FRAME_MS;
    }

    if (mLatencyTimeMs[0] > logThresholdMs ||
        mLatencyTimeMs[1] > logThresholdMs ||
        mLatencyTimeMs[2] > logThresholdMs) {
        ALOGW("%s(), latency_in_ms, %3u, %3u, %3u, writtenBytes: %u, mLatencyUs: %u, spendTimeUs: %u, "
              "writeTimeUs: %u, sleepUs: %u", __FUNCTION__,
              (uint32_t)(mLatencyTimeMs[0] & 0xFFFFFFFF),
              (uint32_t)(mLatencyTimeMs[1] & 0xFFFFFFFF),
              (uint32_t)(mLatencyTimeMs[2] & 0xFFFFFFFF),
              writtenBytes,
              (uint32_t)(mLatencyUs & 0xFFFFFFFF),
              (uint32_t)(spendTimeUs & 0xFFFFFFFF),
              (uint32_t)(writeTimeUs & 0xFFFFFFFF),
              (uint32_t)(sleepUs & 0xFFFFFFFF));
    } else if (getLogEnableByLevel(VOICEMIXER_LOG_LEVEL_PLAYBACK_HANDLER)) {
        ALOGD("%s(), latency_in_ms, %3u, %3u, %3u, writtenBytes: %u, mLatencyUs: %u, spendTimeUs: %u, "
              "writeTimeUs: %u, sleepUs: %u", __FUNCTION__,
              (uint32_t)(mLatencyTimeMs[0] & 0xFFFFFFFF),
              (uint32_t)(mLatencyTimeMs[1] & 0xFFFFFFFF),
              (uint32_t)(mLatencyTimeMs[2] & 0xFFFFFFFF),
              writtenBytes,
              (uint32_t)(mLatencyUs & 0xFFFFFFFF),
              (uint32_t)(spendTimeUs & 0xFFFFFFFF),
              (uint32_t)(writeTimeUs & 0xFFFFFFFF),
              (uint32_t)(sleepUs & 0xFFFFFFFF));
    }

    AUDIO_FREE_POINTER(targetBuf);
    return targetBytes;
}

uint32_t AudioALSAPlaybackHandlerVoiceMixer::chooseTargetSampleRate(uint32_t sampleRate) {
    ALOGD("ChooseTargetSampleRate sampleRate = %d ", sampleRate);
    uint32_t targetSampleRate = 44100;
    if ((sampleRate % 8000) == 0) { // 8K base
        targetSampleRate = 48000;
    }
    return targetSampleRate;
}

int AudioALSAPlaybackHandlerVoiceMixer::configMixType(const uint8_t mixType) {
    if (mixType != mStreamAttributeTarget.mPcmMixTypeDl) {
        ALOGD("%s(), dlMixType(%d ->%d)", __FUNCTION__, mStreamAttributeTarget.mPcmMixTypeDl, mixType);
        mStreamAttributeTarget.mPcmMixTypeDl = mixType;
        return mVoiceMixerPlayer->configMixType(mSpeechDriver, mStreamAttributeTarget.mPcmMixTypeDl);
    } else {
        ALOGV("%s(), the same mPcmMixTypeDl(%d)", __FUNCTION__, mStreamAttributeTarget.mPcmMixTypeDl);
        return NO_ERROR;
    }
}


} // end of namespace android
